// Description of csm_access_usr.c at the end of the file.
// File name for preprozessor commands
#define CSM_ACCESS_USR_SELF

// own definitions
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// csm_access_usr.c is a file according V-team CAN-SW architecture implementation rules
#define CSM_CAN_STACK
// #define CSM_NO_TYPE_REDEFINITION                    // not used anywhere !!

#define CSM_S_IMPORT_INTERFACE_GENERIC_USER
#include "csm_if.h"
#include "csm_stack_m.h"

// Trace class definition for ETG (extented trace generator).
#define TR_CLASS_CSMUSER (TR_COMP_CSM + 0x12)

#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMUSER
    // #define ETG_INLINE
    #include "trcGenProj/Header/csm_access_usr.cpp.trc.h"
  #endif
  #endif
#endif // CSM_S_TRACE

//******************************************************************************
//* defines and macros (scope: file local)
//******************************************************************************
// First part of the name of the semaphore which protects the access functions
#define CSMAL_C_ACCESS_SEM_NAME_PREFIX "CSMALU_SEM_ACC_"

/* Feature switch for freeing system resources. This measure avoids
 * exceptions during system shut down
 */

/** Due to massive dependencies in implementation, the flag for NORMAL/FATAL is
    passed to OSAL_vAssertFunction() as MSB of __LINE__ which should usually not
    occur to be so large :)
    b31 = 1  -->  NORMAL (no assert, only trace, in RELEASE compiled version)
    b31 = 0  -->  FATAL (behaviour as usual)
*/
#define CSMALU_vAssert(_INT_)         ((_INT_) ? (void)(0) : (csmalu_vAssertFunction(#_INT_, __FILE__, (__LINE__)|0x80000000)) )

/**  Use it for FATAL asserts, which should suspend task or system
    in debug and release version
    _INT_ == 0 -> Assert
          == 1 -> NOP
    Handling of error equal in Debug and Release
*/

// currently not used
//#define CSMALU_M_FATAL_ASSERT(_INT_)  ((_INT_) ? (void)(0) : (CSM_vAssertFunction(#_INT_, __FILE__, (__LINE__)&(~0x80000000))) )

/**  Use it for NORMAL asserts, which should suspend task or system
    in debug version but resuming task operation in release version
    _INT_ == 0 -> Assert
          == 1 -> NOP
    Handling of error differs in Debug and Release
*/

// currently not used
// #define CSMALU_M_NORMAL_ASSERT(_INT_)  CSM_vAssert(_INT_)

// Legacy: Default value for the access layer user task priority
#define CSM_ACCESS_SERVICETASK_PRIO         100

//******************************************************************************
//* typedefs (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* variable declaration (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* variable definition (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* function prototypes (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* function prototypes (scope: external access by functionpointer)
//******************************************************************************
// none

/*******************************************************************************
 * function    csmalu_vAssertFunction
 * \doxydocu
 * \brief      This Function is called after an interrupt
 *
 * \param[in]  coszExp
 *              pointer to assert expression
 *
 * \param[in]  coszFile
 *              pointer to sourcefile name
 *
 * \param[in]  u32Line
 *              Line number in source file
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  no
 *
 ******************************************************************************/
void csmalu_vAssertFunction(const char* coszExp, const char* coszFile, DWORD u32Line)
{
  (tVoid)coszExp;
  (tVoid)coszFile;
  (tVoid)u32Line;
  NORMAL_M_ASSERT_ALWAYS();
}

// Feature switch is used only to control doxygen documentation
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
//******************************************************************************
//* function implementation (scope: file local = static)
//******************************************************************************
// none

//******************************************************************************
//*  function implementation (scope: module-global)
//******************************************************************************
// none
#endif // CSM_S_DOXYGEN_HIDE_NON_API_FCT

//******************************************************************************
//* function implementation (scope: CAN stack #1 API)
//******************************************************************************

#undef CSM_ACCESS_USR_SELF /*lint !e750 */ /* file identifier can be used in (project specific) header files - 05.02.08 krv2hi */

/*******************************************************************************
 * function    CSM_lExceptionHandler
 * \doxydocu
 * \brief      This function will be called by the macro CSM_M_MAKE_ERROR
 *             in cases of warnings, errors and severe errors.
 *
 *
 * \param[in]  u8ErrorGrade
 *             error grade
 *             Allowed values:
 *                CSM_C_ERR_INFO
 *                CSM_C_ERR_WARNING
 *                CSM_C_ERR_ERROR
 *                CSM_C_ERR_SEVERE_ERROR
 *
 * \param[in]  u8ErrorLocation
 *             error location
 *             Allowed values:
 *                CSM_C_ERR_LOC_GLB
 *                CSM_C_ERR_LOC_CSM
 *                CSM_C_ERR_LOC_CNM
 *                CSM_C_ERR_LOC_CNP
 *                CSM_C_ERR_LOC_CNP_MCNET
 *                CSM_C_ERR_LOC_CNP_ISO_TP
 *                CSM_C_ERR_LOC_CNP_VAG_TP
 *                CSM_C_ERR_LOC_CBR
 *                CSM_C_ERR_LOC_CDA
 *                CSM_C_ERR_LOC_CSM_STACK
 *                CSM_C_ERR_LOC_CSM_APL
 *
 * \param[in]  u8ErrorGrade
 *             error, code, see modules
 *
 * \return     True/False
 *
 * \access     XXXXXXXXXXXXXXXXXXXXXXXX
 * \reentrant  no
 *
 ******************************************************************************/
#if (CSM_S_EXCEPTION_HANDLER_ACTIVATED == CSM_C_F_ON)
LONG CSM_lExceptionHandler(BYTE u8ErrorGrade, BYTE u8ErrorLocation, WORD u16ErrorCode)
{
  LONG u32ErrorValue = (-(u8ErrorGrade << 24) - (u8ErrorLocation << 16) - u16ErrorCode);

  if(u16ErrorCode != CSM_E_INIT_WARNING)
  {
    //csm_access_usr_u32ErrorCounter++;
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR1(("[0x%08X] CSM_lExceptionHandler - 0x%02X 0x%02X 0x%04X", _hProcessID, u8ErrorGrade, u8ErrorLocation, u16ErrorCode));
    #endif
  }
  return u32ErrorValue;
}
#endif

#undef CSM_CAN_STACK

//*************************************************************************************************************************
//* implementation of the CSM access user class
//*************************************************************************************************************************
//***********************
// static class members
//***********************
tU8  csm_tclCsmAccessUser::_u8AccessSemCounter = 0;
tU32 csm_tclCsmAccessUser::_u32MessageQueueCounter = 0;         // we always start with no message queue
tU8  csm_tclCsmAccessUser::_u8ServiceTaskCounter = 0;

//*************************
// public class functions
//*************************
// constructor with initializer list (max. entries of message queue (size) is a default value as long as not overwritten later on)
csm_tclCsmAccessUser::csm_tclCsmAccessUser()
                     :_pvCsmHandle(CSM_C_INVALID_HANDLE),
                      _u32MessageQueueCount(0),
                      _hParentMessageQueueID(OSAL_C_INVALID_HANDLE),
                      _hServiceMessageQueueID(OSAL_C_INVALID_HANDLE),
                      _hProcMessageQueueID(OSAL_C_INVALID_HANDLE),
                      _u32MaxNrOfServiceQueueElements(CSM_C_SERVICE_MESSAGE_QUEUE_MAX_ENTRIES_DEFAULT),
                      _u32ServiceQueueElementSize(CSM_C_SERVICE_MESSAGE_QUEUE_MAX_MESSAGE_SIZE)
{
  // get process ID
  _hProcessID = OSAL_ProcessWhoAmI();
  _u8AccessSemCounter++;
  _u8AccessSemCount = _u8AccessSemCounter;
  _hAccessSem = _hCreateAccessSemaphore();
}

// destructor
csm_tclCsmAccessUser::~csm_tclCsmAccessUser()
{
  if ( OSAL_C_INVALID_HANDLE != _hProcMessageQueueID )
  {
    OSAL_s32MessageQueueClose( _hProcMessageQueueID );
    _hProcMessageQueueID = OSAL_C_INVALID_HANDLE;
  }

  if ( OSAL_C_INVALID_HANDLE != _hParentMessageQueueID )
  {
    OSAL_s32MessageQueueClose( _hParentMessageQueueID );
    _hParentMessageQueueID = OSAL_C_INVALID_HANDLE;
  }
  _vDeleteAccessSemaphore();
}

//****************************************************************************************************************
// pre-INIT
// is the first function to be called after having created an instance of this class
// this is ensured, if using the CSMIF wrapper from the middleware
// if Nr. of QueueElements is NOT given, a default value is set (this is relevant for the service message queue)
//****************************************************************************************************************
tVoid csm_tclCsmAccessUser::vApplCallbackPreInit(tU32 u32NrOfMsgQueueElements, tU32 u32MsgPayloadSize)
{
  // diese Funktion muss erst komplett beenden, bevor sie erneut aufgerufen werden kann !!!
  OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

  // increment the static counter and store the current value for this object
  _u32MessageQueueCounter++;
  _u32MessageQueueCount = _u32MessageQueueCounter;

  // create message queue for synchronous application requests (len is only 1 queue element)
  _hParentMessageQueueID = _hCreateParentMessageQueue();

  // check required (given) nr of max. queue elements (for asynchronous requests and results)
  if (u32NrOfMsgQueueElements != 0)
  {
    // overwrite our member var (default value) with the given one
    _u32MaxNrOfServiceQueueElements = u32NrOfMsgQueueElements;
  }

  // check required (given) queue element size (for asynchronous requests and results)
  if (u32MsgPayloadSize != 0)
  {
    // overwrite our member var (default value) with the given one, if not too large
    if (u32MsgPayloadSize <= CSMAL_MAX_DATA_PAYLOAD)
    {
      // but keep in mind, that the message len is the payload plus some admin data
      _u32ServiceQueueElementSize = (sizeof(trDataIndMsg)) + u32MsgPayloadSize;
    }
  }

  // check if synchronous queue has been successfully created
  if (_hParentMessageQueueID != OSAL_C_INVALID_HANDLE )
  {
    // now create message queue for asynchronous events and requests
    _hServiceMessageQueueID = _hCreateServiceMessageQueue();

    // check if asynchronous service has been successfully created
    if (_hServiceMessageQueueID != OSAL_C_INVALID_HANDLE )
    {
      // create user mode service task
      if ( OSAL_ERROR == _hCreateServiceTask() )
      {
        // no task created - hier wird das 2. Mal asserted !!
        CSMALU_vAssert(FALSE); /*lint !e506 */ /* Not created task shall cause an assert - krv2hi 2010-09-24 */
      }
      else
      {
        // Send INIT message to CSM process
        if (FALSE == _bInitCSMProc() )
        {
          // the INIT message could NOT be sent to ProcCsm, so we terminate our service task
          // by sending a message via our own service message queue
          // this is the most proper way, since we do not know, what kind of ressources have been used upon creation
          trMsgHeader rMsgHeader;
          rMsgHeader.enMsgType = CSM_ACCESS_TerminateTask;
          OSAL_s32MessageQueuePost( _hServiceMessageQueueID,
                                    (tPCU8)&rMsgHeader,
                                    (tU32)sizeof(rMsgHeader),
                                    OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST
                                  );
          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_ERR(("[0x%08X] vApplCallbackPreInit - Init CSM process not successful", _hProcessID));
          #endif
        }
        else
        {
          // INIT message successfully sent to ProcCsm
          // so now we wait here at the parent message queue !!!
          trCSMParentMessageQueueItem  rMessage;
          tS32                         s32ReadBytes;

          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_USR2(("[0x%08X] vApplCallbackPreInit - Wait at parent message queue 0x%X", _hProcessID, _hParentMessageQueueID));
          #endif

          s32ReadBytes = OSAL_s32MessageQueueWait( _hParentMessageQueueID,
                                                   (tPU8)&rMessage,
                                                   sizeof(rMessage),
                                                   OSAL_NULL,               // equal priorities
                                                   OSAL_C_U32_INFINITE );

          // message received - waiting is over
          if (s32ReadBytes > 0)
          {
            // message has content
            if ( CSM_PARENT_MESSAGE_QUEUE_ID_INIT == rMessage.enID )
            {
              // INIT message !!! - yippie
              // store the given handle !!
              _pvCsmHandle = rMessage.uData.rInit.pvHandle;
              #if (CSM_S_TRACE == CSM_C_F_ON)
              ETG_TRACE_USR1(("[0x%08X] vApplCallbackPreInit - Init message received - Handle:0x%08X", _hProcessID, _pvCsmHandle));
              #endif
            }
            else
            {
              // wrong message !!! - not good
              #if (CSM_S_TRACE == CSM_C_F_ON)
              ETG_TRACE_ERR(("[0x%08X] vApplCallbackPreInit - Wrong message ID 0x%X", _hProcessID, rMessage.enID));
              #endif
            }
          }
          else
          {
            // empty message or error
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_ERR(("[0x%08X] vApplCallbackPreInit - Read error", _hProcessID));
            #endif
          }
        } // WAIT for INIT response message
      }  // service task created
    } // service message queue created
    else
    {
       // service message queue NOT created
       CSMALU_vAssert(FALSE);  /*lint !e506 */ /* Not created task shall cause an assert - krv2hi 2010-09-24 */
    }
  } // parent message queue created
  else
  {
    // parent message queue NOT created
     CSMALU_vAssert(FALSE);  /*lint !e506 */ /* Not created message buffer shall cause an assert - krv2hi 2010-09-24 */
  }

  // now release semaphore so that the next user application can create its CSM IF
  OSAL_s32SemaphorePost(_hAccessSem);
}

//****************************************************************************************************************
// service function, ApplCallbackInit, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32ApplCallbackInit(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, const tVoid* pcvCallBackFkt)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;
    const tU8* pu8AddressField = (const tU8*) pcvAddressField;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_APPL;
    rMessage.uData.rRegisterAppl.hProcessID      = _hProcessID;
    rMessage.uData.rRegisterAppl.u32MsgQueueCnt  = _u32MessageQueueCount;
    rMessage.uData.rRegisterAppl.pvHandle        = _pvCsmHandle;
    rMessage.uData.rRegisterAppl.u8Bus           = u8Bus;
    rMessage.uData.rRegisterAppl.u32ProtocolType = u32ProtocolType;

    // fill all callback function pointers of the message in dependence of the given protocol type
    s32RetVal = _s32FillCallbackFunctionsForApplCallbackInit( &(rMessage.uData.rRegisterAppl), u32ProtocolType, pcvCallBackFkt );

    if (CSM_C_NO_ERROR == s32RetVal)
    {
      // callbacks filled correctly
      for (tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++)
      {
        // fill addressfield of message
        if (pu8AddressField != NULL)
        {
          rMessage.uData.rRegisterAppl.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
        }
        else
        {
          rMessage.uData.rRegisterAppl.au8AddressField[u32Cnt] = 0;
        }
      }

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("[0x%08X] s32ApplCallbackInit - MQcount:0x%04X  Handle:0x%08X  Bus:%X  Prot:%X  AddrField:0x%02X%02X  Callback:0x%08X",
                      _hProcessID,
                      _u32MessageQueueCount,
                      _pvCsmHandle,
                      ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  u8Bus),
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                      rMessage.uData.rRegisterAppl.au8AddressField[0], rMessage.uData.rRegisterAppl.au8AddressField[1], pcvCallBackFkt));
      #endif

      // check semaphore and post message to ProcCsm queue
      OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

      // send message to process queue and wait for answer
      if (OSAL_OK == OSAL_s32MessageQueuePost( _hProcMessageQueueID,
                                               (tPCU8)&rMessage,
                                               (tU32)sizeof(rMessage),
                                               OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST ) )
      {
        // message written successfully, wait for response
        s32RetVal = _s32WaitForGeneralResult((tCString)"s32ApplCallbackInit", CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_APPL);
      }
      else
      {
        // message post failed -> error
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
      }

      // free semaphore
      OSAL_s32SemaphorePost(_hAccessSem);
    } // callbacks filled correctly
  }
  else
  {
    // no valid queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, ApplCallbackDeInit, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32ApplCallbackDeInit()
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_APPL;
    rMessage.uData.rUnregisterAppl.hProcessID     = _hProcessID;
    rMessage.uData.rUnregisterAppl.u32MsgQueueCnt = _u32MessageQueueCount;
    rMessage.uData.rUnregisterAppl.pvHandle       = _pvCsmHandle;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32ApplCallbackDeInit - MQcount:0x%04X  Handle:0x%08X",
                    _hProcessID, _u32MessageQueueCount, _pvCsmHandle));
    #endif

    // check sempahore and post message to ProcCsm queue
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message written, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32ApplCallbackDeInit", CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_APPL);
    }
    else
    {
      // error writing message
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // no valid queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, SignalCallbackInit, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32SignalCallbackInit(tU32 u32SignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_SIGNAL;
    rMessage.uData.rRegisterSignal.hProcessID       = _hProcessID;
    rMessage.uData.rRegisterSignal.u32MsgQueueCnt   = _u32MessageQueueCount;
    rMessage.uData.rRegisterSignal.pvHandle         = _pvCsmHandle;
    rMessage.uData.rRegisterSignal.u32SignalId      = u32SignalId;
    rMessage.uData.rRegisterSignal.vPFNCBRSignalInd = vPFNCBRSignalInd;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32SignalCallbackInit - MQcount:0x%04X  Handle:0x%08X  Signal:0x%08X  Callback:0x%08X",
                    _hProcessID, _u32MessageQueueCount, _pvCsmHandle, u32SignalId, vPFNCBRSignalInd));
    #endif

    // check semaphore and post message to ProcCsm
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message written -> wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32SignalCallbackInit", CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_SIGNAL);
    }
    else
    {
      // message writing failed
      s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN );
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // no valid queues -> error
    s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE );
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, SignalCallbackDeInit, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32SignalCallbackDeInit(tU32 u32SignalId)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL;
    rMessage.uData.rUnregisterSignal.hProcessID     = _hProcessID;
    rMessage.uData.rUnregisterSignal.u32MsgQueueCnt = _u32MessageQueueCount;
    rMessage.uData.rUnregisterSignal.pvHandle       = _pvCsmHandle;
    rMessage.uData.rUnregisterSignal.u32SignalId    = u32SignalId;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32SignalCallbackDeInit - MQcount:0x%04X  Handle:0x%08X  Signal:0x%08X",
                    _hProcessID, _u32MessageQueueCount, _pvCsmHandle, u32SignalId));
    #endif

    // check semaphore and pot messag to ProcCsm
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost( _hProcMessageQueueID,
                                             (tPCU8)&rMessage,
                                             (tU32)sizeof(rMessage),
                                             OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message written -> wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32SignalCallbackDeInit", CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL);
    }
    else
    {
      // error writing message
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // no valid queue -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32SignalCallbackDeInitAll, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32SignalCallbackDeInitAll()
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS;
    rMessage.uData.rUnregisterAllSignals.hProcessID     = _hProcessID;
    rMessage.uData.rUnregisterAllSignals.u32MsgQueueCnt = _u32MessageQueueCount;
    rMessage.uData.rUnregisterAllSignals.pvHandle       = _pvCsmHandle;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32SignalCallbackDeInitAll - MQcount:0x%04X  Handle:0x%08X",
                    _hProcessID, _u32MessageQueueCount, _pvCsmHandle));
    #endif

    // check semaphore and post message
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost( _hProcMessageQueueID,
                                             (tPCU8)&rMessage,
                                             (tU32)sizeof(rMessage),
                                             OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message sent, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32SignalCallbackDeInitAll", CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS);
    }
    else
    {
      // message writing failed
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32BusErrorIndCallbackInit, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32BusErrorIndCallbackInit(tCSM_PFN_BUS_ERROR_IND vPFNBusErrorInd)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND;
    rMessage.uData.rRegisterBusErrorInd.hProcessID      = _hProcessID;
    rMessage.uData.rRegisterBusErrorInd.u32MsgQueueCnt  = _u32MessageQueueCount;
    rMessage.uData.rRegisterBusErrorInd.pvHandle        = _pvCsmHandle;
    rMessage.uData.rRegisterBusErrorInd.vPFNBusErrorInd = vPFNBusErrorInd;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32BusErrorIndCallbackInit - MQcount:0x%04X  Handle:0x%08X  Callback:0x%08X",
                    _hProcessID, _u32MessageQueueCount, _pvCsmHandle, vPFNBusErrorInd));
    #endif

    // check semaphore
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost( _hProcMessageQueueID,
                                             (tPCU8)&rMessage,
                                             (tU32)sizeof(rMessage),
                                             OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message sent, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32BusErrorIndCallbackInit", CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND);
    }
    else
    {
      // message writing failed
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid queue -> error
    s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE );
  }

  #else
  // feature disabled -> no service
  s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_SRVC_UNAVAILABLE);
  #endif
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32CommunicationReq, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32CommunicationReq(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, tU8 u8Action, tU16 u16ApplID)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;
    const tU8* pu8AddressField = (const tU8*) pcvAddressField;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_COMMUNICATION_REQ;
    rMessage.uData.rCommunicationReq.hProcessID      = _hProcessID;
    rMessage.uData.rCommunicationReq.u32MsgQueueCnt  = _u32MessageQueueCount;
    rMessage.uData.rCommunicationReq.u8Bus           = u8Bus;
    rMessage.uData.rCommunicationReq.u32ProtocolType = u32ProtocolType;
    rMessage.uData.rCommunicationReq.u8Action        = u8Action;
    rMessage.uData.rCommunicationReq.u16ApplID       = u16ApplID;

    // loop over all possible addressfields
    for ( tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++ )
    {
      if ( pu8AddressField != NULL )
      {
        // drop given byte
        rMessage.uData.rCommunicationReq.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
      }
      else
      {
        // set byte to 0
        rMessage.uData.rCommunicationReq.au8AddressField[u32Cnt] = 0;
      }
    }

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32CommunicationReq - MQcount:0x%04X  Bus:%X  Prot:0x%08X  AddrField:0x%02X%02X  Action:0x%X  AppID:0x%04X",
                    _hProcessID,
                    _u32MessageQueueCount,
                    ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  u8Bus),
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                    rMessage.uData.rCommunicationReq.au8AddressField[0], rMessage.uData.rCommunicationReq.au8AddressField[1], u8Action, u16ApplID));
    #endif

    // check semaphore and pots message
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message sent, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32CommunicationReq", CSM_PARENT_MESSAGE_QUEUE_ID_COMMUNICATION_REQ);
    }
    else
    {
      // message writing failed
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32GetCommunicationState, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32GetCommunicationState(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, tPU8 pu8ConnectState, tU16 u16ApplID)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rOutMessage;
    const tU8* pu8AddressField = (const tU8*) pcvAddressField;

    rOutMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE;
    rOutMessage.uData.rGetCommunicationState.hProcessID      = _hProcessID;
    rOutMessage.uData.rGetCommunicationState.u32MsgQueueCnt  = _u32MessageQueueCount;
    rOutMessage.uData.rGetCommunicationState.u8Bus           = u8Bus;
    rOutMessage.uData.rGetCommunicationState.u32ProtocolType = u32ProtocolType;
    rOutMessage.uData.rGetCommunicationState.u16ApplID       = u16ApplID;

    // loop over addressfield bytes
    for ( tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++ )
    {
      if ( pu8AddressField != NULL )
      {
        rOutMessage.uData.rGetCommunicationState.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
      }
      else
      {
        rOutMessage.uData.rGetCommunicationState.au8AddressField[u32Cnt] = 0;
      }
    }

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32GetCommunicationState - MQcount:0x%04d  Bus:%X  Prot:0x%08X  AddrField:0x%02X%02X  AppID:0x%04X",
                    _hProcessID,
                    _u32MessageQueueCount,
                    ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  u8Bus),
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                    rOutMessage.uData.rGetCommunicationState.au8AddressField[0], rOutMessage.uData.rGetCommunicationState.au8AddressField[1], u16ApplID));
    #endif

    // check semaphore and post message
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rOutMessage,
                                            (tU32)sizeof(rOutMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message sent, wait for response
      trCSMParentMessageQueueItem  rInMessage;
      tS32  s32ReadBytes;

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR2(("[0x%08X] s32GetCommunicationState - Wait at parent message queue 0x%X", _hProcessID, _hParentMessageQueueID));
      #endif

      s32ReadBytes = OSAL_s32MessageQueueWait(_hParentMessageQueueID,
                                              (tPU8)&rInMessage,
                                              sizeof(rInMessage),
                                              OSAL_NULL,  // equal priorities
                                              OSAL_C_U32_INFINITE);

      // response received
      if (s32ReadBytes > 0)
      {
        // expected ID ?
        if (CSM_PARENT_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE == rInMessage.enID)
        {
          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_USR1(("[0x%08X] s32GetCommunicationState - Answer received - Result:0x%08X  ConnectionState:0x%X",
                          _hProcessID,
                          rInMessage.uData.rGetCommunicationState.s32Result,
                          rInMessage.uData.rGetCommunicationState.u8State));
          #endif

          // get the specific response result
          s32RetVal = rInMessage.uData.rGetCommunicationState.s32Result;
          if (CSM_C_NO_ERROR == s32RetVal)
          {
            *pu8ConnectState = rInMessage.uData.rGetCommunicationState.u8State;
          }
        }
        else
        {
          // unexpected message -> error
          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_ERR(("[0x%08X] s32GetCommunicationState - Wrong message ID:0x%X", _hProcessID, rInMessage.enID));
          #endif
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_EVENT);
        }
      }
      else
      {
        // error reading message
        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_ERR(("[0x%08X] s32GetCommunicationState - Read error", _hProcessID));
        #endif
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_RX_QUEUE_OVERRUN);
      }
    }
    else
    {
      // error writing message
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32DataReq, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32DataReq(tU32 u32ProtocolType, const tVoid* pcvAddressField, tPCU8 pcu8Data, tU16 u16DataLength)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;
  tU32 u32Cnt;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID ) && ( OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    if (NULL != pcu8Data)
    {
      if ( u16DataLength <= CSM_C_ACCESS_LAYER_DATA_REQUEST_MAXLENGTH )
      {
        // create message
        trCSMProcMessageQueueItem  rMessage;
        const tU8* pu8AddressField = (const tU8*) pcvAddressField;

        rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_DATA_REQUEST;
        rMessage.uData.rDataRequest.hProcessID      = _hProcessID;
        rMessage.uData.rDataRequest.u32MsgQueueCnt  = _u32MessageQueueCount;
        rMessage.uData.rDataRequest.u32ProtocolType = u32ProtocolType;

        // fill address field
        for (u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++)
        {
          if (pu8AddressField != NULL)
          {
            rMessage.uData.rDataRequest.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
          }
          else
          {
            rMessage.uData.rDataRequest.au8AddressField[u32Cnt] = 0;
          }
        }

        // fill data
        rMessage.uData.rDataRequest.u16DataLength = u16DataLength;
        for ( u32Cnt = 0; u32Cnt < u16DataLength; u32Cnt++ )
        {
          rMessage.uData.rDataRequest.au8Data[u32Cnt] = *(pcu8Data + u32Cnt);
        }

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR1(("[0x%08X] s32DataReq - MQcount:0x%04X  Prot:0x%08X  AddrField:0x%02X%02X  Len:0x%X  Data:0x%02X",
                        _hProcessID,
                        _u32MessageQueueCount,
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                        rMessage.uData.rDataRequest.au8AddressField[0],
                        rMessage.uData.rDataRequest.au8AddressField[1],
                        u16DataLength,
                        ETG_LIST_LEN(u16DataLength),
                        ETG_LIST_PTR_T8(&rMessage.uData.rDataRequest.au8Data[0])
                      ));
        #endif

        // check semaphore and post message
        OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

        // send message to process queue and wait for answer
        if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                                (tPCU8)&rMessage,
                                                (tU32)sizeof(rMessage),
                                                OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
        {
          // message sent, wait for result
          s32RetVal = _s32WaitForGeneralResult((tCString)"s32DataReq", CSM_PARENT_MESSAGE_QUEUE_ID_DATA_REQUEST);
        }
        else
        {
          // error writing message
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
        }

        // free semaphore
        OSAL_s32SemaphorePost(_hAccessSem);
      }
      else
      {
        // invalid data length
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_DATA_LENGTH_EXCEEDED);
      }
    }
    else
    {
      // invalid data
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  }
  else
  {
    // invalid message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE );
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, s32DataIndProcessed, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32DataIndProcessed(tU32 u32ProtocolType, const tVoid* pcvAddressField)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;
    const tU8* pu8AddressField = (const tU8*) pcvAddressField;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED;
    rMessage.uData.rDataIndProcessed.hProcessID      = _hProcessID;
    rMessage.uData.rDataIndProcessed.u32MsgQueueCnt  = _u32MessageQueueCount;
    rMessage.uData.rDataIndProcessed.u32ProtocolType = u32ProtocolType;

    // fill addressfield
    for (tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++)
    {
      if (pu8AddressField != NULL)
      {
        rMessage.uData.rDataIndProcessed.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
      }
      else
      {
        rMessage.uData.rDataIndProcessed.au8AddressField[u32Cnt] = 0;
      }
    }

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32DataIndProcessed - MQcount:0x%04X  Prot:0x%08X  AddrField:0x%02X%02X",
                    _hProcessID,
                    _u32MessageQueueCount,
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                    rMessage.uData.rDataIndProcessed.au8AddressField[0],
                    rMessage.uData.rDataIndProcessed.au8AddressField[1]));
    #endif

    // check semaphore and post message
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // messag esent, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32DataIndProcessed", CSM_PARENT_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED);
    }
    else
    {
      // message writing failed
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid messag queues -> error
    s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE );
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, SignalWrite, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32SignalWrite(tU32 u32SignalId, tVoid* pvNewSignalData, tU8 u8DataBufferLength, tU8 u8TxType)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID ) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // check if signal data pointer is valid
    if (NULL != pvNewSignalData )
    {
      // check buffer len
      if (u8DataBufferLength <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH)
      {
        // create message for ProcMessage Queue
        trCSMProcMessageQueueItem  rMessage;
        tPU8 pu8DataPointer = (tPU8) pvNewSignalData;

        rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_WRITE;
        rMessage.uData.rSignalWrite.hProcessID     = _hProcessID;
        rMessage.uData.rSignalWrite.u32MsgQueueCnt = _u32MessageQueueCount;
        rMessage.uData.rSignalWrite.u32SignalId    = u32SignalId;
        rMessage.uData.rSignalWrite.u8Length       = u8DataBufferLength;
        rMessage.uData.rSignalWrite.u8TxType       = u8TxType;

        // copy signal data to message buffer
        for(tU8 u8Cnt = 0; u8Cnt < u8DataBufferLength; u8Cnt++ )
        {
          rMessage.uData.rSignalWrite.au8Data[u8Cnt] = *pu8DataPointer;
          pu8DataPointer++;
        }

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR4(("[0x%08X] s32SignalWrite - MQcount:0x%04X  Signal:0x%08X  Len:0x%X  Type:0x%02X",
                        _hProcessID, _u32MessageQueueCount, u32SignalId, u8DataBufferLength, u8TxType));
        #endif

        // check semaphore and post message to ProcCsm queue
        OSAL_s32SemaphoreWait( _hAccessSem, OSAL_C_TIMEOUT_FOREVER );

        // send message to process queue and wait for answer
        if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                                (tPCU8)&rMessage,
                                                (tU32)sizeof(rMessage),
                                                OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
        {
          // message written successfully
          // wait for result message
          s32RetVal = _s32WaitForGeneralResult((tCString)"s32SignalWrite", CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_WRITE);
        }
        else
        {
          // message write error
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
        }

        // free semaphore
        OSAL_s32SemaphorePost(_hAccessSem);
      } // buffer len ok
      else
      {
        // invalid buffer len -> error
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_DATA_LENGTH_EXCEEDED);
      }
    } // signal data pointer is valid
    else
    {
      // invalid signal data pointer -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  } // Message queues are valid
  else
  {
    // invalid message queues -> return error that the CSM is inactive
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, MultipleSignalWrite, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32MultipleSignalWrite(const tCSM_MultipleSignalWrite *prSignalList, const tU8 u8NumberOfSignals)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // check if signal data pointer is valid
    if (NULL != prSignalList)
    {
      // check if nr of signals does not exceed the limit (currently 20)
      if (u8NumberOfSignals <= CSM_C_ACCESS_LAYER_MULTI_SIGNAL_MAXNUMBER)
      {
        // create message for ProcMessage Queue
        trCSMProcMessageQueueItem  rMessage;
        const tCSM_MultipleSignalWrite *prSignalWrite;

        rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE;
        rMessage.uData.rMultipleSignalWrite.hProcessID        = _hProcessID;
        rMessage.uData.rMultipleSignalWrite.u32MsgQueueCnt    = _u32MessageQueueCount;
        rMessage.uData.rMultipleSignalWrite.u8NumberOfSignals = u8NumberOfSignals;

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalWrite - MQcount:0x%04X  NumOfSignals:0x%X",
                        _hProcessID, _u32MessageQueueCount, u8NumberOfSignals));
        #endif

        // loop over given signals
        for (tU8 u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++)
        {
          prSignalWrite = prSignalList + u8SignalCnt;

          if (prSignalWrite->bDataBufferLength <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH)
          {
            tPU8  pu8DataPointer = (tPU8)(prSignalWrite->pvNewSignalData);

            rMessage.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u32SignalId = prSignalWrite->dwSignalId;
            rMessage.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8Length    = prSignalWrite->bDataBufferLength;
            rMessage.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8TxType    = prSignalWrite->bTxType;

            for (tU8 u8DataCnt = 0; u8DataCnt < prSignalWrite->bDataBufferLength; u8DataCnt++)
            {
              rMessage.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].au8Data[u8DataCnt] = *pu8DataPointer;
              pu8DataPointer++;
            }

            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalWrite - SignalCount:0x%02X  Signal:0x%08X  Len:0x%X  Type:0x%02X",
                            _hProcessID, u8SignalCnt + 1, prSignalWrite->dwSignalId, prSignalWrite->bDataBufferLength, prSignalWrite->bTxType));
            #endif
          }
          else
          {
            s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_DATA_LENGTH_EXCEEDED);
            break;  // leave the for-loop
          }
        }

        if (CSM_C_NO_ERROR == s32RetVal)
        {
          // check semaphore and post message to ProcCsm queue
          OSAL_s32SemaphoreWait( _hAccessSem, OSAL_C_TIMEOUT_FOREVER );

          // send message to process queue and wait for answer
          if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                                  (tPCU8)&rMessage,
                                                  (tU32)sizeof(rMessage),
                                                  OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
          {
            // message written successfully
            // wait for result message
            s32RetVal = _s32WaitForGeneralResult((tCString)"s32MultipleSignalWrite", CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE);
          }
          else
          {
            // message write error
            s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
          }

          // free semaphore
          OSAL_s32SemaphorePost(_hAccessSem);
        }
      }
      else
      {
        // too many signals -> error
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_PARA);
      }
    }
    else
    {
      // invalid pointer -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  }
  else
  {
    // invalid message queues -> return error that the CSM is inactive
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, SignalRead, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32SignalRead(tU32 u32SignalId, tVoid* pvActSignalData, tU8 u8DataBufferLength, tPU32 pu32SignalStatus)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // check for valid parameters
    if ((NULL != pvActSignalData ) && (NULL != pu32SignalStatus))
    {
      trCSMProcMessageQueueItem  rOutMessage;

      rOutMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_READ;
      rOutMessage.uData.rSignalRead.hProcessID     = _hProcessID;
      rOutMessage.uData.rSignalRead.u32MsgQueueCnt = _u32MessageQueueCount;
      rOutMessage.uData.rSignalRead.u32SignalId    = u32SignalId;
      rOutMessage.uData.rSignalRead.u8Length       = u8DataBufferLength;

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("[0x%08X] s32SignalRead - MQcount:0x%04X  Signal:0x%08X  Len:0x%X",
                      _hProcessID, _u32MessageQueueCount, u32SignalId, u8DataBufferLength));
      #endif

      // check semaphore and post message to ProcCsm queue
      OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

      if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                              (tPCU8)&rOutMessage,
                                              (tU32)sizeof(rOutMessage),
                                              OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
      {
        trCSMParentMessageQueueItem  rInMessage;
        tS32  s32ReadBytes;

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR4(("[0x%08X] s32SignalRead - Wait at parent message queue 0x%X", _hProcessID, _hParentMessageQueueID));
        #endif

        // wait for response message
        s32ReadBytes = OSAL_s32MessageQueueWait(_hParentMessageQueueID,
                                                (tPU8)&rInMessage,
                                                sizeof(rInMessage),
                                                OSAL_NULL,  // equal priorities
                                                OSAL_C_U32_INFINITE);

        // check for valid message
        if (s32ReadBytes > 0)
        {
          if (CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_READ == rInMessage.enID)
          {
            // expected message type
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR4(("[0x%08X] s32SignalRead - Answer received - Result:0x%08X  SignalStatus:0x%04X",
                            _hProcessID,
                            rInMessage.uData.rSignalRead.s32Result,
                            rInMessage.uData.rSignalRead.u32SignalStatus));
            #endif

            s32RetVal = rInMessage.uData.rSignalRead.s32Result;

            if (CSM_C_NO_ERROR == s32RetVal)
            {
              tPU8  pu8DataPointer = (tPU8)pvActSignalData;
              *pu32SignalStatus = rInMessage.uData.rSignalRead.u32SignalStatus;
              for (tU8 u8Cnt = 0; u8Cnt < u8DataBufferLength; u8Cnt++)
              {
                *pu8DataPointer = rInMessage.uData.rSignalRead.au8Data[u8Cnt];
                pu8DataPointer++;
              }
            }
          }
          else
          {
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_ERR(("[0x%08X] s32SignalRead - Wrong message ID:0x%X", _hProcessID, rInMessage.enID));
            #endif
            s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_EVENT);
          }
        } // valid message
        else
        {
          // error reading the response message
          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_ERR(("[0x%08X] s32SignalRead - Read error", _hProcessID));
          #endif
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_RX_QUEUE_OVERRUN);
        }
      } // post message OK
      else
      {
        // write message error
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
      }

      // free semaphore
      OSAL_s32SemaphorePost(_hAccessSem);
    } // valid parameters
    else
    {
      // invalid parameters -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  } // valid message queues
  else
  {
    // no message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, MultipleSignalRead, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32MultipleSignalRead(const tCSM_MultipleSignalRead *prSignalList, const tU8 u8NumberOfSignals)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // check signal list pointer
    if (NULL != prSignalList )
    {
      if (u8NumberOfSignals <= CSM_C_ACCESS_LAYER_MULTI_SIGNAL_MAXNUMBER)
      {
        // create message for ProcMessage Queue
        trCSMProcMessageQueueItem  rOutMessage;
        const tCSM_MultipleSignalRead *prSignalRead;
        tU8 u8SignalCnt;

        rOutMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ;
        rOutMessage.uData.rMultipleSignalRead.hProcessID        = _hProcessID;
        rOutMessage.uData.rMultipleSignalRead.u32MsgQueueCnt    = _u32MessageQueueCount;
        rOutMessage.uData.rMultipleSignalRead.u8NumberOfSignals = u8NumberOfSignals;

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalRead - MQcount:0x%04X  NumOfSignals:0x%X",
                        _hProcessID, _u32MessageQueueCount, u8NumberOfSignals));
        #endif

        // loop over given signals
        for ( u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
        {
          prSignalRead = prSignalList + u8SignalCnt;

          rOutMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId = prSignalRead->dwSignalId;
          rOutMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length    = prSignalRead->bDataBufferLength;

          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalRead - SignalCount:0x%02X  Signal:0x%08X  Len:0x%X",
                          _hProcessID, u8SignalCnt + 1, prSignalRead->dwSignalId, prSignalRead->bDataBufferLength));
          #endif
        }

        // check semaphore and post message
        OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

        // send message to process queue and wait for answer
        if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                                (tPCU8)&rOutMessage,
                                                (tU32)sizeof(rOutMessage),
                                                OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
        {
          // message successfully written, so wait for response message
          trCSMParentMessageQueueItem  rInMessage;
          tS32  s32ReadBytes;

          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalRead - Wait at parent message queue 0x%X", _hProcessID, _hParentMessageQueueID));
          #endif

          // wait at queue
          s32ReadBytes = OSAL_s32MessageQueueWait(_hParentMessageQueueID,
                                                  (tPU8)&rInMessage,
                                                  sizeof(rInMessage),
                                                  OSAL_NULL,  // equal priorities
                                                  OSAL_C_U32_INFINITE);

          // check received response
          if (s32ReadBytes > 0)
          {
            if (CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ == rInMessage.enID)
            {
              // expected messag etype
              #if (CSM_S_TRACE == CSM_C_F_ON)
              ETG_TRACE_USR4(("[0x%08X] s32MultipleSignalRead - Answer received - Result:0x%08X  NumOfSignals:0x%X",
                               _hProcessID,
                               rInMessage.uData.rMultipleSignalRead.s32Result,
                               rInMessage.uData.rMultipleSignalRead.u8NumberOfSignals));
              #endif

              s32RetVal = rInMessage.uData.rMultipleSignalRead.s32Result;
              if (CSM_C_NO_ERROR == s32RetVal)
              {
                // chek nr of signals
                if (u8NumberOfSignals == rInMessage.uData.rMultipleSignalRead.u8NumberOfSignals)
                {
                  // loop over signals in message
                  for (u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
                  {
                    prSignalRead = prSignalList + u8SignalCnt;
                    if ((prSignalRead->dwSignalId == rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId)
                        &&
                        (prSignalRead->bDataBufferLength == rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length)
                       )
                    {
                      tPU8  pu8DataPointer = (tPU8)prSignalRead->pvActSignalData;
                      *(prSignalRead->pdwSignalStatus) = rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalStatus;
                      for (tU8 u8Cnt = 0; u8Cnt < prSignalRead->bDataBufferLength; u8Cnt++)
                      {
                        *pu8DataPointer = rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].au8Data[u8Cnt];
                        pu8DataPointer++;
                      }
                    }
                    else
                    {
                      #if (CSM_S_TRACE == CSM_C_F_ON)
                      ETG_TRACE_ERR(("[0x%08X] s32MultipleSignalRead - Wrong output for signal count:0x%02X  SentSignal:0x%08X  ReceivedSignal:0x%08X  SentLen:0x%X  ReceivedLen:0x%X",
                                     _hProcessID, u8SignalCnt,
                                     prSignalRead->dwSignalId,        rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId,
                                     prSignalRead->bDataBufferLength, rInMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length));
                      #endif
                      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_GENERAL_ERROR);
                      break;  // leave for-loop
                    }
                  }
                }
                else
                {
                  // error
                  #if (CSM_S_TRACE == CSM_C_F_ON)
                  ETG_TRACE_ERR(("[0x%08X] s32MultipleSignalRead - Wrong number of signals - Sent:0x%X  Received:0x%X",
                                  _hProcessID, u8NumberOfSignals, rInMessage.uData.rMultipleSignalRead.u8NumberOfSignals));
                  #endif
                  s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_GENERAL_ERROR);
                }
              }
            }
            else
            {
              // not the expected message ID
              #if (CSM_S_TRACE == CSM_C_F_ON)
              ETG_TRACE_ERR(("[0x%08X] s32MultipleSignalRead - Wrong message ID:0x%X", _hProcessID, rInMessage.enID));
              #endif
              s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_EVENT);
            }
          }
          else
          {
            // read error
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_ERR(("[0x%08X] s32MultipleSignalRead - Read error", _hProcessID));
            #endif
            s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_RX_QUEUE_OVERRUN);
          }
        }
        else
        {
          // error when writing message to queue
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
        }

        // free semaphore
        OSAL_s32SemaphorePost(_hAccessSem);
      }
      else
      {
        // nr of signals not correct
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_PARA);
      }
    }
    else
    {
      // invalid parameters -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  }
  else
  {
    // invalid message queues -> return error that the CSM is inactive
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, GetAllStateIndicationsAgain, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32GetAllStateIndicationsAgain() const
{
  tS32 s32RetVal;

  // this is for CBR only
  #if ((CSM_S_CBR_AVAILABLE == CSM_C_F_ON) && (CSM_S_CCS_USED == CSM_C_F_ON))
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN;
    rMessage.uData.rGetAllStateIndAgain.hProcessID     = _hProcessID;
    rMessage.uData.rGetAllStateIndAgain.u32MsgQueueCnt = _u32MessageQueueCount;

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32GetAllStateIndicationsAgain - MQcount:0x%04X",
                    _hProcessID, _u32MessageQueueCount));
    #endif

    // check semaphore and post message to ProcMessage queue
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message esuccessfully sent, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32GetAllStateIndicationsAgain", CSM_PARENT_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN);
    }
    else
    {
      // message post failed -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid message queues -> return error that the CSM is inactive
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  #else // no CBR and no CCS -> error
  s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_SRVC_UNAVAILABLE);
  #endif
  return s32RetVal;
}

//****************************************************************************************************************
// service function, ConnectionTest, synchron
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32ConTest(tU32 u32ProtocolType, const tVoid* pcvAddressField, tU8 u8Action)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    // create message
    trCSMProcMessageQueueItem  rMessage;
    const tU8* pu8AddressField = (const tU8*) pcvAddressField;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_CON_TEST;
    rMessage.uData.rConTest.hProcessID      = _hProcessID;
    rMessage.uData.rConTest.u32MsgQueueCnt  = _u32MessageQueueCount;
    rMessage.uData.rConTest.u32ProtocolType = u32ProtocolType;
    rMessage.uData.rConTest.u8Action        = u8Action;

    // loop over addressfiled bytes
    for ( tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++)
    {
      if (pu8AddressField != NULL )
      {
        // get addressfield byte and drop it to in message
        rMessage.uData.rConTest.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
      }
      else
      {
        // fill message addressfield with 0
        rMessage.uData.rConTest.au8AddressField[u32Cnt] = 0;
      }
    }

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] s32ConTest - MQcount:0x%04X  Prot:0x%08X  AddrField:0x%02X%02X  Action:0x%X",
                    _hProcessID,
                    _u32MessageQueueCount,
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                    rMessage.uData.rConTest.au8AddressField[0],
                    rMessage.uData.rConTest.au8AddressField[1],
                    u8Action));
    #endif

    // check semaphore and post message
    OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

    // send message to process queue and wait for answer
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      // message witten successfully, wait for response
      s32RetVal = _s32WaitForGeneralResult((tCString)"s32ConTest", CSM_PARENT_MESSAGE_QUEUE_ID_CON_TEST);
    }
    else
    {
      // error when writing message
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
    }

    // free semaphore
    OSAL_s32SemaphorePost(_hAccessSem);
  }
  else
  {
    // invalid message queues -> return error that the CSM is inactive
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//****************************************************************************************************************
// service function, DataRead, synchron (used by VW BAP protocol)
//****************************************************************************************************************
tS32 csm_tclCsmAccessUser::s32DataRead(tU32 u32ProtocolType, const tVoid* pcvAddressField, tPU8 pu8Data, tU16 u16DataLength)
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // check if required message queues are valid
  if ((OSAL_C_INVALID_HANDLE != _hProcMessageQueueID) && (OSAL_C_INVALID_HANDLE != _hParentMessageQueueID))
  {
    if (NULL != pu8Data)
    {
      // create message
      trCSMProcMessageQueueItem  rMessage;
      const tU8* pu8AddressField = (const tU8*) pcvAddressField;

      rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_DATA_READ;
      rMessage.uData.rDataRead.hProcessID      = _hProcessID;
      rMessage.uData.rDataRead.u32MsgQueueCnt  = _u32MessageQueueCount;
      rMessage.uData.rDataRead.u32ProtocolType = u32ProtocolType;

      // loop over addressfield bytes
      for (tU32 u32Cnt = 0; u32Cnt < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; u32Cnt++)
      {
        if (pu8AddressField != NULL)
        {
          rMessage.uData.rDataRead.au8AddressField[u32Cnt] = *(pu8AddressField + u32Cnt);
        }
        else
        {
          // fill message addressfield with 0
          rMessage.uData.rDataRead.au8AddressField[u32Cnt] = 0;
        }
      }

      // add data len to message
      rMessage.uData.rDataRead.u16DataLength = u16DataLength;

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("[0x%08X] s32DataRead - MQcount:0x%04X  Protocol:0x%08X  AddrField:0x%02X%02X  Len:0x%X",
                      _hProcessID,
                      _u32MessageQueueCount,
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) u32ProtocolType),
                      rMessage.uData.rDataRead.au8AddressField[0],
                      rMessage.uData.rDataRead.au8AddressField[1],
                      u16DataLength));
      #endif

      // check semaphore and post message
      OSAL_s32SemaphoreWait(_hAccessSem, OSAL_C_TIMEOUT_FOREVER);

      // send message to process queue and wait for answer
      if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                              (tPCU8)&rMessage,
                                              (tU32)sizeof(rMessage),
                                              OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
      {
        // message written successfully
        trCSMParentMessageQueueItem  rInMessage;
        tS32  s32ReadBytes;

        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR2(("[0x%08X] s32DataRead - Wait at parent message queue 0x%X", _hProcessID, _hParentMessageQueueID));
        #endif

        // wait for response
        s32ReadBytes = OSAL_s32MessageQueueWait(_hParentMessageQueueID,
                                                (tPU8)&rInMessage,
                                                sizeof(rInMessage),
                                                OSAL_NULL,  // equal priorities
                                                OSAL_C_U32_INFINITE);
        // response received
        if (s32ReadBytes > 0)
        {
          if (CSM_PARENT_MESSAGE_QUEUE_ID_DATA_READ == rInMessage.enID)
          {
            // expected message received
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("[0x%08X] s32DataRead - Answer received - Result:0x%08X",
                            _hProcessID, rInMessage.uData.rDataRead.s32Result));
            #endif

            // get CSM result
            s32RetVal = rInMessage.uData.rDataRead.s32Result;

            if (CSM_C_NO_ERROR == s32RetVal)
            {
              // read back the requested bytes
              for (tU16 u16Cnt = 0; u16Cnt < u16DataLength; u16Cnt++)
              {
                *pu8Data = rInMessage.uData.rDataRead.au8Data[u16Cnt];
                pu8Data++;
              }
            }
          }
          else
          {
            // unexpected message
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_ERR(("[0x%08X] s32DataRead - Wrong message ID:0x%X", _hProcessID, rInMessage.enID));
            #endif
            s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_EVENT);
          }
        }
        else
        {
          // error reading message
          #if (CSM_S_TRACE == CSM_C_F_ON)
          ETG_TRACE_ERR(("[0x%08X] s32DataRead - Read error", _hProcessID));
          #endif
          s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_RX_QUEUE_OVERRUN);
        }
      }
      else
      {
        // error writing message
        s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_TX_QUEUE_OVERRUN);
      }

      // free semaphore
      OSAL_s32SemaphorePost(_hAccessSem);
    }
    else
    {
      // invalid parameters -> error
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_NULL_POINTER);
    }
  }
  else
  {
    // no message queues -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_STACK_INACTIVE);
  }
  return s32RetVal;
}

//**************************
// private class functions
//**************************

/*******************************************************************************
 * function    _hCreateAccessSemaphore
 * \doxydocu
 * \brief      creates an OSAL semaphore used for access control
 *
 * \param[in]  None
 *
 * \return     on success: semaphore handle
 *             on error:   OSAL_C_INVALID_HANDLE
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
OSAL_tSemHandle csm_tclCsmAccessUser::_hCreateAccessSemaphore()
{
  OSAL_tSemHandle  hSemID         = OSAL_C_INVALID_HANDLE;
  CHAR             acNamePrefix[] = CSMAL_C_ACCESS_SEM_NAME_PREFIX;
  CHAR             acSemaphoreName[OSAL_C_U32_MAX_NAMELENGTH];

  // create semaphore name
  sprintf(acSemaphoreName, "%s%08x_%04d", acNamePrefix, _hProcessID, _u8AccessSemCount);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("[0x%08X] Creation of semaphore - %s", _hProcessID, acSemaphoreName));
  #endif

  if (OSAL_ERROR == OSAL_s32SemaphoreCreate(acSemaphoreName, &hSemID, 1))
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  return hSemID;
}

/*******************************************************************************
 * function    _vDeleteAccessSemaphore
 * \doxydocu
 * \brief      deletes the OSAL semaphore used for access control
 *
 * \param[in]  None
 *
 * \return     Void
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
tVoid csm_tclCsmAccessUser::_vDeleteAccessSemaphore()
{
  CHAR  acNamePrefix[] = CSMAL_C_ACCESS_SEM_NAME_PREFIX;
  CHAR  acSemaphoreName[OSAL_C_U32_MAX_NAMELENGTH];

  // create semaphore name
  sprintf(acSemaphoreName, "%s%08x_%04d", acNamePrefix, _hProcessID, _u8AccessSemCount);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("[0x%08X] Deleting semaphore - %s", _hProcessID, acSemaphoreName));
  #endif

  if (OSAL_C_INVALID_HANDLE != _hAccessSem)
  {
    OSAL_s32SemaphoreClose(_hAccessSem);
    OSAL_s32SemaphoreDelete(acSemaphoreName);
    _hAccessSem = OSAL_C_INVALID_HANDLE;
  }
}

/*******************************************************************************
 * function    _hCreateParentMessageQueue
 * \doxydocu
 * \brief      creates an OSAL message queue
 *             used by the Pre-Init whenever an Application creates a communication object
 *             Queue is used for simulating (implementing) synchronous service calls -> thread is waiting for result of CSM
 *
 * \param[in]  None
 *
 * \return     on success: message queue handle
 *             on error:   OSAL_C_INVALID_HANDLE
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
OSAL_tMQueueHandle csm_tclCsmAccessUser::_hCreateParentMessageQueue()
{
  OSAL_tMQueueHandle hMessageQueueID = OSAL_C_INVALID_HANDLE;
  CHAR  acNamePrefix[] = CSMAL_C_PARENT_MESSAGE_QUEUE_NAME_PREFIX;
  CHAR  acMessageQueueName[OSAL_C_U32_MAX_NAMELENGTH];

  // create the queue name
  sprintf(acMessageQueueName, "%s%08x_%04u", acNamePrefix, _hProcessID, (unsigned int) _u32MessageQueueCount);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("[0x%08X] Name of created parent message queue - %s", _hProcessID, acMessageQueueName));
  #endif

  // Create osal message queue (the element size is mainly determined by the size of a union used in the MsgQItem)
  // the queue will only contain 1 single element, since this queue is for simulating a synchron function call !!
  if (OSAL_ERROR == OSAL_s32MessageQueueCreate(acMessageQueueName,
                                               1,
                                               sizeof(trCSMParentMessageQueueItem),
                                               OSAL_EN_READWRITE,
                                               &hMessageQueueID))
  {
    // in case of error get the error code
    tU32 u32ErrorCode = OSAL_u32ErrorCode();
    ETG_TRACE_FATAL(("[0x%08X] CreateParentMessageQueue - Could not create parent message queue - ErrorCode: %X", _hProcessID, u32ErrorCode));
    NORMAL_M_ASSERT_ALWAYS();
  }

  return hMessageQueueID;
}

/*******************************************************************************
 * function    _hCreateServiceMessageQueue
 * \doxydocu
 * \brief      creates an OSAL message queue
 *             used by the Pre-Init whenever an Application creates a communication object
 *             Queue is used for sending callback data from CSM process to user process
 *
 * \param[in]  None
 *
 * \return     on success: message queue handle
 *             on error:   OSAL_C_INVALID_HANDLE
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
OSAL_tMQueueHandle csm_tclCsmAccessUser::_hCreateServiceMessageQueue()
{
  OSAL_tMQueueHandle hMessageQueueID = OSAL_C_INVALID_HANDLE;
  CHAR  acMessageQueueName[OSAL_C_U32_MAX_NAMELENGTH];
  CHAR  acNamePrefix[] = CSMAL_C_SERVICE_MESSAGE_QUEUE_NAME_PREFIX;

  // create the queue name
  sprintf(acMessageQueueName, "%s%08x_%04u", acNamePrefix, _hProcessID, (unsigned int) _u32MessageQueueCount );

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("[0x%08X] Name of created service message queue - %s", _hProcessID, acMessageQueueName));
  #endif

  // Create osal message queue (the element size is mainly determined by the size of a union used in the MsgQItem)
  // the queue will contain a previously given max. number of elements.
  if (OSAL_ERROR == OSAL_s32MessageQueueCreate(acMessageQueueName,
                                               _u32MaxNrOfServiceQueueElements,
                                               _u32ServiceQueueElementSize,
                                               OSAL_EN_READWRITE,
                                               &hMessageQueueID))
  {
    // in case of error get the error code
    tU32 u32ErrorCode = OSAL_u32ErrorCode();
    ETG_TRACE_FATAL(("[0x%08X] CreateServiceMessageQueue - Could not create service message queue - ErrorCode: %X", _hProcessID, u32ErrorCode));
    NORMAL_M_ASSERT_ALWAYS();
  }

  return hMessageQueueID;
}

/*******************************************************************************
 * function    _vCsmUserModeServiceTask
 * \doxydocu
 * \brief      worker function for the user mode service task.
 *             waits at the service message queue to handle asynchronous messages
 *
 * \param[in]  pointer to the thread argument structure
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
tVoid csm_tclCsmAccessUser::_vCsmUserModeServiceTask(const tVoid * pcvArg)
{
  // upon task creation, we have passed the THIS pointer which is now passed here as argument
  // we can now cast it to get access to all our instance (THIS) data
  const csm_tclCsmAccessUser * poMyself = (const csm_tclCsmAccessUser *) pcvArg;

  if (poMyself)
  {
    // instance seems to be valid
    BOOL fTerminateTask = FALSE;

    #if (CSMAL_S_DATA_IND_STRUCT_WITH_DATA_ARRAY == CSM_C_F_ON)
    BYTE msg[sizeof(trDataIndMsg)];
    #else
    BYTE msg[sizeof(trDataIndMsg) + CSMAL_MAX_DATA_PAYLOAD];
    #endif

    void *     pvMsg = msg;
    pMsgHeader pHeader;
    DWORD      dwQueuePriority = OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST;

    // this thread function runs endlessly until termination message comes
    while (!fTerminateTask)
    {
      // wait at the service message queue referenced by an instance related queue ID !!!
      if (0 == OSAL_s32MessageQueueWait(poMyself->_hServiceMessageQueueID,
                                        (tPU8)pvMsg,
                                        sizeof(msg),
                                        (tPU32)&dwQueuePriority,
                                        OSAL_C_TIMEOUT_FOREVER)
                                       )
      {
        break;   // empty message -> error, exit task
      }

      // probably valid message received -> check message type
      pHeader = (pMsgHeader) pvMsg;

      switch(pHeader->enMsgType)
      {
        // Thread TERMINATION
        case CSM_ACCESS_TerminateTask:
        {
          // quit loop
          fTerminateTask = TRUE;
        }
        break;
        // Signal Indication
        case CSM_ACCESS_SignalInd:
        {
          poMyself->_vHandleSignalIndication(pvMsg);
        }
        break;
        // Communication Confirmation
        case CSM_ACCESS_CommunicationCon:
        {
          poMyself->_vHandleCommunicationConfirmation(pvMsg);
        }
        break;
        // Communication Indication
        case CSM_ACCESS_CommunicationInd:
        {
          poMyself->_vHandleCommunicationIndication(pvMsg);
        }
        break;

        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
          // Data Confirmation
          case CSM_ACCESS_DataCon:
          {
            poMyself->_vHandleDataConfirmation(pvMsg);
          }
          break;
          // Data Indication
          case CSM_ACCESS_DataInd:
          {
            poMyself->_vHandleDataIndication(pvMsg);
          }
          break;

          #if (CSM_S_ISO_TP == CSM_C_F_ON)
          #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
          // First Frame Indication
          case CSM_ACCESS_DataIndFF:
          {
            poMyself->_vHandleDataIndicationFF(pvMsg);
          }
          break;
          #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
          #endif // CSM_S_ISO_TP

          #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
          // Data Error Indication
          case CSM_ACCESS_DataErrorInd:
          {
            poMyself->_vHandleDataErrorIndication(pvMsg);
          }
          break;
          #endif
        #endif // CSM_S_CNP_AVAILABLE

        #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
          // Bus Error Indication
          case CSM_ACCESS_BusErrorInd:
          {
            poMyself->_vHandleBusErrorIndication(pvMsg);
          }
          break;
        #endif

        // NoEvent or unknown event
        case CSM_ACCESS_NoEvent:
        default:
        {
          // do nothing
        }
        break;
      } // switch MsgType
    } // while loop
  } // valid argument pointer received

  // end task if invalid argument pointer has been passed to this function
  OSAL_vThreadExit();
}

/*******************************************************************************
 * function    _hCreateServiceTask
 * \doxydocu
 * \brief      Reads priority value from registry and
 *             creates a user mode service task, which reads callback data from
 *             a message queue and calls the callbacks
 *
 * \param[in]  None
 *
 * \return     ID of the created service task
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
OSAL_tThreadID csm_tclCsmAccessUser::_hCreateServiceTask()
{
  OSAL_trThreadAttribute  rAttr;
  OSAL_tThreadID          hThreadId;
  tS32                    s32Prio   = CSM_ACCESS_SERVICETASK_PRIO;          // legacy stuff, not used in LINUX
  tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "";

  // increase our global task counter
  _u8ServiceTaskCounter++;

  // create the thread name
  sprintf(szThreadName, "CSMALU_TH_%08x_%03d", _hProcessID, _u8ServiceTaskCounter);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("[0x%08X] Creation of Service Thread - %s", _hProcessID, szThreadName));
  #endif

  // fill attribute struct with task/thread data
  rAttr.szName       = szThreadName;
  rAttr.s32StackSize = 4096;
  rAttr.u32Priority  = (tU32) s32Prio;
  rAttr.pfEntry      = (OSAL_tpfThreadEntry)_vCsmUserModeServiceTask; // the thread handler function itself !!
  rAttr.pvArg        = (tPVoid)this;                                  // pvArg can be freely used to pass any data
                                                                      // the one and only call of the worker function
                                                                      // will get pvArg as argument again.
  // spawn the thread
  hThreadId = OSAL_ThreadSpawn(&rAttr);

  // check if successfully created
  if (hThreadId == OSAL_ERROR)
  {
    ETG_TRACE_FATAL(("[0x%08X] CreateServiceTask - Could not create service thread - ThreadId: %X", _hProcessID, hThreadId));
    CSMALU_vAssert(FALSE);  /*lint !e506 */ /* Not started task shall cause an assert - krv2hi 2010-09-24 */
  }

  return hThreadId;
}

/*******************************************************************************
 * function    _bInitCSMProc
 * \doxydocu
 * \brief      Init CSM process by sending message with own thread ID
 *             and message queue handle
 *
 * \param[in]  None
 *
 * \return     tBool
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
tBool csm_tclCsmAccessUser::_bInitCSMProc()
{
  tBool  bRetVal = FALSE;

  // open the message queue provided by ProcCsm
  if (OSAL_OK == OSAL_s32MessageQueueOpen(CSM_C_PROC_MESSAGE_QUEUE_NAME, OSAL_EN_READWRITE, &_hProcMessageQueueID))
  {
    // create message queue item of type INIT and pass our process ID and our message queue number (count)
    trCSMProcMessageQueueItem  rMessage;

    rMessage.enID = CSM_PROC_MESSAGE_QUEUE_ID_INIT;
    rMessage.uData.rInit.hProcessID             = _hProcessID;                 // process ID of the user Application using this instance
    rMessage.uData.rInit.u32MsgQueueCnt         = _u32MessageQueueCount;       // message queue number of this instance
    rMessage.uData.rInit.u32MsgQueueElementSize = _u32ServiceQueueElementSize; // message queue element size of this instance

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] _bInitCSMProc - MQcount:%d  MQelementSize:%d",
                    _hProcessID,
                    _u32MessageQueueCount,
                    _u32ServiceQueueElementSize
                   ));
    #endif

    // now send the message to ProcCsm
    if (OSAL_OK == OSAL_s32MessageQueuePost(_hProcMessageQueueID,
                                            (tPCU8)&rMessage,
                                            (tU32)sizeof(rMessage),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
    {
      bRetVal = TRUE;
    }
  }
  return bRetVal;
}

/*******************************************************************************
 * function    _vFillCallbackFunctionsForApplCallbackInit
 * \doxydocu
 * \brief      Fill all necessary call back function pointers to message queue item
 *
 * \param[in]  - pointer to message queue item to be filled
 *             - protocol type
 *             - pointer to protocol type specific callback function pointer array
 *
 * \return     tS32
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
tS32 csm_tclCsmAccessUser::_s32FillCallbackFunctionsForApplCallbackInit( trCSMProcMessageQueueItemRegisterAppl *prRegisterAppl,
                                                                         tU32 u32ProtocolType,
                                                                         const tVoid* pcvCallBackFkt) const
{
  tS32 s32RetVal = CSM_C_NO_ERROR;

  // pre-assign all callbacks of given struct with NULL
  prRegisterAppl->pcvCallBackCommunicationCon = NULL;
  prRegisterAppl->pcvCallBackCommunicationInd = NULL;
  prRegisterAppl->pcvCallBackDataCon          = NULL;
  prRegisterAppl->pcvCallBackDataInd          = NULL;
  prRegisterAppl->pcvCallBackDDataCon         = NULL;
  prRegisterAppl->pcvCallBackDDataInd         = NULL;
  prRegisterAppl->pcvCallBackDataIndFF        = NULL;
  prRegisterAppl->pcvCallBackDataErrorInd     = NULL;

  // now evaluate protocol type
  switch(u32ProtocolType)
  {
    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      // cast given callback struct to real target type
      const tCSM_LOCAL_CTRL_APPL_CALLBACK * pCallBackFkt = (const tCSM_LOCAL_CTRL_APPL_CALLBACK *) pcvCallBackFkt;
      // fill given struct with real target data
      prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)(pCallBackFkt->pfvCommunicationCon);   /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)(pCallBackFkt->pfvCommunicationInd);   /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->u16ApplId = pCallBackFkt->u16ApplID;
    }
    break;
    #endif // LCTRL

    #ifdef CSM_C_PTYPE_GM_NWM
    case CSM_C_PTYPE_GM_NWM:
    {
      tCSM_GMLAN_NWM_APPL_CALLBACK* pCallBackFkt = (tCSM_GMLAN_NWM_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;
    }
    break;
    #endif // GM LAN

    #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
    case CSM_C_PTYPE_USED_STD_USDT_ISO:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        const tCSM_ISO_TP_USDT_APPL_CALLBACK * pCallBackFkt = (const tCSM_ISO_TP_USDT_APPL_CALLBACK *)pcvCallBackFkt;
        prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;         /*lint !e611 suspicious yes, but works*/
        prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;         /*lint !e611 suspicious yes, but works*/
        prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvDataCon;                  /*lint !e611 suspicious yes, but works*/
        prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvDataInd;                  /*lint !e611 suspicious yes, but works*/
        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataIndFF        = (tPCVoid)pCallBackFkt->pfvDataIndFF;
        #endif
        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataErrorInd     = (tPCVoid)pCallBackFkt->pfvDataErrorInd;
        #endif
        #endif // ISO TP
      #endif // CNP
    }
    break;
    #endif // USDT ISO

    #ifdef CSM_C_PTYPE_USED_STD_LIN_DIAG
    case CSM_C_PTYPE_USED_STD_LIN_DIAG:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        tCSM_ISO_TP_USDT_APPL_CALLBACK* pCallBackFkt = (tCSM_ISO_TP_USDT_APPL_CALLBACK*)pcvCallBackFkt;
        prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;
        prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;
        prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvDataCon;
        prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvDataInd;
        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataIndFF        = (tPCVoid)pCallBackFkt->pfvDataIndFF;
        #endif
        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataErrorInd     = (tPCVoid)pCallBackFkt->pfvDataErrorInd;
        #endif
        #endif // ISO TP
     #endif // CNP
    }
    break;
    #endif // LIN Diag

    #ifdef CSM_C_PTYPE_USED_STD_LIN_USDT
    case CSM_C_PTYPE_USED_STD_LIN_USDT:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        tCSM_ISO_TP_USDT_APPL_CALLBACK* pCallBackFkt = (tCSM_ISO_TP_USDT_APPL_CALLBACK*)pcvCallBackFkt;
        prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;
        prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;
        prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvDataCon;
        prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvDataInd;
        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataIndFF        = (tPCVoid)pCallBackFkt->pfvDataIndFF;
        #endif
        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        prRegisterAppl->pcvCallBackDataErrorInd     = (tPCVoid)pCallBackFkt->pfvDataErrorInd;
        #endif
        #endif // ISO TP
      #endif // CNP
    }
    break;
    #endif // LIN USDT

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
    case CSM_C_PTYPE_USED_STD_UUDT_TX:
    {
      #if (CSM_S_UUDT_TX == CSM_C_F_ON)
      tCSM_UUDT_TX_APPL_CALLBACK* pCallBackFkt = (tCSM_UUDT_TX_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackDataCon = (tPCVoid)pCallBackFkt->pfvDataCon;
      #endif
    }
    break;
    #endif // UUDT_TX

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
    case CSM_C_PTYPE_USED_STD_UUDT_RX:
    {
      #if (CSM_S_UUDT_RX == CSM_C_F_ON)
      tCSM_UUDT_RX_APPL_CALLBACK* pCallBackFkt = (tCSM_UUDT_RX_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackDataInd = (tPCVoid)pCallBackFkt->pfvDataInd;
      #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      prRegisterAppl->pcvCallBackDataErrorInd  = (tPCVoid)pCallBackFkt->pfvDataErrorInd;
      #endif
      #endif
    }
    break;
    #endif // UUDT_RX

    #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
    case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
    {
      const tCSM_BR_SIGNAL_APPL_CALLBACK * pCallBackFkt = (const tCSM_BR_SIGNAL_APPL_CALLBACK *) pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon; /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd; /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->u16ApplId = pCallBackFkt->u16ApplID;
    }
    break;
    #endif //BR_SIGNAL

    #ifdef CSM_C_PTYPE_MCNET_ASDT_STD
    case CSM_C_PTYPE_MCNET_ASDT_STD:
    {
      // not supported yet
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    #endif

    #ifdef CSM_C_PTYPE_MCNET_USDT
    case CSM_C_PTYPE_MCNET_USDT:
    {
      // not supported yet
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    #endif

    #ifdef CSM_C_PTYPE_VW_BAP
    case CSM_C_PTYPE_VW_BAP:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      tCSM_VW_BAP_APPL_CALLBACK* pCallBackFkt = (tCSM_VW_BAP_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;
      prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvDataCon;
      prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvDataInd;
      #endif
    }
    break;
    #endif // VW_BAP

    #ifdef CSM_C_PTYPE_PAG_OSEK_NWM
    case CSM_C_PTYPE_PAG_OSEK_NWM:
    {
      tCSM_OSEK_NWM_APPL_CALLBACK* pCallBackFkt = (tCSM_OSEK_NWM_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;
    }
    break;
    #endif // PAG_OSEK_NWM

    #ifdef CSM_C_PTYPE_RN_MPDT_C
    case CSM_C_PTYPE_RN_MPDT_C:
    case CSM_C_PTYPE_RN_MPDT_D:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      const tCSM_MPDT_APPL_CALLBACK * pCallBackFkt = (const tCSM_MPDT_APPL_CALLBACK *) pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;  /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;  /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvCDataCon;          /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvCDataInd;          /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackDDataCon         = (tPCVoid)pCallBackFkt->pfvDDataCon;          /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->pcvCallBackDDataInd         = (tPCVoid)pCallBackFkt->pfvDDataInd;          /*lint !e611 suspicious yes, but works*/
      #endif
    }
    break;
    #endif // RN_MPDT_C and D

  #ifdef CSM_C_PTYPE_J1939
    case CSM_C_PTYPE_J1939:
    {
      tCSM_J1939_APPL_CALLBACK* pCallBackFkt = (tCSM_J1939_APPL_CALLBACK*)pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationCon = (tPCVoid)pCallBackFkt->pfvCommunicationCon;
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;
      prRegisterAppl->pcvCallBackDataCon          = (tPCVoid)pCallBackFkt->pfvDataCon;
      prRegisterAppl->pcvCallBackDataInd          = (tPCVoid)pCallBackFkt->pfvDataInd;
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      prRegisterAppl->pcvCallBackDataIndFF        = (tPCVoid)pCallBackFkt->pfvDataIndFF;
    #endif
    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
      prRegisterAppl->pcvCallBackDataErrorInd     = (tPCVoid)pCallBackFkt->pfvDataErrorInd;
    #endif

    }
    break;
  #endif /* CSM_C_PTYPE_J1939 */

    #ifdef CSM_C_PTYPE_RN_OSEKI_NWM
    case CSM_C_PTYPE_RN_OSEKI_NWM:
    {
      const tCSM_OSEKI_NWM_APPL_CALLBACK * pCallBackFkt = (const tCSM_OSEKI_NWM_APPL_CALLBACK *) pcvCallBackFkt;
      prRegisterAppl->pcvCallBackCommunicationInd = (tPCVoid)pCallBackFkt->pfvCommunicationInd;  /*lint !e611 suspicious yes, but works*/
      prRegisterAppl->u16ApplId = pCallBackFkt->u16ApplID;
    }
    break;
    #endif // RN_OSEKI_NWM

    default:
    {
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    break;
  } // switch protocol type

  return s32RetVal;
}

/*******************************************************************************
 * function    _s32WaitForGeneralResult
 * \doxydocu
 * \brief      Wait at the parent message queue for a general result message
 *
 * \param[in]  DebugString and the type of message to be waited for
 *
 * \return     tS32
 *
 * \access     ?
 * \reentrant  ?
 *
 ******************************************************************************/
tS32 csm_tclCsmAccessUser::_s32WaitForGeneralResult(tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID)
{
  tS32  s32RetVal;
  trCSMParentMessageQueueItem  rInMessage;
  tS32  s32ReadBytes;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("[0x%08X] Wait at parent message queue - 0x%X for %s", _hProcessID, _hParentMessageQueueID, coszDebugString));
  #endif

  // wait at the Parent-Message Queue for response from ProcCsm
  s32ReadBytes = OSAL_s32MessageQueueWait(_hParentMessageQueueID,
                                          (tPU8)&rInMessage,
                                          sizeof(rInMessage),
                                          OSAL_NULL,             // equal priorities
                                          OSAL_C_U32_INFINITE);

  // message received
  if (s32ReadBytes > 0)
  {
    if (enItemID == rInMessage.enID)
    {
      // expected response message type
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("[0x%08X] General Result - 0x%08X received for %s", _hProcessID, rInMessage.uData.rGeneralResult.s32Result, coszDebugString));
      #endif
      // take over the result
      s32RetVal = rInMessage.uData.rGeneralResult.s32Result;
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("[0x%08X] Wrong message ID - 0x%X for %s", _hProcessID, rInMessage.enID, coszDebugString));
      #endif
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_INVALID_EVENT);
    }
  } // valid message
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("[0x%08X] ERROR - %s - Read error for response message", _hProcessID, coszDebugString));
    #endif
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_USR, CSM_E_RX_QUEUE_OVERRUN);
  }
  return s32RetVal;
}

/*******************************************************************************
 * function    _vHandleSignalIndication
 * \doxydocu
 * \brief      Handles occurrence of signal indication
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
tVoid csm_tclCsmAccessUser::_vHandleSignalIndication(tVoid * pvMsg) const
{
  pSignalIndMsg pMsg = (pSignalIndMsg) pvMsg;

  if ( NULL != pMsg->vPFNCBRSignalInd )
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] _vHandleSignalIndication - UsrHandle:0x%08X  Callback:0x%08X  Signal:0x%08X  State:0x%08X",
                    _hProcessID,
                    pMsg->rMsgHeader.pvAccessUsrHandle,
                    pMsg->vPFNCBRSignalInd,
                    pMsg->dwSignalId,
                    pMsg->dwState));
    #endif
    pMsg->vPFNCBRSignalInd(pMsg->rMsgHeader.pvAccessUsrHandle, pMsg->dwSignalId, pMsg->dwState);
  }
}

/*******************************************************************************
 * function    _vHandleCommunicationIndication
 * \doxydocu
 * \brief      Handles occurrance of communication indications
 *            (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
tVoid csm_tclCsmAccessUser::_vHandleCommunicationIndication(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pCommunicationIndMsg pMsg = (pCommunicationIndMsg) pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPCommunicationInd != NULL)
  {
    // call callback function
    pMsg->vPFNTPCommunicationInd(pMsg->rMsgHeader.pvAccessUsrHandle,
                                 pMsg->bBus,
                                 pMsg->dwProtocolType,
                                 pMsg->abAddressField,
                                 pMsg->bConnectState,
                                 pMsg->wAppId);
  }
}

/*******************************************************************************
 * function    _vHandleCommunicationConfirmation
 * \doxydocu
 * \brief      Handles occurrance of communication confirmations
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
tVoid csm_tclCsmAccessUser::_vHandleCommunicationConfirmation(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pCommunicationConMsg pMsg = (pCommunicationConMsg)pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPCommunicationCon != NULL)
  {
    // call callback function
    pMsg->vPFNTPCommunicationCon(pMsg->rMsgHeader.pvAccessUsrHandle,
                                 pMsg->bBus,
                                 pMsg->dwProtocolType,
                                 pMsg->abAddressField,
                                 pMsg->bConnectState,
                                 pMsg->wAppId);
  }
}

/*******************************************************************************
 * function    _vHandleDataIndication
 * \doxydocu
 * \brief      Handles occurrance of data indications
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
tVoid csm_tclCsmAccessUser::_vHandleDataIndication(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pDataIndMsg pMsg = (pDataIndMsg) pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPDataInd != NULL)
  {
    // call callback function
    pMsg->vPFNTPDataInd(pMsg->rMsgHeader.pvAccessUsrHandle,
                        pMsg->dwProtocolType,
                        pMsg->abAddressField,
                        #if (CSMAL_S_DATA_IND_STRUCT_WITH_DATA_ARRAY == CSM_C_F_ON)
                          pMsg->abData,
                        #else
                          ((BYTE *) pvMsg)+sizeof(trDataIndMsg),
                        #endif
                        pMsg->wDataLength);
  }
}
#endif

/*******************************************************************************
 * function    _vHandleDataConfirmation
 * \doxydocu
 * \brief      Handles occurrance of data confirmations
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
tVoid csm_tclCsmAccessUser::_vHandleDataConfirmation(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pDataConMsg pMsg = (pDataConMsg) pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPDataCon != NULL)
  {
    // call callback function
    pMsg->vPFNTPDataCon(pMsg->rMsgHeader.pvAccessUsrHandle,
                        pMsg->dwProtocolType,
                        pMsg->abAddressField,
                        pMsg->bTransferResult);
  }
}
#endif

/*******************************************************************************
 * function    _vHandleDataIndicationFF
 * \doxydocu
 * \brief      Handles occurrance of first frame data indications
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_ISO_TP == CSM_C_F_ON)
#if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
tVoid csm_tclCsmAccessUser::_vHandleDataIndicationFF(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pDataIndFFMsg pMsg = (pDataIndFFMsg) pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPDataIndFF != NULL)
  {
    // call callback function
    pMsg->vPFNTPDataIndFF(pMsg->rMsgHeader.pvAccessUsrHandle,
                          pMsg->dwProtocolType,
                          pMsg->abAddressField,
                          pMsg->wDataLength);
  }
}
#endif
#endif
#endif

/*******************************************************************************
 * function    _vHandleDataErrorIndication
 * \doxydocu
 * \brief      Handles occurrance of data error indications
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     none
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
tVoid csm_tclCsmAccessUser::_vHandleDataErrorIndication(tVoid * pvMsg) const
{
  // cast message data to appropriate type
  pDataErrorIndMsg pMsg = (pDataErrorIndMsg) pvMsg;

  // hopefully the data are containing the registered callback function
  if(pMsg->vPFNTPDataErrorInd != NULL)
  {
    // call callback function
    pMsg->vPFNTPDataErrorInd(pMsg->rMsgHeader.pvAccessUsrHandle,
                             pMsg->dwProtocolType,
                             pMsg->abAddressField,
                             pMsg->dwDataErrorCode);
  }
}
#endif
#endif

/*******************************************************************************
 * function    _vHandleBusErrorIndication
 * \doxydocu
 * \brief      Handles occurrance of bus error indications
 *             (calls registered callback)
 *
 * \param[in]  pvMsg
 *             message data
 *
 * \return     tVoid
 *
 * \access     user mode service task
 * \reentrant  no (unless application callback is)
 *
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
tVoid csm_tclCsmAccessUser::_vHandleBusErrorIndication(tVoid * pvMsg ) const
{
  pBusErrorIndMsg pMsg = (pBusErrorIndMsg)pvMsg;

  if ( NULL != pMsg->vPFNBusErrorInd )
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("[0x%08X] _vHandleBusErrorIndication - UsrHandle:0x%08X  Callback:0x%08X  Bus:%X  ErrorCode:0x%04X",
                    _hProcessID,
                    pMsg->rMsgHeader.pvAccessUsrHandle,
                    pMsg->vPFNBusErrorInd,
                    ETG_CENUM(tCSM_BUS_NUMBER,    (tU8)  pMsg->bBus),
                    pMsg->wDataErrorCode));
    #endif

    pMsg->vPFNBusErrorInd(pMsg->rMsgHeader.pvAccessUsrHandle, pMsg->bBus, pMsg->wDataErrorCode);
  }
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

/*******************************************************************************
 * \doxydocu
 * \file          csm_access_usr.cpp
 * \brief         Functionality for usermode access to CSM
  *               This library must be linked to the user mode app.
 *
 * \project       Nissan LCN2 kai
 * \path          /di_can/adapt/osal/csm_access_usr/sources/
 *
 * \ingroup       CSM_ACCESS_USR
 *
 * \authors       VTeam
 *
 * COPYRIGHT      (c) 2011 Bosch GmbH
 *
 * \history_begin
 *
 * 17.11.11  /main/1 Kollai
 * - Initial revision.
 *
 * 16.05.12  /main/2 Kollai Kempen
 * - completely reworked
 * - new class csm_tclCsmAccessUser
 * - Communication beyond process boundary via Message-Queue
 *
 * 23.10.12  /main/3 Kempen Borck
 * - annoying debug output featured.
 *
 * 08.07.13  /main/5
 * - update from KPK for gen3 enviroment.
 *
 * 12.07.13  /main/6
 * - missing include added.
 *
 * 15.07.13  /main/7
 * - update from Reinhold for BAP support.
 *
 * 15.07.13  /main/8  Prhl
 * - ApplId added for in _s32FillCallbackFunctionsForApplCallbackInit() to some protocol types
 *
 * 20.11.13  /main/9
 * - trace functionality uses now ETG traces
 *
 * 02.06.15  /main/10 M. Prhl
 * - support for CSM_C_PTYPE_RN_OSEKI_NWM added.
 *
 * 01.09.15  /main/11 M. Prhl
 * - cast added to avoid warning with windows osal
 *
 * 20.10.16  /main/12 A.Borck
 * - reviewed and comments added
 * - service message queue: number of elements upon creation can be determined by middleware application/CSMIF
 * - some static functions converted to methods
 * - intermediate state - not yet finished !!!
 *
 * 20.10.16  /main/13 A.Borck
 * - trace output reworked completely
 * - 2 callbacks added for DataInd and DataConf (used by MPDT D-Data)
 *
 * 22.11.16  /main/14 A.Borck
 * - default value for number of service queue entries moved to header file
 * - the element size of a service queue entry can be stored in a class member
 *   a default value is taken but can be overwritten by the application using the CSMIF
 * - Service Message Queue Size is now depending on the number of elements and element size !!!
 * - the INIT message to the CSM PROC is containg this element size now for each access user app
 * - Before sending a message to Service Queue, the data length has to be checked (in access kernel) !!!
 *
 * 30.11.16  /main/15 A.Borck
 * - some trace outputs modified, some code alignments, no functional changes
 *
 * 01.03.17  /main/16 A.Borck
 * - LINT issues fixed
 *
 * 19.05.17  /main/17 A.Borck
 * - trace output for SignalRead set under USER4 to avoid annoying, cyclic output. USER3 is now the preferred trace class.
 *
 * 31.05.17  /main/18 A.Borck
 * - LINT issues fixed
 *
 * 10.08.17  /main/19 A.Borck
 * - final optimisations regarding TRACE output -> USER3 should be used as standard, USER4 is with CBR output
 *
 * \history_end
 *//**** END OF FILE **********************************************************/
