/* Description of csm_access_usr.h at the end of the file.             */
#ifndef CSM_ACCESS_USR_HEADER
/** File name for preprozessor commands */
#define CSM_ACCESS_USR_HEADER

// ProcCsm.h is required by ALL applications which intend to communicate with the CSM process
// via the "csm_access_user interface". It is containing important typedefs to setup
// messages to the CSM process.
#include "../../csm_proc/ProcCsm.h"

/*******************************************************************************
 * defines and macros (scope: csm_access_usr module local)
 ******************************************************************************/
// The max. number of message queue entries can now be requested/determined by the Application through
// the CSMIF class. Anyway, to be compatible, a default value should be assumed upon construction of our object.
// #ToDo BKA2HI: as long as not all Applications do support the new params, we have to reserve a large amount
// of service message queue entries to avoid a queue overflow due to the large amount of signal indications
#define CSM_C_SERVICE_MESSAGE_QUEUE_MAX_ENTRIES_DEFAULT     100

// The maximum message size handled by message queue has to be adjusted to the size of
// the largest message that shall be handled. If not using the data array, the payload
// for TP has to be added.
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #if (CSMAL_S_DATA_IND_STRUCT_WITH_DATA_ARRAY == CSM_C_F_ON)
    #define CSM_C_SERVICE_MESSAGE_QUEUE_MAX_MESSAGE_SIZE    (sizeof(trDataIndMsg))
  #else
    #define CSM_C_SERVICE_MESSAGE_QUEUE_MAX_MESSAGE_SIZE    (sizeof(trDataIndMsg) + CSMAL_MAX_DATA_PAYLOAD)
  #endif
#else
  #define CSM_C_SERVICE_MESSAGE_QUEUE_MAX_MESSAGE_SIZE      (sizeof(trCommunicationIndMsg))
#endif

/*******************************************************************************
 * typedefs (scope: csm_access_usr module local)
 ******************************************************************************/
// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tPVoid  pvHandle;
} trCSMParentMessageQueueItemInit;

// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tS32  s32Result;
} trCSMParentMessageQueueItemGeneralResult;

// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tS32  s32Result;
   tU8    u8State;
} trCSMParentMessageQueueItemGetCommunicationState;

// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tS32  s32Result;
   DWORD  u32SignalStatus;
   tU8   au8Data[CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH];
} trCSMParentMessageQueueItemSignalRead;

// Typ: Hilfselement fr die nachfolgende Struktur
typedef struct
{
   tU32  u32SignalId;
   tU32  u32SignalStatus;
   tU8    u8Length;
   tU8   au8Data[CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH];
} trCSMParentMessageQueueItemSingleSignalReadData;

// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tS32  s32Result;
   tU8    u8NumberOfSignals;
   trCSMParentMessageQueueItemSingleSignalReadData  arSignalData[CSM_C_ACCESS_LAYER_MULTI_SIGNAL_MAXNUMBER];
} trCSMParentMessageQueueItemMultipleSignalRead;

// Typ: Struktur die fr die Messsage Queue Union verwendet wird
typedef struct
{
   tS32  s32Result;
   tU8   au8Data[CSM_C_ACCESS_LAYER_DATA_READ_MAXLENGTH];
} trCSMParentMessageQueueItemDataRead;

// TYP: Union, die mageblich die Lnge eines Message-Queue Elements bestimmt
typedef union
{
   trCSMParentMessageQueueItemInit                    rInit;
   trCSMParentMessageQueueItemGeneralResult           rGeneralResult;
   trCSMParentMessageQueueItemGetCommunicationState   rGetCommunicationState;
   trCSMParentMessageQueueItemSignalRead              rSignalRead;
   trCSMParentMessageQueueItemMultipleSignalRead      rMultipleSignalRead;
   trCSMParentMessageQueueItemDataRead                rDataRead;
} tuCSMParentMessageQueueItemData;

//Typ: Enum zur Erzeugung von Message-Queue Element-Typ IDs
typedef enum
{
   CSM_PARENT_MESSAGE_QUEUE_ID_INIT = 1,
   CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_APPL,
   CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_APPL,
   CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_SIGNAL,
   CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL,
   CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS,
   CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND,
   CSM_PARENT_MESSAGE_QUEUE_ID_COMMUNICATION_REQ,
   CSM_PARENT_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE,
   CSM_PARENT_MESSAGE_QUEUE_ID_DATA_REQUEST,
   CSM_PARENT_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED,
   CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_WRITE,
   CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE,
   CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_READ,
   CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ,
   CSM_PARENT_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN,
   CSM_PARENT_MESSAGE_QUEUE_ID_CON_TEST,
   CSM_PARENT_MESSAGE_QUEUE_ID_DATA_READ,
   CSM_PARENT_MESSAGE_QUEUE_ID_MAX
} tenCSMParentMessageQueueItemID;

// Typ: Struktur aus der jedes Message-Queue Element besteht
typedef struct
{
   tenCSMParentMessageQueueItemID   enID;
    tuCSMParentMessageQueueItemData  uData;
} trCSMParentMessageQueueItem;

/*******************************************************************************
 * variable definition (scope: csm_access_usr module local)
 ******************************************************************************/
/* --none-- */

/*******************************************************************************
 * function prototypes (scope: csm_access_usr module local)
 ******************************************************************************/
class csm_tclCsmAccessUser
{
  public:
    // De/Constructor
    csm_tclCsmAccessUser();
    virtual ~csm_tclCsmAccessUser();

    // pre-Init (default values are used if caller does not apply the params)
    tVoid vApplCallbackPreInit(tU32 u32NrOfMsgQueueElements = CSM_C_SERVICE_MESSAGE_QUEUE_MAX_ENTRIES_DEFAULT,
                               tU32 u32MsgPayloadSize       = CSMAL_MAX_DATA_PAYLOAD // for test: CSMAL_MIN_DATA_PAYLOAD else CSMAL_MAX_DATA_PAYLOAD
                              );

    // CSM Interface mapping
    tS32 s32ApplCallbackInit(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, const tVoid* pcvCallBackFkt);
    tS32 s32ApplCallbackDeInit();
    tS32 s32SignalCallbackInit(tU32 u32SignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd);
    tS32 s32SignalCallbackDeInit(tU32 u32SignalId);
    tS32 s32SignalCallbackDeInitAll();
    tS32 s32BusErrorIndCallbackInit(tCSM_PFN_BUS_ERROR_IND vPFNBusErrorInd);
    tS32 s32CommunicationReq(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, tU8 u8Action, tU16 u16ApplID);
    tS32 s32GetCommunicationState(tU8 u8Bus, tU32 u32ProtocolType, const tVoid* pcvAddressField, tPU8 pu8ConnectState, tU16 u16ApplID);
    tS32 s32DataReq(tU32 u32ProtocolType, const tVoid* pcvAddressField, tPCU8 pcu8Data, tU16 u16DataLength);
    tS32 s32DataIndProcessed(tU32 u32ProtocolType, const tVoid* pcvAddressField);
    tS32 s32SignalWrite(tU32 u32SignalId, tVoid* pvNewSignalData, tU8 u8DataBufferLength, tU8 u8TxType);
    tS32 s32MultipleSignalWrite(const tCSM_MultipleSignalWrite *prSignalList, const tU8 u8NumberOfSignals);
    tS32 s32SignalRead(tU32 u32SignalId, tVoid* pvActSignalData, tU8 u8DataBufferLength, tPU32 pu32SignalStatus);
    tS32 s32MultipleSignalRead(const tCSM_MultipleSignalRead *prSignalList, const tU8 u8NumberOfSignals);
    tS32 s32GetAllStateIndicationsAgain() const;
    tS32 s32ConTest(tU32 u32ProtocolType, const tVoid* pcvAddressField, tU8 u8Action);
    tS32 s32DataRead(tU32 u32ProtocolType, const tVoid* pcvAddressField, tPU8 pu8Data, tU16 u16DataLength);

  private:
    //******************************************************************
    // members
    //******************************************************************
    static tU8         _u8AccessSemCounter;
    static tU32        _u32MessageQueueCounter;        // overall counter of created message queues
    static tU8         _u8ServiceTaskCounter;

    tPVoid             _pvCsmHandle;
    OSAL_tSemHandle    _hAccessSem;
    tU8                _u8AccessSemCount;

    OSAL_tProcessID    _hProcessID;
    tU32               _u32MessageQueueCount;          // message queue number for the instance of this class

    OSAL_tMQueueHandle _hParentMessageQueueID;         // message queue handle for synchronous results from CSM
    OSAL_tMQueueHandle _hServiceMessageQueueID;        // message queue handle for asynchronous data from CSM
    OSAL_tMQueueHandle _hProcMessageQueueID;           // message queue handle for requests to CSM

    tU32               _u32MaxNrOfServiceQueueElements;  // max. number of elements in the service message queue
    tU32               _u32ServiceQueueElementSize;      // element size of the service message queue

    //******************************************************************
    // methods
    //******************************************************************
    // service queue thread function
    static tVoid       _vCsmUserModeServiceTask(const tVoid *pcvArg ); // thread function without hidden THIS pointer passing !!!

    // service thread handler functions
    tVoid              _vHandleSignalIndication(tVoid * pvMsg) const;
    tVoid              _vHandleCommunicationIndication(tVoid * pvMsg) const;
    tVoid              _vHandleCommunicationConfirmation(tVoid * pvMsg) const;

    #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      tVoid            _vHandleDataConfirmation(tVoid * pvMsg) const;
      tVoid            _vHandleDataIndication(tVoid * pvMsg) const;

      #if (CSM_S_ISO_TP == CSM_C_F_ON)
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      tVoid            _vHandleDataIndicationFF(tVoid * pvMsg) const;
      #endif
      #endif

      #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      tVoid            _vHandleDataErrorIndication(tVoid * pvMsg) const;
      #endif
    #endif

    #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
    tVoid              _vHandleBusErrorIndication(tVoid * pvMsg) const;
    #endif

    // Init and Exit functions
    OSAL_tSemHandle    _hCreateAccessSemaphore();
    tVoid              _vDeleteAccessSemaphore();

    OSAL_tMQueueHandle _hCreateParentMessageQueue();
    OSAL_tMQueueHandle _hCreateServiceMessageQueue();
    OSAL_tThreadID     _hCreateServiceTask();

    tBool              _bInitCSMProc();

    // other functions
    tS32               _s32FillCallbackFunctionsForApplCallbackInit(trCSMProcMessageQueueItemRegisterAppl *prRegisterAppl, tU32 u32ProtocolType, const tVoid* pcvCallBackFkt) const;
    tS32               _s32WaitForGeneralResult( tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID );
};
#else // CSM_ACCESS_USR_HEADER
  #error "CSM_ACCESS_USR.H included several times"
#endif

/*******************************************************************************
 * \doxydocu
 * \file          csm_access_usr.h
 * \brief         Module internal header of CSM access layer (user part).
 *
 * \see           CAN-SW Architecture, written by VTeam
 * \see
 *
 * \sw_component  CAN stack
 * \project       Nissan LCN2 kai
 * \path          /di_can/adapt/osal/csm_access_usr/include
 *
 * \ingroup       CSM_ACCESS_LAYER
 * \ingroup       CSM_ACCESS_USR
 *
 * \authors       VTeam
 *
 * COPYRIGHT      (c) 2011 Bosch GmbH
 *
 * \history_begin
 *
 * 17.11.12  main\1  Kollai
 * - Initial revision.
 *
 * 16.05.12  main\2  Kollai-Kempen
 * - reworked completely
 *
 * 20.10.16  main\3  A.Borck
 * - intermediate state while reworking for MPDT D-Data
 *
 * 22.11.16  main\4  A.Borck
 * - default value for number of service queue entries increased from 10 to 100
 *   since some access user apps have a large amount of signal indications
 * - the default value can be overwritten by parametrizing the CSMIF class
 * - the element size of a service queue entry can be adapted as well now and is stored in a class member
 *
 * 26.01.17  main\5  A.Borck
 * - including "ProcCsm.h" now -> this is required for all USER applications (also for our own test process)
 *
 * 01.03.17  main\6  A.Borck
 * - made some methods const
 *
 * \history_end
 *******************************************************************************/
