/*******************************************************************************************
* FILE:			 dl_V850UpdateCtrlManager.cpp
*
* SW-COMPONENT: Software Download
*
* DESCRIPTION:  This file is a main thread for V850DLCtrl component. Which is invoked from 
*					 V850 Client handler. This class implements the download sequence for V850.  
*				    This will wait for event posted be Client handler and Inc Receiver and 
*               processed events.
*
* AUTHOR:		 Aditya Kumar Jha
*
* COPYRIGHT:	 (c) 2013 â€“ 2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
* ------------------------------------------------------------------------------------------------
* Date 				| Rev. 				| Author 		   	| Modification
* ------------------------------------------------------------------------------------------------
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					18.02.2014	Aditya Kumar Jha : 
*					Change for new trace class TR_CLASS_SWUPDATE_V850.
*					Added functionality for CRC and Version read.
*					19.02.2014	Aditya Kumar Jha : Version 1.0
*					Added functionality to write SW Version info into registry.
*					20.02.2014	Aditya Kumar Jha : Version 1.1
*					Re-arrange the function for maintainability. 
*					added vGetMountPoint() function to read mount point.
*					03.04.2014	Aditya Kumar Jha : Version 1.2
*					Added u8GetUpdateState() and chage the functionality for execute the update.
*              14.03.2014	Aditya Kumar Jha : Change for ERROR CODE.
*              18.03.2014	Aditya Kumar Jha : Change for block ID and Negative response handle.
*              28.03.2014	Aditya Kumar Jha : Change for writing and reading SW Version info into registry.
*              07.04.2014	Aditya Kumar Jha : Change for BLOCK ID and checking for .dnl file at persistence.
*              13.05.2014	Aditya Kumar Jha : Correction for the issue, while sending the data packet to V850.
*              21.05.2014	Aditya Kumar Jha : Remove mutex and introduce semaphore. Remove delay at begining.
*              12.06.2014  Aditya Kumar Jha , Neera.K.Ponnappa : Implemented No CRC and No Reset
*              30.06.2014  Neera.K.Ponnappa : Implemented Security Bytes and Jtag ID
*              22.07.2014  Neera.K.Ponnappa : Bug Fix - CMG3GB-863
                                              Implemented reset with switch back to BL Mode after download 
*              25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
*              31.07.2014  Neera.K.Ponnappa : Lint Warnings solved
*              01.08.2014	Aditya Kumar Jha : IS1515_CRQ 4166-2: Prevention of: Linux root user can bypass
*															 signature check of V850 code during software update.
*              04.08.2014	Aditya Kumar Jha : Review points incorporated.
*              06.08.2014  Neera.K.Ponnappa : Bug Fix CMG3GB-reset issue when Security Bytes/Jtag Bytes executed.
*              2014-08-27  Neera.K.Ponnappa,Apoorva K R  Implemented:CMG3G-5311  IS1604_Optimize Logging (update.log)@V850 Update Ctrl
*              16.09.2014	Aditya Kumar Jha : Fixes for Jira ticket CMG3GB-1128.
*              18.09.2014  Aditya Kumar Jha : added bIsSignByteSend() and replcae system("sleep time") call with sleep(time).
*              08.01.2015  Neera.K.Ponnappa : Fix - CMG3GB-1498
                                              [AID] Cyclic resets caused by "swu_common_v850_app_out.out" - stack smashing detected
*              27.02.2015  Neera.K.Ponnappa : Fix - CMG3GB - 1684
                                              V850 Default timeout increased from 5s to 15s.
*              04.03.2015  Neera.K.Ponnappa : Increased the v850 wait time between commands 10 60 and 10 01(switch from BL mode to Default session) from 3s to 12s.											  
*--------------------------------------------------------------------------------------------------------------------
* 12-03-2015		| 2.0	| VRI7COB (RBEI)	| Enabling TTFI's Traces;Ticket Number:CMG3GB-1324
*--------------------------------------------------------------------------------------------------------------------
* 17-03-2015		| 2.1	| VRI7COB (RBEI)	| Implementation of Change Request From Lothar.
* 					|		|					| After BL Update & Reset is send, Poll V850 to Get the Response For Active Session
* 					|		|					| Polling has to be done with time period of 2 sec for 8 cycles.
*---------------------------------------------------------------------------------------------------------------------
* 23-03-2015		| 2.2	| VRI7COB (RBEI)	| Review Issues are corrected. & CMG3GB-860 is fixed.
* --------------------------------------------------------------------------------------------------------------------
* 21-05-2015		| 2.3	| VRI7COB (RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 					|		|					| Parameter "screenreinit"
*---------------------------------------------------------------------------------------------------
* 24-06-2015        | 2.4   | VRI7COB (RBEI)    | Fixing the V850 Hang due to Not Processing
*                   |       |                   | all Messages in Message Queue. GMMY17-4737
*---------------------------------------------------------------------------------------------------
* 14-07-2015        | 2.5   | VRI7COB (RBEI)    | Fixing the Reset Issues CMG3GB-2166
* 				    |       |  				    | After 10-60 ACK From V850 Retry For BL Mode (Lothar Changes)
*---------------------------------------------------------------------------------------------------
* 25-08-2015        | 2.6   | AGL5KOR (RBEI)    | Fixing v850update.log cutoff - CMG3GB-2380 
* 				    |       |  				    | [SW Update]Persistent partition usage for SW update logs is around 40MB
*---------------------------------------------------------------------------------------------------
*31-08-2015			| 2.7	| VRI7COB (RBEI)	| Fixing the Reset Issue on Normal Startup CMG3GB-2291
*					|		|					| pthread_cancel() and pthread_join() API is used to handle it.
*---------------------------------------------------------------------------------------------------
*31-08-2015			| 2.8	| VRI7COB (RBEI)	| Updater Should be Downgrade Compactible.
*					|		|					| Changes are Made to make sure that Old AUTOSAR Version is Communicated to Update the New AUTOSAR.
*					|		|					| Ticket is CMG3GB-2402
*---------------------------------------------------------------------------------------------------
*25-09-2015			| 2.9	| VRI7COB (RBEI)	| Fixing the Ticket CMG3GB-2492.
					|		|					| Reseting the Flag 
*---------------------------------------------------------------------------------------------------
*19-02-2016			| 3.0	| VRI7COB (RBEI)	| Implementing CRQ CMG3G-9596
*---------------------------------------------------------------------------------------------------
*25-04-2016			| 3.1	| VRI7COB (RBEI)	| Fixing the Issue GMMY17-13603
*----------------------------------------------------------------------------------------------------
*18-05-2016			| 3.2	| VRI7COB (RBEI)	| Posting the Semaphore in case of setting the JTAG Value.
*					|		|					|  Ticket Number: CMG3GB-3323
*----------------------------------------------------------------------------------------------------
*******************************************************************************************************************************/
#define ETG_ENABLED
#include "trace_interface.h"
#include "ai_sw_update/common/base/imp/swupd_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN	
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclV850UpdateCtrlManager::
	#define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_V850
	#include "trcGenProj/Header/dl_V850UpdateCtrlManager.cpp.trc.h"
#endif

// OSAL types are required for Datapool
#define OSAL_S_IMPORT_INTERFACE_TYPES
#include "osal_if.h"



#define DEFAULT_TIMEOUT					15 //timeout in seconds
#define ERASE_TIMEOUT					30 //timeout in seconds
#define TRANSIT_TIME					10 //wait time after bootloader update in seconds
#define BLTODEF_TIME                    12//wait time for switch from BL mode to extented session.

#define BLTOAPP_ACTIVESESSION_TIME		 2 //2 seconds
#define BLTOAPP_RETRY_COUNT				 8 // 8 * 2 seconds

#define CRC_VALUE_SIZE					8 //size of CRC Value

#define DL_PER_DEVICE   "/dev/mmcblk1p3"
#define SWUPDATE "/swupdate/"
#define DNL_FORMAT ".dnl"
#define DEVICE_NAME "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS"
#define APP_VERSION_KEY	"SCC_APP_VERSION"
#define BL_VERSION_KEY	"SCC_BL_VERSION"
#define BM_VERSION_KEY	"SCC_BM_VERSION"

#define APP_LABEL_KEY	"SCC_APP_LABEL"
#define BL_LABEL_KEY	"SCC_BL_LABEL"
#define BM_LABEL_KEY	"SCC_BM_LABEL"

#define APP_BLOCKID			0x4024
//#define BL_BLOCKID			0x4023
#define BLU_BLOCKID			0x8300
#define APP_MEM_ADDRESS		0x0000E000
#define VERSION_REQ_LENGTH 5

#include <pthread.h>
#include <stdlib.h>
#include <sys/timerfd.h>
#include <dirent.h>
#include <sstream>
#include <cstring>
#include "dl_V850MessageDefine.h"
#include "ai_sw_update/common/base/imp/RegistryAccess.h"
#include "dl_V850UpdateCtrlManager.h"

tS32 sockfd;
struct sockaddr_in serv_addr, local_addr, remote_addr;
struct hostent *server, *local, *remote;
sk_dgram *dgram;
static bool bSecurityAccess = false;
static bool bJtagAccess =false;

#define CRC_INFO_PASS 7

/*enum CRCInfo_State {
  eCRCInfoUnInit = 0,          //!< The consistency check is not initialised.
  eCRCInfoPending,             //!< A check is ongoing, there is no result yet.
  eCRCInfoIPending2,          //!< A check is ongoing, there is no result yet.
  eCRCInfoFailAreaDescriptor,  //!< A check failed because the AreaDescriptor was not found.
  eCRCInfoFailLayoutCRC,       //!< The CRC over the AreaDescriptor itself failed.
  eCRCInfoFailLayoutVersion,   //!< The AreaDescriptor is used in a version that does not match.
  eCRCInfoFailCRC,             //!< The CRC over the memory area did not match the one stored in the AreaDescriptor.
  eCRCInfoPass                 //!< The consistency check passed all tests.
};*/


static dl_tclV850UpdateCtrlManager* s_ptrself = NULL;

/******************************************************************************/
/*                                                                            */
/* STATIC MEMBER VARIABLES                                                    */
/*                                                                            */
/******************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bIsV850DownloadSuccess = false;
tBool dl_tclV850UpdateCtrlManager::m_bIsSignByteSend = false;

/*******************************************************************************************
* FUNCTION:		dl_tclV850UpdateCtrlManager
* DESCRIPTION:	Private Constructor
* PARAMETER:
*					NULL			
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
*					1.1 Adding the ETG Register File, Channel & Open Port
********************************************************************************************/
dl_tclV850UpdateCtrlManager::dl_tclV850UpdateCtrlManager()
:m_mainloop()
{	
	ETG_I_REGISTER_FILE();
	ETG_I_REGISTER_CHN(0);

	g_fCallback = NULL;
	m_listOfOpera.clear();
	if (sem_init(&m_updateCtrlSem, 0, 0) == -1)
	{
		error("sem_init");
	}

	//If socket is not closed in perivous run 
	//due to any reset in between program.
	if( sockfd != -1 )
	{
		close(sockfd);
	}

	m_chSWVersion[0] = '\n';
	m_chSWLabel[0] = '\n';
	m_chCheckSumVal[0] = '\n';
	m_chCmdBuff[0] = '\n';
	m_u32BlockSize = 0;
	m_u32RBlockSize = 0;
	m_u32LastChuckData = 0;
	m_pFile = NULL;
	m_u32MemoryAddress = 0;
	m_szFilePath = NULL;
	m_szFileName = NULL;
	m_pclV850MessageFactory = NULL;
	m_pMessageQueue = NULL;
	m_pclIncSender = NULL;
	m_pclIncReceiver = NULL;
	m_u8DlState = 0;
	m_evfdClientReq = EP_HANDLE_INVALID;
	m_evfdIncRx = EP_HANDLE_INVALID;
	m_evFdTimer = EP_HANDLE_INVALID;
	m_bIsReadAllSWVersion = false;
	m_sectimeout = DEFAULT_TIMEOUT;
	m_bIsBLUpdate = false;
	m_bIsDnlFile = false;
	m_bIsBLMode = false;
	m_chErrorCode = 0;
	m_bIsBLProgMode = false;
	m_bIsBLSuccess = false;
	m_u16BlockID = 0;
	m_s8CheckSumResult = -1;
	m_s8SwVersionResult = -1;
	m_s8SwLabelResult = -1;
	m_bResetEnabled = true; 
	m_bCrcEnabled = true;
	s_ptrself = this;
	m_szSecValToWrite = NULL;
	m_szSecValToWrite = new tChar[DEFAULT_MSGSZ];
	if(NULL == m_szSecValToWrite)
	{
		printf("ERROR: Failed to allocate memory, exiting!!\n");
	}
	m_szJtagValToWrite = NULL;
	m_szJtagValToWrite = new tChar[DEFAULT_MSGSZ];
	if(NULL == m_szJtagValToWrite)
	{
		printf("ERROR: Failed to allocate memory, exiting!!\n");
	}
	//Start of CRQ CMG3G-9596
	m_szJtagCRCValToCheck = NULL;
	m_szJtagCRCValToCheck = new tChar[DEFAULT_MSGSZ];
	if(NULL == m_szJtagCRCValToCheck)
	{
		printf("ERROR: Failed to allocate memory, exiting!!\n");
	}
	//End of CRQ CMG3G-9596

	//m_bIsSignByteSend = false;
	m_bSecFlag = false;
	m_bJtagFlag = false;
	m_bResetFlag = false;
	m_bCrcFlag = false;
	m_bVerFlag = false;
	m_bUpdateReqFlag = false;
	m_bIsBLCmdSend = false;
	m_u32RetryCount = 0;
	m_eRetryAfter = EN_RETRY_AFTER_INVALID;
	m_v850DlCtrlMgrThread = 0;
	UpdateLog::vSetLogFile("v850update.log");
	UpdateLog::vUpdateLogOpen();
	
	//Start of CRQ CMG3G-9596
	m_u8ProcessingClientMsg = DL_REQUEST_INVALID;
	m_u8SecurityBitVal = 0;
	m_eJTAGValidationResult = E_JTAG_VALIDATION_INVALID;
	//End of CRQ CMG3G-9596
	//Coverity fix for 17697
	m_bIsAppMode = false;
	std::memset(&m_DFInfo, 0, sizeof(Flash_strTYInfo));
}


/*******************************************************************************************
* FUNCTION:		~dl_tclV850UpdateCtrlManager
* DESCRIPTION:	Destructor
* PARAMETER:
*					NULL			
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
*					1.1 Adding the ETG UnRegister File, Channel & Close Port
********************************************************************************************/
dl_tclV850UpdateCtrlManager::~dl_tclV850UpdateCtrlManager()
{
	UpdateLog::vUpdateLogClose();
	sem_destroy(&m_updateCtrlSem);

	m_mainloop.vStopMainLoop();//Stop mainloop
	s_ptrself = NULL; //Set pointer to NULL

	close(m_evfdClientReq);
	close(m_evfdIncRx);
	//coverity fix for 18198
	//close(m_evFdTimer);
	

	vMessageTermination();

	if( NULL != m_pclV850MessageFactory )
	{
		delete m_pclV850MessageFactory;
		m_pclV850MessageFactory = NULL;
	}

	if( NULL != m_pMessageQueue )
	{
		delete m_pMessageQueue;
		m_pMessageQueue = NULL;
	}

	if( NULL != m_pclIncSender )
	{
		delete m_pclIncSender;
		m_pclIncSender = NULL;
	}

	if( NULL != m_pclIncReceiver )
	{
		delete m_pclIncReceiver;
		m_pclIncReceiver = NULL;
	}

	if(NULL != 	m_szSecValToWrite)
	{
		delete [] m_szSecValToWrite;
	}
	if(NULL != 	m_szJtagValToWrite)
	{
		delete [] m_szJtagValToWrite;
	}
	//Coverity fix for 20250
	if (NULL != m_szJtagCRCValToCheck)
	{
		delete [] m_szJtagCRCValToCheck;
	}

	m_chSWVersion[0] = '\n';
	m_chSWLabel[0] = '\n';
	vCloseBinaryFile();

	//close timer fd
	if(m_evFdTimer >= 0)
	{
		close(m_evFdTimer);
	}
	m_szFilePath = NULL;
	m_szFileName = NULL;	
	m_szSecValToWrite = NULL;
        m_szJtagValToWrite = NULL;    

	ETG_I_UNREGISTER_CHN();
	ETG_I_UNREGISTER_FILE();

}

/*******************************************************************************************
* FUNCTION:		bInitV850Message
* DESCRIPTION:	Message initialization for V850 and create a thread for incReceiver
*					to listen on download port.
* PARAMETER:
*					NULL  
*					 
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bInitV850Message(  T_fCallback fCallback, tBool bIsAppStartOnly )
{	
	tCString method = "bInitV850Message";
	ETG_TRACE_USR4 (("%s was entered", method));

	tBool bRetVal = false;
	//asign call back to globle
	g_fCallback = fCallback;

	if ( bInitComponents() )
	{
		//Create FDs
		vCreateAndAddEventFD();
		
		//Create a socket to listen download port
		if ( (s32SocketCreation ( DOWNLOAD_PORT ) == 1) 
			&&
			(dgram != NULL) )
		{
			m_pclIncReceiver->s32CreateIncReceiverThread( dgram );
			bRetVal = true;
		}

		//Create a seperate working thread for 					
		pthread_create( &m_v850DlCtrlMgrThread, NULL, &dl_tclV850UpdateCtrlManager::v850DlCtrlMgrThreadCallBack, s_ptrself );

		//wait for 3 seconds
		//vDelaySimulation();

		//Assign the download state to init state
		m_u8DlState = DL_INIT_STATE;
		
		/* Read current mode */
		vReadActiveSession();

		/* Case for initiate the application without any parameter */
		if( bIsAppStartOnly )
		{
			vStartInitOperation();
		}

	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;
}

/*******************************************************************************************
* FUNCTION:		bInitV850Message
* DESCRIPTION:	Message initialization for V850 and create a thread for incReceiver
*					to listen on download port.
* PARAMETER:
*					NULL  
*					 
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bInitComponents( void )
{
	tCString method = "bInitComponents";
	ETG_TRACE_USR4 (("%s was entered", method));

	tBool bRetVal = false;
	//instantiate the v850 message compositor
	m_pclV850MessageFactory = new dl_tclV850MessageFactory();

	//create message queue
	m_pMessageQueue = new MessageQueue();

	//instantiate the INCSender
	m_pclIncSender = new dl_tclIncSender();

	//instantiate the INCReceiver
	m_pclIncReceiver = new dl_tclIncReceiver();

	if ( ( NULL != m_pclV850MessageFactory )
		&& 
		( NULL != m_pclIncSender ) 
		&&
		( NULL != m_pclIncReceiver ) )
	{
		UpdateLog::vUpdateLogTrace("%s Initialising the Sender,Reciver,MessageFactory,Message Queue ", DL_LOG_INFO );
		bRetVal = true;
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;
}
/*******************************************************************************************
* FUNCTION:		vReadActiveSession
* DESCRIPTION:	This function read the current mode of V850.
*					
* PARAMETER:
*					NULL  
*					 
* RETURNVALUE:	tVoid
*
* HISTORY:
*					03.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vReadActiveSession( void )
{
	tCString method = "vReadActiveSession";
	ETG_TRACE_USR4 (("%s was entered", method));

	//Read Active Session State
	INFO_MSG (INFO_LEVEL_1, "INFO: Reading Active Session State\n");
	UpdateLog::vUpdateLogTrace("%s Reading Active Session State", DL_LOG_INFO );
	bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );
	//Wait till get the active state
	vWaitForSemRelease();

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vStartInitOperation
* DESCRIPTION:	This function perform the initial operation like reading active state, checking 
*					persence of binary data at persistence partition and reading version info.
*					
* PARAMETER:
*					NULL  
*					 
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					13.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vStartInitOperation( void )
{
	
	tCString method = "vStartInitOperation";
	ETG_TRACE_USR4 (("%s was entered", method));

	tBool bIsDnlFileAvailable = false;	
	//if available, start the download process
	if( true == m_bIsBLMode )
	{	

		//validate the presence of .dnl file at persistant partitions
		std::string sMountPoint = "";
		vGetMountPoint(DL_PER_DEVICE, sMountPoint);
		sMountPoint.append(SWUPDATE);
		
		bIsDnlFileAvailable = bCheckBinaryFile(sMountPoint, ".dnl");

		//if .dnl file is not present than check for App.bin file
		if( false == bIsDnlFileAvailable )
		{
			bIsDnlFileAvailable = bCheckBinaryFile(sMountPoint, "App.bin");
			
		}

		if( true == bIsDnlFileAvailable )
		{
			//Assign the file name and path to a const character array
			//Coverity fix for 18240 
			vSetFilePath(sMountPoint.c_str());
			vSetFileName("-app");
			m_bUpdateReqFlag = true;
			UpdateLog::vUpdateLogTrace("%s DNL File Available", DL_LOG_INFO );
			bStartV850Download();
			//Wait till download is finished
			vWaitForSemRelease();
		}

	}
	else if(true == m_bIsBLProgMode)
	{	
		UpdateLog::vUpdateLogTrace("%s V850 is in BL_Prog Session", DL_LOG_SERVICE );
	    sem_post(&m_updateCtrlSem);
	    vTerminateProgram();
		OSAL_vSetProcessExitCode(V850_MODE_ERROR);
        OSAL_vProcessExit();	
	} 
	else 
	{
		INFO_MSG (INFO_LEVEL_1, "INFO: Reading Version information...\n");
		UpdateLog::vUpdateLogTrace("%s Reading Version information...", DL_LOG_INFO );
		/* Read out the Software version */
		bReadVersionInfo( true );
	}

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vCreateAndAddEventFD
* DESCRIPTION:	This function creates the event FD for client request and response and add 
*					them to main loop.
* PARAMETER:
*					NULL  
*					 
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					13.11.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vCreateAndAddEventFD()
{
	//create Client Request FD
	m_evfdClientReq = eventfd(0,EFD_NONBLOCK);
	assert(m_evfdClientReq != EP_HANDLE_INVALID);	
	//Add fd to main loop
	if ( m_mainloop.bAddfdtoPoll( m_evfdClientReq, POLLIN, vHandleClientEvent ) )
	{
		ETG_TRACE_USR4 (("m_evfdClientReq fd added to poll"));
	}
	else
	{
		ETG_TRACE_USR4 (("failed to add m_evfdClientReq fd to poll"));
	}

	//create FD for INC receiver
	m_evfdIncRx = eventfd(0,EFD_NONBLOCK);
	assert(m_evfdIncRx != EP_HANDLE_INVALID);

	//Add fd to main loop
	if ( m_mainloop.bAddfdtoPoll( m_evfdIncRx, POLLIN, vHandleDownloadCtrlEvent ) )
	{
		ETG_TRACE_USR4 (("m_evfdIncRx fd added to poll"));
	}
	else
	{
		ETG_TRACE_USR4 (("failed to add m_evfdIncRx fd to poll"));
	}

	//Create timer fd
	m_evFdTimer = timerfd_create(CLOCK_MONOTONIC,TFD_NONBLOCK);
	assert(m_evFdTimer != EP_HANDLE_INVALID);

	//Add timer fd to main loop
	if(m_mainloop.bAddfdtoPoll( m_evFdTimer, POLLIN, vHandleTimeoutEvent ))
	{
		ETG_TRACE_USR4(("m_evFdTimer fd added to poll"));
	}
	else
	{
		ETG_TRACE_USR4(("Failed to add m_evFdTimer fd to poll"));
	}

}
/*******************************************************************************************
* FUNCTION:		s32SocketCreation
* DESCRIPTION:	Create a Socket to for incReceiver
*					to listen on download port.
* PARAMETER:
*					tU32 u32LocalPort
*				
*
* RETURNVALUE:	tS32
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS32 dl_tclV850UpdateCtrlManager::s32SocketCreation (tU32 u32LocalPort) const
{
	tCString method = "s32SocketCreation";
	ETG_TRACE_USR4 (("%s was entered", method));

	sockfd = socket( AF_BOSCH_INC_AUTOSAR, SOCK_STREAM, 0 ); // wait for a message to arrive
	if (sockfd < 0)
	{
		ETG_TRACE_USR4(("DNL: ERROR opening socket"));
		error("ERROR: opening socket");
		return -1; //Coverity fix for 128902
	}
	//Data gram initiation 
	dgram = dgram_init(sockfd, DGRAM_MAX, NULL);
	if (dgram == NULL)
	{
		ETG_TRACE_USR4(("DNL: dgram_init failed"));
		error("ERROR: dgram_init failed\n");
		return -1; //Coverity fix for 128902
	}

	vBindToLocalHost( u32LocalPort );
	vConnectToRemoteHost( u32LocalPort );

	ETG_TRACE_USR4 (("%s was left", method));

	return 1;
}

/*******************************************************************************************
* FUNCTION:		vBindToLocalHost
* DESCRIPTION:	Bind Socket to the local host.
*					
* PARAMETER:
*					tU32 u32LocalPort
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					05.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vBindToLocalHost( tU32 u32LocalPort ) const
{
	tCString method = "vBindToLocalHost";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32PortNumber = u32LocalPort; 
	//local host address, iMX
	local = gethostbyname("scc-local");	  
	if (local == NULL)
	{
		ETG_TRACE_USR4(("DNL: ERROR no such local host"));
		error("ERROR, no such local host");
		return; //Coverity fix for 126791
	}
	local_addr.sin_family = AF_INET;
	memcpy( (char *)&local_addr.sin_addr.s_addr, (char *)local->h_addr, local->h_length );
	local_addr.sin_port = htons(u32PortNumber); 

	//bind to local host
	if ( bind( sockfd, (struct sockaddr *) &local_addr, sizeof(local_addr) ) < 0 )
	{
		ETG_TRACE_USR4(("DNL: ERROR on binding"));
		error("ERROR on binding");
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vConnectToRemoteHost
* DESCRIPTION:	Connect Socket to the remote host.
*					
* PARAMETER:
*					tU32 u32LocalPort
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					05.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vConnectToRemoteHost( tU32 u32LocalPort ) const
{
	tCString method = "vConnectToRemoteHost";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32PortNumber = u32LocalPort; 
	// remote host address, V850
	remote = gethostbyname("scc"); 
	if (remote == NULL)
	{
		ETG_TRACE_USR4(("DNL: ERROR no such remote host"));
		error("ERROR, no such remote host");
		return;  //Coverity fix for 126792
	}
	remote_addr.sin_family = AF_INET;
	memcpy ( (char *) &remote_addr.sin_addr.s_addr, (char *) remote->h_addr, remote->h_length );
	remote_addr.sin_port = htons(u32PortNumber); 

	//Connect to remote host
	if ( connect( sockfd,(struct sockaddr *) &remote_addr, sizeof(remote_addr) ) < 0 )
	{
		ETG_TRACE_USR4(("DNL: ERROR connecting"));
		error("ERROR connecting");
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vDlCtrlMgrWorkedThread
* DESCRIPTION:	This function validate the presence of v850.dnl file at persistant partitions, 
*					if presence it start the download process and Wait for any Events to occurs.
*
* PARAMETER:
*					
*
* RETURNVALUE:	tVoid 
*
* HISTORY:
*					25.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vDlCtrlMgrWorkedThread( )
{
	tCString method = "vDlCtrlMgrWorkedThread";
	ETG_TRACE_USR4 (("%s was entered", method));


	//Wait for Events 
	m_mainloop.vRunMainLoop();


	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vHandleClientEvent
* DESCRIPTION:	callback function invoked by client FD
*					
* PARAMETER:
*					tInt evFd,  
*					tUInt eEventType	
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleClientEvent( tInt evFd, tUInt eEventType ) 
{
			
	if ( s_ptrself != NULL )
	{
		
		uint64_t data = 0;
		eEventType = eEventType; //for lint
		//Coverity fix for 10274
		if (-1 == read(s_ptrself->m_evfdClientReq, &data, sizeof(data))) //Clean the event field by triggering a read
		{
			 ETG_TRACE_USR4(("DNL: file read error"));
                         INFO_MSG(INFO_LEVEL_1, "WARN: file read error");
		}
		s_ptrself->vProcessClientEvent( evFd );
	}
    

}

/*******************************************************************************************
* FUNCTION:		vHandleDownloadCtrlEvent
* DESCRIPTION:	callback function invoked by INC receiver FD
*					
* PARAMETER:
*					tInt evFd,  
*					tUInt eEventType	
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleDownloadCtrlEvent( tInt evFd, tUInt eEventType ) 
{
	if ( s_ptrself != NULL )
	{
		uint64_t data = 0;
		eEventType = eEventType; //for lint
		//Coverity fix for 10276
		if (-1 == read(s_ptrself->m_evfdIncRx, &data, sizeof(data))) //Clean the event field by triggering a read
		{
                         ETG_TRACE_USR4(("DNL: file read error"));
                         INFO_MSG(INFO_LEVEL_1, "WARN: file read error");
                }
		s_ptrself->vProcessDownloadCtrlEvent( evFd );
	}
	
}

/*******************************************************************************************
* FUNCTION:		vHandleTimeoutEvent
* DESCRIPTION:	callback function invoked by timer FD
*					
* PARAMETER:
*					tInt evFd,  
*					tUInt eEventType	
*
* RETURNVALUE:	Void
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleTimeoutEvent( tInt evFd, tUInt eEventType ) 
{
	if ( s_ptrself != NULL )
	{
		uint64_t data = 0;
		eEventType = eEventType; //for lint
		//Coverity fix for 10275
		if (-1 == read(s_ptrself->m_evfdIncRx, &data, sizeof(data))) //Clean the event field by triggering a read
		{
                         ETG_TRACE_USR4(("DNL: file read error"));
                         INFO_MSG(INFO_LEVEL_1, "WARN: file read error");
                }
		s_ptrself->vProcessTimoutEvent( evFd );
	}
	
}

/*******************************************************************************************
* FUNCTION:		vProcessTimoutEvent
* DESCRIPTION:	This function handles the event post by timer handler
*					
* PARAMETER:
*					tInt evFd,  
*
* RETURNVALUE:	Void
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vProcessTimoutEvent( tInt evFd )
{
	tCString method = "vProcessTimoutEvent";
	ETG_TRACE_USR4 (("%s was entered", method));
	ETG_TRACE_USR4(("DL STATE:%u", m_u8DlState));

	/* Request from client */
	if( m_evFdTimer == evFd && !m_bIsBLCmdSend)
	{
		INFO_MSG (INFO_LEVEL_1, "WARN: TimeOut waiting for response from V850 to the request: %s\n", m_chCmdBuff);
		UpdateLog::vUpdateLogTrace("%s TimeOut waiting for response from V850 to the request", DL_LOG_SERVICE );
		vStopTimer();

//		start download process if state is DL_TRANSFER_STATE
		if ( DL_TRANSFER_STATE == m_u8DlState )
		{
			bStartV850Download();
		}
		else if ( true == m_bIsSignByteSend )
		{
			//send transfer exit command
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for Transfer Exit", DL_LOG_INFO );
			bWriteToInc( MSG_TX_REQUEST_TRANSF_EXIT );
			m_bIsSignByteSend = false;
		}
		else
		{
			/* release the download running mutex */
			INFO_MSG (INFO_LEVEL_1, "Time out of send Packet\n");
			UpdateLog::vUpdateLogTrace("%s Time out of send Packet", DL_LOG_INFO );

			if(m_u8DlState == DL_RETRY_STATE )
			{
				if(m_u32RetryCount < BLTOAPP_RETRY_COUNT)
				{
					vSetTimeout(BLTOAPP_ACTIVESESSION_TIME);
					bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );
					m_u32RetryCount++;
					INFO_MSG (INFO_LEVEL_1, "Retrycount:%d\n", m_u32RetryCount);
					UpdateLog::vUpdateLogTrace("%s Retrycount:%d", DL_LOG_INFO,  m_u32RetryCount);
				}
				else
				{
					INFO_MSG (INFO_LEVEL_1, "EXIT:Retrycount:%d\n", m_u32RetryCount);
					UpdateLog::vUpdateLogTrace("%s EXIT:Retrycount:%d", DL_LOG_INFO,  m_u32RetryCount);
					sem_post(&m_updateCtrlSem);
				}
			}
			else
				sem_post(&m_updateCtrlSem);

		}
	}
	else if(m_bIsBLCmdSend)
	{
		m_bIsBLCmdSend = false;
		INFO_MSG (INFO_LEVEL_1, "INFO: No Response For 10-60\n");
		vStopTimer();

		m_u8DlState = DL_RETRY_STATE;
		m_eRetryAfter = EN_RETRY_AFTER_SWITCHBL;
		vSetTimeout(BLTOAPP_ACTIVESESSION_TIME);
		bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );

		m_u32RetryCount = 1;
		INFO_MSG (INFO_LEVEL_1, "Retrycount:%d\n", m_u32RetryCount);
		UpdateLog::vUpdateLogTrace("%s Retrycount:%d", DL_LOG_INFO,  m_u32RetryCount);
	}


}

/*******************************************************************************************
* FUNCTION:		vProcessClientEvent
* DESCRIPTION:	This function handles the event post by client handler
*					
* PARAMETER:
*					tInt evFd,  
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vProcessClientEvent( tInt evFd )
{
	
	tCString method = "vProcessClientEvent";
	ETG_TRACE_USR4 (("%s was entered", method));

	/* Request from client */
	if( (m_evfdClientReq == evFd) 
		&& 
		(NULL != m_pMessageQueue)
		&&
		(!(m_pMessageQueue->mQueueHandlerClientReq.empty())) )
	{
		//m_pMessageQueue->vReqLock();
		tU8 *message = (tU8* )(m_pMessageQueue->mQueueHandlerClientReq.front());
		m_pMessageQueue->mQueueHandlerClientReq.pop();
		//m_pMessageQueue->vReqUnlock();

		if( NULL != message )
		{
			vHandleClientReqMsg( message[0] );			

			/**The Memory allocated is freed here*/
			delete[] message;
			message = NULL;

		}//if( NULL != message )

	}//if( m_evfdClientReq == evFd )


}

/*******************************************************************************************
* FUNCTION:		vHandleClientReqMsg
* DESCRIPTION:	This function handles the messageses in request message queue
*					
* PARAMETER:
*					tU8 u8Message,  
*
* RETURNVALUE:	Void
*
* HISTORY:
*					13.11.2013	Aditya Kumar Jha 
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleClientReqMsg( tU8 u8Message )
{
	tCString method = "vHandleClientReqMsg";
	ETG_TRACE_USR4 (("%s was entered", method));

	//read the download state
	tU8 u8DLState = u8GetDownloadState();
    
	switch( u8Message )
	{
		/* request from client to start the download */
	case DL_REQUEST_START:
		{						
			bStartV850Download( );						

			break;
		}

		/* request from client to abort the download */
	case DL_REQUEST_ABORT:
		{
			if( (DL_INIT_STATE == u8DLState)
				||
				(DL_VERIFY_STATE == u8DLState)
				||
				(DL_SEQ_SETUP_STATE == u8DLState) )
			{
				//Abort//Provided, if this function would finish only after DL is completed

			}
			break;
		}
	/* request from client to read the boot manager Version information */
	/* request from client to read the boot loader Version information */
	/* request from client to read the application Version information */
	case DL_REQUEST_BM_VERSION:
    case DL_REQUEST_BL_VERSION:
    case DL_REQUEST_APP_VERSION:
		{
		  bReadVersionInfo( false, u8Message );
		  break;
		}
       /* request from client to read the Boot manager CRC Value */
	case DL_REQUEST_BM_CRC:
		{
			//send command to read the boot manager CRC
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for BM CRC", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_BM_CRC );
			break;
		}

	/* request from client to read the boot loader CRC Value */
	case DL_REQUEST_BL_CRC:
		{
			//send command to read the boot loader CRC
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for BL CRC", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_BL_CRC );
			break;
		}

	/* request from client to read the application CRC Value */
	case DL_REQUEST_APP_CRC:
		{
			//send command to read the application CRC
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for APP CRC", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_APP_CRC );
			break;
		}
		
	case DL_REQUEST_ECU_RESET:
		{
			//send command to Reset v850
			bStopV850Download();
			break;
		}

	default:
		{
			vHandleClientReqMsgExtended(u8Message);
			break;
			
		}

	}//switch( message )

}


/*******************************************************************************************
* FUNCTION		: vHandleClientReqMsgExtended
* DESCRIPTION	: This function handles the messageses in request message queue
* PARAMETER		: tU8 u8Message,  
* RETURNVALUE   :	Void
* HISTORY		:
* DATE			| VERSION			| AUTHOR	| MODIFICATION
------------------------------------------------------------------------------
* 18-08-2014	| 0.1				| Neera		| Initial
------------------------------------------------------------------------------
* 19-02-2016	| 0.2				| vri7cob	| Changes For CMG3G-9596
*******************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleClientReqMsgExtended( tU8 u8Message )
{
	tCString method = "vHandleClientReqMsg";
	ETG_TRACE_USR4 (("%s was entered", method));
    
	switch( u8Message )
	{	
	case DL_REQUEST_SECURITY_ACCESS:
		m_u8ProcessingClientMsg = DL_REQUEST_SECURITY_ACCESS;
		if(m_szSecValToWrite != NULL)
			bWriteToInc(m_szSecValToWrite);
		break;
	case DL_REQUEST_JTAG_ACCESS:
		m_u8ProcessingClientMsg = DL_REQUEST_JTAG_ACCESS;
		if(m_szJtagValToWrite != NULL)
			bWriteToInc(m_szJtagValToWrite);
		break;
	case DL_REQUEST_READ_SECURITY:
		m_u8ProcessingClientMsg = DL_REQUEST_READ_SECURITY;
		bWriteToInc(MSG_RX_READ_SEC_PROTECT_BYTE);
		break;
	case DL_DO_STARTUP_SEQUENCE:
		m_u8ProcessingClientMsg = DL_DO_STARTUP_SEQUENCE;
		if(m_bIsBLProgMode )
		{
			ETG_TRACE_USR4(("V850 is already BL Programming Mode"));
			INFO_MSG (INFO_LEVEL_1, "INFO: V850 is already BL Programming Mode\n");
			sem_post(&m_updateCtrlSem);
		}
		else
			bStartDnlSequence();
		break;
	case DL_DO_JTAGFINGERPRINT_VALIDATION:
		m_u8ProcessingClientMsg = DL_DO_JTAGFINGERPRINT_VALIDATION;
		if(m_szJtagCRCValToCheck != NULL)
			bWriteToInc(m_szJtagCRCValToCheck);
		break;
	case DL_DO_SCC_RESET:
		m_u8ProcessingClientMsg = DL_DO_SCC_RESET;
		bWriteToInc( MSG_TX_ECU_RESET );
		sem_post(&m_updateCtrlSem);
		break;
	default:
		{
			break;
		}

	}//switch( message )

}

/*******************************************************************************************
* FUNCTION:		bReadVersionInfo
* DESCRIPTION:	This function read the version information of all binary file and assign to the 
*					member variable.
*					
* PARAMETER:
*					tBool bIsReadAll, true:to read all SW version, false:to read particular version
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					02.01.2014	Aditya Kumar Jha
*					Initial Version.
---------------------------------------------------------------------------------------------
*25-04-2016			| 1.1	| VRI7COB (RBEI)	| Fixing the Issue GMMY17-13603
---------------------------------------------------------------------------------------------
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bReadVersionInfo( tBool bIsReadAll, tU8 u8SwVer )
{
	tBool bRetVal = true;

	m_bIsReadAllSWVersion = bIsReadAll;

	if ( true == m_bIsReadAllSWVersion )
	{

		char buffer[VERSION_REQ_LENGTH+1];
		memset( buffer, 0, (VERSION_REQ_LENGTH+1) );	
		buffer[VERSION_REQ_LENGTH] = '\0';
//		//Read all SW Version
//		for ( tU16 u16VerReqID = MSG_RX_READ_VER_BYTE_3; u16VerReqID < MSG_RX_READ_VER_BYTE_MAX; /*u16VerReqID++*/)
//		{
//
//			sprintf( buffer, "%d-%d-%d", MSG_RX_READ_VER_BYTE_1, MSG_RX_READ_VER_BYTE_2, u16VerReqID);
//			bWriteToInc(buffer);
//			u16VerReqID += MSG_RX_READ_VER_OFFSET;
//			//Wait till version read out is finished
//			vWaitForSemRelease();
//		}

		//Boot Manager
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_BM_VERSION);
		sem_wait(&m_updateCtrlSem);
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_BM_LABEL);
		sem_wait(&m_updateCtrlSem);

		//Boot Loader
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_BL_VERSION);
		sem_wait(&m_updateCtrlSem);
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_BL_LABEL);
		sem_wait(&m_updateCtrlSem);

		//Application
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_APP_VERSION);
		sem_wait(&m_updateCtrlSem);
		bWriteToInc(MSG_TX_READ_DATA_IDENTI_APP_LABEL);
		sem_wait(&m_updateCtrlSem);

	}
	else 
	{
		switch( u8SwVer )
		{
		default:
			break;

		case DL_REQUEST_BM_VERSION:
			
			//Read Boot Manager SW Version
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for BM VERSION", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_BM_LABEL );
			break;

		case DL_REQUEST_BL_VERSION:
			
			//Read Boot loader SW Version
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for BL VERSION", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_BL_LABEL );
			break;

		case DL_REQUEST_APP_VERSION:
			//Read application SW Version
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for APP VERSION", DL_LOG_INFO );
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_APP_LABEL );
			break;
		
		}

	}

	return bRetVal;

}
/*******************************************************************************************
* FUNCTION:		vProcessDownloadCtrlEvent
* DESCRIPTION:	This function handles the event post by Inc receiver or V850DownloadCtrlManager
*					
* PARAMETER:
*					tInt evFd,  
*					tUInt eEventType	
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					24-06-2015  VRI7COB (RBEI)
*					Message Queue has to be Processed till it becomes empty.
*					So While Loop is added for that.
*					14-07-2015 VRI7COB (RBEI)
*					CMG3GB-2166 - Fixing the Reset. Checking the NUll Pointer in Message Queue.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vProcessDownloadCtrlEvent( tInt evFd )
{
	tCString method = "vProcessDownloadCtrlEvent";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU8 u8Temp = 0;
	tBool bRetVal = true;

	/* Response from INC Receiver */
	if( (m_evfdIncRx == evFd) && (NULL != m_pMessageQueue) )
	{
	   while( (NULL != m_pMessageQueue)&&
	              (!(m_pMessageQueue->mQueueHandlerIncReceiver.empty()))
	           )
      {
         //m_pMessageQueue->vRespLock();
         tU8 *message = (tU8* )(m_pMessageQueue->mQueueHandlerIncReceiver.front());
         m_pMessageQueue->mQueueHandlerIncReceiver.pop();
         //m_pMessageQueue->vRespUnlock();

         ETG_TRACE_USR4(("DL State:%d", m_u8DlState));

         if( NULL != message )
         {
            //stop timer after receiving any positive or negative response
            if ( message[2] != MSG_NEG_RX && m_u8DlState != DL_RETRY_STATE)
            {
               vStopTimer();
            }

            switch( message[0] )
            {
               /* response from V850 */
            case DL_RESPONSE_RECEIVED:
               {
                  //work on the server response
                  bRetVal = bV850DownloadProcess( &message[2] , message[1]);

                  if(NULL != g_fCallback)
                  {
                     g_fCallback(u8Temp);
                  }
                  break;
               }

            default:
               {
                  break;
               }

            }//switch( message )

            if( false == bRetVal )
            {
               vDisplayNegativeResponse( &message[2] , message[1] );
               //close data file if opened
               vCloseBinaryFile();

               /* The Memory allocated is freed here */
               delete[] message;
               message = NULL;

               //Provided, if this function would finish only after DL is completed
               sem_post(&m_updateCtrlSem);

            }
            else
            {

               /* The Memory allocated is freed here */
               delete[] message;
               message = NULL;
            }
         }
      }

	}//if( m_evfdIncRx == evFd )

	
	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vDisplayNegativeResponse
* DESCRIPTION:	This function Display the negative response received from v850 and 
*					terminate the program.
* PARAMETER:
*					tU8 *pcMessage,
*					tU32 u32MsgSize	
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					18.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vDisplayNegativeResponse( tU8 *pcMessage, tU32 u32MsgSize )
{
	tChar acRecvMsgBuffer[DEFAULT_MSGSZ+1];

	if( NULL != pcMessage && NULL != m_pclV850MessageFactory )    
	{
		memset( acRecvMsgBuffer, 0, (DEFAULT_MSGSZ+1) );
		memcpy( acRecvMsgBuffer, pcMessage, u32MsgSize );
		
		m_pclV850MessageFactory->vHexDump( "RECE", acRecvMsgBuffer , u32MsgSize );
		INFO_MSG (INFO_LEVEL_1, "ERROR: Negative Response from V850\n");
		UpdateLog::vUpdateLogTrace("%s Negative Response from V850", DL_LOG_ERROR );

		//Error Case is handled for Security Access
		if(m_u8ProcessingClientMsg !=  DL_REQUEST_SECURITY_ACCESS)
			vSetErrorCode(acRecvMsgBuffer[2]);

	}

}


/*******************************************************************************************
* FUNCTION:		bV850DownloadProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pcMessage,
*					tU32 u32MsgSize	
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bV850DownloadProcess( tU8 *pcMessage, tU32 u32MsgSize )
{
	
	tCString method = "bV850DownloadProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	tBool bRetVal = false;
	tU8 u8ResponceCode = 0;
	tChar acRecvMsgBuffer[DEFAULT_MSGSZ+1];

	if ((NULL != pcMessage ) && (u32MsgSize <= (DEFAULT_MSGSZ+1)))
	{
		bRetVal = true;

		memset( acRecvMsgBuffer, 0, (DEFAULT_MSGSZ+1) );
		memcpy( acRecvMsgBuffer, pcMessage, u32MsgSize );

		if ( NULL != m_pclV850MessageFactory )
		{
			m_pclV850MessageFactory->vHexDump( "RECE", acRecvMsgBuffer , u32MsgSize );
		}	

		//Get message code
		u8ResponceCode = acRecvMsgBuffer[0];
		ETG_TRACE_USR4(("DNL: Response code %x", u8ResponceCode));

		//Check message code
		switch ( u8ResponceCode )
		{
		case MSG_RX_DIAG_SESS_CTRL:
			{
				//Diagnostic Session Control functionality
				vDiagControlProcess( acRecvMsgBuffer );
				break;
			}
		case MSG_RX_READ_DATA_IDENTI:
			{
				//Read Data By Identifier functionality
				vReadDataByIdentifierProcess( acRecvMsgBuffer, u32MsgSize );
				
				break;
			}
		case MSG_RX_WRITE_DATA_IDENTI:
			{
				//Write Data By Identifier functionality
				vWriteDataByIdentifierProcess( acRecvMsgBuffer );
				break;
			}
		case MSG_RX_SECURITY_ACCESS:
			{
				//Security Access functionality
				vSecurityAccessProcess( acRecvMsgBuffer );
				break;
			}
		case MSG_RX_REQUEST_TRANSF_EXIT:
			{
				vTransitReponse();
				break;
			}
		case MSG_NEG_RX:
				{
					//Negative Response received from V850
					bRetVal = bHandleNegativeResponse( acRecvMsgBuffer );
					break;
				}
		case MSG_RX_ECU_RESET:
			{
				if(m_u8DlState == DL_SUCCESS_BL_STATE)
				{
					m_u8DlState = DL_RETRY_STATE;
					m_eRetryAfter = EN_RETRY_AFTER_RESET;
					sleep(BLTOAPP_ACTIVESESSION_TIME);
					INFO_MSG (INFO_LEVEL_1, "RESET Response Received\n");
					UpdateLog::vUpdateLogTrace("%s RESET Response Received", DL_LOG_INFO);
					vSetTimeout(BLTOAPP_ACTIVESESSION_TIME);
					bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );

					m_u32RetryCount = 1;
					INFO_MSG (INFO_LEVEL_1, "Retrycount:%d\n", m_u32RetryCount);
					UpdateLog::vUpdateLogTrace("%s Retrycount:%d", DL_LOG_INFO,  m_u32RetryCount);
				}

				break;
			}

		default:
			{
				//other functionality
				vRoutineAndRequestProcess( acRecvMsgBuffer );
				break;
			}

		}
	}
	
	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;

}

/*******************************************************************************************
* FUNCTION:		bStartV850Download
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tCString CStrFilePath,
*					tU32 u32MemoryAddress			
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bStartV850Download( )
{
	tCString method = "bStartV850Download";
	ETG_TRACE_USR4 (("%s was entered", method));

	tBool bRetVal = false;	

	if ( (NULL != m_szFileName) && (NULL != m_szFilePath) )
	{
		/* assign download state to varifying state */
		m_u8DlState = DL_VERIFY_STATE;

		/* assign the block ID */
		vAssignBlockID();
		
		/* open binary file */
		m_pFile = fopen(m_szFilePath, "r");

		if ( NULL != m_pFile && ((bRetVal = bInitDnlFile()) == true) )
		{
			/* assign download state to sequence setup state */
			m_u8DlState = DL_SEQ_SETUP_STATE;

			/* start download sequence */
			bStartDnlSequence();
			
		}
		else
		{
			UpdateLog::vUpdateLogTrace("%s DNL: file open error", DL_LOG_ERROR );
			ETG_TRACE_USR4(("DNL: file open error"));
			perror("ERROR: Download file open error\n");
			vSendPacketError();
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;
}

/*******************************************************************************************
* FUNCTION:		vAssignBlockID
* DESCRIPTION:	This function assign the block ID based on binary name 
* PARAMETER:
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					20.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vAssignBlockID( )
{
	if ( NULL != m_szFileName )
	{
		if ( strcmp(m_szFileName, "-bl") == 0 )
		{
			m_bIsBLUpdate = true;
			m_u16BlockID = BLU_BLOCKID;
			UpdateLog::vUpdateLogTrace("%s Updating %s",m_szFileName, DL_LOG_INFO );
			INFO_MSG (INFO_LEVEL_1, "INFO: Updating %s\n", m_szFileName);
			
		}
		else
		{
			m_bIsBLUpdate = false;
			m_u16BlockID = APP_BLOCKID;
			UpdateLog::vUpdateLogTrace("%s Updating %s",m_szFileName, DL_LOG_INFO );
			INFO_MSG (INFO_LEVEL_1, "INFO: Updating %s\n", m_szFileName);
			
		}
	}
}

/*******************************************************************************************
* FUNCTION:		bInitDnlFile
* DESCRIPTION:	This function initilized the .dnl file and get pointer of the file
* PARAMETER:
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					20.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bInitDnlFile( )
{
	tBool bRetVal = false;
	m_bIsDnlFile = bIsFileDnlFormat( m_szFilePath );

	if ( (true == m_bIsDnlFile) && (NULL != m_pclV850MessageFactory) )
	{

		if ((bRetVal = static_cast <tBool> (m_pclV850MessageFactory->DFInfo_ulwInit(m_pFile, &m_DFInfo)) != 1))
		{  
			vCloseBinaryFile();
			return bRetVal;                    /* return error, if wrong dataformat */
		}

		if( !(m_pclV850MessageFactory->iReadBinaryDataPtr(m_pFile, &m_DFInfo, m_u16BlockID)) )
		{
			UpdateLog::vUpdateLogTrace("%s BlockID not found or SEEK error",m_szFileName, DL_LOG_ERROR );
			INFO_MSG (INFO_LEVEL_1, "ERROR: BlockID not found or SEEK error\n");
			return false;									/* seek error or block ID not found in DNL-File */

		}
	}

	bRetVal = true;

	return bRetVal;

}

/*******************************************************************************************
* FUNCTION:		bStartDnlSequence
* DESCRIPTION:	This function start the download sequence
* PARAMETER:
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					20.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bStartDnlSequence( )
{
	tBool bRetVal = false;
	UpdateLog::vUpdateLogTrace("%s Starting Download Sequence", DL_LOG_INFO );
	if ( false == m_bIsBLMode )
	{
		UpdateLog::vUpdateLogTrace("%s Sending INC command to switch from application to bootloader mode", DL_LOG_INFO );
		vSetTimeout(BLTODEF_TIME);
		m_bIsBLCmdSend = true;
		bRetVal = bWriteToInc( MSG_TX_DIAG_SESS_CTRL_BTL );
		vDoDesiredOperations(EN_SCREENREINIT_OPERATION);
	}
	else
	{
		//If Switch to BL is happened already
		INFO_MSG (INFO_LEVEL_1, "INFO: V850 is in BL Mode\n");
		UpdateLog::vUpdateLogTrace("%s Sending INC command to switch to Extended session", DL_LOG_INFO );
		bWriteToInc( MSG_TX_DIAG_SESS_CTRL_EXT );
	}

	return bRetVal;

}

/*******************************************************************************************
* FUNCTION:		vWaitForBootloaderMode
* DESCRIPTION:	This function wait till v850 starts in bootloader mode.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					14.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vWaitForBootloaderMode( void )
{
	while(!m_bIsBLMode)
	{
		vDelaySimulation();
		//Read Active Session State
		INFO_MSG (INFO_LEVEL_2, "INFO: Reading Active Session State\n");
		UpdateLog::vUpdateLogTrace("%s Reading Active Session State", DL_LOG_INFO );
		bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );		
	}

	INFO_MSG (INFO_LEVEL_1, "INFO: Bootloader Mode Active\n");

}
/*******************************************************************************************
* FUNCTION:		bStopV850Download
* DESCRIPTION:	This function stop download communication.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bStopV850Download( )
{
	tCString method = "bStopV850Download";
	ETG_TRACE_USR4 (("%s was entered", method));	
	tBool bRetVal = true;		
    //ECU RESET
	UpdateLog::vUpdateLogTrace("%s Sending INC Command to reset V850", DL_LOG_INFO );
	bWriteToInc( MSG_TX_ECU_RESET );

	// Application invoked with -noreset option, switch back to BL_Mode
	if (false == m_bResetEnabled )
	{
		sleep(1);
		INFO_MSG (INFO_LEVEL_1, "INFO: Switching back to BL_Mode\n");
		UpdateLog::vUpdateLogTrace("%s Switching back to BL_Mode", DL_LOG_INFO );
		bWriteToInc( MSG_TX_DIAG_SESS_CTRL_BTL);
	}

	if ( DL_SUCCESS_STATE == m_u8DlState )
	{
		INFO_MSG (INFO_LEVEL_1, "INFO: V850 Download Success\n");
		UpdateLog::vUpdateLogTrace("%s V850 Download Success!", DL_LOG_INFO );

	}
	else if ( DL_INIT_STATE == m_u8DlState )
	{
		//INFO_MSG (INFO_LEVEL_1, "INFO: V850 Reset\n");

	}
	else
	{
		INFO_MSG (INFO_LEVEL_1, "ERROR: V850 Download Failed\n");
		UpdateLog::vUpdateLogTrace("%s V850 Download Failed!", DL_LOG_ERROR );
	}
	
	vDoDesiredOperations(EN_SCREENREINIT_OPERATION);
	vTerminateProgram();

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;
}

/*******************************************************************************************
* FUNCTION:		vTerminateProgram
* DESCRIPTION:	This function terminates the communication.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					13.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vTerminateProgram( void )
{
	UpdateLog::vUpdateLogTrace("%s Terminating V850 App", DL_LOG_INFO );
	//exit of receiver thread			
	if ( NULL != m_pclIncReceiver )
	{
		UpdateLog::vUpdateLogTrace("%s Exiting Reciever Thread", DL_LOG_INFO );
		m_pclIncReceiver->vSetReceiverThreadStatus( false );
	}

	vDelaySimulation();

	//close data file
	vCloseBinaryFile();

	//vMessageTermination();

	//Provided, if this function would finish only after DL is completed
	sem_post(&m_updateCtrlSem);
}
/*******************************************************************************************
* FUNCTION:		vCloseBinaryFile
* DESCRIPTION:	This function closed the binary file if opened.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					24.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vCloseBinaryFile( void )
{
	//close data file
	if (m_pFile != NULL) 
	{
		UpdateLog::vUpdateLogTrace("%s Closing the Binary File", DL_LOG_INFO );
		fclose(m_pFile);
		m_pFile = NULL;
	}

}

/*******************************************************************************************
* FUNCTION:		bWriteToInc
* DESCRIPTION:	write command to Inc driver
*					
* PARAMETER:
*					tCString string //command string
*				
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bWriteToInc( tCString sCmdString )
{
	tCString method = "bWriteToInc";
	ETG_TRACE_USR4 (("%s was entered", method));

	tChar buffer[DEFAULT_MSGSZ+1];
	tU32 u32Length = 0;
	tU32 u32WriteLength = 0;
	tBool bRetVal = false;

	//check for NULL pointer
	if ( ( dgram != NULL )
		&&
		( NULL != sCmdString ) 
		&&
		( NULL != m_pclV850MessageFactory ) 
		&&
		( NULL != m_pclIncSender ) )
	{

		//copy the command in command buffer
		std::memset( m_chCmdBuff, 0 , sizeof(m_chCmdBuff) );
		strncat(m_chCmdBuff, sCmdString, sizeof(m_chCmdBuff) - 1);//coverity fix for 17325
		
		vStopTimer();
		//pack the message
		u32Length = m_pclV850MessageFactory->u32PackCmdPacket( sCmdString,  buffer, DEFAULT_MSGSZ );
		//Write to inc
		u32WriteLength = m_pclIncSender->s32WriteCommandPacket( dgram, buffer, u32Length );

		//check byte to be written and byte wrote is same
		if ( u32WriteLength != u32Length)
		{
			vSendPacketError();
			return bRetVal;
		}
		else
		{
			bRetVal = true;
		}
		vStartTimer();

	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;

}

/*******************************************************************************************
* FUNCTION:		bWriteDownloadReqToInc
* DESCRIPTION:	write command to Inc driver
*					
* PARAMETER:
*					tCString string //command string
*				
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bWriteDownloadReqToInc( tCString sCmdString, tU16 Typ )
{
	tCString method = "bWriteDownloadReqToInc";
	ETG_TRACE_USR4 (("%s was entered", method));

	tChar buffer[DEFAULT_MSGSZ+1];
	tU32 u32FileLength = 0;
	tU32 u32PackLength = 0;
	tU32 u32WriteLength = 0;
	tBool bRetVal = false;
	tU16 u16BlkID = Typ;

	//check for NULL pointer
	if ( ( dgram != NULL )
		&&
		( NULL != sCmdString ) 
		&&
		( NULL != m_pclV850MessageFactory ) 
		&&
		( NULL != m_pclIncSender ) )
	{
		//get the file size
		if ( true == m_bIsDnlFile )
		{
			u32FileLength = static_cast <tU32>(m_pclV850MessageFactory->u32GetFileLen( &m_DFInfo, u16BlkID ) );
			m_u32MemoryAddress =static_cast <tU32> (m_pclV850MessageFactory->u32GetMemoryAddress( &m_DFInfo, u16BlkID ) );

			INFO_MSG (INFO_LEVEL_1, "INFO: BlockID %04X: length %08X start address %08X\n", u16BlkID, u32FileLength, m_u32MemoryAddress); 
		}
		else
		{
			u32FileLength = m_pclV850MessageFactory->u32GetFileSize( m_szFilePath );
			m_u32MemoryAddress = APP_MEM_ADDRESS;
		}

		//Calculate total number of blocks
		if ( u32FileLength != 0 )
		{
			m_u32BlockSize = u32FileLength / BLOCK_SIZE;

			m_u32LastChuckData = u32FileLength % BLOCK_SIZE;
		}

		//copy the command in command buffer
		std::memset( m_chCmdBuff, 0 , sizeof(m_chCmdBuff) );
		strncat(m_chCmdBuff, sCmdString, sizeof(m_chCmdBuff)-1);//coverity fix for 17326
		
		vStopTimer();
		//pack the message
		u32PackLength = m_pclV850MessageFactory->u32PackDownloadRequestPacket( sCmdString, buffer, (DEFAULT_MSGSZ+1), u32FileLength, m_u32MemoryAddress );

		//Write to inc
		u32WriteLength = m_pclIncSender->s32WriteCommandPacket( dgram, buffer, u32PackLength );

		if ( u32WriteLength != u32PackLength )
		{
			vSendPacketError();
			return bRetVal;
		}
		else
		{
			bRetVal = true;
		}

		vStartTimer();

	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;

}


/*******************************************************************************************
* FUNCTION:		vWriteBinaryDataToInc
* DESCRIPTION:	write command to Inc driver
*					
* PARAMETER:
*					tCString string //command string
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vWriteBinaryDataToInc( tU32 u32BytesToRead )
{
	tCString method = "vWriteBinaryDataToInc";
	ETG_TRACE_USR4 (("%s was entered", method));

	char acBinaryBuffer[DATA_TRANSFER_SIZE+1];
	memset( acBinaryBuffer, 0xFF, (DATA_TRANSFER_SIZE+1) );
	memset( m_chCmdBuff, 0 , (DEFAULT_MSGSZ+1) );

	//pack the message code at 0th index of buffer
	acBinaryBuffer[0] = MSG_TX_TRANSF_DATA;

	//pack the block number at 1th index of buffer
	acBinaryBuffer[1] =static_cast<char>(m_u32RBlockSize);

	//check for NULL pointer
	if ( (sockfd != 0)
		&&
		( NULL != m_pclV850MessageFactory )
		&&
		( NULL != m_pclIncSender ) )
	{
		strcpy(m_chCmdBuff, MSG_TRANSF_DATA);
		vStopTimer();
		//read the data and pack it from 2nd index of buffer 
		if ( u32BytesToRead != m_pclV850MessageFactory->u32ReadBinaryFile( m_pFile, &acBinaryBuffer[2] , u32BytesToRead ) )
		{
			vSendPacketError();
		}

		//Write to inc
		if ( false == m_pclIncSender->bWriteDataPacket( sockfd, acBinaryBuffer, DATA_TRANSFER_SIZE ))
		{
			vSendPacketError();
		}
		else
		{
			vStartTimer();
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}


/*******************************************************************************************
* FUNCTION:		bHandleNegativeResponse
* DESCRIPTION:	This function handle the negative response from V850.
* PARAMETER:
*					 tString pacRecvMsgBuffer
*				
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
*------------------------------------------------------------------------------
* 19-02-2016	| 0.2				| vri7cob	| Changes For CMG3G-9596
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bHandleNegativeResponse( tString pacRecvMsgBuffer )
{
	tCString method = "bHandleNegativeResponse";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU8 u8ResponceCode = 0;
	tBool bRetVal = false;

	if ( NULL != pacRecvMsgBuffer )
	{
		//Get message code
		u8ResponceCode = pacRecvMsgBuffer[1];		
		switch ( u8ResponceCode )
		{
		case MSG_NEG_RX_INVALID_FORMAT:
		case MSG_NEG_RX_NOT_SUPPORT:
		case MSG_NEG_RX_LONG_RESP:
		case MSG_NEG_RX_COND_NOT_CORRECT:
		case MSG_NEG_RX_REQ_SEQ_ERROR:
		case MSG_NEG_RX_ACCESS_DENIED:
		case MSG_NEG_RX_INVALID_KEY:
		case MSG_NEG_RX_EXCEED_ATTEMPTS:
		case MSG_NEG_RX_TIMEDELAY_NOT_EXP:
		case MSG_NEG_RX_UPLOAD_DL_NOT_ACCEPT:
		case MSG_NEG_RX_TRANS_DATA_SUSPEND:
		case MSG_NEG_RX_PROG_FAILURE:
		case MSG_NEG_RX_SEQ_COUNTER:
		case MSG_NEG_RX_VOLTAGE_ERROR1:
		case MSG_NEG_RX_VOLTAGE_ERROR2:
			{
				if(pacRecvMsgBuffer[2] == MSG_NEG_RX_REQ_OUT_OF_RANGE)
					bRetVal = bHandleSignRes( );
				break;
			}
		case MSG_NEG_RX_JTAG_ERROR1:
			if(pacRecvMsgBuffer[2] == MSG_NEG_RX_INVALID_KEY)
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: JTAG FingerPrint Failed\n");
				m_eJTAGValidationResult = E_JTAG_VALIDATION_FAILURE;
			}
			else if(pacRecvMsgBuffer[2] == MSG_NEG_RX_REQ_OUT_OF_RANGE)
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: JTAG FingerPrint Not Implemented\n");
				m_eJTAGValidationResult = E_JTAG_VALIDATION_NOTIMPLEMENTED;
				bRetVal = bHandleSignRes( );
			}
			if(m_u8ProcessingClientMsg == DL_DO_JTAGFINGERPRINT_VALIDATION)
			{
				vStopTimer();
				bRetVal = true;
				sem_post(&m_updateCtrlSem);
			}
			break;
		default:
			{
				bRetVal = true;				
				break;
			}

		}
	}

	ETG_TRACE_USR4 (("%s was left", method));
	return bRetVal;	

}


/*******************************************************************************************
* FUNCTION:		vDiagControlProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*				vDiagControlProcess
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
*					24-06-2015 VRI7COB (RBEI)
*					BL Response is Handled.
*					25-09-2015 VRI7COB (RBEI)
*					m_bIsBLCmdSend is reset when 10-60 command is responded.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vDiagControlProcess( tString pacRecvMsgBuffer )
{
	tCString method = "vDiagControlProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		//Get message code
		u8ResponceCode = pacRecvMsgBuffer[1];

		ETG_TRACE_USR4(("DNL: Response Sub code %d", u8ResponceCode));

		switch ( u8ResponceCode )
		{
		case MSG_RX_DIAG_SESS_CTRL_DEF:
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: Normal session Active\n");
				UpdateLog::vUpdateLogTrace("%s Normal session Active", DL_LOG_INFO );
				//switch to extended session
				bWriteToInc( MSG_TX_DIAG_SESS_CTRL_EXT );
				break;
			}

		case MSG_RX_DIAG_SESS_CTRL_PRG:
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: Programming session Active\n");
				UpdateLog::vUpdateLogTrace("%s Programming session Active", DL_LOG_INFO );
				//verify Programming Session
				UpdateLog::vUpdateLogTrace("%s Verifying Programming session ", DL_LOG_INFO );
				bWriteToInc( MSG_TX_READ_DATA_IDENTI_VER_PRG );
				break;
			}

		case MSG_RX_DIAG_SESS_CTRL_EXT:
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: Extended session Active\n");
				UpdateLog::vUpdateLogTrace("%s Extended session Active ", DL_LOG_INFO );
				//switch to programming session
				bWriteToInc( MSG_TX_DIAG_SESS_CTRL_PRG );
				break;
			}
		case MSG_RX_DIAG_SESS_CTRL_BTL:
         {
        	 INFO_MSG (INFO_LEVEL_1, "INFO: Response For 10-60\n");
            INFO_MSG (INFO_LEVEL_1, "INFO: Switch to BL Mode\n");
            m_bIsBLCmdSend = false;
            UpdateLog::vUpdateLogTrace("%s Switch to BL Mode ", DL_LOG_INFO );

            //Don't Retry when Download completed
            if(m_u8DlState != DL_SUCCESS_STATE)
            {
            	sleep(BLTOAPP_ACTIVESESSION_TIME);
				m_u8DlState = DL_RETRY_STATE;
				m_eRetryAfter = EN_RETRY_AFTER_SWITCHBL;
				vSetTimeout(BLTOAPP_ACTIVESESSION_TIME);
				bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );
				m_u32RetryCount = 1;
				INFO_MSG (INFO_LEVEL_1, "Retrycount:%d\n", m_u32RetryCount);
            }
         }
		   break;

		default:
			break;
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vReadDataByIdentifierProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*					tU32 u32MsgSize	
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vReadDataByIdentifierProcess( tString pacRecvMsgBuffer, tU32 u32MsgSize )
{
	tCString method = "vReadDataByIdentifierProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	
	if ( NULL != pacRecvMsgBuffer )
	{
		
		if( (MSG_RX_READ_DATA_IDENTI_STATE1 == pacRecvMsgBuffer[1]) 
			&& 
			(MSG_RX_READ_DATA_IDENTI_STATE2 == pacRecvMsgBuffer[2]) )
		{
			if(m_u8DlState == DL_RETRY_STATE && m_eRetryAfter == EN_RETRY_AFTER_RESET)
			{
				m_eRetryAfter = EN_RETRY_AFTER_INVALID;
				INFO_MSG (INFO_LEVEL_1, "Active Session Response Recv: Start App Update\n");
				UpdateLog::vUpdateLogTrace("%s Active Session Response Recv: Start App Update", DL_LOG_INFO );
				m_u32RetryCount = 0;
				vStopTimer();
				bStartV850Download();

			}
			else
				vHandleActiveSessionResponse(&pacRecvMsgBuffer[3]);


		}
		else if( (MSG_RX_READ_DATA_IDENTI_VER_PRG1 == pacRecvMsgBuffer[1]) 
			&&
			(MSG_RX_READ_DATA_IDENTI_VER_PRG2 == pacRecvMsgBuffer[2])
			&&
			(MSG_RX_READ_DATA_IDENTI_VER_PRG3 == pacRecvMsgBuffer[3]) )
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: Programming session verified\n");
			UpdateLog::vUpdateLogTrace("%s Programming session verified", DL_LOG_INFO );
			//request seed
			bWriteToInc( MSG_TX_SECURITY_ACCESS_SEED );
		}
		else
		{
			vReadDataByIdentifierProcessExtended(pacRecvMsgBuffer,u32MsgSize);
		}


	}
	
	ETG_TRACE_USR4 (("%s was left", method));

}
/*******************************************************************************************
* FUNCTION:		vReadDataByIdentifierProcessExtended
* DESCRIPTION:	This function start a download communication to flash V850 Chip(This function written to improve maintainability)
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*					tU32 u32MsgSize	
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vReadDataByIdentifierProcessExtended( tString pacRecvMsgBuffer, tU32 u32MsgSize )
{

	if( (MSG_RX_READ_DATA_IDENTI_CRC_AND_VERSION == pacRecvMsgBuffer[1])
			&&
			( (MSG_RX_READ_DATA_IDENTI_BM_BUILD_USER == pacRecvMsgBuffer[2])
			||
			(MSG_RX_READ_DATA_IDENTI_BM_BUILD_USER < pacRecvMsgBuffer[2]
				&&
          MSG_RX_READ_DATA_IDENTI_APP_ID_STRING > pacRecvMsgBuffer[2])
			||
			(MSG_RX_READ_DATA_IDENTI_APP_ID_STRING == pacRecvMsgBuffer[2]) ) )
	{
		vHandleVersionResponse(&pacRecvMsgBuffer[2], u32MsgSize);
	}
	else if( (MSG_RX_READ_DATA_IDENTI_CRC_AND_VERSION == pacRecvMsgBuffer[1])
			&&
			( (MSG_RX_READ_DATA_IDENTI_APP_CRC == pacRecvMsgBuffer[2])
			||
			(MSG_RX_READ_DATA_IDENTI_BM_CRC == pacRecvMsgBuffer[2])
			||
			(MSG_RX_READ_DATA_IDENTI_BL_CRC == pacRecvMsgBuffer[2]) ) )
	{
		vHandleCrcResponse(&pacRecvMsgBuffer[2]);
	}
	else if( (MSG_RX_READ_SECURITY_DATA == pacRecvMsgBuffer[1]) && (MSG_RX_READ_SECURITY_DATA_BYTE == pacRecvMsgBuffer[2]) ) 
	{
		vHandleSecurityResponse(&pacRecvMsgBuffer[3]);
	}
	else
	{			
		//do nothing
		//printf("do nothing %s\n", __func__);
		bStopV850DownloadForSec();
	}
}
/*******************************************************************************************
* FUNCTION:		vHandleVersionResponse
* DESCRIPTION:	This function handles the response of Version read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*					tU32 u32MsgSize	
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleVersionResponse(tString pacRecvMsgBuffer, tU32 u32MsgSize)
{
	tCString method = "vResponseForVersionRead";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		if ( u8ResponceCode == MSG_RX_READ_DATA_IDENTI_BM_BUILD_USER
			||
			( u8ResponceCode > MSG_RX_READ_DATA_IDENTI_BM_BUILD_USER
			&&
			u8ResponceCode < MSG_RX_READ_DATA_IDENTI_BM_ID_STRING )
			||
			u8ResponceCode == MSG_RX_READ_DATA_IDENTI_BM_ID_STRING )
		{
			vHandleBMVersionResponse(pacRecvMsgBuffer, u32MsgSize);

		}
		else if ( u8ResponceCode == MSG_RX_READ_DATA_IDENTI_BL_BUILD_USER
			||
			( u8ResponceCode > MSG_RX_READ_DATA_IDENTI_BL_BUILD_USER
			&&
			u8ResponceCode < MSG_RX_READ_DATA_IDENTI_BL_ID_STRING )
			||
			u8ResponceCode == MSG_RX_READ_DATA_IDENTI_BL_ID_STRING )
		{
			vHandleBLVersionResponse(pacRecvMsgBuffer, u32MsgSize);

		}
		else if ( u8ResponceCode == MSG_RX_READ_DATA_IDENTI_APP_BUILD_USER
			||
			( u8ResponceCode > MSG_RX_READ_DATA_IDENTI_APP_BUILD_USER
			&&
			u8ResponceCode < MSG_RX_READ_DATA_IDENTI_APP_ID_STRING )
			||
			u8ResponceCode == MSG_RX_READ_DATA_IDENTI_APP_ID_STRING )
		{
			vHandleAppVersionResponse(pacRecvMsgBuffer, u32MsgSize);

		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vHandleBMVersionResponse
* DESCRIPTION:	This function handles the response of boot manager version read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*              tU32 u32MsgSize
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.02.2014	Aditya Kumar Jha
*					Initial Version.
*                   25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleBMVersionResponse(tString pacRecvMsgBuffer, tU32 u32MsgSize)
{
	tCString method = "vHandleBMVersionResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;
	
	
	tBool bIsVerOK = false;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		/* Convert it into string */
		
		

		switch ( u8ResponceCode )
		{
		case MSG_RX_READ_DATA_IDENTI_BM_BUILD_USER:
		case MSG_RX_READ_DATA_IDENTI_BM_TIMESTAMP:
		case MSG_RX_READ_DATA_IDENTI_BM_BOSCH_NUMBER:
		case MSG_RX_READ_DATA_IDENTI_BM_OEM_NUMBER:
		case MSG_RX_READ_DATA_IDENTI_BM_USER1:
		case MSG_RX_READ_DATA_IDENTI_BM_USER2:
		case MSG_RX_READ_DATA_IDENTI_BM_FGIU_ID:
		case MSG_RX_READ_DATA_IDENTI_BM_ID_STRING:
			{
				break;
			}
		case MSG_RX_READ_DATA_IDENTI_BM_LABEL:
			{
				UpdateLog::vUpdateLogTrace("%s Response for BM LABEL recieved", DL_LOG_INFO );
				m_chSWLabel[0] = '\n';
				vConvertIntArrayToString( m_chSWLabel, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(BM_LABEL_KEY, m_chSWLabel);
				readFromRegistry(BM_LABEL_KEY);
				sem_post(&m_updateCtrlSem);

				break;
			}
		case MSG_RX_READ_DATA_IDENTI_BM_VERSION:
			{
				UpdateLog::vUpdateLogTrace("%s Response for BM VERSION recieved", DL_LOG_INFO );
				bIsVerOK = true;
				m_chSWVersion[0] = '\n';
				vConvertIntArrayToString( m_chSWVersion, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(BM_VERSION_KEY, m_chSWVersion);
				readFromRegistry(BM_VERSION_KEY);
				sem_post(&m_updateCtrlSem);
				
				break;
			}
		default:
			bIsVerOK = false;
			break;
		}

	}

//	if ( true == bIsVerOK )
//	{
//		// we send INC message to raed the Label Info.
//		UpdateLog::vUpdateLogTrace("%s Sending INC command to read BM LABEL", DL_LOG_INFO );
//		bWriteToInc( MSG_TX_READ_DATA_IDENTI_BM_LABEL );
//
//	}


	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vHandleBLVersionResponse
* DESCRIPTION:	This function handles the response of boot loader version read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*              tU32 u32MsgSize
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.02.2014	Aditya Kumar Jha
*					Initial Version.
*                   25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleBLVersionResponse(tString pacRecvMsgBuffer, tU32 u32MsgSize)
{
	tCString method = "vHandleBLVersionResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;
	
	tBool bIsVerOK   = false;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		/* Convert it into string */
		
		

		switch ( u8ResponceCode )
		{
		case MSG_RX_READ_DATA_IDENTI_BL_BUILD_USER:
		case MSG_RX_READ_DATA_IDENTI_BL_TIMESTAMP:
		case MSG_RX_READ_DATA_IDENTI_BL_BOSCH_NUMBER:
		case MSG_RX_READ_DATA_IDENTI_BL_OEM_NUMBER:
		case MSG_RX_READ_DATA_IDENTI_BL_USER1:
		case MSG_RX_READ_DATA_IDENTI_BL_USER2:
		case MSG_RX_READ_DATA_IDENTI_BL_FGIU_ID:
		case MSG_RX_READ_DATA_IDENTI_BL_ID_STRING:
			{
				break;
			}
		case MSG_RX_READ_DATA_IDENTI_BL_LABEL:
			{
				UpdateLog::vUpdateLogTrace("%s Response for BL LABEL recieved", DL_LOG_INFO );
                m_chSWLabel[0] = '\n';
				vConvertIntArrayToString( m_chSWLabel, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(BL_LABEL_KEY, m_chSWLabel);
				readFromRegistry(BL_LABEL_KEY);
				sem_post(&m_updateCtrlSem);
				break;
			}
		case MSG_RX_READ_DATA_IDENTI_BL_VERSION:
			{
				UpdateLog::vUpdateLogTrace("%s Response for BL VERSION recieved", DL_LOG_INFO );
				bIsVerOK = true;
				m_chSWVersion[0] = '\n';
				vConvertIntArrayToString( m_chSWVersion, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(BL_VERSION_KEY, m_chSWVersion);
				readFromRegistry(BL_VERSION_KEY);
				sem_post(&m_updateCtrlSem);
				break;
			}
		default:
			bIsVerOK = false;
			break;
		}
	}

//	if (true == bIsVerOK )
//	{
//		UpdateLog::vUpdateLogTrace("%s Sending INC command to read BL LABEL ", DL_LOG_INFO );
//		bWriteToInc( MSG_TX_READ_DATA_IDENTI_BL_LABEL );
//
//	}

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vHandleAppVersionResponse
* DESCRIPTION:	This function handles the response of appliaction version read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*              tU32 u32MsgSize
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.02.2014	Aditya Kumar Jha
*					Initial Version.
*                   25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleAppVersionResponse(tString pacRecvMsgBuffer, tU32 u32MsgSize)
{
	tCString method = "vHandleAppVersionResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;
	
	tBool bIsVerOK   = false;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		/* Convert it into string */
		
		

		switch ( u8ResponceCode )
		{
		case MSG_RX_READ_DATA_IDENTI_APP_BUILD_USER:
		case MSG_RX_READ_DATA_IDENTI_APP_TIMESTAMP:
		case MSG_RX_READ_DATA_IDENTI_APP_BOSCH_NUMBER:
	    case MSG_RX_READ_DATA_IDENTI_APP_OEM_NUMBER:
		case MSG_RX_READ_DATA_IDENTI_APP_USER1:
		case MSG_RX_READ_DATA_IDENTI_APP_USER2:
		case MSG_RX_READ_DATA_IDENTI_APP_ID_STRING:
			{
				break;
			}
		case MSG_RX_READ_DATA_IDENTI_APP_LABEL:
			{
				UpdateLog::vUpdateLogTrace("%s Response recieved for APP LABEL ", DL_LOG_INFO );
                m_chSWLabel[0] = '\n';
				vConvertIntArrayToString( m_chSWLabel, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(APP_LABEL_KEY, m_chSWLabel);
				readFromRegistry(APP_LABEL_KEY);
				sem_post(&m_updateCtrlSem);
				break;
			}
		case MSG_RX_READ_DATA_IDENTI_APP_VERSION:
			{
				UpdateLog::vUpdateLogTrace("%s Response recieved for APP VERSION ", DL_LOG_INFO );
				bIsVerOK = true;
				m_chSWVersion[0] = '\n';
				vConvertIntArrayToString( m_chSWVersion, &pacRecvMsgBuffer[1], u32MsgSize );
				writeToRegistry(APP_VERSION_KEY, m_chSWVersion);
				readFromRegistry(APP_VERSION_KEY);
				sem_post(&m_updateCtrlSem);
				break;
			}
		default:
			bIsVerOK = false;
			break;
		}

	}

//	if ( true ==  bIsVerOK )
//	{
//		UpdateLog::vUpdateLogTrace("%s Sending INC command to read APP LABEL ", DL_LOG_INFO );
//		//we send INC message to read the label info
//		bWriteToInc( MSG_TX_READ_DATA_IDENTI_APP_LABEL );
//	}

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vHandleActiveSessionResponse
* DESCRIPTION:	This function handles the response of active session request.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					18.03.2014	Aditya Kumar Jha
*					Initial Version.
*------------------------------------------------------------------------------
* 19-02-2016	| 0.2				| vri7cob	| Changes For CMG3G-9596
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleActiveSessionResponse(tString pacRecvMsgBuffer)
{
	tCString method = "vHandleActiveSessionResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		switch ( u8ResponceCode )
		{
		/* Check if it is in Default state */
		case MSG_RX_READ_DATA_IDENTI_DEF_STATE:
			{
				if ( (MSG_RX_READ_DATA_IDENTI_DS_STATE == pacRecvMsgBuffer[1])
					&&
					((DL_INIT_STATE == m_u8DlState) || (DL_RETRY_STATE == m_u8DlState) ))
				{
					m_bIsBLMode = true;
					INFO_MSG (INFO_LEVEL_1, "INFO: V850 is in BL_Mode@def-session\n");
					UpdateLog::vUpdateLogTrace("%s V850 is in BL_Mode@def-session", DL_LOG_INFO );

					if(m_eRetryAfter == EN_RETRY_AFTER_SWITCHBL && m_u8DlState == DL_RETRY_STATE)
					{
						m_u8DlState = DL_INIT_STATE;
						m_eRetryAfter = EN_RETRY_AFTER_INVALID;
						m_u32RetryCount = 0;
						vStopTimer();
						UpdateLog::vUpdateLogTrace("%s Sending INC command to switch to default session", DL_LOG_INFO );
						INFO_MSG (INFO_LEVEL_1, "INFO: Switch to Default Session\n");
						bWriteToInc( MSG_TX_DIAG_SESS_CTRL_DEF );
					}
					else
						sem_post(&m_updateCtrlSem);

				}
				else if ( (MSG_RX_READ_DATA_IDENTI_DS_STATE_APP == pacRecvMsgBuffer[1])
					&&
					((DL_INIT_STATE == m_u8DlState) || (DL_RETRY_STATE == m_u8DlState)))
				{
					m_bIsBLMode = false;
					INFO_MSG (INFO_LEVEL_1, "INFO: V850 is in Application mode\n");
					UpdateLog::vUpdateLogTrace("%s V850 is in Application mode", DL_LOG_INFO );

					if(m_eRetryAfter == EN_RETRY_AFTER_SWITCHBL && m_u8DlState == DL_RETRY_STATE)
					{
						if(m_u32RetryCount < BLTOAPP_RETRY_COUNT)
						{
							vSetTimeout(BLTOAPP_ACTIVESESSION_TIME);
							bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );
							m_u32RetryCount++;
							INFO_MSG (INFO_LEVEL_1, "INFO: RetryCount:%d\n", m_u32RetryCount);
						}
						else
						{
							INFO_MSG (INFO_LEVEL_1, "ERROR: V850 is not switched to BL After 8 Retries\n");
							UpdateLog::vUpdateLogTrace("%s V850 is not switched to BL After 8 Retries", DL_LOG_INFO );
							sem_post(&m_updateCtrlSem);
						}
					}
					else
						sem_post(&m_updateCtrlSem);
				}
				break;
			}

	
		/* default case */
		default: vHandleActiveSessionResponseExtended(pacRecvMsgBuffer);
			break;
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vHandleActiveSessionResponseExtended
* DESCRIPTION:	This function handles the response of active session request extended from
                vHandleActiveSessionResponse to solve maintability issue.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					18.03.2014	Aditya Kumar Jha
*                   08.12.2014	Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleActiveSessionResponseExtended(tString pacRecvMsgBuffer)
{
	tCString method = "vHandleActiveSessionResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		switch ( u8ResponceCode )
		{
		
		/* Check if it is in programming state */
		case MSG_RX_READ_DATA_IDENTI_PRG_STATE:
			{
				if(MSG_RX_READ_DATA_IDENTI_PS_STATE_UNLOCKED == pacRecvMsgBuffer[1])
				{
					//Write Fingerprint					
					m_bIsBLProgMode = true;
					if(true == m_bIsBLMode && true == m_bUpdateReqFlag)
					{
						INFO_MSG (INFO_LEVEL_1, "INFO: Write FingerPrint..\n");	
						UpdateLog::vUpdateLogTrace("%s Write FingerPrint..", DL_LOG_INFO );
						bWriteToInc( MSG_TX_WRITE_DATA_IDENTI_FPRINT );
					}
					else if (((true == m_bSecFlag)|| (true == m_bJtagFlag)||(true == m_bResetFlag)))
					{
						INFO_MSG (INFO_LEVEL_1, "INFO: V850 is in BL Programming Mode \n");
						sem_post(&m_updateCtrlSem);
					}  
					else
					{					
					   INFO_MSG (INFO_LEVEL_1, "WARN: Reset Required after Diagnostic session \n");
					   UpdateLog::vUpdateLogTrace("%s Reset Required after Diagnostic session ", DL_LOG_SERVICE );
					   sem_post(&m_updateCtrlSem);
					   vTerminateProgram();
					}

				}
				else if( DL_INIT_STATE == m_u8DlState )
				{
					m_bIsBLMode = true;
					INFO_MSG (INFO_LEVEL_1, "INFO: V850 is in BL_Mode@prog-session\n");
					UpdateLog::vUpdateLogTrace("%s V850 is in BL_Mode@prog-session", DL_LOG_INFO );
					sem_post(&m_updateCtrlSem);
				}
				else
				{
					//do nothing
				}
				break;
			}

		/* default case */
		default:
			break;
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		
* DESCRIPTION:	This function handles the response of CRC read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleCrcResponse(tString pacRecvMsgBuffer)
{
	tCString method = "vResponseForCRCRead";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];

		switch ( u8ResponceCode )
		{
		/* message code is response of read BootManager CRC */
		case MSG_RX_READ_DATA_IDENTI_BM_CRC:
			{
				UpdateLog::vUpdateLogTrace("%s Response recieved for BM CRC", DL_LOG_INFO );
				vHandleBMCrcResponse(&pacRecvMsgBuffer[1]);
				break;
			}
		/* message code is response of read Bootloader CRC */
		case MSG_RX_READ_DATA_IDENTI_BL_CRC:
			{
				UpdateLog::vUpdateLogTrace("%s Response recieved for BL CRC", DL_LOG_INFO );
				vHandleBLCrcResponse(&pacRecvMsgBuffer[1]);
				break;
			}
		/* message code is response of read application CRC */
		case MSG_RX_READ_DATA_IDENTI_APP_CRC:
			{
				UpdateLog::vUpdateLogTrace("%s Response recieved for APP CRC", DL_LOG_INFO );
				vHandleAppCrcResponse(&pacRecvMsgBuffer[1]);				
				break;
			}
		/* default case */
		default:
			break;
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vHandleBMCrcResponse
* DESCRIPTION:	This function handles the response of boot manager CRC read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					23.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleBMCrcResponse(tString pacRecvMsgBuffer)
{
	tCString method = "vHandleBMCrcResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tBool bIsCrcOK   = false;

	bIsCrcOK = bReadCrcValue( pacRecvMsgBuffer );

	/* Only request for CRC check sum */
	if ( DL_INIT_STATE == m_u8DlState )
	{
		vTerminateProgram();
	}

	ETG_TRACE_USR4 (("%s was left", method));

}
/*******************************************************************************************
* FUNCTION:		vHandleBLCrcResponse
* DESCRIPTION:	This function handles the response of boot loader CRC read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					23.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleBLCrcResponse(tString pacRecvMsgBuffer)
{
	tCString method = "vHandleBLCrcResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tBool bIsCrcOK   = false;

	bIsCrcOK = bReadCrcValue( pacRecvMsgBuffer );

	/* Only request for CRC check sum */
	if ( DL_INIT_STATE == m_u8DlState )
	{
		vTerminateProgram();
	}
	/* bootloader CRC is ok then download application binary */
	else if( true == bIsCrcOK )
	{

		vAppUpdate();

	}
	/* bootloader CRC is not ok */
	else
	{
		m_u8DlState = DL_FAILED_STATE;
		bStopV850Download();
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vHandleAppCrcResponse
* DESCRIPTION:	This function handles the response of application CRC read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					23.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vHandleAppCrcResponse(tString pacRecvMsgBuffer)
{
	tCString method = "vHandleAppCrcResponse";
	ETG_TRACE_USR4 (("%s was entered", method));
	tBool bIsCrcOK   = false;

	bIsCrcOK = bReadCrcValue( pacRecvMsgBuffer );

	if ( DL_INIT_STATE == m_u8DlState )
	{
		vTerminateProgram();
	}
	else
	{
		if ( false == bIsCrcOK )
		{
			m_u8DlState = DL_FAILED_STATE;
		}
		else
		{
			m_u8DlState = DL_SUCCESS_STATE;
		}
		/* Application CRC is ok then change the state of download and send ECU reset command */
		if ( DL_FAILED_STATE == m_u8DlState || DL_SUCCESS_STATE == m_u8DlState )
		{					
			bStopV850Download();	
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vWriteDataByIdentifierProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
------------------------------------------------------------------------------
* 19-02-2016	| 0.2				| vri7cob	| Changes For CMG3G-9596
------------------------------------------------------------------------------
* 18-05-2016	| 0.3				| VRI7COB	| Posting the Semaphore For JTAG Validation Process.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vWriteDataByIdentifierProcess(tString pacRecvMsgBuffer )
{
	tCString method = "vWriteDataByIdentifierProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	if ( NULL != pacRecvMsgBuffer )
	{
		if( (MSG_RX_WRITE_DATA_IDENTI_FPRINT1 == pacRecvMsgBuffer[1]) 
			&& 
			(MSG_RX_WRITE_DATA_IDENTI_FPRINT2 == pacRecvMsgBuffer[2]) )
		{
			//Set time out value for erase command i.e., 30 secs
			UpdateLog::vUpdateLogTrace("%s Setting Erase Memory Time Out", DL_LOG_INFO );
			vSetTimeout( ERASE_TIMEOUT );

			//Erase memory
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for Erase Memory", DL_LOG_INFO );
			bWriteToInc( MSG_TX_ROUTINE_CTRL_ERASE_MEM );

		}
		if( (MSG_RX_WRITE_DATA_IDENTI_SIGN1 == pacRecvMsgBuffer[1]) 
			&& 
			(MSG_RX_WRITE_DATA_IDENTI_SIGN2 == pacRecvMsgBuffer[2]) )
		{

			m_bIsSignByteSend = false;

			//send transfer exit command
			UpdateLog::vUpdateLogTrace("%s Sending INC Command for Transfer Exit", DL_LOG_INFO );
			bWriteToInc( MSG_TX_REQUEST_TRANSF_EXIT );

		}

		if(pacRecvMsgBuffer[1] == MSG_RX_READ_SECURITY_DATA &&
				pacRecvMsgBuffer[2] == MSG_RX_READ_SECURITY_DATA_BYTE)
		{
            INFO_MSG (INFO_LEVEL_1, "INFO: Security Bit is Set\n");	
            sem_post(&m_updateCtrlSem);
		}

		if(pacRecvMsgBuffer[1] == MSG_RX_READ_SECURITY_DATA &&
						pacRecvMsgBuffer[2] == MSG_RX_READ_JTAG_DATA_BYTE)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: JTAG Bit is Set\n");
			sem_post(&m_updateCtrlSem);
		}
		//CRQ CMG3G-9596
		if(pacRecvMsgBuffer[1] == MSG_RX_WRITE_DATA_IDENTI_SIGN1 &&
				pacRecvMsgBuffer[2] == MSG_RX_WRITE_DATA_IDENTI_SIGN3)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: FingerPrint For JTAG is Passed\n");
			m_eJTAGValidationResult = E_JTAG_VALIDATION_SUCCESS;
			sem_post(&m_updateCtrlSem);
		}

		else
		{
			//do nothing
		}

	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vSecurityAccessProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSecurityAccessProcess( tString pacRecvMsgBuffer )
{
	tCString method = "vSecurityAccessProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU8 u8ResponceCode = 0;
	
	if ( NULL != pacRecvMsgBuffer )
	{
		//Get message code
		u8ResponceCode = pacRecvMsgBuffer[1];
		ETG_TRACE_USR4(("DNL: Response Sub code %d", u8ResponceCode));

		switch ( u8ResponceCode )
		{
		case MSG_RX_SECURITY_ACCESS_SEED:
			{
				//Send key
				UpdateLog::vUpdateLogTrace("%s Response recived for Security Access Seed", DL_LOG_INFO );
				UpdateLog::vUpdateLogTrace("%s Sending INC Command for Security Access Send Key", DL_LOG_INFO );
				bWriteToInc( MSG_TX_SECURITY_ACCESS_SEND_KEY );
				break;
			}

		case MSG_RX_SECURITY_ACCESS_SEND_KEY:
			{
				UpdateLog::vUpdateLogTrace("%s Response recived for Security Access Send Key", DL_LOG_INFO );
				//check for security flag
				if(true == bSecurityAccess)
				{
					vSendSecurityBytes();
					bSecurityAccess = false;
				}
				//check for Jtag flag
				else if(true == bJtagAccess)
				{
					vSendJtagBytes();
					bJtagAccess = false;
				}
				else
				{					
					bWriteToInc( MSG_TX_READ_DATA_IDENTI_STATE );
				}

				break;
			}
		default:
			{
				break;
			}

		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vRoutineAndRequestProcess
* DESCRIPTION:	This function start a download communication to flash V850 Chip.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vRoutineAndRequestProcess( tString pacRecvMsgBuffer )
{
	tCString method = "vRoutineAndRequestProcess";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU8 u8ResponceCode = 0;

	if ( NULL != pacRecvMsgBuffer )
	{
		//Get message code
		u8ResponceCode = pacRecvMsgBuffer[0];

		ETG_TRACE_USR4(("DNL: Response code %d", pacRecvMsgBuffer[0]));

		//check the response code
		switch ( u8ResponceCode )
		{
			//response for routine control
		case MSG_RX_ROUTINE_CTRL:
			{
				m_u32RBlockSize = 0;
				if( (MSG_RX_ROUTINE_CTRL_ERASE_MEM1 == pacRecvMsgBuffer[1]) 
					&& 
					(MSG_RX_ROUTINE_CTRL_ERASE_MEM2 == pacRecvMsgBuffer[2])
					&& 
					(MSG_RX_ROUTINE_CTRL_ERASE_MEM3 == pacRecvMsgBuffer[3])
					&& 
					(MSG_RX_ROUTINE_CTRL_ERASE_MEM4 == pacRecvMsgBuffer[4]) )
				{
					//set download state to transfer state
					m_u8DlState = DL_TRANSFER_STATE;

					//Set time out value to default
					UpdateLog::vUpdateLogTrace("%s Setting Default Time Out", DL_LOG_INFO );
					vSetTimeout( DEFAULT_TIMEOUT );

					//request download
					UpdateLog::vUpdateLogTrace("%s Memory Erased!", DL_LOG_INFO );
					INFO_MSG (INFO_LEVEL_1, "INFO: Memory Erased\n");
					bWriteDownloadReqToInc( MSG_TX_REQUEST_DNL, m_u16BlockID );

				}

				break;
			}

			//response for Download Request
		case MSG_RX_REQUEST_DNL:
			{   
				if(m_pclV850MessageFactory != NULL)
					m_pclV850MessageFactory->vEnableDisableDebugTraces(false);

				if ( MSG_RX_REQUEST_DNL1 == pacRecvMsgBuffer[1] )
				{
					++m_u32RBlockSize;
					INFO_MSG (INFO_LEVEL_1, "INFO: Please Wait, Data Transfer is in progress...\n");
					UpdateLog::vUpdateLogTrace("%s Please Wait, Data Transfer is in progress...", DL_LOG_INFO );
				}
			}
			/*fall through*/
		//response for Data Transfer
		case MSG_RX_TRANSF_DATA:
			{
				vBinaryDataTransfer();				
				break;
			}
		default:
			{
				break;
			}

		}
	}

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vBinaryDataTransfer
* DESCRIPTION:	This function transfer the binary data to V850 in a chunk of block.
* PARAMETER:			
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					20.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vBinaryDataTransfer( )
{
	if ( m_u32RBlockSize > 0 )
	{
		//send data in a block size
		if( (m_u32RBlockSize < m_u32BlockSize)
			|| 
			(m_u32RBlockSize == m_u32BlockSize)	)
		{	
			vWriteBinaryDataToInc( BLOCK_SIZE );

		}
		else if( (m_u32RBlockSize > m_u32BlockSize) 
			&& 
			(m_u32LastChuckData > 0) )
		{
			//send last chuck of data
			vWriteBinaryDataToInc( m_u32LastChuckData );
			m_u32LastChuckData = 0;

		}
		else /* End of file*/
		{
			if(m_pclV850MessageFactory != NULL)
				m_pclV850MessageFactory->vEnableDisableDebugTraces(true);
			//set download state to transfer done state
			m_u8DlState = DL_TRANSFER_DONE_STATE;
			INFO_MSG (INFO_LEVEL_1, "INFO: Data transfer done\n");
			UpdateLog::vUpdateLogTrace("%s Data transfer done", DL_LOG_INFO );

			if ( true == m_bIsDnlFile )
			{
				//send security signature bytes
				UpdateLog::vUpdateLogTrace("%s Sending INC command for security signature bytes", DL_LOG_INFO );
				bWriteSecuritySignToInc( MSG_TX_WRITE_DATA_IDENTI_SECURITY_SIGN, m_u16BlockID );
			}
			else
			{
				//send transfer exit command
				UpdateLog::vUpdateLogTrace("%s Sending INC command for transfer exit", DL_LOG_INFO );
				bWriteToInc( MSG_TX_REQUEST_TRANSF_EXIT );
			}
			
		}
		++m_u32RBlockSize;
	}

}

/*******************************************************************************************
* FUNCTION:		error
* DESCRIPTION:	log the error and exit the program execution.
*					
* PARAMETER:
*					tCString
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::error( tCString msg ) const
{
	perror( msg );
	OSAL_vSetProcessExitCode(2);
    OSAL_vProcessExit();
}

/*******************************************************************************************
* FUNCTION:		vMessageTermination
* DESCRIPTION:	close the socket
*					
* PARAMETER:
*					NULl
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vMessageTermination(void) const
{
	tCString method = "vMessageTermination";
	ETG_TRACE_USR4 (("%s was entered", method));

	if (dgram_exit(dgram) < 0)
		error("dgram_exit failed");

	if( sockfd != -1 )
	{
		close(sockfd);
	}

	ETG_TRACE_USR4 (("%s was left", method));
}

/*******************************************************************************************
* FUNCTION:		vDelaySimulation
* DESCRIPTION:	three second delay
*					
* PARAMETER:
*					NULl
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vDelaySimulation (void) const
{
	sleep(3); // Wait 3s

}

/*******************************************************************************************
* FUNCTION:		u8GetDownloadState
* DESCRIPTION:	Return current download state
*					
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	tU8
*
* HISTORY:
*					30.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU8 dl_tclV850UpdateCtrlManager::u8GetDownloadState() const
{
	return m_u8DlState;
}

/*******************************************************************************************
* FUNCTION:		s32GetIncResEventFD
* DESCRIPTION:	Return DownloadCtrlMgr or Inc Response event FD
*					
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	tS32
*
* HISTORY:
*					30.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS32 dl_tclV850UpdateCtrlManager::s32GetIncResEventFD()
{
	if ( s_ptrself != NULL )
	{
		return s_ptrself->m_evfdIncRx;
	}

	return -1;

}

/*******************************************************************************************
* FUNCTION:		s32GetClientReqEventFD
* DESCRIPTION:	Return client request event FD
*					
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	tS32
*
* HISTORY:
*					30.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS32 dl_tclV850UpdateCtrlManager::s32GetClientReqEventFD() const
{

	return m_evfdClientReq;

}

/*******************************************************************************************
* FUNCTION:		pGetMessageQueuePtr
* DESCRIPTION:	Return message queue pointer
*					
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	MessageQueue *
*
* HISTORY:
*					30.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
MessageQueue * dl_tclV850UpdateCtrlManager::pGetMessageQueuePtr()
{
	if ( s_ptrself != NULL )
	{
		return s_ptrself->m_pMessageQueue;
	}

	return 0;

}

/*******************************************************************************************
* FUNCTION:		pGetdlfinishedmutexPtr
* DESCRIPTION:	Return mutex  pointer
*					
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	pthread_mutex_t *
*
* HISTORY:
*					30.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
pthread_mutex_t * dl_tclV850UpdateCtrlManager::pGetdlfinishedmutexPtr()
{
	return 0;
}

/*******************************************************************************************
* FUNCTION:		vSetFilePath
* DESCRIPTION:	This function set the binary file path.
*					
* PARAMETER:
*					tCString CStrFilePath
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					01.10.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetFilePath( tCString CStrFilePath )
{
	//Assign the file name and path to a const character array 
	m_szFilePath = CStrFilePath;

}

/*******************************************************************************************
* FUNCTION:		vSetFileName
* DESCRIPTION:	This function set the binary file Name to be update.
*					
* PARAMETER:
*					tCString CStrFileName
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					01.10.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetFileName( tCString CStrFileName )
{
	//Assign the file name to a const character array 
	m_szFileName = CStrFileName;

}

/*******************************************************************************************
* FUNCTION:		vSetMemeoryAddress
* DESCRIPTION:	This function set the memory address to be flashed.
*					
* PARAMETER:
*					tU32 u32MemoryAddress
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					01.10.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetMemeoryAddress( tU32 u32MemoryAddress )
{
	//Assign the memory address
	m_u32MemoryAddress = u32MemoryAddress;

}

/*******************************************************************************************
* FUNCTION:		vConvertIntArrayToString
* DESCRIPTION:	This function convert array of integer to a string.
*					
* PARAMETER:
*					tString pcHexString,
*					tString pacRecvMsgBuffer,
*					tU32 u32SizeOfArray	
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					02.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vConvertIntArrayToString( tString pcHexString, tString pacRecvMsgBuffer, tU32 u32SizeOfArray) const
{

	if ( NULL != pcHexString )
	{
		memset ( pcHexString, 0, u32SizeOfArray );

		tU32 u32Index;
		for (u32Index = 0; u32Index < u32SizeOfArray; u32Index++)
		{
			sprintf(pcHexString+u32Index,"%c", pacRecvMsgBuffer[u32Index]);
		}
	}

}

/*******************************************************************************************
* FUNCTION:		vConvertIntArrayToHexString
* DESCRIPTION:	This function convert array of integer to a hex string.
*					
* PARAMETER:
*					tString pcHexString,
*					tString pacRecvMsgBuffer,
*					tU32 u32SizeOfArray	
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					02.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vConvertIntArrayToHexString( tString pcHexString, tString pacRecvMsgBuffer, tU32 u32SizeOfArray) const
{

	if ( NULL != pcHexString )
	{
		/* need 2 characters for a single hex value */
		memset ( pcHexString, 0, 2*u32SizeOfArray );

		tU32 u32Index;
		for (u32Index = 0; u32Index < u32SizeOfArray; u32Index++)
		{
			sprintf(pcHexString+2*u32Index,"%02x", pacRecvMsgBuffer[u32Index]);
		}
	}

}
/*******************************************************************************************
* FUNCTION:		vSetTimeout
* DESCRIPTION:	This function sets the timeout value.
*					
* PARAMETER:
*					time_t sectimeout
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetTimeout( time_t sectimeout )
{
	m_sectimeout = sectimeout;
}

/*******************************************************************************************
* FUNCTION:		vStartTimer
* DESCRIPTION:	This function starts the timer.
*					
* PARAMETER:
*					tVoid
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vStartTimer( void )
{
	//Start timer
	itimerspec startvalue;
	bzero(&startvalue, sizeof(startvalue));

	startvalue.it_value.tv_sec = m_sectimeout;

	//Start the timer
	if( m_evFdTimer != EP_HANDLE_INVALID )
	{
		timerfd_settime(m_evFdTimer, 0, &startvalue, NULL);
	}

}

/*******************************************************************************************
* FUNCTION:		vStopTimer
* DESCRIPTION:	This function stops the timer.
*					
* PARAMETER:
*					tVoid
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vStopTimer( void )
{
	itimerspec stopvalue;
	bzero(&stopvalue, sizeof(stopvalue));
	//Stop the timer
	if( m_evFdTimer != EP_HANDLE_INVALID )
	{
		timerfd_settime(m_evFdTimer, 0, &stopvalue, NULL);
	}

}

/*******************************************************************************************
* FUNCTION:		bCheckBinaryFile
* DESCRIPTION:	This function check the particular binary file inside provided path.
*					
* PARAMETER:
*					tCString CStrFolderPath,
*					tCString CStrFileName
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					06.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bCheckBinaryFile(std::string& CStrFolderPath, tCString CStrPattern) const
{
	DIR *dir;
	struct dirent * rDirent;
	tBool bRet   = false;

	/* Open the root directory */
	dir = opendir(CStrFolderPath.c_str());

	if(!dir)
	{
		// can not open the dir
		INFO_MSG (INFO_LEVEL_1, "INFO: %s Directory is not available\n", CStrFolderPath.c_str());
	}
	else
	{
		//read each content of the directory
		while( (rDirent = readdir(dir)) != NULL )
		{
			if ( strstr(rDirent->d_name, CStrPattern)) 
			{				
				CStrFolderPath.append(rDirent->d_name);
				bRet = true;
				INFO_MSG (INFO_LEVEL_1,  "INFO: found an %s file at %s\n", CStrPattern, CStrFolderPath.c_str() );
				break;
			}
		}

		//Close the directory
		closedir(dir);
	}

	return bRet;
}

/*******************************************************************************************
* FUNCTION:		bReadCrcValue
* DESCRIPTION:	This function reads the CRC value and check the result of CRC request.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tBool 
*
* HISTORY:
*					23.01.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bReadCrcValue(tString pacRecvMsgBuffer)
{
	tCString method = "bReadCrcValue";
	ETG_TRACE_USR4 (("%s was entered", method));
	tBool bRet   = false;

	if ( NULL != pacRecvMsgBuffer )
	{
		vConvertIntArrayToHexString( m_chCheckSumVal, pacRecvMsgBuffer, CRC_VALUE_SIZE );
		if( CRC_INFO_PASS == pacRecvMsgBuffer[CRC_VALUE_SIZE] )
		{			
			INFO_MSG (INFO_LEVEL_1, "INFO: CRC is OK\n");
			UpdateLog::vUpdateLogTrace("%s CRC is OK", DL_LOG_INFO );
			m_s8CheckSumResult = 0;
			bRet = true;
		}
		else
		{
			INFO_MSG (INFO_LEVEL_1, "ERROR: Wrong CRC\n");
			UpdateLog::vUpdateLogTrace("%s Wrong CRC", DL_LOG_ERROR );
			m_s8CheckSumResult =static_cast <tS8> (pacRecvMsgBuffer[CRC_VALUE_SIZE] + 1 );
		}
		
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRet;

}

/*******************************************************************************************
* FUNCTION:		bIsFileDnlFormat
* DESCRIPTION:	This function reads the CRC value and check the result of CRC request.
* PARAMETER:
*					tCString m_szFilePath
*
* RETURNVALUE:	tBool : true if binary format is .dnl format else false
*
* HISTORY:
*					05.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bIsFileDnlFormat( tCString sFilePath ) const
{
	tCString method = "bIsFileDnlFormat";
	ETG_TRACE_USR4 (("%s was entered", method));
	tBool bRet   = false;

	if ( NULL != sFilePath )
	{
		/* Check path contains .dnl */
		if ( strstr(sFilePath, DNL_FORMAT) )
		{
			bRet = true;
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return bRet;

}

/*******************************************************************************************
* FUNCTION:		writeToRegistry
* DESCRIPTION:	This function write the key value into registry.
* PARAMETER:
*					tCString cKeyName
*					tString cKeyValue
* RETURNVALUE:	tVoid 
*
* HISTORY:
*					19.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::writeToRegistry( tCString cKeyName, tString cKeyValue )
{
	ETG_TRACE_USR4 (("writeToRegistry(key=%30s val=%s)", cKeyName, cKeyValue));

	RegistryAccess registry;
	tS8 ret = (tS8) registry.u32RegWriteValue(DEVICE_NAME, cKeyName, cKeyValue);
	m_s8SwVersionResult = ret;

	INFO_MSG (INFO_LEVEL_1, "INFO: Key %s = %s written\n", cKeyName, cKeyValue);
	ETG_TRACE_USR4 (("writeToRegistry(key=%30s ret=%i val=%s)", cKeyName, ret, cKeyValue));

}

/*******************************************************************************************
* FUNCTION:		vGetMountPoint
* DESCRIPTION:	This function gives the mount point of the device.
* PARAMETER:
*					tCString sDevice
*					std::string sMountPoint
* RETURNVALUE:	tVoid 
*
* HISTORY:
*					24.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vGetMountPoint(tCString sDevice, std::string& sMountPoint)
{
	FILE *fileptr;
	char cbuff[BUFSIZ];
	std::string sTempStr = "";

	/* open mounts file */
	fileptr = fopen("/proc/mounts" ,"r");
	if( fileptr == NULL )  
	{
		return;
	}

	/* search for sDevice */
	//Coverity fix for 29952
	while(fgets(cbuff,BUFSIZ,fileptr))
	{
		if ( strstr(cbuff, sDevice) ){
			sTempStr.append(cbuff);
			int first_pos = sTempStr.find_first_of(" ", 0);
			if(first_pos >= 0){
				sTempStr.erase(0,first_pos+1);
				/* Copy string into sMountPoint */
				int second_pos = sTempStr.find_first_of(" ", 0);
				if(second_pos >= 0) {
				sMountPoint.assign(sTempStr.c_str(),second_pos);
				break;
				}
			}
		}
	}
	fclose(fileptr);
}

/*******************************************************************************************
* FUNCTION:		u8GetUpdateState
* DESCRIPTION:	This function returns the download state.
* PARAMETER:
*					tVoid
* RETURNVALUE:	tU8 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU8 dl_tclV850UpdateCtrlManager::u8GetUpdateState()
{
	return m_u8DlState;
}


/*******************************************************************************************
* FUNCTION:		s8GetCheckSumResult
* DESCRIPTION:	This function returns the sw version result.
* PARAMETER:
*					tVoid
* RETURNVALUE:	tS8 
*
* HISTORY:
*					13.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS8 dl_tclV850UpdateCtrlManager::s8GetCheckSumResult()
{
	return m_s8CheckSumResult;
}

/*******************************************************************************************
* FUNCTION:		s8GetSwVersionResult
* DESCRIPTION:	This function returns the sw version result.
* PARAMETER:
*					tVoid
* RETURNVALUE:	tS8 
*
* HISTORY:
*					13.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS8 dl_tclV850UpdateCtrlManager::s8GetSwVersionResult()
{
	return m_s8SwVersionResult;
	//return m_s8SwLabelResult;
}

/*******************************************************************************************
* FUNCTION:		s8GetSwVersionResult
* DESCRIPTION:	This function returns the sw version result.
* PARAMETER:
*					tVoid
* RETURNVALUE:	tS8 
*
* HISTORY:
*					13.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS8 dl_tclV850UpdateCtrlManager::s8GetSwLabelResult()
{
	
	return m_s8SwLabelResult;
}

/*******************************************************************************************
* FUNCTION:		readFromRegistry
* DESCRIPTION:	This function reads the key value from registry.
* PARAMETER:
*					tString cKeyValue
*
* RETURNVALUE:	tVoid 
*
* HISTORY:
*					28.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::readFromRegistry( tCString cKeyName )
{
   UpdateLog::vUpdateLogTrace("%s reading Version from registry", DL_LOG_INFO );
   ETG_TRACE_USR4 (("readFromRegistry(%30s) START", cKeyName));
   tS8   pBuffer[255];

   RegistryAccess registry;
   tS8 ret = (tS8) registry.u32RegReadValue(DEVICE_NAME, cKeyName, pBuffer);
   //Assign the SW version result and result for writing to registry

   INFO_MSG (INFO_LEVEL_1, "INFO: Key %s = %s Read\n", cKeyName, pBuffer);
   ETG_TRACE_USR4 (("readFromRegistry(%30s): ret=%i  END",
                    cKeyName, ret));


}


/*******************************************************************************************
* FUNCTION:		vSetCrcEnable
* DESCRIPTION:	This function sets the Crc Enable to false . 
*					
* PARAMETER:
*					tBool bTempCrcEnable 
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					04.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetCrcEnable(tBool bTempCrcEnable)
{
	m_bCrcEnabled = bTempCrcEnable;                                              
}

/*******************************************************************************************
* FUNCTION:		vSetResetEnable
* DESCRIPTION:	This function sets the Reset Enable to false . 
*					
* PARAMETER:
*					bTempResetEnable  
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					04.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetResetEnable(tBool bTempResetEnable)
{
	m_bResetEnabled = bTempResetEnable;                 
}

/*******************************************************************************************
* FUNCTION		: vSetListOfOpera
* DESCRIPTION	: Set the List of Operations.
* PARAMETER		:
* 			List (I)
* 				- List of Command Line Operations To be Performed.
* RETURNVALUE	: tVoid
* HISTORY		: 20/05/2015 - VRI7COB
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetListOfOpera(std::list<EN_EXTRA_OPERATIONS> listOfOpera)
{
	m_listOfOpera = listOfOpera;
}

/*******************************************************************************************
* FUNCTION:		vAppUpdate
* DESCRIPTION:	This function start application update if required.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vAppUpdate()
{
	if ( true == m_bIsBLUpdate )
	{

		if( true == m_bIsDnlFile ) 
		{
			m_u8DlState = DL_SUCCESS_BL_STATE;
			//ECU RESET
			UpdateLog::vUpdateLogTrace("%s Sending INC command to reset V850", DL_LOG_INFO );
			bWriteToInc( MSG_TX_ECU_RESET );
			vDoDesiredOperations(EN_SCREENREINIT_OPERATION);
			m_bIsBLMode = true;
			m_bIsBLSuccess = true;

			vCloseBinaryFile();
			vSetFileName("-app");
		}
		else
		{
			m_u8DlState = DL_SUCCESS_STATE;
			bStopV850Download();
			sleep(TRANSIT_TIME);
		}
	}
	else
	{
		m_u8DlState = DL_SUCCESS_STATE;
		bStopV850Download();	
	}

}

/*******************************************************************************************
* FUNCTION:		vTransitReponse
* DESCRIPTION:	This function checks for TransitResponse
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vTransitReponse()
{
	//Check CRC is enable
	if(true == m_bCrcEnabled ) 
	{
        INFO_MSG (INFO_LEVEL_1, "INFO: Checking CRC\n");
		UpdateLog::vUpdateLogTrace("%s Checking CRC", DL_LOG_INFO );
		//check CRC
		if ( true == m_bIsBLUpdate )
		{
			/* Read Boot loader CRC Value */
			    UpdateLog::vUpdateLogTrace("%s Reading Boot loader CRC Value", DL_LOG_INFO );
				bWriteToInc( MSG_TX_READ_DATA_IDENTI_BL_CRC );
		}
		else
		{
			/* Read Application CRC Value */
			bWriteToInc( MSG_TX_READ_DATA_IDENTI_APP_CRC );
			UpdateLog::vUpdateLogTrace("%s Reading Application CRC Value", DL_LOG_INFO );
		}   
	}
	else /* CRC disable*/
	{
		vAppUpdate();
	}
    						
					
}
/*******************************************************************************************
* FUNCTION:		vSetSecurityByteVal
* DESCRIPTION:	This function sets the Sets the Security Byte recieved as command line input . 
*					
* PARAMETER: tCString
*					
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					30.06.2014	Neera.K.Ponnappa
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Lint warning
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetSecurityByte( tCString CStrSecByteVal )
{
	if(NULL != 	m_szSecValToWrite)
	{
		memset(m_szSecValToWrite, 0, DEFAULT_MSGSZ);
		strcpy(m_szSecValToWrite, (const char*) MSG_TX_SECURITY_BYTE);
		strcat(m_szSecValToWrite, (const char*)CStrSecByteVal);
		//bSecurityAccess = true;
	}
}

/*******************************************************************************************
* FUNCTION:		vSendSecurityBytes
* DESCRIPTION:	This function forms the Security Byte Access Message to be sent. 
*					
* PARAMETER:    void
*				
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					30.06.2014  Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSendSecurityBytes( )
{		    
		if(NULL != m_szSecValToWrite)
		{
			UpdateLog::vUpdateLogTrace("%s Writing the requested Security Bytes", DL_LOG_INFO );
			bWriteToInc(m_szSecValToWrite); 
			UpdateLog::vUpdateLogTrace("%s Reading Security Bytes", DL_LOG_INFO );
			//bWriteToInc(MSG_RX_READ_SEC_PROTECT_BYTE);
		}
}

/*******************************************************************************************
* FUNCTION:		vSetJtagByte
* DESCRIPTION:	This function sets the Jtag Byte . 
*					
* PARAMETER:
*					tCString 
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					30.06.2014  Neera.K.Ponnappa
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Lint warning
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetJtagByte( tCString CStrJtagByteVal )
{
	if(NULL != 	m_szJtagValToWrite)
	{
		memset(m_szJtagValToWrite, 0, DEFAULT_MSGSZ);
		strcpy(m_szJtagValToWrite, (const char*) MSG_TX_JTAG_BYTE);
		strcat(m_szJtagValToWrite, (const char*)CStrJtagByteVal);
		bJtagAccess = true;
	}
	                                             
}

/*******************************************************************************************
* FUNCTION			: vSetJtagByte
* DESCRIPTION		: This function sets the Jtag CRC Byte . 
* PARAMETER			: tCString 
* HISTORY			:
* DATE			| VERSION			| AUTHOR		| MODIFICATION
*--------------------------------------------------------------------------------------------
* 19-02-2016	| 0.1				| VRI7COB		| CRQ, CMG3G-9596			
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vsetJTAG_CRC_Byte(tCString szJtagCRCByte)
{
	if(m_szJtagCRCValToCheck != NULL)
	{
		memset(m_szJtagCRCValToCheck, 0, DEFAULT_MSGSZ);
		strcpy(m_szJtagCRCValToCheck, (const char*) MSG_TX_VALIDATE_JTAG_BYTES);
		strcat(m_szJtagCRCValToCheck, (const char*)szJtagCRCByte);
		INFO_MSG (INFO_LEVEL_1, "INFO: JTAG FingerPrint Value:%s\n", m_szJtagCRCValToCheck);
	}
}

/*******************************************************************************************
* FUNCTION:		vSendJtagBytes
* DESCRIPTION:	This writes to INC the Jtag ID to be set . 
*					
* PARAMETER:
*					Void 
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					30.06.2014  Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSendJtagBytes( )
{	
		if(NULL != m_szJtagValToWrite)
		{
			ETG_TRACE_USR4 (("Writing the requested Jtag Bytes:%s", m_szJtagValToWrite));
			UpdateLog::vUpdateLogTrace("%s Writing the requested Jtag Bytes", DL_LOG_INFO );
			bWriteToInc(m_szJtagValToWrite); 
			UpdateLog::vUpdateLogTrace("%s Reading Jtag Bytes", DL_LOG_INFO );
			//bWriteToInc(MSG_RX_READ_JTAG_PROTECT_BYTE);
		}	   
		
}

/*******************************************************************************************
* FUNCTION:		vHandleSecurityResponse
* DESCRIPTION:	This function handles the response of Security Byte Setting read.
* PARAMETER:
*					tU8 *pacRecvMsgBuffer
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					30.06.2014  Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/      

tVoid dl_tclV850UpdateCtrlManager::vHandleSecurityResponse(tString pacRecvMsgBuffer)               
{
	tCString method = "vResponseForSecurity";
	ETG_TRACE_USR4 (("%s was entered", method));
	tU8 u8ResponceCode = 0;
	
	if ( NULL != pacRecvMsgBuffer )
	{
		/* Get message code */
		u8ResponceCode = pacRecvMsgBuffer[0];
		m_u8SecurityBitVal = u8ResponceCode;

		if(MSG_RX_SEC_BLOCK_ERASE == (~u8ResponceCode & MSG_RX_SEC_BLOCK_ERASE))
		{
			ETG_TRACE_USR4(("Security Protection for Block Erase is set"));
			printf("INFO: Security Protection for Block Erase is set\n");
			UpdateLog::vUpdateLogTrace("%s Security Protection for Block Erase is set", DL_LOG_INFO );

		}
		
		if(MSG_RX_SEC_CHIP_ERASE == (~u8ResponceCode & MSG_RX_SEC_CHIP_ERASE))
		{
			ETG_TRACE_USR4(("Security Byte for Chip Erase is set"));
			printf("INFO: Security Byte for Chip Erase is set\n");
			UpdateLog::vUpdateLogTrace("%s Security Byte for Chip Erase is set", DL_LOG_INFO );

		}
		
		if(MSG_RX_SEC_WRITE_PROC == (~u8ResponceCode & MSG_RX_SEC_WRITE_PROC))
		{
			ETG_TRACE_USR4(("Security Byte for Write Protection is set"));
			printf("INFO: Security Byte for Write Protection is set\n");
			UpdateLog::vUpdateLogTrace("%s Security Byte for Write Protection is set", DL_LOG_INFO );

		}
		if(MSG_RX_SEC_READ_PROC == (~u8ResponceCode & MSG_RX_SEC_READ_PROC))
		{
			ETG_TRACE_USR4(("Security Byte for read protection is set"));
			printf("INFO: Security Byte for read protection is set\n");
			UpdateLog::vUpdateLogTrace("%s Security Byte for read protection is set", DL_LOG_INFO );

		}

		if(MSG_RX_SEC_BOOT_BLOCK_ERASE == (~u8ResponceCode & MSG_RX_SEC_BOOT_BLOCK_ERASE))
		{
			ETG_TRACE_USR4(("Security Protection for Boot Block Erase is set"));
			printf("INFO: Security Protection for Boot Block Erase is set\n");
			UpdateLog::vUpdateLogTrace("%s Security Protection for Boot Block Erase is set", DL_LOG_INFO );
		}

		if(MSG_RX_SEC_INVALID_SECURITY == u8ResponceCode)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: Security is Not Implemented in V850\n");
		}

	}
	else
	{
		ETG_TRACE_USR4(("ERROR:reading the Security Byte Protection"));
		printf("ERROR: reading the Security Byte Protection");
		UpdateLog::vUpdateLogTrace("%s Error reading the Security Byte Protection", DL_LOG_ERROR );
		sem_post(&m_updateCtrlSem);
	}

	ETG_TRACE_USR4 (("%s was left", method));
	sem_post(&m_updateCtrlSem);
//	bStopV850DownloadForSec();
}
/*******************************************************************************************
* FUNCTION:		vHandleJtagResponse
* DESCRIPTION:	This function handles the response of Jtag Byte.
* PARAMETER:
*				
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					30.06.2014  Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/      

tVoid dl_tclV850UpdateCtrlManager::vHandleJtagResponse()               
{
	tCString method = "vResponseForSecurity";
	ETG_TRACE_USR4 (("%s was entered", method));
	printf("INFO: Jtag ID set Successfully!");
	UpdateLog::vUpdateLogTrace("%s Jtag ID set Successfully!", DL_LOG_INFO );
	ETG_TRACE_USR4 (("%s was left", method));
	bStopV850DownloadForSec(); 
		
}
/*******************************************************************************************
* FUNCTION:		bStopV850DownloadForSec
* DESCRIPTION:	This function stop download communication.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bStopV850DownloadForSec( )
{
	
	tCString method = "bStopV850Download";
	ETG_TRACE_USR4 (("%s was entered", method));	
	tBool bRetVal = true;		

	// Application invoked with -noreset option, switch back to BL_Mode
	if (true == m_bResetEnabled )
	{
	   //ECU RESET
	   UpdateLog::vUpdateLogTrace("%s Sending INC Command to Reset V850", DL_LOG_INFO );
	   bWriteToInc( MSG_TX_ECU_RESET );
	   vDoDesiredOperations(EN_SCREENREINIT_OPERATION);
	}
	else
	{
		INFO_MSG (INFO_LEVEL_1, "INFO: No RESET\n");
		UpdateLog::vUpdateLogTrace("%s No V850 Reset", DL_LOG_INFO );
	   
	}

	if ( DL_SUCCESS_STATE == m_u8DlState )
	{
		INFO_MSG (INFO_LEVEL_1, "INFO: V850 Download Success\n");
		UpdateLog::vUpdateLogTrace("%s V850 Download Success", DL_LOG_INFO );

	}
	else if ( DL_INIT_STATE == m_u8DlState )
	{
		//INFO_MSG (INFO_LEVEL_1, "INFO: V850 Reset\n");

	}
	else
	{
		INFO_MSG (INFO_LEVEL_1, "ERROR: V850 Download Failed\n");
		UpdateLog::vUpdateLogTrace("%s V850 Download Failed", DL_LOG_ERROR );
	}
	vTerminateProgram();

	ETG_TRACE_USR4 (("%s was left", method));

	return bRetVal;
}

/*******************************************************************************************
* FUNCTION:		bWriteSecuritySignToInc
* DESCRIPTION:	write Security Signature byte command to Inc driver
*					
* PARAMETER:
*					tCString sCmdString, //command string
*					tU16 Typ // BlockID		
*					tU32 u32MemBlock // App or BL
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					24.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bWriteSecuritySignToInc( tCString sCmdString, tU32 u32MemBlock )
{
	tCString method = "bWriteSecuritySignToInc";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32FileLength = 0;

	/* To store the command */
	tChar achCmdbuf[DATA_TRANSFER_SIZE+1];
	memset( achCmdbuf, 0x00, (DATA_TRANSFER_SIZE+1) );

	/* To store the RSA data bytes */
	tChar buffer[DATA_TRANSFER_SIZE+1];
	memset( buffer, 0x00, (DATA_TRANSFER_SIZE+1) );
	tU32 u32PackLength = 0;

	//check for NULL pointer
	if ( (sockfd != 0)
		&&
		( NULL != m_pclV850MessageFactory )
		&&
		( NULL != m_pclIncSender ) )
	{

		//get the size of Security Signature and read RSA bytes 
		u32FileLength = m_pclV850MessageFactory->u32GetRSASignature (m_pFile, &m_DFInfo, buffer, u32MemBlock); 

		if ( u32FileLength == 0 )
		{
			//send transfer exit command
			bWriteToInc( MSG_TX_REQUEST_TRANSF_EXIT );
			return false;
		}

		INFO_MSG (INFO_LEVEL_1, "INFO: length of %04X security signature: %08X\n", u32MemBlock, u32FileLength);

		//pack the message
		u32PackLength = m_pclV850MessageFactory->u32PackSecuritySignReqPacket( sCmdString, achCmdbuf, buffer, (DATA_TRANSFER_SIZE+1), u32FileLength);

		//copy the command in command buffer
		std::memset( m_chCmdBuff, 0 , sizeof(m_chCmdBuff) );
		strncat(m_chCmdBuff, sCmdString, sizeof(m_chCmdBuff)-1);//coverity fix for 126980
		vStopTimer();

		//Write to inc
		if ( false == m_pclIncSender->bWriteDataPacket( sockfd, achCmdbuf, u32PackLength ) )
		{
			vSendPacketError();
		}
		else
		{
			vStartTimer();
		}
		//m_pclV850MessageFactory->vHexDump( "SEND", achCmdbuf , u32PackLength );

		//Start the timer
		vStartTimer();
		m_bIsSignByteSend = true;
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return true;

}
/*******************************************************************************************
* FUNCTION:		vSetSecFlag
* DESCRIPTION:	This function Sets the flag for security Byte.
* PARAMETER:
*					NULL		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					05.08.2014	Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/

tVoid dl_tclV850UpdateCtrlManager::vSetSecFlag()
{
	m_bSecFlag = true;                 
}

/*******************************************************************************************
* FUNCTION:		bHandleSignRes
* DESCRIPTION:	Handler for Security Signature response
*					
* PARAMETER:
*					tVoid		
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					25.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bHandleSignRes( )
{
	tBool bRetVal = false;
	if ( true == m_bIsSignByteSend ) 
	{
		//send transfer exit command
		UpdateLog::vUpdateLogTrace("%s Sending INC Command for transfer exit", DL_LOG_INFO );
		bWriteToInc( MSG_TX_REQUEST_TRANSF_EXIT );
		m_bIsSignByteSend = false;
		bRetVal = true;
	}

	return bRetVal;

}

/*******************************************************************************************
* FUNCTION:		s8GetErrorCode
* DESCRIPTION:	Get the error code
*					
* PARAMETER:
*					tVoid		
*
* RETURNVALUE:	tChar
*
* HISTORY:
*					16.09.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tChar dl_tclV850UpdateCtrlManager::s8GetErrorCode()
{
	return m_chErrorCode;
}
	
/*******************************************************************************************
* FUNCTION:		vSetErrorCode
* DESCRIPTION:	Set the error code
*					
* PARAMETER:
*					tChar		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					16.09.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSetErrorCode(tChar chErrorCode )
{
	m_chErrorCode = chErrorCode;
}

/*******************************************************************************************
* FUNCTION:		vSimaphoreRelease
* DESCRIPTION:	Set the error code
*					
* PARAMETER:
*					tVoid		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					16.09.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vWaitForSemRelease()
{
	//Wait for sem post
	sem_wait(&m_updateCtrlSem);

	if( m_chErrorCode != 0 && m_u8DlState != DL_RETRY_STATE )
	{
		OSAL_vSetProcessExitCode(m_chErrorCode);
		OSAL_vProcessExit();
	}

}

/*******************************************************************************************
* FUNCTION:		vSendPacketError
* DESCRIPTION:	error while sending the packet
*					
* PARAMETER:
*					tVoid		
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					16.09.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vSendPacketError()
{
	//set error code
	vSetErrorCode(2);
	//Wait for sem post
	sem_post(&m_updateCtrlSem);
}


/*******************************************************************************************
* FUNCTION:		bIsSignByteSend
* DESCRIPTION:	getter function to check signature bytes are send or not.
*					
* PARAMETER:
*					tVoid		
*
* RETURNVALUE:	tBool true: signature bytes are send / False: signature bytes are not send
*
* HISTORY:
*					18.09.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bIsSignByteSend()
{
	return m_bIsSignByteSend;
}

/*******************************************************************************************
* FUNCTION			:vDoDesiredOperations
* DESCRIPTION		:Do the Desired Operations which is passed as an Argument.
* PARAMETER			:
* 				enum (I) - Enum of Operations To be Performed.
* RETURNVALUE		:tVoid
* HISTORY			:
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::vDoDesiredOperations(EN_EXTRA_OPERATIONS eDesOpera)
{

	for(std::list<EN_EXTRA_OPERATIONS>::iterator Iter = m_listOfOpera.begin(); Iter != m_listOfOpera.end(); Iter++)
	{
		//check Desried Operations is present in List.
		if(*Iter == eDesOpera)
		{
			if(eDesOpera == EN_SCREENREINIT_OPERATION)
			{
				INFO_MSG (INFO_LEVEL_1, "INFO: Screen-Reinit Opera\n");
				if(bDoScreenReInitOpera())
				{
					INFO_MSG (INFO_LEVEL_1, "INFO: Screen-Reinit Success\n");
				}
				else
					INFO_MSG (INFO_LEVEL_1, "INFO: Screen-Reinit Failure\n");
			}
		}
	}
}

/*******************************************************************************************
* FUNCTION			:bDoScreenReInitOpera
* DESCRIPTION		:Do the Screen ReInit Operations
* PARAMETER			:tVoid
* RETURNVALUE		:
* 				tBool (O) - Success or Failure
* HISTORY			:
********************************************************************************************/
tBool dl_tclV850UpdateCtrlManager::bDoScreenReInitOpera()
{
	sleep(1);				//Requirement From Platform Team
	tBool bReturn = false;

	tU8 value = 1;
	tS32 u32FileDesp = open("/sys/devices/soc.0/2100000.aips-bus/21a4000.i2c/i2c-0/0-002c/reinit_ds90uh926", O_WRONLY );

	if (u32FileDesp == -1)
	{
		INFO_MSG (INFO_LEVEL_1, "ERROR: Error in Opening SysFS\n");
		return false;
	}

	if(write(u32FileDesp, &value, sizeof(tU8)) == -1)
	{
		INFO_MSG (INFO_LEVEL_1, "INFO: Writing to SysFS is Success\n");
		bReturn = true;
	}
	else
	{
		INFO_MSG (INFO_LEVEL_1, "ERROR: Writing to SysFS is Failure\n");
		bReturn = false;
	}
	close(u32FileDesp);

	return bReturn;
}

/*******************************************************************************************
* FUNCTION			:KillManagerThread
* DESCRIPTION		:Killing the Manager Thread When exiting the Process.
* PARAMETER			:tVoid
* RETURNVALUE		:tVoid
* HISTORY			:
********************************************************************************************/
tVoid dl_tclV850UpdateCtrlManager::KillManagerThread()
{
	pthread_cancel(m_v850DlCtrlMgrThread);
	pthread_join(m_v850DlCtrlMgrThread, 0);
}
