/*******************************************************************************************
* FILE:			dl_IncReceiver.cpp
*
* SW-COMPONENT:Software Download
*
* DESCRIPTION:	This file create a thread, whose job is to create a socket 
*					and listen to the port. Receive a message which is send to this port 
*					and post a message to V850 core control thread using message queue.
*				
* AUTHOR:		Aditya Kumar Jha
*
* COPYRIGHT:	(c) 2013  2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
* ------------------------------------------------------------------------------------------------
* Date 				| Rev. 				| Author 		   	| Modification
* ------------------------------------------------------------------------------------------------
* 					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*					18.02.2014	Aditya Kumar Jha : Change for new trace class TR_CLASS_SWUPDATE_V850.
*					18.02.2014	Aditya Kumar Jha : Check to filter generic postive response.
*					16.09.2014	Aditya Kumar Jha : Fixes for Jira ticket CMG3GB-1128.
*					18.09.2014	Aditya Kumar Jha : prevent to kill ReceiverThread when signature bytes are sent 
*                                             and negative response is MSG_NEG_RX_REQ_OUT_OF_RANGE.
*---------------------------------------------------------------------------------------------------
* 12-03-2015		| 2.0	| VRI7COB (RBEI)	| Enabling TTFI's Traces;Ticket Number:CMG3GB-1324
*---------------------------------------------------------------------------------------------------
* 24-06-2015        | 2.1   | VRI7COB (RBEI)    | Fixing the V850 Hang due to Not Processing
*                   |       |                   | all Messages in Message Queue. GMMY17-4737
*---------------------------------------------------------------------------------------------------
* 08-04-2016		| 2.2	| VRI7COB (RBEI)	| Fixing the Issue PSARCCB-6728
*					|		|					| Negative Response From v850 - 7F-2E-31 Exit the Receiver Thread.
*					|		|					| ReInitailized the Receiver Buffer to 0
********************************************************************************************/

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclIncReceiver::
    #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_V850
  #include "trcGenProj/Header/dl_IncReceiver.cpp.trc.h"
#endif

#include "dl_V850UpdateCtrlManager.h"
#include "dl_V850MessageDefine.h"
#include "dl_IncReceiver.h"
#include <pthread.h>
#include <algorithm>
#include <iterator>
#include <iomanip>
#include <errno.h>


/*******************************************************************************************
* FUNCTION:		dl_tclIncReceiver
* DESCRIPTION:	Private Constructor
* PARAMETER:
*					NULL
*				
* RETURNVALUE:	Void
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
********************************************************************************************/
dl_tclIncReceiver::dl_tclIncReceiver()
{
	m_RunningRecvThread = false;
	m_pDataGram = NULL;

}

/*******************************************************************************************
* FUNCTION:		~dl_tclIncReceiver
* DESCRIPTION:	Destructor
* PARAMETER:
*					NULL
*				
* RETURNVALUE:	Void
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
********************************************************************************************/
dl_tclIncReceiver::~dl_tclIncReceiver()
{
	m_RunningRecvThread = false;

	if ( NULL != m_pDataGram )
	{
		m_pDataGram = NULL;
	}

}

/*******************************************************************************************
* FUNCTION:		s32CreateIncReceiverThread
* DESCRIPTION:	This funcrion creates a working thread, whose job is to create a socket 
*					and listen to the port. Receive a message which is send to this port 
*					and post a message to V850 core control thread using message queue.
* PARAMETER:
*					sk_dgram * localDatagram 
*				
* RETURNVALUE:	tS32
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS32 dl_tclIncReceiver::s32CreateIncReceiverThread( sk_dgram * localDatagram )
{
	tCString method = "bInitV850DLCtrl";
	ETG_TRACE_USR4 (("%s was entered", method));

	if ( NULL != localDatagram )
	{

		m_pDataGram = localDatagram;

		//Create a seperate working thread for 			
		pthread_t  incReceiverThread;
		pthread_create( &incReceiverThread, NULL, &dl_tclIncReceiver::vIncRecThreadCallBack, this );
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return 0;
}

/*******************************************************************************************
* FUNCTION:		vIncReceiverWorkedThread
* DESCRIPTION:	This funcrion creates a working thread.
*
* PARAMETER:
*					NULL
*				
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*--------------------------------------------------------------------------------------------
*DATE		|	AUTHOR			| MODIFICATION
*-------------------------------------------------------------------------------------------
*08-04-2016	|  VRI7COB (RBEI)	| Fixing the Issue PSARCCB-6728
*			|					| Add the Extra ETG Trace for printing the received data. 
*			|					| To get only this enable 6 level of tracing in ETG.
********************************************************************************************/
tVoid dl_tclIncReceiver::vIncReceiverWorkedThread( )
{
	tCString method = "vIncReceiverWorkedThread";
   ETG_TRACE_USR4 (("%s was entered", method));

	m_RunningRecvThread = true;
	tS32 s32ReceivedMsg = -1;
	tU8 acRecvMsgBuffer[DEFAULT_MSGSZ+3];
	memset(acRecvMsgBuffer, 0, (DEFAULT_MSGSZ+3) );

	while ( m_RunningRecvThread )
	{	
		if ( NULL != m_pDataGram )
		{				
			memset(acRecvMsgBuffer, 0, (DEFAULT_MSGSZ+3) );
			s32ReceivedMsg = (tS32)(ssize_t)(dgram_recv(m_pDataGram, &acRecvMsgBuffer[2], DEFAULT_MSGSZ));

			stringstream ss;
			ss << std::hex << std::setfill('0') << std::setw(2);
			copy( &acRecvMsgBuffer[2], &acRecvMsgBuffer[s32ReceivedMsg + 2], ostream_iterator<int>(ss, "\t"));
			ETG_TRACE_USR2(("Received Data - %s", ss.str().c_str()));

			if (s32ReceivedMsg < 0)
			{
				INFO_MSG (INFO_LEVEL_1, "ERROR: reading from socket\n");
				ETG_TRACE_USR2(("Error:%s", strerror(errno)));
			}
			else if( MSG_POSITIVE_RX_HIGHER_BYTE == acRecvMsgBuffer[2] && MSG_POSITIVE_RX_LOWER_BYTE == acRecvMsgBuffer[4])
			{
				//It is a positive response continue to read the data from recv call
				continue;
			}
			else
			{
				acRecvMsgBuffer[0] = DL_RESPONSE_RECEIVED;
				acRecvMsgBuffer[1] = (tU8)(tS32)s32ReceivedMsg;
				s32ReceivedMsg++;
				s32ReceivedMsg++;
				vPostMsgToV850DLCtrl( acRecvMsgBuffer, s32ReceivedMsg );
			}

			//This commented out, because below check is not needed.
			//All Condition check is handled in Manager Thread. In Assumption that Receiver thread always be in running.
//			if(s32ReceivedMsg > = 3)
//			{
//				if ( (true == dl_tclV850UpdateCtrlManager::bIsSignByteSend())
//					&&
//					(MSG_NEG_RX == acRecvMsgBuffer[2])
//					&&
//					(MSG_NEG_RX_REQ_OUT_OF_RANGE == acRecvMsgBuffer[4]) )
//				{
//					//do nothing
//				}
//				else if( (MSG_POSITIVE_RX_HIGHER_BYTE == acRecvMsgBuffer[2]) && (MSG_POSITIVE_RX_LOWER_BYTE != acRecvMsgBuffer[4]) )
//				{
//					//INFO_MSG (INFO_LEVEL_1, "INFO: negative response\n");
//					m_RunningRecvThread = false;
//				}
//			}
		} 
	} 

	ETG_TRACE_USR4 (("%s was left", method));

}

/*******************************************************************************************
* FUNCTION:		vPostMsgToV850DLCtrl
* DESCRIPTION:	This funcrion post the message to V850 core control thread using message queue.
* PARAMETER:
*					tU8* message
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*					24-06-2015  VRI7COB (RBEI)
*					To know Incoming Message From V850 through TTFi's
*					Extra Trace is added.
********************************************************************************************/
tVoid dl_tclIncReceiver::vPostMsgToV850DLCtrl( tU8* message, tS32 s32ReceivedMsg) const
{
	tCString method = "vPostMsgToV850DLCtrl";
	ETG_TRACE_USR4 (("%s was entered", method));

	ETG_TRACE_USR4(("RECV First Value:0x%02x", message[2]));


	MessageQueue * pMessageQueue = dl_tclV850UpdateCtrlManager::pGetMessageQueuePtr();

	//Check the message queue pointer and post the message over queue
	if( ( NULL != pMessageQueue )
		&&
		( NULL != message ) )
	{
		pMessageQueue->vSendtoRespMsgQ(message, s32ReceivedMsg);

		tS32 s32TempFD = dl_tclV850UpdateCtrlManager::s32GetIncResEventFD();
		if ( s32TempFD != -1 )
		{
			write( s32TempFD, message ,sizeof(uint64_t) );
		}
	}

	ETG_TRACE_USR4 (("%s was left", method));

}


/*******************************************************************************************
* FUNCTION:		vSetReceiverThreadStatus
* DESCRIPTION:	This funcrion sets the receiver thread status to terminate the Inc Receiver thread.
* PARAMETER:
*					bool bStatus
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclIncReceiver::vSetReceiverThreadStatus( tBool bStatus )
{
	m_RunningRecvThread = bStatus;

}
