#ifndef SWU_UTIL_HPP
#define SWU_UTIL_HPP

#include <vector>
#include <set>
#include <string>
#include <map>
#include <stddef.h>
#include "tinyxml/tinyxml.h"
#include "util/swu_types.h"
#include "util/swu_trace.h"
#include "util/swu_securityEngine.h"

class TiXmlElement;

namespace swu {

/**
 * This class represents the Checksum-Entry of an BoschXML file.
 */
class CCkSumElement {
public:
   CCkSumElement() :
               _ENCRYPT_PARAM(""),
               _FILE_SIZE(0),
               _DIGEST_TYPE(swu::tenDigestTypeInvalid),
               _DIGEST_SEGLEN(0){
   }

   bool init(const TiXmlElement *pCurRelElement);

   uint64_t GetSize() const {
      return _FILE_SIZE;
   }

   swu::tenDigestType GetType() const {
      return _DIGEST_TYPE;
   }

   uint32_t GetSeglen() const {
      return _DIGEST_SEGLEN;
   }

   std::string GetEncryptParam() const {
      return _ENCRYPT_PARAM;
   }

   const ::std::vector< ::std::vector< uint8_t > > &GetDigests() const {
      return _DIGEST_BIN;
   }

   ::std::vector< ::std::vector< uint8_t > > &GetDigestsMutable() {
      return _DIGEST_BIN;
   }

private:

   std::string _ENCRYPT_PARAM;
   uint64_t _FILE_SIZE;
   swu::tenDigestType _DIGEST_TYPE;
   uint32_t _DIGEST_SEGLEN;

   ::std::vector< ::std::vector< uint8_t > > _DIGEST_BIN;
};
void forceSegmentationFault();
void registerSignalHandler();

std::string binToHex(const std::vector< unsigned char >& bin);
void binToHex(const unsigned char *bin, size_t len, char *result);
bool hexToBin(const char *str, size_t len, std::vector< unsigned char > &result);
bool hexToBin(const char *str, size_t len, unsigned char *result);
bool hexToBin(std::string str, std::vector< unsigned char > &result);

tU8 bcdToU8(tU8 u8Bcd);

unsigned int extractUint(char const* array);
std::string extractLine(std::string &buffer);
bool extractLine(std::string &buffer, std::string &line);
bool cmdToLines(std::string cmd, std::vector<std::string> &lines, bool trim=true);

// return true if string has been modified
bool purifyString(std::string &str, bool truncate=true);

// remove one line from buffer and return it.
std::string getLine(std::string buffer);

bool streq(const char *a, const char *b);

tUInt stringToUInt(std::string val);
std::string intToString(tUInt val);
std::string u64ToString(int64_t val);
// convert unsigned chars to a string
std::string toString (const std::vector<SWU_BYTE>& data);


::std::set<std::string> splitString(::std::string strIn, char delim=',');
::std::vector<std::string> stringToVector(::std::string strIn, char delim=',');
tUInt replaceSubString(std::string &where, const std::string& what, const std::string& with);
template<class T>
bool hasIntersection(::std::set<T> const &set1, ::std::set<T> const &set2) {
   typename ::std::set<T>::const_iterator iter1=set1.begin();
   typename ::std::set<T>::const_iterator iter2=set2.begin();
   while (iter1!=set1.end() && iter2!=set2.end())
   {
      if (*iter1<*iter2) ++iter1;
      else if (*iter2<*iter1) ++iter2;
      else {
         return true;
      }
   }
   return false;
}

template<class T>
bool findInSet(::std::set<T> const &whereSet, T const &what) {
   return whereSet.find(what) != whereSet.end();
}

template<class T>
void addToSet(::std::set<T>&to, ::std::set<T> const &from) {
   for (typename ::std::set<T>::const_iterator iter=from.begin(); iter!=from.end(); ++iter) {
      to.insert(*iter);
   }
}

bool addNodeToAll(TiXmlNode *root, const char *toNode, TiXmlElement &extraNode);
void storeTiXmlElement(TiXmlElement *xmlElement, std::string pathAndName);
void setTextChild( TiXmlElement* parent , const std::string childName, const std::string childText );
void setTextChild( TiXmlElement* parent , const char *childName, const char *childText );

void removeChildren( TiXmlElement *parent, std::string const childName);

bool getTextFromChildAsBool( TiXmlElement const *pElem, char const *childName);
const char *getTextFromChild( TiXmlElement const *pElem, char const *childName, bool assert = true, char const *defaultRes=0 );

inline const char *getTextFromChildOrEmpty( TiXmlElement const *pElem, char const *childName) {
   return getTextFromChild(pElem, childName, false, "");
};

TiXmlElement *addTextChild( TiXmlElement *pElem, char const *childName,  char const *childText);
TiXmlElement *addTextChild( TiXmlElement *pElem, ::std::string const childName,  ::std::string const childText);
void addUIntChild( TiXmlElement *parent, char const *childName,  tUInt val);
void setUIntChild( TiXmlElement *parent, char const *childName,  tUInt val);

tUInt getUIntFromChild( TiXmlElement const *pElem, char const *childName, tUInt defaultVal=0);

tUInt getNumChildren(TiXmlElement *parent, std::string childName);
tUInt getNumChildrenWithTxtChild(TiXmlElement *parent, std::string childName, std::string withElem, std::string withText);
TiXmlElement *getNthChild(TiXmlElement *parent, std::string childName, tUInt index);

std::string &trimString(::std::string &str);
::std::string uIntToString(tUInt val);

bool bFindPatternInString(::std::string pattern, ::std::string str);
bool bFindPatternInFile(::std::string pattern, ::std::string fileName);

::std::string getMediaType(::std::string mediaPath);

void lowercase(::std::string& str);
void uppercase(::std::string& str);
std::string toLower(char const *chr);

bool hasSuffix(const ::std::string& file, const ::std::string& suffix);

bool stringToBool(std::string val);

template< class T >
class ScopedBuffer {
public:
   ScopedBuffer(size_t size, bool zero = false) :
      _size(size) {
      _buf = new T[size];
      SWU_ASSERT_RETURN(_buf);
      if (zero) {
         memset(_buf, 0, size);
      }
   }
   ~ScopedBuffer() {
      if (_buf) {
         delete[] _buf;
         _buf = 0;
      }
   }
   size_t getSize() {
      return _size;
   }
   T *getBuf() {
      return _buf;
   }
private:
   size_t _size;
   T *_buf;
   ScopedBuffer ( const ScopedBuffer & );  //Coverity fix for 48249
   ScopedBuffer& operator = ( const ScopedBuffer & );  //Coverity fix for 47905
};

template< class T >
class ScopedPointer {

private:
   ScopedPointer & operator=(const ScopedPointer &);
   ScopedPointer(const ScopedPointer &);
public:
   ScopedPointer(T *object) : _obj(object) {
      SWU_ASSERT_RETURN(object);
   }

   ScopedPointer() : _obj(0) {
   }

   ~ScopedPointer() {
      if (_obj) {
         delete _obj;
      }
   }

   T *operator->() {
      SWU_ASSERT_RETURN_VAL(_obj, _obj);
      return _obj;
   }

   T *assign(T * obj)  {
      SWU_ASSERT_RETURN_VAL(obj, obj);
      if (_obj) {
         delete _obj;
      }
      _obj=obj;
      return _obj;
   }
   T *getPtr() {
      return _obj;
   }

   T* release() {
      T *obj=_obj;
      _obj = 0;
      return obj;
   }

private:
   T *_obj;
};

class XmlElemContainer {
public:
   XmlElemContainer(const ::std::string &elemType):
      _elemType(elemType)
   {}
   tVoid add(::std::string const &name, TiXmlElement const *xmlParent);

   TiXmlElement const *pull(std::string const &name);
   TiXmlElement const *get(std::string const &name);

private:
   ::std::map< ::std::string, TiXmlElement const *> _elems;
   ::std::string const _elemType;
};
}

#endif
