#include <sys/stat.h>
#include <cstdlib>
#include <sstream>
#include <iostream>
#include <fstream>
#include <algorithm>
#ifndef LOCAL_BUILD_
#include <regex.h>
#include <pthread.h>
#include <signal.h>
#endif

#include <algorithm>
#include "tinyxml/tinyxml.h"
#include "util/swu_types.h"
#include "util/swu_constants.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_trace.h"
#include "util/swu_execCommand.h"
#include "util/swu_util.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_util.cpp.trc.h"
#endif

using namespace std;

namespace swu {

bool CCkSumElement::init(const TiXmlElement *pCurRelElement) {
#ifdef LOCAL_BUILD_
   return true;
#else
   SWU_ASSERT_RETURN_FALSE(pCurRelElement);
   SWU_ASSERT_RETURN_FALSE(std::string(pCurRelElement->Value()) == "CKSUM");
   _ENCRYPT_PARAM = getTextFromChildOrEmpty(pCurRelElement,"ENCRYPT_PARAM");
   _FILE_SIZE = static_cast<uint64_t>(strtoll(getTextFromChildOrEmpty(pCurRelElement, "SIZE"), 0, 10) );
   string typeStr = getTextFromChildOrEmpty(pCurRelElement, "TYPE");
   if (typeStr == "SHA1") {
      _DIGEST_TYPE = swu::tenDigestTypeSHA1;
   } else if (typeStr == "MD5") {
      _DIGEST_TYPE = swu::tenDigestTypeMD5;
   } else if (typeStr == "NONE") {
      _DIGEST_TYPE = swu::tenDigestTypeNone;
   } else if (typeStr == "SHA256") {
      _DIGEST_TYPE = swu::tenDigestTypeSHA256;
   } else {
      ETG_TRACE_COMP(("Unknown digest type; %s", typeStr.c_str()));
      _DIGEST_TYPE = swu::tenDigestTypeInvalid;
   }
   
   if (_DIGEST_TYPE == swu::tenDigestTypeNone) {
      return true;
   }
   else if(0 != pCurRelElement->FirstChildElement("DIGEST_SEGLEN"))
   {
      _DIGEST_SEGLEN =static_cast<uint32_t> (strtoll(getTextFromChildOrEmpty(pCurRelElement, "DIGEST_SEGLEN"), 0, 10) );
      for (const TiXmlElement *pCksumVALElement = pCurRelElement->FirstChildElement("DIGEST");
           pCksumVALElement;
           pCksumVALElement = pCksumVALElement->NextSiblingElement("DIGEST")) {
         const char *valTextOrig = pCksumVALElement->GetText();
         SWU_ASSERT_RETURN_FALSE(valTextOrig);
         size_t valTextLen = strlen(valTextOrig);
         std::vector< uint8_t > result;
         hexToBin(valTextOrig, valTextLen, result);
         _DIGEST_BIN.push_back(result);
      }
   } else
   {  _DIGEST_SEGLEN = static_cast<uint32_t>(strtoll(getTextFromChildOrEmpty(pCurRelElement, "SEGLEN"), 0, 10) );
      for (const TiXmlElement *pCksumVALElement = pCurRelElement->FirstChildElement("VAL");
              pCksumVALElement;
              pCksumVALElement = pCksumVALElement->NextSiblingElement("VAL")) 
      {
         const char *valTextOrig = pCksumVALElement->GetText();
         SWU_ASSERT_RETURN_FALSE(valTextOrig);
         size_t valTextLen = strlen(valTextOrig);
         std::vector< uint8_t > result;
         hexToBin(valTextOrig, valTextLen, result);
         _DIGEST_BIN.push_back(result);
      }
   }

   return true;
#endif
}



void forceSegmentationFault() {
#ifndef LOCAL_BUILD_
   raise(SIGSEGV);
#endif
}

void signalHandler(int sig) {
#ifndef LOCAL_BUILD_
   ETG_TRACE_FATAL(("Error: signal %d:", sig));
   // get void*'s for all entries on the stack
   // print out all the frames
   traceBacktrace();
   exit(1);
#endif
}

void registerSignalHandler() {
#ifndef LOCAL_BUILD_
   // todo: backtraces do not resolve names.
   // todo: check how backtraces shall be produced in our system
   signal(SIGSEGV, signalHandler);   // install our handler
   signal(SIGABRT, signalHandler);   // install our handler
#endif
}



static const char nibbleToCharArray[]= {
   '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};

std::string binToHex(const std::vector<unsigned char>& bin)
{
   std::string hex;
   for (std::vector<unsigned char>::const_iterator i = bin.begin(); i != bin.end(); i++) {
      hex.push_back(nibbleToCharArray[((*i) >> 4) & 0x0f]);
      hex.push_back(nibbleToCharArray[(*i) & 0x0f]);
   }
   return hex;
}


void binToHex(const unsigned char *bin, size_t len, char *result) {
   const unsigned char *binOrig=bin;
   char *resultOrig=result;
   for (unsigned int i=0; i<len; i++) {
      *result = nibbleToCharArray[((*bin) >> 4) & 0x0f];
      ++result;
      *result = nibbleToCharArray[(*bin) & 0x0f];
      ++result;
      ++bin;
   }
#ifndef LOCAL_BUILD_
   ETG_TRACE_USR1(("hexToBin:hex:%32s bin=%02x",
                   resultOrig,
                   ETG_LIST_LEN(static_cast<unsigned int> (len)), ETG_LIST_PTR_T8(binOrig)));
#endif
}



bool hexToBin(std::string str, std::vector< unsigned char > &result) {
   //const char *strOrig=str;

   return hexToBin(str.c_str(), str.size(), result);
}

bool hexToBin(const char *str, size_t len, std::vector<unsigned char > &result) {
   SWU_ASSERT_RETURN_VAL(str, false);
   if (len %2) {
      return false;
   }
   result.reserve(len/2);
   char const *strOrig=str;
   unsigned char val = 0;
   while (len > 0) {
      unsigned char chr=*str;
      if ((chr >= '0') && (chr <= '9')) {
         val =static_cast<unsigned char> (val + chr - '0');
      } else if ((chr >= 'a') && (chr <= 'f')) {
         val =static_cast<unsigned char> (val + chr + 10 - 'a');
      } else if ((chr >= 'A') && (chr <= 'F')) {
         val =static_cast<unsigned char> (val + chr + 10 - 'A');
      } else {
         SWU_ASSERT_RETURN_FALSE_ALWAYS();
      }
      ++str;
      --len;
      if (len & 1) {
         val = static_cast<unsigned char>(val << 4);
      } else {
         result.push_back(val);
         val = 0;
      }
   }
#ifndef LOCAL_BUILD_
   ETG_TRACE_USR1(("hexToBin:hex:0x%32s bin=%02x", 
                   strOrig,
                   ETG_LIST_LEN(static_cast<unsigned int> (len/2)), ETG_LIST_PTR_T8(&result[0])));
#endif
   
   return true;
}

bool hexToBin(const char *str, size_t len, unsigned char *result) {
   SWU_ASSERT_RETURN_VAL(str, false);
   std::vector<unsigned char > resultVector;
   bool res=hexToBin(str, len, resultVector);
   if (!res) {
      return false;
   }
   memcpy(result, &resultVector[0], resultVector.size());

   return res;

}

tU8 bcdToU8(tU8 u8Bcd) {
   tU8 bcdHigh=static_cast<tU8>(u8Bcd>>4);
   tU8 bcdLow=(u8Bcd & 0x0F);
   return static_cast<tU8>(10 * bcdHigh + bcdLow);
}

unsigned int extractUint(char const* array)
{
   return (unsigned int)array[0] | ((unsigned int)array[1] << 8) |
      ((unsigned int)array[2] << 16) | (((unsigned int)array[3]) << 24);
}

std::string extractLine(std::string &buffer)
{
   std::string result = getLine(buffer);
   size_t resultSize = result.size();
   if(0 != resultSize) {
      buffer.assign(buffer.c_str() + resultSize, buffer.size() - resultSize);
   }
   return result;
}


bool extractLine(std::string &buffer, std::string &line)
{
   if (!buffer.size()) {
      return false;
   }
   line=extractLine(buffer);
   return true;
}

bool cmdToLines(std::string cmd, std::vector<std::string> &lines, bool trim) {
   
   lines.clear();
   TempFile tempFile;
   std::string logFileName=tempFile.get();
   if (execCommand(cmd + " > " + logFileName)) {      
      return false;
   }
   std::string fileContent;
   if (!loadFile(logFileName, fileContent)) {
      ETG_TRACE_ERR(("cmdToLines::store(): could not load=%s", 
                     logFileName.c_str()));      
      return false;
   }

   std::string line;
   while (extractLine(fileContent, line)) {      
      if (trim) {
         trimString(line);         
         if (line.empty()) {
            continue;
         }
      }     
      lines.push_back(line);
   }  
   return true;
}


bool purifyString(std::string &str, bool truncate) {
   bool changed=false;
   std::string res;
   for (std::string::iterator iter= str.begin(); iter!=str.end(); ++iter) {
      char val=*iter;
      
      if (!val) {
         ETG_TRACE_USR4(("purifyString: found 0"));
         changed=true;
         break;
      }
      else if (((unsigned char)val < 32) || ((unsigned char)val>127)) {
         ETG_TRACE_USR4(("purifyString: replace chr=%u with X", val));
         changed =true;
         if (truncate) {
            break;
         } else {
            res+='X';
         }
      }
      else {
         res+=val;
      }
   }
   if (changed) {
      ETG_TRACE_USR4(("purifyString: changed to %s", str.c_str()));
      str=res;
   }
   return changed;
}

bool streq(const char *a, const char *b) {
   return 0==strcmp(a, b);
}

tUInt stringToUInt(std::string val) {
   return (tUInt)strtoul(val.c_str(), 0, 10);
}

string intToString(tUInt val) {
   string res;
   std::stringstream s;
   s << val;
   return s.str();

}

string u64ToString(int64_t val) {
   string res;
   std::stringstream s;
   s << val;
   return s.str();
}

std::string toString (const std::vector<SWU_BYTE>& data)
{
   std::string out;
   out.assign(reinterpret_cast<const char*>(&data[0]), data.size());
   return out;
}

std::set<std::string> splitString(std::string strIn, char delim) {
   std::set<string> strings;
   istringstream f(strIn);
   std::string cur;    
   while (getline(f, cur, delim)) {
      trimString(cur);
      strings.insert(cur);
   }
   return strings;
}

std::vector<std::string> stringToVector(std::string strIn, char delim) {
   std::vector<string> strings;
   istringstream f(strIn);
   std::string cur;    
   while (getline(f, cur, delim)) {
      trimString(cur);
      strings.push_back(cur);
   }
   return strings;
}

tUInt replaceSubString(std::string &where, const std::string &what, const std::string &with){
   size_t pos=0;
   tUInt numFound=0;
   ETG_TRACE_USR1(("replaceSubString:where=%50s, what=%30s with=%s",
                   where.c_str(), what.c_str(), with.c_str()));

   while((pos = where.find(what, pos)) != std::string::npos){
      ETG_TRACE_USR1(("replaceSubString:found occurence of %s", what.c_str()));
      where.replace(pos, what.size(), with);
      pos += with.size();
      numFound++;
   }
   return numFound;
}

tUInt getUIntFromChild(TiXmlElement const *pElem, char const *childName, tUInt defaultRes) {
   const char* txt=getTextFromChild(pElem, childName, false);
   if (!txt) {
      return defaultRes;
   }
   tUInt res=atoi(txt);
   return res;
}

tUInt getNumChildren(TiXmlElement *parent, string childName) {
   tUInt res=0;
   for (TiXmlElement *current=parent->FirstChildElement(childName.c_str()); 
        current;
        current=current->NextSiblingElement(childName.c_str())) {
      ++res;
   }
   return res;
}

tUInt getNumChildrenWithTxtChild(TiXmlElement *parent, std::string childName, std::string withElem, std::string withText)  {
   tUInt res=0;
   for (TiXmlElement *current=parent->FirstChildElement(childName.c_str()); 
        current;
        current=current->NextSiblingElement(childName.c_str())) {
      char const *foundText=swu::getTextFromChild(current, withElem.c_str(), false, 0);
      if (foundText && withText==foundText) {
         ++res;
      }
   }
   return res;
}

TiXmlElement *getNthChild(TiXmlElement *parent, string childName, tUInt index) {
   tUInt res=0;
   TiXmlElement *current=parent->FirstChildElement(childName.c_str());
   for (unsigned int i=0; 
        current && i<index;
        ++i) {
      current=current->NextSiblingElement(childName.c_str());
   }
   return current;
}


const char *getTextFromChild(TiXmlElement const *pElem, char const *childName,
                             bool assert, char const *defaultRes) {

   
   if (assert) {
      SWU_ASSERT_RETURN_VAL(pElem, defaultRes);
      const char *parentName = pElem->Value();
      ETG_TRACE_USR4(("getTextFromChild:parentName=%20s, childName=%s",
                      parentName, childName));
      TiXmlElement const *child = pElem->FirstChildElement(childName);
      SWU_ASSERT_RETURN_VAL(child, defaultRes);
      const char *text = child->GetText();
      SWU_ASSERT_RETURN_VAL(text, defaultRes);
      ETG_TRACE_USR4(("getTextFromChild: res='%s'", text));

      return text;
   } else {
      if (!pElem || !childName) {
         ETG_TRACE_ERR(("getTextFromChild was called for Null"));
         return defaultRes;
      }
      const char *parentName = pElem->Value();
      ETG_TRACE_USR4(("getTextFromChild:parentName=%20s, childName=%s",
                      parentName, childName));
      TiXmlElement const *child = pElem->FirstChildElement(childName);
      if (!child) {
         ETG_TRACE_USR4(("getTextFromChild was called for an Element without subelement %30s, returning default:%s", childName, defaultRes));
         return defaultRes;
      }
      const char *text = child->GetText();
      
      if (text) {
         ETG_TRACE_USR4(("getTextFromChild: res='%s'", text));
         return text;
      } else {
         ETG_TRACE_USR4(("getTextFromChild was called for an Element with empty subelement %s", childName));
         return defaultRes;
      }
   }
}

void removeChildren( TiXmlElement *parent, std::string const childName) {
   ETG_TRACE_USR4(("removeChildren %s START", childName.c_str()));

   while (TiXmlElement *child = parent->FirstChildElement(childName.c_str())) {
      ETG_TRACE_USR4(("found and remove child!"));
      parent->RemoveChild(child);
   }      
   ETG_TRACE_USR4(("removeChildren %s END", childName.c_str()));
}


bool getTextFromChildAsBool( TiXmlElement const *pElem, char const *childName) {
   return stringToBool(getTextFromChildOrEmpty(pElem, childName));
}

TiXmlElement *addTextChild( TiXmlElement *pElem, string const childName,  string const childText) {
   return addTextChild(pElem, childName.c_str(), childText.c_str());
}

TiXmlElement *addTextChild( TiXmlElement *parent, char const *childName,  char const *childText) {
   ETG_TRACE_USR1(("addTextChild:childName=%30s, text=%s", childName, childText));
   TiXmlElement *child = new TiXmlElement(childName);
   SWU_ASSERT_RETURN_VAL(child, 0);
   ETG_TRACE_USR4(("Parent: %s", parent->Value()));
   TiXmlText *text = new TiXmlText(childText);
   child->LinkEndChild(text);
   parent->LinkEndChild(child);
   text = 0;
   return child;
}


bool addNodeToAll(TiXmlNode *root, const char *toNode, TiXmlElement &extraNode)
{
   if((0 == root) || (0 == toNode)) return false;
   for (TiXmlNode* node = root->FirstChild(); node; node = node->NextSibling())
   {
      if(0 == strcmp(node->Value(), toNode))
      {
         if(0 == node->FirstChild()) return false;
         node->InsertEndChild(extraNode);
      }
      if (false == addNodeToAll(node, toNode, extraNode)) return false;
   }
   return true;
}

void storeTiXmlElement(TiXmlElement *xmlElement, string pathAndName) {
   if (!xmlElement) {
      return;
   }
   TiXmlDocument xmlDoc(pathAndName.c_str());
   xmlDoc.InsertEndChild(*xmlElement);
   xmlDoc.SaveFile();
}


void setTextChild( TiXmlElement* parent , ::std::string childName, ::std::string childText ) {
   setTextChild(parent, childName.c_str(), childText.c_str());
}

void setTextChild( TiXmlElement* parent , const char *childName, const char *childText ) {
   ETG_TRACE_USR4(("setTextChild(%15s)=%s", childName, childText));
   TiXmlElement* childElem ;
   childElem =parent->FirstChildElement(childName);
   if (childElem) {
      //      ETG_TRACE_USR4(("setTextChild remove old"));
      // remove text-child
      parent->RemoveChild(childElem);
   }

   if (*childText!=0) {
      // add the text-child
      addTextChild(parent, childName, childText);
   }
}




void addUIntChild( TiXmlElement *parent, char const *childName,  tUInt val){
   char str[20];
   sprintf(str,"%d",val);
   addTextChild(parent, childName, str);
}

void setUIntChild( TiXmlElement *parent, char const *childName,  tUInt val){
   char str[20];
   sprintf(str,"%d",val);
   setTextChild(parent, childName, str);
}


bool bFindPatternInString(string pattern, string str) {
#ifdef LOCAL_BUILD_
   return false;
#else
   // compile regexp
   regex_t           regex;
   if(regcomp(&regex, pattern.c_str(), REG_NOSUB | REG_EXTENDED )) {
      ETG_TRACE_USR4(("bFindStringInFile:Unable to parse regexp %s", pattern.c_str()));
      return false;
   }
   bool matched=false;
   if (!regexec (&regex, str.c_str(), 1, (regmatch_t*)0, 0)) {
      matched=true;
   }

   regfree(&regex);
   ETG_TRACE_USR4(("bFindPatternInString matched=%u pattern=%30s  string=%s", 
                   matched, pattern.c_str(), str.c_str()));
   return matched;
#endif
}


bool bFindPatternInFile(string pattern, string fileName) {
#ifdef LOCAL_BUILD_
   return false;
#else
   string line;
   ifstream inFile;
   inFile.open(fileName.c_str());
   ETG_TRACE_USR4(("bFindStringInFile:pattern=%s", pattern.c_str()));

   if(!(inFile.is_open())) {
      ETG_TRACE_USR4(("bFindStringInFile:Unable to open %s", fileName.c_str()));
      return false;
   }

   // compile regexp
   regex_t           regex;
   if(regcomp(&regex, pattern.c_str(), REG_NOSUB | REG_EXTENDED )) {
      ETG_TRACE_USR4(("bFindStringInFile:Unable to parse regexp %s", pattern.c_str()));
      return false;
   }
   bool matched=false;
   while(!inFile.eof()) {
      getline(inFile,line);
         ETG_TRACE_USR4(("bFindStringInFile check line %s", line.c_str()));
      if (!regexec (&regex, line.c_str(), 1, (regmatch_t*)0, 0)) {
         ETG_TRACE_USR4(("bFindStringInFile Found in line %s", line.c_str()));
         matched=true;
         break;
      }
   }
   regfree(&regex);
   inFile.close();
   return matched;
#endif
}

::std::string getMediaType(::std::string mediaPath) {
   std::string tmp_file = getTempFile();
   std::string cmd = "mount | grep " + mediaPath + " | cut -d ' ' -f 5 > "+ tmp_file;
   std::string mediaType;

   if (system(cmd.c_str())) {
      ETG_TRACE_USR1(("getMediaType(), cmd failed to execute:%s", cmd.c_str()));
      return mediaType;
   }
   if ((!tmp_file.empty()) && (isFile(tmp_file))) {
      if (loadFile(tmp_file, mediaType)) {
         trimString(mediaType);
      }
   }
   ETG_TRACE_USR1(("mediaType is %s", mediaType.c_str()));
   return mediaType;
}

void uppercase(::std::string& str) {
   ::std::transform(str.begin(), str.end(), str.begin(), ::toupper);
}

void lowercase(::std::string& str) {
   ::std::transform(str.begin(), str.end(), str.begin(), ::tolower);
}

std::string toLower(char const *inChr) {
   SWU_ASSERT_RETURN_VAL(inChr, "");
   std::string resStr(inChr);
   std::transform(resStr.begin(), resStr.end(), resStr.begin(), ::tolower);
   return resStr;
}

bool stringToBool(std::string val) {
   swu::lowercase(val);
   if (atoi(val.c_str())) {
      ETG_TRACE_USR1(("stringToBool(%s): true (int)", val.c_str()));
      return true;
   }
   if (val == "true" || val == "yes" || val == "y") {
      ETG_TRACE_USR1(("stringToBool(%s): true (string)", val.c_str()));
      return true;
   }
   return false;

}


bool hasSuffix(const ::std::string& file, const ::std::string& suffix) {
   size_t sep = file.find_last_of(".");
   if (sep == std::string::npos) {
      return false;
   }
   return (0 == file.compare(sep+1, ::std::string::npos, suffix));
}

std::string &trimString(string &str) {
   // trim trailing spaces
   size_t endpos = str.find_last_not_of(" \t\n\r");
   if( string::npos != endpos )
   {
      str = str.substr( 0, endpos+1 );
   }
   
   // trim leading spaces
   size_t startpos = str.find_first_not_of(" \t\n\r");
   if( string::npos != startpos )
   {
      str = str.substr( startpos );
   }
   return str;
}

string uIntToString(tUInt val) {
   std::stringstream ss;
   std::string s;
   ss << val;
   s = ss.str();
   
   return s;
}

tVoid XmlElemContainer::add(string const &name, TiXmlElement const *xmlParent) {
   TiXmlElement const *xmlElem=xmlParent->FirstChildElement(_elemType.c_str());
   if (xmlElem) {
      _elems[name]=xmlElem;
   }
}

TiXmlElement const * XmlElemContainer::pull(string const &name) {
   map<string, TiXmlElement const *>::iterator iter=_elems.find(name);
   if (iter != _elems.end()) {
      TiXmlElement const *res=(*iter).second;
      _elems.erase(iter);
      return res;
   }
   return 0;
}

TiXmlElement const * XmlElemContainer::get(string const &name) {
   map<string, TiXmlElement const *>::iterator iter=_elems.find(name);
   if (iter != _elems.end()) {
      TiXmlElement const *res=(*iter).second;
      return res;
   }
   return 0;
}

std::string getLine(std::string buffer)
{
   std::string result;
   if(0 != buffer.size())
   {
      char *nextLF;
      if (0 == (nextLF = (char *)memchr(buffer.c_str(), Constants::Ascii::LF, buffer.size()))) {
         result = buffer;
      }
      else {
         result.assign(buffer.c_str(), nextLF - buffer.c_str() + 1);
      }
   }
   return result;
}

tBool checkAndConnectToNetwork(std::string path, std::string localMountPath) {
   if (swu::execCommand("mkdir -p " + localMountPath)) {
      ETG_TRACE_ERR(("checkAndConnectToNetwork:mkdir %s failed", localMountPath.c_str())) 
      return false;
   }

   std::string cmd = "mount -t nfs -o proto=tcp,port=2049 " + path + " " + localMountPath;
   for(tU8 retry=0; retry<5; ++retry) {
      if (! execCommand(cmd) ) {
         return true;
      }
   }
   ETG_TRACE_ERR(("checkAndConnectToNetwork:command %s failed", cmd.c_str())) 
   return false;
}

}

