#include <cstdio>
#include "swu_uin.hpp"

#include "util/swu_constants.hpp"
#include "util/swu_sourceNor.hpp"
#include "util/swu_sourceSdc.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_util.hpp"

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_uin.cpp.trc.h"
#endif 

namespace swu
{
// short name for constants class
typedef swu::Constants::SDC SDC;
typedef swu::Constants::Mmc MMC;


// --------------------------------------------------------------- class UinIf
std::string UinIf::getUin () 
{
   if (!load()) {
      ETG_TRACE_ERR(("No UIN loaded. Returning empty string."));
      return "";
   }
   return _uin;
}

// --------------------------------------------------------------- class NORUin
NORUin::NORUin()
{ }

bool NORUin::load()
{
   if (_loaded) {
      return true;
   }

   SourceNOR nor;
   bool readSuccessfully = false;
   std::vector<SWU_BYTE> data;

   ETG_TRACE_USR3(( "Trying to read UIN from first memory location." ));
   readSuccessfully = nor.read(
         data,
         MMC::MTD_TARGET_UIN1_OFFSET,
         MMC::MTD_UIN_BUFFER_SIZE);
   if (readSuccessfully and setUin(data)) {
      _loaded = true;
      return true;
   }

   ETG_TRACE_USR3(( "Trying to read UIN from second memory location." ));
   readSuccessfully = nor.read(
         data,
         MMC::MTD_TARGET_UIN2_OFFSET,
         MMC::MTD_UIN_BUFFER_SIZE);
   if (readSuccessfully and setUin(data)) {
      _loaded = true;
      return true;
   }

   ETG_TRACE_ERR(("Can't read UIN from NOR."));
   return false;
}

bool NORUin::setUin (const std::vector<SWU_BYTE>& data)
{
   std::vector<SWU_BYTE>::const_iterator it = data.begin();
   for ( ; it != data.end() and *it == MMC::MTD_ERASED_MASK; ++it );
   if(it == data.end()) {
      ETG_TRACE_USR3(("UIN is erased (all bytes == 0xFF)")); 
      return false;
   }

   it = data.begin();
   for ( ; it != data.end() and *it == MMC::MTD_ZEROED_MASK; ++it );
   if(it == data.end()) {
      ETG_TRACE_USR3(("UIN is zeroed (all bytes == 0x00)"));
      return false;
   }

   _uin = toString(data);

   return true;
}

// --------------------------------------------------------------- class SdcUin
SdcUin::SdcUin()
{ }

bool SdcUin::load()
{
   if (_loaded) {
      return true;
   }

   SourceSdc sdc;
   std::vector<SWU_BYTE> data;

   ETG_TRACE_USR1(("Loading UIN from : %s", SDC::PATH_UIN));
   if ( ! swu::exists(SDC::PATH_UIN) ) {
      ETG_TRACE_ERR(("Can't find UIN file: %s", SDC::PATH_UIN));
      return false;
   }

   if (! sdc.read(data, SDC::PATH_UIN) ) {
      ETG_TRACE_ERR(( "SDC can't extract the UIN from file %s", SDC::PATH_UIN));
      return false;
   }

   _uin = toString(data);

   _loaded = true;
   return true;
}

// --------------------------------------------------------------- class FileUin
FileUin::FileUin()
{ }

bool FileUin::load()
{
   if (_loaded) {
      return true;
   }

   if ( ! swu::exists(SDC::PATH_UIN) ) {
      ETG_TRACE_ERR(("Can't find UIN file: %s", SDC::PATH_UIN));
      return false;
   }

   _loaded = swu::loadFile(SDC::PATH_UIN, _uin);
   if (! _loaded ) {
      ETG_TRACE_ERR(( "Can't read the UIN from file %s", SDC::PATH_UIN));
   }

   return _loaded;
}

}
