/*
 * swu_securityEngine.cpp
 *
 *  Created on: Jul 30, 2015
 *      Author: sof4hi
 */

#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <regex.h>
#include <cmath>
#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <algorithm>

#include "util/swu_certificate.h"
#include "util/swu_securityEngine.h"
#include "util/swu_targetKey.h"
#include "util/swu_trace.h"
#include "util/swu_util.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_securityEngine.cpp.trc.h"
#endif

namespace swu {

using namespace std;

static ::std::vector<SWU_BYTE> extractDigest(const EVP_MD_CTX *context);

class InputFileHandle {
public:
   enum Mode {
      MODE_UNKNOWN,
      MODE_STDIN,
      MODE_FILE,
      MODE_TFTP
   };

private:
   Mode _mode;
   bool _open;
   int _fd;
   string _inputFileName;
   static const unsigned int  READ_INDEX=0;
   static const unsigned int WRITE_INDEX=1;

   InputFileHandle ( const InputFileHandle & );  //Coverity fix for 48247
   InputFileHandle & operator = ( const InputFileHandle & );  //Coverity fix for 47818

   /** Detects which mode to use internally:
    *  - standard input,
    *  - file or
    *  - TFTP. */
   void detectMode () {
      if (_inputFileName=="-" || _inputFileName=="") {
         _mode = MODE_STDIN;
      }
      else if ( 0 == strstr(_inputFileName.c_str(), ":") ) {
         _mode = MODE_FILE;
      }
      else {
         _mode = MODE_TFTP;
      }
   }

   // Parse the FTP address and fill pathAndName and ipAdr accordingly.
   bool parseTftpAddress(string &pathAndName, string &ipAdr) {
      ETG_TRACE_USR4(("InputFileHandle:isTftpFile() start"))

      string pattern="^/tftp@([0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}:)(.+)";
      regex_t regex;
      pathAndName="";
      ipAdr="";

      if(0 != regcomp(&regex, pattern.c_str(), REG_EXTENDED )) {
         ETG_TRACE_ERR(("tftpFileName : Unable to compile pattern %s", pattern.c_str()));
         return false;
      }

      regmatch_t pmatch[3];
      if (0 != regexec (&regex, _inputFileName.c_str(), 3, (regmatch_t*)pmatch, 0)) {
         ETG_TRACE_FATAL(("tftpFileName : Unable to match regexec %s", _inputFileName.c_str()));
         return false;
      }
      ipAdr=_inputFileName.substr(pmatch[1].rm_so, (pmatch[1].rm_eo - pmatch[1].rm_so) - 1);
      pathAndName=_inputFileName.substr(pmatch[2].rm_so, pmatch[2].rm_eo - pmatch[2].rm_so);
      ETG_TRACE_USR4(("parsed string = %s", pathAndName.substr(pmatch[0].rm_so, pmatch[0].rm_eo - pmatch[0].rm_so).c_str()));
      ETG_TRACE_USR4(("parsed ipAdr = %s", ipAdr.c_str()));
      ETG_TRACE_USR4(("parsed pathAndName = %s", pathAndName.c_str()));
      return true;
   }

   // code for the forked child:
   // deal with redirection of fds and execute the tftp-command
   int getTftpInputFd(string pathAndName, string ipAdr) {
      ETG_TRACE_USR4(("InputFileHandle:getTftpInputFd() start"))

       _fd=-1;
       pid_t childpid;
       int pipeFds[2];
       if (pipe2(pipeFds, O_CLOEXEC) == -1) {
          ETG_TRACE_FATAL(("verifyAndCatenate: create pipe failed"));
          return -1;
       }
       childpid = ::fork(); /* fork a child */

       if(childpid) {   /* we are in  the parent */
          ETG_TRACE_USR4(("InputFileHandle:getTftpInputFd(PARENT) START"))

          if(::close(pipeFds[WRITE_INDEX])) {/*close unused file descriptor*/
             ETG_TRACE_ERR(("InputFileHandle:getTftpInputFd(PARENT): Failed to close output pipe descriptor"));
          }
          if(-1 == childpid) { // an error occured, no child
             ETG_TRACE_FATAL(("InputFileHandle:getTftpInputFd(PARENT): failed spawn child"))
             // close pipe-input, since there is no child to write to the pipe.
             ::close(pipeFds[READ_INDEX]);
             ETG_TRACE_FATAL(("Failed to fork a child\n"));
             return -1;
          }

          ETG_TRACE_USR4(("InputFileHandle:getTftpInputFd(PARENT):  spawn child, reading from fd=%d", pipeFds[READ_INDEX]))
          /* child is ok, return fd where data can be received from child */
          return pipeFds[READ_INDEX];
       }

       ::system("echo InputFileHandle:getTftpInputFd CHILD START >> /tmp/downloadPipe");

       // we are in  the child
       // no need for input
       bool allOk=true;
       if(::close(pipeFds[READ_INDEX])) {
          ::system("echo InputFileHandle:getTftpInputFd CHILD: Failed to close input pipe descriptor >> /tmp/downloadPipe");
       }

       if(STDOUT_FILENO != ::dup2(pipeFds[WRITE_INDEX], STDOUT_FILENO)) {
          // make pipe-output STDOUT
          ::system("echo InputFileHandle:getTftpInputFd CHILD: Failed to redirect child stdout to pipe-output >> /tmp/downloadPipe");
          // and close out-pipe
          allOk=false;
       }

       // close WRITE_INDEX, since we have it now as STDOUT
       if(::close(pipeFds[WRITE_INDEX])) {
          ::system("echo InputFileHandle:getTftpInputFd CHILD: Failed to close output pipe descriptor >> /tmp/downloadPipe");
       }

       if (allOk) {
          ::system("echo InputFileHandle:getTftpInputFd CHILD: call tftp >> /tmp/downloadPipe");
#if 0
          ::system("echo InputFileHandle:getTftpInputFd CHILD: call tftp(%30s:%s)",
                   ipAdr.c_str(),
                   pathAndName.c_str());
#endif
          //          ::execlp("tftp", "tftp", "-g", "-l", "-", "-r", pathAndName.c_str(), ipAdr.c_str(), NULL); /*execute env*/

          // execlp :  _exec_ute the command, with a 
          //           _l_ist of parameters, looking for the command in the
          //           _p_ath defined by the current environment.
          const char* command = "swu_common_tftp_client_out.out";
          ::execlp( command, command, "-f", pathAndName.c_str(), "-i", ipAdr.c_str(), NULL);
          while(1) {
             ::system("echo TFTP process running >> /tmp/downloadPipe");
             ::sync(); ::sync();
             ::sleep(1000);
          }
       }
       while(1) {
          ::system("echo Error in TFTP process >> /tmp/downloadPipe");
          ::sync(); ::sync();
          ::sleep(1000);
       }
       // should never get here
       return -1;
    }

public:

   InputFileHandle(string inputFileName):
      _mode(MODE_UNKNOWN),
      _open(false),
      _fd(-1),
      _inputFileName(inputFileName)
   {
      ETG_TRACE_USR2(("InputFileHandle:CTOR inputFileName=%s", inputFileName.c_str()));
      detectMode();
   }

   ~InputFileHandle() {
      close();
   }

   int getFd() {
      return _fd;
   }

   Mode mode () {
      return _mode;
   }

   void close() {
      ETG_TRACE_USR2(("InputFileHandle:close() _open=%u", _open));
      if (_open) {
         _open = false;
         if (STDIN_FILENO != _fd) {
            ::close(_fd);
         }
      }
   }

   /** @brief Generic open method.
    *
    * Try to assign an input-stream to _fd according to _mode.
    */
   bool open() {
      ETG_TRACE_USR2(("InputFileHandle:open() START"));

      close();

      switch ( _mode ) {
         case MODE_STDIN:
         {
            ETG_TRACE_USR2(("InputFileHandle:open() MODE_STDIN"));
            _fd   = STDIN_FILENO;
            _open = true;
            break;
         }
         case MODE_FILE:
         {
            ETG_TRACE_USR2(("InputFileHandle:open() MODE_FILE"))
            _fd = ::open(_inputFileName.c_str(), O_RDONLY);
            if (-1 == _fd) {
               ETG_TRACE_FATAL(("verifyAndCatenate : Can't open input file"));
            } else {
               _open = true;
            }
            break;
         }
         case MODE_TFTP:
         {
            ETG_TRACE_USR2(("InputFileHandle:open() MODE_TFTP"))
            string pathAndName;
            string ipAdr;
            if (! parseTftpAddress(pathAndName, ipAdr) ) {
               ETG_TRACE_FATAL(("InputFileHandle::open(): Can't parse TFT address."));
            }
            _fd=getTftpInputFd(pathAndName, ipAdr);
            if (-1 == _fd) {
               ETG_TRACE_FATAL(("InputFileHandle::open(): Can't open TFTP address"));
            }
            _open = true;
            break;

         }
         default:
               ETG_TRACE_FATAL(("InputFileHandle::open(): Can't open, mode unknown."));
      }
      return _open;
   }

   /** Return the size of the underlying file.
    *
    * @return -1 if the filesize is unknown, otherwise a positive number
    *         representing the filesize in byte.
    */
   long long int size ()
   {
      long long int s = -1;
      if ( ! _open) { return s; }

      if ( _mode == MODE_FILE) {
         struct stat file_stat;
         if (::fstat(_fd, &file_stat) < 0) {
            ETG_TRACE_ERR(("Can't stat file."));
            return s;
         }
         s = file_stat.st_size;
      }

      return s;
   }
};

SecurityEngine::SecurityEngine() :
      _encryptParam(""),
      _enDigestType(swu::tenDigestTypeNone),
      _blockLength(0),
      _fileLength(0),
      _digests() {;}

SecurityEngine::SecurityEngine(const SecurityEngine &copyFrom) :
            _encryptParam(copyFrom._encryptParam),
            _enDigestType(copyFrom._enDigestType),
            _blockLength(copyFrom._blockLength),
            _fileLength(copyFrom._fileLength),
            _digests(copyFrom._digests) {}

bool SecurityEngine::readParameters(const std::string &file) {
   TiXmlDocument doc;
   if(false == doc.LoadFile(file.c_str())) {
      ETG_TRACE_ERR(("Could not load Security Parameters file %s", file.c_str()));
      ETG_TRACE_ERR((" because %s", doc.ErrorDesc()));
      return false;
   }
   if(false == readFromXml(doc.FirstChildElement(SECURITY_PARAM().c_str()))) {
      ETG_TRACE_ERR(("Could not fill Security Parameters from XML"));
      return false;
   }
   return true;
}

bool SecurityEngine::writeParameters(const std::string &file) {
   TiXmlDocument doc;
   doc.InsertEndChild(toXml());
   if(0 == doc.SaveFile(file.c_str())) {
      ETG_TRACE_ERR(("Could not store Security Parameters file %s", file.c_str()));
      return false;
   }
   return true;
}

bool SecurityEngine::readFromXml(TiXmlElement *e) {
   if(0 == e) {
      ETG_TRACE_ERR(("readFromXml Can't read data from NULL"));
      return false;
   }
   const char *t;
   TiXmlElement *i = e->FirstChildElement(ENCRYPT_PARAM().c_str());
   if((0 == i) || (0 == (t = i->GetText())))
   {
      ETG_TRACE_USR2(("Security Parameters do not contain ENCRYPT_PARAM, use empty"));
      _encryptParam = "";
   }
   else _encryptParam = t;
   if(0 == (i = e->FirstChildElement(DIGEST_TYPE().c_str()))) {
      ETG_TRACE_USR3(("Security Parameters has no %s, use tenDigestTypeNone", DIGEST_TYPE().c_str()));
      _enDigestType = swu::tenDigestTypeNone;
   }
   else if(0 == (t = i->GetText())) {
      ETG_TRACE_USR3(("%s contains no data, use tenDigestTypeNone", DIGEST_TYPE().c_str()));
      _enDigestType = swu::tenDigestTypeNone;
   }
   else
   {
      if(0 == DIGEST_TYPE_NONE().compare(t))          _enDigestType = swu::tenDigestTypeNone;
      else if(0 == DIGEST_TYPE_MD5().compare(t))      _enDigestType = swu::tenDigestTypeMD5;
      else if(0 == DIGEST_TYPE_SHA1().compare(t))     _enDigestType = swu::tenDigestTypeSHA1;
      else if(0 == DIGEST_TYPE_SHA256().compare(t))   _enDigestType = swu::tenDigestTypeSHA256;
      else {
         ETG_TRACE_ERR(("unknown digest type %s", t));
         return false;
      }
   }

   if(0 == (i = e->FirstChildElement(BLOCK_SIZE().c_str()))) {
      ETG_TRACE_USR3(("Security Parameters do not contain %s, use 0", BLOCK_SIZE().c_str()));
      _blockLength = 0;
   }
   else if((0 == (t = i->GetText())) || (1 != sscanf(t, "%ju", &_blockLength))) {
      ETG_TRACE_ERR(("%s contains no data", BLOCK_SIZE().c_str()));
      return false;
   }

   if(0 == (i = e->FirstChildElement(FILE_SIZE().c_str()))) {
      ETG_TRACE_ERR(("Security Parameters do not contain %s", FILE_SIZE().c_str()));
      return false;
   }
   if((0 == (t = i->GetText())) ||(1 != sscanf(t, "%ju", &_fileLength))) {
      ETG_TRACE_ERR(("%s contains no data", FILE_SIZE().c_str()));
      return false;
   }

   TiXmlElement *c = e->FirstChildElement(DIGEST_SERIES().c_str());
   if (c) {
      for (i = c->FirstChildElement(DIGEST_VALUE().c_str()); i != 0;
            i = i->NextSiblingElement(DIGEST_VALUE().c_str())) {
         if(0 == (t = i->GetText())) {
            ETG_TRACE_ERR(("Found an empty %s Field", DIGEST_VALUE().c_str()));
            return false;
         }
         ::std::vector< SWU_BYTE > v = decodeHex(t);
         if(0 == v.size()) {
            ETG_TRACE_ERR(("Could not convert %s to digest", t));
            return false;
         }
         _digests.push_back(v);
      }
   }
   return true;
}

TiXmlElement SecurityEngine::toXml() {
   ETG_TRACE_ERR(("SecurityEngine::toXml START"));
   ETG_TRACE_ERR(("SecurityEngine::toXml SECURITY_PARAM=%s", SECURITY_PARAM().c_str()));
   ETG_TRACE_ERR(("SecurityEngine::toXml ENCRYPT_PARAM()=%s", ENCRYPT_PARAM().c_str()));
   ETG_TRACE_ERR(("SecurityEngine::toXml _encryptParam=%s", _encryptParam.c_str()));

   TiXmlElement result(SECURITY_PARAM().c_str());
   swu::setTextChild(&result, ENCRYPT_PARAM(), _encryptParam);

   std::string sDigestType = DIGEST_TYPE_NONE();
   if(swu::tenDigestTypeSHA1 == _enDigestType)         sDigestType = DIGEST_TYPE_SHA1();
   else if(swu::tenDigestTypeSHA256 == _enDigestType)  sDigestType = DIGEST_TYPE_SHA256();
   else if(swu::tenDigestTypeMD5 == _enDigestType)     sDigestType = DIGEST_TYPE_MD5();
   swu::setTextChild(&result, DIGEST_TYPE(), sDigestType);

   char buff[1024];
   sprintf(buff, "%ju", _fileLength);
   swu::setTextChild(&result, FILE_SIZE().c_str(), buff);
   sprintf(buff, "%ju", _blockLength);
   swu::setTextChild(&result, BLOCK_SIZE().c_str(), buff);

   TiXmlElement series(DIGEST_SERIES().c_str());
   for (::std::vector< ::std::vector< SWU_BYTE > >::iterator it =
         _digests.begin(); it != _digests.end(); ++it) {
      char *pbuff = buff;
      for (::std::vector< SWU_BYTE >::iterator it2 = it->begin();
            it2 != it->end(); ++it2) {
         SWU_BYTE Digit = (*it2 & 0xf0) >> 4;
         if(Digit > 9) Digit =static_cast<swu::SWU_BYTE> (Digit + 'A' - 10);
         else Digit =static_cast<swu::SWU_BYTE> (Digit + '0');
         *pbuff++ = Digit;
         Digit = *it2 & 0x0f;
         if(Digit > 9) Digit =static_cast<swu::SWU_BYTE> (Digit + 'A' - 10);
         else Digit =static_cast<swu::SWU_BYTE> (Digit + '0');
         *pbuff++ = Digit;
         if(pbuff >= buff + sizeof(buff) - 2) {
            ETG_TRACE_ERR(("too long digest array"));
            return result;
         }
      }
      *pbuff = 0;
      TiXmlElement digestElement(DIGEST_VALUE().c_str());
      TiXmlText digestText(buff);
      digestElement.InsertEndChild(digestText);
      series.InsertEndChild(digestElement);
   }
   result.InsertEndChild(series);
   return result;
}

bool SecurityEngine::runThroughStream(int inFile, int *outFile, bool verify,
   EVP_MD_CTX *digestCtx, EVP_CIPHER_CTX *cipherCtx)
{
   if((false == verify) && (swu::tenDigestTypeNone != _enDigestType)
      && (_digests.size() != 0)) {
         ETG_TRACE_FATAL(("checksum vector has to be empty for checksum creation."));
         return false;
   }

   off_t bufferSize;
   if(0 != _blockLength) {
      bufferSize = _blockLength;
      if((0 != _fileLength) && (_fileLength < _blockLength)) bufferSize = _fileLength;
   }
   else { // we use read chunk 16 MB - smaller value influences the performance
      if((_fileLength == 0) || (_fileLength >= 16 * 1024 * 1024)) bufferSize = 16 * 1024 * 1024;
      else bufferSize = _fileLength;
   }
   char *buffer = new char [bufferSize];
   if(0 == buffer) {
      ETG_TRACE_FATAL(("Can't allocate memory for file reading"));
      return false;
   }
   SWU_BYTE* outbuf = new (std::nothrow) SWU_BYTE[2 * bufferSize + EVP_MAX_BLOCK_LENGTH]; //Coverity fix for 48976
   if(0 == outbuf) {
      ETG_TRACE_ERR(("Can't allocate cipher output buffer"));
      delete[] buffer;
      return false;
   }

   off_t readLength = 0;
   off_t act_block = 0;
   int cipherLen = 0;
   int cipherAddLen = 0;

   while (1) {
      int chunk = readChunk(inFile, buffer, bufferSize, readLength);
      if(-1 == chunk) {
         ETG_TRACE_ERR(("Could not read from given handle. Read until now %u chars", readLength));
         delete[] buffer;
         delete[] outbuf;
         return false;
      }
      readLength += chunk;
      if( ! EVP_CipherUpdate(cipherCtx, outbuf + cipherLen, &cipherAddLen, (SWU_BYTE*)buffer, (int)chunk)) {
         ETG_TRACE_ERR(("EVP_CipherUpdate failed, processing %10u chars at position %u",
                 chunk, readLength));
         delete[] buffer;
         delete[] outbuf;
         return false;
      }
      cipherLen += cipherAddLen;
      if(((readLength == _fileLength)) || ((0 == _fileLength) && (((0 == _blockLength) &&
         (chunk != bufferSize)) || ((0 != _blockLength) && (chunk != _blockLength)))))
      {
         if( ! EVP_CipherFinal_ex(cipherCtx, outbuf + cipherLen, &cipherAddLen)) {
            ETG_TRACE_ERR(("Error finalizing output"));
            delete[] buffer;
            delete[] outbuf;
            return false;
         }
         cipherLen += cipherAddLen;
      }

      int digestLen;
      if      (0 == _blockLength)         { digestLen = cipherLen; }
      else if (cipherLen >= _blockLength) { digestLen =static_cast<int> ( _blockLength ); }
      else if (readLength == _fileLength) { digestLen = cipherLen; }
      else                                { digestLen = 0; }

      while(digestLen) {
         if(1 != EVP_DigestUpdate(digestCtx, outbuf, digestLen)) {
            ETG_TRACE_ERR(("SecurityEngine::runThroughStream: Error updating Message digest at position %u", readLength));
            delete[] buffer;
            delete[] outbuf;
            return false;
         }
         // bool isNotZeroBlock     = (_blockLength != 0);
         // bool reachedEOF         = (readLength == _fileLength);
         // bool unknownFileLength  = (_fileLength == 0);
         // bool unknownBlockLength = (_blockLength == 0);

         if((0 != _blockLength) || (readLength == _fileLength) || ((0 == _fileLength) &&
            ((0 == _blockLength) && (chunk != bufferSize))))
         {
            ::std::vector< SWU_BYTE > digest_vector = extractDigest(digestCtx);
            if (true == verify) {
               if (swu::tenDigestTypeNone != _enDigestType) {
                  uint32_t actblock = static_cast<uint32_t> (act_block);
                  if((actblock >= _digests.size()) || (_digests[actblock] != digest_vector))
                  {
                     std::string digest_vector_hex = binToHex(digest_vector);
                     std::string digest_act_block_hex = binToHex(_digests[actblock]);
                     ETG_TRACE_ERR(("Digest calculated: %s", digest_vector_hex.c_str() ));
                     ETG_TRACE_ERR(("Digest read:       %s", digest_act_block_hex.c_str() ));
                     ETG_TRACE_ERR(("Verification Error, _digests[%2d]", act_block ));
                     delete[] buffer;
                     delete[] outbuf;
                     return false;
                  } else ETG_TRACE_USR4(("SecurityEngine::runThroughStream: block %d verified", act_block + 1));
               }
            }
            else {
               _digests.push_back(digest_vector);
               ETG_TRACE_USR4(("SecurityEngine::runThroughStream: block %d calculated", act_block + 1));
            }
            ++act_block;
         }
         if (true != writeChunk(outFile, outbuf, digestLen)) {
            ETG_TRACE_ERR(("SecurityEngine::runThroughStream: Could not write to output handle. Tried to write %d", cipherLen));
            delete[] buffer;
            delete[] outbuf;
            return false;
         }
         cipherLen -= digestLen;
         memmove(outbuf, outbuf + digestLen, digestLen);
         if(0 == _blockLength) digestLen = cipherLen;
         else
         {
            if(cipherLen >= _blockLength) digestLen =static_cast<int> ( _blockLength );
            else if (readLength == _fileLength) digestLen = cipherLen;
            else digestLen = 0;
         }
      }
      if(((readLength == _fileLength)) || ((0 == _fileLength) && (((0 == _blockLength) &&
         (chunk != bufferSize)) || ((0 != _blockLength) && (chunk != _blockLength))))) 
      {
         break;
      }
   }
   delete[] buffer;
   delete[] outbuf;
   return true;
}

static ::std::vector< SWU_BYTE > extractDigest(const EVP_MD_CTX *context) {
   SWU_BYTE digest_buffer[EVP_MD_CTX_size(context)];
   unsigned int digest_length;
   ::std::vector< SWU_BYTE > digest_vector;
   EVP_MD_CTX *backupContext = EVP_MD_CTX_create();
  if((0 == backupContext) || (1 != EVP_MD_CTX_copy(backupContext, context))) {
      ETG_TRACE_ERR(("extractDigest: Could not copy context"));
      return digest_vector;
   }
   if(1 != EVP_DigestFinal_ex(backupContext, digest_buffer, &digest_length)) {
      ETG_TRACE_ERR(("extractDigest: Could not finalize digest check"));
      EVP_MD_CTX_destroy(backupContext);
      return digest_vector;
   }
   EVP_MD_CTX_destroy(backupContext);
   digest_vector.assign(digest_buffer, digest_buffer + digest_length);
   return digest_vector;
}

bool SecurityEngine::writeChunk(int *handle, SWU_BYTE* buffer, off_t writeLength) {
   if ((0 == handle) || (0 == writeLength)) return true;
   off_t actualOutput, neededOutput =  writeLength;
   SWU_BYTE* pBuffer = buffer;
   while (neededOutput > (actualOutput = (write(*handle, pBuffer, static_cast<size_t>(neededOutput))))) {
      if(actualOutput < 0) {
         ETG_TRACE_ERR(("SecurityEngine::writeChunk Could not write %d bytes to output handle", writeLength));
         return false;
      }
      pBuffer += actualOutput;
      neededOutput -= actualOutput;
   }
   return true;
}

int SecurityEngine::readChunk(int handle, char *buffer, off_t bufferSize, off_t readLength) {
   off_t read_next;
   int readBytes = 0;
   char *pbuffer = buffer;
   if (0 != _fileLength) {
      if (0 != _blockLength) {
         read_next = min(_blockLength, _fileLength - readLength);
      } else  {
         read_next = min(bufferSize, _fileLength - readLength);
      }
   }  else {
      if (0 != _blockLength) {
         read_next = _blockLength;
      } else  {
         read_next = bufferSize;
      }
   }
   ETG_TRACE_USR4(("readChunk : read_next=%d", read_next));
   while (read_next > 0) {
      int chunk =static_cast<int> (read(handle, pbuffer, static_cast<size_t> (read_next)) );
      if(-1 == chunk) {
         ETG_TRACE_ERR(("readChunk: Could not read from given handle"));
         return -1;
      }
      if(0 == chunk) break;
      pbuffer += chunk;
      read_next -=  chunk;
      readBytes += chunk;
   }
   if(read_next == 0) return readBytes;
   if(0 == _fileLength) return readBytes;
   ETG_TRACE_ERR(("readChunk: Could not read enough data from given handle"));
   return -1;
}

EVP_CIPHER_CTX * SecurityEngine::initCipherContext()
{
   std::vector< SWU_BYTE > vectorKey;
   std::vector< SWU_BYTE > vectorIV;

   const EVP_CIPHER *cipher = EVP_enc_null();
   if(0 != _encryptParam.size()) {
      TiXmlDocument encryptXml;
      encryptXml.Parse(_encryptParam.c_str());
      if (encryptXml.Error()) {
         ETG_TRACE_ERR(("Tinyxml failed to parse decoded sym decrypt key %s", encryptXml.ErrorDesc()));
         return 0;
      }
      TiXmlElement *root = encryptXml.FirstChildElement(SYM_ENCRYPT_PARAM().c_str());
      if (0 == root) {
         ETG_TRACE_ERR(("initCipherContext: Could not find %s tag", SYM_ENCRYPT_PARAM().c_str()));
         printf("initCipherContext: Could not find %s tag\n", SYM_ENCRYPT_PARAM().c_str());
         return 0;
      }
      TiXmlElement *i;
      if(!(i = root->FirstChildElement(ENCRYPT_TYPE().c_str())) || !(i->GetText()))
      {
         ETG_TRACE_ERR(("Sym Security Parameters has no mandatory tag %s", ENCRYPT_TYPE().c_str()));
         printf("Sym Security Parameters has no mandatory tag %s\n", ENCRYPT_TYPE().c_str());
         return 0;
      }
      if(0 == strcmp(i->GetText(), ENCRYPT_TYPE_NONE().c_str())) cipher = EVP_enc_null();
      else if(0 == strcmp(i->GetText(), ENCRYPT_TYPE_AES_128_CBC().c_str()))    cipher = EVP_aes_128_cbc();
      else if(0 == strcmp(i->GetText(), ENCRYPT_TYPE_AES_192_CBC().c_str()))    cipher = EVP_aes_192_cbc();
      else if(0 == strcmp(i->GetText(), ENCRYPT_TYPE_AES_256_CBC().c_str()))    cipher = EVP_aes_256_cbc();
      else {      /* add further if needed... */
         ETG_TRACE_ERR(("initCipherContext: unsupported cipher %s", i->GetText()));
         printf("initCipherContext: unsupported cipher %s", i->GetText());
         return 0;
      }
      ETG_TRACE_USR3(("initCipherContext: cipher found"));
      if(EVP_enc_null() != cipher)
      {
         if(!(i = root->FirstChildElement(ENCRYPT_KEY().c_str())) || !(i->GetText()))
         {
            ETG_TRACE_ERR(("Sym Security Parameters has no mandatory tag %s", ENCRYPT_KEY().c_str()));
            printf("Sym Security Parameters has no mandatory tag %s\n", ENCRYPT_KEY().c_str());
            return 0;
         }
         vectorKey = decodeHex(i->GetText());
         if(!(i = root->FirstChildElement(ENCRYPT_IV().c_str())) || !(i->GetText()))
         {
            ETG_TRACE_ERR(("Sym Security Parameters has no mandatory tag %s", ENCRYPT_IV().c_str()));
            printf("Sym Security Parameters has no mandatory tag %s\n", ENCRYPT_IV().c_str());
            return 0;
         }
         vectorIV = decodeHex(i->GetText());
      }
   }
   if((int)vectorKey.size() != EVP_CIPHER_key_length(cipher)) {
      ETG_TRACE_ERR(("initCipherContext: Wrong cipher key length %d", vectorKey.size()));
      printf("initCipherContext: Wrong cipher key length %zd\n", vectorKey.size());
      return 0;
   }
   if((int)vectorIV.size() != EVP_CIPHER_iv_length(cipher)) {
      ETG_TRACE_ERR(("initCipherContext: Wrong cipher IV length %d", vectorIV.size()));
      printf("initCipherContext: Wrong cipher IV length %zd\n", vectorIV.size());
      return 0;
   }
   EVP_CIPHER_CTX *cipherCtx = EVP_CIPHER_CTX_new();
   if(1 != EVP_CipherInit_ex(cipherCtx, cipher, NULL, &vectorKey[0], &vectorIV[0], 0)) {
      ETG_TRACE_ERR(("initCipherContext: Can't initialize cipher context"));
      printf("initCipherContext: Can't initialize cipher context\n");
      return 0;
   }
   return cipherCtx;
}

bool SecurityEngine::processAndCatenate(const std::string &inFileName,
   const std::string &outFileName, bool doVerify)
{
   ETG_TRACE_USR2(("processAndCatenate : inFileName=%30s outFileName=%30s doVerify=%u",
                   inFileName.c_str(),
                   outFileName.c_str(),
                   doVerify));

   // Open input file and determine file size
   InputFileHandle inFd(inFileName);
   if ( ! inFd.open()) {
      ETG_TRACE_FATAL(("Can't open input file %s", inFileName.c_str() ));
      return false;
   }
   long long int filesize = inFd.size();
   if(-1 != filesize) {
      _fileLength = static_cast<off_t>(filesize);
   }

   int outputFile            = 0;
   bool result               = false;
   const EVP_MD *digest      = 0;
   EVP_MD_CTX *dgstCtx       = 0;
   EVP_CIPHER_CTX *cipherCtx = 0;

   do { // this loop emulates "goto cleanup;"
      if (outFileName == "-") {
         ETG_TRACE_USR2(("processAndCatenate : using stdout"));
         outputFile = STDOUT_FILENO;
         fflush(stdout);
      }
      else {
         outputFile = ::open(
               outFileName.c_str(),
               O_WRONLY | O_CREAT | O_TRUNC,
               S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
         if (-1 == outputFile) {
            ETG_TRACE_FATAL(("processAndCatenate : Can't open output file %s",
                     outFileName.c_str() ));
            result = false; break;
         }
      }

      switch (_enDigestType) {
         case swu::tenDigestTypeNone:   digest = EVP_md_null(); break;
         case swu::tenDigestTypeSHA1:   digest = EVP_sha1();    break;
         case swu::tenDigestTypeSHA256: digest = EVP_sha256();  break;
         case swu::tenDigestTypeMD5:    digest = EVP_md5();     break;
      }
      if (! digest) {
         ETG_TRACE_FATAL(("Unsupported digest type %d", _enDigestType));
         result = false; break;
      }

      dgstCtx = EVP_MD_CTX_create();
      if( ! dgstCtx) {
         ETG_TRACE_ERR(("Could not create digest context"));
         result = false; break;
      }
      if( ! EVP_DigestInit_ex(dgstCtx, digest, 0)) {
         ETG_TRACE_FATAL(("Could not initialize contexts"));
         result = false; break;
      }
      cipherCtx = initCipherContext();
      if( ! cipherCtx) {
         ETG_TRACE_ERR(("Failed to create Cipher Ctx"));
         result = false; break;
      }

      result = runThroughStream(inFd.getFd(), &outputFile, doVerify, dgstCtx, cipherCtx);
   } while (false);

   // cleanup:
   if(dgstCtx) EVP_MD_CTX_destroy(dgstCtx);
   if(cipherCtx) EVP_CIPHER_CTX_free(cipherCtx);
   
   if (outputFile != STDOUT_FILENO) {
      ::close(outputFile);
   }
   return result;
}

}
