/*
 * swu_robustFile.cpp
 *
 *  Implementation of RobustFile.
 */

#include "swu_robustFile.h"
//#include "swu_types.hpp"
#include "swu_filesystem.h"
#include "swu_execCommand.h"

#include <iostream>
#include <fstream>
#include <string>
#include <queue>
#include <sys/sendfile.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <libgen.h>

using std::string;

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_robustFile.cpp.trc.h"
#endif

//#define RF_PATCH
namespace swu {

const string RobustFile::k_DEFAULT_PATH_EXTENSION = "/.RobustFile";

RobustFile::RobustFile(string path) :
         _status(NOT_INITIALIZED), _file_path(path), _secure_area_path(), _explicit_secure_path(false), _file_copy_path(), _file_crc_path(), _file_crc_copy_path() {
}

RobustFile::RobustFile(string path, string secure_area_path) :
         _status(NOT_INITIALIZED), _file_path(path), _secure_area_path(secure_area_path), _explicit_secure_path(true), _file_copy_path(), _file_crc_path(), _file_crc_copy_path() {
}

bool RobustFile::init(bool lazy) {
   if (_status != NOT_INITIALIZED) {
      return true;
   }

   if (!_explicit_secure_path) {
      std::string parent_directory;
      if (!parentDirname(_file_path, parent_directory)) {
         ETG_TRACE_ERR(("Error: Could not get parent dirname of %s", _file_path.c_str()));
         return false;
      }
      _secure_area_path = parent_directory + k_DEFAULT_PATH_EXTENSION;
   }

   std::string name;
   size_t name_start = _file_path.find_last_of("/");
   if (name_start == string::npos) {
      name = _file_path;
   } else {
      name = _file_path.substr(name_start + 1);
      if (name.length() == 0) {
         ETG_TRACE_ERR(("Error: %s denotes a path, because last char is a / or it is an empty string...", _file_path.c_str()));
         return false;
      }
   }

   if (!lazy) {
      if (!makeDirectoryRecursive(_secure_area_path, k_DEFAULT_DIRMODE)) {
         ETG_TRACE_ERR(("Could not create directory for secure area %s", _secure_area_path.c_str()));
         return false;
      }
   }

   _file_copy_path = _secure_area_path + "/" + name + ".BACKUP";
   _file_crc_path = _secure_area_path + "/" + name + ".CRC";
   _file_crc_copy_path = _secure_area_path + "/" + name + ".CRC_BACKUP";

   _status = INITIALIZED;
   return true;
}

void RobustFile::print_error_out(uint line, const char *msg, ...) {
   // TODO(efs1hi): Decrease output, if stable
   char buffer[400];
   memset(buffer, 0, 400);
   va_list argptr;
   va_start(argptr, msg);
   vsprintf(buffer, msg, argptr);
   va_end(argptr);
   char buffer2[512];
   memset(buffer2, 0, 512);
   sprintf(buffer2, "%s [in swu_robustFile.cpp %d]", buffer, line);
   ETG_TRACE_FATAL(("%s", buffer2));
   ETG_TRACE_ERRMEM(("%s", buffer2));
   bool file_exists;
   struct stat info;
   std::list < std::string > files;
   files.push_back(_secure_area_path);
   files.push_back(_file_path);
   files.push_back(_file_crc_path);
   files.push_back(_file_copy_path);
   files.push_back(_file_crc_copy_path);

   for (std::list < std::string >::iterator it = files.begin(); it != files.end(); ++it) {
      std::string file = *it;
      ETG_TRACE_FATAL(("    checking %s", file.c_str()));
      ETG_TRACE_ERRMEM(("    checking %s", file.c_str()));
      std::string command = "ls -al " + file + " >> /tmp/robust_file_ls_out";
      execCommand(command.c_str());
      std::ifstream infile("/tmp/robust_file_ls_out");
      std::string infile_line;
      while (std::getline(infile, infile_line)) {
         ETG_TRACE_FATAL(("        $> %s", infile_line.c_str()));
         ETG_TRACE_ERRMEM(("        $> %s", infile_line.c_str()));
      }
      infile.close();
      unlink("/tmp/robust_file_ls_out");
      if (!exists(_file_path, file_exists)) {
         ETG_TRACE_FATAL(("        Could not check existence."));
         ETG_TRACE_ERRMEM(("        Could not check existence."));
      } else if (file_exists) {
         if (stat(_file_path.c_str(), &info)) {
            memset(buffer2, 0, 512);
            sprintf(buffer2, "        stat returned error %d", (int)errno);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));
         } else {
            memset(buffer2, 0, 512);
            sprintf(buffer2, "        uid %d", info.st_uid);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));

            memset(buffer2, 0, 512);
            sprintf(buffer2, "        gid %d", info.st_gid);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));

            memset(buffer2, 0, 512);
            sprintf(buffer2, "        mode %d", info.st_mode);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));

            memset(buffer2, 0, 512);
			// cast- for gen3 and gen4 compatibality
            sprintf(buffer2, "        size %d", (int)info.st_size);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));

            memset(buffer2, 0, 512);
			// cast- for gen3 and gen4 compatibality
            sprintf(buffer2, "        blksize %d",(int)info.st_blksize);
            ETG_TRACE_FATAL(("%s", buffer2));
            ETG_TRACE_ERRMEM(("%s", buffer2));
         }
      } else {
         ETG_TRACE_FATAL(("        not existing."));
         ETG_TRACE_ERRMEM(("        not existing."));
      }
   }
}

bool RobustFile::sync() {
   if ((_status != INITIALIZED) && (!init())) {
      print_error_out(__LINE__, "Error: RobostFile object could not be initialized before using sync.");
      return false;
   }
   //sync to be safe, that original file is stored correctly
   ::sync();
   ::sync();

   //print_error_out(__LINE__, "This is only to test that the error out works in sync.");

#ifdef RF_PATCH
   return true;
#else
   //create new crc version
   if (!genCRC(_file_path, _file_crc_path)) {
      print_error_out(__LINE__, "Could not generate Checksum File.");
      return false;
   }
   (void) chmod(_file_crc_path.c_str(), 0600);
   ::sync();
   ::sync();

   //update secondary file
   if (!copyFile(_file_path, _file_copy_path)) {
      print_error_out(__LINE__, "Could not copy data file.");
      return false;
   }
   (void) chmod(_file_copy_path.c_str(), 0600);
   ::sync();
   ::sync();

   // copy crc to backup location
   if (!copyFile(_file_crc_path, _file_crc_copy_path)) {
      print_error_out(__LINE__, "Could not copy crc file.");
      return false;
   }
   (void) chmod(_file_crc_copy_path.c_str(), 0600);
   (void) chmod(_secure_area_path.c_str(), 0700);
   ::sync();
   ::sync();

   return true;
#endif
}

bool RobustFile::restore() {
   ETG_TRACE_USR1(("RobustFile::restore (%s) START", _file_path.c_str()));
   if ((_status != INITIALIZED) && (!init())) {
      print_error_out(__LINE__, "Error: RobostFile object could not be initialized before using restore.");
      return false;
   }

   //print_error_out(__LINE__, "This is only to test that the error out works in restore.");

   // First check, if the existing version is o.k.
   bool file_exists;
   bool file_crc_exists;
   bool file_copy_exists;
   bool file_crc_copy_exists;
   struct stat file_stats;
   bool file_fits_crc = false;
   bool file_fits_backup_crc = false;
   bool backup_fits_crc = false;
   bool backup_fits_backup_crc = false;
   bool file_fits_backup = false;

   (void) chmod(_secure_area_path.c_str(), 0700);

   if (!exists(_file_path, file_exists)) {
      print_error_out(__LINE__, "Error: Could not check if %s exists.", _file_path.c_str());
      return false;
   }

#ifdef RF_PATCH
   return file_exists;
#else
   if (!exists(_file_copy_path, file_copy_exists)) {
      print_error_out(__LINE__, "Error: Could not check if %s exists.", _file_copy_path.c_str());
      return false;
   }
   if (!exists(_file_crc_path, file_crc_exists)) {
      print_error_out(__LINE__, "Error: Could not check if %s exists.", _file_crc_path.c_str());
      return false;
   }
   if (!exists(_file_crc_copy_path, file_crc_copy_exists)) {
      print_error_out(__LINE__, "Error: Could not check if %s exists.", _file_crc_copy_path.c_str());
      return false;
   }

   //print_error_out(__LINE__, "No error, just checking before chmod");
   std::queue<std::string> missingFiles;
   if (file_exists) {
      stat(_file_path.c_str(), &file_stats);
      if (!(file_stats.st_mode & S_IWUSR)) {
         (void) chmod(_file_path.c_str(), file_stats.st_mode | S_IWUSR);
      }
   } else {
      missingFiles.push(_file_path);
   }
   if (file_copy_exists) {
      stat(_file_copy_path.c_str(), &file_stats);
      if (!(file_stats.st_mode & S_IWUSR)) {
         (void) chmod(_file_copy_path.c_str(), file_stats.st_mode | S_IWUSR);
      }
   } else {
      missingFiles.push(_file_copy_path);
   }
   if (file_crc_exists) {
      stat(_file_crc_path.c_str(), &file_stats);
      if (!(file_stats.st_mode & S_IWUSR)) {
         (void) chmod(_file_crc_path.c_str(), file_stats.st_mode | S_IWUSR);
      }
   } else {
      missingFiles.push(_file_crc_path);
   }
   if (file_crc_copy_exists) {
      stat(_file_crc_copy_path.c_str(), &file_stats);
      if (!(file_stats.st_mode & S_IWUSR)) {
         (void) chmod(_file_crc_copy_path.c_str(), file_stats.st_mode | S_IWUSR);
      }
   } else {
      missingFiles.push(_file_crc_copy_path);
   }

   if (missingFiles.size() > 0) {
      ETG_TRACE_USR4(("some robust file is missing for %s", _file_path.c_str()));     
   }
   else {
      // print_error_out(__LINE__, "Error: missing file(s), first: %s.", 
      //                 missingFiles.front().c_str());     
   }

   //print_error_out(__LINE__, "No error, just checking after chmod");

   // Check if File is there or if copy is available to be used. In that case we have something to recover from...
   if (file_exists) {
      if (!file_copy_exists) {
         if (!copyFile(_file_path, _file_copy_path)) {
            print_error_out(__LINE__, "Error: could not copy back %40s to %40s.", _file_copy_path.c_str(), _file_path.c_str());
            return false; // not recoverable error
         }
         ETG_TRACE_USR1(("RobustFile::restore (%s) (nothing identical, taking copy file) success.", _file_path.c_str()));
         file_fits_backup = true;
      }
   } else if (file_copy_exists) {
      if (!copyFile(_file_copy_path, _file_path)) {
         print_error_out(__LINE__, "Error: could not copy back %40s to %40s.", _file_copy_path.c_str(), _file_path.c_str());
         return false; // not recoverable error
      }
      ETG_TRACE_USR1(("RobustFile::restore (%s) (nothing identical, taking copy file) success.", _file_path.c_str()));
      file_fits_backup = true;
   } else {
      return false;
   }

   // Now lets find out what to use and what needs to be recovered
   if (file_crc_exists) {
      if (!checkCRC(_file_path, _file_crc_path, file_fits_crc)) {
         print_error_out(__LINE__, "Error: Can not check checksum %40s for file %40s", _file_crc_path.c_str(), _file_path.c_str());
      }
      if (!checkCRC(_file_copy_path, _file_crc_path, backup_fits_crc)) {
         print_error_out(__LINE__, "Error: Can not check checksum %40s for file %40s", _file_crc_path.c_str(), _file_path.c_str());
      }
   }

   if (file_crc_copy_exists) {
      if (!checkCRC(_file_path, _file_crc_copy_path, file_fits_backup_crc)) {
         print_error_out(__LINE__, "Error: Can not check checksum %40s for file %40s", _file_crc_copy_path.c_str(), _file_path.c_str());
      }
      if (!checkCRC(_file_copy_path, _file_crc_copy_path, backup_fits_backup_crc)) {
         print_error_out(__LINE__, "Error: Can not check checksum %40s for file %40s", _file_crc_copy_path.c_str(), _file_path.c_str());
      }
   }

   if (!file_fits_backup) {
      if (!compareFiles(_file_path, _file_copy_path, file_fits_backup)) {
         print_error_out(__LINE__, "Error: Could not compare %40s and %40s.", _file_path.c_str(), _file_copy_path.c_str());
      }
   }

   // Now we tested File against CRC and BACKUP.CRC (so also CRC and BACKUP.CRC must be identical) and BACKUP against BACKUP.CRC
   // We also know if file and BACKUP are identical, in case we have a HASH collision
   if (!file_fits_backup && !file_fits_crc && !file_fits_backup_crc && (backup_fits_crc || backup_fits_backup_crc)) {
      // Only in this case we will use the backup!
      if (!copyFile(_file_copy_path, _file_path)) {
         print_error_out(__LINE__, "Error: could not copy back %40s to %40s.", _file_copy_path.c_str(), _file_path.c_str());
         return false; // not recoverable error
      }
      if (!backup_fits_backup_crc) {
         if (backup_fits_crc) {
            // copying the checksum is cheaper then generating it
            if (!copyFile(_file_crc_path, _file_crc_copy_path)) {
               print_error_out(__LINE__, "Could not copy crc file.");
               return false;
            }
         } else {
            if (!genCRC(_file_path, _file_crc_copy_path)) {
               print_error_out(__LINE__, "Could not generate Checksum File.");
               return false;
            }
         }
      }
      if (!backup_fits_crc) {
         if (!copyFile(_file_crc_copy_path, _file_crc_path)) {
            print_error_out(__LINE__, "Could not copy crc file.");
            return false;
         }
      }
   } else {
      if (!file_fits_backup) {
         if (!copyFile(_file_path, _file_copy_path)) {
            print_error_out(__LINE__, "Error: could not copy back %40s to %40s.", _file_copy_path.c_str(), _file_path.c_str());
            return false; // not recoverable error
         }
      }
      if (!file_fits_crc) {
         if (file_fits_backup_crc) {
            // copying the checksum is cheaper then generating it
            if (!copyFile(_file_crc_copy_path, _file_crc_path)) {
               print_error_out(__LINE__, "Could not copy crc file.");
               return false;
            }
         } else {
            if (!genCRC(_file_path, _file_crc_path)) {
               print_error_out(__LINE__, "Could not generate Checksum File.");
               return false;
            }
         }
      }
      if (!file_fits_backup_crc) {
         if (!copyFile(_file_crc_path, _file_crc_copy_path)) {
            print_error_out(__LINE__, "Could not copy crc file.");
            return false;
         }
      }
   }
   return true;
#endif
}

bool RobustFile::remove() {
   ETG_TRACE_USR1(("RobustFile::remove (%s) START", _file_path.c_str()));
   if ((_status != INITIALIZED) && (!init(true))) {
      ETG_TRACE_ERR(("Error: RobostFile object could not be initialized before using remove."));
      return false;
   }
   bool file_exists = false;

#ifndef RF_PATCH
   if (!exists(_file_crc_copy_path, file_exists)) {
      ETG_TRACE_ERR(("Error: Could not check if %s exists", _file_crc_copy_path.c_str()));
      return false;
   }
   if (file_exists) {
      if (unlink(_file_crc_copy_path.c_str())) {
         ETG_TRACE_ERR(("Error: Could not remove %40s, errno %8d", _file_crc_copy_path.c_str(),
         errno));
         return false;
      }
   }

   if (!exists(_file_copy_path, file_exists)) {
      ETG_TRACE_ERR(("Error: Could not check if %s exists", _file_copy_path.c_str()));
      return false;
   }
   if (file_exists) {
      if (unlink(_file_copy_path.c_str())) {
         ETG_TRACE_ERR(("Error: Could not remove %40s, errno %8d", _file_copy_path.c_str(),
         errno));
         return false;
      }
   }

   if (!exists(_file_crc_path, file_exists)) {
      ETG_TRACE_ERR(("Error: Could not check if %s exists", _file_crc_path.c_str()));
      return false;
   }
   if (file_exists) {
      if (unlink(_file_crc_path.c_str())) {
         ETG_TRACE_ERR(("Error: Could not remove %40s, errno %8d", _file_crc_path.c_str(),
         errno));
         return false;
      }
   }

#endif
   if (!exists(_file_path, file_exists)) {
      ETG_TRACE_ERR(("Error: Could not check if %s exists", _file_path.c_str()));
      return false;
   }
   if (file_exists) {
      if (unlink(_file_path.c_str())) {
         ETG_TRACE_ERR(("Error: Could not remove %40s, errno %8d", _file_path.c_str(), errno));
         return false;
      }
   }
   ::sync();

   //   _status = NOT_INITIALIZED;
   return true;
}

}

