
#include <vector>

#include "swu_rcarBootChain.h"
#include "swu_facUtil.h"
#include "swu_constants.hpp"
#include "swu_trace.h"

#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SWUPDATE_UTIL
  #include "trcGenProj/Header/swu_rcarBootChain.cpp.trc.h"
#endif

namespace swu {

static const tU32 _relativeOffsetAddr     = 0x00l;
static const tU32 _relativeCounterAddr    = 0x10l;
static const tU32 _relativeStatusAddr     = 0x20l;
static const tU32 _relativeEthernetAddr   = 0x30l;
static const tU32 _relativeRootFsAddr     = 0x40l;

rcarBootChain::rcarBootChain() :
   _activeSectorChain(0),
   _inValidate(false)
{
}

const bool rcarBootChain::activateRecoveryMode() {

   ETG_TRACE_FATAL(("rcarBootChain::activateRecoveryMode"));
   
   if(writeMagic(_relativeStatusAddr, Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_ACTIVE)) {
      _inValidate = true;
      return true;
   }
   return false;
}

const bool rcarBootChain::activateApplicationMode() {

   ETG_TRACE_FATAL(("rcarBootChain::activateApplicationMode"));
   
   if(writeMagic(_relativeStatusAddr, Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_INACTIVE)) {
      _inValidate = true;
      return true;
   }
   return false;
}

const bool rcarBootChain::switchBootSectors() {

   ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors-START"));
   bool bReturn = false;

    while(1) {

      if(!forceReadMagics()) {
         ETG_TRACE_USR1(("forceReadMagics has failed"));
         break;
      }
      
      std::vector<tU32> magicData_active(MAGIC_BUFFER_SIZE, 0xFFFFFFFF);
      facInfo facDataWrite_active;
      facInfo facDataErase_active;

      std::vector<tU32> magicData_inActive(MAGIC_BUFFER_SIZE, 0xFFFFFFFF);
      facInfo facDataWrite_inActive;
      facInfo facDataErase_inActive;

      
      if(_activeSectorChain == 1) {
         magicData_active.at(0)           = Constants::RCAR::VALID_MAGIC_VALUE;
         magicData_active.at(4)           = _magicAreaData_2.at(4);
         magicData_active.at(8)           = _magicAreaData_2.at(8);
         magicData_active.at(12)          = _magicAreaData_2.at(12);
         magicData_active.at(16)          = _magicAreaData_2.at(16);

         getFacDataForWriteBootChain_2(facDataWrite_active);
         getFacDataForEraseBootChain_2(facDataErase_active);

         magicData_inActive.at(0)           = Constants::RCAR::INVALID_MAGIC_VALUE;
         magicData_inActive.at(4)           = _magicAreaData_1.at(4);
         magicData_inActive.at(8)           = _magicAreaData_1.at(8);
         magicData_inActive.at(12)          = _magicAreaData_1.at(12);
         magicData_inActive.at(16)          = _magicAreaData_1.at(16);

         getFacDataForWriteBootChain_1(facDataWrite_inActive);
         getFacDataForEraseBootChain_1(facDataErase_inActive);
               
      } else {
         magicData_active.at(0)           = Constants::RCAR::VALID_MAGIC_VALUE;
         magicData_active.at(4)           = _magicAreaData_1.at(4);
         magicData_active.at(8)           = _magicAreaData_1.at(8);
         magicData_active.at(12)          = _magicAreaData_1.at(12);
         magicData_active.at(16)          = _magicAreaData_1.at(16);

         getFacDataForWriteBootChain_1(facDataWrite_active);
         getFacDataForEraseBootChain_1(facDataErase_active);

         magicData_inActive.at(0)           = Constants::RCAR::INVALID_MAGIC_VALUE;
         magicData_inActive.at(4)           = _magicAreaData_2.at(4);
         magicData_inActive.at(8)           = _magicAreaData_2.at(8);
         magicData_inActive.at(12)          = _magicAreaData_2.at(12);
         magicData_inActive.at(16)          = _magicAreaData_2.at(16);

         getFacDataForWriteBootChain_2(facDataWrite_inActive);
         getFacDataForEraseBootChain_2(facDataErase_inActive);

      }

      //set active
      facUtil fac;
      if(!fac.erase(facDataErase_active)) {
         ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors:erase(active) of FAC failed check !!!"));
         break;
      }      
      
      if(!fac.write(facDataWrite_active, magicData_active)) {
         ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors:write(active) of FAC failed check !!!"));
         break;
      }

      //set inactive, don't change the flow. inactive is applicable only if active is set properly.
      //otherwise target ends in bricked state.
      if(!fac.erase(facDataErase_inActive)) {
         ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors:erase(inactive) of FAC failed check !!!"));
         break;
      }      
      
      if(!fac.write(facDataWrite_inActive, magicData_inActive)) {
         ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors:write(inactive) of FAC failed check !!!"));
         break;
      }

      if(!forceReadMagics()) {
         break;
      }      
      bReturn = true;        
      break;
    }

    ETG_TRACE_FATAL(("rcarBootChain::switchBootSectors-END"));   
    return bReturn;
}

bool rcarBootChain::usedBootChain(tU32 &activeChain) {
   ETG_TRACE_USR2(("rcarBootChain::usedBootChain - START"));
   bool bReturn = false;

   while(1) {
      
      if(!checkAndReadMagics()) {
         ETG_TRACE_USR1(("checkAndReadMagics has failed"));
         break;
      }

      if(_activeSectorChain == 1) {
         activeChain = _magicAreaData_1[16];
      } else {
         activeChain = _magicAreaData_2[16];
      }
      bReturn = true;
      break;
   }
   
   ETG_TRACE_USR2(("rcarBootChain::usedBootChain - END:status:%u", bReturn));
   return bReturn;
}


const bool rcarBootChain::switchBootChainOne() {

   ETG_TRACE_FATAL(("rcarBootChain::switchBootChainOne"));
   
   if(writeMagic(_relativeRootFsAddr, Constants::RCAR::HYPERFLASH_ROOTFS1_ACTIVE)) {
      return true;
   }
   return false;
}

const bool rcarBootChain::switchBootChainTwo() {

   ETG_TRACE_FATAL(("rcarBootChain::switchBootChainTwo"));
   
   if(writeMagic(_relativeRootFsAddr, Constants::RCAR::HYPERFLASH_ROOTFS2_ACTIVE)) {
      return true;
   }
   return false;
}

bool rcarBootChain::writeMagic(tU32 flashAddr, tU32 val) {
   ETG_TRACE_USR2(("rcarBootChain::writeMagic - START"));

   bool bReturn = false;

   while(1) {

      if(!checkAndReadMagics()) {
         ETG_TRACE_USR1(("checkAndReadMagics has failed"));
         break;
      }

      ETG_TRACE_USR3(("rcarBootChain::writeMagic, flashAddr:0x%x Val:0x%X", flashAddr, val));
      
      tU32 magicIndex = getMagicIndexFromAddr(flashAddr);
      if(magicIndex == 20) {
         break;
      }

      std::vector<tU32> newMagicData(MAGIC_BUFFER_SIZE, 0xFFFFFFFF);
      facInfo facDataWrite;
      facInfo facDataErase;

      //write the magics to inactive chain.
      //switch sector chain, when it's required.
      if(_activeSectorChain == 1) {
         newMagicData.at(0)           = _magicAreaData_2.at(0);
         newMagicData.at(4)           = _magicAreaData_2.at(4);
         newMagicData.at(8)           = _magicAreaData_2.at(8);
         newMagicData.at(12)          = _magicAreaData_2.at(12);
         newMagicData.at(16)          = _magicAreaData_2.at(16);
         newMagicData.at(magicIndex)  = val;
         
         getFacDataForWriteBootChain_2(facDataWrite);
         getFacDataForEraseBootChain_2(facDataErase);
         
      } else {
         newMagicData.at(0)           = _magicAreaData_1.at(0);
         newMagicData.at(4)           = _magicAreaData_1.at(4);
         newMagicData.at(8)           = _magicAreaData_1.at(8);
         newMagicData.at(12)          = _magicAreaData_1.at(12);
         newMagicData.at(16)          = _magicAreaData_1.at(16);
         newMagicData.at(magicIndex)  = val;

         getFacDataForWriteBootChain_1(facDataWrite);
         getFacDataForEraseBootChain_1(facDataErase);
      }

      facUtil fac;
      if(!fac.erase(facDataErase)) {
         ETG_TRACE_FATAL(("erase of FAC failed check !!!"));
         bReturn = false;
         break;
      }      
      
      if(!fac.write(facDataWrite, newMagicData)) {
         ETG_TRACE_FATAL(("write of FAC failed check !!!"));
         bReturn = false;
         break;
      }

      if(!forceReadMagics()) {
         break;
      }      
      
      bReturn = true;
      break;
   }

   ETG_TRACE_USR2(("rcarBootChain::writeMagic,END status:%u", bReturn));
   return bReturn;
}

bool rcarBootChain::writeMagicToActiveSector(tU32 flashAddr, tU32 val) {
   ETG_TRACE_USR2(("rcarBootChain::writeMagicToActiveSector - START"));

   bool bReturn = false;

   while(1) {

      if(!checkAndReadMagics()) {
         ETG_TRACE_USR1(("checkAndReadMagics has failed"));
         break;
      }

      ETG_TRACE_USR3(("rcarBootChain::writeMagicToActiveSector, flashAddr:0x%x Val:0x%X", flashAddr, val));
      
      tU32 magicIndex = getMagicIndexFromAddr(flashAddr);
      if(magicIndex == 20) {
         break;
      }

      std::vector<tU32> newMagicData(MAGIC_BUFFER_SIZE, 0xFFFFFFFF);
      facInfo facDataWrite;
      facInfo facDataErase;

      //write the magics to inactive chain.
      //switch sector chain, when it's required.
      if(_activeSectorChain == 1) {
         newMagicData.at(0)           = _magicAreaData_1.at(0);
         newMagicData.at(4)           = _magicAreaData_1.at(4);
         newMagicData.at(8)           = _magicAreaData_1.at(8);
         newMagicData.at(12)          = _magicAreaData_1.at(12);
         newMagicData.at(16)          = _magicAreaData_1.at(16);
         newMagicData.at(magicIndex)  = val;
         
         getFacDataForWriteBootChain_1(facDataWrite);
         getFacDataForEraseBootChain_1(facDataErase);
         
      } else {
         newMagicData.at(0)           = _magicAreaData_2.at(0);
         newMagicData.at(4)           = _magicAreaData_2.at(4);
         newMagicData.at(8)           = _magicAreaData_2.at(8);
         newMagicData.at(12)          = _magicAreaData_2.at(12);
         newMagicData.at(16)          = _magicAreaData_2.at(16);
         newMagicData.at(magicIndex)  = val;

         getFacDataForWriteBootChain_2(facDataWrite);
         getFacDataForEraseBootChain_2(facDataErase);
      }

      facUtil fac;
      if(!fac.erase(facDataErase)) {
         ETG_TRACE_FATAL(("erase of FAC failed check !!!"));
         bReturn = false;
         break;
      }      
      
      if(!fac.write(facDataWrite, newMagicData)) {
         ETG_TRACE_FATAL(("write of FAC failed check !!!"));
         bReturn = false;
         break;
      }

      if(!forceReadMagics()) {
         break;
      }      
      
      bReturn = true;
      break;
   }

   ETG_TRACE_USR2(("rcarBootChain::writeMagicToActiveSector,END status:%u", bReturn));
   return bReturn;
}




bool rcarBootChain::readMagic(tU32 flashAddr, tU32 &val) {
   ETG_TRACE_USR2(("rcarBootChain::readMagic - START"));

   bool bReturn = false;
   tU32 magicIndex = 20; //invalid index
   val = 0;

   while(1) {

     if(!checkAndReadMagics()) {
         ETG_TRACE_USR1(("checkAndReadMagics has failed"));
         break;
      }

     magicIndex = getMagicIndexFromAddr(flashAddr);
     if(magicIndex == 20) {
        break;
     }

     if(_activeSectorChain == 1) {
        val = _magicAreaData_1.at(magicIndex);
        ETG_TRACE_USR3(("value in activeBootChain 1 is:0x%X", val));
        bReturn = true;
        break;
     }

     if(_activeSectorChain == 2) {
        val = _magicAreaData_2.at(magicIndex);
        ETG_TRACE_USR3(("value in activeBootChain 2 is:0x%X", val));
        bReturn = true;
        break;
     }
     break;
   }

   ETG_TRACE_USR2(("rcarBootChain::readMagic-END,bReturn:%u val:0x%x", bReturn,val));
   return bReturn;
}

bool rcarBootChain::isRecoveryMagicSet(bool &isSet) {
   ETG_TRACE_USR2(("rcarBootChain::isRecoveryMagicSet - START"));

   bool bReturn = false;
   isSet = false;

   while(1) {

      if(!checkAndReadMagics()) {
         ETG_TRACE_USR1(("checkAndReadMagics has failed"));
         break;
      }

      bReturn = true;
      if(_activeSectorChain == 1) {
         if(_magicAreaData_1.at(8) == Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_ACTIVE) {
            isSet = true;
            break;
         }
      }

      if(_activeSectorChain == 2) {
         if(_magicAreaData_2.at(8) == Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_ACTIVE) {
            isSet = true;
            break;
         }
      }
      
      break;
   }
   
   ETG_TRACE_USR2(("rcarBootChain::isRecoveryMagicSet-END:bReturn:%u,isSet:%u", bReturn, isSet));;
   return bReturn;
}

bool rcarBootChain::eraseMagic(tU32 flashAddr) {
   //todo: is erase logic is needed for bootchain ?
   //it may be needed for some other memory not for bootchain memory.
   return true;
}

bool rcarBootChain::forceReadMagics() {
   ETG_TRACE_USR2(("rcarBootChain::forceReadMagics - START"));

   bool bReturn = true;
   tU32 activeChain = 0;
   
   facInfo facInfo_1;
   getFacDataForReadBootChain_1(facInfo_1);

   facInfo facInfo_2;
   getFacDataForReadBootChain_2(facInfo_2);   

   while(1) {

      _magicAreaData_1.clear();
      _magicAreaData_2.clear();
      
      facUtil fac;      
      if(!fac.read(facInfo_1, _magicAreaData_1)) {
         ETG_TRACE_FATAL(("read of FAC failed for bootchain 1"));
         bReturn = false;
         break;
      }

      if(!fac.read(facInfo_2, _magicAreaData_2)) {
         ETG_TRACE_FATAL(("read of FAC failed for bootchain 2"));
         bReturn = false;
         break;
      }

      if((_magicAreaData_1.at(0) == Constants::RCAR::VALID_MAGIC_VALUE) &&
         (_magicAreaData_2.at(0) != Constants::RCAR::VALID_MAGIC_VALUE)) {
         
         ETG_TRACE_USR3(("active boot chain:1"));
         activeChain = 1;
         break;
      }

      if((_magicAreaData_1.at(0) != Constants::RCAR::VALID_MAGIC_VALUE) &&
         (_magicAreaData_2.at(0) == Constants::RCAR::VALID_MAGIC_VALUE)) {
         
         ETG_TRACE_USR3(("active boot chain:2"));
         activeChain = 2;
         break;
      }

      if((_magicAreaData_1.at(0) == Constants::RCAR::VALID_MAGIC_VALUE) &&
         (_magicAreaData_2.at(0) == Constants::RCAR::VALID_MAGIC_VALUE)) {

         ETG_TRACE_USR3(("both bootchain are active with Index 0"));
         
         if(_magicAreaData_1.at(4) >= _magicAreaData_2.at(4)) {            
            ETG_TRACE_USR3(("active boot chain:1"));
            activeChain = 1;
            break;
         } else {            
            ETG_TRACE_USR3(("active boot chain:2"));
            activeChain = 2;
            break;
         }
      }

      break;      
   }
   _activeSectorChain = activeChain;

   printDownloadPipe();
   
   ETG_TRACE_USR2(("rcarBootChain::forceReadMagics-END:chain:%u status:%u", activeChain, bReturn));
   return bReturn;
}

void rcarBootChain::getFacDataForReadBootChain_1(facInfo &info) {
   
   //3072 * 512 = 1572864 (0x180000)   
   info._blockSize = BLOCK_SIZE;
   //info._count = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_BUFFER_SIZE)) / BLOCK_SIZE;
   info._count = 1;
   info._skip = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_OFFSET)) / BLOCK_SIZE ;
   info._seek = 0;   
}

void rcarBootChain::getFacDataForReadBootChain_2(facInfo &info) {

   //3584 * 512 = 1835008 (0x1C0000)
   info._blockSize = BLOCK_SIZE;
   //info._count = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_BUFFER_SIZE)) / BLOCK_SIZE;
   info._count = 1;
   info._skip = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_OFFSET)) / BLOCK_SIZE;
   info._seek = 0;   
}

void rcarBootChain::getFacDataForWriteBootChain_1(facInfo &info) {
   
   //3072 * 512 = 1572864 (0x180000)   
   info._blockSize = BLOCK_SIZE;
   info._count = 1;
   info._skip =  0;
   info._seek = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_OFFSET)) / BLOCK_SIZE;   
}

void rcarBootChain::getFacDataForWriteBootChain_2(facInfo &info) {

   //3584 * 512 = 1835008 (0x1C0000)
   info._blockSize = BLOCK_SIZE;
   info._count = 1;
   info._skip = 0;
   info._seek = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_OFFSET)) / BLOCK_SIZE;   
}

void rcarBootChain::getFacDataForEraseBootChain_1(facInfo &info) {

   //512  * 512 =  262144 (0x40000)
   info._blockSize = BLOCK_SIZE;
   info._count = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_BUFFER_SIZE)) / BLOCK_SIZE;
   info._skip = 0;
   info._seek = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_OFFSET)) / BLOCK_SIZE;   

}

void rcarBootChain::getFacDataForEraseBootChain_2(facInfo &info) {

   //512  * 512 =  262144 (0x40000)
   info._blockSize = BLOCK_SIZE;
   info._count = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_BUFFER_SIZE)) / BLOCK_SIZE;
   info._skip = 0;
   info._seek = (static_cast<tU32> (Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_OFFSET)) / BLOCK_SIZE;   

}



bool rcarBootChain::checkAndReadMagics() {

   if(_activeSectorChain == 0 || _inValidate) {
      ETG_TRACE_USR3(("active boot chain is not available / forcing, reading.."));
      _inValidate = false;
      tU32 activeChain;
      if(!forceReadMagics()) {
         return false;
      }
   }
      
   ETG_TRACE_USR3(("active sector chain is:%u", _activeSectorChain));
   if(_activeSectorChain == 0) {
      ETG_TRACE_FATAL(("active boot chain is still 0, even after reading check !!!"));
      return false;
   }

   if(_magicAreaData_1.size() == 0 || _magicAreaData_2.size() == 0){
      ETG_TRACE_FATAL(("magic area data is not set check !!!"));
      return false;
   }

   return true;

}

tU32 rcarBootChain::getMagicIndexFromAddr(tU32 relativeFlashAddr) {

   ETG_TRACE_USR1(("rcarBootChain::getMagicIndexFromAddr:0x%X", relativeFlashAddr));
   tU32 index = 20; //invalid value;

   switch(relativeFlashAddr) {
      case _relativeOffsetAddr:
         index = 0;
         break;
      case _relativeCounterAddr:
         index = 4;
         break;
      case _relativeStatusAddr:
         index = 8;
         break;
      case _relativeEthernetAddr:
         index = 12;
         break;
      case _relativeRootFsAddr:
         index = 16;
         break;
      default:
         index = 20;
         break;
   }

   ETG_TRACE_USR1(("rcarBootChain::getMagicIndexFromAddr,index:%u", index));
   return index;
}

void rcarBootChain::printDownloadPipe() {

   tChar  tmpBuffer [256]   = {0};

   sprintf(tmpBuffer, "ReadMagicArea1=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
           _magicAreaData_1.at(0),
           _magicAreaData_1.at(4),
           _magicAreaData_1.at(8),
           _magicAreaData_1.at(12),
           _magicAreaData_1.at(16) );
   printToDownloadPipe(tmpBuffer);

   sprintf(tmpBuffer, "ReadMagicArea2=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
           _magicAreaData_2.at(0),
           _magicAreaData_2.at(4),
           _magicAreaData_2.at(8),
           _magicAreaData_2.at(12),
           _magicAreaData_2.at(16) );
   printToDownloadPipe(tmpBuffer);

   

}

void rcarBootChain::dumpBootChainConfig() {

   print2stdout("rcarBootChain::dumpBootChainConfig()\n");

   if(!forceReadMagics()) {
      return;
   }

   print2stdout("Sector 1 (0x180000)\n");
   
   print2stdout("read HYPERFLASH_MAGIC_CHAIN1_OFFSET (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_OFFSET,_magicAreaData_1[0]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN1_COUNTER_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_COUNTER_ADDR, _magicAreaData_1[4]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN1_STATUS_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_STATUS_ADDR, _magicAreaData_1[8]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN1_ETHERNET_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_ETHERNET_ADDR, _magicAreaData_1[12]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN1_ROOTFS_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN1_ROOTFS_ADDR, _magicAreaData_1[16]);
   
   
   //-----------------------------------------------------------------------------------------
   
   print2stdout("Sector 2 (0x1C0000)\n");

   print2stdout("read HYPERFLASH_MAGIC_CHAIN2_OFFSET (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_OFFSET,_magicAreaData_2[0]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN2_COUNTER_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_COUNTER_ADDR, _magicAreaData_2[4]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN2_STATUS_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_STATUS_ADDR, _magicAreaData_2[8]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN2_ETHERNET_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_ETHERNET_ADDR, _magicAreaData_2[12]);

   print2stdout("read HYPERFLASH_MAGIC_CHAIN2_ROOTFS_ADDR (0x%x) = (0x%8.8X)\n", Constants::RCAR::HYPERFLASH_MAGIC_CHAIN2_ROOTFS_ADDR, _magicAreaData_2[16]);
   
   print2stdout("=============================\n");
   if(_activeSectorChain == 1)
   {
      print2stdout("Sector 1 is active\n");
      if(_magicAreaData_1[8]==Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_ACTIVE)
         print2stdout("Recovery Mode is active\n");
      else
         print2stdout("Full Operation Mode is active\n");
      if(_magicAreaData_1[16]==Constants::RCAR::USE_SECOND_BOOT_CHAIN)
         print2stdout("Bootchain 2 is active\n");
      else
      print2stdout("Bootchain 1 is active\n");
   }
   else if(_activeSectorChain == 2)
   {
      print2stdout("Sector 2 is active\n");
      if(_magicAreaData_2[8]==Constants::RCAR::HYPERFLASH_RECOVERY_MAGIC_ACTIVE)
         print2stdout("Recovery Mode is active\n");
      else
         print2stdout("Full Operation Mode is active\n");
      if(_magicAreaData_2[16]==Constants::RCAR::USE_SECOND_BOOT_CHAIN)
         print2stdout("Bootchain 2 is active\n");
      else
         print2stdout("Bootchain 1 is active\n");
   }
  else
  {
     print2stdout("Both sectors are inactive!\n");
     print2stdout("Recovery Mode is active\n");
     print2stdout("Bootchain 1 is active\n");
  }
   
}


}
