#ifndef SWU_PLUGIN_HPP
#define SWU_PLUGIN_HPP

#include <stdio.h>
#include <dlfcn.h>
#include <libgen.h>
#include <set>
#include "util/swu_filesystem.h"
#include "util/swu_singleton.hpp"
#include "util/swu_memberList.hpp"
#include "util/swu_trace.h"

#define SWU_PLUGIN_TRACE_LEVEL_INFO 0
#define SWU_PLUGIN_TRACE_LEVEL_ERR 0

/*
  declare CLASS (must be swu::member of ROOT) with PRIO and create symbol ANCHOR
  ANCHOR can be used to force linking with --undefined=ANCHOR
 */
#define SWU_REGISTER_PLUGIN(ROOT,CLASS,PRIO,ANCHOR)                     \
   static class CLASS##Plugin: public swu::Plugin<ROOT, CLASS> {        \
   public:                                                              \
   CLASS##Plugin():                                                     \
   Plugin(PRIO) {                                                       \
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO,                       \
                     "SWU_REGISTER_PLUGIN %s", #CLASS);                 \
   }                                                                    \
   } _plugin;                                                           \
                                                                        \
   extern "C" {                                                         \
      void *ANCHOR=(void *)&_plugin;                                    \
   }


/*
  each plugin shall register itself by using SWU_REGISTER_PLUGIN
  Owner of the plugins shall activate them by swu::PluginManager<ROOT>::instance()->execute(&memberList)
 */
namespace swu {

template <class COMP>
class PluginBase;

template <class COMP, class MEMBER>
class Plugin;



// use template to have pre-initialization separated by COMP
template <class COMP>
class PluginManager: 
      public Singleton<PluginManager<COMP> > {
   template<class COM, class MBR>
   friend class Plugin;

public:
   PluginManager():
      _first(0)
   {
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager CTOR");
   };


   // load all plugins in directory mathing pattern
   void loadDynPlugins(std::string directory, std::string pattern, char fileType='l') {
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::loadDynPlugins START");

      std::set<std::string> libs=swu::getMatchingFiles(directory, pattern, true);
      for (std::set<std::string>::iterator iter=libs.begin();iter!=libs.end();++iter) {
         std::string lib=*iter;
         purifyString(lib);
         char* buffer = strdup(lib.c_str());

         char const *fileName=basename(buffer);
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::loadDynPlugins: load:%s", fileName);
         loadDynPlugin(fileName);
         free(buffer);
      }
   };

   bool loadDynPlugin(std::string pluginName) {
      void *dlRes=dlopen(pluginName.c_str(), RTLD_NOLOAD);
      if (dlRes) {
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::loadDynPlugin(%s) alread loaded", pluginName.c_str());
	 dlclose(dlRes);  //Coverity fix for 98641.
         return true;
      }
      dlRes=dlopen(pluginName.c_str(), RTLD_LAZY);
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::dlopen (%40s) returned:%p", pluginName.c_str(), dlopen);
      char *dlErrString=dlerror();
      if (dlErrString) {
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::dlerror: returned:%s", dlErrString);
      }
      if(dlRes != NULL)
	  dlclose(dlRes);   //Coverity fix for 98641 and 175662
      return dlRes!=0;
   };

public:
   /* Owner shall call execute to instantiate all plugins registered plugins and add them to its member-list */
   void execute(swu::MemberList<COMP> *memberList) {
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager execute START");
      PluginBase<COMP> *first=_first;
      _first=0;

      while (first) {
         Member<COMP>*member=first->instantiateMember();
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager execute plugin=%p, member=%p (%s) FOUND", first, member, member->getName());
         memberList->addMember(member);
         PluginBase<COMP> *oldFirst=first;
         first=oldFirst->_next;
         oldFirst->_next=0;
      }

      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager execute END");

   };


private:
   // function to be call by friends of type PluginBase
   template<class COM, class MBR>
   void reg(Plugin<COM, MBR> *usr) {
      SWU_ASSERT_RETURN(usr);
      PluginBase<COMP> *newPlugin=usr;
      newPlugin->_next=0;
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::reg() START (%p)", newPlugin);
      PluginBase<COMP> *cur=_first;
      if (!cur) {
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::reg() FIRST");
         _first=newPlugin;
      }
      else {
         while (cur->_prio >= newPlugin->_prio) {
            if (!cur->_next) {
               break;
            }
            cur=cur->_next;
         }
         newPlugin->_next=cur->_next;
         cur->_next=newPlugin;
         SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginManager::reg() LAST");
      }
   };

   PluginBase<COMP> *_first;
};


/*
  only static object may derive from PluginBase
*/
template <class COMP>
class PluginBase {
   template <class COM>
   friend class PluginManager;

protected:
   PluginBase():
      _next(0),
      _prio(0) {
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "PluginBase CTOR");
   };
   virtual ~PluginBase() {
   }
   virtual swu::Member<COMP> *instantiateMember()=0;

protected:
   // intrusive single linked list
   PluginBase<COMP> *_next;
   int _prio;
};


// final template class that 
template <class COMP, class MEMBER>
class Plugin: public PluginBase<COMP> {
public:
   Plugin(int prio=0)
   {
      PluginBase<COMP>::_prio=prio;
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "Plugin CTOR");
      PluginManager<COMP>::instance()->reg(const_cast<Plugin<COMP, MEMBER> *>(this));
   }
   virtual swu::Member<COMP> *instantiateMember() {
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "instantiateMember START");
      Member<COMP> *member=MEMBER::instance();
      SWU_TRACE_VARG(SWU_PLUGIN_TRACE_LEVEL_INFO, "instantiateMember END: %s", member->getName());
      return member;
   };


};

}
#endif
