#include <errno.h>
#include <string>
#include <sstream>
#include <unistd.h>
#include <linux/loop.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include "base/imp/DownloadPipe.h"
#include "swu_execCommand.h"
#include "swu_filesystem.h"

#include "swu_isoContainer.h"

#include "util/swu_trace.h"
#include "util/swu_util.hpp"
#include "util/swu_types.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_isoContainer.cpp.trc.h"
#endif


namespace swu {

static bool _modprobeDone=false;



bool IsoContainer::cleanup() {
   
   ETG_TRACE_USR1(("IsoContainer::cleanup() START mount status:%u", _isMounted));
   
   if (_isMounted) {
      std::string cmd=std::string("umount ") +  _mp;
      if (swu::execCommand(cmd)) {
         ETG_TRACE_FATAL(("IsoContainer::cleanup: umount(%s) failed", _mp.c_str()));         
      }
      //this is required for bind mount
      _isMounted = false;
   } else {
      ETG_TRACE_USR1(("no mount point is exist"));
      if(_deviceId == -1) {         
         return true;
      }
   }

   ETG_TRACE_USR1(("IsoContainer::cleanup _deviceId:%u", _deviceId));
   if (_deviceId != -1) {

      //if it is loop mount, take the original state.
      _isMounted = true;

      std::string cmd = "umount /dev/loop0";
      if (swu::execCommand(cmd)) {
         ETG_TRACE_ERR(("FcSwUpdateOtaIf::unmountOtaData: umount losetup failed"));
      }

      cmd.clear();
      cmd="losetup -d " + getLoopDeviceName();
      if(swu::execCommand(cmd)) {
         ETG_TRACE_FATAL(("IsoContainer::cleanup: losetup delete is failed"));
      }

      _isMounted=false;
  
      int devfd = open("/dev/loop-control", O_RDWR);
      if(devfd != -1) {
         _deviceId = ioctl(devfd, LOOP_CTL_REMOVE, _deviceId);
         _deviceId = -1;
         close(devfd);
      }
   }
  
   ETG_TRACE_USR1(("IsoContainer::cleanup() END"));
   return true;
}

bool IsoContainer::bindMount(std::string destDir) {
   ETG_TRACE_USR2(("IsoContainer::bindMounr: %s", destDir.c_str()));

   if (!swu::makeDirectoryRecursive(_mp, 0700)) {
      ETG_TRACE_FATAL(("IsoContainer::mount(): could not create mp:%s", _mp.c_str())); 
      return false;           
   }

   std::string cmd = "mount --bind " + destDir  + " " + _mp;
   ETG_TRACE_USR2(("cmd -%s", cmd.c_str()));

   if(swu::execCommand(cmd)) {
      ETG_TRACE_USR2(("bind mount fails"));
      return false;
   } else {
      ETG_TRACE_USR2(("bind mount passed"));
      _isMounted = true;
      return true;
   }

}

bool IsoContainer::mount(std::string srcFile, unsigned long long srcOffset, unsigned long long srcLen) {
   _srcFile=srcFile;
   _srcOffset=srcOffset;
   _srcLen=srcLen;
   return mount();
}

bool IsoContainer::mount() {
   printToDownloadPipe("IsoContainer::mount() START");
   cleanup();
   ETG_TRACE_USR3(("modprobe status:%u", _modprobeDone)); 
   bool res=false;
   do {
      if (!_modprobeDone) {
         if (swu::execCommand("modprobe loop")) {
            printToDownloadPipe("IsoContainer::mount() ERROR could not modprobe loop");
            ETG_TRACE_FATAL(("IsoContainer::mount(): could not modprobe loop"));
            break;
         }
         _modprobeDone=true;
      }

      if (_deviceId != -1) {
         printToDownloadPipe("IsoContainer::mount() ERROR _deviceId already set");
         ETG_TRACE_FATAL(("IsoContainer::mount(): _deviceId already set: %d", _deviceId));
         cleanup();
         break;
      }
      int devfd = open("/dev/loop-control", O_RDWR);
      if(devfd < 0) {
         printToDownloadPipe("IsoContainer::mount() ERROR could not open /dev/loop-control ");
         std::string errorText = strerror(errno);
         ETG_TRACE_FATAL(("IsoContainer::mount(): could not open /dev/loop-control: %s", errorText.c_str()));
         break;
      }
      _deviceId = ioctl(devfd, LOOP_CTL_GET_FREE);
      close(devfd);
      if (_deviceId < 0 || devfd < 0) {
         printToDownloadPipe("IsoContainer::mount() ERROR could not get free device-id");
         ETG_TRACE_FATAL(("IsoContainer::mount(): could not get free device-id: %d", _deviceId));
         break;
      }
   
      if (!swu::makeDirectoryRecursive(_mp, 0700)) {
         printToDownloadPipe("IsoContainer::mount() ERROR could not create mp");
         ETG_TRACE_FATAL(("IsoContainer::mount(): could not create mp:%s", _mp.c_str()));
         break;
      }

      if (!isFile(_srcFile)) {
         printToDownloadPipe("IsoContainer::mount() ERROR not existing _srcFile");
         ETG_TRACE_FATAL(("IsoContainer::mount(): not existing _srcFile:%s", _srcFile.c_str()));
         break;
      }

      off_t fileSize=getFileSize(_srcFile);
      off_t calSize=_srcOffset+_srcLen;
      if (calSize>fileSize) {
         printToDownloadPipe("IsoContainer::mount() ERROR invalid len/offset");
         ETG_TRACE_FATAL(("IsoContainer::mount(%50s): invalid len: offset=%u, len=%u fileSize=%u", 
                          _srcFile.c_str(), _srcOffset, _srcLen, fileSize));
         break;
      }

      std::stringstream strStream;
      std::string loopDeviceName=getLoopDeviceName();
      strStream << "losetup";
      if (_srcOffset) {
         strStream << " -o " <<  _srcOffset;
      }
      if (_srcLen) {
         strStream << " --sizelimit " <<  _srcLen;
      }
      strStream << " " << loopDeviceName << " " << _srcFile;
      std::string cmd = strStream.str();

      ETG_TRACE_FATAL(("CMD-losetup:%s", cmd.c_str()));
      
      if (swu::execCommand(cmd)) {
         ETG_TRACE_FATAL(("IsoContainer::mount(): cmd failed:%s", cmd.c_str()));
         printToDownloadPipe("IsoContainer::mount() ERROR mount-cmd failed");
         break;
      }

      cmd=std::string("mount ") + loopDeviceName + " " + _mp;
      ETG_TRACE_FATAL(("CMD-mount:%s", cmd.c_str()));
      
      if (swu::execCommand(cmd)) {
         ETG_TRACE_FATAL(("IsoContainer::mount(): cmd failed:%s", cmd.c_str()));
         printToDownloadPipe("IsoContainer::mount() ERROR mount-cmd failed");
      } else {
         _isMounted = true;
      }  
   } while(false);

   ETG_TRACE_USR2(("mount status:%u", _isMounted));
   if (!_isMounted) {
      cleanup();
   }
   return _isMounted;
}

std::string IsoContainer::getLoopDeviceName() {
   std::string loopDeviceName="/dev/loop";
   std::stringstream strStream;
   strStream << loopDeviceName << _deviceId;
   std::string result = strStream.str();
   return result;
}

IsoContainerChain::~IsoContainerChain() {
   umount();
}
bool IsoContainerChain::mount(std::string configFile) {
   if (_containers.size()) {
      return false;
   }
   bool res=true;      
   std::string fileContent;
   res=swu::loadFile(configFile, fileContent);
   if (!res) {
      ETG_TRACE_ERR(("IsoContainerChain::mount(): could not load configFile=%s", configFile.c_str()));
      return false;
   }
   // for each line in config file
   std::string line;
   std::string mp="";
   while (swu::extractLine(fileContent, line)) {
      trimString(line);
      std::string lastMp=mp;
      std::vector<std::string> items=swu::stringToVector(line, ' ');
      ETG_TRACE_USR4(("IsoContainerChain::mount(): config-line=%s", line.c_str()));
      ETG_TRACE_USR4(("IsoContainerChain::mount(): items.size()=%u", items.size()));
      if (!items.size()) {
         // empty line
         ETG_TRACE_USR2(("IsoContainerChain::mount(): empty line\n"));
         continue;
      } else if ( ! items[0].size() || items[0].at(0) == '#') {
         ETG_TRACE_USR4(("IsoContainerChain::mount(): empty item"));
         // empty item or comment
         continue;
      }else if (items.size() == 1) {
         // last line, the final file with path
         _dataFile=trimString(items[0]);
         ETG_TRACE_USR4(("IsoContainerChain::mount(): one item: mount data-file=%s", _dataFile.c_str()));
         mp = parentDirname(_dataFile);
         if(mp.size() > 0) {
            ETG_TRACE_USR1(("mp:%80s, lastMp:%80s",mp.c_str(), lastMp.c_str()));
            swu::IsoContainer *container=new swu::IsoContainer(mp);
            SWU_ASSERT_RETURN_FALSE(container);
            if(!container->bindMount(lastMp)) {
               ETG_TRACE_FATAL(("IsoContainerChain::mount(): could not bind mount=%50s to %s", lastMp.c_str(), mp.c_str()));
            }
            _containers.push_back(container);
         } else {
            ETG_TRACE_USR1(("mount point path is empty"));
         }
         break;
      }
      else if (items.size() >= 3) {
         std::string isoFile=trimString(items[0]);
         mp=getTempDir();
         tUInt offset=stringToUInt(items[1]);
         tUInt sizelimit=stringToUInt(items[2]);
         ETG_TRACE_USR4(("IsoContainerChain::mount(): offset=%u sizelimit=%u isoFile=%50s mp=%s", offset, sizelimit, isoFile.c_str(), mp.c_str()));
         swu::IsoContainer *container=new swu::IsoContainer(mp, lastMp + "/" + isoFile, offset, sizelimit);
         if (!container->mount()) {
            ETG_TRACE_ERR(("IsoContainerChain::mount(): could not mount=%50s to %s", 
                           isoFile.c_str(), mp.c_str()));
            res=false;
            break;
         }
         _containers.push_back(container);
      }
   } while (0);
   
   if (!res) {
      umount();
   }
   ETG_TRACE_USR4(("IsoContainerChain::mount():DONE\n"));
   return res;
}


void IsoContainerChain::umount() {
   ETG_TRACE_USR1(("IsoContainerChain::umount():containerSize:%u", _containers.size()));
   while (_containers.size()) {
      delete _containers.back();
      _containers.pop_back();
   }
   _dataFile="";
}

std::string IsoContainerChain::getMountPoint() {
   if (!_containers.size()) {
      return "";
   }
   return _containers.back()->getMountPoint();
}

std::string IsoContainerChain::getDataFile() {
   return _dataFile;
}



void IsoContainerChain::fetchLoopDeviceInfos(std::map<std::string, LoopDeviceInfo> &loopDeviceInfos) {
   ETG_TRACE_USR4(("IsoContainerChain::fetchLoopDeviceInfos() START"));
   std::vector<std::string> lines;
   cmdToLines("losetup -a", lines);
   
   for (std::vector<std::string>::iterator iterLines=lines.begin(); iterLines!=lines.end(); ++iterLines) {
      // /dev/loop0: [0811]:47191720 (/mnt/transfer/samba_share/views/dpIsoTest/dp.iso)
      std::string line=*iterLines;
      ETG_TRACE_USR3(("IsoContainerChain::fetchLoopDeviceInfos() check line: %s", line.c_str()));

      std::vector<std::string>items=stringToVector(line, ' ');
      LoopDeviceInfo loopInfo;
         
      tUInt index=0;
      std::string wait="";
      for (std::vector<std::string>::iterator iterItems=items.begin();iterItems!=items.end();++iterItems,++index) {
         std::string item=*iterItems;
         ETG_TRACE_USR2(("IsoContainerChain::fetchLoopDeviceInfos() check item: %s", item.c_str()));
         trimString(item);
         replaceSubString(item, ":", "");
         replaceSubString(item, "(", "");
         replaceSubString(item, ")", "");
         replaceSubString(item, ",", "");
         ETG_TRACE_USR2(("IsoContainerChain::fetchLoopDeviceInfos() item clean: %50s", item.c_str()));

         if (index==0) {
            // /dev/loop0:
            loopInfo.device=item;
         }
         else if (index==2){
            // (/mnt/transfer/samba_share/views/dpIsoTest/dp.iso)
            loopInfo.container=item;
            ETG_TRACE_USR3(("IsoContainerChain::fetchLoopDeviceInfos() container=%s", loopInfo.container.c_str()));

            loopInfo.containerShort=fileName(item);
            ETG_TRACE_USR3(("IsoContainerChain::fetchLoopDeviceInfos() containerShort=%s", loopInfo.containerShort.c_str()));
            
         }
         else if (item=="offset") {
            wait="offset";
         }
         else if (wait=="offset") {
            loopInfo.offset=stringToUInt(item);
            wait="";
         }
         else if (item=="sizelimit") {
            wait="sizelimit";
         }
         else if (wait=="sizelimit") {
            loopInfo.sizeLimit=stringToUInt(item);
            wait="";
         }
      }

      if (!loopInfo.device.empty()) {
         ETG_TRACE_USR3(("IsoContainerChain::fetchLoopDeviceInfos() add device:%s", loopInfo.device.c_str()));
         
         loopDeviceInfos[loopInfo.device]=loopInfo;
      }
   }
   ETG_TRACE_USR4(("IsoContainerChain::fetchLoopDeviceInfos() END "));
}


void IsoContainerChain::fetchMountInfos(std::map<std::string, LoopDeviceInfo> &loopDeviceInfos,
                                        std::map<std::string, MountInfo> &mountInfos) {
   ETG_TRACE_USR4(("IsoContainerChain::fetchMountInfos() START"));

   std::vector<std::string> lines;
   cmdToLines("mount", lines);
   for (std::vector<std::string>::iterator iterLines=lines.begin(); iterLines!=lines.end(); ++iterLines) {
      std::string line=*iterLines;
      std::vector<std::string>items=stringToVector(line, ' ');
      MountInfo mountInfo;
      tUInt index=0;
      ETG_TRACE_USR2(("IsoContainerChain::fetchMountInfos() check line: %s", line.c_str()));

      for (std::vector<std::string>::iterator iterItems=items.begin();iterItems!=items.end();++iterItems,++index) {
         std::string item=*iterItems;
         trimString(item);
         // /dev/loop2 on /tmp/comp type iso9660 (ro)
         ETG_TRACE_USR3(("IsoContainerChain::fetchMountInfos() check item: %s", item.c_str()));

         if (index==0) {
            // /dev/loop2:
            mountInfo.device=item;
            LoopDeviceInfo const &loopDeviceInfo=loopDeviceInfos[mountInfo.device];
            if (loopDeviceInfos[mountInfo.device].device==mountInfo.device) {
               mountInfo.loopDeviceInfo=&loopDeviceInfos[mountInfo.device];
            }
         }
         else if (index==2){
            // /tmp/comp
            mountInfo.mp=item;
         }
         else if (index==4) {
            // iso9660
            mountInfo.type=item;
         }
         else if (item.at(0)=='(') {
            // (ro)
            replaceSubString(item, "(", "");
            replaceSubString(item, ")", "");
            std::vector<std::string> options=swu::stringToVector(item, ',');
            for (std::vector<std::string>::iterator iterOptions=options.begin();
                 iterOptions!=options.end(); ++iterOptions) {
               std::string option=*iterOptions;
               if (option=="bind") {
                  mountInfo.isBindMount=true;
               }
            }
         }
      }
     

      if (!mountInfo.device.empty()) {
         ETG_TRACE_USR2(("IsoContainerChain::fetchMountInfos() add device:%s", mountInfo.device.c_str()));
         mountInfos[mountInfo.device]=mountInfo;
      }

   }
   ETG_TRACE_USR4(("IsoContainerChain::fetchMountInfos() END"));
}

IsoContainerChain::MountInfo const *IsoContainerChain::getMountInfoByMp(std::string &mp, std::map<std::string, MountInfo> &mountInfos) {
   ETG_TRACE_USR4(("IsoContainerChain::getMountInfoByMp(%s) START", mp.c_str()));

   for (std::map<std::string, MountInfo>::iterator iterMountInfos=mountInfos.begin();
        iterMountInfos!=mountInfos.end(); ++ iterMountInfos) {
      MountInfo const &mountInfo=iterMountInfos->second;
      ETG_TRACE_USR2(("IsoContainerChain::getMountInfoByMp(%50s) check %50s", mp.c_str(), mountInfo.mp.c_str()));
      if (isSameFile(mp, mountInfo.mp)) {
         ETG_TRACE_USR3(("IsoContainerChain::getMountInfoByMp(%50s) FOUND dev=%50s", mountInfo.mp.c_str(), mountInfo.device.c_str()));
         return &mountInfo;
      }
   }
   ETG_TRACE_USR4(("IsoContainerChain::getMountInfoByMp(%s) FAILED", mp.c_str()));
   return 0;
}

std::string IsoContainerChain::analyzeParentDir(std::string child, std::map<std::string, MountInfo> &mountInfos) {
   ETG_TRACE_USR4(("IsoContainerChain::analyzeParentDir(%s) START", child.c_str()));

   std::string exportText;
   std::string parentDir=parentDirname(child);
   ETG_TRACE_USR3(("IsoContainerChain::analyzeParentDir(%50s) parentDir=%50s", child.c_str(), parentDir.c_str()));
   // check if it is a mount-point
   MountInfo const *mountInfo=getMountInfoByMp(parentDir, mountInfos);
   if (!mountInfo) {
      ETG_TRACE_USR2(("IsoContainerChain::analyzeParentDir(%s) no mountInfo", child.c_str()));
      return exportText;
   }
   // check if it is an iso-mount
   if (!mountInfo->loopDeviceInfo) {
      ETG_TRACE_USR3(("IsoContainerChain::analyzeParentDir(%s) no loopDeviceInfo", child.c_str()));
      return exportText;
   }

   LoopDeviceInfo const &loopDeviceInfo=*(mountInfo->loopDeviceInfo); 
   // resurse to parent-dir of iso-container
   ETG_TRACE_USR2(("IsoContainerChain::analyzeParentDir(%50s) recurse to %50s", child.c_str(), loopDeviceInfo.container.c_str()));

   exportText=analyzeParentDir(loopDeviceInfo.container, mountInfos);
   ETG_TRACE_USR2(("IsoContainerChain::analyzeParentDir(%50s) returned exportText=%50s", child.c_str(), exportText.c_str()));
   // store settings of loopDevice
   exportText = exportText +
      (exportText.size() ? loopDeviceInfo.containerShort : loopDeviceInfo.container) + 
      " " + intToString(loopDeviceInfo.offset) + 
      " " + intToString(loopDeviceInfo.sizeLimit) + "\n";
   ETG_TRACE_USR2(("IsoContainerChain::analyzeParentDir(%50s) returning exportText=%50s", child.c_str(), exportText.c_str()));
   return exportText;
}


bool IsoContainerChain::store(std::string leafFile, std::string configFile) {   
   ETG_TRACE_USR4(("IsoContainerChain::store(): leafFile: %s", leafFile.c_str()));
   ETG_TRACE_USR4(("IsoContainerChain::store(): configFile: %s", configFile.c_str()));

   std::string tmpFileName="/tmp/isoChain.txt";
   bool res=false;
   if (!exists(leafFile)) {      
      ETG_TRACE_USR2(("IsoContainerChain::store(): not existing leafFile=%s", 
                     leafFile.c_str()));
      return false;
   }
  
   std::map<std::string, LoopDeviceInfo> loopDeviceInfos;
   // **** analyze existing loop-devices *****
   fetchLoopDeviceInfos(loopDeviceInfos);
      
   std::map<std::string, MountInfo> mountInfos;
   // **** analyze existing mounts *****
   fetchMountInfos(loopDeviceInfos, mountInfos );
   
   // ***** analyze starting from leaf-file *****   
   std::string exportText=analyzeParentDir(leafFile, mountInfos) + leafFile + "\n";
   // ***** store the result ******
   ETG_TRACE_USR4(("IsoContainerChain::   configFile=%s", configFile.c_str()));
   res=writeFile(exportText, configFile);

   return res;
} 

}
