/*
 * swu_filesystem.h
 *
 *  Created on: Oct 10, 2013
 *      Author: efs1hi
 */

#ifndef SWU_FILESYSTEM_H_
#define SWU_FILESYSTEM_H_

#if 0 // peha todo moved to swu_constants.hpp
#define CR        (0x0D)
#define LF        (0x0A)
#endif

#include <string>
#include <vector>
#include <unistd.h>
#include <string.h>
#include "swu_types.h"
#include "swu_util.hpp"
#include "swu_constants.hpp"
#include "swu_singleton.hpp"
#include "swu_crypto.hpp"

namespace swu {


bool getFileSize(const std::string& path, off_t &size);

off_t getFileSize(const std::string& path);

bool copyFile(std::string source_filename, std::string destination_filename);

bool compareFiles(std::string filename1, std::string filename2, bool &same);

bool isSameFile(std::string const &nameA, std::string const &nameB);
bool isSameFile(const char* nameA, const char* nameB);

bool genCRCValueFromFile(std::string filename,
                         std::vector< unsigned char > &digest);

bool genCRC(std::string source_filename, std::string crc_filename);

bool checkCRC(std::string source_filename, std::string crc_filename,
              bool &valid);

bool exists(std::string pathname, bool &existing);

bool exists(std::string pathname);

bool symlinkExists(std::string pathname);

bool isDirectory(std::string dirname, bool &directory);

bool isDirectory(std::string dirname);

/** @brief Checks if the path starts with '/'. */
bool isAbsolutePath(const std::string& path);

bool isFile(std::string filename, bool &file);

bool isFile(std::string filename);

bool makeDirectoryRecursive(std::string directoryname, int mode, bool isLeaf=true);

bool removeDirectoryRecursive(std::string directoryname, bool keepRoot=false);

bool removeFile(std::string filename);

// get parent-dir, checks that file exists
bool parentDirname(std::string pathname, std::string &parent);
std::string parentDirname(std::string pathname);

// get fileName without directory, checks that file exists
bool getFileNamePortion(std::string filePathAndName, std::string &fileName);
std::string getFileNamePortion(std::string filePathAndName);

// split pathname, no check for existence
std::string dirName(std::string pathname);
// split pathname, no check for existence
std::string fileName(std::string pathname);



std::string getTempDir();

std::string getTempFile();

class TempFile {
public:
   TempFile() {
      _fileName=getTempFile();
   }
   ~TempFile() {
      removeFile(_fileName);
   }

   std::string const &get() {
      return _fileName;
   }
private:
   std::string _fileName;
};

std::set<std::string> getMatchingFiles(std::string const &dir, std::string const &pattern, bool sarchLinks=false);

std::string realPath(std::string const &path);

bool loadFile(std::string pathname, std::string &data);
bool loadFile(const std::string& pathname, std::vector<SWU_BYTE>& data);
bool loadFile(const std::string& pathname, std::vector<tU32>& data);

bool writeFile(const std::string& content, std::string fileName);
bool writeFile(const std::vector<tU32> content, std::string fileName);

bool matchSHA256Hashes(const std::string &hash, const std::string &signedHash, const std::string &Key);

bool calculateSHA256Hash(std::string & data, std::string & hash);

bool calculateSHA256HashFromFile(const std::string &fileName, std::string &hash);

/**
 * Checks if the message has the given checksum
 *
 * @param msg Pointer holding the message to be checked
 * @param msg_len length of the message in bytes
 * @param checksum Pointer holding the checksum for the message
 * @param checksum_len length of the checksum
 * @param result Outgoing param giving back if the checksum was correct
 * @returns false on error, true if checksum could be checked (also if the checksum was wrong!)
 */
bool checkChecksum(const unsigned char *msg, size_t msg_len,
      const unsigned char *checksum, size_t checksum_len, bool &result);

/**
 * This method can create or verify checksums. It is mostly designed to
 * be called by the more convenient functions genChecksum and verifyChecksum.
 *
 * @param inData A Stream to read the data from, that shall be verified or
 *               the checksums shall be created for.
 * @param checksum An object containing the checksum data. If verify is false
 *                 the vector with the checksums has to be empty (but anything
 *                 else has to be filled) and the checksums will be added. If
 *                 verefy is true the vector has to hold a checksum for each
 *                 block.
 * @param verify if true the function takes the checksums from the checksum
 *               vector and checks if they are valid. if false the checksum
 *               are computed and added to the vector.
 * @param stop_on_error only used if verify is true. Then this method tells
 *                      if the check should immediately leave if a wrong
 *                      checksum was found or if it should read the whole
 *                      stream (an stream it to out if given).
 * @param out if this is not null all data read from inData is written to out
 * @param result has to not null for verification. Then the parameter tells
 *               if the verification succeeded or of the checksum was wrong.
 * @return false on error, true if the verification could be checked or the
 *         checksums could be created. A wrong checksum in verification
 *         also result in a return value true, as it could be checked. Only
 *         result tells then if the the verification was successful.
 */
bool runThroughStream(::std::istream &inData, CCkSumElement const &checksumData,
      bool verify, bool stop_on_error, ::std::ostream *out, bool *result,
      ::std::vector< ::std::vector< uint8_t > > *checksums);

/**
 * Create a checksum.
 *
 * @param inData A Stream with the data the checksums shall be created for.
 * @param checksum An object, that shall hold the checksums. Any fields but
 *                 the checksum vector have to be initialized correctly.
 * @param out if this is not null all data read from inData is written to out
 * @return false on error, true if the checksums could be computed
 */
bool genChecksum(::std::istream &inData, CCkSumElement &checksum,
      ::std::ostream *out = NULL);

/**
 * Verify a checksum.
 *
 * @param inData A Stream this the data to be verified
 * @param checksum All checksum Data
 * @param result A boolean holding the result of the verification (the return
 *               value does not tell if the checksum was valid, only if it
 *               could be checked correctly)
 * @param stop_on_error if true the method will stop immediately after
 *                      finding a block with an invalid checksum. If not, it
 *                      will read also the rest of the file (helpful if
 *                      streaming to out)
 * @param out if this is not null all data read from inData is written to out
 * @return false on error, true if the verification could be checked. A wrong
 *         checksum also result in a return value true, as it could be checked.
 *         Only result tells then if the the verification was successful.
 */
bool verifyChecksum(::std::istream &inData, CCkSumElement const &checksum,
      bool &result, bool stop_on_error = true, ::std::ostream *out = NULL);

// to avoid md5-header here. correct lenght will be checked by static assert in source-file
#define SWU_MD5_DIGEST_LEN 16
class FileCrc {
public:
   FileCrc();
   bool generate(std::string filename);

   bool generateAndWrite(std::string filename, std::string crcDestFile);
   bool doRead(std::string crcSourceFile);
   bool doWrite(std::string crcDestFile);

   bool isValid() {
      return _valid;
   }
   bool operator==(FileCrc const &other);

private:
   unsigned char _digest[SWU_MD5_DIGEST_LEN];
   bool _valid;
};


class ScopedFd {
private:
   ScopedFd & operator=(const ScopedFd &);
   ScopedFd(const ScopedFd &);

public:
   ScopedFd(std::string pathname) :
      _fd(-1),
      _pathname(pathname) {
   }

   ~ScopedFd() {
      doClose();
   }

   bool doOpen(int flags);
   bool doOpenWithMode(int flags, mode_t emode);
   bool doClose();

   int getFd() {
      return _fd;
   }
   bool isOpen() {
      return _fd != -1;
   }
   std::string getName() {
      return _pathname;
   }

private:
   int _fd;
   std::string _pathname;
};

class TestFilesystem : public Singleton<TestFilesystem> {
public:
   void vInit();
   void testExecScript(const char *script);
};
}
#endif
