/*
 * swu_filesystem.cpp
 *
 *  Created on: Oct 10, 2013
 *      Author: efs1hi
 */
//#include "main/swu_component.h"
#include <errno.h>
#include <dirent.h>
#include <fstream>
#include <fcntl.h>
#include <iterator>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include <unistd.h>
#include <sstream>
#include <iomanip>
#include <fstream>
#include <algorithm>
#include <sys/sendfile.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <openssl/rsa.h>
#include <openssl/md5.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#include "swu_execCommand.h"
#include "swu_filesystem.h"
//#include <sys/types.h>

#include "util/swu_trace.h"
#include "util/swu_util.hpp"
//peha todo
// #include "main/swu_systemData.h" 

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_filesystem.cpp.trc.h"
#endif

#define SYSTEM_CALL_MAX_BUFFER 256

using namespace std;

namespace swu {


static unsigned char SHA256DigestHeader[] = { 0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x04, 0x20 };


bool calculateSHA256Hash(string & buffer, string & hash) {
   SHA256_CTX context;
   unsigned char md[SHA256_DIGEST_LENGTH];
   if (!SHA256_Init(&context) || !SHA256_Update(&context, buffer.data(), buffer.length()) || !SHA256_Final(md, &context)) {
      ETG_TRACE_ERR(("calculateSHA256Hash: Can't calculate SHA256"));
      return false;
   }
   char md_copy[SHA256_DIGEST_LENGTH];
   memcpy(md_copy, md, sizeof(md_copy));
   hash.assign(md_copy, sizeof(md_copy));
   return true;
}

/**
 * Calculates SHA256 checksum from the content of a file. Returns a string
 * with 64 hexadecimal digits.
 */
bool calculateSHA256HashFromFile(const string &fileName, string &hash) {
   ifstream file(fileName.c_str());
   stringstream buffer;
   /* read complete file content to a buffer */
   buffer << file.rdbuf();
   if (!file.bad()) {
      string content = buffer.str();
      string h;
      if (swu::calculateSHA256Hash(content, h)) {
         stringstream ss;
         for (uint32_t i = 0; i < h.size(); i++) { // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
            ss << setfill('0') << hex << setw(2) << static_cast < int >(h[i]);
         }
         hash = ss.str();
         return true;
      } else {
         return false;
      }
   }
   ETG_TRACE_ERR(("calculateSHA256HashFromFile: can't read from file %s", fileName.c_str()));
   return false;
}

bool matchSHA256Hashes(const string &hash, const string &signedHash, const string &key) {

   OpenSSL_add_all_algorithms();
   ERR_load_BIO_strings();
   ERR_load_crypto_strings();

   BIO *keybio = BIO_new_mem_buf((void *) key.c_str(),static_cast<int> (key.size() ));
   if (0 == keybio) {
      ETG_TRACE_ERR(("%s: Failed to create key BIO", __func__));
      return false;
   }
   RSA *rsa = 0;
   if (0 == (rsa = PEM_read_bio_RSA_PUBKEY(keybio, &rsa, NULL, NULL))) {
      ETG_TRACE_ERR(("%s: Failed to convert key to RSA", __func__));
      return false;
   }
   BIO_free_all(keybio);
   unsigned char decrypted[4098] = { 0 };
   unsigned char encrypted[4098] = { 0 };
   memcpy(encrypted, signedHash.c_str(), signedHash.size());
   int decrypted_length = RSA_public_decrypt(static_cast<int> (signedHash.size() ), encrypted, decrypted, rsa, RSA_PKCS1_PADDING);
   //  int decrypted_length =static_cast<int> (RSA_public_decrypt(signedHash.size(), encrypted, decrypted, rsa, RSA_PKCS1_PADDING);
   if (decrypted_length != (sizeof(SHA256DigestHeader) + SHA256_DIGEST_LENGTH)) {
      ETG_TRACE_ERR(("matchSHA256Hashes: Decrypted Length = %d, failed", decrypted_length));
      return false;
   }
   RSA_free(rsa);
   if (0 != memcmp(SHA256DigestHeader, decrypted, sizeof(SHA256DigestHeader))) {
      ETG_TRACE_ERR(("%s: PKCS #1 SHA-256 DigestInfo mismatch", __func__));
      return false;
   }
   if (0 != memcmp(hash.c_str(), decrypted + sizeof(SHA256DigestHeader), SHA256_DIGEST_LENGTH)) {
      ETG_TRACE_ERR(("%s: decrypted/given hash mismatch", __func__));
      return false;
   }
   return true;
}

bool ScopedFd::doOpen(int flags) {
   _fd = open(_pathname.c_str(), flags);
   ETG_TRACE_USR4(("ScopedFd::doOpen(): (flags=0x%08x) fd=%d  %s", flags, _fd, _pathname.c_str()));
   if (_fd == -1) {
      ETG_TRACE_ERRMEM(("ScopedFd::doOpen(): could not open (flags=0x%08x, errno=%8d) %s", flags, errno, _pathname.c_str()));
      ETG_TRACE_ERR(("ScopedFd::doOpen(): could not open (flags=0x%08x, errno=%8d) %s", flags, errno, _pathname.c_str()));

      std::string command = "/bin/mount >> /tmp/robust_file_ls_out";
      execCommand(command.c_str());
      std::ifstream infile("/tmp/filesystem_ls_out");
      std::string infile_line;
      while (std::getline(infile, infile_line)) {
         ETG_TRACE_FATAL(("        MOUNT $> %s", infile_line.c_str()));
         ETG_TRACE_ERRMEM(("        MOUNT $> %s", infile_line.c_str()));
      }
      infile.close();
      unlink("/tmp/filesystem_ls_out");

      command = "ls -al " + _pathname + " >> /tmp/robust_file_ls_out";
      execCommand(command.c_str());
      std::ifstream infile2("/tmp/filesystem_ls_out");
      while (std::getline(infile2, infile_line)) {
         ETG_TRACE_FATAL(("        $> %s", infile_line.c_str()));
         ETG_TRACE_ERRMEM(("        $> %s", infile_line.c_str()));
      }
      infile2.close();
      unlink("/tmp/filesystem_ls_out");
   }
   return _fd != -1;
}

bool ScopedFd::doOpenWithMode(int flags, mode_t emode) {
	_fd = open(_pathname.c_str(), flags, emode);
	ETG_TRACE_USR4(("ScopedFd::doOpen(): (flags=0x%08x) fd=%d  %s", flags, _fd, _pathname.c_str()));
	if (_fd == -1) {
	  ETG_TRACE_ERRMEM(("ScopedFd::doOpen(): could not open (flags=0x%08x, errno=%8d) %s", flags, errno, _pathname.c_str()));
	  ETG_TRACE_ERR(("ScopedFd::doOpen(): could not open (flags=0x%08x, errno=%8d) %s", flags, errno, _pathname.c_str()));

	  std::string command = "/bin/mount >> /tmp/robust_file_ls_out";
	  execCommand(command.c_str());
	  std::ifstream infile("/tmp/filesystem_ls_out");
	  std::string infile_line;
	  while (std::getline(infile, infile_line)) {
		 ETG_TRACE_FATAL(("        MOUNT $> %s", infile_line.c_str()));
		 ETG_TRACE_ERRMEM(("        MOUNT $> %s", infile_line.c_str()));
	  }
	  infile.close();
	  unlink("/tmp/filesystem_ls_out");

	  command = "ls -al " + _pathname + " >> /tmp/robust_file_ls_out";
	  execCommand(command.c_str());
	  std::ifstream infile2("/tmp/filesystem_ls_out");
	  while (std::getline(infile2, infile_line)) {
		 ETG_TRACE_FATAL(("        $> %s", infile_line.c_str()));
		 ETG_TRACE_ERRMEM(("        $> %s", infile_line.c_str()));
	  }
	  infile2.close();
	  unlink("/tmp/filesystem_ls_out");
   }
   return _fd != -1;
}

bool ScopedFd::doClose() {
   if (!isOpen()) {
      return true;
   }
   _fd = close(_fd) ? _fd : -1;
   if (_fd != -1) {
      ETG_TRACE_ERR(("ScopedFd::doClose(): could not close %s", _pathname.c_str()));
      return false;
   }
   return true;
}

bool FileCrc::doWrite(std::string crcDestFile) {
   ScopedFd destFd(crcDestFile);
   if (!destFd.doOpen(O_WRONLY | O_CREAT | O_TRUNC)) {
      ETG_TRACE_ERRMEM(("Error opening (errno %8d) %s for writing", errno, destFd.getName().c_str()));
      ETG_TRACE_ERR(("Error opening (errno %8d) %s for writing", errno, destFd.getName().c_str()));
      return false;
   }

   char cksSumChar[MD5_DIGEST_LENGTH * 2];
   binToHex(&_digest[0], MD5_DIGEST_LENGTH, cksSumChar);

   if (MD5_DIGEST_LENGTH * 2 != write(destFd.getFd(), cksSumChar, MD5_DIGEST_LENGTH * 2)) {
      ETG_TRACE_ERRMEM(("ERROR: Could not write %d bytes (errno=%8d) to %100s", MD5_DIGEST_LENGTH, errno, crcDestFile.c_str()));
      ETG_TRACE_ERR(("ERROR: Could not write %d bytes (errno=%8d) to %100s", MD5_DIGEST_LENGTH, errno, crcDestFile.c_str()));
      return false;
   }
   return true;
}

bool FileCrc::doRead(std::string crcSourceFile) {
   _valid = false;
   off_t fileSize;
   if (!getFileSize(crcSourceFile, fileSize)) {
      ETG_TRACE_ERR(("FileCrc::doRead(): Error getting file Size (%s)", crcSourceFile.c_str()));
   }
   if (fileSize != MD5_DIGEST_LENGTH * 2) {
      ETG_TRACE_ERR(("FileCrc::doRead(): invalid len (%s)", crcSourceFile.c_str()));
      return false;
   }
   ScopedFd fd(crcSourceFile);
   if (!fd.doOpen(O_RDONLY)) {
      ETG_TRACE_ERR(("FileCrc::doRead(): could not open (%s)", crcSourceFile.c_str()));
      return false;
   }
   char inbuf[MD5_DIGEST_LENGTH * 2];
   ssize_t readLen = read(fd.getFd(), inbuf, static_cast<size_t> (fileSize));  // fileSize: gen3armmake, gen3lsim: conversion to 'size_t {aka unsigned int}' from 'off_t {aka long long int}'
   if (readLen != fileSize) { //gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions so readLen is updated to ssize_t
      ETG_TRACE_ERR(("FileCrc::doRead(): could not read (%s)", crcSourceFile.c_str()));
      return false;
   }
   hexToBin(inbuf, MD5_DIGEST_LENGTH * 2, _digest);

   _valid = true;
   return true;

}

bool getFileSize(const std::string& path, off_t &size) {
   struct stat dir_stat;
   size = 0;
   memset(&dir_stat, 0, sizeof(dir_stat));
   int r = stat(path.c_str(), &dir_stat);
   if (r != 0) {
      // TODO(eff): Logging needed!
      ETG_TRACE_ERR(("ERROR: coud not get stats for %s.", path.c_str()));
      return false;
   }
   ETG_TRACE_USR1(("getFileSize %50s:sizeof(off_t)=%u size=%u",
                    path.c_str(), sizeof(off_t), dir_stat.st_size));
   size = dir_stat.st_size;
   return true;
}

off_t getFileSize(const std::string& path) {
   off_t size;
   return getFileSize(path, size) ? size : 0;
}

bool copyFile(std::string source_filename, std::string destination_filename) {
   bool result;
   ETG_TRACE_USR4(("copyFile %100s --> %s.", source_filename.c_str(), destination_filename.c_str()));
   if (!exists(source_filename, result)) {
      ETG_TRACE_ERR(("Error while trying to check if source %s exists.", source_filename.c_str()));
      return false;
   }
   if (!result) {
      ETG_TRACE_ERR(("Source of copy (%s) does not exist.", source_filename.c_str()));
      return false;
   }
   if (!isFile(source_filename, result)) {
      ETG_TRACE_ERR(("Error while trying to check if source %s is a file.", source_filename.c_str()));
      return false;
   }
   if (!result) {
      ETG_TRACE_ERR(("Source of copy (%s) is not a file.", source_filename.c_str()));
      return false;
   }

   if (!exists(destination_filename, result)) {
      ETG_TRACE_ERR(("Error while trying to check if destination %s exists.", destination_filename.c_str()));
      return false;
   }
   if (result) {
      // Destination exists
      if (!isFile(destination_filename, result)) {
         ETG_TRACE_ERR(("Error while trying to check if destination %s is a file.", destination_filename.c_str()));
         return false;
      }
      if (!result) {
         ETG_TRACE_ERR(("Destination %s exists, but is no file.", destination_filename.c_str()));
         return false;
      }
      // The file exists and is a file, so we shall overwrite it.
      if (unlink(destination_filename.c_str())) {
         ETG_TRACE_ERR(("Error removing existing version of file, I shall copy to (%100s), errno %8d", destination_filename.c_str(), errno));
         return false;
      }
   } else {
      // Destination does not exist. Check if directory exists and create it if not.
      std::string directory_path;
      if (!parentDirname(destination_filename, directory_path)) {
         ETG_TRACE_ERR(("Error getting parent directory of %s", destination_filename.c_str()));
         return false;
      }

      if (!exists(directory_path, result)) {
         ETG_TRACE_ERR(("Unable to check if directory %s exists.", directory_path.c_str()));
         return false;
      }
      if (!result) {
         ETG_TRACE_ERR(("parent directory of %100s (%s) does not exist.", destination_filename.c_str(), directory_path.c_str()));
         return false;
      }
      if (!isDirectory(directory_path, result)) {
         ETG_TRACE_ERR(("Error checking if %s is a path.", directory_path.c_str()));
         return false;
      }
      if (!result) {
         ETG_TRACE_ERR(("parent %100s of %s is not a directory.", directory_path.c_str(), destination_filename.c_str()));
         return false;
      }
   }

   ScopedFd source_fd(source_filename);
   if (!source_fd.doOpen(O_RDONLY)) {
      std::string strName = source_fd.getName(); //Coverity fix for 209484
      ETG_TRACE_ERR(("Error opening %100s for reading, errno %8d", strName.c_str(), errno));
      return false;
   }

   struct stat source_status;
   memset(&source_status, 0, sizeof(source_status));
   if (stat(source_filename.c_str(), &source_status)) {
	   ETG_TRACE_ERR(("Error getting stat of %100s, errno %8d", source_filename.c_str(), errno));
	   return false;
   }

   ScopedFd dest_fd(destination_filename);
   if (!dest_fd.doOpenWithMode(O_WRONLY | O_CREAT | O_TRUNC, source_status.st_mode)) {
      std::string strName = dest_fd.getName(); //Coverity fix for 209484
      ETG_TRACE_ERR(("Error opening %100s for writing, errno %8d", strName.c_str(), errno));
      return false;
   }

//   ScopedFd dest_fd(destination_filename);
//   if (!dest_fd.doOpen(O_WRONLY | O_CREAT | O_TRUNC)) {
//	   ETG_TRACE_ERR(("Error opening %100s for writing, errno %8d", dest_fd.getName().c_str(), errno));
//	   return false;
//   }

   ssize_t copied = sendfile(dest_fd.getFd(), source_fd.getFd(), NULL, static_cast<size_t> (source_status.st_size)); // source_status.st_size: gen3armmake, gen3lsim: conversion to 'size_t {aka unsigned int}' from 'off_t {aka long long int}'
   if (copied == -1) {
      ETG_TRACE_ERR(("Error sending data of %100s to %50s (errno %8d)", source_filename.c_str(), destination_filename.c_str(), errno));
      return false;
   }
   if (copied != source_status.st_size) {
      ETG_TRACE_ERR(("Error sending data of %80s to %80s, copied %8d bytes, but should be %8d (errno %8d)", source_filename.c_str(), destination_filename.c_str(), copied, source_status.st_size, errno));
      return false;
   }
   return true;
}

bool compareFiles(std::string filename1, std::string filename2, bool &same) {
   bool b_result;
   if (!exists(filename1, b_result)) {
      ETG_TRACE_ERR(("Error while trying to check if %s exists.", filename1.c_str()));
      return false;
   }
   if (!b_result) {
      ETG_TRACE_ERR(("Source of copy (%s) does not exist.", filename1.c_str()));
      return false;
   }
   if (!isFile(filename1, b_result)) {
      ETG_TRACE_ERR(("Error while trying to check if source %s is a file.", filename1.c_str()));
      return false;
   }
   if (!b_result) {
      ETG_TRACE_ERR(("Source of copy (%s) is not a file.", filename1.c_str()));
      return false;
   }

   if (!exists(filename2, b_result)) {
      ETG_TRACE_ERR(("Error while trying to check if %s exists.", filename2.c_str()));
      return false;
   }
   if (!b_result) {
      ETG_TRACE_ERR(("Source of copy (%s) does not exist.", filename2.c_str()));
      return false;
   }
   if (!isFile(filename2, b_result)) {
      ETG_TRACE_ERR(("Error while trying to check if source %s is a file.", filename2.c_str()));
      return false;
   }
   if (!b_result) {
      ETG_TRACE_ERR(("Source of copy (%s) is not a file.", filename2.c_str()));
      return false;
   }

   struct stat file1_status;
   struct stat file2_status;
   memset(&file1_status, 0, sizeof(file1_status));
   memset(&file2_status, 0, sizeof(file2_status));
   if (stat(filename1.c_str(), &file1_status)) {
      ETG_TRACE_ERR(("Error getting stat (errno %8d) of %s", errno, filename1.c_str()));
      return false;
   }
   if (stat(filename2.c_str(), &file2_status)) {
      ETG_TRACE_ERR(("Error getting stat (errno %8d) of %s", errno, filename2.c_str()));
      return false;
   }

   if (file1_status.st_size != file2_status.st_size) {
      same = false;
      return true;
   }

   ScopedBuffer < char > buff1(4096, true);
   char *buffer1 = buff1.getBuf();
   ScopedBuffer < char > buff2(4096, true);
   char *buffer2 = buff2.getBuf();
   SWU_ASSERT_RETURN_FALSE(buffer1 && buffer2);

   ScopedFd fd1(filename1);
   if (!fd1.doOpen(O_RDONLY)) {
      ETG_TRACE_ERR(("unable to open file (errno %8d) %s for reading",
      errno, filename1.c_str()));
      return false;

   }

   ScopedFd fd2(filename2);
   if (!fd2.doOpen(O_RDONLY)) {
      ETG_TRACE_ERR(("unable to open file (errno %8d) %s for reading",
      errno, filename2.c_str()));
      return false;
   }

   ssize_t read1 = 0;
   ssize_t read2 = 0;
   int result = 0;

   do {
      do {
         read1 = read(fd1.getFd(), buffer1, 4096);
         if ((read1 < 0) && (errno != EINTR)) {
            ETG_TRACE_ERR(("unable to read (errno %8d) from file %s", errno, filename1.c_str()));
            result = -1;
         }
      } while ((!read1) && (errno == EINTR));
      if (result == 0) {
         int step_read2 = 0;
         read2 = 0;
         do {
            step_read2 =static_cast<int> (read(fd2.getFd(), buffer2 + read2, read1 - read2) );
            read2 += step_read2;
            if ((read2 < 0) && (errno != EINTR)) {
               ETG_TRACE_ERR(("unable to read (errno %8d) from file %s, ",
               errno, filename1.c_str()));
               result = -1;
            }
         } while ((read2 < read1) && ((step_read2 != 0) || (errno == EINTR)) && (!result));

         if ((!result) && ((read1 != read2) || (memcmp(buffer1, buffer2, read1)))) {
            result = 1;
         }
      }
   } while ((read1 != 0) && (read2 != 0) && (result == 0));

   if (!fd1.doClose()) {
      ETG_TRACE_ERR(("Error closing (errno %8d) %s", errno, fd1.getName().c_str()));
      result = -1;
   }
   if (!fd2.doClose()) {
      ETG_TRACE_ERR(("Error closing (errno %8d) %s", errno, fd2.getName().c_str()));
      result = -1;
   }

   if (result == -1)
      return false;

   same = (result == 0);
   return true;
}

bool isSameFile(const char* nameA, const char* nameB)
{
   struct stat statA;
   struct stat statB; 
   if (stat(nameA, &statA)) {
      ETG_TRACE_ERR(("IsSameFile: could not stat %s", nameA));
   }
   if (stat(nameB, &statB)) {
      ETG_TRACE_ERR(("IsSameFile: could not stat %s", nameB));
   }

    return statA.st_dev == statB.st_dev && statA.st_ino == statB.st_ino;
}

bool isSameFile(std::string const &nameA, std::string const &nameB) {
   return isSameFile(nameA.c_str(), nameB.c_str());
}


bool FileCrc::generate(std::string filename) {
   _valid = false;
   MD5_CTX context;
   if (!MD5_Init(&context)) {
      ETG_TRACE_ERRMEM(("Error Initializing MD5 context."));
      ETG_TRACE_ERR(("Error Initializing MD5 context."));
      return false;
   }
   ScopedBuffer < char > buffer(4096);
   if (!buffer.getBuf()) {
      ETG_TRACE_ERRMEM(("unable to allocate memory for buffer"));
      ETG_TRACE_ERR(("unable to allocate memory for buffer"));
      return false;
   }
   ssize_t read_bytes = 0;
   ScopedFd fd1(filename);
   if (!fd1.doOpen(O_RDONLY)) {
      ETG_TRACE_ERRMEM(("FileCrc::generate(): could not open for reading (%s)", filename.c_str()));
      ETG_TRACE_ERR(("FileCrc::generate(): could not open for reading (%s)", filename.c_str()));
      return false;
   }
   do {
      do {
         read_bytes = read(fd1.getFd(), buffer.getBuf(), buffer.getSize());
         if ((read_bytes < 0) && errno && (errno != EINTR)) {
            ETG_TRACE_ERRMEM(("Error: Could not read (errno %8d) from %s", errno, filename.c_str()));
            ETG_TRACE_ERR(("Error: Could not read (errno %8d) from %s", errno, filename.c_str()));
            return false;
         }
      } while ((!read_bytes) && (errno == EINTR));
      if (!MD5_Update(&context, buffer.getBuf(), read_bytes)) {
         ETG_TRACE_ERRMEM(("unable to update MD5 context."));
         ETG_TRACE_ERR(("unable to update MD5 context."));
         return false;
      }
   } while (read_bytes);

   if (!MD5_Final(_digest, &context)) {
      ETG_TRACE_ERRMEM(("unable to final MD5 digest."));
      ETG_TRACE_ERR(("unable to final MD5 digest."));
      return false;
   }
   _valid = true;
   return true;
}

FileCrc::FileCrc() :
   _valid(false){
   _digest[0]='\0';
   SWU_STATIC_ASSERT(SWU_MD5_DIGEST_LEN == MD5_DIGEST_LENGTH);
}

bool FileCrc::operator==(FileCrc const &other) {
   if (!(_valid && other._valid)) {
      ETG_TRACE_ERR(("FileCrc::equal: _valid=%u other._valid=%u.", _valid, other._valid));
      return false;
   }
   if (memcmp(_digest, other._digest, MD5_DIGEST_LENGTH)) {
      ETG_TRACE_ERR(("FileCrc::equal: different data!!!"));
      return false;
   }
   return true;
}

bool FileCrc::generateAndWrite(std::string filename, std::string crcDestFile) {
   if (!generate(filename)) {
      ETG_TRACE_ERRMEM(("unable to generate Checksum for %s", filename.c_str()));
      ETG_TRACE_FATAL(("unable to generate Checksum for %s", filename.c_str()));
      return false;
   }
   if (!doWrite(crcDestFile)) {
      ETG_TRACE_ERRMEM(("unable to write Checksum for %s", filename.c_str()));
      ETG_TRACE_FATAL(("unable to write Checksum for %s", filename.c_str()));
      return false;
   }
   return true;
}

bool genCRC(std::string source_filename, std::string crc_filename) {
   FileCrc fileCrc;
   return fileCrc.generateAndWrite(source_filename, crc_filename);
}

bool checkCRC(std::string source_filename, std::string crc_filename, bool &equal) {
   ETG_TRACE_USR2(("checkCRC: %100s, %s", source_filename.c_str(), crc_filename.c_str()));

   FileCrc calcCrc;
   calcCrc.generate(source_filename);
   FileCrc storedCrc;
   storedCrc.doRead(crc_filename);

   equal = (calcCrc == storedCrc);
   bool res = calcCrc.isValid() && storedCrc.isValid();
   if (!(equal && res)) {
      ETG_TRACE_COMP(("checkCRC: res=%d equal=%d file:%s", res, equal, source_filename.c_str()));
   }
   return res;
}

bool exists(std::string pathname, bool &existing) {
   struct stat path_stat;
   memset(&path_stat, 0, sizeof(path_stat));
   int r = stat(pathname.c_str(), &path_stat);
   if (r == 0) {
      existing = true;
      return true;
   } else if (errno == ENOENT) {
      existing = false;
      return true;
   }
   ETG_TRACE_ERR(("error trying to get stat (errno %8d)of file %s", errno, pathname.c_str()));
   return false;
}

// simple version
bool exists(std::string pathname) {
   bool bRes = false;
   return exists(pathname, bRes) && bRes;
}

bool symlinkExists(std::string pathname) {
   struct stat path_stat;
   memset(&path_stat, 0, sizeof(path_stat));
   int r = lstat(pathname.c_str(), &path_stat);
   if (r == 0) {
      return true;
   }else {
      ETG_TRACE_ERR(("error trying to get lstat (errno %8d)of file %s", errno, pathname.c_str()));
      return false;
   }

}

bool isDirectory(std::string dirname, bool &directory) {
   directory = false;
   struct stat dir_stat;
   memset(&dir_stat, 0, sizeof(dir_stat));
   if (stat(dirname.c_str(), &dir_stat)) {
      ETG_TRACE_ERR(("Unable to get stat of (errno %8d) %s, ", errno, dirname.c_str()));
      return false;
   }
   directory = S_ISDIR(dir_stat.st_mode);
   return true;
}

// simple version
bool isDirectory(std::string dirname) {
   bool bRes = false;
   return isDirectory(dirname, bRes) && bRes;
}

bool isAbsolutePath(const ::std::string& path) {
   if (path.length() > 0 && path[0] == '/') {
      return true;
   }
   return false;
}

bool isNetworkPath(const ::std::string& path) {
   //compiler warning fix. Patter with no escape sequence to identify the network path
   //std::string pattern = "\b\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}\b\:";
   std::string pattern = "((([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){3}(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\:))";

   if (path.length() > 0 && bFindPatternInString(pattern, path)) {
      return true;
   }
   // Remove this trace once working
   //ETG_TRACE_ERR(("Unable to match %s with %s", path.c_str(), pattern.c_str()));
   return false;
}

bool isFile(std::string filename, bool &file) {
   file = false;
   struct stat file_stat;
   memset(&file_stat, 0, sizeof(file_stat));
   if (stat(filename.c_str(), &file_stat)) {
      ETG_TRACE_ERR(("Unable to get stat of (errno %8d)%s ", errno, filename.c_str()));
      return false;
   }
   file = S_ISREG(file_stat.st_mode);
   return true;
}

// simple version
bool isFile(std::string filename) {
   bool bRes = false;
   return isFile(filename, bRes) && bRes;
}

bool makeDirectoryRecursive(std::string directoryname, int mode, bool isLeaf) {
   bool b_result;
   if (!exists(directoryname, b_result)) {
      ETG_TRACE_ERR(("Error checking if %s exists.", directoryname.c_str()));
      return false;
   }
   if (b_result) {
      // the path exists allready! If it is a directory, everything is o.k...
      if (!isDirectory(directoryname, b_result)) {
         ETG_TRACE_ERR(("Error checking if the existing path %s is a directory.", directoryname.c_str()));
         return false;
      }
      return b_result;
   }

   std::string directory_path;
   if (!parentDirname(directoryname, directory_path)) {
      ETG_TRACE_ERR(("Error getting parent directory of %s", directoryname.c_str()));
      return false;
   }

   // child-directories are not forced to the correct mode (isLeaf=false), see below
   if (!makeDirectoryRecursive(directory_path, mode, false))
      return false;

   if (mkdir(directoryname.c_str(), mode) != 0) {
      ETG_TRACE_ERR(("Error creating directory (errno %8d) %s", errno, directoryname.c_str()));
      return false;
   }
   if (isLeaf) {
      /*
        This is the directory we have been requested to create (isLeaf=true as default-value)
        Since we face a strange bug that mkdir does not apply the correct mode,
        so we do it manually here.
       */
      (void)chmod(directoryname.c_str(), mode);
   }
   return true;
}

bool removeDirectoryRecursive(std::string directoryname, bool keepRoot)
{
   DIR *directory = opendir(directoryname.c_str());
   struct dirent *directoryEntry = NULL;
   if (directory == NULL)
   {
      ETG_TRACE_ERR(("Cannot get directory entry for %s", directoryname.c_str()));
      return false;
   }

   while ((directoryEntry = readdir(directory)) != 0)
   {
      // CHECKME: What about lost+found? We can't remove that folder easily
      if (strcmp(directoryEntry->d_name, ".") && strcmp(directoryEntry->d_name, ".."))
      {
         std::string entryName = directoryname + "/" + directoryEntry->d_name;
         bool check_dir;
         if (!isDirectory(entryName, check_dir)) {
            ETG_TRACE_ERR(("Could not check if entity is a directory %s", entryName.c_str()));
            closedir(directory);
            return false;
         }
         if (check_dir)
         {
            if (!removeDirectoryRecursive(entryName))
            {
               ETG_TRACE_ERR(("Could not remove directory %s", entryName.c_str()));
               closedir(directory);
               return false;
            }
         }
         else
         {
            unlink(entryName.c_str());
         }
      }
   }

   closedir(directory);
   if (!keepRoot && !rmdir(directoryname.c_str()))
   {
      ETG_TRACE_ERR(("Can not remove directory %s", directoryname.c_str()));
      ETG_TRACE_ERR(("    Error is %s", strerror(errno)));
      ETG_TRACE_ERR(("    Errno is %d", errno));
      if (errno)
      {
         ETG_TRACE_ERR(("    Aborting, rmdir failed"));
         return false;
      }
      else
      {
         ETG_TRACE_ERR(("    Continue, rmdir ok"));
      }
   }
   return true;
}

bool removeFile(std::string filename) {
   int result = unlink(filename.c_str());
   if (result && result != ENOENT) {
      ETG_TRACE_ERR(("Error %d While deleting %s", result, filename.c_str()));
      return false;
   }
   return true;
}


bool parentDirname(std::string pathname, std::string &parent) {
   char* buffer = strdup(pathname.c_str());
   char* p = dirname(buffer);
   if (!p) {
      ETG_TRACE_ERR(("Was not able to get destination of %s", pathname.c_str()));
      free(buffer);
      return false;
   }
   parent = std::string(p);
   free(buffer);

   return true;
}

std::string parentDirname(std::string pathname) {
   std::string parentDir;
   return  parentDirname(pathname, parentDir) ? parentDir : "";
}

bool getFileNamePortion(std::string filePathAndName, std::string &fileName) {
   if (!isFile(filePathAndName)) {
      return false;
   }
   char* buffer = strdup(filePathAndName.c_str());
   char* p = basename(buffer);
   if (!p) {
      ETG_TRACE_ERR(("Was not able to get filename of %s", filePathAndName.c_str()));
      free(buffer);
      return false;
   }
   fileName = std::string(p);
   free(buffer);

   return true;
}

std::string getFileNamePortion(std::string filePathAndName) {
   std::string fileName;
   return  getFileNamePortion(filePathAndName, fileName) ? fileName : "";
}

std::string dirName(std::string pathname) {
   if (!pathname.size()) {
      return "";
   }
   size_t slashPos=pathname.find_last_of("/");
   if (slashPos != std::string::npos) {
      return "";
   }
   else {
      ETG_TRACE_ERR(("writeFile(): found slash at %u", slashPos));
      pathname=pathname.substr(0, slashPos);
      ETG_TRACE_ERR(("writeFile(): pathname: %s", pathname));
   }
   while (pathname.size() && pathname.at(pathname.length()-1)=='/') {
      pathname=pathname.substr(0, pathname.size() -1);
   }
   return pathname;
}

std::string fileName(std::string pathname) {
   size_t slashPos=pathname.find_last_of("/");
   if (slashPos == std::string::npos) {
      return pathname;
   }
   pathname=pathname.substr(slashPos+1);
   return pathname;

}

std::string getTempDir() {
   char tmpDirName[] = "/tmp/swupd/tempDir.XXXXXX";
   mode_t oldMask=umask(S_IRUSR | S_IWUSR);
   if (!mkdtemp(tmpDirName)) {
      ETG_TRACE_ERR(("getTempDir: could not create tmp-dir"));
      makeDirectoryRecursive(tmpDirName, 0700);
   }
   return tmpDirName;
}


std::string getTempFile() {
   char tmpFileName[] = "/tmp/swupd/tempFile.XXXXXX";
   mode_t oldMask=umask(S_IRUSR | S_IWUSR);
   int fd=mkstemp(tmpFileName);
   (void) umask(oldMask);
   if (fd<0) {
      ETG_TRACE_ERR(("getTempDir: could not create tmp-file"));
      return "/tmp/swupd/tempFile.DEFAULT";
   }
   close(fd);
   return tmpFileName;
}

std::set<std::string> getMatchingFiles(std::string const &dir, std::string const &pattern="*", bool searchLinks) {
   ETG_TRACE_USR4(("getMatchingFiles() START pattern=%20s dir=%s", pattern.c_str(), dir.c_str()));
   std::set<std::string> res;
   TempFile tempFile;
   std::string logFileName=tempFile.get();


   string cmd="find " + dir + " -maxdepth 1 -type " + (searchLinks ? "l" : "f") + " -name \"" + pattern + "\" >> " + logFileName;
   ETG_TRACE_USR4(("Matching File cmd - %s", cmd.c_str()));
   ETG_TRACE_FATAL(("Matching File cmd - %s", cmd.c_str()));
   //execCommand(cmd.c_str());
   system(cmd.c_str());

   ifstream logData;
   logData.open(logFileName.c_str());
   if (!logData.is_open()) {
      return res;
   }

   std::string line;
   while (getline(logData, line)) {
      if (line.size()) {
         ETG_TRACE_USR4(("getMatchingFiles() found:%s", line.c_str()));
         ETG_TRACE_FATAL(("getMatchingFiles() found:%s", line.c_str()));
         res.insert(line);
      }
   }
   
   return res;
}


std::string realPath(std::string const &path) {
   ETG_TRACE_USR4(("realPath(%s) START", path.c_str()));
   TempFile tempFile;
   std::string logFileName=tempFile.get();

   string cmd=string("realpath ") + path + " >> " + logFileName;
   ETG_TRACE_ERR(("realPath: call: %s", cmd.c_str()));
   if (execCommand(cmd.c_str())) {
      ETG_TRACE_ERR(("realPath: cmd failed:%s", cmd.c_str()));      
      return path;
   }

   ifstream logData;
   logData.open(logFileName.c_str());
   if (!logData.is_open()) {
      ETG_TRACE_ERR(("realPath: cant open logfile: %s", logFileName.c_str()));
      return path;
   }
   std::string line;
   getline(logData, line);
   logData.close();

   ETG_TRACE_USR4(("realPath(%30s)=%s ", path.c_str(), line.c_str()));

   return line;
}


bool loadFile(std::string pathname, std::string &data) {
   std::ifstream input(pathname.c_str(), ios_base::binary | ios_base::in);
   if (!input.good()) {
      ETG_TRACE_ERR(("Could not open file %s", pathname.c_str()));
      return false;
   }
   data = "";
   input.seekg(0, std::ios::end);
   if (!input.good()) {
      ETG_TRACE_ERR(("Could not seek to end of file %s", pathname.c_str()));
      return false;
   }
   int length =static_cast<int> (input.tellg() );
   data.resize(static_cast<size_t> (input.tellg()));  // gen3armmake, gen4rcar: conversion to 'std::__cxx11::basic_string<char>::size_type {aka unsigned int}' from 'std::streamoff {aka long long int}'
   if (length < 0) {
      ETG_TRACE_ERR(("Could not get length of %s", pathname.c_str()));
      return false;
   }
   input.seekg(0, std::ios::beg);
   if (!input.good()) {
      ETG_TRACE_ERR(("Could not seek to start of file %s", pathname.c_str()));
      return false;
   }
   input.read(&data[0], data.size());
   if (!input.good()) {
      ETG_TRACE_ERR(("Could not read file %s", pathname.c_str()));
      return false;
   }
   return true;
}

bool loadFile(const std::string& pathname, std::vector<SWU_BYTE>& data)
{
   std::ifstream file(pathname.c_str(), std::ios::binary);
   if ( !file ) {
      ETG_TRACE_ERR(("Could not open file %s", pathname.c_str() ));
      return false;
   }

   // get filesize to reserve space in buffer
   struct stat statbuf;
   if (::stat(pathname.c_str(), &statbuf) == -1) {
      ETG_TRACE_ERR(("Could not stat file %s", pathname.c_str() ));
      return false;
   }
   if (statbuf.st_size == 0) {
      ETG_TRACE_USR3(( "Warning: loading empty file %s", pathname.c_str() ));
   }

   // copy file content to buffer
   try {
      data.resize(static_cast<size_t> (statbuf.st_size));   // gen3armmake, gen4rcar: conversion to 'std::vector<unsigned char>::size_type {aka unsigned int}' from '__off64_t {aka long long int}'

      file.read(reinterpret_cast<char *>(&data[0]), static_cast<size_t> (statbuf.st_size)); // gen3armmake, gen4rcar: conversion to 'std::streamsize {aka int}' from '__off64_t {aka long long int}'
   } catch (std::exception& e) {
      ETG_TRACE_ERR(( "Could not copy file to buffer: %s", e.what() ));
      return false;
   }
   return true;
}

bool loadFile(const std::string& pathname, std::vector<tU32>& data)
{
   std::ifstream file(pathname.c_str(), std::ios::binary);
   if ( !file ) {
      ETG_TRACE_ERR(("Could not open file %s", pathname.c_str() ));
      return false;
   }

   // get filesize to reserve space in buffer
   struct stat statbuf;
   if (::stat(pathname.c_str(), &statbuf) == -1) {
      ETG_TRACE_ERR(("Could not stat file %s", pathname.c_str() ));
      return false;
   }
   if (statbuf.st_size == 0) {
      ETG_TRACE_USR3(( "Warning: loading empty file %s", pathname.c_str() ));
   }

   // copy file content to buffer
   try {
      data.resize(static_cast<size_t> (statbuf.st_size));   // gen3armmake, gen4rcar: conversion to 'std::vector<unsigned int>::size_type {aka unsigned int}' from '__off64_t {aka long long int}'

      file.read(reinterpret_cast<char *>(&data[0]), static_cast<size_t> (statbuf.st_size));  // gen3armmake, gen4rcar: conversion to 'std::streamsize {aka int}' from '__off64_t {aka long long int}'
   } catch (std::exception& e) {
      ETG_TRACE_ERR(( "Could not copy file to buffer: %s", e.what() ));
      return false;
   }
   return true;
}



bool writeFile(const std::string& content, std::string fileName) {
   std::string dir=dirName(fileName);
   if (dir.size() && !makeDirectoryRecursive(dir, 0700)) {
      ETG_TRACE_ERR(("writeFile(): could not create dir=%s", 
                     dir.c_str()));
      return false;
   }

   ScopedFd destFd(fileName);
   
   if (!destFd.doOpen(O_WRONLY | O_CREAT | O_TRUNC)) {
      ETG_TRACE_ERRMEM(("writeFile:Error opening (errno %8d) %s for writing", errno, destFd.getName().c_str()));
      ETG_TRACE_ERR(("writeFile:Error opening (errno %8d) %s for writing", errno, destFd.getName().c_str()));
      return false;
   }
   if (static_cast<ssize_t> (content.size()) != write(destFd.getFd(), content.c_str(), content.size())) {   // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      ETG_TRACE_ERR(("writeFile(): write to file=%s failed", fileName.c_str()));
   }
   
   return true;
}

bool writeFile(const std::vector<tU32> content, std::string fileName) {
   std::string dir=dirName(fileName);
   if (dir.size() && !makeDirectoryRecursive(dir, 0700)) {
      ETG_TRACE_ERR(("writeFile(): could not create dir=%s", 
                     dir.c_str()));
      return false;
   }

   std::ofstream file(fileName.c_str());
   for (tU16 index = 0; index < content.size(); index++) {      
      file.write((const char*)&content.at(index), sizeof(tU32));
   }
   file.close();
   
   return true;
}



class ScopedContext {
public:
   ScopedContext(EVP_MD_CTX *context) :
         _context(context) {
   }

   ~ScopedContext() {
      if (_context) {
         EVP_MD_CTX_destroy(_context);
        _context=0;      
      }

   }

private:
   EVP_MD_CTX *_context;
};





bool checkChecksum(const unsigned char *msg, size_t msg_len, const unsigned char *checksum, size_t checksum_len, bool &result) {
#if 0 // peha todo
   if (Config::instance()->getIgnoreCkSumAndSig()) {
      result = true;
      return true;
   }
#endif

   if (!msg) {
      ETG_TRACE_ERR(("Can not compute checksum of NULL"));
      return false;
   }

   const EVP_MD *digest = EVP_sha1();
   SWU_ASSERT_RETURN_FALSE(digest);

   if (static_cast<int> (checksum_len) != EVP_MD_size(digest)) {  // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      stringstream s;
      s << checksum_len << " instead of " << EVP_MD_size(digest);
      ETG_TRACE_ERR(("Signature has wrong length: %s", s.str().c_str()));
      return false;
   }

   EVP_MD_CTX *context = EVP_MD_CTX_create();
   SWU_ASSERT_RETURN_FALSE(context);
   ScopedContext scopedContext(context);
   if (1 != EVP_DigestInit_ex(context, digest, NULL)) {
      ETG_TRACE_ERR(("could not initialize context"));
      return false;
   }

   unsigned char digest_buffer[EVP_MD_size(digest)];
   if (1 != EVP_DigestUpdate(context, msg, msg_len)) {
      ETG_TRACE_ERR(("Error updating Message digest with length %d", msg_len));
      return false;
   }

   unsigned int digest_length;
   if (1 != EVP_DigestFinal_ex(context, digest_buffer, &digest_length)) {
      ETG_TRACE_ERR(("Could not finalize Message check"));
      return false;
   }
   if (digest_length != checksum_len) {
      stringstream s;
      s << digest_length << " instead of " << checksum_len;
      ETG_TRACE_ERR(("Got a checksum with wrong size: %s", s.str().c_str()));
      return false;
   }

   result = !memcmp(digest_buffer, checksum, checksum_len);
   return true;
}


} //namespace
