#ifndef SWU_CRYPTO_H_677111280

#define SWU_CRYPTO_H_677111280

#include <ctime>
#include <vector>
#include <string>
#include <openssl/asn1t.h>
#include <openssl/x509.h>

namespace swu {

enum tenParserType {
   ParserTypeBosch = 0u,
   ParserTypeCMS   = 1u
};

/** @brief Defines where certificates and decryption keys shall be searched. */
enum tenCertificateAndKeySource {
   CertificateAndKeySourceUnknown = 0u,
   CertificateAndKeySourceNOR     = 1u,
   CertificateAndKeySourceSDC     = 2u,
   CertificateAndKeySourceFile    = 3u
};

/** @brief Type of digest used for verification of files. */
enum tenDigestType {
   tenDigestTypeInvalid = 0u,
   tenDigestTypeSHA1    = 1u,
   tenDigestTypeMD5     = 2u,
   tenDigestTypeNone    = 3u,
   tenDigestTypeSHA256  = 4u,
};
 

// should be used instead of uint8_t, since openssl works
// with unsigned char everywhere.
typedef unsigned char SWU_BYTE;

/** @brief Convert a time in ASN1 format to Unix time.
 *
 * The ASN1 defines two formats for time: YYmmddHHMMSS or YYYYmmddHHMMSS. These
 * are stored in an ASN1_TIME struct in the libcrypto. This function converts
 * this representation to a Unix time. This will not take into account any
 * timezone information.
 */
time_t ASN1_GetTimeT(const ASN1_TIME* time);

/** Parse a string containing a time description following a given format
 * and return it as a POSIX time. The returned value will always be >= 0.
 * Example:
 * parseTime("05-03-2016, 11:23:54", "%d-%m-%Y, %H:%M:S") returns 1457173380
 *
 * @param time_string The string which contains the time description.
 * @param timeFormat  The format of the time, follows the format of
 *                    @see strptime().
 */
time_t parseTimeString(const std::string& time_string, const std::string& time_format);

/** Write a POSIX time into a string, following the given format.
 *
 * @param time_posix  The POSIX timestamp which should be written as a string.
 * @param time_format The format of the time, follows the format of
 *                    @see strftime().
 */
std::string writeTimeString(const time_t& time_posix, const std::string& time_format);

/** @brief Converts a string in Base64 encoding to a dynamic byte array.
 *
 * A Base64 representation of a binary byte array needs one third more space
 * but uses only the characters "A-Za-z+/"
 */
::std::vector<SWU_BYTE> decodeBase64(const std::string& encStr);

/** @brief Converts a string containing a hexadecimal number to a dynamic byte
 *         array.
 */
::std::vector< SWU_BYTE > decodeHex(const ::std::string& hex);
::std::vector< SWU_BYTE > decodeHex(const char* in);

/** @brief Converts a digest in string representation to a dynamic byte array.
 */
::std::vector< SWU_BYTE > decodeDigest(const ::std::string& enc, const tenDigestType& digestType);

/** @brief Converts a string containing a PEM encoded X509 certificate
 * to a libcrypto X509 structure.
 */
X509 * stringToX509Cert(const std::string &cert);

/** @brief Returns true, if the given string is PEM encoded. It does this
 *         by checking if it begins with ASCII string "-----BEGIN"
 */
bool isPEM(const ::std::vector< SWU_BYTE >& data);

/** @brief Callback for X509_verify (used in CMS_verify).
 *
 * This will make the verification ignore validity times of certificates.
 * Other errors will not be suppressed.
 */
int X509_verify_cb(int ok, X509_STORE_CTX *ctx);
}

#endif // include guard
