#include <fstream>
#include <iostream>
#include "swu_filesystem.h"
#include "swu_configBase.hpp"
#include "swu_configItem.hpp"

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_configItem.cpp.trc.h"
#endif 

using namespace std;

namespace swu {

void ConfigItemBase::traceAction(const char *action) {
   ETG_TRACE_USR4(("ConfigItem(%50s)::%s", 
                   ConfigItemBase::_name.c_str(), 
                   action));
}

void ConfigItemBase::traceVal(string const &val) {
   ETG_TRACE_USR1(("ConfigItem(%50s)::traceVal(string):%s", 
                   ConfigItemBase::_name.c_str(),
                   val.c_str()));
}

void ConfigItemBase::traceVal(tU32 const &val) {
   ETG_TRACE_USR1(("ConfigItem(%50s)::traceVal(tU32):%u", 
                   ConfigItemBase::_name.c_str(),
                   val));
}

void ConfigItemBase::traceVal(vector<tU8> const &val) {
   ETG_TRACE_USR1(("ConfigItem(%50s)::traceVal(vector<tU8>):%02x", 
                   ConfigItemBase::_name.c_str(),
                   ETG_LIST_LEN(static_cast<unsigned int> (val.size())), ETG_LIST_PTR_T8(&val[0])));
}

void ConfigItemBase::traceReadAsBool(bool res) {
   ETG_TRACE_USR1(("ConfigItem(%50s)::readAsBool():_valid=%u res=%s", 
                   ConfigItemBase::_name.c_str(),
                   ConfigItemBase::_valid, res ? "true" : "false"));
}

void ConfigItemBase::traceWrite(bool doSim) {
   ETG_TRACE_USR1(("ConfigItem(%50s)::write():_doSimulate=%u doSim=%u", 
                   ConfigItemBase::_name.c_str(),
                   ConfigItemBase::_doSimulate,
                   doSim));
}

MarkerFileItemDescription::MarkerFileItemDescription(char const *markerFileName,  uid_t user, gid_t group, mode_t perm):
   _markerFileName(markerFileName),
   _user(user),
   _group(group),
   _perm(perm)
{}

bool MarkerFileItemDescription::write(tU32 const &activate) {
   string cmd;
   bool hasMarkerFile=exists(_markerFileName);
   ETG_TRACE_USR4(("MarkerFileItemDescription::write() activate=%u hasMarkerFile=%u _markerFileName=%s",
                   activate,
                   hasMarkerFile,
                   _markerFileName.c_str()));
   if (activate == hasMarkerFile) {
      return true;
   }
   else if (activate)  {
      ofstream dummyFile(_markerFileName.c_str());
      dummyFile << "dummy";
      dummyFile.close();
      (void)chmod(_markerFileName.c_str(), _perm);
      (void)chown(_markerFileName.c_str(), _user, _group);
   }
   else {
      unlink(_markerFileName.c_str());
   }

   return true;
}

bool MarkerFileItemDescription::read(tU32 &activate) {
   activate=exists(_markerFileName);
   ETG_TRACE_USR4(("MarkerFileItemDescription::read() res=%u  _markerFileName=%s",
                   activate,
                   _markerFileName.c_str()));
   return activate;
}




bool FileItemDescriptionHelpers::writeFileItem(string const &fileName, string const &content) {
   string cmd;
   bool hasMarkerFile=exists(fileName);
   bool contentEmpty=content.empty();
   ETG_TRACE_USR4(("StringFileItemDescription::write() content=%60s fileName=%s",
                   content.c_str(),
                   fileName.c_str()));
   if (contentEmpty) {
      if (hasMarkerFile && contentEmpty) {
         unlink(fileName.c_str());
      }
   }
   else  {
      ofstream dummyFile(fileName.c_str());
      dummyFile << content;
      dummyFile.close();
   }
   return true;
}

bool FileItemDescriptionHelpers::readFileItem(string const &fileName, string &content) {
   bool fileExists=exists(fileName);
   ETG_TRACE_USR4(("readFileItem<string>() fileExists=%u  _markerFileName=%s",
                   fileExists,
                   fileName.c_str()));
   content="";
   if (fileExists) {
      ifstream in(fileName.c_str());
      getline(in, content);
      ETG_TRACE_USR4(("readFileItem<string>() content=%s",
                      content.c_str()));
   }
   return true;
}


bool FileItemDescriptionHelpers::writeFileItem(string const &fileName, tU32 const &content) {
   string cmd;
   bool hasMarkerFile=exists(fileName);
   bool contentEmpty=(content==0);
   ETG_TRACE_USR4(("writeFileItem<tU32>() content=%u fileName=%s",
                   content,
                   fileName.c_str()));
   if (contentEmpty) {
      if (hasMarkerFile && contentEmpty) {
         unlink(fileName.c_str());
      }
   }
   else  {
      ofstream dummyFile(fileName.c_str());
      dummyFile << content;
      dummyFile.close();
   }
   return true;
}

bool FileItemDescriptionHelpers::readFileItem(string const &fileName, tU32 &content) {
   bool fileExists=exists(fileName);
   ETG_TRACE_USR4(("readFileItem<tU32>() fileExists=%u  fileName=%s",
                   fileExists,
                   fileName.c_str()));
   content=0;
   if (fileExists) {
      ifstream in(fileName.c_str());
      string str;
      getline(in, str);
      content=static_cast<tU32> (strtol(str.c_str(), 0, 0) );
      ETG_TRACE_USR4(("readFileItem<tU32>() content=%u",
                      content));
   }
   return true;
}


bool FileItemDescriptionHelpers::writeFileItem(string const &fileName, std::vector<tU8> const &content) {
   string cmd;
   bool hasMarkerFile=exists(fileName);
   std::string contentString=binToHex(content);
   ETG_TRACE_USR4(("writeFileItem<std::vector<tU8> >(%80s) content=%02x",
                   fileName.c_str(), ETG_LIST_LEN(static_cast<unsigned int> (content.size())), ETG_LIST_PTR_T8(&content[0])));
   if (contentString.empty()) {
      if (hasMarkerFile) {
         unlink(fileName.c_str());
      }
   }
   else  {
      ofstream dummyFile(contentString.c_str());
      dummyFile << contentString.c_str();
      dummyFile.close();
   }
   return true;
}


bool FileItemDescriptionHelpers::readFileItem(string const &fileName, std::vector<tU8> &content) {
   content.clear();
   bool fileExists=exists(fileName);
   ETG_TRACE_USR4(("readFileItem<std::vector<tU8>>() fileExists=%u  fileName=%80s content=%02x",
                   fileExists,
                   fileName.c_str(),
                   ETG_LIST_LEN(static_cast<unsigned int> (content.size())), ETG_LIST_PTR_T8(&content[0])));
   if (fileExists) {
      ifstream in(fileName.c_str());
      string str;
      getline(in, str);
      hexToBin(str, content);
      ETG_TRACE_USR4(("readFileItem<vector<tU8> >() content=%s",
                      str.c_str()));
   }
   return true;
}


bool ConfigItemBase::read(bool force) {
   traceAction("read()");
      
   bool doRead=
      hasItemDescription() && // we need an item-decription to read
      !(_doSimulate &&  // don't read in simulation
        (!ConfigItemBase::_valid || force || getVolatile())); // only read if invalid or we are forced
      
   if (doRead) {
      _valid=readVal();
   }
   else {
      _valid=true;
   }

   if (! _valid) {
      setDefault();
   }
   return _valid;

}

bool ConfigItemBase::write() {
   traceWrite(_doSimulate);
      
   if (_doSimulate || !hasItemDescription()) {
      _valid=true;
      return true;
   }
   if (hasItemDescription()) {
      _valid=writeVal();
   }
   return  _valid;
}

ConfigItemSliceDescription::ConfigItemSliceDescription(ConfigItem<vector<tU8> > *parent, tU32 offset, tU32 numBytes, tU32 mask):
   _parent(parent),
   _offset(offset),
   _numBytes(numBytes),
   _mask(mask)
{
   if (_numBytes > 4) {
      _numBytes=4;
   }
}

bool ConfigItemSliceDescription::write(tU32 const &val) {
   vector<tU8> parentVal;
   ETG_TRACE_ERR(("ConfigItemSliceDescription()::write(%u) START", val));
   bool success= _parent->get(parentVal);
   if (!success) {
      ETG_TRACE_ERR(("ConfigItemSliceDescription()::write() _parent->get() failed"));
      //      return false;
   }
   if (_numBytes + _offset  > parentVal.size()) {
      ETG_TRACE_ERR(("ConfigItemSliceDescription()::write Invalid size: _numBytes=%u _offset=%u  > parentSize=%u"
    		  	  	  , _numBytes , _offset, parentVal.size()));
      return false;
   }
   tU32 u32TmpVal=val;
   ETG_TRACE_ERR(("ConfigItemSliceDescription()::write _numBytes=%u _offset=%u parentSize=%u val=%u valHex=0x%08x, mask=0x%08x"
		   	   	   ,_numBytes , _offset, parentVal.size(), val, val, _mask));
   ETG_TRACE_ERR(("ConfigItemSliceDescription()::write(%08x) START", u32TmpVal));
   if (_mask) {
      tU32 mask=_mask;
      while (!(mask & 1)) {
         mask>>=1;
         u32TmpVal<<=1;
      }
      u32TmpVal &=_mask;
   }

   tU32 u32ParentVal=0;
   for (tU32 i=0; i<_numBytes;++i) {
      u32ParentVal<<8;
      u32ParentVal |= parentVal[_offset + i];
   }
   ETG_TRACE_ERR(("ConfigItemSliceDescription()::write() Step 2 u32TmpVal=%08x u32ParentVal=0x%08x ", u32TmpVal, u32ParentVal));
   if (_mask) {
      u32ParentVal&=(~_mask);
      u32ParentVal |= u32TmpVal;
   } else{
	   u32ParentVal = u32TmpVal;
   }
   ETG_TRACE_ERR(("ConfigItemSliceDescription()::write() Step 3 u32TmpVal=%08x u32ParentVal=0x%08x ", u32TmpVal, u32ParentVal));
   for (unsigned int i=0; i<_numBytes; ++i) {
      parentVal[_offset + (_numBytes -1) -i]=((u32ParentVal>>(i*8)) & 0xFF);
   }
   bool res=_parent->set(parentVal);
   ETG_TRACE_USR4(("ConfigItemSliceDescription()::write res=%u", res));
   return _parent->set(parentVal);

}


bool ConfigItemSliceDescription::read(tU32 &val) {
   vector<tU8> &parentVal= _parent->get();
   ETG_TRACE_USR4(("ConfigItemSliceDescription()::read Start"));

   
   if (_numBytes + _offset  > parentVal.size()) {
      ETG_TRACE_ERR(("ConfigItemSliceDescription()::read Invalid size: _numBytes=%u _offset=u  > parentSize=%u",
                     _numBytes + _offset, parentVal.size()));
      return false;
   }
   val=0;
   for (tU32 i=0; i<_numBytes;++i) {
      val<<=8;
      val+=parentVal[_offset + i];
   }

   ETG_TRACE_USR4(("ConfigItemSliceDescription()::raw val =%u (0x%08x) mask=0x%08x", val, val, _mask));

   if (_mask) {
      val&=_mask;
      tU32 mask=_mask;
      while (!(mask & 1)) {
         mask>>=1;
         val>>=1;
      }
   }
   ETG_TRACE_USR4(("ConfigItemSliceDescription()::final val =%u (0x%08x)", val, val));

   return true;
}

ConfigItemVectorSliceDescription::ConfigItemVectorSliceDescription(ConfigItem<vector<tU8> > *parent, tU32 offset, tU32 numBytes):
   _parent(parent),
   _offset(offset),
   _numBytes(numBytes)
{}


bool ConfigItemVectorSliceDescription::write(vector<tU8> const &val) {
   vector<tU8> parentVal;
   bool success= _parent->get(parentVal);
   if (!success) {
      ETG_TRACE_ERR(("ConfigItemVectorSliceDescription()::write _parent->get() failed"));
      //      return false;
   }
   if (_numBytes + _offset  > parentVal.size()) {
      ETG_TRACE_ERR(("ConfigItemVectorSliceDescription()::write Invalid size: _numBytes=%u _offset=u  > parentSize=%u",
                     _numBytes + _offset, parentVal.size()));
      return false;
   }
   for (tU32 i=0; i<_numBytes;++i) {
      tU8 u8Val=0;
      if (i < val.size()) {
         u8Val=val[i];
      }
      parentVal[_offset + i]=u8Val;
   }
   bool res=_parent->set(parentVal);
   ETG_TRACE_USR4(("ConfigItemVectorSliceDescription()::write res=%u", res));
   return res;
}

bool ConfigItemVectorSliceDescription::read(vector<tU8> &val) {
   vector<tU8> &parentVal= _parent->get();
   
   if (_numBytes + _offset  > parentVal.size()) {
      return false;
   }
   val.resize(_numBytes);
   for (tU32 i=0; i<_numBytes;++i) {
      val[i]=parentVal[_offset + i];
   }
   return true;
}

void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<tU32> *item) {
   cfg->addItem(item);
         
}
void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<std::string> *item) {
   cfg->addItem(item);
         
}
void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<vector<tU8> > *item) {
   cfg->addItem(item);
         
}

void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<tU32> *item) {
   cfg->onItemChanged(item);
         
}
void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<std::string> *item) {
   cfg->onItemChanged(item);
         
}
void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<vector<tU8> > *item) {
   cfg->onItemChanged(item);
         
}

      
}


