
#include <string>
#include "util/swu_configBase.hpp"

#include "util/swu_registry.h"
#include "util/swu_dataStore.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_environment.hpp"
#include "util/swu_kds.h"
#include "util/swu_dataStore.hpp"
// todo: let param refer to another config-item, e.g. "EnterFullOperation"

namespace swu {

class ItemBaseData {
private:
   std::string _name;
   std::string _itemType;
   std::string _valType;
   TiXmlElement const *_root;
   DataStore *_dataStore;
   bool _valid;
   template<class CONFIG, class VALTYPE>
   friend class ItemInfo;

public:

   ItemBaseData(TiXmlElement const *root, swu::DataStore *dataStore);


   bool isValid() {
      return _valid;
   }

   std::string &getValType() {
      SWU_TRACE_VARG_USR4("ItemBaseData:getValType() name=%s valType=%s", _name.c_str(), _valType.c_str());
      return _valType;
   }
   std::string &getName() {
      return _name;
   }

   void invalidate() {
      _valid=false;
   }

private: // for friends ...

   TiXmlElement const *getRoot() {
      return _root;
   }

   bool strToVal(std::string const &str, std::string &res);
   bool strToVal(std::string const &str, tU32 &res);
   bool strToVal(std::string const &str, std::vector<tU8> &res);

   bool appendToVal(std::string &val, std::string appendVal);
   bool appendToVal(tU32 &val, tU32 appendVal);
   bool appendToVal(std::vector<tU8> &val, std::vector<tU8> appendVal);


   DataStore *getDataStore() {
      return _dataStore;
   }

   std::string readParam(TiXmlElement const *params, std::string paramName);
};


template<class CONFIG, class VALTYPE>
class ItemInfo {
private:
   CONFIG *_config;
   swu::ConfigItem<VALTYPE> *_item;
   ItemBaseData *_base;
   TiXmlElement const *_params;


public:
   
   // only ctor
   ItemInfo(ItemBaseData *base, CONFIG *config):
      _base(base), 
      _config(config),
      _item(0),
      _params(0) {
      SWU_TRACE_VARG_USR4("ItemInfo::CTOR(%s) START", _base->getName().c_str());

      SWU_ASSERT_RETURN(_base);
      SWU_ASSERT_RETURN(_base->getRoot());
      SWU_ASSERT_RETURN(_config);
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::CTOR(%s) base is invalid", _base->getName().c_str());
         return;
      }
      _item=_config->getItem((VALTYPE*) 0, _base->getName());
      if (!_item)  {
         SWU_TRACE_VARG_ERR("ItemInfo::CTOR(%s) item not found", _base->getName().c_str());
         invalidate();
         return;
      }

      TiXmlElement const *description=_base->getRoot()->FirstChildElement("DESCRIPTION");
      if (description) {
         SWU_TRACE_VARG_USR4("ItemInfo::CTOR(%s) got description", _base->getName().c_str());
         _params=description->FirstChildElement("PARAMS");
         if (!_params) {
            SWU_TRACE_VARG_ERR("ItemInfo::CTOR(%s) PARAMS MISSING", _base->getName().c_str());
            invalidate();
         }
      }
      
   };
   

   // check if everything is OK
   bool isValid() {
      if (!_base) {
         return false;
      }
      if (!_base->isValid()) {
         return false;
      }
      return true;
   }



   // to be called when something went wrong
   void invalidate() {
      _base->invalidate();
   }

   
   // apply config-changes according to xml-input
   bool apply() {
      bool res=true;
      setVal();
      setDefaultVal();
      setDescription();
      return isValid();
   }

private:
   
   char const *getBaseName() {
      return _base->getName().c_str();
   }
   // set the value of the config-item of CONFIG
   void setVal() {
      if (!isValid()) {
         return;
      }
      if (!_base->getRoot()->FirstChildElement("VAL")) {
         // ok, val not VAL not configured
         return;
      }

      std::string strVal=swu::getTextFromChildOrEmpty(_base->getRoot(), "VAL");

      std::string marker="cfgItem:";
      VALTYPE val;
      if (!strVal.compare(0, marker.size(), marker.c_str())) {
         std::string cfgItemName=strVal;
         cfgItemName.erase(0, marker.size());
         SWU_TRACE_VARG_USR4("setVal() try cfgItemName:%s", cfgItemName.c_str());
         ConfigItem<VALTYPE> *configItem=_config->getItem((VALTYPE *)0, cfgItemName);
         if (configItem) {
            val = configItem->get();
         } else {
            SWU_TRACE_VARG_ERR("setVal() config-item not found:%s", cfgItemName.c_str());
            invalidate();
            return;
         }
      } else {
         if (!_base->strToVal(strVal, val)) {
            // error, object invalidated by base
            invalidate();
            return;
         }
      }
      
      strVal =  swu::getTextFromChildOrEmpty(_base->getRoot(), "APPEND");
      if(strVal.size() > 0) {
         VALTYPE strAppendVal;
         if(readConfigItem(strVal, strAppendVal)) {
            _base->appendToVal(val, strAppendVal);
         }
      } 

      _item->set(val);
   }

   // set the value of the config-item of CONFIG
   void setDefaultVal() {
      if (!isValid()) {
         return;
      }
      if (!_base->getRoot()->FirstChildElement("DEFAULT_VAL")) {
         // ok, DEFAULT_VAL not configured
         return;
      }
      std::string strVal=swu::getTextFromChildOrEmpty(_base->getRoot(), "DEFAULT_VAL");
      VALTYPE val;
      
      if (!_base->strToVal(strVal, val)) {
         // error, object invalidated by base
         return;
      }
      _item->setDefaultVal(val);
   }



   template<class PARAMTYPE>
   PARAMTYPE paramNotZero(PARAMTYPE param) {
      if (!param) {
         invalidate();
      }
      return param;
   }


   /* if the value of a parameter starts with "cfgItem:"
      value shall be read from config.
      return code indicates, that parmeter links to a config-item.
      failure in looking up the config-item will lead to invalidate()

    */
   template<class CFGITEM_VALTYPE>
   bool readConfigItem(std::string strVal, CFGITEM_VALTYPE &val) {
      SWU_TRACE_VARG_USR4("readConfigItem() strVal:%s", strVal.c_str());
      std::string marker="cfgItem:";
      if (!strVal.compare(0, marker.size(), marker.c_str())) {
         std::string cfgItemName=strVal;
         cfgItemName.erase(0, marker.size());
         SWU_TRACE_VARG_USR4("readConfigItem() try cfgItemName:%s", cfgItemName.c_str());
         ConfigItem<CFGITEM_VALTYPE> *configItem=_config->getItem((CFGITEM_VALTYPE *)0, cfgItemName);
         if (configItem) {
            val = configItem->get();
            return true;
         } else {
            SWU_TRACE_VARG_ERR("readConfigItem() config-item not found:%s", cfgItemName.c_str());
            invalidate();
            return false;
         }
      }
      return false;
   } 
   
   std::string paramAsString(std::string paramName) {
      std::string res;
      std::string strVal= _base->readParam(_params, paramName);
      if (readConfigItem(strVal, res )) {
         SWU_TRACE_VARG_USR4("paramAsString(%s) readConfigItem:%s", 
                             paramName.c_str(), res.c_str());
         return res;
      }
      SWU_TRACE_VARG_USR4("paramAsString(%s): %s", 
                          paramName.c_str(), strVal.c_str());
      return strVal;
   }

   tU32 paramAsU32(std::string paramName) {
      tU32 res;
      std::string strVal= _base->readParam(_params, paramName);
      if (readConfigItem(strVal, res )) {
         SWU_TRACE_VARG_USR4("paramAsU32(%s) readConfigItem:%u", 
                             paramName.c_str(), res);
         return res;
      }
      _base->strToVal(strVal, res);
      SWU_TRACE_VARG_USR4("paramAsU32(%s) readConfigItem:%u", 
                          paramName.c_str(), res);
      return res;
   }

   std::vector<tU8> paramAsVector(std::string paramName) {
      std::vector<tU8> res;
      std::string strVal= _base->readParam(_params, paramName);
      SWU_TRACE_VARG_USR4("paramAsVector(%s) strVal=%s", 
                          paramName.c_str(), strVal);
      if (readConfigItem(strVal, res )) {
         SWU_TRACE_VARG_USR4("paramAsVector() readConfigItem");
         return res;
      }
      _base->strToVal(strVal, res);
      return res;
   }

   bool setDescription(std::string *valType, std::string &descriptionType) {
      SWU_TRACE_VARG_USR4("ItemInfo::setDescription(string, %s)"
                          "unsupported descriptionType:%s", 
                          getBaseName(), descriptionType.c_str());

      return false;
   }


   // set descriptions specific for val-type U32
   bool setDescription(tU32 *valType, std::string &descriptionType) {
      SWU_TRACE_VARG_USR4("ItemInfo::setDescription(U32, %s) descriptionType:%s", 
                          getBaseName(), descriptionType.c_str());
      if (descriptionType=="MarkerFileItemDescription") {
         setDescription<MarkerFileItemDescription>(paramAsString("FILE_NAME").c_str());
      } 
      else {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(tU32,%s) NOT OK descriptionType=%s", 
                            getBaseName(), descriptionType.c_str());
         return false;
      }
      return true;
   }


   bool setDescription(std::vector<tU8> *valType, std::string &descriptionType) {
      SWU_TRACE_VARG_USR4("ItemInfo::setDescription(vector<tU8>, %s) descriptionType:%s", 
                          getBaseName(), descriptionType.c_str());
      return false;
   }


   // set the itemDescription of config-item of CONFIG
   void setDescription() {
      SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) START", getBaseName());

      if (!isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) not valid", getBaseName());
         return;
      }
      
      TiXmlElement const *description=_base->getRoot()->FirstChildElement("DESCRIPTION");
      if (!description) {
         SWU_TRACE_VARG_USR4("ItemInfo::setDescription(%s) no description", getBaseName());
         return;
      }

      TiXmlElement const *_params=description->FirstChildElement("PARAMS");
      if (!_params) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) no params", getBaseName());
         invalidate();
         return;
      }

      std::string valType=_base->getValType();
      std::string descriptionType=swu::getTextFromChild(description, "TYPE", true, "" );  //Coverity fix for 69975
      if (descriptionType=="NoItemDescription") {
         _item->removeDescription();
      }
      else if (descriptionType=="KdsItemDescription") {
         // using alway param-fns to call setDescription<XXX>.
         // so description will only be created if all parameters are OK

         // Below compiler warnings are reported for DESCR *desc=new DESCR(p1, p2, p3, p4);
         // gen3armmake, gen3x86make, gen4rcar, gen4lsim: conversion to 'tU16 {aka short unsigned int}' from 'unsigned int'
         // the data retieved is tU32 and stored into tU16
         // so the conversion is made explicit
         setDescription<KdsItemDescription<VALTYPE> >(
                                                      static_cast<uint_least16_t> (paramAsU32("TABLE")),
                                                      static_cast<uint_least16_t> (paramAsU32("TABLE_LEN")),
                                                      static_cast<uint_least16_t> (paramAsU32("OFFSET")),
                                                      static_cast<uint_least16_t> (paramAsU32("LEN"))
                                                      );
      }
      else if (descriptionType=="DataStoreItemDescription") {
         setDescription<DataStoreItemDescription<VALTYPE> >(
                                                            paramNotZero(_base->getDataStore()),
                                                            paramAsString("KEY")
                                                            );
      }
      else if (descriptionType=="RegistryItemDescription") {
         setDescription<RegistryItemDescription<VALTYPE> >(
                                                      paramAsString("PATH"),
                                                      paramAsString("KEY")
                                                      );
      }
      else if (descriptionType=="FileItemDescription") {
         setDescription<FileItemDescription<VALTYPE> >(paramAsString("FILE_NAME").c_str());
      } 

      else {
         // try description that is specific for a certain valtype
         SWU_TRACE_VARG_USR4(
                            "ItemInfo::setDescription(%s) try type specific descriptions", 
                            getBaseName());
         if (!setDescription((VALTYPE *)0, descriptionType)) {
            SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) unsupported descriptionType %s", 
                               getBaseName(), descriptionType.c_str());
         }
      }
      
   }


   // template-fns to create item-descriptions with different numbers of parameters
   template<typename DESCR, class PARAM1> 
   void setDescription( PARAM1 p1) {
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) invalid",getBaseName());
         return;
      }
      DESCR *desc;
      desc=new DESCR(p1);
      _item->setDescription(desc);
   }

   template<class DESCR, class PARAM1, class PARAM2> 
   void setDescription( PARAM1 p1, PARAM2 p2) {
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) invalid", getBaseName());
         return;
      }
      DESCR *desc=new DESCR(p1, p2);
      _item->setDescription(desc);
   }

   template<class DESCR, class PARAM1, class PARAM2, class PARAM3> 
   void setDescription( PARAM1 p1, PARAM2 p2, PARAM3 p3) {
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) invalid", getBaseName());
         return;
      }
      DESCR *desc=new DESCR(p1, p2, p3);
      _item->setDescription(desc);
   }

   template<class DESCR, class PARAM1, class PARAM2, class PARAM3, class PARAM4>
   void setDescription( PARAM1 p1, PARAM2 p2, PARAM3 p3, PARAM4 p4) {
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) invalid", getBaseName() );
         return;
      }
      DESCR *desc=new DESCR(p1, p2, p3, p4);
      _item->setDescription(desc);
   }

   template<class DESCR, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5>
   void setDescription( PARAM1 p1, PARAM2 p2, PARAM3 p3, PARAM4 p4, PARAM5 p5) {
      if (!_base->isValid()) {
         SWU_TRACE_VARG_ERR("ItemInfo::setDescription(%s) invalid", getBaseName() );
         return;
      }
      DESCR *desc=new DESCR(p1, p2, p3, p4, p5);
      _item->setDescription(desc);
   }

};



template<class CONFIG>
static bool setConfigXml(std::string xmlPath, CONFIG *config, DataStore *dataStore=0) {
   SWU_TRACE_VARG_USR1("setConfigXml::setConfigXml():xmlPath:%s", xmlPath.c_str());
   if (!swu::isFile(xmlPath)) {
      SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():could not find file:%s", xmlPath.c_str());
      return false;
   }
   TiXmlDocument xmlDoc(xmlPath.c_str());
   if (!xmlDoc.LoadFile()) {
      SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():could not load %s", xmlPath.c_str());
      return false;
   }

   TiXmlElement *root=xmlDoc.FirstChildElement("CONFIG_XML");
   if (!root) {
      SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():could not find root element CONFIG_XML in xmlDoc %s", xmlPath.c_str());
      return false;
   }
      
   TiXmlElement *items=root->FirstChildElement("ITEMS");
   if (!items) {
      SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():could not find section in ITEMS:", xmlPath.c_str());
      return false;
   }
      
   tU32 numItems=0;
   for (TiXmlElement *item=items->FirstChildElement("ITEM");
        item;
        item=item->NextSiblingElement("ITEM")) {
      ++numItems;
      SWU_TRACE_VARG_USR4("ConfigFactory::setConfigXml():START ITEM %u", numItems);

      ItemBaseData itemBaseData(item, dataStore);
      SWU_TRACE_VARG_USR4("ConfigFactory::setConfigXml():ItemBaseData CTOR DONE");

      if (!itemBaseData.isValid()) {
         SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():invalid ITEM:%u", numItems);
         continue;
      }
      SWU_TRACE_VARG_USR4("ConfigFactory::setConfigXml():ITEM name=%s", itemBaseData.getName().c_str());
      std::string valType=itemBaseData.getValType();
      SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():got valtype:%u", numItems);

      if (valType=="U32") {
         ItemInfo<CONFIG, tU32> itemInfo(&itemBaseData, config);
         itemInfo.apply();
      }
      else if (valType=="String") {
         ItemInfo<CONFIG, std::string> itemInfo(&itemBaseData, config);
         itemInfo.apply();
      }
      else if (valType=="Vector") {
         ItemInfo<CONFIG, std::vector<tU8> > itemInfo(&itemBaseData, config);
         itemInfo.apply();
      }
      else {
         SWU_TRACE_VARG_ERR("ConfigFactory::setConfigXml():item invalid valType:%s", valType.c_str());
      }
         
       
   }
}


}


