#ifndef SWU_CMS_H
#define SWU_CMS_H

#include <string>
#include "util/swu_crypto.hpp"
#include "util/swu_caCert.h"
#include "util/swu_constants.hpp"
#include "util/swu_uin.hpp"
#include "tinyxml/tinyxml.h"

namespace swu {

class CMS {
public:

   /** @brief C'tor with CA certificate.
    *  Sets the root certificate, which is internally used to verify the parsed
    *  CMS, see CMS::validate().
    *
    *  @param caCert  A CA root certificate to validate() against.
    */
   CMS (CACertIf* caCert, ::time_t lastAuthenticTime=0);

   /** @brief Parse the given file.
    *
    * Verify that it is a CMS file and validate()
    * it against the CA certificate. Finally copy the payload 
    * to CMS::_message, @see CMS::message(). 
    *
    * @param file           Signed data string in CMS syntax.
    * @param allowEmptyCert Select if an empty certificate is accepted.
    * @return               True if validation succeeded, false otherwise.
    * */
   bool parse(const ::std::string &file, bool allowEmptyCert);

   /** @brief Verify CMS data against the root certificate and return the payload.
    *
    * @param outStr    Payload of the CMS is returned in this.
    * @param inData    Signed data string in CMS syntax.
    * @param allowEmptyCert  Select if an empty certificate is accepted.
    * @return          True if validation succeeded, false otherwise.
    */
   bool validate(::std::string& outStr, const ::std::string& msgStr, bool allowEmptyCert);

   /** @brief Checks if the string begins with Constants::Cert::MIME_STRING(). */
   bool isMIME(const ::std::string& msgStr) const;

   /** @brief Extract filename and hash value from description in message. */
   bool extractXmlNameAndDigest(::std::string &name, ::std::string &digest, swu::tenDigestType& type);

   /** @brief Extract encryption parameters from description in message.
    * @param encryptParam  Output variable. Encryption parameters will be written to this.
    * @param uin           The target's Unique Identification Number. In case of target-specific
    *                      Updates, this is used to find the correct parameter set.
    * @param allowEmptyCert  Select if an empty certificate is accepted.
    * @return              True if parameters could be extracted, false otherwise.
    * */
   bool extractEncryptParam(::std::string& encryptParam, const ::std::string& uin, bool allowEmptyCert);

   /** Looks, if the XML message contains <meta> and <licensefile> tags. If they
    * exist, this update is considered encrypted. If they
    * are not found in the XML, the update is considered unencrypted.
    *
    * @return  True if the update sources are encrypted, otherwise false.
    */
   bool isUpdateEncrypted ();

   /** @brief Return the message. This should be called after a call to
    * CMS::parse(), otherwise it will be empty. */
   ::std::string message () const;

   void setLastAuthenticTime(const ::time_t& lastAuthenticTime);
   ::time_t getLastAuthenticTime() const;

private:
   ::std::string     _path;
   ::std::string     _message;
   CACertIf*         _caCert;

   // time used for the verification of CMS
   ::time_t          _lastAuthenticTime;

   TiXmlDocument _xmlDoc;
};

}
#endif
