#include <cstdlib>
#include <cstring>
#include <fstream>

#include <sys/stat.h>
#include <unistd.h>

#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <openssl/err.h>

#include "util/swu_types.h"
#include "util/swu_filesystem.h"
#include "util/swu_certificate.h"
#include "util/swu_caCert.h"
#include "util/swu_trace.h"
#include "util/swu_crypto.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_certificate.cpp.trc.h"
#endif

#define ONE_DAY 86400

using namespace std;

namespace swu {

   bool CodeCert::validateCertificate(const string &caCertStr, const string &codeCertStr) const
   {
      bool result = true;

      X509* codeCert           = 0;
      X509* caCert             = 0;

      do { // this loop emulates "goto cleanup;" in error handling
         codeCert = stringToX509Cert(codeCertStr);
         if ( ! codeCert ) {
            ETG_TRACE_ERR(("Can't read code certificate."));
            result = false; break;
         }

         caCert = stringToX509Cert(caCertStr);
         if ( ! caCert ) {
            ETG_TRACE_ERR(("Can't read CA certificate."));
            result = false; break;
         }
         
         result = validateCertificate(caCert, codeCert);
      } while ( false );


      if(caCert)        X509_free(caCert);
      if(codeCert)      X509_free(codeCert);
   
      return result;
   }

   bool CodeCert::validateCertificate(X509* caCert, X509* codeCert) const
   {
      bool result = true;

      X509_STORE* trusted_certs = 0;
      X509_STORE_CTX* storeCtx  = 0;
      BIO* diagnosisBio         = 0;
      X509_NAME* certSubject    = 0;

      do { // this loop emulates "goto cleanup;" in error handling
         
         trusted_certs = X509_STORE_new();
         if (0 == trusted_certs ) {
            ETG_TRACE_ERR(("Can't initialize trusted_certs."));
            result = false; break;
         }
         X509_STORE_add_cert(trusted_certs, caCert);

         storeCtx = X509_STORE_CTX_new();
         if (0 == storeCtx) {
            ETG_TRACE_ERR(("Can't create context for trusted_certs"));
            result = false; break;
         }
         if (0 == X509_STORE_CTX_init(storeCtx, trusted_certs, codeCert, NULL) ) {
            ETG_TRACE_ERR(("Can't initialize context for trusted_certs"));
            result = false; break;
         }

         // Set the current time to the codeCerts "Not Before" time + one day.
         // This should also be smaller than "Not After" time.
         // This means that expiration of the codeCert will not be considered in 
         // the verification.
         time_t cur_time = swu::ASN1_GetTimeT(X509_get_notBefore(codeCert)) + ONE_DAY - 1;
         X509_STORE_CTX_set_time(storeCtx, 0, cur_time);
         X509_STORE_CTX_set_flags(storeCtx, X509_V_FLAG_USE_CHECK_TIME | X509_V_FLAG_CHECK_SS_SIGNATURE);

         if ( 0 == X509_verify_cert(storeCtx) ) {
            result = false;

            // some diagnosis before returning
            ERR_load_BIO_strings();
            ERR_load_crypto_strings();

            ETG_TRACE_ERR(("Can't verify certificate. Reason: %s", X509_verify_cert_error_string(storeCtx->error)));
            //  get the offending certificate causing the failure
            diagnosisBio = BIO_new(BIO_s_mem());
            if ( ! diagnosisBio) {
               ETG_TRACE_ERR(("Can't create BIO for diagnosis"));
               break;
            }
            certSubject = X509_get_subject_name(codeCert);
            X509_NAME_print_ex(diagnosisBio, certSubject, 0, XN_FLAG_MULTILINE);
            char *ptr = 0;
            BIO_get_mem_data(diagnosisBio, &ptr);
            ETG_TRACE_ERR(("%s", ptr));
         }
      } while ( false );

      // cleanup:
      if(certSubject)   X509_NAME_free(certSubject);
      if(diagnosisBio)  BIO_free(diagnosisBio);
      if(storeCtx)      X509_STORE_CTX_free(storeCtx);
      if(trusted_certs) X509_STORE_free(trusted_certs);

      return result;
   }

   bool CodeCert::extractPublicKeyFromCertificate(const string &certStr, string &keyStr) const
   {
      bool result = true;

      X509* cert     = 0;
      EVP_PKEY* pkey = 0;
      BIO* keyBio    = 0;

      do { // this loop emulates "goto cleanup;" in error handling
         cert = stringToX509Cert(certStr);
         if ( ! cert ) {
            ETG_TRACE_ERR(("Can't read certificate"));
            result = false;
            break;
         }
         pkey = X509_get_pubkey(cert);
         if ( ! pkey) {
            ETG_TRACE_ERR(("Can't extract EVP_PKEY from certificate"));
            result = false;
            break;
         }
         switch (pkey->type) {
            case EVP_PKEY_RSA:
               ETG_TRACE_USR3(("%d bit RSA Key\n", EVP_PKEY_bits(pkey) ));
               break;
            case EVP_PKEY_DSA:
               ETG_TRACE_USR3(("%d bit DSA Key\n", EVP_PKEY_bits(pkey) ));
               break;
            default:
               ETG_TRACE_USR3(("%d bit unknown Key\n", EVP_PKEY_bits(pkey) ));
               break;
         }
         keyBio = BIO_new(BIO_s_mem());
         if ( ! keyBio ) {
            ETG_TRACE_ERR(("Can't create BIO for key"));
            result = false;
            break;
         }
         if ( ! PEM_write_bio_PUBKEY(keyBio, pkey) ) {
            ETG_TRACE_ERR(("Can't read EVP_PKEY"));
            result = false;
            break;
         }
         char *ptr = 0;
         // set ptr to the memory section of keyBio
         long length = BIO_get_mem_data(keyBio, &ptr);
         if ((uint32_t)length > Constants::Mmc::MTD_CERTIFICATE_BUFFER_SIZE) {
            ETG_TRACE_ERR(("EVP_PKEY is too long"));
            result = false; break;
         }
         keyStr.assign(ptr, (size_t)length);
      } while ( false );

      // cleanup:
      if(keyBio)  BIO_free(keyBio);
      if(pkey)    EVP_PKEY_free(pkey);
      if(cert)    X509_free(cert);

      return result;
   }

   bool CodeCert::checkSignature(const std::string &msgStr, const std::string &sigStr,
         const std::string &keyStr) const
   {
      // To understand how OpenSSL verify works have a look at following:
      // http://wiki.openssl.org/index.php/EVP_Signing_and_Verifying
      bool result           = true;

      BIO* keyBio           = 0;
      EVP_PKEY* evpPubKey   = 0;
      RSA* pkeyRsa          = 0;
      EVP_MD_CTX* digestCtx = 0;

      do { // this loop emulates "goto cleanup;" in error handling
         keyBio = BIO_new(BIO_s_mem());
         if (! keyBio) {
            ETG_TRACE_ERR(("Could not build keyBio from key."));
            result = false; break;
         }
         BIO_puts(keyBio, keyStr.c_str());

         pkeyRsa = PEM_read_bio_RSA_PUBKEY(keyBio, NULL, NULL, NULL);
         if (!pkeyRsa) {
            ETG_TRACE_ERR(("Could not read public key"));
            result = false; break;
         }
         evpPubKey = EVP_PKEY_new();
         if ( ! evpPubKey ) {
            ETG_TRACE_ERR(("Could not create EVP key object"));
            result = false; break;
         }

         if (! EVP_PKEY_set1_RSA(evpPubKey, pkeyRsa) ) {
            ETG_TRACE_ERR(("Could not set RSA public key as envelope key"));
            result = false; break;
         }

         digestCtx = EVP_MD_CTX_create();
         if (! digestCtx ) {
            ETG_TRACE_ERR(("Could not create EVP message digest context."));
            result = false; break;
         }

         if ( ! EVP_DigestVerifyInit(digestCtx, NULL, EVP_sha256(), NULL, evpPubKey) ) {
            ETG_TRACE_ERR(("Could not initialize verification context"));
            result = false; break;
         }
         if ( ! EVP_DigestVerifyUpdate(digestCtx, msgStr.c_str(), msgStr.size())) {
            ETG_TRACE_ERR(("Could not update verification context"));
            result = false; break;
         }
         size_t sigLen = sigStr.length();
         SWU_BYTE sigCStr[sigLen];
         memcpy((char*)sigCStr, sigStr.c_str(), sigLen);
         if ( ! EVP_DigestVerifyFinal(digestCtx, sigCStr, sigLen) ) {
            ETG_TRACE_ERR(("Could not verify the message."));
            result = false;
         }
      } while ( false );

      // cleanup:
      if (digestCtx)  EVP_MD_CTX_destroy(digestCtx);
      if (pkeyRsa)    RSA_free(pkeyRsa);
      if (evpPubKey)  EVP_PKEY_free(evpPubKey);
      if (keyBio)     BIO_free(keyBio);

      return result;
   }


// -------------------------------------------------------------- TestSignature

   void TestSignature::TestSignature::vInit() const
   {
   }

      void TestSignature::testCheckSignature(const char *messageFile, const char *signatureFile, const char *keyFile) const
      {
         ETG_TRACE_USR4(("swu_filesystem testCheckSignature message file %s", messageFile));
         ETG_TRACE_USR4(("swu_filesystem testCheckSignature signature binary file %s", signatureFile));
         ETG_TRACE_USR4(("swu_filesystem testCheckSignature key file %s", keyFile));
         string message;
         string signature;
         string key;
         if (!loadFile(messageFile, message)) {
            ETG_TRACE_ERR(("Could not load message file %s", messageFile));
            return;
         }
         if (!loadFile(signatureFile, signature)) {
            ETG_TRACE_ERR(("Could not load signature file %s", signatureFile));
            return;
         }
         if (!loadFile(keyFile, key)) {
            ETG_TRACE_ERR(("Could not load key file %s", keyFile));
            return;
         }
         // if (!CodeCert::instance()->checkSignature(message, signature, key)) {
         //    ETG_TRACE_COMP(("swu_filesystem testCheckSignature failed"));
         //    return;
         // }
         ETG_TRACE_COMP(("swu_filesystem testCheckSignature passed"));
      }

      void TestSignature::testCheckSignatureWithSystemkey(const char *messageFile, const char *signatureFile) const
      {
         ETG_TRACE_USR4(("swu_filesystem testCheckSignature message file %s", messageFile));
         ETG_TRACE_USR4(("swu_filesystem testCheckSignature signature binary file %s", signatureFile));
         string message;
         string signature;
         if (!loadFile(messageFile, message)) {
            ETG_TRACE_ERR(("Could not load message file %s", messageFile));
            return;
         }
         if (!loadFile(signatureFile, signature)) {
            ETG_TRACE_ERR(("Could not load signature file %s", signatureFile));
            return;
         }
         // std::string certificate;
         // bool isCertEmpty;
         swu::CACertIf * caCert = swu::NORCaCert::instance();
         if ( ! caCert->isLoaded()) {
            ETG_TRACE_ERR(("%s: read certificate failed", __func__));
            return;
         }
         if (caCert->isEmpty()) {
            ETG_TRACE_USR3(("%s Certificate is not set, returning true without signature check", __func__));
            return;
         }
         // std::string key;
         // if (false == CodeCert::instance()->extractPublicKeyFromCertificate(certificate, key)) {
         //    ETG_TRACE_ERR(("%s: extract key public from certificate failed", __func__));
         //    return;
         // }
         // if (!CodeCert::instance()->checkSignature(message, signature, key)) {
         //    ETG_TRACE_COMP(("swu_filesystem testCheckSignature failed"));
         //    return;
         // }
         ETG_TRACE_COMP(("swu_filesystem testCheckSignature passed"));
      }
}
