#ifndef SWU_ASF_UTIL_HPP
#define SWU_ASF_UTIL_HPP

#include <sstream>
#include <iostream>
#include "asf/core/BaseComponent.h"
#include "asf/core/Logger.h"
#include "asf/core/Proxy.h"
#include "asf/core/ApplicationIF.h"
#include <boost/ptr_container/ptr_vector.hpp>

#include "util/swu_types.h"
#include "util/swu_msg.hpp"
#include "util/swu_singleton.hpp"

#include "util/swu_trace.h"



namespace swu {

template <class COMP>
::std::vector< COMP > convertBoostToStlVector(::boost::ptr_vector< COMP > boostInfo) {
   ::std::vector< COMP > stlInfo;
   class ::boost::ptr_vector< COMP >::iterator Iter;
   for(Iter = boostInfo.begin(); Iter != boostInfo.end(); Iter++) {
      COMP val = *Iter;
      stlInfo.push_back(val);
   }
   return stlInfo;
}

#ifdef VARIANT_S_FTR_ENABLE_SWU_DBUS
#define SWU_ASF_GET_DATA(ASF_OBJ_PTR, MEMBER_NAME) ASF_OBJ_PTR->get##MEMBER_NAME()
#else
#define SWU_ASF_GET_DATA(ASF_OBJ_PTR, MEMBER_NAME) ASF_OBJ_PTR->getData()
#endif

class AsfTraceHelper {
public:
   static void traceAsfMsgName(tUInt traceClass, tUInt traceLevel,  std::string const &name, std::string const &line);
   static void traceSendResult(char const *method, act_t storedAct, bool bOk);
   static void traceAct(const char *action, act_t act);
};

template <class PROXY, class COMP>
struct Msg_ProxyAvailability: public Msg< Msg_ProxyAvailability<PROXY, COMP>, COMP > 
{
   Msg_ProxyAvailability(bool available_=false):available(available_){}
   Msg_ProxyAvailability(PROXY *, bool available_):available(available_){}
   virtual ~Msg_ProxyAvailability() {
   }
   virtual tVoid vTrace() {  }
   bool available;
};


template <class USER, class PROXY>
class ProxyAccess;

class ProxyAccessBase {
public:
   ProxyAccessBase():
      _bAvail(false)
   {}
   virtual ~ProxyAccessBase() {
   }
   virtual boost::shared_ptr<asf::core::Proxy> getProxyBase()=0;
   virtual void notifyAvail(bool avail)=0;
   void init(std::string port);

   std::string const &getPortName() {
      return _port;
   }
   bool isAvail() {
      return _bAvail;
   }
protected:
   void traceCreateProxy();

   std::string _port;
   bool _bAvail;
};

class ProxyList {
public:
   boost::shared_ptr<asf::core::Proxy> getProxy(std::string port);
   void onAvail(boost::shared_ptr<asf::core::Proxy>, bool avail);
   void registerProxy(std::string port, ProxyAccessBase *proxy);
   std::map <std::string, ProxyAccessBase * > const &getList() {
      return _proxies;
   }
private:
   std::map <std::string, ProxyAccessBase * > _proxies;
};

/*
  class to create proxy outside of component.
  Shall be used by project-specific part to create project-specific proxies.
  So we can add proxies to the component without changing the header-file of the component.
 */
template <class USER, class PROXY>
class ProxyAccess: public ProxyAccessBase,
                   public Singleton<ProxyAccess<USER, PROXY> > {
   friend class Singleton<ProxyAccess<USER, PROXY> >;
public:
   typedef typename USER::compType compType;
   /* only accessible constructor. */
   ProxyAccess()
   {}
   virtual boost::shared_ptr<asf::core::Proxy> getProxyBase() {
      return getProxy();
   }
   virtual void notifyAvail(bool avail) {
      if (_bAvail != avail) {
         _bAvail = avail;
         Msg_ProxyAvailability<PROXY, compType> msg(_bAvail);
         msg.iNotifyByNonMember(this);

      }
   }


  void setProxy(boost::shared_ptr< PROXY > proxy, std::string port) {
    _proxy=proxy;
    ProxyAccessBase::_port=port;
    USER::instance()->_proxyList.registerProxy(_port, this);
    
  }

  boost::shared_ptr< PROXY > getProxy() {
      /*
        check if proxy has already been created ...
       */
      if (!_proxy) {
         /*
           if not, create it now ...
          */
         ProxyAccessBase::traceCreateProxy();
         _proxy = PROXY::createProxy(_port,  *USER::instance());
         /*
           and register it to get it back next time calling _comp->_proxyList.getProxy(port)
          */
         USER::instance()->_proxyList.registerProxy(_port, this);
         // since we are no real singleton, register here our instance.
      }
      return _proxy;
   }

private:
   boost::shared_ptr< PROXY > _proxy;
};




  /*
    AutoProperty
    Utility for properties that handle register and unregister automatically following service-availability
  */
  class AutoProperty {
  public:
    virtual ~AutoProperty() {
    }

    virtual tVoid vOnSrvAvail()=0;
    virtual tVoid vOnSrvUnavail()=0;
    virtual boost::shared_ptr<asf::core::Proxy> getProxy() = 0;
    virtual const char *getName()=0;
  };




#define AUTO_PROPERTY_BASE(COMP, USER, PROXY_GETTER, PROPERTY, LINE, REGISTER, DEREGISTER, DEREGISTERALL) \
   class AutoProperty##LINE: public ::swu::AutoProperty                 \
  {									\
  public:								\
     AutoProperty##LINE() { COMP::_autoPropertyList.push_back(this);}; \
    virtual tVoid vOnSrvAvail() {					\
      USER *comp=USER::instance();	\
      ::swu::traceError("AutoProperty Register:"  #PROPERTY);           \
      comp->PROXY_GETTER()->send##PROPERTY##REGISTER(*comp);		\
    }									\
    virtual tVoid vOnSrvUnavail()  {					\
      ::swu::traceError("AutoProperty: Deregister" #PROPERTY);	\
    }									\
    virtual boost::shared_ptr<asf::core::Proxy> getProxy()  {		\
      return USER::instance()->PROXY_GETTER();		\
    }									\
    virtual const char *getName() {					\
      return #PROXY_GETTER "::" #PROPERTY;				\
    }									\
  } autoProperty##LINE

template<class COMP, class PROXY>
class NotifyAvail: public AutoProperty {
   public:
   NotifyAvail() {COMP::_autoPropertyList.push_back(this);};
      virtual tVoid vOnSrvAvail() {
         Msg_ProxyAvailability<PROXY, typename COMP::compType> msg((PROXY*) 0, true);
         msg.iNotifyByNonMember((COMP *)0);
      }
      virtual tVoid vOnSrvUnavail()  {
         Msg_ProxyAvailability<PROXY, typename COMP::compType> msg((PROXY*) 0, false);
         msg.iNotifyByNonMember((COMP *)0);
    }
    virtual boost::shared_ptr<asf::core::Proxy> getProxy()  {
       return COMP::instance()->getProxy((PROXY*) 0);
    }
    virtual const char *getName() {
       return "";
    }
};



  template <class T>
  std::string asfToString(char const *name, T const &val) {
    std::string json=name;
    json+= ":";
    serializeJson(val,json);
    return json;
  }

#define SWU_ASF_TO_STRING(name) asfToString(#name, name).c_str()


  template <class T>
  void traceAsfMsg(tUInt traceClass, tUInt traceLevel,  T const &val, std::string const &name="") {
    std::string json;
    serializeJson(val,json);
    std::istringstream stream(json);

    while(!stream.eof()) {
      std::string line;
      getline(stream, line);
      //if (!line.size()) {
	//break;
      //}
      AsfTraceHelper::traceAsfMsgName(traceClass, traceLevel, name, line);
    }
  }

#define SWU_TRACE_ASF_MSG(CLS, LVL, VAL, NAME)	\
    if (etg_bIsTraceActive(CLS, LVL)) {			\
      ::swu::traceAsfMsg(CLS, LVL,  VAL, NAME);		\
    }							\

#define SWU_TRACE_ASF_MSG_VAL(CLS, LVL, VAL)	\
    if (etg_bIsTraceActive(CLS, LVL)) {			\
      ::swu::traceAsfMsg(CLS, LVL,  VAL, #VAL);		\
    }							\



/*
  storing of act per request
 */

template <class ASFREQ>
class AsfReqActHolder {
public:
   static void set(const ::boost::shared_ptr< ASFREQ >& payload) {
      _act=payload->getAct();
      AsfTraceHelper::traceAct("set", _act);
   }
   static act_t get() {
      act_t act=_act;
      AsfTraceHelper::traceAct("get", _act);
      _act=0;
      return act;
   }
   static bool isActive() {
      return _act != 0;
   }
   
private:
   static act_t _act;
   
};
template <typename ASFREQ> act_t AsfReqActHolder<ASFREQ>::_act = 0;

// helper for macro below
#define SWU_VA_COMMA(...) SWU_GET_10TH_ARG(,##__VA_ARGS__,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,SWU_COMMA,)
#define SWU_GET_10TH_ARG(a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,...) a10
#define SWU_COMMA ,


/*
  send response if act has been stored.
  This avoid asserts in ASF when asf-messages are not answered in sequence or if request have been spoofed via ttfis.
 */

#define SWU_SEND_ASF_RESULT_IMPL(NS, STUB, METHOD_NAME, ...) {           \
      act_t storedAct =::swu::AsfReqActHolder<NS::METHOD_NAME##Request>::get() ; \
      ::swu::AsfTraceHelper::traceSendResult( #METHOD_NAME, storedAct, bOk);   \
      if (storedAct) {                                                  \
         NS::STUB::send##METHOD_NAME##Response(__VA_ARGS__ SWU_VA_COMMA(__VA_ARGS__) storedAct); \
      }                                                                 \
   }

#define IS_ASF_RESPONSE_PENDING(NS, METHOD_NAME) ::swu::AsfReqActHolder<NS::METHOD_NAME##Request>::isActive()

  /*
    Dispatching of messages
    Utilities to broadcast received asf-method-request to registered internal clients.
    If no internal client was registered, a negative response is send to the asf-client. (bOk==FALSE)
    macros are generating complete method  "onXXXRequest"
  */
#define AUTO_DISPATCH_REQ_0(NS, METHOD_NAME)				\
  virtual void on##METHOD_NAME##Request (const ::boost::shared_ptr< NS::METHOD_NAME##Request >& payload) { \
    Msg_##METHOD_NAME##Request msg(payload);				\
    ::swu::AsfReqActHolder< NS::METHOD_NAME##Request >::set(payload);          \
    if (!notify(msg)) {                                        \
      send##METHOD_NAME##Response(FALSE);				\
    }									\
  }




#define AUTO_DISPATCH_REQ_1_(NS, NS_T, METHOD_NAME,RSP_ARG_TYPE)	\
  virtual void on##METHOD_NAME##Request (const ::boost::shared_ptr< NS::METHOD_NAME##Request >& payload) { \
    Msg_##METHOD_NAME##Request msg(payload);				\
    ::swu::AsfReqActHolder< NS::METHOD_NAME##Request >::set(payload);          \
    if (!notify(msg)) {                                        \
      RSP_ARG_TYPE arg;						\
      send##METHOD_NAME##Response(FALSE,arg);				\
    }									\
  }

#define AUTO_DISPATCH_REQ_1(NS, METHOD_NAME, ...)	\
  AUTO_DISPATCH_REQ_1_(NS, NS##_T, METHOD_NAME,__VA_ARGS__)

#define AUTO_DISPATCH_REQ_5_(NS, NS_T, METHOD_NAME,RSP_ARG_TYPE1,RSP_ARG_TYPE2,RSP_ARG_TYPE3,RSP_ARG_TYPE4,RSP_ARG_TYPE5)	\
  virtual void on##METHOD_NAME##Request (const ::boost::shared_ptr< NS::METHOD_NAME##Request >& payload) { \
    Msg_##METHOD_NAME##Request msg(payload);				\
    ::swu::AsfReqActHolder< NS::METHOD_NAME##Request >::set(payload);          \
    if (!notify(msg)) {                                        \
      RSP_ARG_TYPE1 arg1;						\
      RSP_ARG_TYPE2 arg2;						\
      RSP_ARG_TYPE3 arg3;						\
      RSP_ARG_TYPE4 arg4;						\
      RSP_ARG_TYPE5 arg5;						\
      send##METHOD_NAME##Response(FALSE,arg1,arg2,arg3,arg4,arg5);	\
    }									\
  }


#define AUTO_DISPATCH_REQ_5(NS, METHOD_NAME, ...) \
   AUTO_DISPATCH_REQ_5_(NS, NS##_T, METHOD_NAME, __VA_ARGS__)


#define AUTO_DISPATCH_RES(NS, PROXY_TYPE, MSG_TYPE)			\
  virtual void on##MSG_TYPE (const ::boost::shared_ptr< NS::PROXY_TYPE >& proxy, const ::boost::shared_ptr< NS::MSG_TYPE >& payload) { \
    Msg_##MSG_TYPE msg(proxy, payload);					\
    notify(msg);							\
  }


#define AUTO_DISPATCH_ERROR(NS, PROXY_TYPE, MSG_TYPE)			\
  virtual void on##MSG_TYPE (const ::boost::shared_ptr< NS::PROXY_TYPE >& proxy, const ::boost::shared_ptr< NS::MSG_TYPE >& payload) { \
    (void)proxy;  \
    (void)payload;   \
    swu::traceError(("Received error-msg " #NS "::" #MSG_TYPE));		\
  }


  // macros to define dispatchable wraper-classes around XXXRequest received from asf-client
#define DISPATCHABLE_MSG_DECL(COMP, NS, TYPE)                            \
   struct Msg_##TYPE: public ::swu::Msg<Msg_##TYPE, COMP> {             \
    Msg_##TYPE(const ::boost::shared_ptr< NS::TYPE> _payload):	\
    payload(_payload) {};					\
    virtual tVoid vTrace() {					\
       ::swu::traceInfo(" Msg_" #NS "::" #TYPE);                \
    }								\
    const ::boost::shared_ptr<NS::TYPE> payload;		\
  }



#define DISPATCHABLE_MSG_DECL_PROXY(COMP, NS, PROXY_TYPE, TYPE)		\
   struct Msg_##TYPE: public ::swu::Msg<Msg_##TYPE, COMP> {             \
    Msg_##TYPE(const ::boost::shared_ptr< NS::PROXY_TYPE >& proxy_, const ::boost::shared_ptr< NS::TYPE> payload_): \
    payload(payload_), proxy(proxy_) {};				\
    virtual tVoid vTrace() {						\
       ::swu::traceInfo(" Msg_" #NS "::" #TYPE);                        \
    }									\
    const ::boost::shared_ptr<NS::TYPE> payload;			\
    const ::boost::shared_ptr<NS::PROXY_TYPE> proxy;			\
  }




}


#endif
