#include "swu_HyperFlashBootChain.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "swu_execCommand.h"
#include "base/imp/DownloadPipe.h"
#include <fcntl.h>        // for open() command and flags.
#include <unistd.h>       // for pwrite32/pread64.
#include <mtd/mtd-abi.h>  // for MTD ioctl commands and structs.
#include <string>
#include <sstream>
#include <iostream>
#include "swu_trace.h"

#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SWUPDATE_UTIL
  #include "trcGenProj/Header/swu_HyperFlashBootChain.cpp.trc.h"
#endif

namespace swu {

swu_HyperFlashBootChain* swu_HyperFlashBootChain::m_pInstance = NULL;

swu_HyperFlashBootChain* swu_HyperFlashBootChain::Instance()
{
    if (!m_pInstance)   // Only allow one instance of class to be generated.
        m_pInstance = new swu_HyperFlashBootChain;

    return m_pInstance;
}


swu_HyperFlashBootChain::swu_HyperFlashBootChain()
{
    block_size = BlOCK_SIZE_HYPERFLASH;
    count = COUNT_HYPERFLASH;
}

swu_HyperFlashBootChain::~swu_HyperFlashBootChain()
{

}
/*
* FUNCTION:   bool swu_HyperFlashBootChain::bWriteMagicToRawFlash
*
* DESCRIPTION:  Write the magic to the raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic
*                   tU32     u32Value        e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
bool swu_HyperFlashBootChain:: bWriteMagicToRawFlash (unsigned int u32FlashMagicOffset, unsigned int u32Value)
{
  // Read sector 1 and 2
  // Determine selected sector (A)
  // Update the other sector (B) with new data:
  // - copy all data from selected sector (A)
  // - increment the counter by 1
  // - set the sector to valid
  //
  // Clear sector A:
  // - invalidate sector (flag)
  // - counter = 0
  // - copy bootchain from sector B
  //
  // Serialize active sector to NOR flash with MTD_IO functions. Erase existing NOR storage before.

    bool  bRetVal  = FALSE;
    tU32   u32FlashAdressToSet  = 0;
    tU32   u32FlashAdressToClear= 0;
    tU32   u32MagicAreaSelected = 0;
    tU32   u32MagicIndex        = 0;
    tU32   au32MagicArea1[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};
    tU32   au32MagicArea2[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};
    tPVoid pvMagicBufferToSet                   = NULL;
    tPVoid pvMagicBufferToClear                 = NULL;

    ETG_TRACE_USR4 (("bWriteMagicToRawFlash:: Offset %d", u32FlashMagicOffset));
    print2errmem("DNL: bWriteMagicToRawFlash %08X at Offset %08X\n", u32Value, u32FlashMagicOffset);
    ETG_TRACE_FATAL(("bWriteMagicToRawFlash:: %08X at Offset %08X", u32Value, u32FlashMagicOffset));

    // Read both magic areas from flash
    bRetVal = bReadMagicAreas (
        u32FlashMagicOffset,
        &u32MagicIndex,
        &au32MagicArea1[0],
        &au32MagicArea2[0],
        &u32MagicAreaSelected);

    if (u32MagicIndex >= DNL_MAGIC_ARRAY_SIZE_HYPERFLASH) {
        ETG_TRACE_FATAL(("Magic Value Index out of range while writing (%d)", u32MagicIndex));
        return false;
    }

    if(u32MagicAreaSelected == 1)   // First magic area selected:
    {
       ETG_TRACE_FATAL(("u32MagicAreaSelected=1"));
        // Write the new data to the second magic area
        pvMagicBufferToSet                = &au32MagicArea2[0];
        u32FlashAdressToSet               = DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH;
        au32MagicArea2[0]                 = DL_MAGIC_SECTOR_VALID_HYPERFLASH; // Magic valid
        au32MagicArea2[4]                 = au32MagicArea1[4] + 1;    // Increment the counter
        au32MagicArea2[8]                 = au32MagicArea1[8];        // Copy the same DNL state
        au32MagicArea2[12]                = au32MagicArea1[12];       // Copy the same DNL source
        au32MagicArea2[16]                = au32MagicArea1[16];       // Copy the same BootChain
        au32MagicArea2[u32MagicIndex]     = u32Value;                 // Set the new value
        // Clear the old magic:
        pvMagicBufferToClear              = &au32MagicArea1[0];
        u32FlashAdressToClear             = DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH;
        au32MagicArea1[ 0]                = DL_MAGIC_SECTOR_INVALID_HYPERFLASH;  // Magic invalid
        au32MagicArea1[ 4]                = 0;                        // Reset counter
        au32MagicArea1[16]                = au32MagicArea2[16];       // Copy the same BootChain
    }
    else                            // Second magic area selected:
    {
       ETG_TRACE_FATAL(("u32MagicAreaSelected=2"));
        // Write the new data to the first magic area
        pvMagicBufferToSet                = &au32MagicArea1[0];
        u32FlashAdressToSet               = DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH;
        au32MagicArea1[0]                 = DL_MAGIC_SECTOR_VALID_HYPERFLASH;    // Magic valid
        au32MagicArea1[4]                 = au32MagicArea2[4] + 1;    // Increment the counter
        au32MagicArea1[8]                 = au32MagicArea2[8];        // Copy the same DNL state
        au32MagicArea1[12]                = au32MagicArea2[12];       // Copy the same DNL source
        au32MagicArea1[16]                = au32MagicArea2[16];       // Copy the same BootChain
        au32MagicArea1[u32MagicIndex]     = u32Value;                 // Set the new value
        // Clear the old magic:
        pvMagicBufferToClear              = &au32MagicArea2[0];
        u32FlashAdressToClear             = DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH;
        au32MagicArea2[ 0]                = DL_MAGIC_SECTOR_INVALID_HYPERFLASH;  // Magic invalid
        au32MagicArea2[ 4]                = 0;                        // Reset the counter
        au32MagicArea2[16]                = au32MagicArea1[16];       // Copy the same BootChain
    }
    // Write the new magic area to raw flash
    bRetVal = writeToFac( (tPU8)pvMagicBufferToSet,u32FlashAdressToSet,TRUE);
    if (bRetVal) // if setting of new magic was OK:
    {
       ETG_TRACE_FATAL(( "writing succesful, erasing old magic"));
        // Clear the old magic area
        bRetVal = writeToFac((tPU8)pvMagicBufferToClear,u32FlashAdressToClear,TRUE);
        if(bRetVal){
           ETG_TRACE_FATAL(("erasing old magic is succesful"));
        }
        else{
           ETG_TRACE_FATAL(("erasing old magic failed"));
        }
    }
    else
    {
       ETG_TRACE_FATAL(("error in writing"));
        //@todo: handle error case
    }
    ETG_TRACE_USR4 (("bWriteMagicToRawFlash was left"));
    return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   bool swu_HyperFlashBootChain::bReadMagicAreas
*
* DESCRIPTION:  Write the magic to the raw nor flash
*
*
* PARAMETER INPUT:  tU32   u32FlashMagicOffset e.g. 0x00000020 for DNL MagicRecoveryDownload
*                          0x00000030 for DNL MagicEthernetSource
*                          0x00000040 for DNL MagicBootChainSelection
*
* PARAMETER OUTPUT: tPU32  pu32MagicIndex     Index of DWORD in u32FlashMagicArea1[] :  0, 4, 8, 12, 16 if OK
*                                                            20 in case of error
*         tPU32  pu32FlashMagicArea1    u32FlashMagicArea1[20]
*         tPU32  pu32FlashMagicArea2    u32FlashMagicArea2[20]
*         tPU32  pu32MagicAreaSelected  0: none, 1: Area1, 2: Area2
*
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
bool swu_HyperFlashBootChain::bReadMagicAreas (
    tU32  u32FlashMagicOffset,
    tPU32 pu32MagicIndex,
    tPU32 pu32FlashMagicArea1,
    tPU32 pu32FlashMagicArea2,
    tPU32 pu32MagicAreaSelected)
{
  // Read Sector 1 from flash
  // - Read (size of sector) bytes
  // - dump read raw bytes into file /tmp/downloadPipe -> ABSTRACT
  //
  // Read sector 2 from flash (same implementation as above)
  //
  // Check whether number of read bytes equals size of defined bytes per sector (#define)
  // - error case: write to /tmp/downloadPipe -> ABSTRACT
  //
  // Check which sector is valid, return selected sector to caller(1 or 2), return 0 in error case

    ETG_TRACE_USR4 (("bReadMagicAreas was entered"));   
    bool  bRetVal        = FALSE;
    int    fd             = -1;
    int    retcode1       = 0;
    int    retcode2       = 0;
    std::string devname;
    tChar  tmpBuffer [256]   = {0};

    ETG_TRACE_USR4 (("DNL: ------------------------"));
    ETG_TRACE_USR4 (("DNL: bReadMagicAreas"));

    *pu32MagicAreaSelected = 0;   // Initialze with invalid value
    *pu32MagicIndex        = 20;  // Initialze with invalid value

    // Check magic offset and set the index
    // e.g. u32FlashMagicOffset is 0x00, 0x10, 0x20, 0x30, 0x40
    switch (u32FlashMagicOffset)
    {
        // @todo: optimize: u32FlashMagicOffset can be directly mapped to pu32MagicIndex, no subtractions are required.
        case DL_MAGIC_SECTOR_VALID_FLAG_HYPERFLASH   - DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH:  // 0x00160000 - 0x00160000 = 0x00000000  @todo: lint: Info 778: Constant expression evaluates to 0 in operation '-': disable warning
            *pu32MagicIndex = 0;
            break;

        case DL_MAGIC_SECTOR_COUNTER_FLAG_HYPERFLASH - DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH:  // 0x00160010 - 0x00160000 = 0x00000010
            *pu32MagicIndex = 4;
            break;

        case DL_UPDATE_STATUS_FLAG_HYPERFLASH        - DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH:  // 0x00160020 - 0x00160000 = 0x00000020
            *pu32MagicIndex = 8;
            break;

        case DL_ETHERNET_FLAG_HYPERFLASH       - DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH:  // 0x00160030 - 0x00160000 = 0x00000030
            *pu32MagicIndex = 12;
            break;

        case DL_ROOTFS_FLAG_HYPERFLASH         - DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH:  // 0x00160040 - 0x00160000 = 0x00000040
            *pu32MagicIndex = 16;
            break;

        default: // Error
            *pu32MagicIndex = 20;
            break;
    }

    off_t u32FlashAdress = 0;
    off_t u32FlashDevOffset = 0;

    // Read magic area 1 from the flash
    u32FlashAdress = DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH;
    retcode1 = readFromFac(u32FlashAdress,pu32FlashMagicArea1);

    ETG_TRACE_FATAL(("readFromFac:retcode1:%d",retcode1));
    // @todo: replace tracing
    sprintf(tmpBuffer, "ReadMagicArea1=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
        (unsigned int) *pu32FlashMagicArea1,
        (unsigned int) *(pu32FlashMagicArea1+4),
        (unsigned int) *(pu32FlashMagicArea1+8),
        (unsigned int) *(pu32FlashMagicArea1+12),
        (unsigned int) *(pu32FlashMagicArea1+16) );

    printToDownloadPipe(tmpBuffer);

    ETG_TRACE_FATAL(("ReadMagicArea1=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
                     (unsigned int) *pu32FlashMagicArea1,
                     (unsigned int) *(pu32FlashMagicArea1+4),
                     (unsigned int) *(pu32FlashMagicArea1+8),
                     (unsigned int) *(pu32FlashMagicArea1+12),
                     (unsigned int) *(pu32FlashMagicArea1+16) ));

    // Read magic area 2 from the flash
    u32FlashAdress = DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH;
    retcode2 = readFromFac(u32FlashAdress,pu32FlashMagicArea2);

    ETG_TRACE_FATAL(( "readFromFac:retcode2:%d",retcode2));

    // @todo: replace tracing
    sprintf(tmpBuffer, "ReadMagicArea2=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
        (unsigned int) *pu32FlashMagicArea2,
        (unsigned int) *(pu32FlashMagicArea2+4),
        (unsigned int) *(pu32FlashMagicArea2+8),
        (unsigned int) *(pu32FlashMagicArea2+12),
        (unsigned int) *(pu32FlashMagicArea2+16) );

    printToDownloadPipe(tmpBuffer);

    ETG_TRACE_FATAL(( "ReadMagicArea2=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
                      (unsigned int) *pu32FlashMagicArea2,
                      (unsigned int) *(pu32FlashMagicArea2+4),
                      (unsigned int) *(pu32FlashMagicArea2+8),
                      (unsigned int) *(pu32FlashMagicArea2+12),
                      (unsigned int) *(pu32FlashMagicArea2+16) ));

    // Check both magic areas
    if( (retcode1 == DNL_MAGIC_AREA_SIZE_HYPERFLASH) && (retcode2 == DNL_MAGIC_AREA_SIZE_HYPERFLASH) ) // read OK?
    {
        bRetVal = TRUE;
    }
    else // read error
    {
        // @todo: replace tracing
        ETG_TRACE_USR4 (("DNL: bReadMagicAreas: read flash device ERROR"));
        printToDownloadPipe("read flash device ERROR");        
    }

    // Check which magic area is valid
    if ( (*pu32FlashMagicArea1 == DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (*pu32FlashMagicArea2 != DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        *pu32MagicAreaSelected = 1; // only area 1 is valid
    }
    if ( (*pu32FlashMagicArea1 != DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (*pu32FlashMagicArea2 == DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        *pu32MagicAreaSelected = 2; // only area 2 is valid
    }
    if ( (*pu32FlashMagicArea1 == DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (*pu32FlashMagicArea2 == DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        // both areas are valid
        // Check also the counters; magic with higher counter is selected
        if ( *(pu32FlashMagicArea1 + 4) >= *(pu32FlashMagicArea2 + 4) )
        {
            *pu32MagicAreaSelected = 1;
        }
        else
        {
            *pu32MagicAreaSelected = 2;
        }
    }
    if ( (*pu32FlashMagicArea1 != DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (*pu32FlashMagicArea2 != DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        *pu32MagicAreaSelected = 0; // both are invalid  //@todo: check how this error case is handled in calling functions
    }

    if (*pu32MagicIndex == 20) bRetVal  = FALSE;

    ETG_TRACE_USR4 (("bReadMagicAreas was left"));
    return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   bool swu_HyperFlashBootChain::bReadMagicFromRawFlash
*
* DESCRIPTION:  Read the flag content from raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic, 0x00000040 for BootChain
*
* PARAMETER OUTPUT: tPU32    pu32Value         e.g  0x2342ABCD or  0xFFFFFFFF
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
bool swu_HyperFlashBootChain::bReadMagicFromRawFlash (const unsigned int inFlashAddress, unsigned int& outValue)
{
    // in: inFlashAddress is semantically the flash magic offset, e.g. 0x00, 0x10, 0x20, ...


    //u32MagicIndex: Index of DWORD in u32FlashMagicArea1[] :  0, 4, 8, 12, 16 if OK, 20 in case of error


    // Read sector 1 and 2
    // Determine selected sector
    // out: 4 bytes stored at the address of the requested magic offset of the selected sector, in error case 0
    // Write read raw bytes to file /tmp/downloadPipe -> ABSTRACT

    ETG_TRACE_USR4 (("bReadMagicFromRawFlash was entered"));
    bool   bRetVal              = FALSE;

    tU32    u32MagicIndex        = 0;
    tU32    u32MagicAreaSelected = 0;
    tU32    au32MagicArea1[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0}; //@todo: only first element is set to 0, init also the other ones
    tU32    au32MagicArea2[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0}; //@todo: only first element is set to 0, init also the other ones
    tChar   tmpBuffer [256]     = {0};

    // Read both magic areas from flash
    bRetVal = bReadMagicAreas (inFlashAddress, &u32MagicIndex, &au32MagicArea1[0], &au32MagicArea2[0], &u32MagicAreaSelected);

    if (u32MagicIndex >= DNL_MAGIC_ARRAY_SIZE_HYPERFLASH) {
        ETG_TRACE_FATAL(("Magic Value Index out of range (%d)", u32MagicIndex));
        return false;
    }

    if (u32MagicAreaSelected == 1)
    {
        outValue = au32MagicArea1[u32MagicIndex];
        bRetVal = TRUE;
    }
    else if (u32MagicAreaSelected == 2)
    {
        outValue = au32MagicArea2[u32MagicIndex];
        bRetVal = TRUE;
    }
    else
    {
        outValue = 0; // invalid
    }

    ETG_TRACE_USR4 (("DNL: bReadMagic bRet=%d Area=%d Value=0x%8X.X", bRetVal, u32MagicAreaSelected, outValue));

    sprintf(tmpBuffer, "ReadMagic Area=%d Offset=%d Index=%d Value=0x%8.8X",
        (int)u32MagicAreaSelected, (int)inFlashAddress,(int)u32MagicIndex, outValue);
    printToDownloadPipe(tmpBuffer);
    ETG_TRACE_USR4 (("bReadMagicFromRawFlash was left"));

    return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   bool swu_HyperFlashBootChain::bWriteMagicBootChain
*
* DESCRIPTION:  Write the magic to the raw nor flash
*       and set the active boot partition register in the eMMC
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic
*                   tU32     u32Value        e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
bool swu_HyperFlashBootChain::bWriteMagicBootChain(const unsigned int inFlashAddress, const unsigned int inValue)
{

    // inFlashAddress = BOOTCHAIN_MAGIC_ADDRESS (0x40, set from lua script)
    // update bootchain flag in NOR, see above
    // additionally:
    // change mmc to use the correct fast boot area (must be synced to the boot chain configuration in NOR



    ETG_TRACE_USR4 (("bWriteMagicBootChain was entered"));

    bool  bRetVal  = FALSE;
    // Write magic to raw nor flash
    bRetVal = bWriteMagicToRawFlash (inFlashAddress, inValue);

    ETG_TRACE_USR4 (("bWriteMagicBootChain was left"));
    return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   bool swu_HyperFlashBootChain::bReadFlagFromRawFlash
*
* DESCRIPTION:  Read the flag content from raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashAdress e.g. 0x400E0000 for RootPartitionNumber, 0x400E0010 for DNL Magic
*                   tPU32    pu32Value    e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
bool swu_HyperFlashBootChain:: bReadFlagFromRawFlash (off_t u32FlashAdress, tPU32 pu32Value)
{

    // MTD_IO: Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
    // calculate the flash address which shall be read
    // open the device
    // read 4 bytes from flash address
    //   log error case to /tmp/downloadPipe
    // caution: no proper error indication, no error handling

    ETG_TRACE_USR4(("bReadFlagFromRawFlash was entered"));

    bool bRetVal = FALSE;

    tChar tmpBuffer [256] = {0};
    int fd;
    int retcode;
    fd      = -1;

    retcode = readFromFac(u32FlashAdress,pu32Value);

    if(retcode != 4)
    {
        printToDownloadPipe("read flash device ERROR");
    }
    else
    {
        sprintf(tmpBuffer, "read flag from flash val=0x%8.8X", (unsigned int)*pu32Value);
        printToDownloadPipe(tmpBuffer);
        bRetVal = TRUE; // Read OK
    }
    ETG_TRACE_USR4 (("bReadFlagFromRawFlash was left"));
    return bRetVal;
}

/*
* FUNCTION:   int swu_HyperFlashBootChain::writeToFac
*
* DESCRIPTION: Writes to Flash memory
*
*
* PARAMETER INPUT:  const tU8 *pu8BufferPointer
                    off_t inDestinationOffset
                    bool flag
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      int
*
*/

int swu_HyperFlashBootChain:: writeToFac(const tU8 *pu8BufferPointer,off_t inDestinationOffset,bool flag)
{
    int retcode = 0;
    int fd = -1;
    std::string sysCmd = "";
    std::string InputFile = FAC_FILE_TO_WRITE_HYPERFLASH;
    std::string WriteCommand = FAC_WRITE_COMMAND_HYPERFLASH;

    int DestinationOffset = retNoOfBytesToSkip(inDestinationOffset);

    if (flag == true)
    {
       EraseFac(DestinationOffset,flag); //need to check,how much to clear ??
    }

    fd = open(InputFile.c_str(),O_RDWR | O_CREAT);
    if(fd >= 0)
    {
        retcode =static_cast<int> (write(fd, (tPU8)pu8BufferPointer, DNL_MAGIC_AREA_SIZE_HYPERFLASH));
        ETG_TRACE_FATAL(( "writeToFac:retcode:%d", retcode));
        close(fd);
    }

    std::ostringstream str1;
    str1 << DestinationOffset;
    std::string offset = str1.str();

    sysCmd = WriteCommand + InputFile + BLOCK_SIZE_STRING + block_size + SEEK_STRING + offset + COUNT_STRING + count;

    ETG_TRACE_FATAL(( "writeToFac:sysCmd:%s", sysCmd.c_str()));
    execCommand(sysCmd.c_str());

    // sleep(120); //currently due to platform issues ,we need to wait for 1 min

    execCommand("rm -rf /tmp/InputFile");
    return retcode;
}

/*
* FUNCTION:   int swu_HyperFlashBootChain::readFromFac
*
* DESCRIPTION: Reads from Flash memory
*
*
* PARAMETER INPUT:  u32FlashAdress
                    tPU32 pu32FlashMagicArea
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      int
*
*/

int swu_HyperFlashBootChain:: readFromFac(off_t u32FlashAdress,tPU32 pu32FlashMagicArea)
{
    int retcode = 0;
    int fd = -1;
    std::string OutputFile = FAC_FILE_TO_READ_HYPERFLASH;
    std::string ReadCommand = FAC_READ_COMMAND_HYPERFLASH;
    std::string sysCmd = "";
    std::ostringstream str1;
    std::string offset = "";
    size_t inDestinationOffset =0;

    if (u32FlashAdress == DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH)
    {
        inDestinationOffset = retNoOfBytesToSkip(DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH);
        str1 << inDestinationOffset;
        offset = str1.str();
    }
    else
    {
        inDestinationOffset = retNoOfBytesToSkip(DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH);
        str1 << inDestinationOffset;
        offset = str1.str();
    }

    sysCmd = ReadCommand + OutputFile + BLOCK_SIZE_STRING + block_size + SKIP_STRING + offset + COUNT_STRING + count;

    ETG_TRACE_FATAL(("readFromFac:sysCmd:%s", sysCmd.c_str()));
    execCommand(sysCmd.c_str());
    // sleep(120); //currently due to platform issues ,we need to wait for 1 min
    fd = open(OutputFile.c_str(),O_RDONLY);
    if(fd >= 0)
    {
        retcode =static_cast<int> (read(fd, pu32FlashMagicArea, DNL_MAGIC_AREA_SIZE_HYPERFLASH));
        ETG_TRACE_FATAL(( "readFromFac:retcode:%d", retcode));
        close(fd);
    }
    execCommand("rm -rf /tmp/OutputFile");
    return retcode;

}

/*
* FUNCTION:   int swu_HyperFlashBootChain::retNoOfBytesToSkip
*
* DESCRIPTION: returns number of bytes to skip
*
*
* PARAMETER INPUT: off_t starting_address
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      int
*
*/

int swu_HyperFlashBootChain:: retNoOfBytesToSkip(off_t starting_address)
{
    int bs =0;
    int inDestinationOffset = 0 ;
    std::stringstream block(block_size);
    block >> bs;
    
    if(bs == 0) {
       return 0;
    }
    
    if(starting_address == DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH){
        inDestinationOffset = (DL_MAGIC_SECTOR_VALID_FLAG_HYPERFLASH_DECIMAL/bs);
    }
    else if(starting_address == DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH){
        inDestinationOffset = (DL_MAGIC_SECTOR_VALID_FLAG_2_HYPERFLASH_DECIMAL/bs);
    }
    else{
       ETG_TRACE_FATAL(("Error in retNoOfBytesToSkip"));
    }
    return inDestinationOffset;
}

/*
* FUNCTION:   int swu_HyperFlashBootChain::EraseFac
*
* DESCRIPTION: Erases flash memory
*
*
* PARAMETER INPUT: off_t inDestinationOffset
                   bool flag
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      int
*
*/

void swu_HyperFlashBootChain:: EraseFac(int inDestinationOffset,bool flag)
{
    std::string EraseCommand = FAC_ERASE_COMMAND_HYPERFLASH;
    std::ostringstream str1;
    str1 << inDestinationOffset;
    std::string offset = str1.str();
    std::string sysCmd = EraseCommand + block_size + COUNT_STRING + count + SEEK_STRING + offset;
    ETG_TRACE_FATAL(("EraseFac:sysCmd:%s", sysCmd.c_str()));
    execCommand(sysCmd.c_str());
    // sleep(120); //currently due to platform issues ,we need to wait for 1 min
}

/*
* FUNCTION:   const bool swu_HyperFlashBootChain::activateRecoveryMode
*
* DESCRIPTION: activate RecoveryMode
*
*
* PARAMETER INPUT:
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

const bool swu_HyperFlashBootChain::activateRecoveryMode()
{
    print2errmem("swu_HyperFlashBootChain::activateRecoveryMode executed\n");
    return bWriteMagicToRawFlash(DL_UPDATE_STATUS_FLAG_OFFSET_HYPERFLASH, DL_UPDATE_STATE_RECOVERY_HYPERFLASH);
}

/*
* FUNCTION:   const bool swu_HyperFlashBootChain::activateApplicationMode
*
* DESCRIPTION: activate ApplicationMode
*
*
* PARAMETER INPUT:
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

const bool swu_HyperFlashBootChain::activateApplicationMode()
{
    print2errmem("swu_HyperFlashBootChain::activateApplicationMode executed\n");
    return bWriteMagicToRawFlash(DL_UPDATE_STATUS_FLAG_OFFSET_HYPERFLASH, DL_UPDATE_STATE_SUCCESS_HYPERFLASH);
}

/*
* FUNCTION:   const bool swu_HyperFlashBootChain::switchBootChainOne
*
* DESCRIPTION: switch BootChainOne
*
*
* PARAMETER INPUT:
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

const bool swu_HyperFlashBootChain::switchBootChainOne()
{
    print2errmem("swu_HyperFlashBootChain::switchBootChainOne executed\n");
    return bWriteMagicToRawFlash(DL_ROOTFS_FLAG_OFFSET_HYPERFLASH, DL_ROOTFS_1_HYPERFLASH);
}

/*
* FUNCTION:   const bool swu_HyperFlashBootChain::switchBootChainTwo
*
* DESCRIPTION: switch BootChainTwo
*
*
* PARAMETER INPUT:
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

const bool swu_HyperFlashBootChain::switchBootChainTwo()
{
    print2errmem("swu_HyperFlashBootChain::switchBootChainTwo executed\n");
    return bWriteMagicToRawFlash(DL_ROOTFS_FLAG_OFFSET_HYPERFLASH, DL_ROOTFS_2_HYPERFLASH);
}

/*
* FUNCTION:   bool swu_HyperFlashBootChain::isRecoveryMagicSet
*
* DESCRIPTION: Returns if the recoverydnl magic is selected or not
*
*
* PARAMETER INPUT:  isSet
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

bool swu_HyperFlashBootChain::isRecoveryMagicSet(bool &isSet)
{
    int MagicAreaSelected = 0;
    tU32  au32MagicArea1[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};
    tU32  au32MagicArea2[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};
    areaSelected(MagicAreaSelected);
    if(MagicAreaSelected == 1)
    {
        readFromFac(DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH,au32MagicArea1);
        if(au32MagicArea1[8] == DL_UPDATE_STATE_RECOVERY_HYPERFLASH)
        {
            isSet = true;
        }
        else
        {
            isSet = false;
        }
    }
    else
    {
        readFromFac(DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH,au32MagicArea2);
        if(au32MagicArea2[8] == DL_UPDATE_STATE_RECOVERY_HYPERFLASH)
        {
            isSet = true;
        }
        else
        {
            isSet = false;
        }
    }

    ETG_TRACE_FATAL(("isRecoveryMagicSet is %d",isSet));

    return true;
}

/*
* FUNCTION:   void swu_HyperFlashBootChain::usedBootChain
*
* DESCRIPTION: Returns which boot chain is selected
*
*
* PARAMETER INPUT:  chain
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

bool swu_HyperFlashBootChain::usedBootChain(int &chain)
{
    int MagicAreaSelected = 0;
    areaSelected(MagicAreaSelected);
    if(MagicAreaSelected == 1)
    {
        chain = 1;
    }
    else{
        chain = 2;
    }

    ETG_TRACE_FATAL(("usedBootChain is %d",chain));

    return true;

}

/*
* FUNCTION:   bool swu_HyperFlashBootChain::areaSelected
*
* DESCRIPTION: Returns which magic area is selected
*
*
* PARAMETER INPUT:  MagicAreaSelected
*
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      bool
*
*/

bool swu_HyperFlashBootChain::areaSelected(int &MagicAreaSelected)
{
    int retcode1 =0;
    int retcode2 =0;
    off_t u32FlashAdress;
    tU32  au32MagicArea1[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};
    tU32  au32MagicArea2[DNL_MAGIC_ARRAY_SIZE_HYPERFLASH] = {0};

    // Read magic area 1 from the flash
    u32FlashAdress = DNL_MAGIC1_BASE_ADDRESS_HYPERFLASH;
    retcode1 = readFromFac(u32FlashAdress,au32MagicArea1);
    ETG_TRACE_USR3(("swu_HyperFlashBootChain::areaSelected:retcode1:%d", retcode1));

    // Read magic area 2 from the flash
    u32FlashAdress = DNL_MAGIC2_BASE_ADDRESS_HYPERFLASH;
    retcode2 = readFromFac(u32FlashAdress,au32MagicArea2);
    ETG_TRACE_USR3(("swu_HyperFlashBootChain::areaSelected:retcode2:%d", retcode2));


    // Check both magic areas
    if( (retcode1 != DNL_MAGIC_AREA_SIZE_HYPERFLASH) && (retcode2 != DNL_MAGIC_AREA_SIZE_HYPERFLASH) ) {
        ETG_TRACE_USR4 (("DNL: bReadMagicAreas: read flash device ERROR"));        
    }

    // Check which magic area is valid
    if ( (au32MagicArea1[0] == DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (au32MagicArea2[0] != DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        MagicAreaSelected = 1; // only area 1 is valid
    }

    if ( (au32MagicArea1[0] != DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (au32MagicArea2[0] == DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        MagicAreaSelected = 2; // only area 2 is valid
    }

    if ( (au32MagicArea1[0]== DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (au32MagicArea2[0] == DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        // both areas are valid
        // Check also the counters; magic with higher counter is selected
        if ( au32MagicArea1[4] >= au32MagicArea2[4] )
        {
            MagicAreaSelected = 1;
        }
        else
        {
            MagicAreaSelected = 2;
        }
    }

    if ( (au32MagicArea1[0] != DL_MAGIC_SECTOR_VALID_HYPERFLASH) && (au32MagicArea2[0] != DL_MAGIC_SECTOR_VALID_HYPERFLASH))
    {
        MagicAreaSelected = 0; // both are invalid  //@todo: check how this error case is handled in calling functions
    }

    ETG_TRACE_FATAL(("MagicAreaSelected is %d",MagicAreaSelected));

    return true;
}

} // namespace swu
