/************************************************************************
 * File: teseo_SWUpdater.cpp
 * SW-Component: 
 * Description: Receives teseo SW update related commands from sw update manager and
 *				processes accordingly. Mainly implements the teseo sw update by
 *				using linux GNSS proxy. Also supports teseo version query.
 *              
 * Author:
 *	Rohit.ChannappiahRaman@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 14.10.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
 * 27.08.2014 - Implemented CMG3G-5318 optimized logging using update.log - Apoorva K R (RBEI/ECA1)
 * 11.02.2015 - Removed GNSS being accessed in read only mode in bFetchTeseoVersion- Apoorva K R (RBEI/ECA1)
 * 6/4/2015 - Implemented CMG3G-6728 IS2030_Teseo update only if CRC changed@Integration Tooling - Apoorva K.R (RBEI/ECA1)
 * 2/7/2015 - Fixed CMG3GB-2169 - Segmentation fault when calculating CRC for teseo firmware - Apoorva K.R (RBEI/ECA1)
 *					CMG3GB-2167 - Exit codes for CRC calculation are treated as error codes while writing to errmem
 * 1/1/2016 - Implemented CMG3G-9339 Teseo-3 support in SWUpdate - Anusha Ghali (RBEI/ECA1)
*---------------------------------------------------------------------------------------------------
* 1/2/2016          | AGL5KOR (RBEI)    |     CMG3G-9450 - teseo update ctrl app to be modified to write and read CRC
* 				    | Anusha Ghali	    | 
*---------------------------------------------------------------------------------------------------
 ***********************************************************************/

#include "pthread.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_TESEO
  #include "trcGenProj/Header/teseo_SWUpdater.cpp.trc.h"
#endif

#include "teseo_SWUpdateMgr.h"
#include "teseo_SWBinaryHandler.h"
#include "teseo_SWUpdater.h"
#include "teseo_Global.h"

#define GNSS_DEVICE_STRING "/dev/gnss"
#define NUM_OF_CHARS_IN_U32 10

/************************************************************************
* FUNCTION: teseo_SWUpdater()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_SWUpdater::teseo_SWUpdater()
{
	m_gnssFd=0;
	m_strFlasherBinPath="";
	m_strTeseoFWBinPath="";
	m_u32FlashedFwCRC = 0;
	m_u32NewFwCRC = 0;
}


/************************************************************************
* FUNCTION: ~teseo_SWUpdater
* DESCRIPTION: Destruction
* PARAMETER: None
* RETURNVALUE: None
*************************************************************************/
teseo_SWUpdater::~teseo_SWUpdater()
{
	if(m_gnssFd)
	{	
		if ( OSAL_s32IOClose( m_gnssFd ) == OSAL_ERROR )
		{
			DEBUG_TRACE(TRACE_LEVEL_1, "WARN: ~teseo_SWUpdater - error in closing %s", GNSS_DEVICE_STRING);
		}
	}	
}

/************************************************************************
* FUNCTION: bInitSWUpdater
* DESCRIPTION: Creates worker thread 
* PARAMETER: None*
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bInitSWUpdater(std::string &strFlasherPath,
									  std::string &strTeseoFWBinPath)
{
	m_strFlasherBinPath = strFlasherPath;
	m_strTeseoFWBinPath = strTeseoFWBinPath;

	pthread_t rTeseoSWUpdaterThread;	
	if(pthread_create\
		(&rTeseoSWUpdaterThread, NULL,\
			&teseo_SWUpdater::vTeseoSWUpdaterThreadCallback, this))
	{
		ETG_TRACE_USR4 (("ERROR: failed to create updater thread"));
		UpdateLog::vUpdateLogTrace ("%s failed to create updater thread", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: failed to create updater thread");
		return false;
	}
	return true;
}

/************************************************************************
* FUNCTION: u8CompareCRC
* DESCRIPTION: This function compares the CRC of the new and flashed firmware
*				
* PARAMETER: std::string &strTeseoFWBinPath - Path to new teseo f/w image 
* RETURNVALUE: tU8
*************************************************************************/	
tU8 teseo_SWUpdater::u8CompareCRC (std::string &strTeseoFWBinPath)
{
	tU8 u8Res = TESEO_SWUPDATE_CRC_CALC_ERROR;
	tU8 u8CRCFromGNSSExitCode = 0;
	tU8 u8CRCCalcExitCode = 0;
	bool bFlasherCrc = false;
	
	m_strTeseoFWBinPath = strTeseoFWBinPath;
	u8CalcCRCForFwUpdate (u8CRCCalcExitCode,bFlasherCrc);
	
	if (TESEO_SWUPDATE_CRC_CALC_OK != u8CRCCalcExitCode )
	{
		UpdateLog::vUpdateLogTrace ("%s Error while calculating CRC for new fw!!!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_2, "ERROR: Error while calculating CRC for new fw");
		u8Res = u8CRCCalcExitCode;
		return u8Res;
	}
	
	ETG_TRACE_USR4 (("INFO: CRC calc of new fw done. Getting CRC of flashed fw"));
							
	u8GetCRCFromGNSS (u8CRCFromGNSSExitCode);
	
	if (TESEO_SWUPDATE_CRC_CALC_OK != u8CRCFromGNSSExitCode)
	{
		UpdateLog::vUpdateLogTrace ("%s Error while getting flashed fw CRC from GNSS!!!", DL_LOG_ERROR);
		
		DEBUG_TRACE(TRACE_LEVEL_2, "ERROR: Error while getting flashed fw CRC from GNSS!!!");
			
		u8Res = u8CRCFromGNSSExitCode;
		return u8Res;
	}
	
	if (m_u32NewFwCRC != m_u32FlashedFwCRC)
	{
		UpdateLog::vUpdateLogTrace ("%s CRC values of new and flashed images are NOT same! update needed!!!", DL_LOG_INFO);
			
		ETG_TRACE_USR4 (("INFO: CRC values of new and flashed images are NOT same! update needed!!!"));
							
		DEBUG_TRACE (TRACE_LEVEL_2, "INFO: CRC values of new and flashed images are NOT same! update needed");
					
		u8Res = TESEO_SWUPDATE_FW_CRCS_DIFFERENT;
	}
	else
	{
		UpdateLog::vUpdateLogTrace ("%s CRC values of new and flashed images are same! no update needed!!!", DL_LOG_INFO);
				
		ETG_TRACE_USR4 (("INFO: CRC values of new and flashed images are same! no update needed!!!"));
							
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: CRC values of new and flashed images are same! no update needed!");
				
		u8Res = TESEO_SWUPDATE_FW_CRCS_SAME;
	}	

	return u8Res; 
}


/************************************************************************
* FUNCTION: teseo_print2stdout
*
* DESCRIPTION: print to stdout
*
* PARAMETER: 
*
* RETURNVALUE: None
*************************************************************************/
void teseo_SWUpdater::teseo_print2stdout(const char *Format, ...)
{
  int     fd=dup(STDOUT_FILENO);
  //Coverity fix for 15979
  if(-1 == fd)
  {
	fprintf(stderr,"Dup is failed in teseo_SWUpdater.cpp file\n");
	return;
  }
  char    msg[1024];
  va_list arglst;

  va_start(arglst, Format);
  vsnprintf(msg, sizeof(msg), Format, arglst);
  va_end(arglst);
  write(fd, msg, strlen(msg));
  close(fd);
}

/************************************************************************
* FUNCTION: vTeseoSWUpdaterThreadFunction
* DESCRIPTION:The worker thread waits on events relating to teseo sw update 
*			from teseo sw update Manager and processes those events accordingly.
*				
* PARAMETER: None*
* RETURNVALUE: None
*************************************************************************/	
tVoid teseo_SWUpdater::vTeseoSWUpdaterThreadFunction()
{
	while(true)
	{
		OSAL_tEventMask u32ResultMask = 0;
		ETG_TRACE_USR4 (("INFO: SWUpdater waiting for Manager events"));
		UpdateLog::vUpdateLogTrace ("%s SWUpdater waiting for Manager events", DL_LOG_INFO);
		DEBUG_TRACE(TRACE_LEVEL_2,"INFO: SWUpdater waiting for Manager events");

		//wait for events from DL client and response receiver
		if ( OSAL_ERROR == OSAL_s32EventWait\
							( teseo_SWUpdateMgr::m_evHandleSWUpdater,
							 (EV_TESEO_SWUPDATE_MGR),
							 OSAL_EN_EVENTMASK_OR,
							 OSAL_C_TIMEOUT_FOREVER,
							 &u32ResultMask ))
		{
			ETG_TRACE_USR4 (("ERROR: teseo_SWUpdateMgr::vSWUpdateMgrThreadFunction OSAL_s32EventWait error"));
			DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: OSAL_s32EventWait failed in %s %d",__FILE__,__LINE__);  
			break;
		}
		/* Event wait returned with a sucess, Clear the event */
		if(OSAL_ERROR == OSAL_s32EventPost\
			( teseo_SWUpdateMgr::m_evHandleSWUpdater,
				(OSAL_tEventMask) ~u32ResultMask, OSAL_EN_EVENTMASK_AND))
		{
		   ETG_TRACE_USR4 (("ERROR: teseo_SWUpdateMgr::vSWUpdateMgrThreadFunction OSAL_s32EventPost error"));
		   DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: OSAL_s32EventPost failed in %s %d",__FILE__,__LINE__);
		   break;
		}
		
		/* Check event */
		if (( u32ResultMask == EV_TESEO_SWUPDATE_MGR ))
		{
			vProcessEvent();
		}
	}	
}

/************************************************************************
* FUNCTION: bWriteCRCToFile
* DESCRIPTION: This function writes the CRC to file in the specified path
*				
* PARAMETER: const std::string& strPathToFile - Path to file 
			 tU32& u32Crc32	- CRC to be written to file
* RETURNVALUE: tBool
*************************************************************************/	
tBool teseo_SWUpdater::bWriteCRCToFile (const std::string& strPathToFile,\
							tU32& u32Crc32)
{
	FILE *pFileToWriteCRC;
	char* cBuff = new char [9];
	if (NULL == cBuff)
		return false;
	
	memset (cBuff, 0, 9);
	
	pFileToWriteCRC = fopen (strPathToFile.c_str () ,"w+");
	if (NULL == pFileToWriteCRC)  
	{
		UpdateLog::vUpdateLogTrace ("%s fileopen failed in bWriteCRCToFile", DL_LOG_ERROR);
		
		DEBUG_TRACE( TRACE_LEVEL_2, "ERROR: failed to open %s!!", strPathToFile.c_str ());
						
		DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Please make sure to have write access to the folder!");
						
		delete[] cBuff;
		return false;
	}	
	
	else
	{
		UpdateLog::vUpdateLogTrace ("%s writing crc to file...", DL_LOG_INFO);
		sprintf (cBuff, "%08x",(unsigned int) u32Crc32);
		fputs (cBuff, pFileToWriteCRC);
		fflush(pFileToWriteCRC);  //Coverity fix for 127027
		if (EOF == fclose (pFileToWriteCRC) )
		{
			DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: problem with fclose");
			delete[] cBuff;
			return false;
		}
		//fflush(pFileToWriteCRC);
	}
	
	
	delete[] cBuff;
	return true;
}

/************************************************************************
* FUNCTION: u8CalcCRCForFwUpdate
* DESCRIPTION: This function calculates CRC of the firmware to be flashed
*				
* PARAMETER: const std::string& strPathToFile - Path to file to write CRC
			 std::string& strTeseoFWBinPath	- Path to new teseo fw
			 tU8& u8CRCCalcExitCode - exit code of the function
* RETURNVALUE: tVoid
*************************************************************************/	
tU8 teseo_SWUpdater::u8CalcCRCForFwUpdate (tU8& u8CRCCalcExitCode, \
				bool bFlasherCrc,\
				const std::string& strTeseoFWBinPath, \
				const std::string& strPathToFile)
{
	ETG_TRACE_USR4 (("INFO: bCalcCRCForFwUpdate entered"));
	u8CRCCalcExitCode = TESEO_SWUPDATE_CRC_CALC_OK;
	
	if (bFlasherCrc)
	{
		if (m_strFlasherBinPath.empty ())
		{
			m_strFlasherBinPath = strTeseoFWBinPath;
			m_strTeseoFWBinPath = "";
		}
	}
	else
	{
		if (m_strTeseoFWBinPath.empty ())
		{
			m_strTeseoFWBinPath = strTeseoFWBinPath;
			m_strFlasherBinPath = "";
		}
	}
		
	UpdateLog::vUpdateLogTrace ("%s teseo_SWUpdater::bCalcCRCForFwUpdate opened file %s", DL_LOG_INFO, strTeseoFWBinPath.c_str ());

	DEBUG_TRACE (TRACE_LEVEL_2, "INFO: teseo_SWUpdater::bCalcCRCForFwUpdate opened file %s", strTeseoFWBinPath.c_str ());
	
	// Calculate CRC of the binary - firmware or bootloader
	u8CalcCRCForBin(u8CRCCalcExitCode);
	
	if (!strPathToFile.empty ())
	{
		ETG_TRACE_USR4 (("INFO: Writing CRC of new fw to file %s", strPathToFile.c_str()));
		DEBUG_TRACE (TRACE_LEVEL_2, "INFO: Writing CRC of new fw to file %s", strPathToFile.c_str());	
		
		if (false == bWriteCRCToFile (strPathToFile, m_u32NewFwCRC))
		{
			DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Error while writing CRC to %s", strPathToFile.c_str());	
			ETG_TRACE_USR4 (("ERROR: Error while writing CRC to %s", strPathToFile.c_str()));				
				
			u8CRCCalcExitCode = TESEO_SWUPDATE_CRC_CALC_ERROR;
			return u8CRCCalcExitCode;
		}
	}
	return u8CRCCalcExitCode;
}

/************************************************************************
* FUNCTION: u8CalcCRCForBin
* DESCRIPTION: This function calculates CRC of the firmware and bootloader to be flashed
*				
* PARAMETER: tVoid
* RETURNVALUE: tU8
*************************************************************************/	
tU8 teseo_SWUpdater::u8CalcCRCForBin(tU8& u8CRCCalcExitCode)
{
    if ( !m_strFlasherBinPath.empty() )
	{
		DEBUG_TRACE (TRACE_LEVEL_2, "INFO: Found flasher file for CRC calculation");
	    if(false == m_BinaryHandler.bOpenFile(m_strFlasherBinPath))
	    {
		   bPostStatusEvent(TESEO_SWUPDATE_BL_OPEN_ERROR,\
		   EV_TESEO_SWUPDATE_STATUS);
		   return false;
	    }
		if (false == m_BinaryHandler.bCalcCRC32(m_u32NewFwCRC,m_strFlasherBinPath))
		{
		   UpdateLog::vUpdateLogTrace ("%s Crc calculation failed for new teseo bootloader!!!", DL_LOG_ERROR);
		   ETG_TRACE_USR4 (("ERROR: Crc calculation failed for new teseo bootloader!"));
		   DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Crc calculation failed for new teseo bootloader!");
		   u8CRCCalcExitCode = TESEO_SWUPDATE_CRC_CALC_ERROR;
		   return u8CRCCalcExitCode;
		}
		
	}
	else if (false == m_BinaryHandler.bCalcCRC32ForFixedSize(m_u32NewFwCRC,\
					m_strTeseoFWBinPath))
	{
		UpdateLog::vUpdateLogTrace ("%s Crc calculation failed for new teseo fw!!!", DL_LOG_ERROR);
		ETG_TRACE_USR4 (("ERROR: Crc calculation failed for new teseo fw!"));
		DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Crc calculation failed for new teseo fw!");
		u8CRCCalcExitCode = TESEO_SWUPDATE_CRC_CALC_ERROR;
		return u8CRCCalcExitCode;
	}
	
	return u8CRCCalcExitCode;
}
/************************************************************************
* FUNCTION: u8GetCRCFromGNSS
* DESCRIPTION: This function gets CRC of the firmware that is already flashed
*				in the chip using dev_gnss interface
* PARAMETER: const std::string& strPathToFile - Path to file 
			 tU8& u8CRCFromGNSSExitCode - exit code of the function
			 
* RETURNVALUE: tVoid
*************************************************************************/	
tU8 teseo_SWUpdater::u8GetCRCFromGNSS \
		(tU8& u8CRCFromGNSSExitCode,const std::string& strPathToFile)
{

	u8CRCFromGNSSExitCode = TESEO_SWUPDATE_CRC_CALC_OK;
	m_gnssFd = OSAL_IOOpen( GNSS_DEVICE_STRING, OSAL_EN_WRITEONLY); 
	if(OSAL_ERROR == m_gnssFd)
	{
		UpdateLog::vUpdateLogTrace ("%s OSAL_IOOpen error while getting CRC of flashed FW from GNSS!!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_2, "OSAL_IOOpen error while getting CRC of flashed FW from GNSS!!");
		u8CRCFromGNSSExitCode = TESEO_SWUPDATE_GNSS_OPEN_ERROR;
		return u8CRCFromGNSSExitCode;
	}
	
	if(OSAL_ERROR == OSAL_s32IOControl\
		( m_gnssFd, OSAL_C_S32_IOCTL_GNSS_GET_GNSS_CHIP_CRC, \
			(intptr_t)&m_u32FlashedFwCRC) )
	{
		UpdateLog::vUpdateLogTrace ("%s OSAL_s32IOControl error while getting CRC of flashed FW from GNSS. Proceeding with update!!!", DL_LOG_ERROR);
		
		ETG_TRACE_USR4 (("INFO: OSAL IOctl Error while getting CRC of flashed FW from GNSS! Proceeding with update"));
		
		DEBUG_TRACE(TRACE_LEVEL_2, "ERROR: OSAL_s32IOControl error while getting CRC of flashed FW from GNSS!!");
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Proceeding with update since ioctl to GNSS failed");
	}

	if (!strPathToFile.empty())
	{
		ETG_TRACE_USR4 (("INFO: Writing CRC of flashed FW to %s", strPathToFile.c_str()));
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Writing CRC of flashed FW to %s", strPathToFile.c_str());	
			
		if (false == bWriteCRCToFile (strPathToFile, m_u32FlashedFwCRC))
		{
			DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Error while writing CRC of flashed FW to %s", strPathToFile.c_str());	
			UpdateLog::vUpdateLogTrace ("%s Error while writing CRC of flashed FW to %s!!", DL_LOG_ERROR, strPathToFile.c_str());
		}
	}		
	
	if ( OSAL_ERROR == OSAL_s32IOClose ( m_gnssFd ) )
	{
		UpdateLog::vUpdateLogTrace ("%s OSAL_s32IOClose error while getting CRC of flashed FW from GNSS!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_2, "ERROR: OSAL_s32IOClose error while getting CRC of flashed FW from GNSS!!!");
		u8CRCFromGNSSExitCode = TESEO_SWUPDATE_GNSS_CLOSE_ERROR;
		return u8CRCFromGNSSExitCode;
	}	
	
	m_gnssFd=0;
	
	return u8CRCFromGNSSExitCode;
}

/************************************************************************
* FUNCTION: vProcessEvent
* DESCRIPTION: process events from manager
* PARAMETER: None*
* RETURNVALUE: void
*************************************************************************/
tVoid teseo_SWUpdater::vProcessEvent()
{
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: teseo_SWUpdater event RXed is %d", teseo_SWUpdateMgr::m_evMgrToSWUpdater);
	switch(teseo_SWUpdateMgr::m_evMgrToSWUpdater)
	{
		case EV_TESEO_SWUPDATE_START:
		UpdateLog::vUpdateLogTrace("%s teseo_SWUpdater evnt RXed is EV_TESEO_SWUPDATE_START", DL_LOG_INFO);
		DEBUG_TRACE (TRACE_LEVEL_2,"INFO: teseo_SWUpdater evnt RXed is EV_TESEO_SWUPDATE_START");
		vTeseoSWUpdate();
		break;
		
		case EV_TESEO_SWUPDATE_ABORT:
		//TBD
		UpdateLog::vUpdateLogTrace ("%s teseo_SWUpdater evnt RXed is EV_TESEO_SWUPDATE_ABORT", DL_LOG_INFO);
		teseo_SWUpdateMgr::m_SWUpdateStatus = TESEO_SWUPDATE_ABORTED;
		pthread_exit(0);
		break;

		case EV_TESEO_SWUPDATE_EXIT:
		//exit
		pthread_exit(0);	
		break;
		
		default:
		ETG_TRACE_USR4 (("WARN: teseo_SWUpdater::vProcessEvent Invalid event"));
		DEBUG_TRACE(TRACE_LEVEL_1,"WARN: teseo_SWUpdater event RXed is invalid");
		break;	
	}
}

/************************************************************************
* FUNCTION: vTeseoSWUpdate
* DESCRIPTION: Wrapper for Actual teseo SW update 
* PARAMETER: None
* RETURNVALUE: void
*************************************************************************/
tVoid teseo_SWUpdater::vTeseoSWUpdate()
{
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: Opening GNSS device !!!!!!");
	UpdateLog::vUpdateLogTrace ("%s Opening GNSS device", DL_LOG_INFO);
    m_gnssFd = OSAL_IOOpen( GNSS_DEVICE_STRING, OSAL_EN_WRITEONLY); 
	if(OSAL_ERROR != m_gnssFd)
	{
		vUpdateStatus(TESEO_SWUPDATE_STARTED, EV_TESEO_SWUPDATE_STATUS);

		// Actual teseo update using GNSS proxy drivers
		if (false == vTeseoSWUpdateSequence())
			return;
			
		/*Close GNSS Proxy*/
		UpdateLog::vUpdateLogTrace ("%s  closing GNSS device !!!!!", DL_LOG_INFO);
		DEBUG_TRACE(TRACE_LEVEL_1,"INFO: closing GNSS device !!!!!");
		if ( OSAL_s32IOClose( m_gnssFd ) == OSAL_ERROR )
		{
			bPostStatusEvent(TESEO_SWUPDATE_GNSS_CLOSE_ERROR, \
				EV_TESEO_SWUPDATE_STATUS);
			UpdateLog::vUpdateLogTrace ("%s  closing GNSS device failed !!!!!", DL_LOG_ERROR);
			DEBUG_TRACE(TRACE_LEVEL_1,"ERROR: closing GNSS device failed !!!!!");
		}	
		m_gnssFd = 0;
		//notify manager of successful teseo sw update	
		bPostStatusEvent(TESEO_SWUPDATE_SUCCESS, EV_TESEO_SWUPDATE_STATUS);
	}
	else
	{
		UpdateLog::vUpdateLogTrace ("%s Opening GNSS device failed!!!!!!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1,"ERROR: Opening GNSS device failed!!!!!!");
		bPostStatusEvent(TESEO_SWUPDATE_GNSS_OPEN_ERROR, EV_TESEO_SWUPDATE_STATUS);
	}	
}

/************************************************************************
* FUNCTION: vTeseoSWUpdateSequence
* DESCRIPTION: Actual teseo SW update using linux GNSS proxy driver
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::vTeseoSWUpdateSequence()
{
	// Notify teseo chip type to GNSS driver
	DEBUG_TRACE (TRACE_LEVEL_1,"INFO: Notifying teseo chip type (%s) to GNSS- start",strTeseoVersionToGNSS.c_str());
	UpdateLog::vUpdateLogTrace ("%s Notifying teseo chip type (%s) to GNSS- start", DL_LOG_INFO,strTeseoVersionToGNSS.c_str() );
	if(false == bNotifyTeseoChipTypeToGNSS())
		return false;
	UpdateLog::vUpdateLogTrace ("%s Notifying teseo chip type (%s) to GNSS - OK", DL_LOG_INFO,strTeseoVersionToGNSS.c_str());
	DEBUG_TRACE(TRACE_LEVEL_1,"INFO: Notifying teseo chip type (%s) to GNSS - OK",strTeseoVersionToGNSS.c_str());

	//Send Teseo Boot loader - flasher.bin
	DEBUG_TRACE (TRACE_LEVEL_1,"INFO: Flashing Teseo BootLoader - start");
	UpdateLog::vUpdateLogTrace ("%s Flashing Teseo BootLoader - start", DL_LOG_INFO);
	if(false == bSendTeseoBootLoader())
		return false;
	UpdateLog::vUpdateLogTrace ("%s Flashing Teseo BootLoader - OK", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_1,"INFO: Flashing Teseo BootLoader - OK");
		
	//Send Teseo firmware
	UpdateLog::vUpdateLogTrace ("%s Flashing Teseo firmware - start", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_1,"INFO: Flashing Teseo firmware - start");
	if(false == bSendTeseoFirmware())
		return false;
	UpdateLog::vUpdateLogTrace ("%s Flashing Teseo firmware - OK", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_1,"INFO: Flashing Teseo firmware - OK");
	return true;	
}
/************************************************************************
* FUNCTION: bSendTeseoFirmware
* DESCRIPTION: Send teseo firmware binary using linux GNSS proxy driver.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bSendTeseoFirmware()
{
	//send teseo FW size and CRC
	if(false == m_BinaryHandler.bOpenFile(m_strTeseoFWBinPath))
	{
		UpdateLog::vUpdateLogTrace ("%s Opening file %s failed!!!!!!", DL_LOG_ERROR, m_strTeseoFWBinPath.c_str());
		DEBUG_TRACE(TRACE_LEVEL_1,"ERROR: Opening file %s failed", m_strTeseoFWBinPath.c_str());
		ETG_TRACE_USR4 (("ERROR: Opening file %s failed", m_strTeseoFWBinPath.c_str()));
			
		bPostStatusEvent(TESEO_SWUPDATE_FW_OPEN_ERROR, EV_TESEO_SWUPDATE_STATUS);
		return false;
	}
	UpdateLog::vUpdateLogTrace("%s opened file %s with size %d bytes", DL_LOG_INFO, m_strTeseoFWBinPath.c_str(), m_BinaryHandler.s32GetFileSize());
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: opened file %s with size %d bytes", m_strTeseoFWBinPath.c_str(), m_BinaryHandler.s32GetFileSize());

	if(false == bSendFWBinSizeAndCRC())
	{
		UpdateLog::vUpdateLogTrace ("%s Sending teseo firmware size and CRC failed!!!!!!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: Sending teseo firmware size and CRC failed");
		ETG_TRACE_USR4 (("ERROR: Sending teseo firmware size and CRC failed"));
		vCloseFile();
		return false;
	}

	//flash teseo firmware
	vUpdateStatus (TESEO_SWUPDATE_FLASHING_FIRMWARE, EV_TESEO_SWUPDATE_STATUS);
	if(false == bWriteTeseoSWBytes())
	{
		vCloseFile();
		return false;
	}
	
	//close flasher.bin
	if(false == vCloseFile())
	{
		bPostStatusEvent(TESEO_FILE_CLOSE_ERROR, EV_TESEO_SWUPDATE_STATUS);
		return false; 
	}
	
	UpdateLog::vUpdateLogTrace ("%s teseo_SWUpdater::bSendTeseoFirmware - OK", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: teseo_SWUpdater::bSendTeseoFirmware - OK");
	return true;
}

/************************************************************************
* FUNCTION: bSendTeseoBootLoader
* DESCRIPTION: Send teseo bootloader binary using linux GNSS proxy driver.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bSendTeseoBootLoader()
{
	//Load Teseo Boot loader - flasher.bin
	vUpdateStatus(TESEO_SWUPDATE_FLASHING_FLASHER_BIN,\
		EV_TESEO_SWUPDATE_STATUS);	
	if(false == m_BinaryHandler.bOpenFile(m_strFlasherBinPath))
	{
		bPostStatusEvent(TESEO_SWUPDATE_BL_OPEN_ERROR,\
			EV_TESEO_SWUPDATE_STATUS);
		return false;
	}

	if(false == bSendBootloaderSizeAndCRC())
	{
		UpdateLog::vUpdateLogTrace ("%s Sending teseo bootloader size and CRC failed!!!!!!", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: Sending teseo bootloader size and CRC failed");
		ETG_TRACE_USR4 (("ERROR: Sending teseo bootloader size and CRC failed"));
		vCloseFile();
		return false;
	}
	
	//Send flasher.bin			
	if(false == bWriteTeseoSWBytes())
	{	
		vCloseFile();
		return false;
	}
	
	UpdateLog::vUpdateLogTrace ("%s  closing Teseo Boot loader", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: closing Teseo Boot loader");
	//close flasher.bin
	if(false == vCloseFile())
	{
		bPostStatusEvent(TESEO_FILE_CLOSE_ERROR, EV_TESEO_SWUPDATE_STATUS);
		return false; 
	}
	
	return true;
}

/************************************************************************
* FUNCTION: vCloseFile
* DESCRIPTION: Send teseo firmware image options - size,CRC using linux GNSS proxy.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::vCloseFile() 
{
	if(false == m_BinaryHandler.bCloseFile())
	{
		UpdateLog::vUpdateLogTrace ("%s teseo_SWUpdater::vCloseFile failed", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1,"ERROR: teseo_SWUpdater::vCloseFile failed");
		ETG_TRACE_USR4 (("ERROR: teseo_SWUpdater::vCloseFile failed"));
		return false; 
	}
	return true;
}

/************************************************************************
* FUNCTION: bSendBootloaderSizeAndCRC
* DESCRIPTION: Send teseo bootloader image options - size,CRC using linux GNSS proxy.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bSendBootloaderSizeAndCRC()
{
	tU32 u32Crc=0;
	
	//Configuration to send flasher.bin
	trImageOptions imgOpts;
	imgOpts.enType = IMAGE_TESEO_FLASHER_BIN;
	imgOpts.u32Size = m_BinaryHandler.s32GetFileSize();
	
	UpdateLog::vUpdateLogTrace ("%s Opened Teseo Boot loader file ", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: Opened Teseo Boot loader file %s ", m_BinaryHandler.s32GetFileName().c_str());
	UpdateLog::vUpdateLogTrace ("%s  teseo Boot Loader file size - %d bytes", DL_LOG_INFO, imgOpts.u32Size);
	DEBUG_TRACE(TRACE_LEVEL_1,"INFO: teseo Boot Loader file size - %d bytes",(int) imgOpts.u32Size);
	
	if(false == m_BinaryHandler.bCalcCRC32(u32Crc,m_strFlasherBinPath))
	{
		bPostStatusEvent(TESEO_SWUPDATE_CRC_ERROR, EV_TESEO_SWUPDATE_STATUS);
		return false;
	}
	imgOpts.u32ChkSum = u32Crc;	
	UpdateLog::vUpdateLogTrace ("%s INFO: Bootloader CRC calculation - OK", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: Bootloader CRC calculation - OK");
	
	//ioctl to configure sending of teseo fw
	if(OSAL_ERROR == OSAL_s32IOControl\
		( m_gnssFd, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (intptr_t)&imgOpts) )
	{
		bPostStatusEvent(TESEO_SWUPDATE_IOCTL_ERROR, \
			EV_TESEO_SWUPDATE_STATUS);
		vCloseFile();
		return false;
	}
	return true;
}

/************************************************************************
* FUNCTION: bSendFWBinSizeAndCRC
* DESCRIPTION: Send teseo firmware image options - size,CRC using linux GNSS proxy.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bSendFWBinSizeAndCRC()
{
	tU32 u32Crc=0;
	std::string strCurFileName = ""; // must be empty to differentiate between firmware and flasher CRC calculation
	
	//Configuration to send teseo fw
	trImageOptions imgOpts;
	imgOpts.enType = IMAGE_TESEO_FIRMWARE;
	imgOpts.u32Size = m_BinaryHandler.s32GetFileSize();
	
	UpdateLog::vUpdateLogTrace ("%s CRC calculation for firmware - Start", DL_LOG_INFO);
	DEBUG_TRACE (TRACE_LEVEL_2,"INFO: CRC calculation for firmware - Start");
	
	if(false == m_BinaryHandler.bCalcCRC32(u32Crc,strCurFileName))
	{
		bPostStatusEvent(TESEO_SWUPDATE_CRC_ERROR, EV_TESEO_SWUPDATE_STATUS);
		return false;
	}
	imgOpts.u32ChkSum = u32Crc;	
	UpdateLog::vUpdateLogTrace ("%s INFO: CRC calculation - OK", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2,"INFO: CRC calculation - OK");
	
	//ioctl to configure sending of teseo fw
	if(OSAL_ERROR == OSAL_s32IOControl\
		( m_gnssFd, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (intptr_t)&imgOpts) )
	{
		bPostStatusEvent(TESEO_SWUPDATE_IOCTL_ERROR, EV_TESEO_SWUPDATE_STATUS);
		UpdateLog::vUpdateLogTrace ("%s INFO: IO Control error in bSendFWBinSizeAndCRC", DL_LOG_ERROR);
		return false;
	}
	return true;
}

/************************************************************************
* FUNCTION: bWriteTeseoSWBytes
* DESCRIPTION: interface to flash teseo binary.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bWriteTeseoSWBytes()
{
	//Read the file in chunks of 16KB/5KB and transfer it to the GNSS proxy
	tU8* pu8Buf = new tU8[u32gTeseo_Fw_Chunk_Size_Bytes];
	if(NULL == pu8Buf)
		return false;
	
	//Needed to get rid of custodial pointer - lint warning
	tU8* tmpbuf = pu8Buf;

	tS32 s32TeseoBytesRead=0;
	tS32 s32TeseoBytesSent=0;
	tF32 iFlashProgress=0;
	do
	{
		memset(pu8Buf, 0, u32gTeseo_Fw_Chunk_Size_Bytes);
		//read teseo firmware binary in configured size	
		if(false == m_BinaryHandler.bReadFile(u32gTeseo_Fw_Chunk_Size_Bytes, \
			tmpbuf, s32TeseoBytesRead))
		{
			if(s32TeseoBytesRead)
			{
				delete[] pu8Buf;
				return false;
			}
		}

		DEBUG_TRACE(TRACE_LEVEL_2,"INFO: after reading %d bytes from file ", s32TeseoBytesRead);
		
		if(s32TeseoBytesRead > 0)
		{
			s32TeseoBytesSent = OSAL_s32IOWrite(m_gnssFd, (tPCS8)pu8Buf,\
				(tU32) s32TeseoBytesRead);
				
			UpdateLog::vUpdateLogTrace ("%s teseo flashed bytes	%d", DL_LOG_INFO, s32TeseoBytesSent);
			DEBUG_TRACE(TRACE_LEVEL_2, "INFO: teseo flashed bytes	%d", s32TeseoBytesSent);		
				
			if(OSAL_ERROR == s32TeseoBytesSent)
			{
				bPostStatusEvent\
					(TESEO_OSAL_WRITE_ERROR, EV_TESEO_SWUPDATE_STATUS);
				delete[] pu8Buf;
				return false;
			}
			//update the flashed progress
			iFlashProgress +=static_cast<tF32> (s32TeseoBytesSent);
			tF32 iFlashProgPercent = \
				(iFlashProgress/(tF32)(m_BinaryHandler.s32GetFileSize())) * 100;			
			printf("\rINFO: Flashing %d%% complete",(tS32)iFlashProgPercent);			
			fflush(stdout);
		}
	}
	
	while(s32TeseoBytesRead > 0);
		tmpbuf = NULL;
	delete[] pu8Buf;
	
	return true;
}

/************************************************************************
* FUNCTION: bNotifyTeseoChipTypeToGNSS
* DESCRIPTION: Notifying Teseo chip type to GNSS driver.
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bNotifyTeseoChipTypeToGNSS()
{
		// Notify the GNSS driver about teseo chip type
		tU32 u32GnssChipType = GNSS_CHIP_TYPE_INVALID;
	
		if(strTeseoVersionToGNSS=="teseo_3")
		{
			u32GnssChipType = GNSS_CHIP_TYPE_TESEO_3;
		}
		else if (strTeseoVersionToGNSS=="teseo_2")
		{
			u32GnssChipType = GNSS_CHIP_TYPE_TESEO_2;
		}
		else
		{
			u32GnssChipType = GNSS_CHIP_TYPE_UNKNOWN;
		}
                ETG_TRACE_USR4 (("INFO: bNotifyTeseoChipTypeToGNSS: Notifying Teseo chip type %u (%s) to GNSS",u32GnssChipType, strTeseoVersionToGNSS.c_str()));
		if(OSAL_ERROR == OSAL_s32IOControl\
		( m_gnssFd, OSAL_C_S32_IOCTL_GNSS_SET_CHIP_TYPE, \
			(tU32)u32GnssChipType) ) 
		{
			bPostStatusEvent(TESEO_GNSS_NOTIFY_TYPE_ERROR, EV_TESEO_SWUPDATE_STATUS);
			UpdateLog::vUpdateLogTrace ("%s OSAL_s32IOControl error while Notifying Teseo chip type (%s) to GNSS!!!", DL_LOG_ERROR,strTeseoVersionToGNSS.c_str());
		
			ETG_TRACE_USR4 (("INFO: OSAL IOctl Error while Notifying Teseo chip type %s to GNSS!!!",strTeseoVersionToGNSS.c_str()));
		
			DEBUG_TRACE(TRACE_LEVEL_2, "ERROR: OSAL_s32IOControl error while Notifying Teseo chip type %s to GNSS!!!",strTeseoVersionToGNSS.c_str());
			
			vCloseFile();
			return false;
		}
		
		return true;
}

/************************************************************************
* FUNCTION: bPostStatusEvent
* DESCRIPTION: post status event to manager
* PARAMETER: tU8 u8UpdtState, tU8 u8Ev
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bPostStatusEvent(tU8 u8UpdtState, tU8 u8Ev) const
{
	teseo_SWUpdateMgr::m_SWUpdateState=u8UpdtState;
	teseo_SWUpdateMgr::m_evSWUpdaterToMgr=u8Ev;
	return bPostEvent();
}

/************************************************************************
* FUNCTION: vUpdateStatus
* DESCRIPTION: update status for manager
* PARAMETER: tU8 u8UpdtState, tU8 u8Ev
* RETURNVALUE: none
*************************************************************************/
tVoid teseo_SWUpdater::vUpdateStatus(tU8 u8UpdtState, tU8 u8Ev) const
{
	teseo_SWUpdateMgr::m_SWUpdateState=u8UpdtState;
	teseo_SWUpdateMgr::m_evSWUpdaterToMgr=u8Ev;	
}
		
/************************************************************************
* FUNCTION: bPostEvent
*
* DESCRIPTION: post event to update manager
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdater::bPostEvent() const
{
   if(OSAL_ERROR == OSAL_s32EventPost( teseo_SWUpdateMgr::m_evHandleMgr,
									 (OSAL_tEventMask) EV_TESEO_SW_UPDATER,
									 OSAL_EN_EVENTMASK_OR))
   {
       ETG_TRACE_USR4 (("ERROR: teseo_SWUpdateMgr::bPostEvent OSAL_s32EventPost error"));
	   DEBUG_TRACE (TRACE_LEVEL_1,"ERROR: teseo_SWUpdateMgr::bPostEvent OSAL_s32EventPost error");
	   return false;
   }
   return true;
}
