/************************************************************************
 * File: teseo_SWBinaryHandler.cpp
 * SW-Component: 
 * Description: Provides interfaces for reading the teseo binary files
 *				and calculate CRC32 for the teseo firmware.
 *				CRC is calculated first over the length of the file and then
 *				over the entire file.
 * Author:
 *	Rohit.ChannappiahRaman@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 14.10.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
 * 27.08.2013 - Implemented CMG3G-5318 optimized logging using update.log - Apoorva K R (RBEI/ECA1)
 * 6/4/2015 - Implemented CMG3G-6728 IS2030_Teseo update only if CRC changed@Integration Tooling - Apoorva K.R (RBEI/ECA1)
 * 1/1/2016 - Implemented CMG3G-9339 Teseo-3 support in SWUpdate - Anusha Ghali (RBEI/ECA1)
*---------------------------------------------------------------------------------------------------
* 1/2/2016          | AGL5KOR (RBEI)    |     CMG3G-9450 - teseo update ctrl app to be modified to write and read CRC
* 				    | Anusha Ghali	    | 
*---------------------------------------------------------------------------------------------------
 ***********************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_TESEO
  #include "trcGenProj/Header/teseo_SWBinaryHandler.cpp.trc.h"
#endif

#include "teseo_Global.h"
#include "teseo_SWBinaryHandler.h"

/* ====================================================================== */
/*  COPYRIGHT (C) 1986 Gary S. Brown.  You may use this program, or       */
/*  code or tables extracted from it, as desired without restriction.     */
/*                                                                        */
/*  First, the polynomial itself and its table of feedback terms.  The    */
/*  polynomial is                                                         */
/*  X^32+X^26+X^23+X^22+X^16+X^12+X^11+X^10+X^8+X^7+X^5+X^4+X^2+X^1+X^0   */
/*                                                                        */
/* ====================================================================== */
static const tU32 aCrc32Table[] = {
      0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
      0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
      0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
      0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
      0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
      0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
      0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
      0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
      0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
      0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
      0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
      0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
      0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
      0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
      0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
      0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
      0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
      0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
      0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
      0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
      0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
      0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
      0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
      0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
      0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
      0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
      0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
      0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
      0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
      0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
      0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
      0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
      0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
      0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
      0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
      0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
      0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
      0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
      0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
      0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
      0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
      0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
      0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
      0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
      0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
      0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
      0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
      0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
      0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
      0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
      0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
      0x2d02ef8dL
 };
   
/************************************************************************
* FUNCTION: teseo_SWBinaryHandler()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_SWBinaryHandler::teseo_SWBinaryHandler()
{
	m_s32FileSize=0;
	m_pFile=NULL;
	m_strFileName="";
}

/************************************************************************
* FUNCTION: ~teseo_SWBinaryHandler()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_SWBinaryHandler::~teseo_SWBinaryHandler()
{
	bCloseFile();
}

/************************************************************************
* FUNCTION: bLoadFile
*
* DESCRIPTION: initializes the client handler
*
* PARAMETER: std::string& strFlasherPath,
*			  std::string& strTeseoFWBinPath,
*			  rTeseoBinImgOptions& rImgOpts
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWBinaryHandler::bOpenFile(std::string& strFilePath)
{
	tBool bRet=false;
	m_s32FileSize=0;
	
	m_pFile = fopen( strFilePath.c_str(), "r" );
	
	if(NULL != m_pFile)
	{
		if(u32CalcFileSize())
		{
			bRet=true;
			m_strFileName=strFilePath;
		}	
	}
	return bRet;
}

/************************************************************************
* FUNCTION: bCloseFile()
*
* DESCRIPTION: Close the file
*
* PARAMETER: None
*
* RETURNVALUE: tBool
*************************************************************************/
tBool teseo_SWBinaryHandler::bCloseFile()
{
	m_s32FileSize=0;
	m_strFileName="";
	
	if(NULL != m_pFile)
	{
		if(fclose(m_pFile))
		{
			return false;
		}		
		m_pFile=NULL;
	}
	return true;
}


/************************************************************************
* FUNCTION: u32CalcFileSize
*
* DESCRIPTION: calculate file size
*
* PARAMETER: std::string path
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWBinaryHandler::u32CalcFileSize()
{
	if(NULL == m_pFile)
		return false;
		
	// set the file pointer to end of file
	if(fseek( m_pFile, 0, SEEK_END ) < 0)
		return false;
		
	// get the file size	
	m_s32FileSize =static_cast<tS32> (ftell( m_pFile ) );
	if(m_s32FileSize <= 0)
	{
		return false;
	}
	rewind(m_pFile);
	return true;
}

/************************************************************************
* FUNCTION: s32GetFileSize
*
* DESCRIPTION: get file size
*
* PARAMETER: none
* RETURNVALUE: tS32
*************************************************************************/
tS32 teseo_SWBinaryHandler::s32GetFileSize() const
{
	return m_s32FileSize;
}

/************************************************************************
* FUNCTION: s32GetFileName
*
* DESCRIPTION: get file name
*
* PARAMETER: none
* RETURNVALUE: string
*************************************************************************/
std::string teseo_SWBinaryHandler::s32GetFileName() const
{
	return m_strFileName;
}


/************************************************************************
* FUNCTION: bCalcCRC32ForFixedSize
*
* DESCRIPTION: calculate CRC for teseo new FW
*
* PARAMETER: tU32 &u32Crc - Calculated CRC
*			 std::string& strFilePath - Path to Teseo FW
*
* RETURNVALUE: tBool
*************************************************************************/
tBool  teseo_SWBinaryHandler::bCalcCRC32ForFixedSize\
	(tU32 &u32Crc, std::string& strFilePath)
{
	tU8* u8Buff = NULL;
	u8Buff = new tU8 [u32gTeseo_Fw_Chunk_Size_Bytes];
	
	
	tU8 cErasedByte = 0xff;
	size_t rNumBytesRead = 0;
	size_t rNumBytesLeft = 0;
	tU32 u32Crc32 = 0;
	tS32 i = 0;
	
	if(NULL == u8Buff)
	{
		DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: u8Buff null in bCalcCRC32ForFixedSize!");
		return false;
	}
	
	memset (u8Buff, 0, u32gTeseo_Fw_Chunk_Size_Bytes);
	
	if (false == bOpenFile (strFilePath))
	{
		DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: bOpenFile error in bCalcCRC32ForFixedSize");
		delete[] u8Buff;
		return false;		
	}
	
	rNumBytesLeft = m_s32FileSize;	
	size_t sizeOfBuffer = size_t(sizeof(u8Buff));  //Coverity fix for 126971
    do
    {
	rNumBytesRead = fread ( u8Buff, 1, sizeOfBuffer , m_pFile );
    	rNumBytesLeft -= rNumBytesRead;
    	u32Crc32 = u32EtherCrc32Accumulate ( u32Crc32, u8Buff,static_cast<tU32> (rNumBytesRead)) ;
    } while (rNumBytesLeft != 0);
	
    for (i=0; i < (0xffffc - (signed) m_s32FileSize) ; ++i )
    {
    	u32Crc32 = u32EtherCrc32Accumulate ( u32Crc32, &cErasedByte , (int) 1);
    }

    u32Crc = u32Crc32;
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Calculating CRC for new image SUCCESSFUL in Binary Hndl");
	bCloseFile ();

	delete[] u8Buff;
	return true;
}

/************************************************************************
* FUNCTION: bCalcCRC32
*
* DESCRIPTION: interface to calculate CRC32 for the already loaded file.
*			   CRC32 is first calculated on file size and then over 	 
*			   the entire file.	
* PARAMETER: none
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWBinaryHandler::bCalcCRC32 (tU32 &u32Crc,std::string& strFilePath)
{
	
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: teseo_SWBinaryHandler::bCalcCRC32 - start");
	
	if(NULL == m_pFile)
		return false;
		
	u32Crc=0;
	
	//First calculate crc over file length
	u32Crc = u32EtherCrc32Accumulate\
		(u32Crc, (tU8 *)&m_s32FileSize, (tU32)sizeof(m_s32FileSize));

	if (!bCalcCRCFlasher32(u32Crc,strFilePath))
	{
		UpdateLog::vUpdateLogTrace ("%s Calculating CRC for FIRMWARE", DL_LOG_INFO);
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Calculating CRC for FIRMWARE");
	}
	
	UpdateLog::vUpdateLogTrace ("%s teseo_SWBinaryHandler::bCalcCRC32 for file length - done", DL_LOG_INFO);
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: teseo_SWBinaryHandler::bCalcCRC32 for file length - done");
	
	//Then calculate crc over the file 
	tU8* u8Buff = NULL;
	u8Buff = new tU8[u32gTeseo_Fw_Chunk_Size_Bytes];
	if(NULL == u8Buff)
		return false;
		
	memset(u8Buff, 0, u32gTeseo_Fw_Chunk_Size_Bytes);
	rewind (m_pFile);
	
	tS32 u32BytesRead=0;
	
	//to remove custodial pointer lint warning
	tU8* tmpBuff = u8Buff;
	
	//lint -sem(teseo_SWBinaryHandler::bReadFile,custodial(2))
	while(true == bReadFile(u32gTeseo_Fw_Chunk_Size_Bytes, tmpBuff, u32BytesRead))
	{
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: %d bytes read from file", u32BytesRead);
		u32Crc = u32EtherCrc32Accumulate(u32Crc, u8Buff, (tU32) u32BytesRead);	
		memset(u8Buff, 0, u32gTeseo_Fw_Chunk_Size_Bytes);
	}
	
	delete[] u8Buff;
	
	//fread error case
	if(u32BytesRead)
	{
		return false;
	}
    rewind (m_pFile);
	
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: teseo_SWBinaryHandler::bCalcCRC32 - End");
	return true;
}

/************************************************************************
* FUNCTION: bCalcCRCFlasher32
* DESCRIPTION: Additional step to calculate CRC of bootloader		
* PARAMETER: u32Crc, strFilePath
* RETURNVALUE: bool.	
*************************************************************************/
tBool teseo_SWBinaryHandler::bCalcCRCFlasher32(tU32 &u32Crc,std::string& strFilePath)
{
	string strCurFileName = "";
	tU32 u32EntryAdr = 0;
	tBool res = false;
	strCurFileName = s32GetFileName();
	
	if ( !strcmp( strCurFileName.c_str() , strFilePath.c_str() ) )
	{
		res = true;
		UpdateLog::vUpdateLogTrace ("%s Calculating CRC for FLASHER", DL_LOG_INFO);
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Calculating CRC for FLASHER");
		u32Crc = u32EtherCrc32Accumulate\
			(u32Crc, (tU8 *)&u32EntryAdr, (tU32)sizeof(m_s32FileSize));
	}
	return res;
}

/************************************************************************
* FUNCTION: u32ReadFile
* DESCRIPTION: interface to read the already loaded file in specified  
*			   chunks to the input buffer.		
* PARAMETER: tU32 u32BytesToRead, tU8* buff, tS32 s32BytesRead
* RETURNVALUE: ts32 - false indicates EOF or an error case. The caller has to identify the 
*					  error return by evaluating EOF.	
*************************************************************************/
tBool teseo_SWBinaryHandler::bReadFile(tU32 u32BytesToRead, \
	tU8* buff, tS32& u32BytesRead)
{
	if(NULL == buff || NULL == m_pFile)
		return false;
	
	//Needed to get rid of custodial pointer - lint warning
	tU8* lBuf = NULL;
	lBuf = buff;	
	memset(buff, 0, u32BytesToRead);	

	//read u32BytesToRead bytes from file
	u32BytesRead =static_cast<tS32> (fread( buff, (unsigned)1, u32BytesToRead, m_pFile) );

	if(u32BytesRead <= 0)
	{
		//keep file ptr ready for next ops
		rewind(m_pFile);

		//fread error
		return false;
	}
	return true;
}

/************************************************************************
* FUNCTION: u32EtherCrc32Accumulate
* DESCRIPTION: Return a 32-bit CRC of the contents of the buffer accumulating the
*   result from a previous CRC calculation. This uses the Ethernet FCS
*   algorithm..		
* PARAMETER: tU32 u32Crc32val, tU8* pu8Buff, tU32 len
* RETURNVALUE: tu32
*************************************************************************/
/*lint -e{429} suppress 429 for next statement/block*/
tU32 teseo_SWBinaryHandler::u32EtherCrc32Accumulate(tU32 u32Crc32val, \
	const tU8* pu8Buff, tU32 len) const
{
  tU32 u32Indx=0;

  u32Crc32val = u32Crc32val ^ 0xffffffff;
  for (u32Indx = 0;  u32Indx < len;  u32Indx++) 
  {
      u32Crc32val = aCrc32Table[(u32Crc32val ^ pu8Buff[u32Indx]) & 0xff] ^ (u32Crc32val >> 8);
  }
  return u32Crc32val ^ 0xffffffff;
}
