/************************************************************************
 * File: teseo_ClientInterface.cpp
 * SW-Component: 
 * Description: Provides interfaces for teseo DL client to initiate and control 
 * 				teseo sw update
 * Author:
 *	Rohit.ChannappiahRaman@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 14.10.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
 * 27.08.2013 - Implemented CMG3G-5318 optimized logging using update.log - Apoorva K R (RBEI/ECA1)
 * 6/4/2015	Apoorva K.R - Implemented CMG3G-6728 IS2030_Teseo update only if CRC changed@Integration Tooling.
*---------------------------------------------------------------------------------------------------
* 25-08-2015        | AGL5KOR (RBEI)    | Fixing teseoupdate.log cutoff - CMG3GB-2380 
* 				    | Anusha Ghali	    | [SW Update]Persistent partition usage for SW update logs is around 40MB
*---------------------------------------------------------------------------------------------------
* 1/2/2016          | AGL5KOR (RBEI)    |     CMG3G-9450 - teseo update ctrl app to be modified to write and read CRC
* 				    | Anusha Ghali	    | 
*---------------------------------------------------------------------------------------------------
 ***********************************************************************/

#include <unistd.h>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_TESEO
  #include "trcGenProj/Header/teseo_ClientInterface.cpp.trc.h"
#endif

#include "teseo_ClientInterface.h"

//TBD
using namespace std;
#include <iostream>

/************************************************************************
* FUNCTION: teseo_ClientInterface()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_ClientInterface::teseo_ClientInterface()
{
}

/************************************************************************
* FUNCTION: ~teseo_ClientInterface()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_ClientInterface::~teseo_ClientInterface()
{
}


/************************************************************************
* FUNCTION: bInitClientInterface
*
* DESCRIPTION: initializes the client handler
*
* PARAMETER: std::string& strFlasherPath,
*			  std::string& strTeseoFWBinPath,
*			  rTeseoBinImgOptions& rImgOpts
* RETURNVALUE: bool
*************************************************************************/
tU8 teseo_ClientInterface::bInitClientInterface(std::string& strFlasherPath,
											  std::string& strTeseoFWBinPath)
{
	UpdateLog::vSetLogFile ("teseoupdate.log");
	UpdateLog::vUpdateLogOpen ();

	if(m_swUpdateMgr.bInitUpdateMgr(strFlasherPath, strTeseoFWBinPath))
	{
		ETG_TRACE_USR4 (("INFO: Client interface successfully created!"));
		UpdateLog::vUpdateLogTrace ("%s Client interface successfully created!", DL_LOG_INFO);	
		m_swUpdateMgr.m_SWUpdateState = TESEO_CLIENT_INTF_CREATION_OK;		
	}
	else
	{
		ETG_TRACE_USR4 (("ERROR: Failed to initialize Update Client interface!!!"));
		UpdateLog::vUpdateLogTrace ("%s Failed to initialize Update Client interface", DL_LOG_ERROR);
		DEBUG_TRACE (TRACE_LEVEL_1,"ERROR: Failed to initialize Update Client interface");
	}

	UpdateLog::vUpdateLogClose();
	return m_swUpdateMgr.m_SWUpdateState;
}

/************************************************************************
* FUNCTION: u8InitClientInterfaceForCRC
* DESCRIPTION: This function initializes teseo_SWUpdateMgr for CRC calc
* PARAMETER: const std::string& strPathToFile - Path to file 
			 std::string& strTeseoFWBinPath	- path to new f/w to be flashed
*			 bool bCRCFromGNSS - Get CRC of flashed teseo firmware
*			 bool bCRCFromUpdtCtl - Get CRC of new teseo firmware
* RETURNVALUE: tU8
*************************************************************************/	
tU8 teseo_ClientInterface::u8InitClientInterfaceForCRC\
	(std::string& strTeseoFWBinPath, const std::string& strPathToFile,\
		bool bCRCFromGNSS, bool bCRCFromUpdtCtl,bool bFlasherCrc)
{
	UpdateLog::vSetLogFile ("teseoupdate.log");
	UpdateLog::vUpdateLogOpen();
    UpdateLog::vUpdateLogClose();				
	return (m_swUpdateMgr.u8InitUpdateMgrForCRC \
			(strTeseoFWBinPath, strPathToFile, bCRCFromGNSS, \
			  bCRCFromUpdtCtl, bFlasherCrc));
}

/************************************************************************
* FUNCTION: bExitClientInterface
*
* DESCRIPTION: the client handler exits
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_ClientInterface::bExitClientInterface()
{
	pthread_mutex_lock( &(m_swUpdateMgr.m_swUpdateMutex));
	m_swUpdateMgr.m_evSWUpdateClientToMgr = EV_TESEO_SWUPDATE_EXIT;
	UpdateLog::vUpdateLogTrace ("%s Exiting Client Interface", DL_LOG_INFO);
	
    if(false != bPostEvent())
    {
        pthread_mutex_lock( &(m_swUpdateMgr.m_swUpdateMutex));
	}
	//Give time for manager and updater threads to exit
	sleep(1);
	pthread_mutex_unlock( &(m_swUpdateMgr.m_swUpdateMutex));		
	return true;	
}

/************************************************************************
* FUNCTION: bOnStartSWUpdate
*
* DESCRIPTION: interface to start teseo sw update
*
* PARAMETER: T_fCallback fptr_ClientCb
* RETURNVALUE: tu8 - status code
*************************************************************************/
tU8 teseo_ClientInterface::bOnStartSWUpdate(T_fCallback fptr_ClientCb)
{
	DEBUG_TRACE(TRACE_LEVEL_1, "INFO: Please don't restart the device or remove the download media till teseo update is complete");
	ETG_TRACE_USR4 (("INFO: Please don't restart the device or remove the download media till teseo update is complete"));
	UpdateLog::vUpdateLogTrace ("%s Starting Teseo Swupdate", DL_LOG_INFO);

	m_swUpdateMgr.m_fptrClientCb = fptr_ClientCb;
	pthread_mutex_lock( &(m_swUpdateMgr.m_swUpdateMutex));
	m_swUpdateMgr.m_evSWUpdateClientToMgr = EV_TESEO_SWUPDATE_START;
	
    if(false != bPostEvent())
    {
        pthread_mutex_lock( &(m_swUpdateMgr.m_swUpdateMutex));
	    pthread_mutex_unlock( &(m_swUpdateMgr.m_swUpdateMutex));
    }	
    else
    {
        pthread_mutex_unlock( &(m_swUpdateMgr.m_swUpdateMutex));
    }
	
	return m_swUpdateMgr.m_SWUpdateState;	
}
	
/************************************************************************
* FUNCTION: bOnAbortSWUpdate
*
* DESCRIPTION: abort sw update 
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_ClientInterface::bOnAbortSWUpdate() const
{
	UpdateLog::vUpdateLogTrace ("%s Aborting Teseo SWUpdate", DL_LOG_INFO);
	ETG_TRACE_USR4 (("INFO: Aborting Teseo SWUpdate"));
	teseo_SWUpdateMgr::m_evSWUpdateClientToMgr = EV_TESEO_SWUPDATE_ABORT;
	return bPostEvent();
}

/************************************************************************
* FUNCTION: bOnGetTeseoSWUpdateState
*
* DESCRIPTION: 
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tU8 teseo_ClientInterface::bOnGetTeseoSWUpdateState() const
{
	return teseo_SWUpdateMgr::m_SWUpdateState;
}

/************************************************************************
* FUNCTION: bOnGetTeseoSWUpdateStatus
*
* DESCRIPTION: 
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tU8 teseo_ClientInterface::bOnGetTeseoSWUpdateStatus() const
{
	return teseo_SWUpdateMgr::m_SWUpdateStatus;
}

/************************************************************************
* FUNCTION: bPostEvent
*
* DESCRIPTION: post event to update manager
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_ClientInterface::bPostEvent() const
{
   if(OSAL_ERROR == OSAL_s32EventPost( teseo_SWUpdateMgr::m_evHandleMgr,
									 (OSAL_tEventMask) EV_TESEO_SWUPDATE_MGR,
									 OSAL_EN_EVENTMASK_OR))
   {
	   DEBUG_TRACE(TRACE_LEVEL_1,"INFO: teseo_ClientInterface::bPostEvent OSAL_s32EventPost error");
       ETG_TRACE_USR4 (("INFO: teseo_ClientInterface::bPostEvent OSAL_s32EventPost error"));
	   return false;
   }
   return true;
}
