/************************************************************************
 * FILE: 			sxm_update_ctrl_app.cpp
 *
 * SW-COMPONENT: 	swu_common_sxm_lib
 *
 * DESCRIPTION: 	A CLI Application for performing SXM Firmware update
 *
 * AUTHOR: 			Logesh Gopalakrishnan (RBEI/ECA)
 *
 * COPYRIGHT: 		(c) Robert Bosch Engineering and Business Solutions Ltd.
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 *  20.06.2016  Rev. 1.1 Logesh
 * 				Choose firmware from a dir based on HType.
************************************************************************/

#include <iostream>
#include <fstream>
#include <string>
#include <getopt.h>
#include "string.h"
#include "sxm_enums.h"
#include "sxm_gpio.h"
#include "sxm_module_handler.h"
#include "sxm_version_utility.h"
#include "sxm_sirius_update.h"
#include "ErrorHandler.h"
#include "RegistryAccess.h"


#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN

  #include "trcGenProj/Header/sxm_update_ctrl_app.cpp.trc.h"
#endif


#define LOG_CONSOLE(format, args...) \
	fprintf(stderr, format, ##args); \
	fprintf(stderr, "\n")


#define DEVICE_NAME "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS"

#define SXM_UPDATE_CTRL_APP_VERSION "1.1.0"

#define OSAL_EXIT(n) 	{ \
							OSAL_vSetProcessExitCode(n); \
							OSAL_vProcessExit();		 \
						}

const char *SXM_ERROR_CODE_DESC[] = {
	"No Error",
	"Invalid Arguments. Try -h for help.\n",
	"Could not power on the device\n",
	"OSAL driver initialization failed\n",
	"SMS library initialization failed\n",
	"Cannot continue when SXM is not initialized.\n",
	"SRM object could not be created. Initialization failed\n",
	"Module object could not be created. Initialization failed\n",
	"Trying to down-grade. Down-grade not allowed\n",
	"This version of firmware cannot be updated\n",
	"Cannot continue when an update is in progress\n",
	"Firmware update process failed\n"
};

extern char *optarg;

using namespace SiriusModuleUpdateControlLib;
using namespace std;

/*
 * Global Functions
 */
tVoid cbProgressUpdate(tU8 progress);
tVoid presentDebuggingOptions(tVoid);
tVoid logInErrMem(SXM_ERROR_CODE_ENUM);
SXM_ERROR_CODE_ENUM doFirmwareUpdate(tVoid);
tVoid constructUpdateControllers(tVoid);
tVoid destructUpdateControllers(tVoid);
tVoid updateVersionInRegistry(tVoid);
tVoid printVersionInformation(tVoid);
tVoid validateInputFile(const string& sFirmwareFileName);
tVoid help(tBool logError = true);


/*
 * Global Pointers
 */
SiriusUpdate *poSiriusUpdater;
SiriusModuleHandler *poSiriusModule;
SXM_GPIO* poGPIOController;
SXM_VersionUtility* poVersionUtility;

string sFirmwareBase;
tBool bLoggingEnabled;
tBool bForceUpdate;

typedef enum sxm_cli_action
{
	SXM_ACTION_UPDATE_FIRMWARE,
	SXM_ACTION_DEBUG,
	SXM_ACTION_UPDATE_REGISTRY,
	SXM_ACTION_PRINT_VERSION

}SXM_ACTION;


/************************************************************************
 * FUNCTION: main
 *
 * DESCRIPTION: Starting point of control
 *
 * PARAMETER:
 * 				int, char*[]
 *
 * RETURNVALUE: int
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
int main(int argc, char *argv[]) {
	vInitPlatformEtg();

	// To initialize SWUpdate Error Memory
	ai_sw_update::common::iInitializeSwUpdateErrorArray();

	// Disable stdout buffering
	setbuf(stdout, NULL);

	bForceUpdate = FALSE;
	SXM_ACTION action = SXM_ACTION_UPDATE_FIRMWARE;

	// Read command line arguments
	if (argc > 1)
	{
		// evaluate command line options
		int indexes = 0;
		int  c;

		static struct option long_options[]=
		{
			// Dont forget to update help
			{"bin",		 		required_argument,	0,  'b'},		// Path to SXM firmware or a base directory
			{"force",		 	no_argument,       	0,  'f'},		// force same-grade or down-grade
			{"debug",    		no_argument,       	0,  'd'},		// Debug step by step
			{"help",     		no_argument,       	0,  'h'},		// Display command line options
			{"updatereg",	 	no_argument, 		0,  'r'},		// Update registry
			{"version",		 	no_argument, 		0,  'v'},		// Print version
			{"printlog",	 	no_argument, 		0,  'p'},		// Print log
			{0,               	0,            		0,   0 }
			// Dont forget to update help
		};

		while ((c = getopt_long_only (argc, argv, "b:fdhrvp", long_options, &indexes)) != -1)
		{
			switch (c)
			{
				case 'b':									// Path to SXM firmware or a base directory
					sFirmwareBase = optarg;
				break;

				case 'f':									// Force an update to happen even if the version is similar
					bForceUpdate = TRUE;
				break;

				case 'd':									// Debug step by step
					action = SXM_ACTION_DEBUG;
				break;

				case 'r':									// Update registry
					action = SXM_ACTION_UPDATE_REGISTRY;
				break;

				case 'v':									// Print application and SXM version
					action = SXM_ACTION_PRINT_VERSION;
				break;

				case 'p':									// Print traces in console
					bLoggingEnabled = TRUE;
				break;

				case 'h':									// Display command line options
					help(false);
				break;
				case ':':		//arguments missing
				case '?':		//unrecognized option
				default:
					help();
				break;
			}
		}

	}

	else
	{
		help();
	}

	printf("SXM_UPDATE_CTRL_VER = %s\n", SXM_UPDATE_CTRL_APP_VERSION);

	switch(action)
	{

		case SXM_ACTION_DEBUG:

				// If firmware file is not specified exit the program
				validateInputFile(sFirmwareBase);

				constructUpdateControllers();

				presentDebuggingOptions();

				destructUpdateControllers();

		break;

		case SXM_ACTION_UPDATE_REGISTRY:

				constructUpdateControllers();

				updateVersionInRegistry();

				destructUpdateControllers();

		break;

		case SXM_ACTION_PRINT_VERSION:

				constructUpdateControllers();

				printVersionInformation();

				destructUpdateControllers();

		break;

		case SXM_ACTION_UPDATE_FIRMWARE:

				validateInputFile(sFirmwareBase);

				constructUpdateControllers();

				SXM_ERROR_CODE_ENUM error = doFirmwareUpdate();

				destructUpdateControllers();

				OSAL_EXIT( (tInt)error);

		break;
	}


	OSAL_EXIT(0);

}

/************************************************************************
 * FUNCTION: doFirmwareUpdate
 *
 * DESCRIPTION: Initializes, updates firmware and deinitializes SXM
 * 				On error logs in error memory
 *
 * PARAMETER:
 * 				const string&
 *
 * RETURNVALUE: SXM_ERROR_CODE_ENUM
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SXM_ERROR_CODE_ENUM doFirmwareUpdate()
{
	SXM_ERROR_CODE_ENUM error = SXM_NO_ERROR;
	// Start the 3-Step update process

	if( ! poSiriusUpdater)
	{
		LOG_CONSOLE("Updater not initialized");
		error = SXM_ERROR_INITIALIZATION_FAILURE;
		logInErrMem(error);
	}

	if( ! poSiriusUpdater->initialize())
	{
		error = poSiriusUpdater->getError();
		LOG_CONSOLE("Initialization Failed %d", error);
		logInErrMem(error);
		return error;
	}

	if( ! poSiriusUpdater->updateFirmware(sFirmwareBase, bForceUpdate, cbProgressUpdate) )
	{
		error = poSiriusUpdater->getError();
		LOG_CONSOLE("Update Failed %d", error);
		logInErrMem(error);

		// Deinitialize after failure. Do not return immediately.
	}

	if( ! poSiriusUpdater->deinitialize())
	{
		error = poSiriusUpdater->getError();
		LOG_CONSOLE("De-initialization Failed %d", error);
		logInErrMem(error);
		return error;
	}

	if(error == SXM_NO_ERROR)
	{
        LOG_CONSOLE("Update Completed Successfully.");
	}

	return error;
}

/************************************************************************
 * FUNCTION: updateVersionInRegistry
 *
 * DESCRIPTION: Initializes and gets the version information and writes in registry
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid updateVersionInRegistry()
{
	try {
	if ( poSiriusUpdater && poSiriusUpdater->initialize() )
	{
		RegistryAccess registry;
		registry.u32RegWriteValue(DEVICE_NAME, "SXM_SERIALNO_VER", const_cast<char*>(poSiriusUpdater->getSerialVersion().c_str()));
		registry.u32RegWriteValue(DEVICE_NAME, "SXM_HW_VER", const_cast<char*>(poSiriusUpdater->getHardwareVersion().c_str()));
		registry.u32RegWriteValue(DEVICE_NAME, "SXM_SW_VER", const_cast<char*>(poSiriusUpdater->getSoftwareVersion().c_str()));

		printf("SXM_SERIALNO_VER = %s\n", poSiriusUpdater->getSerialVersion().c_str());
		printf("SXM_HW_VER = %s\n", poSiriusUpdater->getHardwareVersion().c_str());
		printf("SXM_SW_VER = %s\n", poSiriusUpdater->getSoftwareVersion().c_str());
		fflush(stdout);
		poSiriusUpdater->deinitialize();
	}
	else
	{
		LOG_CONSOLE("Sirius Updater Cannot be initialized. Could not update version in registry.");
	}

	} catch (std::exception& e) {
		LOG_CONSOLE(( "updateVersionInRegistry failed" ));
		OSAL_EXIT((tInt)SXM_ERROR_INITIALIZATION_FAILURE);
	}

}

/************************************************************************
 * FUNCTION: printVersionInformation
 *
 * DESCRIPTION: Initializes and gets the version information and logs in console
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid printVersionInformation()
{
	try {
	if ( poSiriusUpdater && poSiriusUpdater->initialize() )
	{
		printf("SXM_HW_TYPE = %s\n", poSiriusUpdater->getHardwareType().c_str());
		printf("SXM_SERIALNO_VER = %s\n", poSiriusUpdater->getSerialVersion().c_str());
		printf("SXM_HW_VER = %s\n", poSiriusUpdater->getHardwareVersion().c_str());
		printf("SXM_SW_VER = %s\n", poSiriusUpdater->getSoftwareVersion().c_str());
		fflush(stdout);
		poSiriusUpdater->deinitialize();
	}
	else
	{
		LOG_CONSOLE("Sirius Updater Cannot be initialized. Could not update version in registry.");
	}

	} catch (std::exception& e) {
		LOG_CONSOLE(( "printVersionInformation failed" ));
		OSAL_EXIT((tInt)SXM_ERROR_INITIALIZATION_FAILURE);
	}

}

/************************************************************************
 * FUNCTION: cbProgressUpdate
 *
 * DESCRIPTION: Call back function for getting progress information
 *
 * PARAMETER:
 * 				tU8
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid cbProgressUpdate(tU8 progress)
{
	if(bLoggingEnabled) return; // If enabled, it is done by sxm sirius updater

	if(progress % 10 == 0)
	{
		fprintf(stdout, "Updating...%d%%\n", progress);fflush(stdout);
	}
}

/************************************************************************
 * FUNCTION: constructUpdateControllers
 *
 * DESCRIPTION: Construct all necessary controllers for SXM handling
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid constructUpdateControllers()
{
	SXM_ERROR_CODE_ENUM error = SXM_ERROR_INITIALIZATION_FAILURE;

    try {

	// Create new object for updating sirius module
	poSiriusUpdater = SiriusUpdate::instance();

	poSiriusModule = SiriusModuleHandler::instance();

	poGPIOController = new tclSXM_GPIO();

	poVersionUtility = new tclSXM_VersionUtility();

	poSiriusModule->setControllers(poGPIOController, poVersionUtility);

	poSiriusUpdater->setSiriusModuleHandler(poSiriusModule);

	} catch (std::exception& e) {
		LOG_CONSOLE(( "Construction of objects failed" ));
		logInErrMem(error);
		OSAL_EXIT((tInt)error);
	}

	if(!poSiriusUpdater)
	{
		LOG_CONSOLE("Construction of objects failed");
		logInErrMem(error);
		OSAL_EXIT((tInt)error);
	}
}

/************************************************************************
 * FUNCTION: destructUpdateControllers
 *
 * DESCRIPTION: Delete all allocated pointers for handling SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid destructUpdateControllers()
{
	delete(poSiriusUpdater);
	delete(poSiriusModule);
	delete(poVersionUtility);
	delete(poGPIOController);

}

/************************************************************************
 * FUNCTION: validateInputFile
 *
 * DESCRIPTION: Check if the input file is present, readable and not empty
 *
 * PARAMETER:
 * 				const string&
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 * 	20.06.2016 	Rev. 1.1 Logesh
 * 				Removed readability check. It is done by firmwareUpdate API
************************************************************************/
tVoid validateInputFile(const string& sFirmwareFileName) {

	// If firmware file is not specified exit the program
	if (sFirmwareFileName.empty()) 
	{
		LOG_CONSOLE("%s", SXM_ERROR_CODE_DESC[SXM_ERROR_INVALID_ARGUMENTS]);
		help();
		OSAL_EXIT(SXM_ERROR_INVALID_ARGUMENTS);
	}

}


/************************************************************************
 * FUNCTION: logInErrMem
 *
 * DESCRIPTION: Log the error in SwUpdate Error pipe and Errmem
 *
 * PARAMETER:
 * 				SXM_ERROR_CODE_ENUM
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid logInErrMem(SXM_ERROR_CODE_ENUM iExitCode)
{
	ai_sw_update::common::iSetSwUpdateError((int)ai_sw_update::common::SW_UPDATE_ERROR_DEVICE_SXM, \
                    (int)ai_sw_update::common::SW_UPDATE_ERROR_TYPE_VERIFY, \
                    "SXM UPDATE error:", SXM_ERROR_CODE_DESC[iExitCode]);
}

/************************************************************************
 * FUNCTION: help
 *
 * DESCRIPTION: Show supported options and examples.
 * 					If parameter is true log in errmem.
 *
 * PARAMETER:
 * 				SXM_ERROR_CODE_ENUM
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid help(tBool logError)
{
	cout <<"\n \
Options supported:\n \
	--bin		(-b)\n \
	--force		(-f)\n \
	--version	(-v)\n \
	--updatereg	(-r)\n \
	--printlog	(-p)\n \
	--debug		(-d)\n \
	--help		(-h)\n \
\n \
Usage Examples: \n \
	For SXM Update: \n \
		swu_common_sxm_app_out.out -b </path/to/firmware/dir/ or /firmware.bin> \n \
	For SXM Force Update Equal Version: \n \
		swu_common_sxm_app_out.out -b </path/to/firmware/dir/ or /firmware.bin>> -f \n \
	For debugging: \n \
		swu_common_sxm_app_out.out -b </path/to/firmware/dir/ or /firmware.bin>> -d \n \
	Update version in registry \n \
		swu_common_sxm_app_out.out -r \n \
	Print version information \n \
		swu_common_sxm_app_out.out -v \n \
	Print logs in console \n \
		swu_common_sxm_app_out.out -b </path/to/firmware/dir/ or /firmware.bin>> -p \n \
	Display options & usage: \n \
		swu_common_sxm_app_out.out -h \n" << endl;

	if(logError)
	{
			logInErrMem(SXM_ERROR_INVALID_ARGUMENTS);
			OSAL_EXIT(SXM_ERROR_INVALID_ARGUMENTS);
	}
	else
	{
		OSAL_EXIT(0);
	}
}


/************************************************************************
 * FUNCTION: presentDebuggingOptions
 *
 * DESCRIPTION: Present functionalities to be called indivudually for debugging
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid presentDebuggingOptions()
{

	int choise;
	SXM_ERROR_CODE_ENUM eErrorCode;

	do {

		LOG_CONSOLE(
				"  \n 1. power on  \n 2. initialize  \n 3. Update  \n 4. deinitialize  \n 5. power off  \n 6. Version Details  "
				"  \n 7. Initialize()  \n 8. UpdateFirmWare()  \n 9. DeInitialize() "
				"  \n 0. Exit  \nEnter choice: ");
		cin >> choise;

		switch (choise) {

		case 1: 	// Power On the device

			if (!poSiriusModule->powerOn()) {
				LOG_CONSOLE("SXM_ERROR_POWER_ON_FAILURE");
			}

			break;

		case 2:		// Initialize the handler

			eErrorCode = poSiriusModule->initialize();

			if (eErrorCode != SXM_NO_ERROR) {
				LOG_CONSOLE("SXM_ERROR_INITIALIZATION_FAILURE %d", eErrorCode);
			}

			break;

		case 3:		// Do Firmware Update

			eErrorCode = poSiriusModule->firmwareUpdate(sFirmwareBase, bForceUpdate, cbProgressUpdate);

			if (eErrorCode != SXM_NO_ERROR) {
				LOG_CONSOLE("SXM_ERROR_UPDATE_FAILURE (%d)", eErrorCode);
			}

			break;

		case 4:		// De-initialize the handler

			poSiriusModule->deInitialize();

			break;

		case 5:		// Power off the device

			if (!poSiriusModule->powerOff()) {
				LOG_CONSOLE("Could not properly power off the device");
			}

			break;

		case 6:		// Get firmware version of Running software and firmware file
			LOG_CONSOLE("Hardware Type    = %s", poSiriusModule->getHardwareTypeOfModule().c_str());
			LOG_CONSOLE("Serial Version   = %s", poSiriusModule->getSerialVersionOfModule().c_str());
			LOG_CONSOLE("Hardware Version = %s", poSiriusModule->getHardwareVersionOfModule().c_str());
			LOG_CONSOLE("Software Version = %s", poSiriusModule->getSoftwareVersionOfModule().c_str());
			LOG_CONSOLE("Firmware Version = %s", poSiriusModule->getSoftwareVersionOfFirmware(sFirmwareBase).c_str());
			LOG_CONSOLE("Firmware HW Type =: %s", poSiriusModule->getHardwareTypeFromFirmware(sFirmwareBase).c_str());

			break;

		case 7:
			if( ! poSiriusUpdater->initialize() ) {
				LOG_CONSOLE("%d", poSiriusUpdater->getError());
			}
			LOG_CONSOLE("Initialize complete");
			break;

		case 8:

			if( ! poSiriusUpdater->updateFirmware(sFirmwareBase, bForceUpdate, cbProgressUpdate) ) {
				LOG_CONSOLE("%d", poSiriusUpdater->getError());
			}
			LOG_CONSOLE("Update Success");

			break;

		case 9:
			if( ! poSiriusUpdater->deinitialize() ) {
				LOG_CONSOLE("DeInitialization Failed");
			}
			LOG_CONSOLE("De-Initialize complete");

			break;

		case 0:		// Exit the application
			return;

		default:
			continue;
		}

	} while (1);
}

