/************************************************************************
* File: fbhmi_Utility.h
* SW-Component: 
* Description: utilty functions used by fbhmi_surface for drawing on linux frame buffer
*   
* Author:
*	Rohit.ChannappiahRaman@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 06.06.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
* 18.03.2014 - Lint warning correction - Aditya Kumar Jha
***********************************************************************/

#include "fbhmi_Utility.h"

static const _tChar aCharUtf8Bytes[256] = {
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, 1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2, 3,3,3,3,3,3,3,3,4,4,4,4,5,5,5,5
};

static const _tU32 aU32OffsetsFromUTF8[6] = { 0x00000000UL, 0x00003080UL, 0x000E2080UL, 
0x03C82080UL, 0xFA082080UL, 0x82082080UL };

/************************************************************************
* FUNCTION: fbhmi_Utility
* DESCRIPTION: constructor
* PARAMETER: 
* RETURNVALUE: None
*************************************************************************/
fbhmi_Utility::fbhmi_Utility()
{
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: ~fbhmi_Utility
* DESCRIPTION: Destructor
* PARAMETER: 
* RETURNVALUE: None
*************************************************************************/
fbhmi_Utility::~fbhmi_Utility()
{
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: u32Ustrlen
* DESCRIPTION: gets string length from pointer to string
* PARAMETER: _tU32 *
* RETURNVALUE: _tU32
*************************************************************************/
_tU32 fbhmi_Utility::u32UstrLen(_tU32 *pu32str)
{
	size_t u32Size = 0;
	while (*pu32str != 0)
	{
		u32Size++;
		pu32str++;
	}
	DEBUG_TRACE("%u", u32Size);
	return u32Size;
}

/************************************************************************
* FUNCTION: vUstrcat
* DESCRIPTION: appends the given character to the buffer string
* PARAMETER: buffer *, _tU32
* RETURNVALUE: _tVoid
*************************************************************************/
_tVoid fbhmi_Utility::vUstrCat(_tU32 *pu32Buff, _tU32 u32chr)
{
	DEBUG_TRACE("%s", __func__);
	_tU32 u32Size = fbhmi_Utility::u32UstrLen(pu32Buff);
	pu32Buff[u32Size] = u32chr;
	pu32Buff[u32Size+1] = 0;
}

/************************************************************************
* FUNCTION: pu32Ustrdup
* DESCRIPTION: duplicates the given string and returns the duplicate string.
* PARAMETER: buffer *
* RETURNVALUE: _tU32*
*************************************************************************/
_tU32* fbhmi_Utility::pu32UstrDup(_tU32 *pu32Buff)
{

	_tU32 u32Size = fbhmi_Utility::u32UstrLen(pu32Buff);
	_tU32 *pu32dup = (_tU32 *)malloc(sizeof(_tU32)*(u32Size+1));

	if (NULL == pu32dup) {
		DEBUG_TRACE("Unable to get memory");
		return NULL;
	}
	DEBUG_TRACE("Len: %u", u32Size);

	for (_tU32 u32X=0; u32X < u32Size; u32X++) {
		pu32dup[u32X] = pu32Buff[u32X];
	}
	pu32dup[u32Size] = 0;
	return pu32dup;
}

/************************************************************************
* FUNCTION: vSplitStringSpaces
* DESCRIPTION: The given string is split based on spaces and populated in 
*				the given vector
* PARAMETER: _tU32 *, vector<_tU32 *> &
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Utility::vSplitStringSpaces(_tU32 *pu32Wbuf, vector<_tU32 *> &vecParts)
{
	DEBUG_TRACE("%s", __func__);
	_tU32 u32Len = fbhmi_Utility::u32UstrLen(pu32Wbuf);

	_tU32 *pu32Buff = (_tU32 *)malloc(sizeof(_tU32)*(u32Len+1));
	/*To remove Lint warning*/
	_tU32 *pu32Custodyptr;
	pu32Custodyptr = pu32Buff;
	if (NULL == pu32Buff) {
		DEBUG_TRACE("Unable to get memory");
		return ;
	}

	pu32Buff[0] = 0;

	for (_tU32 u32X = 0; u32X < u32Len; u32X++) {

		fbhmi_Utility::vUstrCat(pu32Buff, pu32Wbuf[u32X]);
		if (pu32Wbuf[u32X] == ' ') {
			vecParts.push_back(pu32UstrDup(pu32Buff));
			pu32Buff[0] = 0;
		}
	}
	vecParts.push_back(pu32UstrDup(pu32Buff));
	free(pu32Buff);

}

/************************************************************************
* FUNCTION: vSplitStringDirection
* DESCRIPTION: The given string is split based on spaces and populated in 
*				the given vector
* PARAMETER: _tU32 *, vector<_tU32 *> &
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Utility::vSplitStringDirection(_tU32 *pu32Wbuf, vector<_tU32 *> &vecParts)
{
	_tU32 u32Len = fbhmi_Utility::u32UstrLen(pu32Wbuf);
	DEBUG_TRACE("Lentgh: %u", u32Len);
	vecParts.push_back(pu32UstrDup(pu32Wbuf));
	u32Len = fbhmi_Utility::u32UstrLen(pu32Wbuf);
	DEBUG_TRACE("AfterLentgh: %u", u32Len);
	return;
}

/************************************************************************
* FUNCTION: bIsWhiteChar
* DESCRIPTION: checks for control characters, return true if a control char
* PARAMETER: _tU32
* RETURNVALUE: _tBool
*************************************************************************/
_tBool fbhmi_Utility::bIsWhiteChar(_tU32 u32Chr)
{
	DEBUG_TRACE("%s", __func__);
	if ((u32Chr <=32))
		return true;
	else
		return false;
}

/************************************************************************
* FUNCTION: u32GetPresentation
* DESCRIPTION: 
* PARAMETER: _tU32 *, _tU32& 
* RETURNVALUE: _tU32
*************************************************************************/
_tU32 fbhmi_Utility::u32GetPresentation(_tU32 *pu32Part, _tU32& pu32X)
{
	_tU32 u32Pres;
	DEBUG_TRACE("%s", __func__);

	u32Pres = pu32Part[pu32X];
	return u32Pres;
}

/************************************************************************
* FUNCTION: u32GetPresentionAdvance
* DESCRIPTION: 
* PARAMETER: _tU32 ,_tBool &
* RETURNVALUE: _tU32
*************************************************************************/
_tS32 fbhmi_Utility::u32GetPresentionAdvance(_tU32 u32Pres,_tBool &pbToDraw)
{
	fbhmi_Font *pFont = fbhmi_Surface::pGetInstance()->pGetFont();
	_tS32 s32Adv = 0;
	pbToDraw = false;
	DEBUG_TRACE("%s", __func__);
	if(NULL == pFont) {
		DEBUG_TRACE("Unable to get Font pointer");
		return 0;
	}

	switch (u32Pres) {
		  case 0x9:
			  s32Adv = pFont->s32GetPixelSize()/2;
			  break;
		  case 0xa:
		  case 0xd:
		  case 0xfeff:
		  case 0xfffe:
		  case 0x200e:
		  case 0x200f:
			  break;
		  default:
			  s32Adv = pFont->s32GetAdvanceX(u32Pres);
			  pbToDraw = true;
			  break;
	}
	return s32Adv;
}

/************************************************************************
* FUNCTION: u32GetStringPixelLength
* DESCRIPTION: gets pixel length for the given string
* PARAMETER: _tU32 *
* RETURNVALUE: _tU32
*************************************************************************/
_tU32 fbhmi_Utility::u32GetStringPixelLength(_tU32 *pu32Part)
{
	_tU32 u32PixelLen = 0;
	_tBool bToDraw;
	DEBUG_TRACE("%s", __func__);

	_tU32 u32Len = fbhmi_Utility::u32UstrLen(pu32Part);

	//lint -e850
	for (_tU32 u32X=0; u32X < u32Len; u32X++) {

		_tU32 u32Pres = u32GetPresentation(pu32Part, u32X);
		u32PixelLen += u32GetPresentionAdvance(u32Pres, bToDraw);
	}
	return u32PixelLen;
}

/************************************************************************
* FUNCTION: vConvlatin9toutf32
* DESCRIPTION: converts latin9 character to utf32 character
* PARAMETER: _tChar *, _tChar *, length
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Utility::vConvlatin9toutf32(_tChar *pcharLatin, _tU32 *pcharUtf32, _tU32 u32Len)
{
	//to remove lint
	u32Len = u32Len;
	DEBUG_TRACE("Len: %u", u32Len);
	_tChar* pcharUtf8 = pcharLatin2utf8((const _tChar*)pcharLatin);
	if(NULL == pcharUtf8){
		DEBUG_TRACE("failed to convert latin 2 utf8");
		return;
	}
	if(!bUtf8toutf32(pcharUtf8, pcharUtf32))
		printf("Failed to convert to UTF32\n");
	free(pcharUtf8);
}

/************************************************************************
* FUNCTION: pcharLatin2utf8
* DESCRIPTION: converts latin character to utf8
* PARAMETER: const pointer to the const latin character
* RETURNVALUE: _tChar*
*************************************************************************/
_tChar* fbhmi_Utility::pcharLatin2utf8(const _tChar *const pcharString)
{
	_tChar   *pcharResult;
	_tU32  u32N = 0;
	DEBUG_TRACE("%s", __func__);

	if (NULL == pcharString) 
	{
		DEBUG_TRACE("String pointer is NULL");
		return NULL;
	}

	const _tU8  *pu8TmpS = (const _tU8 *)pcharString;
	while (*pu8TmpS)
	{
		if (*pu8TmpS < 128) 
		{
			pu8TmpS++; 
			u32N+= 1;
		} 
		else if (*pu8TmpS == 164) 
		{
			pu8TmpS++; 
			u32N += 3;
		} 
		else 
		{
			pu8TmpS++; 
			u32N += 2;
		}
	}

	/* Allocate n+1 (to n+7) bytes for the converted string. */
	pcharResult = (_tChar *) malloc((u32N | 7) + 1);
	if (!pcharResult)
		return NULL;

	/* Clear the tail of the string, setting the trailing NUL. */
	memset(pcharResult + (u32N | 7) - 7, 0, 8);

	if (u32N) {
		const _tU8  *pu8S = (const _tU8 *)pcharString;
		_tU8        *pu8D = (_tU8 *)pcharResult;

		while (*pu8S)
			if (*pu8S < 128) {
				*(pu8D++) = *(pu8S++);
			} else
				if (*pu8S < 192) switch (*pu8S) {
					 case 164: *(pu8D++) = 226; *(pu8D++) = 130; *(pu8D++) = 172; pu8S++; break;
					 case 166: *(pu8D++) = 197; *(pu8D++) = 160; pu8S++; break;
					 case 168: *(pu8D++) = 197; *(pu8D++) = 161; pu8S++; break;
					 case 180: *(pu8D++) = 197; *(pu8D++) = 189; pu8S++; break;
					 case 184: *(pu8D++) = 197; *(pu8D++) = 190; pu8S++; break;
					 case 188: *(pu8D++) = 197; *(pu8D++) = 146; pu8S++; break;
					 case 189: *(pu8D++) = 197; *(pu8D++) = 147; pu8S++; break;
					 case 190: *(pu8D++) = 197; *(pu8D++) = 184; pu8S++; break;
					 default:  *(pu8D++) = 194; *(pu8D++) = *(pu8S++); break;
				} else {
					*(pu8D++) = 195;
					*(pu8D++) = *(pu8S++) - 64;
				}
	}

	/* Done. Remember to free() the resulting string when no longer needed. */
	return pcharResult;
}

/************************************************************************
* FUNCTION: bIsLegalUTF8
* DESCRIPTION: checks if the given string is a legal utf8 character
* PARAMETER: character *, length
* RETURNVALUE: _tBool
*************************************************************************/
_tBool fbhmi_Utility::bIsLegalUTF8(const _tChar *pcharSource, _tS32 s32Length)
{
	DEBUG_TRACE("%s", __func__);
	_tChar charA;
	if(NULL == pcharSource) {
		DEBUG_TRACE("NULL input string");
		return false;
	}
	const _tChar *pcharSrcptr = pcharSource+s32Length;
	switch (s32Length) {
	 default: return false;
		 /* Everything else falls through when "true"... */
	 case 4: if ((charA= (*--pcharSrcptr)) < 0x80 || charA > 0xBF) 
					return false;
		 /*fall through*/
	 case 3: if ((charA = (*--pcharSrcptr)) < 0x80 || charA > 0xBF) 
					return false;
		 /*fall through*/
	 case 2: if ((charA = (*--pcharSrcptr)) < 0x80 || charA > 0xBF) 
					return false;


		 switch (*pcharSource) {
			 /* no fall-through in this inner switch */
	 case 0xE0: if (charA < 0xA0) return false; break;
	 case 0xED: if (charA > 0x9F) return false; break;
	 case 0xF0: if (charA < 0x90) return false; break;
	 case 0xF4: if (charA > 0x8F) return false; break;
	 default:   if (charA < 0x80) return false;
		 }

		 /*fall through*/
	 case 1: if (*pcharSource >= 0x80 && *pcharSource < 0xC2) return false;
	}
	if (*pcharSource > 0xF4) return false;
	return true;
}

/************************************************************************
* FUNCTION: bUtf8toutf32
* DESCRIPTION: converts utf8 character to utf32
* PARAMETER: source character *, destination character *
* RETURNVALUE: _tBool
*************************************************************************/
_tBool fbhmi_Utility::bUtf8toutf32(const _tChar* pcharSource, _tU32* pcharUtf32)
{
	DEBUG_TRACE("%s", __func__);
	_tBool bResult = true;
	_tU16 u16CharLen = 0;
	if(NULL == pcharSource || NULL == pcharUtf32) {
		DEBUG_TRACE("%s ptr is NULL",(NULL == pcharSource )?"pcharSource ":"pcharUtf32");
		return false;
	}
	_tU16 u16Utf8len = strlen(pcharSource) + 1;
	_tU32 *pu32Dst = pcharUtf32;

	while (*pcharSource) {
		_tU32 u32ch = 0;
		u16CharLen = aCharUtf8Bytes[(_tS32) *pcharSource];

		if (u16CharLen >= u16Utf8len) {
			bResult = false;
			break;
		}

		if (!bIsLegalUTF8(pcharSource, u16CharLen+1)) {
			bResult = false;
			break;
		}

		switch (u16CharLen) {
					default:
						break;
					case 5: u32ch += *pcharSource++; u32ch <<= 6;
						/*fall through*/
					case 4: u32ch += *pcharSource++; u32ch <<= 6;
						/*fall through*/
					case 3: u32ch += *pcharSource++; u32ch <<= 6;
						/*fall through*/
					case 2: u32ch += *pcharSource++; u32ch <<= 6;
						/*fall through*/
					case 1: u32ch += *pcharSource++; u32ch <<= 6;
						/*fall through*/
					case 0: u32ch += *pcharSource++;
		}
		u32ch -= aU32OffsetsFromUTF8[u16CharLen];

		if (u32ch <= UNI_MAX_LEGAL_UTF32) {

			if (u32ch >= UNI_SUR_HIGH_START && u32ch <= UNI_SUR_LOW_END) {
				*pu32Dst++ = UNI_REPLACEMENT_CHAR;
			} else {
				*pu32Dst++ = u32ch;
			}
		} else {
			bResult = false;
			*pu32Dst++ = UNI_REPLACEMENT_CHAR;
		}
	}
	return bResult;
}
