/************************************************************************
* File: fbhmi_Surface.cpp
* SW-Component:
* Description: The class helps in drawing the graphics using the linux framebuffer.
* Author:
*   Rohit.ChannappiahRaman@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 10.06.2013 - Skeleton code - Rohit.ChannappiahRaman@in.bosch.com
* 13.06.2013 - Initial Version - Gururaj.B@in.bosch.com
***********************************************************************/

#include <string.h>
#include <stdlib.h>
#include "Fonts/fbhmi_Font.h"
#include "fbhmi_Surface.h"
#include "fbhmi_Global.h"

// Initialize static variables
_tBool fbhmi_Surface::m_bCreated = false;
fbhmi_Surface* fbhmi_Surface::m_pSurfaceHandle = NULL;
fbhmiDisplayType fbhmi_Surface::DisplayType=FBHMI_DISPLAY_FB;


/************************************************************************
* FUNCTION: fbhmi_Surface::pGetInstance()
*
* DESCRIPTION: Create fbhmi_Surface singleton instance
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Surface* fbhmi_Surface::pGetInstance()
{
    if(false == m_bCreated)
    {
        m_pSurfaceHandle = new fbhmi_Surface(FBHMI_SCREEN_DEF_WIDTH,
            FBHMI_SCREEN_DEF_HEIGHT);

        m_bCreated = true;
    }
    return m_pSurfaceHandle;
}


/************************************************************************
* FUNCTION: fbhmi_Surface::fbhmi_Surface()
*
* DESCRIPTION: private Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Surface::fbhmi_Surface(_tU32 u32x, _tU32 u32y)
{
	DEBUG_TRACE("%s", __func__);
    // Create framebuffer or kms handler object
    if(DisplayType==FBHMI_DISPLAY_KMS)
      this->m_pGfxHandle = fbhmi_KmsHandler::pCreateInstance();
    else
      this->m_pGfxHandle = fbhmi_FbHandler::pCreateInstance();

    if(NULL != this->m_pGfxHandle)
    {
        this->m_u32XRes = m_pGfxHandle->u32GetXRes();
        this->m_u32YRes = m_pGfxHandle->u32GetYRes();
    }
    else
    {
        // Debug Assert and proceed with only surface of size u32x*u32y
        // This will make the FBHMI work even if the framebuffer device initialization fails
        this->m_u32XRes = u32x;
        this->m_u32YRes = u32y;
    }

    this->m_pu32Memory = (_tU32 *)malloc(FBHMI_BYTES_PER_PIXEL*this->m_u32XRes*this->m_u32YRes);
    memset(this->m_pu32Memory, FBHMI_ZERO, FBHMI_BYTES_PER_PIXEL*this->m_u32XRes*this->m_u32YRes);
    this->m_pFontHandle = NULL;
}

/************************************************************************
* FUNCTION: fbhmi_Surface::~fbhmi_Surface()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Surface::~fbhmi_Surface()
{
	vDeleteSurfaceHandler();
}

/************************************************************************
* FUNCTION: fbhmi_Surface::vDeleteSurfaceHandler()
*
* DESCRIPTION: Delete surface handler pointer
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vDeleteSurfaceHandler()
{
	DEBUG_TRACE("%s", __func__);
    free(this->m_pu32Memory);

    if (NULL != m_pGfxHandle) {
        delete m_pGfxHandle;
        m_pGfxHandle = NULL;
    }

	 if(NULL != m_pFontHandle)
	 {
        delete m_pFontHandle;
		  m_pFontHandle = NULL;
	 }

    if(NULL != m_pSurfaceHandle)
	 {
        delete m_pSurfaceHandle;
		  m_pSurfaceHandle = NULL;
	 }
    else
	 {
        DEBUG_TRACE("Surface pointer is NULL");
	 }
}

/************************************************************************
* FUNCTION: fbhmi_Surface::vClearSurface()
*
* DESCRIPTION: Clear surface
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vClearSurface()
{
	DEBUG_TRACE("%s", __func__);
    vSetColor(FBHMI_COLOR_BLACK);
    vFillRectangle(FBHMI_SCREEN_DEF_X, FBHMI_SCREEN_DEF_Y,
        m_u32XRes, m_u32YRes);
}


/************************************************************************
* FUNCTION: fbhmi_Surface::fbhmi_vSetPixel()
*
* DESCRIPTION: Set a given pixel with color
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vSetPixel(_tU32 u32x, _tU32 u32y)
{
    if (u32x >= this->m_u32XRes)
        return;
    if (u32y >= this->m_u32YRes)
        return;

    // multiply by 4 is done by pointer arithmetics
    _tU32 u32Addr = this->m_u32XRes*u32y + u32x;
    *(this->m_pu32Memory + u32Addr) = m_tclColor.getARGB();
}


/************************************************************************
* FUNCTION: fbhmi_Surface::vDrawLine()
*
* DESCRIPTION: Set a given pixel with color
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vDrawLine(_tU32 u32x1, _tU32 u32y1, _tU32 u32x2, _tU32 u32y2)
{
    _tS32 s32Dy = u32y2 - u32y1;
    _tS32 s32Dx = u32x2 - u32x1;
    _tS32 s32StepX, s32StepY, s32Fraction;
	DEBUG_TRACE("%s", __func__);

    if (s32Dy < 0) { s32Dy = -s32Dy; s32StepY = -1; } else {s32StepY = 1; }
    if (s32Dx < 0) { s32Dx = -s32Dx; s32StepX = -1; } else {s32StepX = 1; }

    s32Dy <<= 1;
    s32Dx <<= 1;

    vSetPixel(u32x1, u32y1);
    if (s32Dx > s32Dy) {
        s32Fraction = s32Dy - (s32Dx >> 1);
        while (u32x1 != u32x2) {
            if (s32Fraction >= 0) {
                u32y1 += s32StepY;
                s32Fraction -= s32Dx;
            }
            u32x1 += s32StepX;
            s32Fraction += s32Dy;
            vSetPixel(u32x1, u32y1);
        }
    } else {
        s32Fraction = s32Dx - (s32Dy >> 1);
        while (u32y1 != u32y2) {
            if (s32Fraction >=0) {
                u32x1 += s32StepX;
                s32Fraction -= s32Dy;
            }
            u32y1 += s32StepY;
            s32Fraction += s32Dx;
            vSetPixel(u32x1, u32y1);
        }
    }
}


/************************************************************************
* FUNCTION: fbhmi_Surface::vFillRectangle()
*
* DESCRIPTION: Fill rectangle with given color
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vFillRectangle(_tU32 u32x, _tU32 u32y, _tU32 u32Width, _tU32 u32Height)
{
	DEBUG_TRACE("x:%u Y:%u W:%u H:%u RGB: 0x%04X",u32x, u32y, u32Width, u32Height, m_tclColor.getARGB());
    // TODO: if rescalable set: do rescale
    if (u32x >= this->m_u32XRes)
        return;
    if (u32y >= this->m_u32YRes)
        return;

    _tU32 u32lx, u32ly;

    for (u32ly=0; u32ly < u32Height; u32ly++) {
        for (u32lx=0; u32lx < u32Width; u32lx++) {

            if ( (u32x+u32lx) >= this->m_u32XRes)
                continue;
            if ( (u32y+u32ly) >= this->m_u32YRes)
                continue;

            _tS32 s32Addr = this->m_u32XRes*(u32y + u32ly) + u32x + u32lx;
            *(this->m_pu32Memory + s32Addr) = m_tclColor.getARGB();
        }
    }
}

/************************************************************************
* FUNCTION: fbhmi_Surface::vDrawChar()
*
* DESCRIPTION: Draw given character using current font
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vDrawChar(_tU32 u32Ch, _tU32 u32x, _tU32 u32y)
{
	DEBUG_TRACE("%s", __func__);
    if (this->m_pFontHandle == NULL) {
        // TODO: gub3kor: load default font
        return;
    }

    if (u32x >= m_u32XRes)
        return;
    if (u32y >= m_u32YRes)
        return;

    m_pFontHandle->vDrawChar(this, u32Ch, u32x, u32y);
};

/************************************************************************
* FUNCTION: fbhmi_Surface::u32DrawStringNormal()
*
* DESCRIPTION: Draw given character using current font
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tU32 fbhmi_Surface::u32DrawStringNormal(_tU32 *pu32Wbuf, _tU32 u32Xx, _tU32 u32Yy)
{
    _tU32 stLen = fbhmi_Utility::u32UstrLen(pu32Wbuf);

	for (_tU32 stx=0; stx < stLen; stx++)
    {
        _tU32 u32Advance;
        _tBool bDraw;
        u32Advance = fbhmi_Utility::u32GetPresentionAdvance(*(pu32Wbuf+stx), bDraw);
		if (bDraw) {
			DEBUG_TRACE("Ch: %u, Len: %u, Advance: %u", pu32Wbuf[stx], stLen, u32Advance);
            fbhmi_Surface::m_pSurfaceHandle->vDrawChar(pu32Wbuf[stx], u32Xx, u32Yy);
		}
        u32Xx += u32Advance;
    }

    return u32Xx;
}

/************************************************************************
* FUNCTION: fbhmi_Surface::vDrawString()
*
* DESCRIPTION: Draw given character using current font
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vDrawString(_tU32 *pU32Wbuf, _tU32 u32Xx, _tU32 u32Yy)
{
    vector<_tU32 *> vecParts;
	DEBUG_TRACE("%s", __func__);
    fbhmi_Utility::vSplitStringDirection(pU32Wbuf,vecParts);

    for (_tU32 u32x=0; u32x<vecParts.size(); u32x++) {

        _tU32 *pu32Part = vecParts[u32x];
        if (pu32Part[0] == 0) {
            free(pu32Part);
            continue;
        }
        u32Xx = u32DrawStringNormal(pu32Part, u32Xx, u32Yy);
        free(pu32Part);
    }
}


/************************************************************************
* FUNCTION: fbhmi_Surface::vDrawToFrameBuffer()
*
* DESCRIPTION: Draw the Surface Memory to Framebuffer
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Surface::vDrawToFrameBuffer()
{
    // Full update
    _tU32 u32y = 0, u32x = 0;
	DEBUG_TRACE("%s", __func__);
    for (u32y = 0; u32y < m_u32YRes; u32y++)
    {
        for (u32x = 0; u32x < m_u32XRes; u32x++)
        {
            _tU32 u32Color = *(m_pu32Memory + u32y*m_u32XRes + u32x);
            if(NULL != m_pGfxHandle)
                m_pGfxHandle->vSetPixel(u32x, u32y, u32Color);
        }
    }
}
