/************************************************************************
 * File: fbhmi_QueueHandler.cpp
 * SW-Component: 
 * Description: Manages the queue for handling HMI events
 * Author:
 *	Rohit.ChannappiahRaman@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 06.06.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
 * 14.06.2013 - Adapted for first release - Gururaj.B@in.bosch.com
 ***********************************************************************/
#include <stdlib.h>
#include "fbhmi_QueueHandler.h"


/************************************************************************
* FUNCTION: fbhmi_QueueHandler::fbhmi_QueueHandler()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_QueueHandler::fbhmi_QueueHandler()
{
	pthread_mutex_init(&mq_mutex, NULL);
	pthread_cond_init (&mq_count, NULL);
	m_isQLocked = false;
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::~fbhmi_QueueHandler()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_QueueHandler::~fbhmi_QueueHandler()
{
	DEBUG_TRACE("%s", __func__);
	while(!bIsQueueEmpty())
	{
		fbhmi_Request* pReq = m_qRequestQueue.front();
		m_qRequestQueue.pop();
		delete pReq;
	}
	
	pthread_cond_destroy(&mq_count);
	pthread_mutex_destroy(&mq_mutex);
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::pQueueFront()
*
* DESCRIPTION: Pop front element and return
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Request* fbhmi_QueueHandler::pQueueFront()
{
	fbhmi_Request * pReq = m_qRequestQueue.front();
	DEBUG_TRACE("Request: %u Screen: %u Widget: %u",
		pReq->m_u16RequestID, pReq->m_u16ScreenID, pReq->m_u16WidgetID);

	return pReq;
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::bIsQueueEmpty()
*
* DESCRIPTION: Check if Queue is empty
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tBool fbhmi_QueueHandler::bIsQueueEmpty() const
{
	return  m_qRequestQueue.empty();
}


/************************************************************************
* FUNCTION: fbhmi_QueueHandler::vPopQueue()
*
* DESCRIPTION: Pop and discard elements
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_QueueHandler::vPopQueue()
{
	DEBUG_TRACE("%s", __func__);
	 m_qRequestQueue.pop();
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::bPushQueue(fbhmi_Request* pReq)
*
* DESCRIPTION: Push object to the queue
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tBool fbhmi_QueueHandler::bPushQueue(fbhmi_Request* pReq)
{
	pthread_mutex_lock(&mq_mutex);

	DEBUG_TRACE("Request: %u Screen: %u Widget: %u",
	pReq->m_u16RequestID, pReq->m_u16ScreenID, pReq->m_u16WidgetID);
	m_qRequestQueue.push(pReq);
	if(m_isQLocked)
		pthread_cond_signal(&mq_count);
	pthread_mutex_unlock(&mq_mutex);
	return true;
}

/************************************************************************
* FUNCTION: fbhmi_Surface::pGetInstance()
*
* DESCRIPTION: Create fbhmi_Surface singleton instance
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tU32 fbhmi_QueueHandler::u32GetQueueSize() const
{
	DEBUG_TRACE("%s", __func__);
	return m_qRequestQueue.size();
}
/************************************************************************
* FUNCTION: fbhmi_QueueHandler::vWaitMQ()
*
* DESCRIPTION: waits on message Queue
*
* PARAMETER: None
*
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_QueueHandler::vWaitMQ()
{
	_tBool bRet=false;
	if(!pthread_cond_wait(&mq_count, &mq_mutex))
		bRet=true;
	return bRet;	
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::vLockMQ()
*
* DESCRIPTION: locks message Queue
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tBool fbhmi_QueueHandler::vLockMQ()
{
	_tBool bRet=false;
	if(!pthread_mutex_lock(&mq_mutex))
		bRet=true;
	return bRet;	
}

/************************************************************************
* FUNCTION: fbhmi_QueueHandler::vUnlockMQ()
*
* DESCRIPTION: unlocks message Queue
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tBool fbhmi_QueueHandler::vUnlockMQ()
{
	_tBool bRet=false;
	if(!pthread_mutex_unlock(&mq_mutex))
		bRet=true;
	return bRet;	
}
