/************************************************************************
 * File: fbhmi_FbHandler.cpp
 * SW-Component: ASCII HMI for recovery download
 *
 * Description:
 *   This is a singleton class used to initialize and use the Linux
 *   framebuffer for displaying characters.
 * Author:
 *   Gururaj.B@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 21.05.2013 - Initial version - Gururaj.B@in.bosch.com
 ************************************************************************/

//--------------------------------------------------------------------------
// includes
//--------------------------------------------------------------------------
#include <iostream>
using namespace std;
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include "fbhmi_GfxHandler.h"
#include <linux/kd.h>
#include <stdlib.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#define SCREEN_X_RESOLUTION	400
#define SCREEN_Y_RESOLUTION	240
#define FILE_TO_CHECK			4

#define U32_MAX_PATHLENGTH			512
#define UPPER_BYTE_MASK				0x00FF

//--------------------------------------------------------------------------
// Initialize static variables
//--------------------------------------------------------------------------
fbhmi_GfxHandler* fbhmi_GfxHandler::m_pGfxHandle=NULL;

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::fbhmi_FbHandler()
 *
 * DESCRIPTION: private Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_GfxHandler::fbhmi_GfxHandler()
{
  DEBUG_TRACE("%s", __func__);
  m_u32YRes=0;
  m_u32XRes=0;
  memset(&m_stGfxFix, 0, sizeof(m_stGfxFix));
  memset(&m_stGfxVar, 0, sizeof(m_stGfxVar));
  m_pGfxMemory=NULL;
  m_pGfxMemReal=NULL;
  m_u64GfxMemOffset=0;
  m_u32GfxPixelBytes=0;
  m_bVirtualGfx=false;

  signal(SIGUSR1, SigHandler);
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::~fbhmi_FbHandler()
 *
 * DESCRIPTION: Destructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_GfxHandler::~fbhmi_GfxHandler()
{
  DEBUG_TRACE("%s", __func__);
  if(NULL!=m_pGfxHandle)
    delete m_pGfxHandle;
  m_pGfxHandle=NULL;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::getInstance()
 *
 * DESCRIPTION: Static function to return instance of the class.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Pointer to instance of the class
 *************************************************************************/
fbhmi_GfxHandler* fbhmi_GfxHandler::pGetInstance()
{
  DEBUG_TRACE("%s", __func__);
  return m_pGfxHandle;
}


/************************************************************************
 * FUNCTION: fbhmi_GfxHandler::SigHandler(int Sig)
 *
 * DESCRIPTION: Signal handler, so one can use "kill -10 <HMI_PROC_ID>
 *              to generate a screen shot as /tmp/screenshot.tga
 *
 * PARAMETER: Signal
 *
 * RETURNVALUE: void
 *************************************************************************/
void fbhmi_GfxHandler::SigHandler(int Sig)
{
  ScreenShot("/tmp/screenshot.tga");
  return;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vDisableCursor(void)
 *
 * DESCRIPTION: IDisable cursor on framebuffer console
 *
 * PARAMETER: void
 *
 * RETURNVALUE: void
 *************************************************************************/
void fbhmi_GfxHandler::vDisableCursor()
{
  DEBUG_TRACE("%s", __func__);
  _tS32 s32ttyFd;

  // Disable cursor on framebuffer console
  s32ttyFd=open("/dev/tty0", O_WRONLY);

  if(s32ttyFd<0)
  {
    DEBUG_TRACE("Failed to open /dev/tty0!!!");
  }
  else
  {
    ioctl(s32ttyFd, KDSETMODE, KD_GRAPHICS);
    close(s32ttyFd);
  }
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vClearScreen()
 *
 * DESCRIPTION: Clear Screen
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void fbhmi_GfxHandler::vClearScreen()
{
  DEBUG_TRACE("%s", __func__);
  _tU32 u32y;
  _tU32 u32Yres;
  _tU32 u32Xres;

  if(!m_bVirtualGfx)
  {
    u32Yres=m_stGfxVar.yres;
    u32Xres=m_stGfxVar.xres;
  }
  else
  {
    u32Yres=m_stGfxVar.yres_virtual;
    u32Xres=m_stGfxVar.xres_virtual;
  }

  for(u32y=m_stGfxVar.yoffset; u32y<u32Yres; u32y++)
  {
    memset(m_pGfxMemory+u32y*m_stGfxFix.line_length+m_stGfxVar.xoffset*m_u32GfxPixelBytes, 0, u32Xres*m_stGfxVar.bits_per_pixel/8);
  }
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vSetPixel(_tS32 x, _tS32 y, uint32_t u32Rgba)
 *
 * DESCRIPTION: Set a framebuffer pixel with the color information supplied.
 *
 * PARAMETER: 	1. x = x co-ordinate
 * 				2. y = y co-ordinate
 * 				3. u32 = The color value with alpha blending.
 *
 * RETURNVALUE: None
 ************************************************************************/
void fbhmi_GfxHandler::vSetPixel(_tU32 u32x, _tU32 u32y, _tU32 u32Rgba)
{
  _tU8 u8Red, u8Green, u8Blue, u8Alpha;

  u8Alpha=(u32Rgba>>24)&0xFF;
  u8Red=(u32Rgba>>16)&0xFF;
  u8Green=(u32Rgba>>8)&0xFF;
  u8Blue=u32Rgba&0xFF;

  if(!m_bVirtualGfx)
  {
    if(u32x>=m_stGfxVar.xres)
      return;
    if(u32y>=m_stGfxVar.yres)
      return;
  }
  else
  {
    if(u32x>=m_stGfxVar.xres_virtual)
      return;
    if(u32y>=m_stGfxVar.yres_virtual)
      return;
  }

  _tU32 u32Char=
      ((u8Alpha>>(8-m_stGfxVar.transp.length))<<m_stGfxVar.transp.offset)|
      ((u8Red>>(8-m_stGfxVar.red.length))<<m_stGfxVar.red.offset)|
      ((u8Green>>(8-m_stGfxVar.green.length))<<m_stGfxVar.green.offset)|
      ((u8Blue>>(8-m_stGfxVar.blue.length))<<m_stGfxVar.blue.offset);

  _tU64 u64Offset=(u32y+m_stGfxVar.yoffset)*m_stGfxFix.line_length+
      (u32x+m_stGfxVar.xoffset)*m_u32GfxPixelBytes;
  _tU16 *u16Dest=(_tU16 *)(m_pGfxMemory+u64Offset);
  _tU64 *u64Dest=(_tU64 *)(m_pGfxMemory+u64Offset);
  _tU8 *u8Dest=(_tU8*)(m_pGfxMemory+u64Offset);

  switch(m_stGfxVar.bits_per_pixel)
  {
  case 15:
  case 16:
    *u16Dest=u32Char;
    break;
  case 24:
    *u8Dest=u32Char&0xff;
    *(u8Dest+1)=(u32Char&0xff00)>>8;
    *(u8Dest+2)=(u32Char&0xff0000)>>16;
    break;
  case 32:
    *u64Dest=u32Char;
    break;
  default:
    break;
  }
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vGetPixel(_tU32 u32x, _tU32 u32y)
 *
 * DESCRIPTION: Read a pixel from the framebuffer and return it as
 *              an RGBA value, packed into a single int
 *
 * PARAMETER:   1. x = x co-ordinate
 *        2. y = y co-ordinate
 *
 * RETURNVALUE: RGBA value in the framebuffer at the specified position
 ************************************************************************/
_tU32 fbhmi_GfxHandler::vGetPixel(_tU32 u32x, _tU32 u32y)
{
  _tU8 u8Red, u8Green, u8Blue, u8Alpha;

  if(!m_bVirtualGfx)
  {
    if(u32x>=m_stGfxVar.xres)
      return 0;
    if(u32y>=m_stGfxVar.yres)
      return 0;
  }
  else
  {
    if(u32x>=m_stGfxVar.xres_virtual)
      return 0;
    if(u32y>=m_stGfxVar.yres_virtual)
      return 0;
  }

  _tU64 u64Offset=(u32y+m_stGfxVar.yoffset)*m_stGfxFix.line_length+(u32x+m_stGfxVar.xoffset)*m_u32GfxPixelBytes;
  _tU16 *u16Dest=(_tU16 *)(m_pGfxMemory+u64Offset);
  _tU64 *u64Dest=(_tU64 *)(m_pGfxMemory+u64Offset);
  _tU8 *u8Dest=(_tU8*)(m_pGfxMemory+u64Offset);

  _tU32 u32Char=0;
  switch(m_stGfxVar.bits_per_pixel)
  {
  case 15:
  case 16:
    u32Char=*u16Dest;
    break;
  case 24:
    u32Char=*u8Dest&0xff;
    u32Char|=*(u8Dest+1)<<8;
    u32Char|=*(u8Dest+2)<<16;
    u32Char|=*(u8Dest+3)<<24;
    break;
  case 32:
    u32Char=*u64Dest;
    break;
  default:
    break;
  }

  u8Alpha=(u32Char>>m_stGfxVar.transp.offset)<<(8-m_stGfxVar.transp.length);
  u8Red  =(u32Char>>m_stGfxVar.red.offset)   <<(8-m_stGfxVar.red.length);
  u8Green=(u32Char>>m_stGfxVar.green.offset) <<(8-m_stGfxVar.green.length);
  u8Blue =(u32Char>>m_stGfxVar.blue.offset)  <<(8-m_stGfxVar.blue.length);

  return ((u8Alpha&0xFF)<<24)|((u8Red&0xFF)<<16)|((u8Green&0xFF)<<8)|(u8Blue&0xFF);
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::u32GetXRes()
 *
 * DESCRIPTION: Get display resolution X-coordinate
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Resolution in number of pixels
 ************************************************************************/
_tU32 fbhmi_GfxHandler::u32GetXRes()
{
  DEBUG_TRACE("%s", __func__);
  return m_u32XRes;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::u32GetYRes()
 *
 * DESCRIPTION: Get display resolution Y-coordinate
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Resolution in number of pixels
 ************************************************************************/
_tU32 fbhmi_GfxHandler::u32GetYRes()
{
  DEBUG_TRACE("%s", __func__);
  return m_u32YRes;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vReadFileContent
 *
 * DESCRIPTION: This function read the file content and put it into
 *					the buffer passed as parameter.
 *
 * PARAMETER: const char *, _tU32
 *
 * RETURNVALUE: _tS32
 ************************************************************************/
_tS32 fbhmi_GfxHandler::s32ReadFileContent(char * pchFilePath, char * acTempBuff)
{
  DEBUG_TRACE("%s", __func__);
  _tS32 s32FileSize=-1;
  _tS32 s32ReadByte=-1;
  FILE * pFilePtr;

  DEBUG_TRACE("%s", __func__);

  if((NULL!=pchFilePath) && (NULL!=acTempBuff))
  {
    //open the file
    pFilePtr=fopen(pchFilePath, "r");

    if(pFilePtr==NULL)
    {
      DEBUG_TRACE("Error : Failed to open %s file - %s", pchFilePath, strerror(errno));
      return s32ReadByte;
    }

    //read the size of the file
    fseek(pFilePtr, 0, SEEK_END);
    s32FileSize=ftell(pFilePtr);
    rewind(pFilePtr);

    //read the file contents
    s32ReadByte=fread(acTempBuff, 1, s32FileSize, pFilePtr);

    //error case
    if(s32ReadByte<s32FileSize)
    {
      DEBUG_TRACE("ERROR: Can not read");
      fclose(pFilePtr);
      return -1;
    }

    acTempBuff[s32FileSize+1]='\0';
    //close the file
    fclose(pFilePtr);
  }

  return s32ReadByte;
}

/************************************************************************
 * FUNCTION: fbhmi_GfxHandler::ScreenShot(const char *FileName)
 *
 * DESCRIPTION: This function reads the framebuffer and saves its content
 *              to a file in .tga format.
 *
 * PARAMETER: FileName    Path and filename for the targa file
 *
 * RETURNVALUE:
 ************************************************************************/
void fbhmi_GfxHandler::ScreenShot(const char *FileName)
{
  fbhmi_GfxHandler  *inst=pGetInstance();
  unsigned char     tga_header[18];
  _tU32             color;
  unsigned char     r, g, b, a;
  _tU32             *line_buff;
  FILE              *file;

  memset(tga_header, 0, sizeof(tga_header));
  tga_header[2]=2;                          // Uncompressed RGBA Data
  tga_header[12]=inst->m_u32XRes&0xff;      // Low Byte of Width
  tga_header[13]=(inst->m_u32XRes>>8)&0xff; // High Byte of Width
  tga_header[14]=inst->m_u32YRes&0xff;      // Low Byte of Height
  tga_header[15]=(inst->m_u32YRes>>8)&0xff; // High Byte of Height
  tga_header[16]=32;                        // Bits per Pixel
  tga_header[17]=0x20;                      // Data is stored top to bottom

  if((file=fopen(FileName, "wb"))==NULL)
    return;
  fwrite(tga_header, 1, sizeof(tga_header), file);

  line_buff=(_tU32 *)malloc(sizeof(_tU32)*inst->m_u32XRes);
  for(int y=0; y<inst->m_u32YRes; y++)
  {
    for(int x=0; x<inst->m_u32XRes; x++)
    {
      color=inst->vGetPixel(x, y);
      r=color&0xff;
      g=(color>>8)&0xff;
      b=(color>>16)&0xff;
      a=0xff;
      line_buff[x]=(r<<16)|(g<<8)|b|(a<<24);
    }
    fwrite(line_buff, inst->m_u32XRes, sizeof(_tU32), file);
  }
  free(line_buff);
  fclose(file);
}
