/************************************************************************
 * File: fbhmi_FbHandler.cpp
 * SW-Component: ASCII HMI for recovery download
 *
 * Description:
 *   This is a singleton class used to initialize and use the Linux
 *   framebuffer for displaying characters.
 * Author:
 *   Gururaj.B@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 21.05.2013 - Initial version - Gururaj.B@in.bosch.com
 ************************************************************************/

//--------------------------------------------------------------------------
// includes
//--------------------------------------------------------------------------
#include <iostream>
using namespace std;
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <linux/fb.h>
#include "fbhmi_FbHandler.h"
#include <linux/kd.h>
#include <stdlib.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>

#define SCREEN_X_RESOLUTION	400
#define SCREEN_Y_RESOLUTION	240
#define FILE_TO_CHECK			4

#define DTB_CONFIG_DISPLAY_PATH	"/proc/device-tree/soc/display@di0/"
#define DTB_ID							"/proc/device-tree/dtbid"

#define U32_MAX_PATHLENGTH			512
#define VWMIB_B0_DTB_ID				0x26
#define VWMIB_B_DTB_ID				0x28
#define UPPER_BYTE_MASK				0x00FF

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::fbhmi_FbHandler()
 *
 * DESCRIPTION: private Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_FbHandler::fbhmi_FbHandler() :
    fbhmi_GfxHandler()
{
  DEBUG_TRACE("%s", __func__);
  // Initialize framebuffer device
  m_s32FdFrameBuffer=-1;
  bInitFrameBuffer(FBHMI_FRAMEBUFF_DEVICE);
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::~fbhmi_FbHandler()
 *
 * DESCRIPTION: Destructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_FbHandler::~fbhmi_FbHandler()
{
  DEBUG_TRACE("%s", __func__);
  bCleanupFrameBuffer();
  fbhmi_GfxHandler::m_pGfxHandle=NULL;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::getInstance()
 *
 * DESCRIPTION: Static function to return instance of the class.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Pointer to instance of the class
 *************************************************************************/
fbhmi_GfxHandler* fbhmi_FbHandler::pCreateInstance()
{
  DEBUG_TRACE("%s", __func__);
  if(NULL==fbhmi_GfxHandler::m_pGfxHandle)
  {
    fbhmi_GfxHandler::m_pGfxHandle=new fbhmi_FbHandler();
  }
  return fbhmi_GfxHandler::m_pGfxHandle;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::bInitFrameBuffer(_tChar *fbName)
 *
 * DESCRIPTION: Initialize the framebuffer device and structures
 *
 * PARAMETER: Name of the framebuffer device
 *
 * RETURNVALUE: 1 = SUCCESS, 0 = FAILURE
 *************************************************************************/
_tBool fbhmi_FbHandler::bInitFrameBuffer(const _tChar *pFbName)
{
  DEBUG_TRACE("%s", __func__);
  _tBool bResult=false;
  _tU64 u64PageMask;

  // Disable cursor on framebuffer console
  vDisableCursor();

  // Open the frame buffer device
  m_s32FdFrameBuffer=open(pFbName, O_RDWR);

  // Get Screen info from fb driver
  if(bGetFbInfo()==false)
  {
    return bResult;
  }

  // Only packed pixels supported.
  if(m_stGfxFix.type!=FB_TYPE_PACKED_PIXELS)
  {
    return bResult;
  }

  //check for compiler flag enable
#ifdef VARIANT_S_FTR_ENABLE_DTB_CONFIG

  //verify the DTB file
  if ( true == bVerifyDtbConfig() )
  {
    //take display config from DTBs
    vOverrideDisplayConfig();
  }

#endif
  //TODO: jha1kor: Workaround for Recovery download screen shifted in VW B sample
  _tU32 u32RetValue=u32ReadDtbID();

  //compare the board ID with VWMIB B0 and B target
  if(VWMIB_B0_DTB_ID==u32RetValue || VWMIB_B_DTB_ID==u32RetValue)
  {
    m_stGfxVar.xoffset=m_stGfxVar.xres-SCREEN_X_RESOLUTION;
    m_stGfxVar.xres=SCREEN_X_RESOLUTION;
    m_stGfxVar.yres=SCREEN_Y_RESOLUTION;
  }

  // Check number of bits per pixel supported.
  if(m_stGfxVar.bits_per_pixel!=15 && m_stGfxVar.bits_per_pixel!=16 && m_stGfxVar.bits_per_pixel!=24 && m_stGfxVar.bits_per_pixel!=32)
  {
    return bResult;
  }

  if(m_stGfxVar.bits_per_pixel==15)
    m_u32GfxPixelBytes=2;
  else
    m_u32GfxPixelBytes=m_stGfxVar.bits_per_pixel/8;

  // Map framebuffer memory to new memory segment
  // (first parameter to mmap is null)
  u64PageMask=getpagesize()-1;
  m_u64GfxMemOffset=m_stGfxFix.smem_start&u64PageMask;

  m_pGfxMemReal=(_tU8*)mmap(NULL, m_stGfxFix.smem_len+m_u64GfxMemOffset, PROT_READ|PROT_WRITE, MAP_SHARED, m_s32FdFrameBuffer, 0);

  if(-1L==(long)m_pGfxMemReal)
  {
    return bResult;
  }

  m_pGfxMemory=m_pGfxMemReal+m_u64GfxMemOffset;

  //assign x-resolution and y-resolution
  if(!m_bVirtualGfx)
  {
    m_u32YRes=m_stGfxVar.yres;
    m_u32XRes=m_stGfxVar.xres;
  }
  else
  {
    m_u32YRes=m_stGfxVar.yres_virtual;
    m_u32XRes=m_stGfxVar.xres_virtual;
  }

  //Clear the screen
  vClearScreen();
  return true;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::bGetFbInfo(void)
 *
 * DESCRIPTION: IDisable cursor on framebuffer console
 *
 * PARAMETER: void
 *
 * RETURNVALUE: _tBool
 *************************************************************************/
_tBool fbhmi_FbHandler::bGetFbInfo()
{
  _tBool bResult=false;

  if(m_s32FdFrameBuffer==-1)
  {
    return bResult;
  }

  // Get variable and fixed screen info
  if((ioctl(m_s32FdFrameBuffer, FBIOGET_VSCREENINFO, &m_stGfxVar)!=-1) && (ioctl(m_s32FdFrameBuffer, FBIOGET_FSCREENINFO, &m_stGfxFix)!=-1))
  {
    bResult=true;
  }

  return bResult;

}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::bCleanupFrameBuffer()
 *
 * DESCRIPTION: De-Initialize and clean up framebuffer structures
 *
 * PARAMETER: None
 *
 * RETURNVALUE: 1 = SUCCESS, 0 = FAILURE
 *************************************************************************/
_tBool fbhmi_FbHandler::bCleanupFrameBuffer()
{
  DEBUG_TRACE("%s", __func__);
  munmap(m_pGfxMemReal, m_stGfxFix.smem_len+m_u64GfxMemOffset);
  close(m_s32FdFrameBuffer);
  return true;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vOverrideDisplayConfig
 *
 * DESCRIPTION: Override resolution and offset from DTBs
 *
 * PARAMETER: None
 *
 * RETURNVALUE: void
 ************************************************************************/
void fbhmi_FbHandler::vOverrideDisplayConfig()
{
  DIR *dir;
  struct dirent * rDirent;
  _tChar acDirFilePathName[U32_MAX_PATHLENGTH];
  _tChar acTempBuff[U32_MAX_PATHLENGTH];

  DEBUG_TRACE("%s", __func__);

  // Open the display@di0 directory
  dir=opendir(DTB_CONFIG_DISPLAY_PATH);
  if(!dir)
  {
    // can not open the dir
    DEBUG_TRACE("Can not open %s", DTB_CONFIG_DISPLAY_PATH);
    return;
  }

  //read each content of the directory
  while((rDirent=readdir(dir))!=NULL)
  {
    if((!strcmp(rDirent->d_name, ".")) || (!strcmp(rDirent->d_name, "..")))
    {
      continue;
    }

    //read each file
    memset(acDirFilePathName, 0, U32_MAX_PATHLENGTH);
    sprintf(acDirFilePathName, "%s/%s", DTB_CONFIG_DISPLAY_PATH, rDirent->d_name);

    if(s32ReadFileContent(acDirFilePathName, acTempBuff)==-1)
    {
      continue;
    }

    //convert the value into u32 format
    _tU32 u32Value=u32ConvertRaw32(acTempBuff);

    vSetDtbsConfigure(rDirent->d_name, u32Value);

    DEBUG_TRACE("%s Contents are: %d", rDirent->d_name, u32Value);

  }

  //Close the directory
  closedir(dir);

}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::bVerifyDtbConfig
 *
 * DESCRIPTION: verify all four parameters are present in DTBs
 *
 * PARAMETER: None
 *
 * RETURNVALUE: _tBool
 ************************************************************************/
_tBool fbhmi_FbHandler::bVerifyDtbConfig()
{
  DIR *dir;
  struct dirent * rDirent;
  _tBool bRetVal=false;
  _tU8 u8FileCount=0;

  DEBUG_TRACE("%s", __func__);

  // Open the display@di0 directory
  dir=opendir(DTB_CONFIG_DISPLAY_PATH);
  if(!dir)
  {
    // can not open the dir
    DEBUG_TRACE("Can not open %s", DTB_CONFIG_DISPLAY_PATH);
    return bRetVal;
  }

  //check xres, yres, xoffset, yoffset are preset in DTBs
  while((rDirent=readdir(dir))!=NULL)
  {
    if((!strcmp(rDirent->d_name, "xres")) || (!strcmp(rDirent->d_name, "yres")) || (!strcmp(rDirent->d_name, "xoffset")) || (!strcmp(rDirent->d_name, "yoffset")))
    {
      ++u8FileCount;
    }

    if(u8FileCount==FILE_TO_CHECK)
    {
      bRetVal=true;
      break;
    }

  }

  //Close the directory
  closedir(dir);

  return bRetVal;
}


/************************************************************************
 * FUNCTION: fbhmi_FbHandler::u32ConvertRaw32
 *
 * DESCRIPTION: convert value
 *
 * PARAMETER: const char *
 *
 * RETURNVALUE: _tU32 Value
 ************************************************************************/
_tU32 fbhmi_FbHandler::u32ConvertRaw32(const char * pchTempStr)
{
  _tU32 u32Value=0;
  DEBUG_TRACE("%s", __func__);

  if(NULL!=pchTempStr)
  {
    for(_tU8 u8Index=0; u8Index<4; ++u8Index)
    {
      _tU32 u32Val=(_tU32)pchTempStr[3-u8Index];
      u32Val&=0xFF;
      u32Value+=u32Val<<u8Index*8;
    }
  }
  return u32Value;
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::vSetDtbsConfigure
 *
 * DESCRIPTION: This function override the display configuration value.
 *
 * PARAMETER: const char *, _tU32
 *
 * RETURNVALUE: void
 ************************************************************************/
void fbhmi_FbHandler::vSetDtbsConfigure(const char * pchTempStr, _tU32 u32TempVal)
{
  DEBUG_TRACE("%s", __func__);

  if(NULL!=pchTempStr)
  {
    if(!strcmp(pchTempStr, "xres"))
    {
      m_stGfxVar.xres=u32TempVal;
    }
    else if(!strcmp(pchTempStr, "yres"))
    {
      m_stGfxVar.yres=u32TempVal;
    }
    else if(!strcmp(pchTempStr, "xoffset"))
    {
      m_stGfxVar.xoffset=u32TempVal;
    }
    else if(!strcmp(pchTempStr, "yoffset"))
    {
      m_stGfxVar.yoffset=u32TempVal;
    }
  }
}

/************************************************************************
 * FUNCTION: fbhmi_FbHandler::u32ReadDtbID
 *
 * DESCRIPTION: This function reads the DTB ID.
 *
 * PARAMETER: void
 *
 * RETURNVALUE: _tU32
 ************************************************************************/
_tU32 fbhmi_FbHandler::u32ReadDtbID()
{
  DEBUG_TRACE("%s", __func__);
  _tU32 u32RetValue=0;
  _tChar acTempBuff[U32_MAX_PATHLENGTH];

  //read dtbid file
  if(s32ReadFileContent(const_cast<_tChar*>(DTB_ID), acTempBuff)!=-1)
  {
    //convert the value into u32 format
    u32RetValue=u32ConvertRaw32(acTempBuff);
    u32RetValue&=UPPER_BYTE_MASK;
  }

  return u32RetValue;
}
