/************************************************************************
* File: fbhmi_Screen.cpp
* SW-Component: ASCII HMI for recovery download
*
* Description:
*   Screen class definition. 
* Author:
*   Gururaj.B@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 13.06.2013 - Initial version - Gururaj.B@in.bosch.com
* 01.08.2013 - Fix for text clear issue - Ankur Tank 
* 19.03.2014 - Read BBX from BDF Font for Y-offset- Aditya Kumar Jha
* 08.09.2014 - Implemented Functions to Clear and Draw Screen title- Apoorva K R
* 			 - Changed vUpdateWidget function to implement screen title changes
***********************************************************************/
#include "fbhmi_Screen.h"

/************************************************************************
* FUNCTION: fbhmi_Screen::fbhmi_Screen(fbhmi_Request* pReq)
*
* DESCRIPTION: Constructor 1
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Screen::fbhmi_Screen(fbhmi_Request* pReq)
:fbhmi_Widget(pReq)
{
	m_bRescalable = false;
	DEBUG_TRACE("%s", __func__);
}


/************************************************************************
* FUNCTION: fbhmi_Screen::fbhmi_Screen(...)
*
* DESCRIPTION: Constructor 2
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Screen::fbhmi_Screen(_tU16 u16Id ,\
									_tU32 u32x ,\
									_tU32 u32y ,\
									_tU32 u32Height ,\
									_tU32 u32Width  ,\
									_tU16 u16BdrWidth ,\
									_tU32 u32BgColor ,\
									_tU32 u32FgColor ,\
									_tU32 u32BdrColor ,\
									_tU32 u32FillColor,\
									string sText ,\
									_tU8 u8Align ,\
									_tBool bVisible)
{
	m_bRescalable = false;

	m_u16Id         = u16Id;
	m_u32x          = u32x;
	m_u32y          = u32y;
	m_u32Height     = u32Height;
	m_u32Width      = u32Width;
	m_u16BdrWidth   = u16BdrWidth;
	m_u32BgColor    = u32BgColor;
	m_u32FgColor    = u32FgColor;
	m_u32BdrColor   = u32BdrColor;
	m_u32FillColor  = u32FillColor;
	m_strText       = sText;
	m_u8Align       = u8Align;
	m_bVisible      = bVisible;
	m_u32PbarVal    = FBHMI_WIDGET_DEF_PBARVAL;
	m_u32PbarMax    = FBHMI_WIDGET_DEF_PBARMAX;
	DEBUG_TRACE("%s", __func__);

}

/************************************************************************
* FUNCTION: fbhmi_Screen::fbhmi_Screen(fbhmi_Request* pReq)
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Screen::~fbhmi_Screen()
{
	// Delete all widgets
	for(std::vector<fbhmi_Widget*>::iterator pIt = m_vecChildWidgets.begin(); pIt != m_vecChildWidgets.end(); pIt++)
	{
		delete *pIt;
	}
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: fbhmi_Screen::vDrawWidget()
*
* DESCRIPTION: Draw widget function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vDrawWidget()
{
	//_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth(); changes done to remove Lint Warnings

	if (NULL == m_pSurfaceHandle) {
		DEBUG_TRACE("Surface pointer is NULL");
		return ;
	}

	_tU32 u32FontHeight = m_pSurfaceHandle->m_pFontHandle->u32GetFontHeight();

	if(m_u16BdrWidth > FBHMI_WIDGET_DEF_BWIDTH)
	{
		//Draw Border if present
		m_pSurfaceHandle->vSetColor(m_u32BdrColor);
		m_pSurfaceHandle->vFillRectangle(m_u32x, m_u32y, m_u32Width, m_u32Height);
	}
	// Draw Background
	m_pSurfaceHandle->vSetColor(m_u32BgColor);
	m_pSurfaceHandle->vFillRectangle(m_u32x + m_u16BdrWidth,
		m_u32y + m_u16BdrWidth,
		m_u32Width - m_u16BdrWidth * 2,
		m_u32Height - m_u16BdrWidth * 2);
	if(true == m_bVisible)
	{

		_tU32 u32AlignedX =0, u32AlignedY = 0;
		_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth();
		_tU32 u32TmpTextWidth = (m_u32Width-m_u16BdrWidth*2)/u32FontWidth;

		/*Fix for text clear issue If text size if more than label width
		clip the tail of string and append three dots*/
		if ( m_strText.size() >  u32TmpTextWidth)
		{
			m_strText.resize(u32TmpTextWidth-LABELTXT_TRAILING_DOT_SIZE);
			m_strText.append("...");
		}

		vGetAlignedXY(&u32AlignedX, &u32AlignedY, m_strText.size());

		// Draw background for the title
		m_pSurfaceHandle->vSetColor(m_u32BgColor);
		m_pSurfaceHandle->vFillRectangle(u32AlignedX,
			FBHMI_SCREEN_DEF_TITLE_Y,
			u32FontWidth*m_strText.length(),
			u32FontHeight);

		// Draw Text
		_tU32 au32Buff[(m_strText.size() + 1) * sizeof(_tU32)];

		memset(au32Buff, 0, (m_strText.size()+1) * sizeof(_tU32));
		fbhmi_Utility::vConvlatin9toutf32(const_cast<_tChar*>(m_strText.c_str()), au32Buff, m_strText.size());


		m_pSurfaceHandle->vSetColor(m_u32FgColor);
		m_pSurfaceHandle->vDrawString(au32Buff, u32AlignedX, u32AlignedY);
	}
	m_pSurfaceHandle->vDrawToFrameBuffer();
	DEBUG_TRACE("Label Text: %s", m_strText.c_str());
}

/************************************************************************
* FUNCTION: fbhmi_Screen::vDrawTitle()
*
* DESCRIPTION: Draw Title function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vDrawTitle()
{

	if (NULL == m_pSurfaceHandle) {
		DEBUG_TRACE("Surface pointer is NULL");
		return ;
	}

	_tU32 u32FontHeight = m_pSurfaceHandle->m_pFontHandle->u32GetFontHeight();

	if(true == m_bVisible)
	{

		_tU32 u32AlignedX =0, u32AlignedY = 0;
		_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth();
		_tU32 u32TmpTextWidth = (m_u32Width-m_u16BdrWidth*2)/u32FontWidth;

		/*Fix for text clear issue If text size if more than label width
		clip the tail of string and append three dots*/
		if ( m_strText.size() >  u32TmpTextWidth)
		{
			m_strText.resize(u32TmpTextWidth-LABELTXT_TRAILING_DOT_SIZE);
			m_strText.append("...");
		}

		vGetAlignedXY(&u32AlignedX, &u32AlignedY, m_strText.size());

		// Draw background for the title
		m_pSurfaceHandle->vSetColor(m_u32BgColor);
		m_pSurfaceHandle->vFillRectangle(u32AlignedX,
			FBHMI_SCREEN_DEF_TITLE_Y,
			u32FontWidth*m_strText.length(),
			u32FontHeight);

		// Draw Text
		_tU32 au32Buff[(m_strText.size() + 1) * sizeof(_tU32)];

		memset(au32Buff, 0, (m_strText.size()+1) * sizeof(_tU32));
		fbhmi_Utility::vConvlatin9toutf32(const_cast<_tChar*>(m_strText.c_str()), au32Buff, m_strText.size());


		m_pSurfaceHandle->vSetColor(m_u32FgColor);
		m_pSurfaceHandle->vDrawString(au32Buff, u32AlignedX, u32AlignedY);
	}
	m_pSurfaceHandle->vDrawToFrameBuffer();
	DEBUG_TRACE("Title Text: %s", m_strText.c_str());
}

/************************************************************************
* FUNCTION: fbhmi_Screen::vClearTitle()
*
* DESCRIPTION: Clear Title function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vClearTitle()
{

	if (NULL == m_pSurfaceHandle) {
		DEBUG_TRACE("Surface pointer is NULL");
		return ;
	}

	_tU32 u32FontHeight = m_pSurfaceHandle->m_pFontHandle->u32GetFontHeight();

	if(true == m_bVisible)
	{

		_tU32 u32AlignedX =0, u32AlignedY = 0;
		_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth();
		_tU32 u32TmpTextWidth = (m_u32Width-m_u16BdrWidth*2)/u32FontWidth;

		/*Fix for text clear issue If text size if more than label width
		clip the tail of string and append three dots*/
		if ( m_strText.size() >  u32TmpTextWidth)
		{
			m_strText.resize(u32TmpTextWidth-LABELTXT_TRAILING_DOT_SIZE);
			m_strText.append("...");
		}

		vGetAlignedXY(&u32AlignedX, &u32AlignedY, m_strText.size());

		// Draw background for the title
		m_pSurfaceHandle->vSetColor(m_u32BgColor);
		m_pSurfaceHandle->vFillRectangle(u32AlignedX,
			FBHMI_SCREEN_DEF_TITLE_Y,
			u32FontWidth*m_strText.length(),
			u32FontHeight);
	}
	m_pSurfaceHandle->vDrawToFrameBuffer();
	DEBUG_TRACE("Clear Title Text");
}


/************************************************************************
* FUNCTION: fbhmi_Screen::bAddChilvGetAlignedXY
*
* DESCRIPTION: Draw widget function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vGetAlignedXY(_tU32* pu32AlndX, _tU32 *pu32AlndY, _tU32 u32Strlen)
{

	if (NULL == m_pSurfaceHandle) {
		DEBUG_TRACE("Surface pointer is NULL");
		return ;
	}
	//To remove lint 
	u32Strlen = u32Strlen;

	_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth();
	_tU32 u32FontHeight = m_pSurfaceHandle->m_pFontHandle->u32GetFontHeight();

	//jha1kor: Read BBX from BDF Font for Y-offset
	_tS32 s32FontYOffset = m_pSurfaceHandle->m_pFontHandle->s32GetFontYPlus();

	*pu32AlndX = m_u32x;
	
	*pu32AlndY = FBHMI_SCREEN_DEF_TITLE_Y+u32FontHeight+s32FontYOffset;

	switch(m_u8Align)
	{
	case FBHMI_TXT_ALIGN_LEFT:
		break;
	case FBHMI_TXT_ALIGN_RIGHT:
		*pu32AlndX = (m_u32Width) - (u32FontWidth * m_strText.size());
		break;
	case FBHMI_TXT_ALIGN_CENTER:
	default:
		*pu32AlndX = (m_u32Width/2) - ((u32FontWidth * m_strText.size())/2);
		break;
	}

	DEBUG_TRACE("X:%u Y:%u StrLen: %u", *pu32AlndX, *pu32AlndY, u32Strlen);
}

/************************************************************************
* FUNCTION: fbhmi_Screen::bAddChildWidget(fbhmi_Request* pReq)
*
* DESCRIPTION: Draw widget function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tBool fbhmi_Screen::bAddChildWidget(fbhmi_Request* pReq)
{
	DEBUG_TRACE("%s", __func__);
	_tBool bRet = false;
	if (NULL == pReq) {
		DEBUG_TRACE("Request pointer is NULL"); 
		return bRet;
	}
	_tU16 u16ReqId = pReq->u16GetRequestID();
	fbhmi_Widget* pWidget = NULL;

	switch(u16ReqId)
	{
	case FBHMI_EVNT_CREATE_LABEL:
		pWidget = new fbhmi_Label(pReq);
		break;

	case FBHMI_EVNT_CREATE_PROGRESSBAR:
		pWidget = new fbhmi_ProgressBar(pReq);
		break;

	default:
		break;
	}

	/*To remove lint warning*/
	fbhmi_Widget* pCustodyptr; 

	pCustodyptr = pWidget;

	// Add widget to child list
	if(NULL != pWidget)
	{
		m_vecChildWidgets.push_back(pWidget);
		DEBUG_TRACE("Added widget to screen.. :-)");
		pWidget->vDrawWidget();
		bRet = true;
	} 
	else
	{
		DEBUG_TRACE("Failed to add widget to screen!!");
	}
	return bRet;
}

/************************************************************************
* FUNCTION: fbhmi_Screen::pGetChildWidget(_tU16 u16Id)
*
* DESCRIPTION: Draw widget function
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_Widget* fbhmi_Screen::pGetChildWidget(_tU16 u16Id)
{
	fbhmi_Widget* pWidget = NULL;
	for(std::vector<fbhmi_Widget*>::iterator pIt = m_vecChildWidgets.begin(); pIt != m_vecChildWidgets.end(); pIt++)
	{
		DEBUG_TRACE("WidgetID Req: %u, WidgetID Found: %u", u16Id, (*pIt)->m_u16Id);
		if(u16Id == (*pIt)->m_u16Id)
		{
			pWidget = *pIt;
			break;
		}
	}
	return pWidget;
}

/************************************************************************
* FUNCTION: fbhmi_Screen::vClearScreen()
*
* DESCRIPTION: Clear Widget stuff from Screen
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vClearScreen()
{
	DEBUG_TRACE("%s", __func__);
	// Delete all widgets
	for(std::vector<fbhmi_Widget*>::iterator pIt = m_vecChildWidgets.begin(); pIt != m_vecChildWidgets.end(); pIt++)
	{
		delete *pIt;		
	}
	m_vecChildWidgets.clear();

	this->vDrawWidget();
}

/************************************************************************
* FUNCTION: fbhmi_Screen::vClearScreen()
*
* DESCRIPTION: Clear Widget stuff from Screen
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tU16  fbhmi_Screen::u16GetScreenId()
{
	DEBUG_TRACE("%s", __func__);
	return m_u16Id;
}

/************************************************************************
* FUNCTION: fbhmi_Widget::vUpdateWidget()
*
* DESCRIPTION: Update widget properties
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_Screen::vUpdateWidget(fbhmi_Request* pReq)
{
	DEBUG_TRACE("%s", __func__);

	if(NULL != pReq)
	{
		if(FBHMI_EVNT_UPDATE_TITLE == pReq->m_u16RequestID)
		{
			DEBUG_TRACE("FBHMI_EVNT_UPDATE_TITLE event received in screen..");
            fbhmi_TextReq * pclTxtReq;
            pclTxtReq = dynamic_cast<fbhmi_TextReq*>(pReq);
            if(NULL != pclTxtReq)
            {
            	this->vClearTitle();
            	m_strText.clear();
            	m_strText = pclTxtReq->m_strText;
            	m_u8Align = pclTxtReq->m_u8Align;
            	m_bVisible = pclTxtReq->m_bVisible;
            	DEBUG_TRACE("Updating Screen Title: %s", m_strText.c_str());
            	this->vDrawTitle();
            }

		}
		else
		{
			fbhmi_Widget* pWidget = pGetChildWidget(pReq->m_u16WidgetID);

			if(NULL == pWidget)
			{
				DEBUG_TRACE("Some pointer is NULL");
				return;
			}
			pWidget->vClearWidget(this->m_u32BgColor);
			pWidget->vUpdateWidget(pReq);
		}

	}
}

