/************************************************************************
* File: fbhmi_ProgressBar.cpp
* SW-Component: ASCII HMI for recovery download
*
* Description:
*   This is a singleton class used to initialize and use the Linux
*   framebuffer for displaying characters. 
* Author:
*   Adityakumar.jha@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 07.06.2013 - Initial version - Adityakumar.jha@in.bosch.com
* 01.08.2013 - Fix for text clear issue - Ankur Tank
* 18.03.2014 - Fix for ProgressBar filling - Aditya Kumar Jha
************************************************************************/

#include <string>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "../fbhmi_Utility.h"
#include "../fbhmi_Surface.h"
#include "fbhmi_ProgressBar.h"
using namespace std;

/************************************************************************
* FUNCTION: fbhmi_ProgressBar::fbhmi_ProgressBar()
*
* DESCRIPTION: Constructor
*
* PARAMETER: nID : Widget ID
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_ProgressBar::fbhmi_ProgressBar(_tU16 u16Id = FBHMI_WIDGET_DEF_ID,
												 _tU32 u32X = FBHMI_WIDGET_DEF_X,
												 _tU32 u32Y = FBHMI_WIDGET_DEF_Y,
												 _tU32 u32Height = FBHMI_WIDGET_DEF_HEIGHT,
												 _tU32 u32Width = FBHMI_WIDGET_DEF_WIDTH,
												 _tU16 u16BdrWidth = FBHMI_WIDGET_DEF_BWIDTH,
												 _tU32 u32BgColor = FBHMI_WIDGET_DEF_BGCOLOR,
												 _tU32 u32FgColor = FBHMI_WIDGET_DEF_FGCOLOR,
												 _tU32 u32BdrColor = FBHMI_WIDGET_DEF_BDRCOLOR,
												 _tU32 u32FillColor = FBHMI_WIDGET_DEF_FILCOLOR,
												 string sText = FBHMI_WIDGET_DEF_TEXT,
												 _tU8 u8Align = FBHMI_WIDGET_DEF_ALIGN,
												 _tBool bVisible = FBHMI_WIDGET_DEF_VISIBLE,
												 _tU32 u32PbarVal = FBHMI_WIDGET_DEF_PBARVAL,
												 _tU32 u32PbarMax = FBHMI_WIDGET_DEF_PBARMAX): fbhmi_Widget(u16Id,\
												 u32X, u32Y, u32Height, u32Width,\
												 u16BdrWidth, u32BgColor,\
												 u32FgColor, u32BdrColor,\
												 u32FillColor, sText,\
												 u8Align, bVisible,\
												 u32PbarVal, u32PbarMax)
{
	m_u16Id        = u16Id;
	m_u32x         = u32X;
	m_u32y         = u32Y;
	m_u32Height    = u32Height;
	m_u32Width     = u32Width;
	m_u16BdrWidth  = u16BdrWidth;
	m_u32BgColor   = u32BgColor;
	m_u32FgColor   = u32FgColor;
	m_u32BdrColor  = u32BdrColor;
	m_u32FillColor = u32FillColor;
	m_strText      = sText;
	m_u8Align      = u8Align;
	m_bVisible     = bVisible;
	m_u32PbarVal   = u32PbarVal;
	m_u32PbarMax   = u32PbarMax;
	DEBUG_TRACE("%s", __func__);

}
/************************************************************************
* FUNCTION: fbhmi_ProgressBar::fbhmi_ProgressBar()
*
* DESCRIPTION: Parameterized constructor
*
* PARAMETER: 
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_ProgressBar::fbhmi_ProgressBar(fbhmi_Request *pReq)
{
	fbhmi_WidgetReq *pclWidReq = dynamic_cast<fbhmi_WidgetReq *>(pReq);

	if(NULL != pReq && NULL != pclWidReq)
	{
		m_u16Id         = pclWidReq->m_u16WidgetID;
		m_u32x          = pclWidReq->m_u32x;
		m_u32y          = pclWidReq->m_u32y ;
		m_u32Height     = pclWidReq->m_u32height ;
		m_u32Width      = pclWidReq->m_u32width   ;
		m_u16BdrWidth   = pclWidReq->m_u32borderWidth ;
		m_u32BgColor    = pclWidReq->m_u32bgColor;
		m_u32FgColor    = pclWidReq->m_u32fgColor;
		m_u32BdrColor   = pclWidReq->m_u32borderColor;
		m_u32FillColor  = pclWidReq->m_u32fillColor;
		m_strText       = pclWidReq->m_strText;
		m_u8Align       = pclWidReq->m_u32textAlign;
		m_bVisible      = pclWidReq->m_bvisibility;
		m_u32PbarVal    = pclWidReq->m_u32value;
		m_u32PbarMax    = pclWidReq->m_u32maxValue;
	}

	m_pSurfaceHandle = fbhmi_Surface::pGetInstance();

	if(NULL == m_pSurfaceHandle)
	{
		// Debug Log
	}
	DEBUG_TRACE("%s", __func__);

}

/************************************************************************
* FUNCTION: fbhmi_ProgressBar::~fbhmi_ProgressBar()
*
* DESCRIPTION: Destructor
*
* PARAMETER: 
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_ProgressBar::~fbhmi_ProgressBar()
{
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: fbhmi_ProgressBar::vSetCurrentSize()
*
* DESCRIPTION: sets current position
*
* PARAMETER: Current size
*
* RETURNVALUE: _tVoid
*************************************************************************/
_tVoid fbhmi_ProgressBar::vUpdateValues(_tU32 u32PbarVal, _tU32 u32MaxValue)
{
	m_u32PbarVal = u32PbarVal;
	m_u32PbarMax = u32MaxValue;
	this->vDrawWidget();
	DEBUG_TRACE("%s", __func__);
}


/************************************************************************
* FUNCTION: fbhmi_ProgressBar::vDrawWidget()
*
* DESCRIPTION: Draw the Progressbar widget
*
* PARAMETER: Graphics *
*
* RETURNVALUE: _tVoid
*************************************************************************/
_tVoid fbhmi_ProgressBar::vDrawWidget()
{
	_tU32 u32TempWidth = m_u32Width;
	_tU32 u32TempHeight = m_u32Height;
	if (NULL == m_pSurfaceHandle)
	{
		DEBUG_TRACE("Surface pointer is NULL");
		return;
	}
	if(m_u16BdrWidth > FBHMI_WIDGET_DEF_BWIDTH)
	{
		//Draw Border if present
		m_pSurfaceHandle->vSetColor(m_u32BdrColor);
		m_pSurfaceHandle->vFillRectangle(m_u32x, m_u32y, m_u32Width, m_u32Height);
		u32TempWidth = m_u32Width - m_u16BdrWidth * 2;
		u32TempHeight = m_u32Height - m_u16BdrWidth * 2;
	}

	// Draw Background

	m_pSurfaceHandle->vSetColor(m_u32BgColor);
	m_pSurfaceHandle->vFillRectangle(m_u32x + m_u16BdrWidth,\
		m_u32y + m_u16BdrWidth,\
		u32TempWidth ,\
		u32TempHeight);
	if(true == m_bVisible)
	{
		// Fill progress
		_tU32 u32BarFillWidth = u32GetBarFillWidth();
		m_pSurfaceHandle->vSetColor(m_u32FillColor);
		m_pSurfaceHandle->vFillRectangle(m_u32x + m_u16BdrWidth,\
			m_u32y + m_u16BdrWidth,\
			u32BarFillWidth,\
			u32TempHeight);

		/* Draw Text 
		Fix for text clear issue If text size if more than label width
		clip the tail of string and append three dots*/
		_tU32 u32FontWidth = m_pSurfaceHandle->m_pFontHandle->u32GetFontWidth();
		_tU32 u32TmpTextWidth = (m_u32Width-m_u16BdrWidth*2)/u32FontWidth;

		if ( m_strText.size() >  u32TmpTextWidth)
		{
			m_strText.resize(u32TmpTextWidth-LABELTXT_TRAILING_DOT_SIZE);
			m_strText.append("...");
		}
		_tU32 au32Buff[(m_strText.size()+1) * sizeof(_tU32)];
		_tU32 u32AlignedX =0, u32AlignedY = 0;
		m_pSurfaceHandle->vSetColor(m_u32FgColor);
		memset(au32Buff, 0, (m_strText.size()+1) * sizeof(_tU32));
		fbhmi_Utility::vConvlatin9toutf32(const_cast<_tChar*>(m_strText.c_str()),au32Buff, m_strText.size());
		vGetAlignedXY(&u32AlignedX, &u32AlignedY, m_strText.size());
		m_pSurfaceHandle->vDrawString(au32Buff, u32AlignedX, u32AlignedY);
	}
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: fbhmi_ProgressBar::u32GetBarFillWidth()
*
* DESCRIPTION: Reuturns the Bar width to be filled based on width and
*              ProgressBar status
*
* PARAMETER: Graphics *
*
* RETURNVALUE: tU32
*************************************************************************/
_tU32 fbhmi_ProgressBar::u32GetBarFillWidth()
{
	/** Currently progressbar min value is not defined so considering 
	it as zero */
	_tU32 u32BarFillWidth = 0;
	if(0 == m_u32PbarMax){
		DEBUG_TRACE("Error! ProgressBar max is Zero.");
		return 0;
	}

	/** Progress bar current value must not exceed maximum value
	If it does then set Progress bar to maximum width*/
	if (m_u32PbarVal >=  m_u32PbarMax)
	{
		u32BarFillWidth = m_u32Width - m_u16BdrWidth * 2;
	}
	else if ( ((m_u32PbarVal * m_u32Width)/m_u32PbarMax) > m_u16BdrWidth )
	{
		u32BarFillWidth = ((m_u32PbarVal * m_u32Width)/m_u32PbarMax) - m_u16BdrWidth;
	}

	DEBUG_TRACE("%u", u32BarFillWidth);

	return u32BarFillWidth;
}

