/************************************************************************
 * File: mcp_CsmProxy.cpp
 * SW-Component:
 *
 * Description:
 *	Calls APIs of MCP library and handles the responses
 *
 * Author:
 *   RamaGopalReddy.Bommireddy@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 31.12.2019 - Initial version - RamaGopalReddy.Bommireddy@in.bosch.com
 * 28.02.2020 - Revision 1.0    - Surparaju.Pavankumar@in.bosch.com
 ***********************************************************************/
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_MCP
#define ETG_I_FILE_PREFIX 				mcp_CsmProxy::
#include "trcGenProj/Header/mcp_CsmProxy.cpp.trc.h"
#endif

#define ETG_ENABLED
#ifndef __SW_UPDATE_UNIT_TESTING__
#include "trace_interface.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

#include "util/swu_util.hpp"
#include "mcp_CsmProxy.h"
#include "mcp_fsmMCPUpdActions.h"


#ifndef CSM_C_NO_HANDLE_PTR
#define CSM_C_NO_HANDLE_PTR NULL
#endif

tCSM_ISO_TP_USDT_APPL_CALLBACK callbackData =
{
   NULL,
   NULL,
   mcp_CsmProxy::mcp_vCSMDataConfirmation,          // callback
   mcp_CsmProxy::mcp_vCSMDataIndication,           // callback
};

/************************************************************************
 * FUNCTION: mcp_CsmProxy()
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
mcp_CsmProxy::mcp_CsmProxy():_pCsmAccessUser(NULL)
{
    ETG_TRACE_USR4 (("MCPINFO: mcp_CsmProxy constructor called"));
}


/************************************************************************
 * FUNCTION: ~mcp_CsmProxy
 * DESCRIPTION: Destruction
 * PARAMETER: None
 * RETURNVALUE: None
 *************************************************************************/
mcp_CsmProxy::~mcp_CsmProxy()
{ 
    ETG_TRACE_USR4 (("MCPINFO: mcp_CsmProxy destructor called"));
}

/************************************************************************
 * FUNCTION: vInit()
 *
 * DESCRIPTION: Intializes the CSM access user
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_CsmProxy::vInit() {

   ETG_TRACE_USR4(("mcp_CsmProxy::vInit() START"));
   if(_pCsmAccessUser == NULL)
   {
      ETG_TRACE_USR4(("mcp_CsmProxy::vInit pCsmAccessUser object creation"));
      _pCsmAccessUser = new csm_tclCsmAccessUser();
       mcp_vRegisterCsmCallbacks();
   }
   else
   {
      ETG_TRACE_USR4(("mcp_CsmProxy::vInit _pCsmAccessUser Object is already created and call backs are registered"));
   }
   ETG_TRACE_USR4(("mcp_CsmProxy::vInit() END"));
}

/************************************************************************
 * FUNCTION: vDeInit()
 *
 * DESCRIPTION: Deletes the CSM access user
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_CsmProxy::vDeInit() {
   ETG_TRACE_USR4(("mcp_CsmProxy::vDeInit()"));

   mcp_vUnregisterCsmCallbacks();

   if (_pCsmAccessUser) {
      delete _pCsmAccessUser;
      _pCsmAccessUser = NULL;
   }
}

/************************************************************************
 * FUNCTION: mcp_vSetAddressField()
 *
 * DESCRIPTION: Setting the address fields
 *
 * PARAMETER: tU8*
 *
 * RETURNVALUE: None
 *************************************************************************/
tVoid mcp_CsmProxy::mcp_vSetAddressField(tU8* abAddressField)
{
   ETG_TRACE_COMP(("mcp_CsmProxy::mcp_vSetAddressField"));
   abAddressField[ 0] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO; // master
   abAddressField[ 1] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO_MCAN_CDIAG; // TCU
}

/************************************************************************
 * FUNCTION: mcp_vRegisterApplCallbackInit()
 *
 * DESCRIPTION: Register the application callbacks
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tS32
 *************************************************************************/
tS32 mcp_CsmProxy::mcp_vRegisterApplCallbackInit()
{
   tU8 abAddressField[2] = {0x00,0x00};
   abAddressField[ 0] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO; // master
   abAddressField[ 1] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO_MCAN_CDIAG; // TCU
   return _pCsmAccessUser->s32ApplCallbackInit(CSM_C_CAN_BUS_VCAN, CSM_C_PTYPE_DC_USDT_ISO, &abAddressField, &callbackData);
}

/************************************************************************
 * FUNCTION: mcp_vRegisterCsmCallbacks()
 *
 * DESCRIPTION: Register the csm callbacks
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
tVoid mcp_CsmProxy::mcp_vRegisterCsmCallbacks(tVoid) {
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vRegisterCsmCallbacks"));

   if (_pCsmAccessUser == NULL) {
      ETG_TRACE_ERR(("mcp_CsmProxy::mcp_vRegisterCsmCallbacks _pCsmAccessUser is null"));
      return;
   }

   _pCsmAccessUser->vApplCallbackPreInit(MAX_QUEUE_ELEMENTS, MCP_TX_BUFFER_SIZE);
   if(mcp_vRegisterApplCallbackInit()!=CSM_C_NO_ERROR)
   {
      ETG_TRACE_ERR(("mcp_CsmProxy::mcp_vRegisterCsmCallbacks Cannot register to 's32ApplCallbackInit' for MCP"));
   }  
}

/************************************************************************
 * FUNCTION: mcp_vUnregisterCsmCallbacks()
 *
 * DESCRIPTION: Deregister csm callbacks
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
tVoid mcp_CsmProxy::mcp_vUnregisterCsmCallbacks(tVoid) {
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vUnRegisterCsmCallbacks"));

   if (_pCsmAccessUser == NULL) {
      ETG_TRACE_ERR(("mcp_CsmProxy::mcp_vUnregisterCsmCallbacks _pCsmAccessUser is null"));
      return;
   }
   _pCsmAccessUser->s32ApplCallbackDeInit();
}

/************************************************************************
 * FUNCTION: sendUdsDataMessage()
 *
 * DESCRIPTION: Sends UDS message to CsmAccessUser
 *
 * PARAMETER: tPCU8,tU16,tU32
 *
 * RETURNVALUE: tS32
 *************************************************************************/
tS32 mcp_CsmProxy::sendUdsDataMessage(tPCU8 mcp_pcu8TxBuffer, tU16 mcp_u16TxBufferLen, tU32 pcvAddressField) {

   tS32 s32Result = CSM_C_GENERAL;
   // To fix CID 2750549
   tU8 abAddressField[4] = {0x00, 0x00, 0x00, 0x00};
   
   abAddressField[ 0] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO; // master
   abAddressField[ 1] = CSM_C_ADDRESS_FIELD_INF4CV_USDT_ISO_MCAN_CDIAG; // TCU
   
   if (_pCsmAccessUser)
   {
      // Address filed to be filed accordingly
      ETG_TRACE_USR4(("mcp_CsmProxy::sendUDSDataMessage pcvAddressField=%p", abAddressField));
      s32Result = _pCsmAccessUser->s32DataReq(CSM_C_PTYPE_DC_USDT_ISO, &abAddressField, mcp_pcu8TxBuffer, mcp_u16TxBufferLen);
   }
   else
   {
      ETG_TRACE_ERR(("mcp_CsmProxy::sendUDSDataMessage _pCsmAccessUser is null"));
   }
   if (s32Result != CSM_C_NO_ERROR)
   {
      ETG_TRACE_ERR(("mcp_CsmProxy::sendUDSDataMessage s32Result Error = %x",s32Result));
      ETG_TRACE_ERR(("mcp_CsmProxy::sendUDSDataMessage Failed"));
   }
   else
   {
      ETG_TRACE_USR4(("mcp_CsmProxy::sendUDSDataMessage Success"));
   }   
  
   return s32Result;
}

/************************************************************************
 * FUNCTION: mcp_vCSMDataConfirmation()
 *
 * DESCRIPTION: Receives data confirmation from CsmAccessUser
 *
 * PARAMETER: void*,tU32,const void*,tU8
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_CsmProxy::mcp_vCSMDataConfirmation(void* handle, tU32 protocol, const void* addrField, tU8 transactionResult) {
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataConfirmation"));

   tU8* pAddrField = const_cast<tU8*>((const tU8*) addrField);

   tU8 bData0 = *((((tU8 *)pAddrField)+0));
   tU8 bData1 = *((((tU8 *)pAddrField)+1));
   
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataConfirmation called - State:%d  Prot:0x%08X  AddrField:0x%02X%02X",
    transactionResult,
    protocol,
    bData0,bData1
    ));

    
    if( CSM_C_CONF_OK == transactionResult ) {             
        
        mcp_Request msg(swupd_fsmMCPUpd::evDataSentConfirmation);
        mcp_ClientHandler::pclGetQueueHandle()->bPushQueue(msg);    
    } 
    else 
    {
        mcp_Request msg(swupd_fsmMCPUpd::evDataNotSentConfirmation);
        mcp_ClientHandler::pclGetQueueHandle()->bPushQueue(msg);
    }
}

/************************************************************************
 * FUNCTION: mcp_CsmProxy()
 *
 * DESCRIPTION: Receives data indication from CsmAccessUser
 *
 * PARAMETER: void*,tU32,const void*,tU8*,tU16
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_CsmProxy::mcp_vCSMDataIndication(void* handle, tU32 protocol, const void* addrField, tU8* pData, tU16 u16DataLength)
{
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataIndication start"));
   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataIndication DataLength=%d", u16DataLength));
   tU8* pAddrField = const_cast<tU8*>((const tU8*) addrField);

   tU8 bData0 = *((((tU8 *)pAddrField)+0));
   tU8 bData1 = *((((tU8 *)pAddrField)+1));

   ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataIndication called - Prot:0x%08X  AddrField:0x%02X%02X Data:%02X",
            protocol,
            bData0,bData1,
            ETG_LIST_LEN(u16DataLength),
            ETG_LIST_PTR_T8(pData)
            ));

    std::vector<uint_fast8_t> udsBytes(pData, pData+u16DataLength); 
    mcp_Request msg(swupd_fsmMCPUpd::evResponseReceived,udsBytes,u16DataLength);
    mcp_ClientHandler::pclGetQueueHandle()->bPushQueue(msg);
    ETG_TRACE_USR4(("mcp_CsmProxy::mcp_vCSMDataIndication end"));

}
