/*************************************************************************
* FILE:         kdsread.c
* PROJECT:      Gen3 Platform
* SW-COMPONENT: SW Update
*----------------------------------------------------------------------
*
* DESCRIPTION:  Application to read KDS data
*
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
**************************************************************************/

/* build with
buildproduct --os=linux --env=GEN3ARMMAKE --buildmode=build --mode=release --noprecreate --info --alldeps=none swu_common_kds_out
*/

//#define OSAL_S_IMPORT_INTERFACE_GENERIC
//#include "osal_if.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

//#include "ostypes.h"
//#include "oserror.h"
#include "kdsread.h"
#include "kds_const.h"
//extern "C" {
  #include "dev_kds.h"

  void eh_reboot(void);
  void eh_reboot() {;}
//}

//--------------------------------------------------------------------------------
int main(int argc, char *argv[])
//--------------------------------------------------------------------------------
{

  printf ("\x1b[32mINFO\x1b[0m\t Start reading KDS values\n");
  char filename[MAXSTRLEN] = "/proc/self/fd/1";
  char prefix[MAXSTRLEN] = "";

  int error = 0;
  int verbose = 0;
  opterr = 1;
  int opt;
  while ((opt = getopt (argc, argv, "evo:")) != -1) {
    switch (opt)
    {
      case 'e':
        strcpy (prefix, "export ");
        break;
      case 'o':
        strncpy(filename, optarg, MAXSTRLEN-1);
        break;
      case 'v':
        verbose = 1;
        break;
      case '?':
        if (optopt == 'o') {
          fprintf (stderr, "Option -%c requires a output path/filename as argument.\n", optopt);
        } else if (isprint (optopt)) {
          fprintf (stderr, "Unknown option `-%c'.\n", optopt);
        } else {
          fprintf (stderr, "Unknown option character `\\x%x'.\n", optopt);
        }
        // fall through to error
      default:
        error = 1;
    }
  }

  int index;
  for (index = optind; index < argc; index++) {
    printf ("Non-option argument %s\n", argv[index]);
  }

  if (!error) {
    error = get_kds (prefix, filename);
  }
  _exit(error);
}


int get_kds(char prefix[], char filename[])
{
   int error=3;
   FILE *hfile=fopen(filename, "w");
   if(hfile==NULL)
   {
      hfile=stdout;
      printf("\x1b[32mERROR\x1b[0m\t file creation failed! -> Abort\n\n");
      error=1;
   }
   else
   {
      tU32 openresult=KDS_IOOpen();
      if(OSAL_E_NOERROR!=openresult)
      {
         fprintf(hfile, "# ERROR: KDS_IOOpen failed\n");
         fprintf(hfile, "  %sKDS_exit=\"1\"\n", prefix);
         printf("\x1b[32mERROR\x1b[0m\t KDS_IOOpen failed! -> Abort\n\n");
         error=openresult;
      }
      else
      {
         fprintf(hfile, "#!/bin/ash\n#\n# This file is autogenerated, do not edit!\n#\n# KDS values:\n#\n");

         get_kds_SWUPD_VariantInfo(hfile, prefix);
         get_kds_ECU_Partnumber(hfile, prefix);

         int oem=get_kds_CMVariantCoding(hfile, prefix);
         switch(oem)
         {
         case VW:
         case Seat:
         case Skoda:
         case Audi:
            get_kds_MIB2_VariantCoding(hfile, prefix);
            get_kds_MIB2_HardwareVersion(hfile, prefix);
            break;
         case Suzuki:
            // no additional items
            break;
         case GM:
            // no additional items
            break;
         case JACC:
            get_kds_JACC_VariantCoding(hfile, prefix);
            break;
         default:
            ; // error
         }
         KDS_s32IOClose();
         error=0;
      }
      fprintf(hfile, "\n# overall result:\n  %sKDS_exit=\"0\" \n# OK\n", prefix);
      fclose(hfile);
      printf("\x1b[32mINFO\x1b[0m\t Write data to file '%s'\n", filename);
   }
   return error;
}

//KDS_KEY_ECU_PARTNUMBER

int get_kds_ECU_Partnumber(FILE * hfile, char prefix[])
{
  tU16 kds_len = 10;
  char CM_ECU_Partnumber[KDS_MAX_ENTRY_LENGTH]={0};
  int status = read_kds(KDS_KEY_ECU_PARTNUMBER, kds_len, CM_ECU_Partnumber );
  if ( 0 == status )
  {
    hexprint(hfile, CM_ECU_Partnumber, "CM_ECU_Partnumber", prefix, kds_len);
    CM_ECU_Partnumber[kds_len]=0;
    fprintf (hfile, "  %sKDS_ECU_Partnumber=\"%s\"\n", prefix, CM_ECU_Partnumber);
  }
  else
  {
    fprintf (hfile, "# INFO: No KDS entry for 'CM_ECU_Partnumber'\n");
  }
  return status;
}


int get_kds_MIB2_HardwareVersion(FILE * hfile, char prefix[])
{
  tU16 kds_len = 3;
  char MIB2_HardwareVersion[KDS_MAX_ENTRY_LENGTH]={0};
  int status = read_kds(KDS_KEY_HARDWARE_VERSION, kds_len, MIB2_HardwareVersion );
  if ( 0 == status )
  {
    hexprint(hfile, MIB2_HardwareVersion, "MIB2_HardwareVersion", prefix, kds_len);
    MIB2_HardwareVersion[kds_len]=0;
    fprintf (hfile, "  %sKDS_MIB2_HWVER=\"%s\"\n", prefix, MIB2_HardwareVersion);
  }
  else
  {
    fprintf (hfile, "# INFO: No KDS entry for 'MIB2_HardwareVersion'\n");
  }
  return status;
}

int get_kds_JACC_VariantCoding(FILE * hfile, char prefix[])
{
  tU16 kds_len = 16;
  char JACC_VariantCoding[KDS_MAX_ENTRY_LENGTH]={0};
  int status = read_kds(KDS_KEY_CUSTOMER_VARIANTCODING, kds_len, JACC_VariantCoding );
  if ( 0 == status )
  {
    hexprint(hfile, JACC_VariantCoding, "JACC_VariantCoding", prefix, kds_len);
  }
  else
  {
    fprintf (hfile, "# INFO: No KDS entry for 'JACC_VariantCoding'\n");
  }
  return status;
}

int get_kds_MIB2_VariantCoding(FILE * hfile, char prefix[])
{
    tU16 kds_len = 16;
    char MIB2_VariantCoding[KDS_MAX_ENTRY_LENGTH]={0};
    int status = read_kds(KDS_KEY_CUSTOMER_VARIANTCODING, kds_len, MIB2_VariantCoding );
    if ( 0 == status )
    {
      hexprint(hfile, MIB2_VariantCoding, "MIB2_VariantCoding", prefix, kds_len);

      char Platform = MIB2_VariantCoding[KDS_MIB2_PLATFORM];
      switch ( Platform ) {
        case KDS_MIB2_PLATFORM_MQB:
          fprintf (hfile, "  %sKDS_MIB2_PLATFORM=\"mqb\"\n", prefix);
        break;
        case KDS_MIB2_PLATFORM_PQ:
          fprintf (hfile, "  %sKDS_MIB2_PLATFORM=\"pq\"\n", prefix);
        break;
        default:
          fprintf (hfile, "  %sKDS_MIB2_PLATFORM=\"not available\"\n", prefix);
      }
      char Market = MIB2_VariantCoding[KDS_MIB2_REGION];
      switch ( Market ) {
        case 0:
          fprintf (hfile, "  %sKDS_MIB2_REGION=\"unknown\"\n", prefix);
        break;
        case KDS_MIB2_REGION_ROW:
          fprintf (hfile, "  %sKDS_MIB2_REGION=\"row\"\n", prefix);
        break;
        case KDS_MIB2_REGION_USA:
          fprintf (hfile, "  %sKDS_MIB2_REGION=\"nar\"\n", prefix);
        break;
        default:
          fprintf (hfile, "  %sKDS_MIB2_REGION=\"undefined\"\n", prefix);
      }
    }
    else
    {
      fprintf (hfile, "# INFO: No KDS entry for 'MIB2_VariantCoding'\n");
    }
    return status;
}

int get_kds_CMVariantCoding(FILE * hfile, char prefix[])
{
   tU16 kds_len=32;
   char CMVariantCoding[KDS_MAX_ENTRY_LENGTH]={ 0 };
   char oem=0;
   int status=read_kds(KDS_KEY_CMVARIANTCODING, kds_len, CMVariantCoding);
   if(0==status)
   {
      hexprint(hfile, CMVariantCoding, "CMVariantCoding", prefix, kds_len);

      varprint(CMVariantCoding[KDS_CMVC_HW_ASSEMBLY]&KDS_MASK_DAB_TUNER, hfile, prefix, "KDS_ADR_DAB", "with_dab", "no_dab");

      if(CMVariantCoding[KDS_CMVC_HW_ASSEMBLY]&KDS_MASK_DAB_TUNER)
      {
         fprintf(hfile, "  %sKDS_ADR3=\"with_dab\"\n", prefix);
      }
      else
      {
         fprintf(hfile, "  %sKDS_ADR3=\"no_dab\"\n", prefix);
      }

      if(CMVariantCoding[KDS_CMVC_HW_ASSEMBLY]&KDS_MASK_CD)
      {
         fprintf(hfile, "  %sKDS_CD=\"with_cd\"\n", prefix);
      }
      else
      {
         fprintf(hfile, "  %sKDS_CD=\"no_cd\"\n", prefix);
      }
      if(CMVariantCoding[KDS_CMVC_DISPLAYTYPE]==KDS_DISPLAYTYPE_COLOR)
      {
         fprintf(hfile, "  %sKDS_DISPLAY=\"color\"\n", prefix);
      }
      else
      {
         fprintf(hfile, "  %sKDS_Display=\"grey\"\n", prefix);
      }
      oem=CMVariantCoding[KDS_CMVC_OEM_TYPE];
      fprintf(hfile, "  %sKDS_OEM_ID=\"0x%02x\"\n", prefix, oem);
      fprintf(hfile, "  %sKDS_OEM_NR=\"%d\"\n", prefix, oem);
      char oemcount=(sizeof(oem_names)/sizeof(oem_names[0]));
      if(oem<oemcount)
      {
         fprintf(hfile, "  %sKDS_OEM_NAME=\"%s\"\n", prefix, oem_names[oem]);
      }
      else
      {
         fprintf(hfile, "  %sKDS_OEM_NAME=\"(value out of range %d > %d)\"\n", prefix, oem, oemcount);
      }
      char cUsbInfo=CMVariantCoding[KDS_CMVC_USB_INFO];
      fprintf(hfile, "  %sKDS_USB_INFO=\"%02x\"\n", prefix, cUsbInfo);

      char dtbid=CMVariantCoding[KDS_CMVC_DTBID];
      fprintf(hfile, "  %sKDS_DTBID=\"%02x\"\n", prefix, dtbid);
   }
   else
   {
      fprintf(hfile, "# INFO: No KDS entry for 'CMVariantCoding'\n");
   }
   return oem;
}

int varprint(int condition, FILE * hfile, char prefix[], char key[], char value_true[], char value_false[])
{
  //char value* = (condition ? value_true : value_false);
  fprintf (hfile, "  %s%s=\"%s\"\n", prefix, key, condition ? value_true : value_false);
  return 0;
}

int get_kds_SWUPD_VariantInfo(FILE * hfile, char prefix[])
{
  char SWUPD_VariantInfo[KDS_MAX_ENTRY_LENGTH]={0};
  tU32 num = 0;
  tS32 readresult = read_kds(KDS_KEY_SWUPD_VARIANT_INFO, 5, SWUPD_VariantInfo );
  if ( 0 == readresult)
  {
    hexprint(hfile, SWUPD_VariantInfo, "SWUPD_VariantInfo", prefix, 5);

    tU32 cvtype = SWUPD_VariantInfo[0];
    set_numvar(hfile, "KDS_CMSWUPConfigurationVersionType", cvtype,  prefix);

    tU32 hwid = SWUPD_VariantInfo[1];
    set_numvar(hfile, "KDS_SWUPD_HWID", hwid,  prefix);

    tU32 swid = SWUPD_VariantInfo[2];
    set_numvar(hfile, "KDS_SWUPD_SWID", swid,  prefix);

    tU32 i;
    tU32 var_lo = 0;
    for (i=3; i < 5; i++) {
      var_lo = (var_lo<<8) + SWUPD_VariantInfo[i];
    }
    set_numvar(hfile, "KDS_SWUPD_Variant_lo", var_lo,  prefix);
  }
  else
  {
    fprintf (hfile, "# ERROR: KDS entry for 'SWUPD_VariantInfo' not found\n");
  }
  return readresult;
}

int hex2num32 (char in[])
{
    int num = 0;
    tU32 i = 0;
    for (i=0; i < 4; i++) {
      num = (num<<8) + in[i];
    }
    return num;
}

int read_kds(tU16 u16Entry, tU16 u16datalength, char pu8data[] )
{
  int error = 1;
  tsKDSEntry sKDSEntryData;

  if (u16datalength > KDS_MAX_ENTRY_LENGTH)
  {
    u16datalength = KDS_MAX_ENTRY_LENGTH;
  }

  sKDSEntryData.u16Entry        = u16Entry;
  sKDSEntryData.u16EntryLength  = u16datalength; // KDS_MAX_ENTRY_LENGTH
  sKDSEntryData.u16EntryFlags   = M_KDS_ENTRY_FLAG_NONE;
  memset (sKDSEntryData.au8EntryData, 0, KDS_MAX_ENTRY_LENGTH);

  tU32 readresult = KDS_s32IORead((tPCS8) &sKDSEntryData, sizeof(sKDSEntryData));
  if (OSAL_E_NOERROR == readresult || sizeof(sKDSEntryData) == readresult ) {
    error = 0;
    memcpy ( pu8data, sKDSEntryData.au8EntryData, u16datalength);
    //printf ("INFO: KDS_s32IORead for entry '0x%04x' result '0x%04x'\n", u16Entry, readresult);
  //} else {
    //printf ("ERROR: KDS_s32IORead for entry '0x%04x' error '0x%04x'\n", u16Entry, readresult);
  }
  return error;
}


void set_numvar(FILE *hfile, char varname[], tU32 numvalue, char prefix[])
{
  char valuestring[MAXSTRLEN];
  //memset(valuestring, 0, sizeof(valuestring));
  sprintf(valuestring, "%d", numvalue);
  setenv(varname, valuestring, 1);
  if (hfile) {
    fprintf (hfile, "  %s%s=\"%s\"\n\n", prefix, varname, valuestring);
  }
}

void set_var(FILE *hfile, char varname[], char strvalue[], char prefix[])
{
  setenv(varname, strvalue, 1);
  if (hfile) {
    fprintf (hfile, "  %s%s=\"%s\"\n\n", prefix, varname, strvalue);
  }
}

void bin2hex(char bin[], char hex[], int len)
{
  int i;
  for (i=0; i < len; i++) {
    sprintf (hex + 2*i, "%02x", bin[i]);
  }
  hex[2*i] = 0;
}

void hexprint(FILE *hfile, char kds_bin[], const char name[], const char prefix[], int len)
{
  char hexstring[2*KDS_MAX_ENTRY_LENGTH];
  memset(hexstring, 0, sizeof(hexstring));
  bin2hex(kds_bin, hexstring, len);
  fprintf (hfile, "\n# hex values for '%s'\n", name);
  fprintf (hfile, "  %sKDS_%s=\"0x%s\"\n\n", prefix, name, hexstring);
  fprintf (hfile, "# extracted values for '%s'\n", name);
}
