
/************************************************************************
*FILE					: swuhmi_cairoadapter.cpp
*SW-COMPONENT			: SW-Update
*DESCRIPTION			: CAIRO GRAPHICS Adapter.
*DESCRIPTION			: It is Graphics Interface where we can achieve all the Drawing Funtionality.
*AUTHOR					: VRI7COB (RBEI/ECA)
*COPYRIGHT				: (c) 1996 – 2000 Blaupunkt Werke GmbH
*HISTORY				:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
13-07-2016	| 0.2		| AGL5KOR			| CMG3G-10993 - Generic Way of Handling the Long Text in Cairo Adapter 
28-02-2016	| 0.3		| SCK7COB			| CMG3G-12559 - Font faces option in cario adapter 
13-07-2017	| 0.4		| ALA5COB			| CMG3G-14492-Improvements in the recovery hmi
************************************************************************/

#include "swuhmi_cairoadapter.h"
#include <algorithm>

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "SWUHMI_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_SWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_HMI
#include "trcGenProj/Header/swuhmi_cairoadapter.cpp.trc.h"
#endif

bool BothAreSpaces(char lhs, char rhs) { return (lhs == rhs) && (lhs == ' '); }

/************************************************************************
*FUNCTION			: cairoadapter
*DESCRIPTION		: Constructor of cairoadapter
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
cairoadapter::cairoadapter()
{
	m_pCairoSurface 		= NULL;
	m_pCairo				= NULL;
	m_FreeTypeLib			= NULL;
	m_pChinesefont			= NULL;
	m_u32Width				= 0;
	m_u32Height				= 0;
	m_u32Stride				= 0;
	memset(&m_ft_cairo_user_data_key, 0, sizeof (cairo_user_data_key_t));

	m_U8FrameBuffer			= NULL;
	m_pfont					= NULL;
	m_dLineOffset			= 0.0;
	m_strDelimiter.clear();
}

/************************************************************************
*FUNCTION			: cairoadapter
*DESCRIPTION		: Destructor of cairoadapter
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
cairoadapter::~cairoadapter()
{
	// TODO Auto-generated destructor stub
	if(m_pCairoSurface != NULL)		
		m_pCairoSurface = NULL;

	if(m_pCairo != NULL) 		
		m_pCairo = NULL;

	if(m_FreeTypeLib != NULL) 		
		m_FreeTypeLib = NULL;

	if(m_pChinesefont != NULL) 		
		m_pChinesefont = NULL;

	if(m_pfont != NULL) 
		m_pfont = NULL;

	if(m_U8FrameBuffer != NULL)
		m_U8FrameBuffer = NULL;
}

/*******************************************************************************************
*FUNCTION			: InitializeCairoAdapter
*DESCRIPTION		: Initialize the Cario Graphics
*PARAMETER			: None
*RETURNVALUE		: bool - Successs / Failure
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
13-07-2017	| 0.2		| ALA5COB			| CMG3G-14492-Improvements in the recovery hmi
********************************************************************************************/
tBool cairoadapter::InitializeCairoAdapter()
{
   tBool bReturn = false;
   ETG_TRACE_USR4(("InitializeCairoAdapter"));

   if(m_DRMRenderInst.InitializeDRMRender() == 0){
      bReturn = true;
      ETG_TRACE_USR4(("InitializeDRMRender - Success"));
		
      if(m_DRMRenderInst.SetDrawEnv())
      {
         ETG_TRACE_USR4(("Draw Env is set Properly"));

         m_U8FrameBuffer 		= m_DRMRenderInst.GetFrameBuffer();
         m_u32Width				= m_DRMRenderInst.GetWidth();
         m_u32Height				= m_DRMRenderInst.GetHeight();
         m_u32Stride				= m_DRMRenderInst.GetStride();
         m_dLineOffset			= (tDouble) (m_u32Width - WIDTH_ADJUSTMENT_VAL);

         ETG_TRACE_USR4(("Screen Width:%u, Height:%u", m_u32Width, m_u32Height));

         InitFreeTypeLib();
			bReturn = true;
      }
      else{
         fprintf(stderr,"swuhmi_cairoadapterDraw Env is not set Properly..... \n");
      }
   }
   else{
      fprintf(stderr,"swuhmi_cairoadapter: InitializeDRMRender - failed..... \n");      
   }
   return bReturn;
}

/************************************************************************
*FUNCTION			: ReSurface
*DESCRIPTION		: Changing the Buffer of DRM.
*PARAMETER			: None
*RETURNVALUE		: tVoid
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
11-05-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::ReSurface() {

	if(m_DRMRenderInst.SetDrawEnv())
	{
		m_U8FrameBuffer 		= m_DRMRenderInst.GetFrameBuffer();
		m_u32Width				= m_DRMRenderInst.GetWidth();
		m_u32Height				= m_DRMRenderInst.GetHeight();
		m_u32Stride				= m_DRMRenderInst.GetStride();

		DestroySurface();
		CreateSurface();

		cairo_set_font_face(m_pCairo, m_pfont);
	}

}

/************************************************************************
*FUNCTION			: InitFreeTypeLib
*DESCRIPTION		: Initialize the Font File
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::InitFreeTypeLib()
{
	ETG_TRACE_USR4(("InitFreeTypeLib"));
	FT_Error result = 0;
	/* Try to create the FreeType library handle on-the-fly */
	if (NULL == m_FreeTypeLib)
	{
		result = FT_Init_FreeType(&m_FreeTypeLib);
		if (0 != result)
		{
			ETG_TRACE_USR4(("Error in Initalize the LibFreeType"));
			fprintf(stderr, "cairo_font_from_ttf_file: Failed to initialize "
							"FreeType library\n");
		}
		else
		{
			ETG_TRACE_USR4(("LibFreeType is Created Successfully"));
		}
	}
}

/************************************************************************
*FUNCTION			: CreateSurface
*DESCRIPTION		: Creating the Cairo Surface
*PARAMETER			: None
*RETURNVALUE		: bool - Success / Failure
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tBool cairoadapter::CreateSurface()
{
	tBool bReturn = false;
	ETG_TRACE_USR4(("Create Cairo Surface"));
	/* First, we create the cairo surface from the mapped DRM buffer */
	m_pCairoSurface = cairo_image_surface_create_for_data(m_U8FrameBuffer,
												  CAIRO_FORMAT_ARGB32,
												  m_u32Width,
												  m_u32Height,
												  m_u32Stride);
	if(m_pCairoSurface != NULL)
	{
		ETG_TRACE_USR4(("Cairo Surface is created Successfully"));
		bReturn = true;
		 /* Secondly, create a cairo context for the surface */
		m_pCairo = cairo_create(m_pCairoSurface);
		cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
		cairo_paint(m_pCairo);

	}
	else
	{
		ETG_TRACE_USR4(("CAiro Surface is not created Successfully"));
	}
	return bReturn;
}

tVoid cairoadapter::DestroySurface() {

	ETG_TRACE_USR4(("Destroy Cairo Surface"));

	cairo_surface_destroy(m_pCairoSurface);
	cairo_destroy(m_pCairo);

}

/************************************************************************
*FUNCTION			: ClearSurface
*DESCRIPTION		: clearing the Cairo Surface
*PARAMETER			: None
*RETURNVALUE		: None
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::ClearSurface()
{
	ETG_TRACE_USR2(("ClearSurface"));
	cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
	cairo_paint(m_pCairo);
	m_DRMRenderInst.FlipThePage();
}

/************************************************************************
*FUNCTION			: FlipThePage
*DESCRIPTION		: To Fliping the Image through DRM
*PARAMETER			: None
*RETURNVALUE		: None
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::FlipThePage()
{
	m_DRMRenderInst.FlipThePage();
}

tVoid cairoadapter::SetDelimiter(string strDelimiter) {
	ETG_TRACE_USR4(("Delimiter - %s", strDelimiter.c_str()));
	m_strDelimiter = strDelimiter;
}

/************************************************************************
*FUNCTION			: SetBackGndImage
*DESCRIPTION		: Set the Back Ground Image - Logo
*PARAMETER			: None
*RETURNVALUE		: None
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::SetBackGndImage(string strPath,float xPos,float yPos)
{
	ETG_TRACE_USR4(("SetBackGndImage"));

	cairo_surface_t *background;
	background = cairo_image_surface_create_from_png(strPath.c_str());
	cairo_set_source_surface(m_pCairo, background, xPos, yPos);
	cairo_paint_with_alpha(m_pCairo, 0.3);
	cairo_surface_destroy(background);

	cairo_paint(m_pCairo);
}

/************************************************************************
*FUNCTION			: UpdateImage
*DESCRIPTION		: Updating the Image.
*PARAMETER			: struct (I)
*PARAMETER				Image Information.
*RETURNVALUE		: None
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::UpdateImage(ImageInfo stImageInfo)
{
	ETG_TRACE_USR4(("UpdateImage"));

	//Clear Previous Image
	if(!stImageInfo.PrevImagePath.empty())
	{
		cairo_surface_t* PrevLogo;
		tDouble Prev_logo_width;
		tDouble Prev_logo_height;
		PrevLogo = cairo_image_surface_create_from_png(stImageInfo.PrevImagePath.c_str());
		Prev_logo_width = cairo_image_surface_get_width(PrevLogo);
		Prev_logo_height = cairo_image_surface_get_height(PrevLogo);
		cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
		cairo_rectangle(m_pCairo, (tDouble)stImageInfo.u16XPos, (tDouble)stImageInfo.u16YPos, Prev_logo_width, Prev_logo_height);
		cairo_fill(m_pCairo);
		cairo_surface_destroy(PrevLogo);
	}

	cairo_surface_t *statusLogo;
	int logo_width;
	int logo_height;
	statusLogo = cairo_image_surface_create_from_png(stImageInfo.ImagePath.c_str());
	logo_width = cairo_image_surface_get_width(statusLogo);
	logo_height = cairo_image_surface_get_height(statusLogo);
	cairo_set_source_surface(m_pCairo, statusLogo, (tDouble)stImageInfo.u16XPos, (tDouble)stImageInfo.u16YPos);
	cairo_surface_destroy(statusLogo);
	cairo_paint(m_pCairo);

}

/************************************************************************
*FUNCTION			: CreateFont
*DESCRIPTION		: Creating the Font Through Lib FreeType.
*PARAMETER			: string (I)
*PARAMETER				Font File Name with Path
*RETURNVALUE		: bool - Success / Failure
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
28-02-2016	| 0.2		| SCK7COB			| CMG3G-12559 - Font faces option in cario adapter 
************************************************************************/
tBool cairoadapter::CreateFont(string strPath,bool useSynthFlags,int synthFlags)
{
	ETG_TRACE_USR4(("CreateFont"));
	ETG_TRACE_USR4(("Font Path:%s", strPath.c_str()));

	tBool bReturn = false;
	FT_Error result = 0;
	FT_Face face;
	cairo_status_t status = CAIRO_STATUS_SUCCESS;

	if (strPath.empty())
	{
		fprintf(stderr, "cairo_font_from_ttf_file: font_file path was not given\n");
		return bReturn;
	}

	ETG_TRACE_USR4(("FT_New_Face Creation"));
	result = FT_New_Face(m_FreeTypeLib, strPath.c_str(), 0, &face);
	if (0 != result)
	{
		fprintf(stderr, "cairo_font_from_ttf_file: Failed to create face\n");
		return bReturn;
	}
	ETG_TRACE_USR4(("FT_New_Face Done"));

	m_pfont = cairo_ft_font_face_create_for_ft_face(face, 0);
	if (m_pfont == NULL)
	{
		fprintf(stderr, "cairo_font_from_ttf_file: Failed to create cairo font "
						"from FT_Face\n");
		FT_Done_Face(face);

		return bReturn;
	}

	status = cairo_font_face_set_user_data(m_pfont,
	                                           &m_ft_cairo_user_data_key,
	                                           face,
	                                           (cairo_destroy_func_t) FT_Done_Face);

	if (status != CAIRO_STATUS_SUCCESS)
	{
		cairo_font_face_destroy(m_pfont);
		FT_Done_Face(face);

		return bReturn;
	}

	bReturn = true;
	if(useSynthFlags)
	{
		cairo_ft_font_face_set_synthesize(m_pfont , synthFlags);
	}

    /* Start rendering some bold text in size 22px on the buffer */
    cairo_set_font_face(m_pCairo, m_pfont);

    return bReturn;
}

/************************************************************************
*FUNCTION			: CreateChineseFont
*DESCRIPTION		: Creating the chinese Font Through Lib FreeType.
*PARAMETER			: string (I)
*PARAMETER				Font File Name with Path
*RETURNVALUE		: bool - Success / Failure
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
28-02-2016	| 0.2		| SCK7COB			| CMG3G-12559 - Font faces option in cario adapter 
************************************************************************/
tBool cairoadapter::CreateChineseFont(string strPath,bool useSynthFlags,int synthFlags)
{
	ETG_TRACE_USR4(("CreateChineseFont"));
	ETG_TRACE_USR4(("Font Path:%s", strPath.c_str()));

	tBool bReturn = false;
	FT_Error result = 0;
	FT_Face face;
	cairo_status_t status = CAIRO_STATUS_SUCCESS;

	if (strPath.empty())
	{
		fprintf(stderr, "cairo_font_from_ttf_file: font_file path was not given\n");
		return bReturn;
	}

	ETG_TRACE_USR4(("FT_New_Face Creation"));
	result = FT_New_Face(m_FreeTypeLib, strPath.c_str(), 0, &face);
	if (0 != result)
	{
		fprintf(stderr, "cairo_font_from_ttf_file: Failed to create face\n");
		return bReturn;
	}
	ETG_TRACE_USR4(("FT_New_Face Done"));

	m_pChinesefont = cairo_ft_font_face_create_for_ft_face(face, 0);
	if (m_pfont == NULL)
	{
		fprintf(stderr, "cairo_font_from_ttf_file: Failed to create cairo font "
						"from FT_Face\n");
		FT_Done_Face(face);

		return bReturn;
	}

	status = cairo_font_face_set_user_data(m_pChinesefont,
	                                           &m_ft_cairo_user_data_key,
	                                           face,
	                                           (cairo_destroy_func_t) FT_Done_Face);

	if (status != CAIRO_STATUS_SUCCESS)
	{
		cairo_font_face_destroy(m_pfont);
		FT_Done_Face(face);

		return bReturn;
	}

	bReturn = true;

	if(useSynthFlags)
	{
		cairo_ft_font_face_set_synthesize(m_pChinesefont , synthFlags);
	}

    /* Start rendering some bold text in size 22px on the buffer */
    cairo_set_font_face(m_pCairo, m_pChinesefont);

	return bReturn;
}

/************************************************************************
*FUNCTION			: DrawText
*DESCRIPTION		: Drawing the Text
*PARAMETER			: string (I)
*PARAMETER				Label Information
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
21-06-2016  | 0.2       | AGL5KOR           | Changed the logic to handle multi '\n' in the string
13-07-2016  | 0.3       | AGL5KOR           | Changed the logic to check delimiter and split text into multi-line if required
************************************************************************/
tVoid cairoadapter::DrawText(LabelInfo stLabelInfo)
{
	ETG_TRACE_USR4(("DrawText"));

	ETG_TRACE_USR4(("Delimiter-%s", m_strDelimiter.c_str()));

	if(!m_strDelimiter.empty())
		DrawTextWithDelimiter(stLabelInfo);
	else
		DrawTextWithoutDelimiter(stLabelInfo);
}

/************************************************************************
*FUNCTION			: DrawTranslatedTextWithoutDelimiter
*DESCRIPTION		: Drawing the tanslated text on the screen depending on the screen width
*PARAMETER			: struct (I)
*PARAMETER				Label Information.
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
13-07-2016  | 0.1       | VRI7COB           | Initial
************************************************************************/
tVoid cairoadapter::DrawTextWithoutDelimiter(LabelInfo stLabelInfo, bool IsTranslatedText){

	ETG_TRACE_USR1(("DrawTextWithoutDelimiter - Start"));

	ETG_TRACE_USR4(("Translated Text Status:%d", IsTranslatedText));
	ETG_TRACE_USR4(("Text-%s", stLabelInfo.strText.c_str()));

	cairo_set_font_size(m_pCairo, (tDouble)stLabelInfo.u16Height);

	cairo_text_extents_t extents;
	cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);
	tDouble dTextWidth = extents.width;
	tDouble dTextHeight = extents.height;
	// gen3armmake, gen3lsim, gen4rcar, gen4lsim : conversion to 'etg_tF32 {aka float}' from 'tDouble {aka double}'
	ETG_TRACE_USR4(("dTextWidth:%lf, m_dLineOffset:%lf", static_cast<float> (dTextWidth), static_cast<float> (m_dLineOffset)));

	string strInput = stLabelInfo.strText;
	std::string::iterator new_end = std::unique(strInput.begin(), strInput.end(), BothAreSpaces);
	strInput.erase(new_end, strInput.end());
	stLabelInfo.strText = strInput;

	if(dTextWidth < m_dLineOffset) {
		ETG_TRACE_USR4(("No need to Spilt the Strings"));
		if(IsTranslatedText)
			DrawAbNormalText(stLabelInfo);
		else
			DrawNormalText(stLabelInfo);
	}
	else {
		ETG_TRACE_USR4(("Strings need to Spilt"));
		string strTextToBeDrawn = stLabelInfo.strText;
		string strLine2 = strTextToBeDrawn;
		tU16 count=0;
		LabelInfo TempLabelInfo = {0};

		while (!strLine2.empty()) {
			cairo_text_extents_t extent;
			cairo_text_extents (m_pCairo, strTextToBeDrawn.c_str(), &extent);
			dTextWidth = extent.width;
			strLine2.clear();

			list<string> words;
			istringstream is( strTextToBeDrawn );
			string word;
			while ( is >> word )
				words.push_back( word );

			while ( dTextWidth > m_dLineOffset ) {
				strLine2.insert(0,words.back());
				strLine2.insert(0," ");
				words.pop_back();

				size_t found = strTextToBeDrawn.find_last_of(" ");
				strTextToBeDrawn=strTextToBeDrawn.substr(0,found);
				cairo_text_extents (m_pCairo, strTextToBeDrawn.c_str(), &extent);
				dTextWidth = extent.width;
			}

			ETG_TRACE_USR4(("strTextToBeDrawn :%s", strTextToBeDrawn.c_str()));
			ETG_TRACE_USR4(("strLine2 :%s", strLine2.c_str()));

			//Don't try to do memcpy, causes Double Memory Corruption.
			//http://stackoverflow.com/questions/9187444/why-do-i-get-double-free-or-corruption
			TempLabelInfo.eAlign = stLabelInfo.eAlign;
			TempLabelInfo.stcolor = stLabelInfo.stcolor;
			TempLabelInfo.strPrevText = "";
			TempLabelInfo.u16Height = stLabelInfo.u16Height;
			TempLabelInfo.u16XPos = stLabelInfo.u16XPos;
			TempLabelInfo.u16YPos = (tU16)(stLabelInfo.u16YPos + ((dTextHeight + EXTRA_LINE_SPACE_BETWEEN_LINES ) * count));
			TempLabelInfo.strText = strTextToBeDrawn;

			if(IsTranslatedText)
				DrawAbNormalText(TempLabelInfo);
			else
				DrawNormalText(TempLabelInfo);
			count++;

			strTextToBeDrawn.clear();
			strTextToBeDrawn = strLine2;
		}
	}
	ETG_TRACE_USR1(("DrawTextWithoutDelimiter - End"));
}

tVoid cairoadapter::DrawTextWithDelimiter(LabelInfo stLabelInfo, bool IsTranslatedText){
	ETG_TRACE_USR1(("DrawTextWithDelimiter - Start"));
	ETG_TRACE_USR4(("Translated Text Status:%d", IsTranslatedText));

	cairo_set_font_size(m_pCairo, (tDouble)stLabelInfo.u16Height);

	cairo_text_extents_t extents;
	cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);
	tDouble dTextHeight = extents.height;

	string strcurrentText = stLabelInfo.strText;
	string delim(m_strDelimiter);
	size_t start=0;
	size_t end = strcurrentText.find(delim);

	if (end == std::string::npos) {
		ETG_TRACE_USR4(("No need to Spilt the Strings"));
		if(IsTranslatedText) {
                   DrawAbNormalText(stLabelInfo);
		} else {
                   DrawNormalText(stLabelInfo);
                }
	}
	else {
		ETG_TRACE_USR4(("Need to Spilt the Strings"));
		list<string> token;
		int count = 0;
		LabelInfo TempLabelInfo = {0};

		while (end != std::string::npos){
			token.push_back(strcurrentText.substr(start, end - start)) ;
			start = end + delim.length();
			end = strcurrentText.find(delim, start);
		}
		token.push_back(strcurrentText.substr(start, end));

		while(!(token.empty())) {
			string Line1 = token.front();
			token.pop_front();

			//Don't try to do memcpy, causes Double Memory Corruption.
			//http://stackoverflow.com/questions/9187444/why-do-i-get-double-free-or-corruption
			TempLabelInfo.eAlign = stLabelInfo.eAlign;
			TempLabelInfo.stcolor = stLabelInfo.stcolor;
			TempLabelInfo.strPrevText = "";
			TempLabelInfo.u16Height = stLabelInfo.u16Height;
			TempLabelInfo.u16XPos = stLabelInfo.u16XPos;
			TempLabelInfo.u16YPos = (tU16)(stLabelInfo.u16YPos + ((dTextHeight + EXTRA_LINE_SPACE_BETWEEN_LINES ) * count));
			TempLabelInfo.strText = Line1;

			if(IsTranslatedText)
				DrawAbNormalText(TempLabelInfo);
			else
				DrawNormalText(TempLabelInfo);
			count++;
		}

	}
	ETG_TRACE_USR1(("DrawTextWithDelimiter - End"));
}


tVoid cairoadapter::DrawNormalText(LabelInfo stLabelInfo) {

	ETG_TRACE_USR1(("DrawNormalText - Start"));

	cairo_text_extents_t extents;
	cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);

	if(stLabelInfo.eAlign == E_LABEL_HORIZONTAL_CENTER_ALIGNMENT) {
		tDouble x;
		x = m_u32Width / 2.0 - (extents.width + extents.x_bearing) / 2.0;
		cairo_move_to(m_pCairo, x, (tDouble)stLabelInfo.u16YPos);
	}
	else
		cairo_move_to(m_pCairo, (tDouble)stLabelInfo.u16XPos, (tDouble)stLabelInfo.u16YPos);

	cairo_set_source_rgb(m_pCairo, stLabelInfo.stcolor.u8Red / 255.0, stLabelInfo.stcolor.u8Green / 255.0, stLabelInfo.stcolor.u8Blue / 255.0);
	cairo_show_text(m_pCairo, stLabelInfo.strText.c_str());

	ETG_TRACE_USR1(("DrawNormalText - End"));
}

tVoid cairoadapter::DrawAbNormalText(LabelInfo stLabelInfo) {

	ETG_TRACE_USR1(("DrawAbNormalText - Start"));

	cairo_glyph_t *glyphs = NULL;
	cairo_status_t status;
	int num_glyphs;

	cairo_text_extents_t extents;
	cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);

	if(stLabelInfo.eAlign == E_LABEL_HORIZONTAL_CENTER_ALIGNMENT)
		stLabelInfo.u16XPos = (tU16)(m_u32Width / 2.0 - (extents.width + extents.x_bearing) / 2.0);

	cairo_set_source_rgb(m_pCairo, stLabelInfo.stcolor.u8Red / 255.0, stLabelInfo.stcolor.u8Green / 255.0, stLabelInfo.stcolor.u8Blue / 255.0);
	status = cairo_scaled_font_text_to_glyphs (cairo_get_scaled_font (m_pCairo),
													   (tDouble)stLabelInfo.u16XPos, (tDouble)stLabelInfo.u16YPos,
													   stLabelInfo.strText.c_str(),(static_cast<int> (stLabelInfo.strText.length() )),
													   &glyphs, &num_glyphs,
													   NULL, NULL,
													   NULL);
	if (status == CAIRO_STATUS_SUCCESS)
		cairo_show_glyphs (m_pCairo, glyphs, num_glyphs);

	if ( NULL != glyphs)
		cairo_glyph_free (glyphs);

	ETG_TRACE_USR1(("DrawAbNormalText - End"));
}

/************************************************************************
*FUNCTION			: UpdateText
*DESCRIPTION		: Updating the Text
*PARAMETER			: struct (I)
*PARAMETER				Label Information
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::UpdateText(LabelInfo stLabelInfo)
{
	ETG_TRACE_USR4(("UpdateText" ));

	tU16 u16FontHeight = 0;
	cairo_set_font_size(m_pCairo, (tDouble)stLabelInfo.u16Height);
	u16FontHeight = stLabelInfo.u16Height;
	tU16 u16ExtraFontHeight =static_cast<tU16> (u16FontHeight + EXTRA_FONT_HEIGHT);

	//Clear the Text
	cairo_text_extents_t extents;
	cairo_text_extents(m_pCairo, stLabelInfo.strPrevText.c_str(), &extents);
	// gen3armmake, gen3x86make, gen4rcar, gen4lsim: conversion to 'etg_tF32 {aka float}' from 'double'
	ETG_TRACE_USR4(("Previous Text Extends Width:%lf, Height:%lf ", static_cast<float> (extents.width), static_cast<float> (extents.height)));

	string strPrevText = stLabelInfo.strPrevText;
	size_t PrevNewLinePos1 = strPrevText.find("\n");
	if(PrevNewLinePos1 != string::npos)
	{
		string Line1 = strPrevText.substr(0, PrevNewLinePos1);
		cairo_text_extents_t PrevLine1extents;
		cairo_text_extents(m_pCairo, Line1.c_str(), &PrevLine1extents);
		cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
		cairo_rectangle(m_pCairo, (tDouble)stLabelInfo.u16XPos, (tDouble)(stLabelInfo.u16YPos - u16FontHeight), PrevLine1extents.width + EXTRA_FONT_WIDTH, (tDouble)u16ExtraFontHeight);
		cairo_fill(m_pCairo);

		string Line2 = strPrevText.substr(PrevNewLinePos1, strPrevText.length());
		cairo_text_extents_t PrevLine2extents;
		cairo_text_extents(m_pCairo, Line2.c_str(), &PrevLine2extents);
		cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
		cairo_rectangle(m_pCairo, (tDouble) stLabelInfo.u16XPos, (tDouble) stLabelInfo.u16YPos + EXTRA_LINE_SPACE_BETWEEN_LINES, PrevLine2extents.width + EXTRA_FONT_WIDTH, (tDouble)(PrevLine1extents.height + u16ExtraFontHeight));
		cairo_fill(m_pCairo);

	}
	else
	{
		cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
		if(stLabelInfo.eAlign == E_LABEL_HORIZONTAL_CENTER_ALIGNMENT)
		{			
			cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);
			tDouble x;
			x = m_u32Width / 2.0 - (extents.width + extents.x_bearing) / 2.0;
			cairo_rectangle(m_pCairo, x, (tDouble)(stLabelInfo.u16YPos - u16FontHeight), extents.width + EXTRA_FONT_WIDTH, (tDouble)u16ExtraFontHeight);			
		}
		else
			cairo_rectangle(m_pCairo, (tDouble)stLabelInfo.u16XPos, (tDouble)(stLabelInfo.u16YPos - u16FontHeight), extents.width + EXTRA_FONT_WIDTH, (tDouble)u16ExtraFontHeight);
		cairo_fill(m_pCairo);
	}


	//Draw the Text
	string strcurrentText = stLabelInfo.strText;
	size_t newlinePos1 = strcurrentText.find("\n");
	if(newlinePos1 != string::npos)
	{
		string Line1 = strcurrentText.substr(0, newlinePos1);
		cairo_move_to(m_pCairo, (tDouble)stLabelInfo.u16XPos, (tDouble)stLabelInfo.u16YPos);
		cairo_set_source_rgb(m_pCairo, stLabelInfo.stcolor.u8Red / 255.0, stLabelInfo.stcolor.u8Green / 255.0, stLabelInfo.stcolor.u8Blue / 255.0);
		cairo_show_text(m_pCairo, Line1.c_str());

		cairo_text_extents_t Line1extents;
		cairo_text_extents(m_pCairo, Line1.c_str(), &Line1extents);

		string Line2 = strcurrentText.substr(newlinePos1 + 1, strcurrentText.length());
		cairo_move_to(m_pCairo, (tDouble)stLabelInfo.u16XPos, stLabelInfo.u16YPos + Line1extents.height);
		cairo_set_source_rgb(m_pCairo, stLabelInfo.stcolor.u8Red / 255.0, stLabelInfo.stcolor.u8Green / 255.0, stLabelInfo.stcolor.u8Blue / 255.0);
		cairo_show_text(m_pCairo, Line2.c_str());

	}
	else
	{
		if(stLabelInfo.eAlign == E_LABEL_HORIZONTAL_CENTER_ALIGNMENT)
		{			
			cairo_text_extents (m_pCairo, stLabelInfo.strText.c_str(), &extents);
			tDouble x;
			x = m_u32Width / 2.0 - (extents.width + extents.x_bearing) / 2.0;
			cairo_move_to(m_pCairo, x, (tDouble)stLabelInfo.u16YPos);
		}
		else
			cairo_move_to(m_pCairo, (tDouble)stLabelInfo.u16XPos, (tDouble)stLabelInfo.u16YPos);

		cairo_set_source_rgb(m_pCairo, stLabelInfo.stcolor.u8Red / 255.0, stLabelInfo.stcolor.u8Green / 255.0, stLabelInfo.stcolor.u8Blue / 255.0);
		cairo_show_text(m_pCairo, stLabelInfo.strText.c_str());
	}
}

/************************************************************************
*FUNCTION			: DrawOutLine
*DESCRIPTION		: Drawing the Outline
*PARAMETER			: struct (I)
*PARAMETER				Label Information
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::DrawOutLine(LineInfo stLineInfo)
{
	ETG_TRACE_USR4(("DrawOutLine"));
	cairo_set_source_rgb(m_pCairo, stLineInfo.stcolor.u8Red / 255.0, stLineInfo.stcolor.u8Green / 255.0, stLineInfo.stcolor.u8Blue / 255.0);
    cairo_set_line_width(m_pCairo, (tDouble)stLineInfo.u16Width);
    cairo_rectangle(m_pCairo, (tDouble)stLineInfo.u16XPos, (tDouble)stLineInfo.u16YPos, (tDouble)(m_u32Width - CAIRO_OUTLINE_XPOS), (tDouble)(m_u32Height - CAIRO_OUTLINE_YPOS));
    cairo_stroke(m_pCairo);

}

/************************************************************************
*FUNCTION			: DrawProgressBar
*DESCRIPTION		: Drawing the ProgressBar
*PARAMETER			: struct (I)
*PARAMETER				Progress Bar Information
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::DrawProgressBar(ProgressBarInfo stProgressInfo)
{
	ETG_TRACE_USR4(("DrawProgressBar"));

	tU16 u16ProgressBarWidth = stProgressInfo.u16Width;
	tU16 u16XPos = stProgressInfo.u16XPos;
	tDouble radius = stProgressInfo.u16Height / 2.0;
	tDouble degrees = M_PI / 180.0;

	cairo_set_line_width(m_pCairo, (tDouble)stProgressInfo.u16BorderWidth);
	cairo_set_source_rgb(m_pCairo, stProgressInfo.stBorderColor.u8Red / 255.0, stProgressInfo.stBorderColor.u8Green / 255.0,
											stProgressInfo.stBorderColor.u8Blue / 255.0);
	cairo_new_sub_path (m_pCairo);
	cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 90 * degrees, 180 * degrees);
	cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + radius, radius, 180 * degrees, 270 * degrees);
	cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + radius, radius, -90 * degrees, 0 * degrees);
	cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 0 * degrees, 90 * degrees);
	cairo_close_path (m_pCairo);
	cairo_stroke(m_pCairo);

	cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
	cairo_new_sub_path (m_pCairo);
	cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 90 * degrees, 180 * degrees);
	cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + radius, radius, 180 * degrees, 270 * degrees);
	cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + radius, radius, -90 * degrees, 0 * degrees);
	cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 0 * degrees, 90 * degrees);
	cairo_close_path (m_pCairo);
	cairo_fill(m_pCairo);

	tU16 u16FillXPos =static_cast<tU16> (u16XPos + 5);
	tU16 u16FillYPos =static_cast<tU16> (stProgressInfo.u16YPos + 3);
	tU16 u16FillHeight =static_cast<tU16> (stProgressInfo.u16Height - 5);

	tDouble Fillradius = u16FillHeight / 2.0;
	tU16 u16FillWidth = stProgressInfo.u16PercentageFillValue;

	if(u16FillWidth != 0)
	{
		cairo_set_source_rgb(m_pCairo, stProgressInfo.stDefaultFillColor.u8Red / 255.0, stProgressInfo.stDefaultFillColor.u8Green / 255.0,
												stProgressInfo.stDefaultFillColor.u8Blue / 255.0);
		cairo_new_sub_path (m_pCairo);
		cairo_arc (m_pCairo, u16FillXPos + Fillradius, u16FillYPos + u16FillHeight - Fillradius, Fillradius, 90 * degrees, 180 * degrees);
		cairo_arc (m_pCairo, u16FillXPos + Fillradius, u16FillYPos + Fillradius, Fillradius, 180 * degrees, 270 * degrees);
		cairo_arc (m_pCairo, u16FillXPos + u16FillWidth - Fillradius, u16FillYPos + Fillradius, Fillradius, -90 * degrees, 0 * degrees);
		cairo_arc (m_pCairo, u16FillXPos + u16FillWidth - Fillradius, u16FillYPos + u16FillHeight - Fillradius, Fillradius, 0 * degrees, 90 * degrees);
		cairo_close_path (m_pCairo);
		cairo_fill(m_pCairo);
	}

	if(stProgressInfo.u16PercentageCompleted != 0)
	{
		cairo_t* pProgressTextCr = NULL;
		cairo_surface_t* pProgressTextSurface = NULL;
		pProgressTextSurface = cairo_surface_create_similar(cairo_get_target(m_pCairo), CAIRO_CONTENT_COLOR_ALPHA, m_u32Width, m_u32Height);
		pProgressTextCr = cairo_create(pProgressTextSurface);

		cairo_set_font_face(pProgressTextCr, m_pfont);
		cairo_set_font_size(pProgressTextCr, (tDouble)stProgressInfo.u16FontHeight);

		//Draw Percentage
		cairo_text_extents_t extents;
		cairo_text_extents(m_pCairo, "99%", &extents);
		cairo_move_to(pProgressTextCr, (tDouble)(stProgressInfo.u16XPos + u16ProgressBarWidth/2.0), (tDouble)(stProgressInfo.u16YPos + 25));
		cairo_set_source_rgb(pProgressTextCr, stProgressInfo.stBorderColor.u8Red / 255.0, stProgressInfo.stBorderColor.u8Green / 255.0,
													stProgressInfo.stBorderColor.u8Blue / 255.0);

		std::ostringstream out;
		out << stProgressInfo.u16PercentageCompleted  <<  "%";
		cairo_show_text(pProgressTextCr, out.str().c_str());

		ETG_TRACE_USR4(("Percentage Completed:%s", out.str().c_str()));

		cairo_set_operator(pProgressTextCr, CAIRO_OPERATOR_SOURCE);
		cairo_set_source_surface(m_pCairo, pProgressTextSurface, 0.0, 0.0);
		cairo_paint(m_pCairo);

		cairo_surface_destroy(pProgressTextSurface);
		cairo_destroy(pProgressTextCr);
	}

}

/************************************************************************
*FUNCTION			: DrawProgressBar1
*DESCRIPTION		: Drawing the ProgressBar1
*PARAMETER			: struct (I)
*PARAMETER				Progress Bar Information
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::DrawProgressBar1(ProgressBarInfo stProgressInfo)
{
	ETG_TRACE_USR4(("DrawProgressBar1"));

	cairo_set_source_rgb(m_pCairo, 0.0, 0.0, 0.0);
	cairo_rectangle(m_pCairo, (tDouble)stProgressInfo.u16XPos, (tDouble)stProgressInfo.u16YPos, (tDouble)(stProgressInfo.u16Width *  TOTAL_PROGRESS_BAR), (tDouble)stProgressInfo.u16Height);
	cairo_fill(m_pCairo);

	tU16 u16ProgressBarWidth = stProgressInfo.u16Width;
	tU16 u16XPos = stProgressInfo.u16XPos;
	tU16 u16FillProgressCount = stProgressInfo.u16PercentageCompleted;

	tDouble radius = stProgressInfo.u16Height / 2.0;
	tDouble degrees = M_PI / 180.0;

	for(tU16 u16Count = 1; u16Count <= TOTAL_PROGRESS_BAR; u16Count++)
	{
		if(u16Count == FIRST_PROGRESS_BAR)
		{
			cairo_new_sub_path (m_pCairo);
			cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 90 * degrees, 180 * degrees);
			cairo_arc (m_pCairo, u16XPos + radius, stProgressInfo.u16YPos + radius, radius, 180 * degrees, 270 * degrees);
			cairo_rectangle(m_pCairo, (tDouble)u16XPos, (tDouble)stProgressInfo.u16YPos, (tDouble)u16ProgressBarWidth, (tDouble)stProgressInfo.u16Height);
			cairo_close_path (m_pCairo);

		}
		else if(u16Count == LAST_PROGRESS_BAR)
		{
			cairo_new_sub_path (m_pCairo);
			cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + radius, radius, -90 * degrees, 0 * degrees);
			cairo_arc (m_pCairo, u16XPos + stProgressInfo.u16Width - radius, stProgressInfo.u16YPos + stProgressInfo.u16Height - radius, radius, 0 * degrees, 90 * degrees);
			cairo_rectangle(m_pCairo, (tDouble)u16XPos, (tDouble)stProgressInfo.u16YPos, (tDouble)u16ProgressBarWidth, (tDouble)stProgressInfo.u16Height);
			cairo_close_path (m_pCairo);
		}
		else
			cairo_rectangle(m_pCairo, (tDouble)u16XPos, (tDouble)stProgressInfo.u16YPos, (tDouble)u16ProgressBarWidth, (tDouble)stProgressInfo.u16Height);

		if(u16Count <= u16FillProgressCount )
			cairo_set_source_rgb(m_pCairo, stProgressInfo.stCompletePercentageColor.u8Red / 255.0, stProgressInfo.stCompletePercentageColor.u8Green / 255.0,
															stProgressInfo.stCompletePercentageColor.u8Blue / 255.0);
		else
			cairo_set_source_rgb(m_pCairo, stProgressInfo.stDefaultFillColor.u8Red / 255.0, stProgressInfo.stDefaultFillColor.u8Green / 255.0,
															stProgressInfo.stDefaultFillColor.u8Blue / 255.0);
		cairo_fill_preserve(m_pCairo);

		cairo_set_line_width(m_pCairo, (tDouble)stProgressInfo.u16BorderWidth);
		cairo_set_source_rgb(m_pCairo, stProgressInfo.stBorderColor.u8Red / 255.0, stProgressInfo.stBorderColor.u8Green / 255.0,
											stProgressInfo.stBorderColor.u8Blue / 255.0);
		cairo_stroke(m_pCairo);

		u16XPos =static_cast<tU16> (u16XPos + u16ProgressBarWidth);
	}
}

/************************************************************************
*FUNCTION			: TakeScreenShot
*DESCRIPTION		: Capturing the Screen in the form of png
*PARAMETER			: string (I)
*PARAMETER				FileName
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::TakeScreenShot(string strFileName)
{
	cairo_surface_t *surface;
	string strFileNameWithPath("");
	strFileNameWithPath.assign("/var/opt/bosch/persistent/swupdate/");
	strFileNameWithPath.append(strFileName);
	strFileNameWithPath.append(".png");
	surface = cairo_image_surface_create_for_data(m_U8FrameBuffer,
												  CAIRO_FORMAT_ARGB32,
												  m_u32Width,
												  m_u32Height,
												  m_u32Stride);
	cairo_surface_write_to_png(surface,strFileNameWithPath.c_str());
	cairo_surface_destroy(surface);

}

/************************************************************************
*FUNCTION			: DrawTranslatedText
*DESCRIPTION		: Draw the Translated Text.
*PARAMETER			: struct (I)
*PARAMETER				Label Information.
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
13-07-2016  | 0.3       | AGL5KOR           | Changed the logic to check delimiter and split text into multi-line if required 
************************************************************************/
tVoid cairoadapter::DrawTranslatedText(LabelInfo stLabelInfo)
{
	ETG_TRACE_USR4(("DrawTranslatedText"));
	
	//check if delimiter is empty
	if(!m_strDelimiter.empty())
		DrawTextWithDelimiter(stLabelInfo, true);
	else
		DrawTextWithoutDelimiter(stLabelInfo, true);
}



/************************************************************************
*FUNCTION			: DrawTestImage
*DESCRIPTION		: Draw the Test Image
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
tVoid cairoadapter::DrawTestImage()
{
	cairo_surface_t *surface;
	cairo_surface_t *background;
	cairo_surface_t *logo;
	cairo_t *cr;
	int logo_width;
	int logo_height;

	/* First, we create the cairo surface from the mapped DRM buffer */
	surface = cairo_image_surface_create_for_data(m_U8FrameBuffer,
												  CAIRO_FORMAT_ARGB32,
												  m_u32Width,
												  m_u32Height,
												  m_u32Stride);
	if (surface == NULL)
	{
		ETG_TRACE_USR4(("Failed to create cairo surface."));
		return;
	}

	 /* Secondly, create a cairo context for the surface */
	cr = cairo_create(surface);

	/* Now clear the buffer with black */
	cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
	cairo_paint(cr);

	 /* Now we paint a background image with reduced opacity.
	     *
	     * Images can be loaded directly into cairo surfaces and then used to draw
	     * them on the target.
	     */
	background = cairo_image_surface_create_from_png("Resources/JAC.png");
	cairo_set_source_surface(cr, background, 0.0, 0.0);
	cairo_paint_with_alpha(cr, 0.3);
	cairo_surface_destroy(background);

	/* Draw yellow rectangle on top. This is a simple example of drawing a
	     * geometrical figure. The first, the paint color is set, then the rectangle
	     * is prepared and its outline painted using cairo_stroke().
	     *
	     * For an example of a filled rectangle refer to draw_progress_bar().
	     */
	cairo_set_source_rgb(cr, 0.0, 1.0, 0.0);
	cairo_set_line_width(cr, 4.0);
	cairo_rectangle(cr, 4.0, 4.0, m_u32Width - 8.0, m_u32Height - 8.0);
	cairo_stroke(cr);

	  /* draw logo bottom right */
	logo = cairo_image_surface_create_from_png("Resources/bosch2.png");
	logo_width = cairo_image_surface_get_width(logo);
	logo_height = cairo_image_surface_get_height(logo);
	cairo_set_source_surface(cr, logo,
								m_u32Width - logo_width - 10.0,
								m_u32Height - logo_height - 10.0);
	cairo_paint(cr);
	cairo_surface_destroy(logo);

	cairo_destroy(cr);
	cairo_surface_destroy(surface);
}

