/**
 * @addtogroup ai_sw_update
 * @{
 *
 * @file       Fingerprint.h
 * @author     Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date       Thu 14 Apr 2016, 13:20
 * @copyright  Robert Bosch Car Multimedia GmbH
 * @brief      The fingerprint of a filesystem tree consists of digests, i.e.,
 *             MD5 sums of all files in the tree, together with the files stats
 *             and an overall digest to ensure that there are no changes in the
 *             tree structure.
 */

#ifndef FINGERPRINT_H
#define FINGERPRINT_H

#include <map>
#include <vector>
#include <string>

#include "DigestStat.h"

#define FINGERPRINT_COMMENT_CHAR ';'

namespace fingerprint {

typedef std::string FileName;
/** A map is used for storing the files metadata. Thus, it is possible to 
 *  compute a unique digest of the whole file tree. */
typedef std::map< FileName, DigestStat > FileMap;

/** A list of file names that are excluded from fingerprinting. */
typedef std::vector<FileName> ExcludedList;

/** @brief Representation of a file tree fingerprint.
 */
class Fingerprint
{
   public:
      Fingerprint ();
      
      /** Print the complete map of filenames and MD5 sums.
       *
       * Format:
       *    ;f <fingerprint>
       *    <MD5sum1 [32]>  <filename1>
       *    ; <Stat1 []> <filename1>
       *    <MD5sum2 [32]>  <filename2>
       *    ; <Stat2 []> <filename2>
       *    ...
       */
      void fprint (FILE* file);
      /** @brief Read a map of filenames and MD5 sums from file.
       *
       *  The file should have the format described in fprint(). 
       *  Alternatively, a plain md5 sum file can be read.  */
      bool fread (FILE* file, const FileName& prefix = "");

      /** @brief Use User ID information from stat while checking. */
      void useUid (bool b);
      /** @brief Use Group ID information from stat while checking. */
      void useGid (bool b);
      /** @brief Use access right information from stat while checking. */
      void useMode (bool b);
      /** @brief Switch MD5 mode. This is used to check a plain MD5 sum file,
       * i.e., one that was build with a md5sum command line call.  */
      void useCap (bool b);
      /** @brief Use file capabilities of file while checking. */
      void onlyMd5Sum (bool b);
      /** @brief Set the maximum number of errors written to output while 
       * checking. */
      void setErrMax (unsigned int m);
      /** Walks over the member FileMap and the one from 'other' and compares
       *  the entries. It is assumed that this object is the reference and
       *  'other' reflects the state of the filesystem. */
      bool check(const Fingerprint& other) const;
      /** Compute a digest of the complete FileMap. */
      void computeRootDigest();
      /** Recomputes root digest and compares to the stored one.
       * @return True, if the digests are the same. False, if they are
       *          different, or there was none stored before. */
      bool checkRootDigest ();

      size_t numFiles () const;
      Digest rootDigest () const;

      /** Return the root directory of the fingerprint. */
      FileName root () const;
      /** Set the root of the fingerprint. */
      void setRoot(const FileName& root);

      /** Return the DigestStat information for a given FileName. */
      DigestStat& map(const FileName& fn);

      /** Direct access to the list of excluded files. */
      ExcludedList& excludedList ();
      /** Replace currect list with the given one. */
      void setExcludedList (const ExcludedList& el);
      /** Return if the given file name is on the excluded list. */
      bool isExcluded (const FileName& fn) const;
      /** Add a file name to the list of excluded files. */
      void addExcluded (const FileName& fn);

   private:
      /** A mapping from filenames to their digests and filesystem stats. */
      FileMap _map;

      // A set of paths which are excluded from fingerprinting.
      ExcludedList _excluded;

      /** Digest of the complete _map. */
      Digest   _root_digest;
      FileName _root;

      /** Should only the basic md5sum of files (not directories be used in
       * check() ? */
      bool _only_md5sum; 
      /** Should the GID of a file be used for check()? */
      bool _use_gid;
      /** Should the UID of a file be used for check()? */
      bool _use_uid;
      /** Should the MODE of a file be used for check()? */
      bool _use_mode;
      /** Should the CAPABILITIES of a file be used for check()? */
      bool _use_cap;
      /** Maximum number of errors to output while checking. */
      unsigned int  _errmax;

      /** Emulates the behavior of 'md5sum'. Walks over the member FileMap and 
       * checks, if the 'other' map's entries have the same digests. */
      bool checkMd5Sum(const Fingerprint& fp_fs) const;
      /** Complete check of all entries in the maps. */
      bool checkDigestAndStat(const Fingerprint& fp_fs) const;
};


} // ns fingerprint

#endif
/** @} */


