/*
 * @file       Fingerprint.cpp
 * @author     Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date       Fri 25 Nov 2016, 09:34
 * @copyright  Robert Bosch Car Multimedia GmbH
 */
#include <algorithm>
#include <cstdio>
#include <climits>
#include <openssl/evp.h>
#include "Fingerprint.h"

#include "my_etg.h"

using namespace fingerprint;

// ----------------------------------------------------------------------------
// -------------------------------------------------------------------- HELPERS
/** Remove prefix from beginning of path, if path begins with prefix. 
 *
 * @return True, if prefix was erased from path, false otherwise.
 */
void stripPrefix(FileName& path, const FileName& prefix) {
   if (prefix.empty()) return; 

   if(path.length() >= prefix.length() 
         && std::equal(prefix.begin(), prefix.end(), path.begin()))
   {
      ETG_TRACE_USR4(("Before Stripping: %s", path.c_str() ));
      path.erase(0, prefix.length());
      ETG_TRACE_USR4(("           After: %s", path.c_str() ));
   }
}

Fingerprint::Fingerprint ()
   : _only_md5sum(false), _use_gid(true), _use_uid(true), _use_mode(true), _use_cap(true), _errmax(UINT_MAX)
{}

void Fingerprint::fprint (FILE* file)
{
   // Print fingerprint
   char dig_hex[DIGEST_STR_LEN];
   char stat_hex[STAT_STR_LEN];

   _root_digest.print(dig_hex, DIGEST_STR_LEN);
   fprintf(file, "%cf %s %s\n", FINGERPRINT_COMMENT_CHAR, dig_hex, _root.c_str());

   for (ExcludedList::const_iterator it = _excluded.begin();
         it != _excluded.end();
         ++it)
   {
      fprintf(file, "%ce %s\n", FINGERPRINT_COMMENT_CHAR, it->c_str());
   }

   for (FileMap::const_iterator tuple = _map.begin();
         tuple != _map.end();
         ++tuple)
   {
      const FileName& fn = tuple->first;
      const DigestStat& ds = tuple->second;
      if ( ds.isRegularFile() || ds.isLinkToReg() ) {
         ds.printDigest(dig_hex, DIGEST_STR_LEN);
         fprintf(file, "%s  %s\n", dig_hex, fn.c_str());

         if (ds.isCapDigestSet()) {
            ds.printCapDigest(dig_hex, DIGEST_STR_LEN);
            fprintf(file, "%cc %s %s\n", FINGERPRINT_COMMENT_CHAR, dig_hex, fn.c_str());
         }
      }
      ds.printStat(stat_hex, STAT_STR_LEN);
      fprintf(file, "%c %s %s\n", FINGERPRINT_COMMENT_CHAR, stat_hex, fn.c_str());
   }
}

bool Fingerprint::fread (FILE* file, const FileName& prefix)
{
   char* buf = 0;
   size_t buflen = 0;

   _map.clear();

   ETG_TRACE_USR3(("Stripping \"%s\" from filenames", prefix.c_str() ));
   for (int read =(static_cast<int> (getline(&buf, &buflen, file))), line_num = 1;
        read > 0;
        read =(static_cast<int> ( getline(&buf, &buflen, file))), ++line_num)
   {
      // replace newline character by null
      buf[read-1] = 0;

      if (buf[0] != FINGERPRINT_COMMENT_CHAR) { // read digest
         if (read < 2*DIGEST_LEN + 3) { // hex + sep + at least 1 char filename
            ETG_TRACE_FATAL(("Reading digest: line %i too short.", line_num));
            return false;
         }

         FileName file_name(buf+2+2*DIGEST_LEN);
         stripPrefix(file_name, prefix);
         _map[file_name].readDigest(buf, 2*DIGEST_LEN);
      }
      else { // read comment
         if (buf[1] == 'f' && buf[2] == ' ') { // read fingerprint
            if (read < 3 + 2*DIGEST_LEN + 2) { // prefix + fingerprint + min 1 char
               ETG_TRACE_FATAL(("Reading fingerprint: line %i too short.", line_num));
               return false;
            }

            _root_digest.read(buf + 3, 2*DIGEST_LEN);
            _root.assign(buf + 4 + 2*DIGEST_LEN);
         }
         else if (buf[1] == 'e' && buf[2] == ' ') { // read exclude
            if (read < 4) { // prefix + min 1 char
               ETG_TRACE_FATAL(("Reading exclude: line %i too short.", line_num));
               return false;
            }
            FileName ex;
            ex.assign(buf+3);
            _excluded.push_back(ex);
         }
         else if (buf[1] == ' ') { // read stat
            if (read < 2 + 2*STAT_LEN + 2) { // prefix + stat hex + filename
               ETG_TRACE_FATAL(("Reading stat: line %i too short.", line_num));
               return false;
            }

            FileName file_name;
            file_name.assign(buf+3+2*STAT_LEN);
            stripPrefix(file_name, prefix);

            _map[file_name].readStat(buf+2, 2*STAT_LEN);
         }
         else if (buf[1] == 'c') {
            if (read < 2 + 2*DIGEST_LEN + 2) { // prefix + capability hex + filename
               ETG_TRACE_FATAL(("Reading cap digest: line %i too short.", line_num));
               return false;
            }
            FileName file_name;
            file_name.assign(buf+4+2*DIGEST_LEN);
            _map[file_name].readCapDigest(buf+3, 2*DIGEST_LEN);
         }
      }
   }
   return true;
}

void Fingerprint::computeRootDigest()
{
   union { uid_t uid; unsigned char uc[UID_LEN]; } uid2uc;
   union { gid_t gid; unsigned char uc[GID_LEN]; } gid2uc;
   union { mode_t mode; unsigned char uc[MODE_LEN]; } mode2uc;

   EVP_MD_CTX* ctx = EVP_MD_CTX_create();
   EVP_DigestInit(ctx, EVP_md5());
   for (FileMap::const_iterator tuple = _map.begin();
         tuple != _map.end();
         ++tuple)
   {
      const DigestStat& ds = tuple->second;
      uid2uc.uid = ds.uid();
      gid2uc.gid = ds.gid();
      mode2uc.mode = ds.mode();
      Digest digest = ds.digest();
      Digest capDigest = ds.capDigest();
      // char hex[DIGEST_STR_LEN];
      // ds.digest.print(hex);
      // fprintf(stderr, "%s%d%d%d \t %s\n", hex, uid2uc.uid, gid2uc.gid, mode2uc.mode, tuple->first.c_str());
      EVP_DigestUpdate(ctx, digest.x, fingerprint::DIGEST_LEN);
      EVP_DigestUpdate(ctx, capDigest.x, fingerprint::DIGEST_LEN);
      EVP_DigestUpdate(ctx, uid2uc.uc, fingerprint::UID_LEN);
      EVP_DigestUpdate(ctx, gid2uc.uc, fingerprint::GID_LEN);
      EVP_DigestUpdate(ctx, mode2uc.uc, fingerprint::MODE_LEN);
      // Fix [blp4hi, 2018-02-02] include filename in digest
      const std::string & filename = tuple->first;
      EVP_DigestUpdate(ctx, filename.c_str(), filename.size());
   }
   EVP_DigestFinal(ctx, _root_digest.x, 0);
   EVP_MD_CTX_destroy(ctx);
}

bool Fingerprint::checkRootDigest ()
{
   Digest md(_root_digest);
   computeRootDigest();
   return md == _root_digest;
}

void Fingerprint::useUid (bool b)
{ _use_uid = b; }

void Fingerprint::useGid (bool b)
{ _use_gid = b; }

void Fingerprint::useMode (bool b)
{ _use_mode = b; }

void Fingerprint::useCap(bool b)
{ _use_cap = b; }

void Fingerprint::onlyMd5Sum (bool b)
{ _only_md5sum = b; }

void Fingerprint::setErrMax (unsigned int m)
{ _errmax = m; }

bool Fingerprint::check(const Fingerprint& fp_fs) const
{
   // check fingerprint first, unless stats should not be used
   if ( _use_uid && _use_gid && _use_mode && _use_cap) {
      if (_root_digest == fp_fs._root_digest) {
         ETG_TRACE_USR1(("Fingerprints are the same."));
         return true;
      }
      else {
         ETG_TRACE_FATAL(("Fingerprints are different. Doing full analysis."));
      }
   }

   return _only_md5sum ? checkMd5Sum(fp_fs) : checkDigestAndStat(fp_fs);
}

bool Fingerprint::checkDigestAndStat(const Fingerprint& fp_fs) const
{
   // walk over both maps to analyze errors.
   unsigned int err_count = 0;
   FileMap::const_iterator i = _map.begin();
   FileMap::const_iterator i_fs = fp_fs._map.begin();

   while (i != _map.end() && i_fs != fp_fs._map.end())  {
      if (err_count >= _errmax) {
         ETG_TRACE_FATAL(("Maximum number of errors reached." ));
         return false;
      }

      const FileName &name = i->first;
      const FileName &name_fs = i_fs->first;
      const DigestStat &stat = i->second;
      const DigestStat &stat_fs = i_fs->second;

      // @todo (blp4hi, 04.04.2016)
      // skip links from the reference file, since they are currently also skipped
      // in the filewalk. See fingerprint_tool.cpp. This should be changed. 
      if ( stat.isSymlink() ) { ++i; continue; }

      int comparison = name.compare(name_fs);
      if (comparison < 0) { // name < name_fs
         ETG_TRACE_FATAL(("File not found: %s", name.c_str() ));
         ETG_TRACE_FATAL(("        (Found: %s)", name_fs.c_str() ));
         ++i;
         ++err_count;
         continue;
      }
      else if (comparison > 0) { // name > name_fs
         ETG_TRACE_FATAL(("Unexpected file found: %s", name_fs.c_str() ));
         ++i_fs;
         ++err_count;
         continue;
      }
      else if ( stat.digest() != stat_fs.digest() ) {
         ETG_TRACE_FATAL(("Digests differ on: %s", name.c_str() ));
         ++err_count;
      }
      else if ( _use_uid && (stat.uid() != stat_fs.uid()) ) {
         ETG_TRACE_FATAL(("UIDs differ on: %s", name.c_str() ));
         ETG_TRACE_FATAL(("UID is %u, should be %u", stat.uid(), stat_fs.uid() ));
         ++err_count;
      }
      else if ( _use_gid && (stat.gid() != stat_fs.gid()) ) {
         ETG_TRACE_FATAL(("GIDs differ on: %s", name.c_str() ));
         ++err_count;
      }
      else if ( _use_mode && (stat.mode() != stat_fs.mode()) ) {
         ETG_TRACE_FATAL(("MODEs differ on: %s", name.c_str() ));
         ++err_count;
      }
      else if ( _use_cap && (stat.capDigest() != stat_fs.capDigest())) {
    	  ETG_TRACE_FATAL(("CAPs differ on: %s", name.c_str() ));
    	  ++err_count;
      }

      ++i;
      ++i_fs;
   }

   // there may be more files in the reference
   while ( i != _map.end() ) {
      const FileName &name = i->first;
      ETG_TRACE_FATAL(("File not found: %s", name.c_str() ));
      ++i;
      if (++err_count >= _errmax) {
         ETG_TRACE_FATAL(("Maximum number of errors reached." ));
         return false;
      }
   }

   // there may be more files on the filesystem
   while ( i_fs != fp_fs._map.end() ) {
      const FileName &name_fs = i_fs->first;
      ETG_TRACE_FATAL(("Unexpected file found: %s", name_fs.c_str() ));
      ++i_fs;
      if (++err_count >= _errmax) {
         ETG_TRACE_FATAL(("Maximum number of errors reached." ));
         return false;
      }
   }

   return (err_count == 0);
}

bool Fingerprint::checkMd5Sum(const Fingerprint& fp_fs) const
{
   unsigned int err_count = 0;

   FileMap::const_iterator i_fs;
   for (FileMap::const_iterator i = _map.begin(); i != _map.end(); i++) {
      if (err_count >= _errmax) {
         ETG_TRACE_FATAL(("Maximum number of errors reached." ));
         return false;
      }

      if ( ! i->second.isDigestSet() ) {
         continue;
      }
      i_fs = fp_fs._map.find( i->first );
      if (i_fs == fp_fs._map.end()) {
         ETG_TRACE_FATAL(("File not found: %s", i->first.c_str() ));
         ++err_count;
         continue;
      }

      if ( i->second.digest() != i_fs->second.digest() ) {
         ETG_TRACE_FATAL(("Digests differ on: %s", i->first.c_str() ));
         ++err_count;
      }
   }

   return (err_count == 0);
}


size_t Fingerprint::numFiles () const
{ return _map.size(); }

Digest Fingerprint::rootDigest() const
{ return _root_digest; }

FileName Fingerprint::root() const
{ return _root; }

void Fingerprint::setRoot(const FileName& root_)
{ _root = root_; }

DigestStat& Fingerprint::map(const FileName& fn)
{ return _map[fn]; }

ExcludedList& Fingerprint::excludedList ()
{ return _excluded; }

bool Fingerprint::isExcluded (const FileName& fn) const
{ return (std::find(_excluded.begin(), _excluded.end(), fn) != _excluded.end()); }

void Fingerprint::addExcluded (const FileName& fn)
{ _excluded.push_back(fn); }

void Fingerprint::setExcludedList (const ExcludedList& el)
{ _excluded = el; }

