/*
 * fcswupd_parserBosch.cpp
 *
 *  Created on: Apr 15, 2014
 *      Author: efs1hi
 */

#include <sys/stat.h>
#include <dirent.h>
#include "tinyxml/tinyxml.h"
#include "util/swu_filesystem.h"
#include "util/swu_util.hpp"
#include "util/fcswupd_parserUtil.h"
#include "util/fcswupd_globalLog.h"
#include "fcswupd_parserBosch.h"
#include "main/fcswupd_propDevMgr.h"
#include "main/fcswupd_systemData.h"
#include "main/fcswupd_main.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_PARSER
#include "trcGenProj/Header/fcswupd_parserBosch.cpp.trc.h"
#endif

using namespace std;

namespace fcswupdate {

BoschParser::BoschParser() : _document(0), _errorCode(tenSwUpdateError_OK) {
}

TiXmlDocument *BoschParser::readBoschXmlFromMedium(trSourceInfo const &sourceInfo, tenSwUpdateError *errorCode) {
   tenSwUpdateError errorCodeLocal;
   ETG_TRACE_USR1(("BoschParser::readBoschXmlFromMedium: START"));
   if (!errorCode) {
      errorCode=&errorCodeLocal;
   }
   *errorCode=tenSwUpdateError_OK;
   std::string filePath = sourceInfo.getInfoFilePathAndName();

   TiXmlDocument *document = new TiXmlDocument(filePath.c_str());
   SWU_ASSERT_RETURN_VAL(document, 0);
   swu::ScopedPointer<TiXmlDocument> scopedDoc(document);
   if (!document->LoadFile()) {
      *errorCode = tenSwUpdateError_ERROR_METAINFO_READ;
      const char *errorDesc = document->ErrorDesc();
      int row = document->ErrorRow();
      int col = document->ErrorCol();
      printf("readBoschXmlFromMedium: Error at row %d, col %d: %s", row, col, errorDesc);
      ETG_TRACE_ERR(("readBoschXmlFromMedium: Could not load bosch.xml from %s", filePath.c_str()));
      ETG_TRACE_ERR(("Reason: %s", errorDesc));
      ETG_TRACE_ERR(("Error at row: %d", row));
      ETG_TRACE_ERR(("Error at col: %d", col));
      return 0;
      //SWU_ASSERT_RETURN_VAL_ALWAYS(result);
   }

   // apply pre-filters to raw-document
   if (_preFilters) {
      ReleaseFilterContext context;
      context.sourceInfo=sourceInfo;
      *errorCode=_preFilters->apply(document, context);
      if (*errorCode != tenSwUpdateError_OK) {
         ETG_TRACE_USR1(("FcSwUpdCore::readBoschXmlFromMedium _bxmlFilterChain failed"));
         return 0;
      }
   }

   ETG_TRACE_USR2(("readBoschXmlFromMedium():END"));
   return scopedDoc.release();

}


#define SET_PARSER_ERROR(ERR)            \
   if (errorCode) { *errorCode=ERR; }           \
   _errorCode=ERR;                              \

#define SET_PARSER_ERROR_RETURN(ERR)                            \
   SET_PARSER_ERROR(ERR)                                        \
   _errorCode=ERR;                                              \
   if (_errorCode != tenSwUpdateError_OK) { return result; }

#define SET_PARSER_ERROR_CONT(ERR)                      \
   SET_PARSER_ERROR(ERR)                                \
   _errorCode=ERR;                                      \
   if (_errorCode != tenSwUpdateError_OK) { continue; }

std::list<TiXmlElement> BoschParser::getReleaseList(trSourceInfo const &sourceInfo, tenSwUpdateError *errorCode) {
   std::string relDir=sourceInfo.path;
   std::string relFile=sourceInfo.infoFile;
   tenSwUpdateError errCode;
   std::list<TiXmlElement> result;
   ETG_TRACE_USR1(("ParserBosch::getReleaseList %s", relDir.c_str()));
   SET_PARSER_ERROR_RETURN(tenSwUpdateError_OK);

   if (! relDir.c_str()) {
      ETG_TRACE_ERR(("No path given. Stop parsing."));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }

   else if (!swu::isDirectory(relDir)) {
      ETG_TRACE_ERR(("Not a directory:%s", relDir.c_str()));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }
   else if (!relFile.empty() && !swu::isFile(relDir + "/" + relFile)) {
      ETG_TRACE_ERR(("File %30s does not exist in dir:%s", relFile.c_str(), relDir.c_str()));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }


   ETG_TRACE_USR1(("Parsing path '%s'", relDir.c_str()));
   if (!relFile.empty()) {
      ETG_TRACE_USR1(("looking only for file '%s'", relFile.c_str()));
   }

   struct dirent *dent=0;
   DIR *dir=opendir(relDir.c_str());

   if (dir) {
      while (0 != (dent = readdir(dir))) {
         char* filename = dent->d_name;

         if( ! swu::hasSuffix(filename, "xml") ) {
            continue;
         }

         if (!relFile.empty() && !swu::isSameFile(relDir + "/" + relFile, relDir + "/" + filename)) {
            continue;
         }

         trSourceInfo sourceInfo1;
         if(_srcFilters && Config::instance()->cfg_ApplyXMLFilterChain.readAsBool() && sourceInfo.enSourceType != tenSourceType_SCOMO_INDEX) {
            ETG_TRACE_USR4(("BoschParser::getReleaseList:before filter srcPath:%s", sourceInfo.getInfoFilePathAndName().c_str()));
            ReleaseFilterContext context;
            context.sourceInfo=sourceInfo;
            
            errCode=_srcFilters->apply(sourceInfo.getInfoFilePathAndName(), context);
            SET_PARSER_ERROR_RETURN(errCode);
            
            sourceInfo1 = context.sourceInfo;
            ETG_TRACE_USR4(("BoschParser::getReleaseList:after filter srcPath:%s", sourceInfo1.getInfoFilePathAndName().c_str()));
         } else {
            sourceInfo1=sourceInfo;
            sourceInfo1.infoFile=filename;
         }
                  
         TiXmlDocument *doc=readBoschXmlFromMedium(sourceInfo1,  &errCode);
         if (!doc) {
            ETG_TRACE_USR1(("Parser getReleaseList: FOUND nothing _errorCode=%u", _errorCode));
            SET_PARSER_ERROR_CONT(errCode);
         }
         ETG_TRACE_USR1(("Parser getReleaseList: FOUND Release"));
         TiXmlElement *root = doc->FirstChildElement();
         SWU_ASSERT_RETURN_VAL(root, result);
         TiXmlElement *overall = root->FirstChildElement("OVERALL");
         SWU_ASSERT_RETURN_VAL(overall, result);
         if (!relFile.empty()) {
            swu::setTextChild(overall, "INFO_FILE", relFile.c_str());
         }

         swu::setTextChild(overall, "MEDIUM_DRIVE", relDir.c_str());
         swu::setUIntChild(overall, "SOURCE_TYPE", (tU32)sourceInfo1.enSourceType);
         // calculate file hash and add entry in DOM tree
         std::string hash;
         if (swu::calculateSHA256HashFromFile(sourceInfo1.getInfoFilePathAndName(), hash)) {
            swu::setTextChild(overall, "BXML_CHECKSUM", hash);
            ETG_TRACE_USR1(("Added node BXML_CHECKSUM [%s]", hash.c_str()));
         }
      
         ReleaseFilterContext context;
         context.sourceInfo=sourceInfo1;
         errCode =_postFilters->apply(doc, context);
         
         if (errCode != tenSwUpdateError_OK) {
            ETG_TRACE_ERR(("ERR !postFilters, error-code=%u", _errorCode ));
            SET_PARSER_ERROR_CONT(errCode);
         }

         result.push_back(*((TiXmlElement *)overall->Clone()));
         ETG_TRACE_USR1(("Loaded successfully bosch.xml from %s", relDir.c_str()));
      }
      closedir (dir);
   }
   return result;
}

bool BoschParser::setOverallSection(TiXmlElement const &overallSection) {
   ETG_TRACE_USR1(("BoschParser::setOverallSection"));
   std::string mediumPathP = swu::getTextFromChildOrEmpty(&overallSection, "MEDIUM_DRIVE");
   ETG_TRACE_USR1(("BoschParser::setOverallSection: got mediumPath=%s", mediumPathP.c_str()));
   if (mediumPathP.empty())
   {
      ETG_TRACE_ERR(("BoschParser::setOverallSection: missing MEDIUM_DRIVE"));
      _errorCode = tenSwUpdateError_ERROR_METAINFO_NOT_FOUND;
      return false;
   }
   trSourceInfo sourceInfo=PropDevMgr::instance()->getSourceInfo(mediumPathP);
   sourceInfo.infoFile=swu::getTextFromChildOrEmpty(&overallSection, "INFO_FILE");
   if (sourceInfo.infoFile.empty())
   {
      ETG_TRACE_ERR(("BoschParser::setOverallSection: missing INFO_FILE"));
      _errorCode = tenSwUpdateError_ERROR_METAINFO_NOT_FOUND;
      return false;
   }
   
   ETG_TRACE_USR1(("Found medium path %s", sourceInfo.path.c_str()));
   if (_document) {
      delete _document;
      _document = 0;
   }

   if(_srcFilters && Config::instance()->cfg_ApplyXMLFilterChain.readAsBool() && sourceInfo.enSourceType != tenSourceType_SCOMO_INDEX) {
      ETG_TRACE_USR4(("BoschParser::setOverallSection:before filter srcPath:%s", sourceInfo.getInfoFilePathAndName().c_str()));
      ReleaseFilterContext context;
      context.sourceInfo=sourceInfo;
      
      _errorCode=_srcFilters->apply(sourceInfo.getInfoFilePathAndName(), context);
      if(_errorCode != tenSwUpdateError_OK) {
         return false;
      }
      
      sourceInfo = context.sourceInfo;
      ETG_TRACE_USR4(("BoschParser::setOverallSection:after filter srcPath:%s", sourceInfo.getInfoFilePathAndName().c_str()));
   } 
   
   _document=readBoschXmlFromMedium(sourceInfo, &_errorCode);
   if (!_document) {
      return false;
   }
   
   TiXmlElement *root = _document->FirstChildElement();
   if (!root) {
      _errorCode=tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND;
      return false;
   }
   
   TiXmlElement *overall = root->FirstChildElement("OVERALL");
   if (!overall) {
      _errorCode=tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND;
      return false;
   }
   swu::setTextChild(overall, "MEDIUM_DRIVE", sourceInfo.path);
   swu::setTextChild(overall, "INFO_FILE", sourceInfo.infoFile);
   const char *bxmlChecksum = swu::getTextFromChild(&overallSection, "BXML_CHECKSUM", false);
   if(bxmlChecksum) {
      swu::setTextChild(overall, "BXML_CHECKSUM", bxmlChecksum);
   } else {
      std::string hash;
      if (swu::calculateSHA256HashFromFile(sourceInfo.getInfoFilePathAndName(), hash))
      {
         swu::setTextChild(overall, "BXML_CHECKSUM", hash);
      }
   }
   if (_postFilters) {
         ReleaseFilterContext context;
         context.sourceInfo=sourceInfo;
         _errorCode=_postFilters->apply(_document, context);
   }
   return _errorCode==tenSwUpdateError_OK;
}

TiXmlDocument *BoschParser::getReleaseXml(size_t id) {
   (void)id; 
   if (_document) {
      TiXmlDocument *doc=_document;
      _document=0;
      return doc;
   }
   return 0;
}

tenSwUpdateError BoschParser::getParserError(void) const {
   return _errorCode;
}

}
