#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include "util/swu_util.hpp"
#include "util/swu_constants.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_bootChain.h"
#include "util/swu_execCommand.h"
#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "config/fcswupd_config.hpp"
#include "main/fcswupd_component.h"
#include "main/fcswupd_srv.h"
#include "main/fcswupd_main.h"
#include "main/fcswupd_mainMessages.hpp"
#include "main/fcswupd_systemData.h"
#include "main/fcswupd_srvGeneric.h"
#include "main/fcswupd_spmIf.h"
#include "main/fcswupd_campaignmanager.h"
#include "ctrl/fcswupd_ctrl.h"
#include "main/fcswupd_spmIf.h"
#include "main/fcswupd_propDevMgr.h"
#include "util/swu_execCommand.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>




#define ETG_I_FILE_PREFIX fcswupdate::SrvGeneric::instance()->

#include "util/fcswupd_trace.hpp"
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_MIB_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_srvGeneric.cpp.trc.h"
#endif


namespace fcswupdate {


VerifyChecksumWorker::VerifyChecksumWorker(act_t act, FCSWUPD_NS_FCSWUPD_T::tenVerifyType verifyType, std::string  area):
      _act(act),
      _verifyType(verifyType),
      _area(area), 
      _verifyResult(FCSWUPD_NS_FCSWUPD_T::tenVerifyResult__OtherError),
      _success(false) {
   ETG_TRACE_USR1((" Msg_ChecksumWorkerFinished:CTOR _act=%u verifyType=%u area=%s", 
                   _act, verifyType, area.c_str()));

}


void VerifyChecksumWorker::Msg_ChecksumWorkerFinished::vTrace() {
   ETG_TRACE_USR1((" Msg_ChecksumWorkerFinished(worker=%p)", worker));

}

void VerifyChecksumWorker::notifyDone() {
   ETG_TRACE_COMP(("VerifyChecksumWorker::notifyDone()"));
   // Receiver will take responsibility for deleting *this*
   Msg_ChecksumWorkerFinished *msg=new Msg_ChecksumWorkerFinished(this);
   msg->bNotifyLater();
   msg=0;
}

void VerifyChecksumWorker::getAsfResult(FCSWUPD_NS_FCSWUPD_T::trVerifyChecksumResult &asfResult) {
  asfResult.setEnType(_verifyType);
  asfResult.setArea(_area);
  asfResult.setChecksum(_checksum);
  asfResult.setEnResult(_verifyResult);
}

bool
VerifyChecksumWorker::parseChecksumFile( const ::std::string & outputFile, std::string &ckSum) {
	ETG_TRACE_COMP(("VerifyChecksumWorker::parseChecksumFile()"));


	const int fingerprintSize( 36 ); // Generous estimation
	const int lineSize( 2 * fingerprintSize ); // Generous estimation
	char line[ lineSize ];
	char fingerprintBuf[ fingerprintSize ];
	char successBuf[ lineSize ];
        std::string successString;
        ckSum="";
        char *check=0;
        
	::std::FILE *fp;
	fp = ::std::fopen(outputFile.c_str(), "r");
        if( !fp ) {
           ETG_TRACE_COMP(("VerifyChecksumWorker::parseChecksumFile() - could not open outputFile"));
           return false;
        }
        
        while((ckSum.empty() || successString.empty()) &&  ! feof( fp ) ) {
           check = fgets( line, lineSize, fp );
           if( check == line ) {
              if (ckSum.empty() && sscanf( line, "Fingerprint=\"%[^\"]s", fingerprintBuf )) {
                 fingerprintBuf[fingerprintSize - 1]=0;
                 ETG_TRACE_COMP(("VerifyChecksumWorker::parseChecksumFile() - fingerprint found"));
                 ckSum = fingerprintBuf; // Performs a copy
              }
              
              if( successString.empty() && sscanf( line, "SUCCESS=\"%[^\"]s", successBuf ) ) {
                 successBuf[lineSize - 1]=0;
                 ETG_TRACE_COMP(("VerifyChecksumWorker::parseChecksumFile() - SUCCESS found"));
                 successString=successBuf;
                 if (successString !="Check passed") {
                    ETG_TRACE_COMP(("VerifyChecksumWorker::parseChecksumFile() - SUCCESS not ok:%s", 
                                    successBuf));
                    break;
                 }
              }
           }
        } // End else

        fclose( fp );
              
        bool success= !ckSum.empty() && successString =="Check passed";
        return success;
}

void
VerifyChecksumWorker::theadFn() {
	ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn()"));

	// Calculate the checksum here...
	std::string commandLine;
	std::string outputFile;
	bool checksumUpdated( false );
	bool initialChecksPassed( false );

	if(( _verifyType != FCSWUPD_NS_FCSWUPD_T::tenVerifyType__Fingerprint ) && ( _verifyType != FCSWUPD_NS_FCSWUPD_T::tenVerifyType__Md5Sum )) {
           ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn(): invalid check-type:%u", _verifyType));
           notifyDone();
           return;
	} // End if

	// Try: -
	// /opt/bosch/checksum/check_cksum.sh -m raw -a nor -c 121212
	// As described in: -
	// https://inside-docupedia.bosch.com/confluence/pages/viewpage.action?pageId=359083129

        if (_area=="AREA_RAW_NOR") {
		commandLine = "/opt/bosch/checksum/check_cksum.sh -m raw -a nor -c 121212";
		outputFile = "/tmp/chksums_check/raw_nor.chk";
        } else if (_area=="AREA_FFS_ROOT") {
		commandLine = "/opt/bosch/checksum/check_cksum.sh -m filesystem -a root -p root -c 121212";
		outputFile = "/tmp/chksums_check/filesystem_root.chk";
        } else if (_area=="AREA_RAW_EMMC") {
           commandLine = "/opt/bosch/checksum/check_cksum.sh -m raw -a emmc_user -c 121212";
           outputFile = "/tmp/chksums_check/raw_emmc_user.chk";
        } else if (_area=="AREA_RAW_BOOT") {
           commandLine = "/opt/bosch/checksum/check_cksum.sh -m raw -a boot -c 121212";
           outputFile = "/tmp/chksums_check/raw_boot.chk";
        } else if (_area=="AREA_FFS_STATIC") {
           outputFile = "/tmp/chksums_check/filesystem_static.chk";
        } else if (_area=="AREA_V850_BOOTLOADER") {
           commandLine = "/opt/bosch/checksum/check_cksum.sh -m v850 -a bootloader -c 121212";
           outputFile = "/tmp/chksums_check/v850_bootloader.chk";
        } else if (_area=="AREA_V850_APPLICATION") {
           commandLine = "/opt/bosch/checksum/check_cksum.sh -m v850 -a application -c 121212";
           outputFile = "/tmp/chksums_check/v850_application.chk";
        } else {
           ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn() - path not found"));
           _verifyResult = FCSWUPD_NS_FCSWUPD_T::tenVerifyResult__PathNotFound;
           notifyDone();
           return;
	} 
        ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn(): cmd-line=:%s", commandLine.c_str()));

        const int32_t retval = swu::execCommand( commandLine.c_str() );
        if( !retval ) {
           bool success = parseChecksumFile( outputFile,  _checksum);
           // parseChecksumFile() will change the value of success
           if( success) {
              ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn() - check passed"));
              _verifyResult = FCSWUPD_NS_FCSWUPD_T::tenVerifyResult__OK;
           }
           else {
              ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn() - error"));
              _verifyResult = FCSWUPD_NS_FCSWUPD_T::tenVerifyResult__OtherError;
           } // End else
        }
        else {
           ETG_TRACE_COMP(("VerifyChecksumWorker::theadFn() - path not found"));
           _verifyResult = FCSWUPD_NS_FCSWUPD_T::tenVerifyResult__PathNotFound;
        } // End else
        
        notifyDone();
}


// Now send a response using the original act
tVoid SrvGeneric::vProcess(VerifyChecksumWorker::Msg_ChecksumWorkerFinished *pMsg) {
   SWU_ASSERT_RETURN(pMsg);
   SWU_ASSERT_RETURN(pMsg->worker);
   ETG_TRACE_COMP(("SrvGeneric::vProcess(Msg_ChecksumWorkerFinished) worker=%p", pMsg->worker));

  VerifyChecksumWorker *worker=(VerifyChecksumWorker *) pMsg->worker;
  if (!_checksumWorkers.erase(worker))  {
     ETG_TRACE_COMP(("SrvGeneric::vProcess(Msg_ChecksumWorkerFinished) worker=%p not registerd", pMsg->worker));
     return;
  }
  // now fetch result and act from worker and send the
  // response to the Verify Checksum message
  FCSWUPD_NS_FCSWUPD_T::trVerifyChecksumResult trResult;
  act_t act;
  worker->getAsfResult(trResult);
  FcSwUpdSrv::instance()->sendVerifyChecksumResponse( true,
						      trResult,
						      worker->getAct());

  // and delete it (clearing the thread up)
  delete worker;

}

tVoid SrvGeneric::vProcess(Msg_VerifyChecksumRequest *pMsg) {
  ETG_TRACE_COMP(("SrvGeneric::vProcess(Msg_VerifyChecksumRequest)"));

  /*
   * Put the values inside the VerifyChecksumWorker object
   */
  //  FCSWUPD_NS_FCSWUPD_T::trVerifyChecksumResult=pMsg->payload->getTrVerifyChecksumResult();
  FCSWUPD_NS_FCSWUPD_T::trVerifyChecksumRequest verfiyRequest= pMsg->payload->getVerifyRequest();
  FCSWUPD_NS_FCSWUPD_T::tenVerifyType verifyType = verfiyRequest.getEnVerifyType();
  std::string area = verfiyRequest.getArea();
  act_t act =  pMsg->payload->getAct();

  // don't check the request-type here, but pass all parameters to thread.

  // Create a thread object in which we can calculate a checksum
  VerifyChecksumWorker *worker = new VerifyChecksumWorker( act, verifyType, area );
  if (worker) {
     _checksumWorkers.insert(worker);
  }
  // Now start the thread
  worker->start();

  // The VerifyChecksumWorker() object will send a response.
  // We need do nothing more here.
}

SrvGeneric::SrvGeneric() {
   ETG_I_REGISTER_FILE();
   Msg_ProgressInformation::vSubscribe(this);
   Msg_GetCacheDirectoryRequest::vSubscribe(this);
   Msg_EnterRecoveryModeRequest::vSubscribe(this);
   Msg_StoreHistoryToStickRequest::vSubscribe(this);
   Msg_GetUpdateFileRequest::vSubscribe(this);
   Msg_DeleteHistoryRequestRequest::vSubscribe(this);
   Msg_VerifyChecksumRequest::vSubscribe(this);
   VerifyChecksumWorker::Msg_ChecksumWorkerFinished::vSubscribe(this);
}

SrvGeneric::~SrvGeneric() {
   ETG_I_UNREGISTER_FILE();
   vDeInit();
}


void SrvGeneric::vInit() {}

void SrvGeneric::vDeInit() {
   for (std::set<VerifyChecksumWorker *>::iterator iter=_checksumWorkers.begin();
        iter!=_checksumWorkers.end();
        ++iter) {
      delete *iter;
   }
   _checksumWorkers.clear();
}


void SrvGeneric::traceState() {
   ETG_TRACE_COMP(("  todo: store activated features"));
}


void SrvGeneric::vProcess(Msg_UpdateSelectReleaseByFileResult *pMsg) {
   Msg_UpdateSelectReleaseByFileResult::vUnSubscribe(this);
   if (!_postImageRelease.isRunning()) {
      return;
   }
   bool success=pMsg->success;
   if (!success) {
      _postImageRelease.done(success);
   }
}

ETG_I_CMD_DEFINE((simPostImageInstallation, "simPostImageInstallation"))
void SrvGeneric::simPostImageInstallation() {
   _postImageRelease.forceInstallation();

}

void SrvGeneric::PostImageRelease::forceInstallation() {
   Config *cfg=Config::instance();
   std::string bxmlFile=Config::getPersitentDirName() + "/" + getPostImageSubDir() + "/bosch.xml";
   if (!swu::exists(bxmlFile)) {
      ETG_TRACE_ERR(("PostImageRelease::forceInstallation(): missing file %s", bxmlFile.c_str()));
      return;
   }
   std::string bxmlFileTmp=cfg->cfg_SwuTmpDir.get() + "/bosch.xml";
   (void)swu::copyFile(bxmlFile, bxmlFileTmp);
   swu::removeDirectoryRecursive(Config::getPersitentDirName(), true);
   swu::makeDirectoryRecursive(getPostImageSubDir(), 0700);
   (void)swu::copyFile(bxmlFileTmp, bxmlFile);
   cfg->cfg_AutoHandlePostImgPending.set(true);
   ::sync();
   ::sync();

   ETG_TRACE_ERR(("PostImageRelease::forceInstallation: rebooting target to conduct post-image installation"));

   SpmIf::instance()->reboot();

}

std::string const &SrvGeneric::PostImageRelease::getPostImageBxmlDir() {
   static std::string postImgPath=Config::getPersitentDirName() + "/" + getPostImageSubDir();
   return postImgPath;
}


void SrvGeneric::PostImageRelease::done(bool success) {
   ETG_TRACE_ERRMEM(("SWUPDATE: postImage done, success=%u", success));
   Config *cfg=Config::instance();
 
   // restore configuration
   cfg->cfg_ParserType.set(_origParserType);
   cfg->cfg_CtrlCreateWithPersistentProgress.set(_origCreatePersistentProgress);
   cfg->cfg_CtrlForceRecovery.set(_origForceRecovery);
   cfg->cfg_SilentUpdate.set(_origSilentUpdate);

   if (success) {
      cfg->cfg_AutoHandlePostImgDone.set(1);
      cfg->cfg_AutoHandlePostImgPending.set(false);
   }

   Ctrl::setSendAutoInstallResponse(false);

   ReleaseFilterFileExistence filter;
   FcSwUpdCore::instance()->activateReleaseFilter(filter.getName(), true);
   cfg->cfg_ParserType.set(_origParserType);

   _running=false;
}

bool SrvGeneric::PostImageRelease::start() {
   ETG_TRACE_COMP(("SrvGeneric::PostImageRelease::start()"));

   _running=true;
   Config *cfg=Config::instance();
      
   // store current configuration
   _origParserType=cfg->cfg_ParserType.get();
   _origCreatePersistentProgress=cfg->cfg_CtrlCreateWithPersistentProgress.get();
   _origForceRecovery=cfg->cfg_CtrlForceRecovery.get();
   _origSilentUpdate=cfg->cfg_SilentUpdate.get();

   // set new configuration
   cfg->cfg_ParserType.set((tU32)swu::ParserTypeBosch);
   cfg->cfg_CtrlCreateWithPersistentProgress.set(false);
   // disable forcing recovery-mode
   cfg->cfg_CtrlForceRecovery.set(0);
   cfg->cfg_SilentUpdate.set(true);


   // deactivate release-filter for files (there is only the bosch.xml)
   ReleaseFilterFileExistence filter;
   FcSwUpdCore::instance()->activateReleaseFilter(filter.getName(), false);


   // Configure Ctrl that the reqests are not sent to SwUpdCore
   Ctrl::setSendAutoInstallResponse(true);

   // create asf-message
   const ::boost::shared_ptr< FCSWUPD_NS_FCSWUPD::UpdateSelectReleaseByFileRequest > payload(new FCSWUPD_NS_FCSWUPD::UpdateSelectReleaseByFileRequest());
   FCSWUPD_NS_FCSWUPD_T::trReleaseAccess releaseAccess;
   releaseAccess.setMountPoint(Config::getPersitentDirName());
   releaseAccess.setPath(getPostImageSubDir());
   releaseAccess.setFile("bosch.xml");
   releaseAccess.setUseExtendedOptions(true);
   FCSWUPD_NS_FCSWUPD_T::trReleaseFilter asfReleaseFilter(FCSWUPD_NS_FCSWUPD_T::tenUpdOptions__Any);
   releaseAccess.setReleaseFilter(asfReleaseFilter);
   releaseAccess.setUpdateOption(FCSWUPD_NS_FCSWUPD_T::tenUpdOptions__Any);
   releaseAccess.setDoApply(true);
   releaseAccess.setPostImageInstallationMode(true);
   payload->setReleaseAccess(releaseAccess);

   
   /* todo: if something goes wrong, we have to reset made changes above ...
      handle cfg_CtrlCreateWithPersistentProgress.get()
      vProcess(Msg_UpdateSelectReleaseByFileRequest) has to send a internal message back
      to handle failure-cases.
      Msg_UpdateSelectReleaseByFileResult

    */
   // we want to handle the asf-message async
   // use the existing wrapper-type, the is normaly used by AUTO_DISPATCH_REQ
   // create asf-message-wrapper
   Msg_UpdateSelectReleaseByFileRequest *msg= new Msg_UpdateSelectReleaseByFileRequest(payload);
   msg->bNotifyLater();
   msg=0;
   ETG_TRACE_COMP(("AutoHandlePostImageRelease END"));
   return true;
}

bool SrvGeneric::PostImageRelease::isPending() {
   Config *cfg=Config::instance();
   bool res=true;
   if (cfg->cfg_AutoHandlePostImgDone.get()) {
      res = false;
   }
   else if (!Config::instance()->cfg_EnterFullOperation.get()) {
      res = false;
   }
   else if (!Config::instance()->cfg_FcId.get()) {
      res = false;
   }
   else if (!cfg->cfg_AutoHandlePostImgPending.get()) {
      cfg->cfg_AutoHandlePostImgDone.set(true);
      res = false;
   }
   else if (!swu::exists(getPostImageBxmlDir() + "/bosch.xml")) {
      cfg->cfg_AutoHandlePostImgDone.set(true);
      res = false;
   }
      
   return res;
}



void SrvGeneric::AutoHandlePostImageRelease() {

   if (!_postImageRelease.isPending()) {
      return;
   }
   /* configurator will send this message to report the result.
      success=true will only mean, that a valid release-doc has been created.
    */
   Msg_UpdateSelectReleaseByFileResult::vSubscribe(this);

   ETG_TRACE_COMP(("SrvGeneric::AutoHandlePostImageRelease: found release"));
   if (!_postImageRelease.start()) {
      Msg_UpdateSelectReleaseByFileResult::vUnSubscribe(this);
      _postImageRelease.done(false);
      return;
   }
 
}

void SrvGeneric::setFeature(SrvGeneric::tenFeatures enFeature) {
   ETG_TRACE_COMP(("SrvGeneric::setFeature enFeature=%u", (unsigned)enFeature));
   switch (enFeature) {
      case tenFeatures_GetUpdateHistoryRequest:
         Msg_GetUpdateHistoryRequest::vSubscribe(this);
         break;
      case tenFeatures_GetTargetStateRequest:
         Msg_GetTargetStateRequest::vSubscribe(this);
         break;
#if 0
      case tenFeatures_GetTargetStateByTypeRequest:
         Msg_GetTargetStateByTypeRequest::vSubscribe(this);
         break;
#endif
      case tenFeatures_NotifyCtrlError:
         Msg_NotifyCtrlError::vSubscribe(this);
         break;
      case tenFeatures_NotifyCtrlResult:
         Msg_NotifyCtrlResult::vSubscribe(this);
         break;
      case tenFeatures_NotifyProgress:
         Msg_ProgressInformation::vSubscribe(this);
         break;
      case tenFeatures_SetAllowRecoveryMode:
         Msg_SetAllowRecoveryModeRequest::vSubscribe(this);
         break;
      case tenFeatures_GetUpdateHistoryScopedRequest:
         Msg_GetUpdateHistoryScopedRequest::vSubscribe(this);
         break;
      case tenFeatures_AutoHandlePostImageRelease:
         AutoHandlePostImageRelease();
         break;
      case tenFeatures_BgImageUpdate:
         Msg_DisplayBgImageUpdate::vSubscribe(this);
         break;
      default:
         break;
   }
}

tVoid SrvGeneric::vProcess(Msg_DisplayBgImageUpdate *pMsg) {

   FCSWUPD_NS_SWUCORE_T::trDisplayBgImage const &displayBgImage = SWU_ASF_GET_DATA(pMsg->payload, DisplayBgImage);    
   ETG_TRACE_COMP(("Ctrl::vProcess Msg_DisplayBgImageUpdate imgDirPath:%50s imgName:%50s)", 
                   displayBgImage.getImgDirPath().c_str(), displayBgImage.getImgName().c_str()));

   FCSWUPD_NS_FCSWUPD_T::trBgImageInfo rBgImageInfo = FcSwUpdSrv::instance()->getUpdateBgImageInfo();
   rBgImageInfo.setImgDirPath(displayBgImage.getImgDirPath());
   rBgImageInfo.setImgName(displayBgImage.getImgName());
   FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, rBgImageInfo, UpdateBgImageInfo);
      
}

tVoid SrvGeneric::vProcess(Msg_SetAllowRecoveryModeRequest *pMsg) {
   bool allow=pMsg->payload->getAllow();
   bool oldAllow=Config::instance()->cfg_AllowRebootDuringInstall.get();
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_SetAllowRecoveryModeRequest) allow:%u->%u", oldAllow, allow));
   if (oldAllow != allow) {
      Config::instance()->cfg_AllowRebootDuringInstall.set(allow);
      Msg_ConfigChanged msg;
      notify(msg);
      if(allow){
        Msg_NextRequest req;
        notify(req);
      }
   }
}

tVoid SrvGeneric::vProcess(Msg_EnterRecoveryModeRequest *pMsg) {
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_EnterRecoveryModeRequest)"));
   FcSwUpdSrv::instance()->sendEnterRecoveryModeResponse(true);
   sleep(3);
   SpmIf::instance()->rebootToRecoveryMode();
}


tVoid SrvGeneric::vProcess(Msg_GetUpdateHistoryRequest *pMsg) {
   (void)pMsg;
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetUpdateHistoryRequest)"));
   FCSWUPD_NS_FCSWUPD_T::trUpdLogs logs;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trUpdLog > updateLogs;
   int i = 0;
   swu::ScopedPointer<History> h(History::instance());
   h->vInit();
     
   if (Config::instance()->cfg_IncludeInitialReleaseInHistoryList.get()) {
      //production release.
      TiXmlElement* release = (TiXmlElement*)h->getInitialRelease();
      if(release) {         
         addUpdLog(updateLogs, release);
      }
   }
   for (TiXmlElement const *hr = h->getReleases()->FirstChildElement("RELEASE");
        hr; hr = hr->NextSiblingElement(), ++i)
   {
      addUpdLog(updateLogs, hr);
   }
   logs.setUpdateLogs(updateLogs);

   FcSwUpdSrv::instance()->sendGetUpdateHistoryResponse(true, logs);
}

tVoid SrvGeneric::addUpdLog(::std::vector<FCSWUPD_NS_FCSWUPD_T::trUpdLog> &updateLogs, TiXmlElement const *hr) {
   if (!hr) {
      return;
   }
   FCSWUPD_NS_FCSWUPD_T::trUpdLog *updLog = toUpdLog(hr);
   SWU_ASSERT_RETURN(updLog);
   updateLogs.push_back(*updLog);
   delete updLog;
}


tVoid SrvGeneric::vProcess(Msg_GetUpdateHistoryScopedRequest *pMsg)
{
   swu::ScopedPointer<History> h(History::instance());
   h->vInit();
   int maxNum = pMsg->payload->getNumEntriesMax();
   FCSWUPD_NS_FCSWUPD_T::tenHistoryScope scope = pMsg->payload->getEnScope();
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetUpdateHistoryScopedRequest) asfScope=%u nRequested=%u nExisting=%u", 
                   scope, maxNum, h->getNumReleases()));
   if (maxNum>h->getNumReleases()) {
      maxNum=h->getNumReleases();
   }
   int skip = pMsg->payload->getIndexStart();
   if (!skip) {
      // get releases from end of list
      if (h->getNumReleases()>maxNum) {
         skip=h->getNumReleases()-maxNum;
      }
   }
   int totalReleases = h->getNumReleases();
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetUpdateHistoryScopedRequest) max=%u", maxNum));
   FCSWUPD_NS_FCSWUPD_T::trUpdLogs logs;
   ::std::vector<FCSWUPD_NS_FCSWUPD_T::trUpdLog> updateLogs;
   TiXmlElement const *hr=0;
   switch (scope)
   {
      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__All:
      {
	     //Currently this section is not required for the Diagnosis . This is to improve SOP2 SMART.
         //addUpdLog(updateLogs, h->getInitialReleaseSinceReset());

         int i = 0;
         hr = h->getReleases()->FirstChildElement("RELEASE");
         for ( ; hr && i < skip; hr = hr->NextSiblingElement(), ++i)
         {
            // Skip 'getIndexStart' releases
         }
         for ( ; hr && i < totalReleases; hr = hr->NextSiblingElement(), ++i)
         {
            addUpdLog(updateLogs, hr);
         }
		 //If <RELEASE> section not present in history.xml then add the <INITIAL> section which is production release
		 if(!hr)
		 {
			//production release.
			TiXmlElement* release = (TiXmlElement*)h->getInitialRelease();
			if(release) {
				ETG_TRACE_USR4(("SrvGeneric::vProcess(Msg_GetUpdateHistoryScopedRequest) added production release"));
				addUpdLog(updateLogs, release);
			}
		 }
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__AllSinceFactory:
      {
         if (Config::instance()->cfg_IncludeInitialReleaseInScopedHistoryList.get()) {
            addUpdLog(updateLogs, h->getInitialReleaseSinceReset());
            if (maxNum) {
               --maxNum;
            }
         }
         int i = 0;
         hr = h->getReleases()->FirstChildElement("RELEASE");
         for ( ; hr && i < skip; hr = hr->NextSiblingElement(), ++i)
         {
            // Skip 'getIndexStart' releases
         }
         for ( ; hr && i < maxNum; hr = hr->NextSiblingElement(), ++i)
         {
            addUpdLog(updateLogs, hr);
         }

         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__FirstUsb:
      {
         addUpdLog(updateLogs, h->getFirstUsbRelease());
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__FirstOta:
      {
         addUpdLog(updateLogs, h->getFirstOtaRelease());
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__FirstUsbSinceFactory:
      {
         addUpdLog(updateLogs, h->getFirstUsbReleaseSinceReset());
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__FirstOtaSinceFactory:
      {
         addUpdLog(updateLogs, h->getFirstOtaReleaseSinceReset());
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__Initial:
      {
         addUpdLog(updateLogs, h->getInitialRelease());
         break;
      }

      case FCSWUPD_NS_FCSWUPD_T::tenHistoryScope__InitialSinceFactory:
      {
         addUpdLog(updateLogs, h->getInitialReleaseSinceReset());
         break;
      }

      default:
         ETG_TRACE_ERR(("SrvGeneric:vProcess(Msg_GetUpdateHistoryScopedRequest) Unknown scope=%u", (int)scope));
         break;
   }

   logs.setUpdateLogs(updateLogs);

   FcSwUpdSrv::instance()->sendGetUpdateHistoryScopedResponse(true, scope, logs,
                                                              h->getNumUpdates(), h->getNumUpdatesAfterReset(), h->getNumReleases());
}

tVoid SrvGeneric::vProcess(Msg_DeleteHistoryRequestRequest *pMsg)
{
   bool sinceFactoryResetOnly = pMsg->payload->getSinceFactoryResetOnly();
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_DeleteHistoryRequestRequest=%u)", sinceFactoryResetOnly));
   swu::ScopedPointer<History> h(History::instance());
   h->vInit();
   if (sinceFactoryResetOnly) {
      h->reset();
   } else {
      h->clear();
   }

   FcSwUpdSrv::instance()->sendDeleteHistoryRequestResponse(true);
}

tVoid SrvGeneric::vProcess(Msg_GetTargetStateRequest *pMsg)
{
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetTargetStateRequest)"));

   swu::ScopedPointer<History> h(History::instance());
   h->vInit();
   ETG_TRACE_USR4(("#Elements in History: %u", h->getNumReleases()));
   TiXmlElement const *aktRelease = h->getReleases()->FirstChildElement("RELEASE");
   SWU_ASSERT_RETURN(aktRelease);
   TiXmlElement const *nextRelease;
   while (nextRelease=aktRelease->NextSiblingElement("RELEASE")) {
      aktRelease=nextRelease;
   }

   // removed old emplementation, update-type currently not handled
   // todo: handle pMsg->payload->getEnUpdateType() == FCSWUPD_NS_FCSWUPD_T::tenUpdType__Customer) ???

   FCSWUPD_NS_FCSWUPD_T::trTartgetState asfTargetState;


   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trDeviceData >  asfDevices;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > asfModules;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > asfSubModules;
   trItemContext context(enVersionStyle_Current, false);
   for (XmlItemAccessIter iter(const_cast<TiXmlElement*>(aktRelease), XmlItemAccessIter::enOnLeave);!iter.isDone();iter.next()) {
      XmlItemAccess access(iter);
      
      // todo: eval scope depending on if version matches history???
      context.asfScope=FCSWUPD_NS_FCSWUPD_T::tenScope__None;
      switch (access.enGetType()) {
         case  XmlItemAccess::enType_Release: { 
            // set devices that have been filled prior (enOnLeave)
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u devices to release", asfDevices.size()));
            asfTargetState.setDevices(asfDevices);

            asfDevices.clear();
            Config *cfg=Config::instance();
            std::string relName=cfg->cfg_RunningSwVersionDisplay.get();
            asfTargetState.setName(relName);
            asfTargetState.setFilter(FCSWUPD_NS_FCSWUPD_T::tenUpdType__All);

            
            break;
         }
         case  XmlItemAccess::enType_Device: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u Modules to Device", asfModules.size()));
            asfDevices.push_back(createAsfDevice(access, asfModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numDevices=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Module: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u asfSubModules to Module", asfSubModules.size()));
            asfModules.push_back(createAsfModule(access, asfSubModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numModules=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Submodule: {
            context.fromVersion=access.getFromVersion();
            context.info=access.getVersion();
            context.toVersion=access.getVersion();
            context.info+=context.toVersion;
            asfSubModules.push_back(createAsfSubModule(access, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numSubModules=%u", asfSubModules.size()));
            break;
         }
         default:
            break;
      }
   }


   FcSwUpdSrv::instance()->sendGetTargetStateResponse(true, asfTargetState);
}

tVoid SrvGeneric::vProcess(Msg_GetCacheDirectoryRequest *pMsg)
{
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetCacheDirectoryRequest)"));
   FCSWUPD_NS_FCSWUPD_T::trGetCacheDirectory asfRequest = pMsg->payload->getRequest();

   bool bOk = false;
   ::std::string dir = "";
   int bootChain = swu::BootChain::usedBootChain();
   ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetCacheDirectoryRequest) Bootchain %d used", bootChain));

   tenSwUpdateState enSwUpdateState = CampaignManager::instance()->enGetSwUpdateState();
   if (enSwUpdateState != tenSwUpdateState_Running && !CampaignManager::instance()->isScomoMode())
   {
      const char *BLOCKDEV = "/dev/mmcblk1p";
      const ::std::string MOUNTPOINT("/tmp/swupdate/cache_directory");
      ::std::string device(BLOCKDEV);
      device += bootChain == 1 ? "2" : "1";

      FCSWUPD_NS_FCSWUPD_T::tenCacheDirRequest enAction = asfRequest.getDirRequest();
      if(enAction == FCSWUPD_NS_FCSWUPD_T::tenCacheDirRequest__ReleaseOnly) {
         if(swu::execCommand("umount " + MOUNTPOINT) == 0) {
            ETG_TRACE_COMP(("device is already mounted, it is successfully umounted"));
            bOk = true;
         } else {
            ETG_TRACE_COMP(("device is already mounted, umount fails"));
         }      
      }else {                  
      int status = 0;
         if (enAction == FCSWUPD_NS_FCSWUPD_T::tenCacheDirRequest__AcquireAndClear)
      {
         ETG_TRACE_COMP(("SrvGeneric:vProcess(Msg_GetCacheDirectoryRequest) Clear cache directory"));
         status = swu::execCommand("mkfs -t ext4 " + device);
      }

      if (0 == status && 0 == swu::execCommand("mkdir -p " + MOUNTPOINT))
      {
         if (0 == swu::execCommand("mount -t ext4 -o rw " + device + " " + MOUNTPOINT))
         {
            using namespace swu::Constants;
            bOk = true;
            dir = MOUNTPOINT;
	    //Coverity fix for 89689 and 89688
            if (-1 == (::chown(dir.c_str(), User::pj_nav, Group::pj_nav_share))){
               ETG_TRACE_USR4(("chown failed to change the owner for %s",dir.c_str()));
            }
	   
            
            if (-1 == (::chmod(dir.c_str(), 0770))){
               ETG_TRACE_USR4(("chmod failed to modify the permission for %s",dir.c_str()));
               }
            }
         }
      }
   }

   FcSwUpdSrv::instance()->sendGetCacheDirectoryResponse(bOk, dir);
}

tVoid SrvGeneric::vProcess(Msg_StoreHistoryToStickRequest *pMsg)
{
   bool sinceReset = pMsg->payload->getSinceFactoryReset();
   bool writeToAllSticks = pMsg->payload->getWriteToAllSticks();
   ETG_TRACE_COMP(("SrvGeneric::vProcess(Msg_StoreHistoryToStickRequest) sinceReset %u, writeToAllSticks %u", sinceReset, writeToAllSticks));

   swu::ScopedPointer<History> h(History::instance());
   h->vInit();

   bool bOk = false;
   FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult enResult = FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult__Ok;
   std::map< std::string, trSourceInfo > &mapSources = PropDevMgr::instance()->access()._mapSources;
   for (std::map< std::string, trSourceInfo >::iterator iter = mapSources.begin() ; iter != mapSources.end(); ++iter) {
      trSourceInfo &medium = iter->second;
      if (medium.enSourceType == tenSourceType_USB || medium.enSourceType == tenSourceType_SD) {
         trSourceInfo &info = iter->second;
         std::string realPath = swu::realPath(info.path);
         std::string cmd = "mount -o rw,remount " + realPath;
         std::string fullPath(realPath + "/UpdateHistory.xml");
         if (!swu::execCommand(cmd.c_str()) && h->store(fullPath.c_str(), sinceReset)) {
            ETG_TRACE_COMP(("Msg_StoreHistoryToStickRequest: saved to %s", fullPath.c_str()));
            bOk = true;
         } else {
            ETG_TRACE_ERR(("Msg_StoreHistoryToStickRequest: write to %s failed", fullPath.c_str()));
            enResult = FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult__WriteFailedAtLeastOnce;
         }
      }
      if (bOk && !writeToAllSticks) {
         break;
      }
   }

   if (bOk) {
      (void)swu::execCommand("sync");
   } else if (enResult == FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult__WriteFailedAtLeastOnce) {
      ETG_TRACE_ERR(("Msg_StoreHistoryToStickRequest: All writes failed"));
      enResult = FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult__WriteFailed;
   } else {
      ETG_TRACE_ERR(("Msg_StoreHistoryToStickRequest: No removable media found"));
      enResult = FCSWUPD_NS_FCSWUPD_T::tenStoreHistoryResult__MediumNotFound;
   }

   FcSwUpdSrv::instance()->sendStoreHistoryToStickResponse(true, enResult);
}


FCSWUPD_NS_FCSWUPD_T::trDeviceData SrvGeneric::createAsfDevice(XmlItemAccess &access, ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > &asfModules, trItemContext &context) {

   FCSWUPD_NS_FCSWUPD_T::trDeviceData asfDevice;
   asfDevice.setName(access.getName());
   asfDevice.setModules(asfModules);
   asfModules.clear();
   asfDevice.setEnScope(context.asfScope);
   ETG_TRACE_USR4(("SrvGeneric::createAsfDevice(%s)", access.getName()));
   return asfDevice;
}

FCSWUPD_NS_FCSWUPD_T::trModuleData SrvGeneric::createAsfModule(XmlItemAccess &access, ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > &asfSubModules, trItemContext &context) {

   FCSWUPD_NS_FCSWUPD_T::trModuleData asfModule;
   asfModule.setName(access.getName());
   asfModule.setHwIndex(Config::instance()->cfg_HwId.get());

   asfModule.setSubModules(asfSubModules);
   asfSubModules.clear();
   asfModule.setEnScope(context.asfScope);
   ETG_TRACE_USR4(("SrvGeneric::createAsfModule(%s)", access.getName()));
   return asfModule;
}

FCSWUPD_NS_FCSWUPD_T::trSubModuleData SrvGeneric::createAsfSubModule(XmlItemAccess &access, trItemContext &context) {
   FCSWUPD_NS_FCSWUPD_T::trSubModuleData asfSubModule;
   asfSubModule.setName(access.getName());
   asfSubModule.setFromVersion(context.fromVersion);
   asfSubModule.setToVersion(context.toVersion);
   asfSubModule.setInfo(context.info);
   asfSubModule.setBSelected(context.asfScope!=FCSWUPD_NS_FCSWUPD_T::tenScope__No);
   asfSubModule.setEnScope(context.asfScope);
   ETG_TRACE_USR4(("SrvGeneric::createAsfSubModule(%s)", access.getName()));
   return asfSubModule;
}

FCSWUPD_NS_FCSWUPD_T::trTimeDate SrvGeneric::getAsfTimeData(TiXmlElement const *utcXml) {
   FCSWUPD_NS_FCSWUPD_T::trTimeDate asfTimeDate;
   swu::LocalTime lc(utcXml);
   asfTimeDate.setYear(static_cast<uint16> (lc.getYear() ));
   asfTimeDate.setMonth(static_cast<unsigned char> (lc.getMonth() ));
   asfTimeDate.setDay(static_cast<unsigned char> (lc.getDay() ));
   asfTimeDate.setHour(static_cast<unsigned char> (lc.getHour() ));
   asfTimeDate.setMinute(static_cast<unsigned char> (lc.getMinute() ));
   asfTimeDate.setUtc(lc.getUTC());
   return asfTimeDate;
}


void  SrvGeneric::vProcess(Msg_NotifyCtrlResult *pMsg) {

   Ctrl *pCtrlInst = CampaignManager::instance()->getCtrlInstance();
   SWU_ASSERT_RETURN(pCtrlInst);

   CtrlProgressSection *prgSection=const_cast<CtrlProgressSection *>(pMsg->_progressSection);

   if (_postImageRelease.isRunning() ) {
      ETG_TRACE_FATAL(("SrvGeneric::Msg_NotifyCtrlResult: added post-image release"));
      Msg_UpdateSelectReleaseByFileResult::vUnSubscribe(this);
      _postImageRelease.done(pMsg->_result);      
      pCtrlInst->vUpdateEndRequest();      
      return;
   } else if (prgSection->isScomoModeNormal() || prgSection->isCompatUpdate()) {
      //don't inform anything to HMI in case of oem updaters.
      return;
   }

   TiXmlElement *progXml=const_cast<TiXmlElement *>(prgSection->getProgressReleaseSection());

   FCSWUPD_NS_FCSWUPD_T::trResult asfResult;

   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trDeviceData >  asfDevices;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > asfModules;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > asfSubModules;

   trItemContext context(enVersionStyle_FromTo, true);
   for (XmlItemAccessIter iter(progXml, XmlItemAccessIter::enOnLeave);!iter.isDone();iter.next()) {
      XmlItemAccess access(iter);
      context.asfScope=(FCSWUPD_NS_FCSWUPD_T::tenScope)swu::getUIntFromChild(access.getXml(), "SCOPE");
      switch (access.enGetType()) {
         case  XmlItemAccess::enType_Release: { 
            // set devices that have been filled prior (enOnLeave)
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u devices to release", asfDevices.size()));
            asfResult.setDevices(asfDevices);

            asfDevices.clear();
            asfResult.setName(access.getName());
            asfResult.setBOk(pMsg->_result);
            
            // set utc
            asfResult.setTimeDate(getAsfTimeData(access.getXml()));
            break;
         }
         case  XmlItemAccess::enType_Device: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u Modules to Device", asfModules.size()));
            asfDevices.push_back(createAsfDevice(access, asfModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numDevices=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Module: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u asfSubModules to Module", asfSubModules.size()));
            asfModules.push_back(createAsfModule(access, asfSubModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numModules=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Submodule: {
            context.fromVersion=access.getFromVersion();
            context.info=context.fromVersion;
            if (!context.info.empty()) {
               context.info += " -> ";
            }
            context.toVersion=access.getVersion();
            context.info+=context.toVersion;
            asfSubModules.push_back(createAsfSubModule(access, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numSubModules=%u", asfSubModules.size()));
            break;
         }
         default:
            break;
      }
   }

   FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, asfResult, UpdateResult);
}



void  SrvGeneric::vProcess(Msg_ProgressInformation *pMsg) {
   ProgressInformation const &progress=pMsg->progress;
   ETG_TRACE_USR1(("SrvGeneric::vProcess(Msg_ProgressInformation)"));
   ETG_TRACE_USR1(("progress.Train = %s", progress.train.c_str()));
   ETG_TRACE_USR1(("progress.DeviceName = %s", progress.deviceName.c_str()));
   ETG_TRACE_USR1(("progress.ModuleName = %s", progress.moduleName.c_str()));
   ETG_TRACE_USR1(("progress.SubModuleName = %s", progress.subModuleName.c_str()));
   ETG_TRACE_USR1(("progress.RefKey = %s", progress.refKey.c_str()));
   ETG_TRACE_USR1(("progress.Source = %s", progress.source.c_str()));
   ETG_TRACE_USR1(("progress.line1 = %s", progress.line1.c_str()));
   ETG_TRACE_USR1(("progress.line2 = %s", progress.line2.c_str()));
   ETG_TRACE_USR1(("progress.luaCmd = %s", progress.luaCmd.c_str()));
   ETG_TRACE_USR1(("progress.U32NumAll = %u", progress.numAll));
   ETG_TRACE_USR1(("progress.U32NumComplete = %u", progress.numComplete));
   ETG_TRACE_USR1(("progress.U32NumRunning = %u", progress.numRunning));
   ETG_TRACE_USR1(("progress.U32NumFailed = %u", progress.numFailed));
   ETG_TRACE_USR1(("progress.U32NumNotApplicable = %u", progress.numNotApplicable));
   ETG_TRACE_USR1(("progress.U32NumRemaining = %u", progress.numRemaining));
   ETG_TRACE_USR1(("progress.U32Retries = %u", progress.retries));
   ETG_TRACE_USR1(("progress.U8SubModulePercentComplete = %u", progress.percent));
   ETG_TRACE_USR1(("progress.U8ReleasePercentComplete = %u", progress.percentRelease));
   ETG_TRACE_USR1(("progress.u8ReleasePercentCompleteForPhase = %u", progress.percentReleaseForPhase));
   ETG_TRACE_USR1(("progress.U32EstimatedUpdateTime = %u", progress.estimatedUpdateTime));
   ETG_TRACE_USR1(("progress.u32EstimatedUpdateTimeSecForPhase = %u", progress.estimatedUpdateTimeForPhase));


   FCSWUPD_NS_FCSWUPD_T::trUpdProgress rProgress = FcSwUpdSrv::instance()->getUpdateProgress();
   rProgress.setTrain(progress.train);
   rProgress.setDeviceName(progress.deviceName);
   rProgress.setModuleName(progress.moduleName);
   rProgress.setSubModuleName(progress.subModuleName);
   rProgress.setRefKey(progress.refKey);
   rProgress.setSource(progress.source);
   rProgress.setLine1(progress.line1);
   rProgress.setLine2(progress.line2);
   rProgress.setLuaCmd(progress.luaCmd);
   rProgress.setU32NumAll(progress.numAll);
   rProgress.setU32NumComplete(progress.numComplete);
   rProgress.setU32NumRunning(progress.numRunning);
   rProgress.setU32NumFailed(progress.numFailed);
   rProgress.setU32NumNotApplicable(progress.numNotApplicable);
   rProgress.setU32NumRemaining(progress.numRemaining);
   rProgress.setU32Retries(progress.retries);
   rProgress.setU8SubModulePercentComplete(progress.percent);
   rProgress.setU8ReleasePercentComplete(progress.percentRelease);
   rProgress.setU8ReleasePercentCompleteForPhase(progress.percentReleaseForPhase);
   rProgress.setU32EstimatedUpdateTimeSec(progress.estimatedUpdateTime);
   rProgress.setU32EstimatedUpdateTimeSecForPhase(progress.estimatedUpdateTimeForPhase);
   if(Config::instance()->cfg_UseEstimatedTimePerPhaseOnly.readAsBool()){
     rProgress.setU8ReleasePercentComplete(progress.percentReleaseForPhase);
     rProgress.setU32EstimatedUpdateTimeSec(progress.estimatedUpdateTimeForPhase);
   }
   rProgress.setU32EstimatedCurrentModuleUpdateTimeSec(progress.estimatedCurrentModuleUpdateTime);

   FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, rProgress, UpdateProgress);

}

tVoid SrvGeneric::vProcess (Msg_GetUpdateFileRequest *pMsg) {
   ETG_TRACE_COMP(( "SrvGeneric::vProcess(Msg_GetUpdateFileRequest)"));
   FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__Ok;

   
   do {  // if something goes wrong, call break
      std::map< std::string, trSourceInfo > &mapSources = PropDevMgr::instance()->access()._mapSources;
      std::map< std::string, trSourceInfo >::iterator iter = mapSources.begin();
      for ( ; iter != mapSources.end(); ++iter) {
         trSourceInfo &medium = iter->second;
         if (medium.enSourceType == tenSourceType_USB || medium.enSourceType == tenSourceType_SD) {
            break;
         }
      }

      if (iter == mapSources.end()) {
         ETG_TRACE_ERR(("  No removable media found"));
         asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__MediumNotFound;
         break;
      }

      trSourceInfo &info = iter->second;
      std::string realPath = swu::realPath(info.path);

      // remount the usb to r/w
      if (swu::execCommand(("mount -o remount,rw " + realPath).c_str())) {
         ETG_TRACE_ERR(("  Could not mount rw %s", realPath.c_str()));
         asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__WriteFailed;
         break;
      }

      std::string fullPath = realPath + "/SrvUpdateFile.xml";
      ETG_TRACE_COMP(( "SrvGeneric::Storage Path %s", fullPath.c_str()));

      TiXmlDocument *doc = new TiXmlDocument();
      swu::ScopedPointer<TiXmlDocument> scoped_doc(doc);
      doc->LinkEndChild(new TiXmlDeclaration("1.0", "", ""));
      TiXmlElement *root = doc->LinkEndChild(new TiXmlElement("HUIDS"))->ToElement();

      FCSWUPD_NS_FCSWUPD_T::trUpdState rState = FcSwUpdSrv::instance()->getUpdateState();
      swu::setTextChild(root, "NAVIUNITID", rState.getHeadUnitId().c_str());
      swu::setTextChild(root, "VERSION", rState.getMuVersion().c_str());	  

      tU16 fcid;
      // get fcid, and store in hex
      SystemData::instance()->getFCID(fcid);
      char hexstr[5];
      sprintf(hexstr, "%.4X", fcid);
      swu::setTextChild(root, "FCID", hexstr);

      // get train info
      std::string train = "";
      train = Config::instance()->cfg_RunningSwVersionDisplay.get();
      swu::setTextChild(root, "RELEASE", train.c_str());

      // convert tinyxml-doc to string
      TiXmlPrinter printer;
      printer.SetIndent( "    " );
      doc->Accept( &printer );
      std::string xmltext = printer.CStr();
      swu::ScopedFd fdOut(fullPath);
      if (!fdOut.doOpen(O_WRONLY | O_TRUNC | O_CREAT)) {
         ETG_TRACE_ERR(("  Could not open file for writing"));
         asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__WriteFailed;
         break;
      }
      
      if (static_cast<ssize_t> (xmltext.size()) != write(fdOut.getFd(), (void *)xmltext.c_str(), xmltext.size())) {  // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
         int err=errno;
         ETG_TRACE_ERR(("  Could not write xml (errno=%d)", errno))
            if (errno=ENOSPC) {
               ETG_TRACE_ERR(("  no space left on device"));
               asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__NoSpaceLeftOnDevice;
            }
            else {
               ETG_TRACE_ERR(("  Could not writing xml"));
               asfErr=FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__WriteFailed;
            }
         break;
      }
      fdOut.doClose();
      (void)swu::execCommand("sync");

   } while (0);
   ETG_TRACE_USR4(("SrvGeneric::saved file with asf-code %u", (int)asfErr));

   FcSwUpdSrv::instance()->sendGetUpdateFileResponse(asfErr==FCSWUPD_NS_FCSWUPD_T::tenGetUpdateFileResult__Ok, asfErr);

}



FCSWUPD_NS_FCSWUPD_T::tenErrorId SrvGeneric::convCoreErrorCodeToPrj(uint32_t errorCodeInt) {
   enum FCSWUPD_NS_SWUCORE_T::tenInstallResult errorCode = (enum FCSWUPD_NS_SWUCORE_T::tenInstallResult) errorCodeInt;
   switch (errorCode) {

      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ABORD_ON_REQUEST:
         // this should not end in an error to the HMI...
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_UNKNOWN;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_BUSY:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_DOWNLOAD_MRG_BUSY;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_DATA_CKSUM:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_CHECKSUM;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_DATA_NOT_FOUND:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_READ;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_TOOLS_NOT_FOUND:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_MEDIA_UNAVAILABLE;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_DEVICE_ERROR:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_DEVICE_NOT_READY;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_DEVICE_NOT_READY:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_DEVICE_NOT_READY;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_DEVICE_NOT_SUPPORTED:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_DEVICE_NOT_SUPPORTED;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_IMAGE_FLASHING:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_FLASHING;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_IMAGE_INCOMPATIBLE_VERSION:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_INCOMPATIBLE_VERSION;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_IMAGE_INVALID_SECTOR:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_INVALID_SECTORHEADER;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_IMAGE_READ:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_READ;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_IMAGE_SIZE:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_SIZE;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_NOT_ENOUGH_MEMORY:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_NOT_ENOUGH_MEMORY;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_READ:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_ERG_READ;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_TOOLS_CKSUM:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_CHECKSUM;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_TOOLS_READ:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_READ;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_TOOLS_SIZE:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_IMAGE_SIZE;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_MEDIUM_REMOVED:
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_MEDIUM_REMOVED;
         //return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_MEDIA_UNAVAILABLE; csh2kor
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_UNKNOWN:
         // Here the download core should be extended
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_UNKNOWN;
      case FCSWUPD_NS_SWUCORE_T::tenInstallResult__OK:
         // Here the download core should be extended
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_OK;
      default:
         ETG_TRACE_ERR(("Unknown Error Type from Core: %u", errorCode));
         return FCSWUPD_NS_FCSWUPD_T::tenErrorId__SWL_ERROR_UNKNOWN;

   }
}


tVoid SrvGeneric::vProcess(Msg_NotifyCtrlError *pMsg) {
   ETG_TRACE_USR2(("SrvGeneric:Msg_NotifyCtrlError"));

   std::list< uint32_t > const &errors=pMsg->_errors;
   FCSWUPD_NS_FCSWUPD_T::trErrorIds errorProperty = FcSwUpdSrv::instance()->getUpdateErrors();
   std::vector<FCSWUPD_NS_FCSWUPD_T::tenErrorId> convErrors(errors.size());
   size_t i = 0;
   for (std::list< uint32_t >::const_iterator it = errors.begin(); it != errors.end(); ++it) {
      convErrors[i++] = convCoreErrorCodeToPrj(*it);
   }
   errorProperty.setErrorIds(convErrors);
   FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, errorProperty, UpdateErrors);
}

 



FCSWUPD_NS_FCSWUPD_T::trUpdLog *SrvGeneric::toUpdLog(const TiXmlElement *hr) const
{
   ETG_TRACE_USR4(("SrvGeneric::toUpdLog start"));
   SWU_ASSERT_RETURN_VAL(hr, NULL);
   SWU_ASSERT_RETURN_VAL(std::string(hr->Value()) == "RELEASE", NULL);
   swu::ScopedPointer<FCSWUPD_NS_FCSWUPD_T::trUpdLog> log(new FCSWUPD_NS_FCSWUPD_T::trUpdLog());

   
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trDeviceData >  asfDevices;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > asfModules;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > asfSubModules;

   trItemContext context(enVersionStyle_FromTo, true);
   for (XmlItemAccessIter iter(const_cast<TiXmlElement *>(hr), XmlItemAccessIter::enOnLeave);!iter.isDone();iter.next()) {
      XmlItemAccess access(iter);
      ETG_TRACE_USR4(("SrvGeneric::toUpdLog iter start type=%u", access.enGetType()));
      TiXmlElement *xmlItem=access.getXml();
      // todo: fix scope (add level to context an evaluate it there
      context.asfScope=(FCSWUPD_NS_FCSWUPD_T::tenScope)swu::getUIntFromChild(xmlItem, "SCOPE");
      switch (access.enGetType()) {
         case  XmlItemAccess::enType_Release: { 
            std::string relName=swu::getTextFromChild(hr, "TOVERSION_CUSTOMER", false, "unknown");

            if (relName =="unknown") {
               if (Config::instance()->cfg_UseReleaseNameFromRunningSwLabel.get()) {
                  relName=swu::getTextFromChild(hr, "TOVERSION_LABEL", false, "unknown");
               }
               else {
                  relName=swu::getTextFromChild(hr, "TOVERSION", false, "unknown");
               }
            }
            ETG_TRACE_USR4(("SrvGeneric::toUpdLog RELEASE=%s", relName.c_str()));
            log->setName(relName);
            FCSWUPD_NS_FCSWUPD_T::trTimeDate date;
            swu::LocalTime utc;
            utc.readXml(hr->FirstChildElement("UTC"));
            date.setYear(static_cast<uint16> (utc.getYear() ));
            date.setMonth(static_cast<unsigned char> (utc.getMonth() ));
            date.setDay(static_cast<unsigned char> (utc.getDay() ));
            date.setHour(static_cast<unsigned char> (utc.getHour() ));
            date.setMinute(static_cast<unsigned char> (utc.getMinute() ));
            date.setSeconds(utc.getSecond());
            ETG_TRACE_USR4(("SrvGeneric::toUpdLog RELEASE timestamp: %u:%u:%u %u-%u-%u", 
                            utc.getYear(), utc.getMonth(), utc.getDay(), utc.getHour(), utc.getMinute(), utc.getSecond()));
            log->setIndex(swu::getUIntFromChild(hr, "INDEX"));
            log->setTimeDate(date);
            log->setUpdateType(swu::getTextFromChildOrEmpty(hr, "TYPE"));
            log->setSuccess(swu::getTextFromChildOrEmpty(hr, "COMPLETED"));
            log->setSize(swu::getUIntFromChild(hr, "DNL_SIZE"));
            log->setSuccess((swu::getUIntFromChild(hr, "SUCCESS", 1)));
            tU32 counterKilometers=swu::getUIntFromChild(hr, "TOTAL_DISTANCE");
            // todo:make this handling configurable via Config
            if (counterKilometers== 0xFFFFFFFF || counterKilometers== 0x7FFFFFFF) {
               counterKilometers=0;
            }
            log->setCounterKilometers(counterKilometers);
            ETG_TRACE_USR4(("SrvGeneric::toUpdLog counterKilometers=%u", counterKilometers));

            // set devices that have been filled prior (enOnLeave)
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u devices to release", asfDevices.size()));
            log->setDevices(asfDevices);

            asfDevices.clear();
            break;
         }
         case  XmlItemAccess::enType_Device: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u Modules to Device", asfModules.size()));
            asfDevices.push_back(createAsfDevice(access, asfModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numDevices=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Module: {
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: add %u asfSubModules to Module", asfSubModules.size()));
            asfModules.push_back(createAsfModule(access, asfSubModules, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numModules=%u", asfModules.size()));
            break;
         }
         case  XmlItemAccess::enType_Submodule: {
            context.fromVersion=access.getFromVersion();
            context.info=context.fromVersion;
            if (!context.info.empty()) {
               context.info += " -> ";
            }
            context.toVersion=access.getVersion();
            context.info+=context.toVersion;
            asfSubModules.push_back(createAsfSubModule(access, context));
            ETG_TRACE_USR4(("Msg_NotifyCtrlResult: numSubModules=%u", asfSubModules.size()));
            break;
         }
         default:
            break;
      }
   }
   return log.release();
}






std::string SrvGeneric::getVersionString(const TiXmlElement *hs, trHistoryContext const &context) const
{
   std::string res;

   switch (context.enVersionStyle)
   {
      case enVersionStyle_Current:
      {
         std::string longName = context.curDevice + "/" + context.curModule + "/" + swu::getTextFromChildOrEmpty(hs, "TYPE");
         res = SystemData::instance()->getCurrentModVersion(longName);
         if (res.empty()) {
            res = "unknown";
         }
         ETG_TRACE_USR4(("SrvGeneric::getVersionString: longName=%30s", longName.c_str()));
         break;
      }
      case enVersionStyle_FromTo:
      {
         res=swu::getTextFromChild(hs, "FROMVERSION", false, "");
         if (res.size()) {
            res += " -> ";
         }
         res=swu::getTextFromChild(hs, "VERSION", false, "unknown");
         break;
      }
      case enVersionStyle_To:
      default:
      {
         res = swu::getTextFromChild(hs, "VERSION", false, "unknown");
         break;
      }
   }

   ETG_TRACE_USR4(("SrvGeneric::getVersionString(mode=%u): versionString=%s",
                   ETG_CENUM(tenVersionStyle, context.enVersionStyle), res.c_str()));
   return res;
}


}  // namespace fcswupdate
