#ifndef FCSWUPD_RELEASE_FILTER_IF_H
#define FCSWUPD_RELEASE_FILTER_IF_H

#include <string.h>
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_bxml.h"

#define RELEASE_FILTER_DEVICE_PATTERN "_X"
namespace fcswupdate {

class ReleaseFilterIf;

struct ReleaseFilterContext {
   trSourceInfo sourceInfo;
};

class ReleaseFilterChain {
public:
   ~ReleaseFilterChain();
   void add(ReleaseFilterIf *filter, bool first = false);
   void remove(std::string filterName);
   void activate(std::string filterName, bool doActivate);
   tenSwUpdateError apply(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);


private:
   std::map <std::string, bool> _activations;
   std::list < ReleaseFilterIf * > _filters;
};

class ReleaseFilterIf {
public:
   virtual ~ReleaseFilterIf() {};
   virtual std::string getName()=0;
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);
   virtual tenSwUpdateError filterElem(TiXmlElement * /*root*/) { return tenSwUpdateError_OK; };
};

class ReleaseFilterDefault:public ReleaseFilterIf {
public:
   virtual std::string getName() {
      return "ReleaseFilterDefault";
   };
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);
};


// Filter removes ending of xml-items, e.g. rename RELEASE_X to RELEASE.
// with this method, items can be hidden from old updaters
class ReleaseFilterRename:public ReleaseFilterIf {
public:
   ReleaseFilterRename(std::string ending):
      _ending(ending) 
   {}
  virtual std::string getName() {
	return "ReleaseFilterRename";
   };
protected:
  virtual tenSwUpdateError filterElem(TiXmlElement *root);

private:
std::string _ending;
};

class ReleaseFilterCompat:public ReleaseFilterIf {
public:
   ReleaseFilterCompat();
  virtual std::string getName() {
	return "ReleaseFilterCompat";
   };
protected:
  virtual tenSwUpdateError filterElem(TiXmlElement *root);
private:
  bool checkCompat(TiXmlElement *elem);
   tU32 _installedCompatInd;
};

class ReleaseFilterDnlSize:public ReleaseFilterIf {
public:
   ReleaseFilterDnlSize() {};
  virtual std::string getName() {
	return "ReleaseFilterDnlSize";
   };
protected:
  virtual tenSwUpdateError filterElem(TiXmlElement *root);
};

class SxmTypeFilter:public ReleaseFilterIf  {
public:
   SxmTypeFilter() {};
   virtual std::string getName() {
     return "SxmTypeFilter";
   };
protected:
    virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

};


class BxmlFilterMatch:public ReleaseFilterIf {
public:
   virtual std::string getName() {
      return _name;
   }

   BxmlFilterMatch(std::string key, std::string val):
      _key(key),
      _val(val)
   {
      _name="BxmlFilterMatch_"+_key + "_"+_val;
   };


protected:
  virtual tenSwUpdateError filterElem(TiXmlElement *elem);

private:
   bool matches(TiXmlElement *elem);
   std::string _key;
   std::string _val;
   std::string _name;

};

class BxmlFilterDefaults:public ReleaseFilterIf {
public:
   virtual std::string getName() {
      return "DEFAULTS";
   }


protected:
  virtual tenSwUpdateError filterElem(TiXmlElement *elem);

private:
   bool isDefault(TiXmlElement *elem);
};


class BXmlMoveElemFromParentToChild:public ReleaseFilterIf {
public:
  virtual std::string getName() {
      return _name;
   }

  BXmlMoveElemFromParentToChild(std::string tagName, XmlItemAccess::tenType parentType):
    _tagName(tagName),
    _parentType(parentType)
  {
    _name = "BXmlMoveElemFromParentToChild_"+_tagName;
  }

protected:
  virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

private:
  std::string _name;
  std::string _tagName;
  XmlItemAccess::tenType _parentType;
};

class BxmlFilterBaseNodes: public ReleaseFilterIf {
   virtual std::string getName() {
	return "BxmlFilterBaseNodes";
   };  


   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);
  private:
   tVoid changeRootBxmlToUXml(TiXmlDocument *boschXmlDoc);


};


class ReleaseFilterFileExistence:public ReleaseFilterIf {
public:
  virtual std::string getName() {
	return "ReleaseFilterFileExistence";
   };

   ReleaseFilterFileExistence() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

private:
   tenSwUpdateError checkFileItem(TiXmlElement *fileItem, std::string const &releasePath);

};

class ReleaseFilterMandatoryNodes:public ReleaseFilterIf {
public:
  virtual std::string getName() {
	return "ReleaseFilterMandatoryNodes";
   };

   ReleaseFilterMandatoryNodes() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

};


class ReleaseFilterCheckFcids:public ReleaseFilterIf {
public:
  virtual std::string getName() {
	return "ReleaseFilterCheckFcids";
   };

   ReleaseFilterCheckFcids() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

};


class ReleaseFilterCheckPrjId:public ReleaseFilterIf
{
public:
  virtual std::string getName()
  {
    return "ReleaseFilterCheckPrjID";
   };

   ReleaseFilterCheckPrjId() {};

protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);
};



class ReleaseFilterConvertScomoIndex:public ReleaseFilterIf {
public:
  virtual std::string getName() {
	return "ReleaseFilterConvertScomoIndex";
   };

   ReleaseFilterConvertScomoIndex() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *indexXmlDoc, ReleaseFilterContext const &context);
};


class BxmlFilterExtraArgs : public ReleaseFilterIf
{
public:
  virtual std::string getName() { return "BxmlFilterExtraArgs"; };

  BxmlFilterExtraArgs() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);
   virtual tenSwUpdateError filterElem(TiXmlElement *root);

};


class ReleaseFilterCheckSRK:public ReleaseFilterIf {
public:
  virtual std::string getName() {
	return "ReleaseFilterSRK";
   };

   ReleaseFilterCheckSRK() {};


protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

private:
   tenSwUpdateError getProcessedError(tenSwUpdateError error);

};

class UpdateXmlFilter : public ReleaseFilterIf {

public:
  virtual std::string getName() { return "UpdateXmlFilter"; };

   UpdateXmlFilter():
      _refModule(0) 
   {};

protected:
   virtual tenSwUpdateError doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context);

private:
   tVoid convertBoschToUpdateXml(TiXmlDocument* boschdoc);
   tVoid manipulateInReleaseTag(TiXmlDocument* boschdoc);
   tVoid manipulateInOverallTag(TiXmlDocument* boschdoc);
   tVoid mandatoryTagInOverall(TiXmlDocument* boschdoc);

   TiXmlElement *_refModule;

};


}
#endif
