#include <climits>
#include <limits>

#include "util/swu_util.hpp"
#include "util/swu_xmlDoc.h"
#include "tinyxml/tinyxml.h"
#include "util/swu_filesystem.h"
#include "config/fcswupd_config.hpp"
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_bxml.h"
#include "main/fcswupd_propDevMgr.h"
#include "main/fcswupd_releaseFilterIf.h"
#include "main/fcswupd_systemData.h"
#include "util/fcswupd_trace.hpp"
#include "main/fcswupd_campaignmanager.h"
#include "util/swu_globallog.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_releaseFilterIf.cpp.trc.h"
#endif

namespace fcswupdate {

static const char *_scomoTemplateXmlFile="/opt/bosch/swupdate/fcswupdate/scomoBxmlTemplate.xml";

ReleaseFilterChain::~ReleaseFilterChain() {
   for (std::list < ReleaseFilterIf * >::iterator iter = _filters.begin();
        iter != _filters.end(); ++iter) {
      ReleaseFilterIf *filter = *iter;
      delete filter;
   }
   _filters.clear();
}

void ReleaseFilterChain::activate(std::string filterName, bool doActivate) {
   ETG_TRACE_USR2(("ReleaseFilterChain():activate(%u) filter:%s", doActivate, filterName.c_str()));
   _activations[filterName]=doActivate;
}


void ReleaseFilterChain::add(ReleaseFilterIf *filter, bool first) {
   //ETG_TRACE_USR2(("ReleaseFilterChain():add() filter:%s", filter->getName().c_str())); 

   remove(filter->getName());
   if (first) {
      _filters.push_front(filter);
   } else {
      _filters.push_back(filter);
   }
   _activations[filter->getName()]=true;

}

tenSwUpdateError ReleaseFilterChain::apply(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   tenSwUpdateError res=tenSwUpdateError_OK; 
   for (std::list < ReleaseFilterIf * >::iterator iter = _filters.begin();
        iter != _filters.end(); ++iter) {
      ReleaseFilterIf *filter = *iter;
      ETG_TRACE_USR2(("ReleaseFilterChain():apply() filter:%s START", filter->getName().c_str()));
      if (!_activations[filter->getName()]) {
         ETG_TRACE_USR2(("ReleaseFilterChain():apply() skip deactivated filter"));
         continue;
      }

      
      res = filter->doFilter(boschXmlDoc, context);
      if (res != tenSwUpdateError_OK) {
         ETG_TRACE_USR1(("ReleaseFilterChain():apply() filter(%s) failed", 
                         filter->getName().c_str()));
         swu::LOG_FAILURE("ReleaseFilterChain():apply() filter(%s) failed with error: (%u)", filter->getName().c_str(), res);
         return res;
      }
   }
   return tenSwUpdateError_OK;
}


void ReleaseFilterChain::remove(std::string filterName) {
   //ETG_TRACE_USR2(("ReleaseFilterChain():remove() filter:%s START", filterName.c_str()));

   for (std::list < ReleaseFilterIf * >::iterator iter = _filters.begin();
        iter != _filters.end(); ++iter) {
      if ((*iter)->getName() == filterName) {
         ETG_TRACE_USR2(("ReleaseFilterChain():remove() filter:%s found!", filterName.c_str()));
         _activations[(*iter)->getName()]=true;
         _filters.erase(iter);   
         return;
      }
   }
}

tenSwUpdateError ReleaseFilterIf::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   tenSwUpdateError enError =tenSwUpdateError_ERROR_UNKNOWN ;

   SWU_ASSERT_RETURN_VAL( boschXmlDoc, enError);
   TiXmlElement *root = boschXmlDoc->FirstChildElement();

   enError=filterElem(root);
   return enError;  
}


static bool combineBooleanXml(TiXmlElement *elem, char const* boolElemName, bool combineVal) {
   bool res=combineVal;
   char const *boolStr=swu::getTextFromChildOrEmpty(elem, boolElemName);
   if (boolStr) {
      if (swu::streq(boolStr, "true") || swu::streq(boolStr, "TRUE")) {
         res=true;
      }
      else if (swu::streq(boolStr, "false") || swu::streq(boolStr, "FALSE")) {
         res=false;
      }
   }
   return res;
}

static void setIgnoreCkSum(TiXmlElement *ckSumsRoot) {
   for (TiXmlElement *ckSum=ckSumsRoot->FirstChildElement("CKSUM"); 
        ckSum;
        ckSum=ckSum->NextSiblingElement("CKSUM")) {
      swu::setTextChild( ckSum , "TYPE", "NONE" );
   }
}


static bool combineSkipCkSum(TiXmlElement *elem, bool combineVal) {
   return combineBooleanXml(elem, "SKIPCHECKSUM", combineVal);
}

tenSwUpdateError ReleaseFilterDefault::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   
   tenSwUpdateError enError =tenSwUpdateError_ERROR_UNKNOWN ;


   SWU_ASSERT_RETURN_VAL( boschXmlDoc, enError);
   TiXmlElement *root = boschXmlDoc->FirstChildElement();
   SWU_ASSERT_RETURN_VAL( root, enError);
   TiXmlElement *release = root->FirstChildElement("RELEASE");
   SWU_ASSERT_RETURN_VAL( release, enError);
   TiXmlElement *overall = root->FirstChildElement("OVERALL");
   SWU_ASSERT_RETURN_VAL( overall, enError);
   
   
   bool skipCheckSumOverall=combineSkipCkSum(overall, false);
   ETG_TRACE_USR1(("boschXmlPostProcess:  skipCheckSumOverall=%u", skipCheckSumOverall));
   TiXmlElement * pCustSection = overall->FirstChildElement( "CUSTUPDATE_SECTION" );
   if (pCustSection && skipCheckSumOverall) {
      ETG_TRACE_USR1(("boschXmlPostProcess: setIgnoreCkSum(overall)"));
      setIgnoreCkSum(pCustSection);
   }
   skipCheckSumOverall=combineSkipCkSum(release, false);
   bool skipDevCkSum=false;
   bool skipModCkSum=false;
   bool hasDeltaUpdate=false;
   std::string deltaPrefix="delta_";
   tUInt deltaPrefixSize=static_cast<tUInt> (deltaPrefix.size());
   for (XmlItemAccessIter releaseIter(release, XmlItemAccessIter::enOnEnter);!releaseIter.isDone();releaseIter.next()) {
      XmlItemAccess access=releaseIter.getAccess();
      TiXmlElement *bxmlElem=access.getXml();
      if (!hasDeltaUpdate && bxmlElem) {
         /* check if anywhere "SOURCE_FILE_TYPE" starts with "delta_".
            we need to know this during update, e.g. we must not clear the fc-partition when
            it is still needed for a delta-update
         */
         std::string sourceFileType=swu::getTextFromChildOrEmpty(bxmlElem, "SOURCE_FILE_TYPE");
         if (!sourceFileType.empty()) {
            if (!sourceFileType.compare(0, deltaPrefixSize, deltaPrefix)) {
               hasDeltaUpdate =true;
            }
         }
      }
      if (access.enGetType()==XmlItemAccess::enType_Device) {
         skipDevCkSum=combineSkipCkSum(bxmlElem, skipCheckSumOverall);
      }
      if (access.enGetType()==XmlItemAccess::enType_Module) {
         skipModCkSum=combineSkipCkSum(bxmlElem, skipDevCkSum);
      }
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         if (combineSkipCkSum(bxmlElem, skipModCkSum)) {
            ETG_TRACE_USR1(("boschXmlPostProcess: setIgnoreCkSum(%s)", access.getLongName().c_str()));
            setIgnoreCkSum(bxmlElem);
         }
      }
   }
   
   // indicate in release-section, that the release contains delta-update(s)
   swu::setUIntChild(release, "HAS_DELTA", hasDeltaUpdate ? 1: 0);


   return tenSwUpdateError_OK;
}


tenSwUpdateError ReleaseFilterRename::filterElem(TiXmlElement *elem)
{
    std::string itemValue(elem->Value());	
    std::string newitemValue;

    if (itemValue.length() > _ending.length()) {
       if (!itemValue.compare (itemValue.length() - _ending.length(), _ending.length(), _ending)) {
          newitemValue.assign(itemValue,0,itemValue.length()-2);
          elem->SetValue(newitemValue.c_str());
       }
    }

    TiXmlElement* it = elem->FirstChildElement();
    for(;it;it=it->NextSiblingElement())
    {
	filterElem(it);
    }
    return tenSwUpdateError_OK;
}


tenSwUpdateError ReleaseFilterCompat::filterElem(TiXmlElement *elem)
{
   bool compatSat = false;
   TiXmlElement* it = elem->FirstChildElement();
   TiXmlElement* next_it;

   for(;it;)
   {
	next_it = it->NextSiblingElement();
	if(!checkCompat(it)) {	
		elem->RemoveChild(it);
	} else {
		filterElem(it);
        }
	it = next_it;
    }
   return tenSwUpdateError_OK;
}

ReleaseFilterCompat::ReleaseFilterCompat() 
{
   _installedCompatInd=Config::instance()->cfg_SwuCompatIndex.get();
}

bool ReleaseFilterCompat::checkCompat(TiXmlElement *elem)
{
	bool compatSat = false;
     	int minCompatInd = 0, maxCompatInd = INT_MAX; 

	elem->Attribute("minSwuCompatIndex", &minCompatInd);
	elem->Attribute("maxSwuCompatIndex", &maxCompatInd);


   if((static_cast<int> (_installedCompatInd) >= minCompatInd)  && (static_cast<int> (_installedCompatInd) <= maxCompatInd)) {   // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      compatSat = true;
   } else {
      printf("\n COMPATIBLITY NOT SATISFIED removing item = %s",elem->Value());
   }
   return compatSat;
}


tenSwUpdateError ReleaseFilterDnlSize::filterElem(TiXmlElement *root)
{
	SWU_ASSERT_RETURN_VAL(root, tenSwUpdateError_ERROR_UNKNOWN);
	TiXmlElement *release = root->FirstChildElement("RELEASE");
	SWU_ASSERT_RETURN_VAL(release, tenSwUpdateError_ERROR_UNKNOWN);
	tU64 ownSize = 0;

        // map storing current size of each level in bosch.xml
        tU64 _dnlSizes[(tU32)XmlItemAccess::enType_Invalid]={0};

	for (XmlItemAccessIter releaseIter(release, XmlItemAccessIter::enOnLeave); !releaseIter.isDone(); releaseIter.next())
	{
           XmlItemAccess access = releaseIter.getAccess();
           tUInt iItemType=(tUInt)access.enGetType();
           if (iItemType>XmlItemAccess::enType_Submodule) {
              continue;
           }
           // take curent size, that has already been set by the children
           tU64 itemSize=_dnlSizes[iItemType];
           _dnlSizes[iItemType]=0;
           TiXmlElement *flashItems = access.getXml()->FirstChildElement("FLASH_ITEMS");
           if (flashItems) {
              for (TiXmlElement *flashItem = flashItems->FirstChildElement("FLASH_ITEM");
                   flashItem;
                   flashItem = flashItem->NextSiblingElement("FLASH_ITEM"))
              {
                 TiXmlElement *chkSumXml = flashItem->FirstChildElement("CKSUM");
                 if (!chkSumXml)
                    continue;
                 // add size of own flash-item
                 itemSize += swu::getUIntFromChild(chkSumXml, "SIZE");
              }
           }
           
           swu::setTextChild(access.getXml(), "DNL_SIZE", swu::u64ToString(itemSize));
           ownSize+=itemSize;
           
           if (itemSize && (iItemType>(tU32)XmlItemAccess::enType_Release)) {
              ETG_TRACE_USR4(("ReleaseFilterDnlSize::adding to upper itemSize=%20s: %20s->%s", 
                              swu::u64ToString(itemSize).c_str(), swu::u64ToString(_dnlSizes[iItemType-1]).c_str(), swu::u64ToString(_dnlSizes[iItemType-1]+itemSize).c_str()));
              // add item-size to parent
              _dnlSizes[iItemType-1]+=itemSize;
           };
           // own size has been added to parent, reset it for next item of this level.
           itemSize=0;
	}
        
	return tenSwUpdateError_OK;
}

bool BxmlFilterMatch::matches(TiXmlElement *elem) {
   // filter out elements:
   // example: only keep element DEVICE if target-value FICD is either "0001" or "0002"
   // first format: 
   // try if key is provided as attribute e.g.
   // <DEVICE FCID="0001, 0002">
   const char *valList=elem->Attribute(_key.c_str());
   if (!valList) {
      // second format: 
      // try if key is provided as attribute FILTERKEY and value as attribute FILTERVALUE
      // <DEVICE FILTERKEY="FCID" FILTERVALUE="0001, 0002">
      const char *filterKey=elem->Attribute("FILTERKEY");
      if (filterKey && _key==filterKey  ) {
         valList=elem->Attribute("FILTERVALUE");
      }
   }
   if (!valList) {
      // filter-key does not exist, so element may pass
      return true;
   }

   
   std::set<std::string> vals=swu::splitString(valList);
   bool res=vals.find(_val)!=vals.end();
   return res;
}

tenSwUpdateError BxmlFilterMatch::filterElem(TiXmlElement *elem) {
   bool keep = false;
   TiXmlElement* it = elem->FirstChildElement();
   TiXmlElement* next_it;

   for(;it;)
   {
      next_it = it->NextSiblingElement();
      if(!matches(it)) {	
        // ETG_TRACE_USR4(("filterElem(%s) REMOVE!",it->Value())); 
         elem->RemoveChild(it);
      } else {
         filterElem(it);
      }
      it = next_it;
   }
   return tenSwUpdateError_OK;
   
}

bool BxmlFilterDefaults::isDefault(TiXmlElement *elem) {
   // filter out elements:
   // we check for existence of the arribute "DEFAULT" of for the filterkey "DEFAULT"
   // first format: 
   // try if key is provided as attribute e.g.
   // <DEVICE FCID="0001, 0002">
   static std::string defaultKey="DEFAULT"; 
   bool res = false;
   if (elem->Attribute("DEFAULT")) {
      res = true;
   }
   else {
      // second format: 
      // try if key is provided as attribute FILTERKEY and value as attribute FILTERVALUE
      // <DEVICE FILTERKEY="FCID" FILTERVALUE="0001, 0002">
      const char *filterKey=elem->Attribute("FILTERKEY");
      if (filterKey && defaultKey==filterKey  ) {
         res = true;
      }
   }
   return res;
}


tenSwUpdateError BxmlFilterDefaults::filterElem(TiXmlElement *elem) {
   bool keep = false;
   TiXmlElement* child=0;
   TiXmlElement* next_child;

   std::map<std::string, TiXmlElement *> defaultMap;
   // iterate all children to remove not needed defaults
   for(child=elem->FirstChildElement();child;)
   {
      next_child = child->NextSiblingElement();
      std::string value=child->Value();
      std::map<std::string, TiXmlElement *>::iterator mapIter=defaultMap.find(value);
      
      TiXmlElement *newDefault=isDefault(child) ? child : (TiXmlElement *)0;
      if (mapIter == defaultMap.end()) {
         // we encounter the Value the first time, take whatever we get.
         mapIter=defaultMap.insert(std::pair<std::string, TiXmlElement *>(value, newDefault)).first;
      }
      else {
         TiXmlElement *oldDefault=(*mapIter).second;
         // we already had this Value
         if (newDefault) {
            // discard default-element, we already have something
            ETG_TRACE_ERR(("BxmlFilterDefaults::filterElem %s: remove duplicate default", value.c_str()));
            elem->RemoveChild(child);
         }
         else if (oldDefault) { // && !newDefault
            // we got a non-default, so we have to delete the existing default
            ETG_TRACE_USR1(("BxmlFilterDefaults::filterElem %s: remove default since we found non-default", value.c_str()));
            elem->RemoveChild(oldDefault);
            (*mapIter).second=(TiXmlElement *)0;
         }
      }
      child=next_child;
   }
   
   // now apply same filter to all children
   for(;child;child->NextSiblingElement())
   {
      filterElem(child);
   }

   return tenSwUpdateError_OK;
   
}


tenSwUpdateError BXmlMoveElemFromParentToChild::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {

  tenSwUpdateError enError = tenSwUpdateError_OK ;
  TiXmlElement *release = getReleaseSection(boschXmlDoc);
  SWU_ASSERT_RETURN_VAL(release, tenSwUpdateError_ERROR_METAINFO_PARSE);

  ETG_TRACE_USR1(("BXmlMoveElemFromParentToChild: START %s", _name.c_str()));

  // find all elements of type _parentType
  for (XmlItemAccessIter srcIter(release, XmlItemAccessIter::enOnEnter); !srcIter.isDone(); srcIter.next()) {
     XmlItemAccess srcAccess(srcIter);
     if(srcAccess.enGetType() == _parentType) {
        TiXmlElement *elemToMove=srcAccess.getXml()->FirstChildElement(_tagName.c_str());
        if (!elemToMove) {
           // from-element doesnt contain child _tagName
           continue;
        }
        ETG_TRACE_ERR(("BXmlMoveElemFromParentToChild: src= %s", srcAccess.getLongName().c_str()));
        // get subelement of _parentType, e.g. UPDATE_TIME
        TiXmlElement *childXml = srcAccess.getFirstChildXml();
        if (!childXml) {
           ETG_TRACE_ERR(("BXmlMoveElemFromParentToChild: no bxml-child in  %s", srcAccess.getLongName().c_str()));
           // element has not required child
           continue;
        }
        
        ETG_TRACE_USR1(("BXmlMoveElemFromParentToChild: moving to %s", XmlItemAccess(childXml).getLongName().c_str()));
        
        swu::moveElem(elemToMove, childXml);
     }     
  }
  return enError;
}

tenSwUpdateError BxmlFilterBaseNodes::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   TiXmlElement *root = boschXmlDoc->FirstChildElement();
   if (!root) {
      ETG_TRACE_USR4(("BxmlFilterBaseNodes::doFilter(): root not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }
   TiXmlElement *overall=root->FirstChildElement("OVERALL");
   if(!overall) {
      ETG_TRACE_USR4(("BxmlFilterBaseNodes::doFilter(): overall not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   trSourceInfo const &sourceInfo=context.sourceInfo;
   swu::setTextChild(overall, "MEDIUM_DRIVE", sourceInfo.path);
   swu::setTextChild(overall, "INFO_FILE", sourceInfo.infoFile);
   swu::setUIntChild(overall, "SOURCE_TYPE", (tU32)sourceInfo.enSourceType);


   TiXmlElement *release=root->FirstChildElement("RELEASE");
   if(!release) {
      ETG_TRACE_USR4(("BxmlFilterBaseNodes::doFilter(): release not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   TiXmlElement *Bxmlroot = boschXmlDoc->FirstChildElement("BOSCHXML");
   if(Bxmlroot) {
      ETG_TRACE_USR3(("boschxml has to be changed to updatexml"));
      changeRootBxmlToUXml(boschXmlDoc);
   } else {
      ETG_TRACE_USR3(("Input is in updatexml format no need to convert"));
   }
        

   return tenSwUpdateError_OK;
}

tVoid BxmlFilterBaseNodes::changeRootBxmlToUXml(TiXmlDocument *boschXmlDoc) {

   ETG_TRACE_USR1(("BxmlFilterBaseNodes::changeRootBxmlToUXml - START"));
   TiXmlElement *overAll = getOverallSection(boschXmlDoc);
   TiXmlElement *release = getReleaseSection(boschXmlDoc);
   swu::setUIntChild(overAll, "UPDATEXML_CONVERSION_PENDING", 1);

   TiXmlElement *root = new TiXmlElement("UPDATEXML");
   root->InsertEndChild(*overAll);
   root->InsertEndChild(*release);

   boschXmlDoc->Clear();
   boschXmlDoc->InsertEndChild(*root);
   ETG_TRACE_USR1(("BxmlFilterBaseNodes::changeRootBxmlToUXml - END"));

}

tenSwUpdateError SxmTypeFilter::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {

  tenSwUpdateError enError = tenSwUpdateError_ERROR_UNKNOWN ;
  TiXmlElement *release = getReleaseSection(boschXmlDoc);
  SWU_ASSERT_RETURN_VAL(release, enError);

  std::string hardware_type;

  for (XmlItemAccessIter it(release, XmlItemAccessIter::enOnEnter); !it.isDone(); it.next())
  {
     if (it.getAccess().enGetType() == XmlItemAccess::enType_Device)
     {
        TiXmlElement* deviceSection = it.getXml();
        std::string stNameText = swu::getTextFromChildOrEmpty(deviceSection, "NAME");
        if(stNameText == "SXM") {

           SXMHardwareType hardwareTypeInst;
           hardware_type = hardwareTypeInst.get();

           if(hardware_type.length() > 0 && hardware_type != "unknown" ) {
             
              //have SXM_TYPE in existing EXTRA_ARGS under submodule.
              XmlItemAccess modAccess = it.getAccess().getFirstChild();
              SWU_ASSERT_RETURN_VAL(modAccess.getXml(), enError);
              XmlItemAccess subModAccess = modAccess.getFirstChild();
              SWU_ASSERT_RETURN_VAL(subModAccess.getXml(), enError);
              subModAccess.addExtraArgs("SXM_TYPE", hardware_type);

              BxmlFilterMatch* filter=new BxmlFilterMatch("SXM_TYPE", hardware_type);
              SWU_ASSERT_RETURN_VAL(filter, enError);
              filter->doFilter(boschXmlDoc, context);
              delete filter;

           } else {
              release->RemoveChild(deviceSection);
           }

        }
     }
  }
  return tenSwUpdateError_OK;
}



tenSwUpdateError ReleaseFilterFileExistence::checkFileItem(TiXmlElement *fileItem, std::string const &releasePath) {
   char const *fPath=swu::getTextFromChild(fileItem, "PATH", false);
   if (!fPath) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): item PATH not found"));
      return tenSwUpdateError_ERROR_IMAGE_READ;
   }
   char const *fName=swu::getTextFromChild(fileItem, "FILENAME", false);
   if (!fName) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): item FILENAME not found"));
      return tenSwUpdateError_ERROR_IMAGE_READ;
   }
   std::string FileName=releasePath + "/" + fPath + "/" + fName;
   if (!Config::instance()->cfg_EnterEngineering.readAsBool()) {

      if (!swu::exists(FileName)) {
         ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): file %s does not exist", FileName.c_str()));

         return tenSwUpdateError_ERROR_IMAGE_READ;
      }


   }

   TiXmlElement *chkSumXml=fileItem->FirstChildElement("CKSUM");
   if (!chkSumXml) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): item CKSUM not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND;
   }

   char const *sizeString=swu::getTextFromChild(chkSumXml, "SIZE", false, 0);
   if (!sizeString) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): item CKSUM::SIZE not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND;
   }
   unsigned long long int fileSizeXml=strtoll(sizeString, 0, 10);
   if (fileSizeXml > std::numeric_limits<tU32>::max()) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): file size exceeds limit: %s",
                      sizeString));
      return tenSwUpdateError_ERROR_IMAGE_SIZE;
   }

   if (!Config::instance()->cfg_EnterEngineering.readAsBool()) {
      off_t fileSizeFile=swu::getFileSize(FileName);
         if (static_cast<off_t>(fileSizeXml) != fileSizeFile) {
         ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): fileSizeXml (%u) != fileSizeFile (%u)",
                         fileSizeXml, fileSizeFile));
         return tenSwUpdateError_ERROR_IMAGE_SIZE;
      }
   }
   return  tenSwUpdateError_OK;

}


tenSwUpdateError ReleaseFilterFileExistence::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   TiXmlElement *root = boschXmlDoc->FirstChildElement();
   if (!root) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): root not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }
   TiXmlElement *overall=root->FirstChildElement("OVERALL");
   if(!overall) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): OVERALL not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   char const * overallMediumDrive=swu::getTextFromChild(overall, "MEDIUM_DRIVE", false);
   if (!overallMediumDrive) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): OVERALL::MEDIUM_DRIVE not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
   }
   char const * overallBasePath=swu::getTextFromChild(overall, "BASEPATH", false);
   if (!overallBasePath) {
      ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): OVERALL::BASEPATH not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
   }
   
   bool ignoreDataFiles=false;
   if (swu::getUIntFromChild(overall, "POST_IMAGE_MODE") || Config::instance()->cfg_PostImageInstallationRunning.get()) {
      ignoreDataFiles=true;
   }
   
   std::string overallPath=std::string(overallMediumDrive)+ "/" + overallBasePath;
   if (swu::realPath(overallPath).find("/var/opt/bosch/persistent") != std::string::npos) {
      // release is stored in persistent partition and is expected to be incomplete
      ignoreDataFiles=true;
   }
   std::string releasePath=overallPath;
   for (XmlItemAccessIter releaseIter(root, XmlItemAccessIter::enOnEnter);!releaseIter.isDone();releaseIter.next()) {
      XmlItemAccess access=releaseIter.getAccess();
      
      if (access.enGetType()==XmlItemAccess::enType_Device) {
         releasePath=overallPath;
         char const * relPath=swu::getTextFromChild(access.getXml(), "BASEPATH", false);
         if (!relPath) {
            ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): %s::BASEPATH not found",
                            access.getLongName().c_str()));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }

         releasePath+= "/";
         releasePath+= relPath; 
      }

      TiXmlElement *bxmlElem=access.getXml();
      TiXmlElement *toolsElem=bxmlElem->FirstChildElement("TOOLS");
      if (toolsElem) {
         tenSwUpdateError errorCode=checkFileItem(toolsElem, releasePath);
         if (errorCode !=tenSwUpdateError_OK) {
            std::string strName = access.getLongName();  //Coverity fix for 209457
            ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): %30s::tools-elem not ok, error=%u",
                            strName.c_str(), errorCode));
            return errorCode;
         }
      }
         
      if (ignoreDataFiles) {
         // data-files not needed in post-image-mode
         continue;
      }
      TiXmlElement *flashItems=bxmlElem->FirstChildElement("FLASH_ITEMS");
      if (flashItems) {
         for (TiXmlElement *flashItem=flashItems->FirstChildElement("FLASH_ITEM");
              flashItem;
              flashItem=flashItem->NextSiblingElement("FLASH_ITEM"))
         {
            tenSwUpdateError errorCode=checkFileItem(flashItem, releasePath);
            if (errorCode != tenSwUpdateError_OK) {
	    std::string strName = access.getLongName();  //Coverity fix for 209457
            ETG_TRACE_USR4(("ReleaseFilterFileExistence::checkFileItem(): %30s::flash-item not ok, error=%u",
                            strName.c_str(), errorCode));
               return errorCode;
            }
            // todo: check if we have a tools-item.
            // todo: same check for script-nodes
         }
      }
   }
   return tenSwUpdateError_OK;
}

tenSwUpdateError ReleaseFilterMandatoryNodes::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   TiXmlElement *root = boschXmlDoc->FirstChildElement();
   if (!root) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): root not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }
   TiXmlElement *overall=root->FirstChildElement("OVERALL");
   if(!overall) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): OVERALL not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }
   TiXmlElement *release = getUXmlSection(boschXmlDoc, XmlItemAccess::enType_Release); 
   TiXmlElement* moduleElem = getUXmlSection(boschXmlDoc, XmlItemAccess::enType_Module);
   if(!release || !moduleElem) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): RELEASE or MODULE not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   if (!std::string(swu::getTextFromChildOrEmpty(moduleElem, "FINALNAME")).size()) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): FINALNAME not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
   }
   if (!swu::getTextFromChild(overall, "MEDIUM_DRIVE", false)) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): MEDIUM_DRIVE not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
   }
   if (!swu::getTextFromChild(overall, "BASEPATH", false)) {
      ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): overall::BASEPATH not found"));
      return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
   }
   for (XmlItemAccessIter releaseIter(release, XmlItemAccessIter::enOnEnter);!releaseIter.isDone();releaseIter.next()) {
      XmlItemAccess access=releaseIter.getAccess();
      TiXmlElement *bxmlElem=access.getXml();      
      if (access.enGetType()==XmlItemAccess::enType_Device) {
         if (!std::string(swu::getTextFromChildOrEmpty(bxmlElem, "NAME")).size()) {
            ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): %s::NAME not found",
                            access.getLongName().c_str()));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }

         if (!std::string(swu::getTextFromChildOrEmpty(bxmlElem, "BASEPATH")).size()) {
            ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): device::BASEPATH not found"));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }
      }
      if (access.enGetType()==XmlItemAccess::enType_Module) {
         if (!std::string(swu::getTextFromChildOrEmpty(bxmlElem, "NAME")).size()) {
            ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): %s::NAME not found",
                            access.getLongName().c_str()));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }
      }      
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         if (!std::string(swu::getTextFromChildOrEmpty(bxmlElem, "TYPE")).size()) {
            ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): %s::TYPE not found",
                            access.getLongName().c_str()));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }
         if (!std::string(swu::getTextFromChildOrEmpty(bxmlElem, "VERSION")).size()) {
            ETG_TRACE_USR4(("ReleaseFilterMandatoryNodes::checkFileItem(): %s::VERSION not found",
                            access.getLongName().c_str()));
            return tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; 
         }
      }
   }
   return tenSwUpdateError_OK;     
}

tenSwUpdateError ReleaseFilterCheckFcids::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
      ETG_TRACE_USR4(("ReleaseFilterCheckFcids::doFilter(): START"));

   TiXmlElement *root = boschXmlDoc->FirstChildElement();
   if (!root) {
      ETG_TRACE_USR4(("ReleaseFilterCheckFcids::checkFileItem(): root not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   TiXmlElement* module = getUXmlSection(boschXmlDoc, XmlItemAccess::enType_Module);
   if(!module) {
      ETG_TRACE_USR4(("ReleaseFilterCheckFcids::checkFileItem(): MODULE not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

      //TODO - FCID check for multiple releases, assuming only one release per source at the moment
   std::string targetFCID;
   
   if (!Config::instance()->cfg_FcIdStr.get(targetFCID)) {
       ETG_TRACE_USR1(("No FCID configured in target (KDS)"));
      return tenSwUpdateError_OK;     
   }

   // due to a bug in the media-generation, multiple section FCIDS may exists
   std::string supFCIDstr;
   for (TiXmlElement *fcidsNode=module->FirstChildElement("FCIDS");
        fcidsNode;
        fcidsNode=fcidsNode->NextSiblingElement("FCIDS")
        ) {
      ETG_TRACE_USR1(("Found FCIDs Tag"));
      const char *text = fcidsNode->GetText();
      if (text && (strlen(text)>=4)) {
         ETG_TRACE_USR1(("Found FCIDs text:%s", text));
         // we found something that should be at least one fcid, e.g. "0815"
         if (supFCIDstr.size()) {
            // we already have content, add "," for concatenation
            supFCIDstr+=",";
         }
         supFCIDstr+=text;
      }
   }
   if (supFCIDstr.empty()) {
       ETG_TRACE_USR1(("No FCIDS tag found in bosch.xml"));
      return tenSwUpdateError_OK;     
   }

   ETG_TRACE_USR1(("All FCIDS:%s", supFCIDstr.c_str()));
   std::set<std::string> supFCIDset = swu::splitString(supFCIDstr.c_str(),',');    
   std::set<std::string>::iterator it;
   for (it = supFCIDset.begin(); it != supFCIDset.end(); ++it)
   {
      ETG_TRACE_USR1(("FCID specfied in FCIDS tag %s",it->c_str()));
   }
   
   if (supFCIDset.find(targetFCID) == supFCIDset.end()) {
      ETG_TRACE_COMP(("FCID specified in bosch.xml (%s) not compatible with the target", targetFCID.c_str()));
      return tenSwUpdateError_ERROR_METAINFO_VARIANT_CONFLICT;
   }
         //FCID NOK
   ETG_TRACE_USR1(("FCID specified in bosch.xml is compatible with the target"));
   return tenSwUpdateError_OK;


}


tenSwUpdateError ReleaseFilterCheckPrjId::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context)
{
   ETG_TRACE_USR4(("ReleaseFilterCheckPrjId::doFilter(): START"));

   TiXmlElement *root=boschXmlDoc->FirstChildElement();
   if(!root)
   {
      ETG_TRACE_USR4(("ReleaseFilterCheckPrjId::checkFileItem(): root not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }
   TiXmlElement *overall=root->FirstChildElement("OVERALL");
   if(!overall)
   {
      ETG_TRACE_USR4(("ReleaseFilterCheckPrjId::checkFileItem(): OVERALL not found"));
      return tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
   }

   std::string targetPrjId;
   
   if(!Config::instance()->cfg_PrjIdStr.get(targetPrjId))
   {
      ETG_TRACE_USR1(("No PrjId configured in target (KDS)"));
      return tenSwUpdateError_OK;     
   }

   char const *supPrjId=swu::getTextFromChild(overall, "PRJID", false);
   if(!supPrjId)
   {
      ETG_TRACE_USR1(("No PRJID tag found in bosch.xml"));
      return tenSwUpdateError_OK;     
   }

   ETG_TRACE_USR1(("PrjId specfied in PRJID tag %s", supPrjId));
   
   if(targetPrjId!=supPrjId)
   {
      ETG_TRACE_COMP(("PRJID specified in bosch.xml (%s) not compatible with the target", targetPrjId.c_str()));
      return tenSwUpdateError_ERROR_METAINFO_VARIANT_CONFLICT;
   }
   ETG_TRACE_USR1(("PRJID specified in bosch.xml is compatible with the target"));
   return tenSwUpdateError_OK;
}


#define NAVIGATE_OR_BREAK(ELEMPTR, RESELEMPTR, CHILDNAME, ERR)        \
   TiXmlElement *RESELEMPTR=0;                                     \
   if (!(ELEMPTR)) { res=ERR; break; }                             \
   RESELEMPTR=(ELEMPTR)->FirstChildElement(CHILDNAME);                \
   if (!RESELEMPTR) { res=ERR; break; }

#define ATTRIBUTE_OR_BREAK(ELEMPTR, RESCHRPTR, ATTRIBNAME, ERR) \
   const char *RESCHRPTR=0;                                     \
   if (!(ELEMPTR)) { res=ERR; break; }                          \
   RESCHRPTR=(ELEMPTR)->Attribute(ATTRIBNAME);                 \
   if (!RESCHRPTR) { res=ERR; break; }

#define ATTRIBUTE_OR_DEFAULT(ELEMPTR, RESCHRPTR, ATTRIBNAME,DEFAULT)      \
   const char *RESCHRPTR=0;                                     \
   if (!(ELEMPTR)) { res=tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND; break; }                          \
   RESCHRPTR=(ELEMPTR)->Attribute(ATTRIBNAME);                 \
   if (!RESCHRPTR) { RESCHRPTR = DEFAULT; }


#define REPLACE_OR_BREAK(IN, WHAT, WITH)                                \
   if (!swu::replaceSubString(IN, WHAT, WITH)){ res=tenSwUpdateError_ERROR_UNKNOWN; break; }
   
tenSwUpdateError ReleaseFilterConvertScomoIndex::doFilter(TiXmlDocument *indexXmlDoc, ReleaseFilterContext const &context) {
   trSourceInfo sourceInfo=context.sourceInfo;
   tenSwUpdateError res=tenSwUpdateError_OK;

   do {
      if (sourceInfo.enSourceType != tenSourceType_SCOMO_INDEX) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: Skip, sourceType not SCOMO_INDEX"));
         break;               
      }

      // open index.xml
      std::string indexXmlFile=sourceInfo.path + "/index.xml";
      if (!swu::isFile(indexXmlFile)) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: missing index.xml in %s", sourceInfo.path.c_str()));
         res = tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND;
         break;
      }
      
      swu::ScopedPointer<TiXmlDocument> scopedDoc(new TiXmlDocument);
      TiXmlDocument *indexXml = scopedDoc.getPtr();      
      if (!indexXml->LoadFile(indexXmlFile.c_str())) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: could not load index.xml in %s", sourceInfo.path.c_str()));
         return tenSwUpdateError_ERROR_METAINFO_PARSE;
      }
      
      // get needed xml-nodes
      NAVIGATE_OR_BREAK(indexXml, compPackElement, "componentPackage", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
      NAVIGATE_OR_BREAK(compPackElement, infoElement, "info", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
      NAVIGATE_OR_BREAK(infoElement, infoPackElement, "package", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
      NAVIGATE_OR_BREAK(infoElement, infoCompElement, "component", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
      NAVIGATE_OR_BREAK(compPackElement, contentElement, "content", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
      NAVIGATE_OR_BREAK(contentElement, contentFileElement, "file", tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);

      
      // read needed attributes
      ATTRIBUTE_OR_BREAK(infoCompElement, compObjectId, "objectId", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      ATTRIBUTE_OR_BREAK(infoCompElement, version, "version", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);      
      //ATTRIBUTE_OR_BREAK(installationElement, durationChr, "EstimatedDuration", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      ATTRIBUTE_OR_BREAK(contentFileElement, dataFileName, "name", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      ATTRIBUTE_OR_BREAK(contentFileElement, dataFileFormat, "format", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      ATTRIBUTE_OR_BREAK(contentFileElement, dataFileHash, "hash", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      ATTRIBUTE_OR_BREAK(contentFileElement, dataFilePath, "path", tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
      

      // // parse duration
      // std::vector<std::string> durationElems=swu::stringToVector(durationChr, ':');
      // if(durationElems.size()!=3) {
      //    ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: coud not parse duration: %s", durationChr));
      //    res = tenSwUpdateError_ERROR_METAINFO_PARSE;
      //    break;
      // }
      // tUInt durationUInt=swu::stringToUInt(durationElems[0]) * 3600 
      //    + swu::stringToUInt(durationElems[1]) * 60
      //    + swu::stringToUInt(durationElems[2]);
         
      // std::string duration=swu::intToString(durationUInt);
      std::string duration("60");
      
         off_t contentSize=swu::getFileSize(Config::instance()->cfg_OtaUnpackedDir.get() + "/" + std::string(dataFileName));
      
      if (std::string(compObjectId) == "RDO.SOFTWARE") {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: unsupported objectId it is a firmware object: %s", 
                         compObjectId));
         res = tenSwUpdateError_ERROR_UNKNOWN;
         break;
      }
      // collect data of preinstalled tools-container
      Config *cfg=Config::instance();
      std::string toolsPathAndName=cfg->cfg_DefaultToolsContainerPathAndName.get();
      if (! swu::exists(toolsPathAndName)) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: missing tools-container in: %s", toolsPathAndName.c_str()));
         res = tenSwUpdateError_ERROR_UNKNOWN;
         break;
      }
      
      std::string toolsPath(swu::parentDirname(toolsPathAndName));
      std::string toolsName(swu::getFileNamePortion(toolsPathAndName));
      off_t toolsSize=swu::getFileSize(toolsPathAndName);
      
      std::string toolsHash;
      swu::calculateSHA256HashFromFile(toolsPathAndName, toolsHash);

      if(! swu::exists(_scomoTemplateXmlFile)) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: missing scomoTemplate %s", _scomoTemplateXmlFile));
         res = tenSwUpdateError_ERROR_UNKNOWN;
         break;
      }
      
      // read template of update.xml into string
      std::string resXmlStr;
      if (! swu::loadFile(_scomoTemplateXmlFile, resXmlStr)) {
         res = tenSwUpdateError_ERROR_UNKNOWN;               
         break;               
      }

      ETG_TRACE_USR3(("Replace or Break - START"));
      // replace items in string of update.xml
      REPLACE_OR_BREAK(resXmlStr, "__VERSION__", version);
      REPLACE_OR_BREAK(resXmlStr, "__UPDATETIME__", duration);
      REPLACE_OR_BREAK(resXmlStr, "__DATAFILENAME__", dataFileName);
         REPLACE_OR_BREAK(resXmlStr, "__DATAFILESIZE__", swu::u64ToString(contentSize));
      REPLACE_OR_BREAK(resXmlStr, "__DATAFILEHASH__", dataFileHash);
      REPLACE_OR_BREAK(resXmlStr, "__TOOLSFILENAME__", toolsName);
      REPLACE_OR_BREAK(resXmlStr, "__TOOLSFILEPATH__", toolsPath);
         REPLACE_OR_BREAK(resXmlStr, "__TOOLSFILESIZE__", swu::u64ToString(toolsSize));
      REPLACE_OR_BREAK(resXmlStr, "__TOOLSFILEHASH__", toolsHash);
      //assume that component has 3 maximum phase.
      //todo: get this value from index.xml somehow.
      REPLACE_OR_BREAK(resXmlStr, "__SCOMOPHASE__", "2,3");
      ETG_TRACE_USR3(("Replace or Break - END"));

      /*
        todo: data-driven approach to get module-name etc. from key in iso-file.
        check which key to use in iso-file.
        We need a list of scomo-installers.
        Read component-id from index.xml and feth MODULE_NAME for update.xml.
        NAMEDISPLAY can be the same as NAME
       */
      REPLACE_OR_BREAK(resXmlStr, "__NAMEDISPLAY__", CampaignManager::instance()->getScomoComponentId());
      REPLACE_OR_BREAK(resXmlStr, "__NAME__", CampaignManager::instance()->getScomoComponentId());
          
      // create TiXmlDocument from string
      TiXmlDocument *updateXml=new TiXmlDocument;
      if (!updateXml->Parse(resXmlStr.c_str())) {
         ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: TiXml could not parse generated bosch.xml"));
         delete updateXml;
         res = tenSwUpdateError_ERROR_UNKNOWN;               
         break;
      }
      indexXmlDoc->Clear();
      *indexXmlDoc=*updateXml;
      indexXmlDoc->SaveFile("/tmp/boschIndex.xml");

      delete updateXml;
      updateXml=0;
   } while (0);
   
   if (res != tenSwUpdateError_OK) {
      ETG_TRACE_COMP(("ReleaseFilterConvertScomoIndex: failed, res=%u", res));
   }

   return res;
}



tenSwUpdateError BxmlFilterExtraArgs::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context)
{
   tenSwUpdateError enError = tenSwUpdateError_ERROR_UNKNOWN ;

   TiXmlElement *overall = getOverallSection(boschXmlDoc);
   SWU_ASSERT_RETURN_VAL(overall, enError);
   (void)filterElem(overall);

   TiXmlElement *release = getReleaseSection(boschXmlDoc);
   SWU_ASSERT_RETURN_VAL(release, enError);

   for (XmlItemAccessIter it(release, XmlItemAccessIter::enOnEnter); !it.isDone(); it.next())
   {
      switch (it.getAccess().enGetType())
      {
         case  XmlItemAccess::enType_Release:
         case  XmlItemAccess::enType_Device:
         case  XmlItemAccess::enType_Module:
         case  XmlItemAccess::enType_Submodule:
         {
            (void)filterElem(it.getXml());
            break;
         }

         default:
            break;
      }
   }

   return tenSwUpdateError_OK;
}


tenSwUpdateError BxmlFilterExtraArgs::filterElem(TiXmlElement *elem)
{
   // move all elem->EXTRA_ARGS->EXTRA_ARG to elem->EXTRA_ARG
   int count = 0;
   for (TiXmlElement *extras = elem->FirstChildElement("EXTRA_ARGS"); extras; ) {
      for (TiXmlElement const *arg = extras->FirstChildElement("EXTRA_ARG"); arg; arg = arg->NextSiblingElement("EXTRA_ARG"))
      {
         elem->InsertBeforeChild(extras, *arg);
         ++count;
      }
      
      TiXmlElement *tmp=extras;
      extras=extras->NextSiblingElement("EXTRA_ARGS");
      elem->RemoveChild(tmp);
      ETG_TRACE_USR4(("filterElem moved %d args", count));
   }
   
   // remove all extra-args with missing value
   TiXmlElement *extra = elem->FirstChildElement("EXTRA_ARG"); 
   while(extra) {
      TiXmlElement *curExtra=extra;
      extra=extra->NextSiblingElement("EXTRA_ARG");
      if (!swu::getTextFromChild(curExtra, "VALUE", false)) {
         elem->RemoveChild(curExtra);
      }
   }

   return tenSwUpdateError_OK;

}

tenSwUpdateError ReleaseFilterCheckSRK::getProcessedError(tenSwUpdateError error)
{
   if(error != tenSwUpdateError_OK && Config::instance()->cfg_HmiCfgNeedsSecureBootErrors.readAsBool())
      return tenSwUpdateError_ERROR_IMAGE_SECURE_BOOT_VERIFICATION_FAILED;
   else
      return error;
}

tenSwUpdateError ReleaseFilterCheckSRK::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context)
{
   TiXmlElement *root;
   TiXmlElement *overall;
   TiXmlElement *release;
   TiXmlElement *secure_boot_info;
   tU32         targetSBMFlag;
   std::string  targetSRKHash;
   tU32         targetSRKRevocation;
   char const   *xmlSRKHash;
   char const   *xmlSRKIndexStr;
   tU32         xmlSRKIndex;

   ETG_TRACE_USR4(("ReleaseFilterCheckSRK::doFilter(): START"));

   if (!Config::instance()->cfg_SecureBootModeEnabled.get(targetSBMFlag))
   {
      ETG_TRACE_USR1(("Failed to read SecureBootMode flag from OTP"));
      // CHECKME: Returning no failure might brick targets, returning an error might prevent old versions from updating
      return getProcessedError(tenSwUpdateError_ERROR_OTP_READ_FAILED);
   }
   
   if(!targetSBMFlag)
   {
      ETG_TRACE_USR4(("SecureBootMode is not enabled - skipping verification of SRK hash"));
      return tenSwUpdateError_OK;
   }

   root = boschXmlDoc->FirstChildElement();
   if (!root)
   {
      ETG_TRACE_USR4(("ReleaseFilterCheckSRK::checkFileItem(): root not found"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
   }
   overall = root->FirstChildElement("OVERALL");
   release = root->FirstChildElement("RELEASE");

   if(!release){
      ETG_TRACE_USR4(("ReleaseFilterCheckSRK::checkFileItem(): release not found"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
   }

   //this check is needed for OEM updaters & special stick, if the target is secure boot.
   if(!isTextExistsInSubModule(release, "BOOT")) {
      ETG_TRACE_FATAL(("ReleaseFilterCheckSRK::doFilter: BOOT binary is not present in bosch.xml, no need to check for SRK keys"));
      return tenSwUpdateError_OK;
   }

   if(!overall)
   {
      ETG_TRACE_USR4(("ReleaseFilterCheckSRK::checkFileItem(): OVERALL not found"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
   }
   secure_boot_info=overall->FirstChildElement("SECURE_BOOT_INFO");
   if(!secure_boot_info)
   {
      ETG_TRACE_USR4(("ReleaseFilterCheckSRK::checkFileItem(): SECURE_BOOT_INFO not found"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
   }

   if (!Config::instance()->cfg_SRKHash.get(targetSRKHash))
   {
      ETG_TRACE_USR1(("Failed to read SRK hash from OTP"));
      return getProcessedError(tenSwUpdateError_ERROR_OTP_READ_FAILED);
   }

   if (!Config::instance()->cfg_SRKRevocation.get(targetSRKRevocation))
   {
      ETG_TRACE_USR1(("Failed to read SRK revocation from OTP"));
      return getProcessedError(tenSwUpdateError_ERROR_OTP_READ_FAILED);
   }

   xmlSRKHash = swu::getTextFromChild(secure_boot_info, "SRK_HASH", false);
   xmlSRKIndexStr = swu::getTextFromChild(secure_boot_info, "SRK_INDEX", false);

   if(!xmlSRKHash || !xmlSRKIndexStr)
   {
      ETG_TRACE_USR1(("SRK_HASH and SRK_INDEX required in bosch.xml"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_MANDATORY_TAG_NOT_FOUND);
   }
   xmlSRKIndex=atoi(xmlSRKIndexStr);

   ETG_TRACE_USR1(("SECURE_BOOT_INFO in bosch.xml specifies index %i and hash %s", xmlSRKIndex, xmlSRKHash));
   ETG_TRACE_USR1(("OTP in target specifies revocation %i and hash %s", targetSRKRevocation, targetSRKHash.c_str()));

   if (targetSRKHash!=xmlSRKHash)
   {
      ETG_TRACE_COMP(("SRKHash in bosch.xml does not match the one inside the target"));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_VARIANT_CONFLICT);
   }

   // XML counts SRKs as 1...4
   // OTP counts SRKs as 1...4 but provides status for 1...3 in bits 0...2 only because #4 can't be revoked
   if (targetSRKRevocation&(1<<(xmlSRKIndex-1)))
   {
      ETG_TRACE_COMP(("SRK #%i is revoked in the target", xmlSRKIndex));
      return getProcessedError(tenSwUpdateError_ERROR_METAINFO_VARIANT_CONFLICT);
   }

   ETG_TRACE_USR1(("SECURE_BOOT_INFO provided by bosch.xml is compatible with the target"));
   return tenSwUpdateError_OK;
}

tenSwUpdateError UpdateXmlFilter::doFilter(TiXmlDocument *boschXmlDoc, ReleaseFilterContext const &context) {
   ETG_TRACE_USR4(("UpdateXmlFilter::doFilter - Enter"));
 
   TiXmlElement *overall = getOverallSection(boschXmlDoc);
   SWU_ASSERT_RETURN_VAL(overall, tenSwUpdateError_ERROR_METAINFO_PARSE);

   TiXmlElement *release = getReleaseSection(boschXmlDoc);
   SWU_ASSERT_RETURN_VAL(release, tenSwUpdateError_ERROR_METAINFO_PARSE);

   if(swu::getUIntFromChild(overall, "UPDATEXML_CONVERSION_PENDING")) {
      convertBoschToUpdateXml(boschXmlDoc);
      swu::setUIntChild(overall, "UPDATEXML_CONVERSION_PENDING", 0);
   }
   
   return tenSwUpdateError_OK;
}

tVoid UpdateXmlFilter::convertBoschToUpdateXml(TiXmlDocument* boschdoc) {

   ETG_TRACE_USR4(("UpdateXmlFilter::convertBoschToUpdateXml - Enters"));

   //boschdoc->SaveFile("/tmp/input.xml");

   manipulateInReleaseTag(boschdoc);
   manipulateInOverallTag(boschdoc);
   mandatoryTagInOverall(boschdoc);

   //boschdoc->SaveFile("/tmp/test.xml");

   ETG_TRACE_USR4(("UpdateXmlFilter::convertBoschToUpdateXml - Exit"));
}

tVoid UpdateXmlFilter::manipulateInReleaseTag(TiXmlDocument* boschdoc) {
   ETG_TRACE_USR2(("manipulateInReleaseTag - START"));

   TiXmlElement* bosch_root = boschdoc->FirstChildElement();
   TiXmlElement* bosch_release = bosch_root->FirstChildElement("RELEASE");
   TiXmlElement* bosch_overall = bosch_root->FirstChildElement("OVERALL");
   
   _refModule = 0;
   TiXmlElement* releasecopy = new TiXmlElement("");
   //Coverity fix for 77676
   if( NULL == releasecopy )
   {
	ETG_TRACE_ERR(("new operator failed to allocate the memory for releasecopy in fcswupd_releaseFilterIf.cpp file\n"));
	return;
   }
   *releasecopy = *bosch_release;

   for (XmlItemAccessIter srcIter(releasecopy, XmlItemAccessIter::enOnBoth); !srcIter.isDone(); srcIter.next()) {
      XmlItemAccess srcAccess(srcIter);			
      XmlItemAccess copyAccess(bosch_release, srcAccess.getLongName());
      XmlItemAccessIter::tenDirection endir = srcIter.getDirection();

      switch (srcAccess.enGetType()){
         case XmlItemAccess::enType_Release:
            if (endir == XmlItemAccessIter::enLeave) {
               copyAccess.removeChildRecur();
               XmlItemAccess childAccess = copyAccess.getFirstChild();
               if(childAccess.getXml()) {
                  childAccess.moveChildFrom(copyAccess, "HWID");
                  childAccess.moveChildFrom(copyAccess, "VARIANT");
                  childAccess.moveChildFrom(copyAccess, "MUVERSION");
                  childAccess.moveChildFrom(copyAccess, "VENDOR");
                  childAccess.moveChildFrom(copyAccess, "BASEPATH");
                  childAccess.moveChildFrom(copyAccess, "TOOLS");
                  childAccess.moveChildFrom(copyAccess, "BXML_VERSION");     
                  childAccess.moveChildFrom(copyAccess, "EXTRA_ARGS");  
               }        
            }
            break;
         case XmlItemAccess::enType_Device:
            if (endir == XmlItemAccessIter::enLeave) {
               XmlItemAccess childAccess = copyAccess.getFirstChild();
               if(childAccess.getXml()) {	       
                  childAccess.moveChildFrom(copyAccess, "FINALIZE");
                  childAccess.moveChildFrom(copyAccess, "PREPARE_ONCE");
                  childAccess.moveChildFrom(copyAccess, "EXTRA_ARGS");
                  //todo:get the device name through config item
                  if (std::string(copyAccess.getName()) == "IMX" || std::string(copyAccess.getName()) == "rcar3"){
                     copyAccess.removeChildRecur();
                  }
                  swu::setTextChild(copyAccess.getXml(), "NAME", "IVI");
                  swu::setTextChild(copyAccess.getXml(), "NAME_DISPLAY", "IVI");
               }
            }			
            break;
         case XmlItemAccess::enType_Module:{
            if (endir == XmlItemAccessIter::enEnter) {				
               if (!_refModule) {
                  _refModule = copyAccess.getXml();
               }
               XmlItemAccess childAccess = copyAccess.getFirstChild();	
               if(childAccess.getXml()) {		
                  childAccess.moveChildRecur(copyAccess, "UPDATE_TIME");
                  childAccess.moveChildRecur(copyAccess, "GROUPS");
                  childAccess.moveChildRecur(copyAccess, "LANG");
                  childAccess.moveChildRecur(copyAccess, "EXTRA_ARGS");
                  childAccess.moveChildRecur(copyAccess, "FINALIZE");
                  childAccess.moveChildRecur(copyAccess, "PREPARE_ONCE");
                  childAccess.moveChildRecur(copyAccess, "NEEDS_RECOVERY_MODE");
                  childAccess.moveChildRecur(copyAccess, "CENTRAL_LOCKS");
                  childAccess.moveChildRecur(copyAccess, "SOURCE_FILE_TYPE");
               }
            } else {
               std::string compId = PropDevMgr::instance()->getComponentId();
               swu::setTextChild(copyAccess.getXml(), "NAME", compId);
               swu::setTextChild(copyAccess.getXml(), "NAME_DISPLAY", compId);
            }           
         }
            break;
         case XmlItemAccess::enType_Submodule:
            if (endir == XmlItemAccessIter::enLeave) {
               XmlItemAccess deviceAccess = copyAccess.getParent().getParent();			       
               if (std::string(deviceAccess.getName()) == "IMX" || std::string(deviceAccess.getName()) == "rcar3") {
                  XmlItemAccess moduleAccess = copyAccess.getParent();
                  copyAccess.changeTextFrom(moduleAccess, "NAME", "TYPE");
               }
               else {
                  copyAccess.changeTextFrom(deviceAccess, "NAME", "TYPE");
               }
               copyAccess.moveChildFrom(deviceAccess, "UPDATE_TIME");
               copyAccess.moveChildFrom(deviceAccess, "NEEDS_RECOVERY_MODE");
               copyAccess.moveChild(_refModule);
            }
            break;
      }
   }

   if (releasecopy) {
      delete releasecopy;
      releasecopy=0;
   }    
   ETG_TRACE_USR2(("manipulateInReleaseTag - END"));
}

tVoid UpdateXmlFilter::manipulateInOverallTag(TiXmlDocument* boschdoc) {
   ETG_TRACE_USR2(("manipulateInOverallTag - START"));

   TiXmlElement *overall = getOverallSection(boschdoc);
   
   swu::moveElem(overall->FirstChildElement("BUILD_LABEL"), _refModule);   
   swu::moveElem(overall->FirstChildElement("FINALNAME"), _refModule);
   swu::moveElem(overall->FirstChildElement("FINALNAME_CUSTOMER"), _refModule);
   swu::moveElem(overall->FirstChildElement("CISSWVERSIONNAME"), _refModule);
   swu::moveElem(overall->FirstChildElement("FCIDS"), _refModule);
   swu::moveElem(overall->FirstChildElement("EXTRA_ARGS"), _refModule);
  
   ETG_TRACE_USR2(("manipulateInOverallTag - END"));
}

tVoid UpdateXmlFilter::mandatoryTagInOverall(TiXmlDocument* boschdoc) {
   ETG_TRACE_USR2(("mandatoryTagInOverall - START"));

   /*
     FINALNAME              - It is required for validating in releaseList of configurator.
     allow_force_dnl        - It is required in configuratorHMIGeneric.
     allow_any_release      - It is required in configuratorHMIGeneric
   */

   TiXmlElement *overall = getOverallSection(boschdoc);

   BXmlAccess access(boschdoc);
   std::string finalName = access.getTextFromModule("FINALNAME");
   swu::setTextChild(overall, "FINALNAME", finalName);

   for(TiXmlElement* extraArgs = access.getModuleSection()->FirstChildElement("EXTRA_ARGS");
       extraArgs ; extraArgs = extraArgs->NextSiblingElement("EXTRA_ARGS")) {

      TiXmlElement* parentElem = 0;
      TiXmlElement* extraArg = extraArgs->FirstChildElement("EXTRA_ARG");				
      while (extraArg){
         std::string key = swu::getTextFromChildOrEmpty(extraArg, "KEY");
         ETG_TRACE_USR3(("Key - %s", key.c_str()));
         if (key == "allow_force_dnl" || key == "ALLOW_FORCE_DNL") {
	    	TiXmlNode *parentNode = extraArg->Parent();
            swu::moveElem(extraArg, overall);	
            parentElem = parentNode->ToElement();
            break;
         }
	 	extraArg = extraArg->NextSiblingElement("EXTRA_ARG");			
      }

      if(parentElem) {
         extraArg = parentElem->FirstChildElement("EXTRA_ARG");
         while (extraArg){
            std::string key = swu::getTextFromChildOrEmpty(extraArg, "KEY");
            ETG_TRACE_USR3(("Key - %s", key.c_str()));
            if (key == "allow_any_release" || key == "ALLOW_ANY_RELEASE") {
	      	swu::moveElem(extraArg, overall);
            break;
            }
            extraArg = extraArg->NextSiblingElement("EXTRA_ARG");
         }
      }
	  
	if(Config::instance()->cfg_AllowMisCmcRelease.get()){
		extraArg = extraArgs->FirstChildElement("EXTRA_ARG");
		while (extraArg){
			std::string key = swu::getTextFromChildOrEmpty(extraArg, "KEY");
		 	ETG_TRACE_USR3(("Key - %s", key.c_str()));
			if (key == "allow_mis_cmc_release" || key == "ALLOW_MIS_CMC_RELEASE") {
				swu::moveElem(extraArg, overall);				
				break;
            }
          extraArg = extraArg->NextSiblingElement("EXTRA_ARG");
        }
	  }	 
   }

   ETG_TRACE_USR2(("mandatoryTagInOverall - END"));
}

}

