#include <algorithm>  

#include <fstream>
#include "config/fcswupd_config.hpp"
#include "util/swu_util.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_bootChain.h"
#include "util/swu_execCommand.h"
#include "util/fcswupd_types.hpp"
#include "main/fcswupd_modVersionIf.h"
#include "main/fcswupd_systemData.h"


#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_modVersionIf.cpp.trc.h"
#endif

using namespace std;

namespace fcswupdate {

std::string InstalledModInfo::_installedDir="/var/opt/bosch/persistent/swupdate/installed/";
std::string InstalledModInfo::_activationsDir="/var/opt/bosch/persistent/swupdate/activations/";
InstalledModInfo::InstalledModInfo(std::string const &modName, bool doRead):
   _modName(modName),
   _valid(false) {
   _fileNameBase=_modName;

   std::transform(_fileNameBase.begin(), _fileNameBase.end(), _fileNameBase.begin(), ::tolower);
   std::replace( _fileNameBase.begin(), _fileNameBase.end(), '/', '_');
   if (doRead) {
      read();
   }
}

std::string InstalledModInfo::getVersion() {
   std::string version=_kvPairs["version"];
   if (version.empty()) {
      ETG_TRACE_USR1(("InstalledModInfo::getVersion(%50s): defaulting to unknown", _modName.c_str()));
      version = "unknown";
   }

   ETG_TRACE_USR1(("InstalledModInfo::getVersion(%50s): %s START",
                   _modName.c_str(), version.c_str()));
   return version;
}





bool InstalledModInfo::read() {
   _kvPairs.clear();
   _valid=false;
   ETG_TRACE_USR1(("InstalledModInfo::read(%50s): fileNameBase=%s START",
                   _modName.c_str(), _fileNameBase.c_str()));

   std::string instFile;
   std::string activationLink=_activationsDir + "/" + _fileNameBase;
   swu::trimString(activationLink);
   ETG_TRACE_USR2(("activationLink:%s", activationLink.c_str()));
   std::string chain="";
   // /var/opt/bosch/persistent/swupdate/activations/ivi_wcs_application
   if (swu::symlinkExists(activationLink)) {
      // link and link-target exist
      char buf[1024] = "";
      // get the link-target:
      size_t len=readlink(activationLink.c_str(), buf, sizeof(buf) -1);
      //Coverity fix for 146928
      if(len > 0){
           buf[len] = '\0';
      }
      else{
           ETG_TRACE_USR1(("readlink failed in fcswupd_modVersionIf file\n"));
      }
      // --> /var/opt/bosch/oem/ivi/wcs/application/1
      activationLink = std::string(buf);
      size_t index=activationLink.find_last_of('/');
      if (index == std::string::npos) {
         ETG_TRACE_USR1(("InstalledModInfo::could not parse activation-link: %s",
                         activationLink.c_str()));
         return false;
      }
      // chain="1"
      chain=activationLink.substr(index + 1);
      instFile=_installedDir + "/scomo/" + _fileNameBase + "_" + chain + ".txt";
   }
   else {
      ETG_TRACE_USR1(("activation Link is not present:%s",activationLink.c_str()));
      chain= (swu::BootChain::usedBootChain()==1  ? "1" : "2");
      instFile=_installedDir + "/" + _fileNameBase + "_" + chain + ".txt";
   }
   if (!swu::isFile(instFile)) {
      instFile=_installedDir + "/" + _fileNameBase + ".txt";
      if (!swu::isFile(instFile)) {
         ETG_TRACE_USR1(("InstalledModInfo::read(%100s): file does not exist: %s",
                         _modName.c_str(), instFile.c_str()));
         return false;
      }
   }

   ifstream inStream;
   inStream.open(instFile.c_str());
   if (!inStream.is_open()) {
      ETG_TRACE_ERR(("InstalledModInfo::read(%100s): coud not open: %s",
                     _modName.c_str(), instFile.c_str()));
      return false;
   }

   std::string line;
   while (getline(inStream, line)) {
      if (!line.size()) {
         continue;
      }
      ETG_TRACE_USR4(("getline() found:%s", line.c_str()));
      std::string key=line.substr(0, line.find(":"));
      swu::trimString(key);
      if (key.empty()) {
         continue;
      }
      std::string val=line;
      val.erase(0, val.find(":") + 1);
      if (val.empty()) {
         continue;
      }
      swu::trimString(val);
      ETG_TRACE_USR4(("InstalledModInfo add key=%30s val=%s", key.c_str(), val.c_str()));
      _kvPairs[key]=val;
   }
   _valid=true;

   return true;
}





class ModVersionIf;

std::string ModVersionIf::getVersion() {
   if (!_versionOk) {
      // only read if version is marked as not ok
      _version=readVersion();
      _versionOk=true;
   }

   ETG_TRACE_USR1(("ModVersionIf::getVersion(%30s):%s",
                   _name.c_str(), _version.c_str()));
   return _version;
}

std::string ModVersionIf::getBoschVersion() {
   if (!_boschVersionOk) {
      _boschVersion = readBoschVersion();
      _boschVersionOk=true;
   }
   ETG_TRACE_USR1(("ModVersionIf::getBoschVersion(%30s):%s",
                   _name.c_str(), _boschVersion.c_str()));
   return _boschVersion;
}


bool ModVersionIf::reg() {
   return SystemData::instance()->registerModVersionIf(this);
}

void ModVersionIf::dereg() {
   return SystemData::instance()->deregisterModVersionIf(this);
}


string ModVersionIf::fetchVersion(std::string processCmd)
{
   string fileName=_name;
   std::replace( fileName.begin(), fileName.end(), '/', '_');
   fileName=Config::instance()->cfg_SwuTmpDir.get() + "/" + fileName;
   std::string fetchVer;
   char verBuff[100];
   string processCmd_=processCmd + " > " + fileName;
   swu::execCommand(processCmd_.c_str());
	
   FILE *file = fopen(fileName.c_str(),"r");
   if(file){
      int res=fscanf(file,"%99s",verBuff);
      fclose(file);
      verBuff[99]=0;
      if (res >0 && res!=EOF) {
         fetchVer.assign(verBuff);
      }
   }
   ETG_TRACE_USR1(("ModVersionIf::fetchVersion(%30s):%s",
                   processCmd.c_str(), fetchVer.c_str()));

   return fetchVer;
}

ModVersionDefaultIf *ModVersionDefaultIf::create(string const &name) {
   ETG_TRACE_USR1(("ModVersionDefaultIf::create(%30s)",
                   name.c_str()));
   ModVersionDefaultIf *modIf=new ModVersionDefaultIf(name);
   return modIf;
}


ModVersionDefaultIf::ModVersionDefaultIf(string const &name): ModVersionIf(name),_modInfo(name)
{};



string ModVersionDefaultIf::readBoschVersion() {
   return _modInfo.getVersion();
}

void InstalledModInfo::setVal(string key, string val) {
      ETG_TRACE_USR1(("InstalledModInfo::setVal(%50s):%50s=%s", 
                      _modName.c_str(), key.c_str(), val.c_str()));
   _kvPairs[key]=val;
}

void InstalledModInfo::write() {
   ETG_TRACE_USR1(("InstalledModInfo::write(%50s)", 
                   _modName.c_str()));
   std::string instFile=_installedDir + "/" + _fileNameBase + ".txt";

   if(!swu::exists(_installedDir)) {
      swu::makeDirectoryRecursive(_installedDir, 0700);
   }

   ofstream outStream;
   outStream.open(instFile.c_str(),std::ofstream::out | std::ofstream::trunc);
   if (!outStream.is_open()) {
      ETG_TRACE_USR1(("InstalledModInfo::write(%50s): coud not open: %s",
                     _modName.c_str(), instFile.c_str()));
      return;
   }

   for (std::map<std::string, std::string>::iterator iter=_kvPairs.begin(); iter!=_kvPairs.end(); ++iter) {
      std::string line=(*iter).first + ":" + (*iter).second;
      ETG_TRACE_USR1(("InstalledModInfo::write():line=%s", line.c_str()));
      outStream << line.c_str() << "\n";
   }
   outStream.close();
   
}



}
