/*
 * fcswupd_history.cpp
 *
 *  Created on: Nov 14, 2013
 *      Author: efs1hi
 */

#include "sstream"
#include "util/swu_filesystem.h"
#include "util/swu_util.hpp"
#include "util/swu_constants.hpp"
#include "util/swu_robustFile.h"
#include "tinyxml/tinyxml.h"
#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "config/fcswupd_config.hpp"
#include "util/fcswupd_cust_update_info.hpp"
#include "main/fcswupd_component.h"
#include "main/fcswupd_srv.h"
#include "main/fcswupd_systemData.h"
#include "ctrl/fcswupd_ctrlXmlUtil.h"
#include "ctrl/fcswupd_ctrl.h"
#include "fcswupd_history.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_HISTORY_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#define ETG_I_FILE_PREFIX fcswupdate::History::instance()->
#include "trcGenProj/Header/fcswupd_history.cpp.trc.h"
#endif

//#define FORCE_PARTIAL_UPDATE 1

/* structure of history.xml
   "<HISTORYXML>"
   "<SPECIAL_RELEASES>"
   "<INITIAL/>"
   "<FIRST_OTA/>"
   "<FIRST_USB/>"
   "<INITIAL_AFTER_RESET/>"
   "<FIRST_OTA_AFTER_RESET/>"
   "<FIRST_USB_AFTER_RESET/>"
   "</SPECIAL_RELEASES>"
   "<RELEASES/>"
   "</HISTORYXML>";
*/

// root of history.xml, rest will be created on the fly
const char *INIT_XML = "<HISTORYXML>"
   "</HISTORYXML>";



namespace fcswupdate {

History::History()
   : _initialized(false),
     _doc(),
     _root(0),
     _releases(0),
     _specialReleases(0)
{
   ETG_TRACE_USR4(("History::CTOR"));
   _historyFileName.clear();
}

History::~History()
{
   ETG_TRACE_USR4(("History::DTOR"));

   _root = 0;
   _releases = 0;
   _specialReleases = 0;   
}

std::string History::getPersistentHistoryFileName()
{
   //return std::string("/var/opt/bosch/persistent/fcswupdate/history2.xml");
   return Config::getPersistentHistoryFileName();
}

tVoid History::setHistoryFileName(std::string FileName) {
   _historyFileName = FileName;
}

tVoid History::vInit()
{
   ETG_TRACE_USR1(("History::vInit START"));
   SWU_ASSERT_RETURN(!_initialized);

   if(_historyFileName.empty()) {
      _historyFileName = getPersistentHistoryFileName();
   }

   ETG_I_REGISTER_FILE();

   bool allOk = false;
   do {
      ETG_TRACE_COMP(("History::vInit(): file-name=%s", _historyFileName.c_str()));
      if (!swu::exists(_historyFileName)) {
         ETG_TRACE_COMP(("History::vInit(): file missing"));
         break;
      }
      swu::RobustFile rf(_historyFileName);
      SWU_ASSERT_RETURN(rf.init());
      if (!rf.restore()) {
         ETG_TRACE_COMP(("History::vInit(): restore failed"));
         rf.remove();
         break;

      }
      _doc.LoadFile(_historyFileName.c_str());

      if (!(_root = _doc.FirstChildElement("HISTORYXML"))) {
         ETG_TRACE_COMP(("History::vInit(): root node HISTORYXML missing"));
         break;
      }
      if (!swu::getUIntFromChild(_root, "HISTORY_VERSION")) {
         ETG_TRACE_COMP(("History::vInit(): wrong HISTORY_VERSION"));
         break;
      }

      if (!_root->FirstChildElement("RELEASES")) {
         ETG_TRACE_COMP(("History::vInit(): missing section RELEASES"));
         break;
      }

      if (!_root->FirstChildElement("SPECIAL_RELEASES")) {
         ETG_TRACE_COMP(("History::vInit(): missing section SPECIAL_RELEASES"));
         break;
      }
      allOk=true;
   } while (0);

   if (!allOk) {
      ETG_TRACE_COMP(("History::vInit(): setting default INIT_XML"));
      // something was not ok, create history-file from scratch
      _doc.Clear();
      TiXmlElement *root=new TiXmlElement("HISTORYXML");
      _doc.LinkEndChild(root);
      swu::setUIntChild(root, "OVERALL_COUNT", 0);
      swu::setUIntChild(root, "SINCE_FACTORY", 0);
      swu::setUIntChild(root, "HISTORY_VERSION", 1);
   }

   _root = _doc.FirstChildElement("HISTORYXML");
   _releases=getChildForced(_root, "RELEASES");
   _specialReleases=getChildForced(_root, "SPECIAL_RELEASES");

   if (!allOk) {
      (void)persist();
   }

   ETG_TRACE_COMP(("History initialized successfully"));
   _initialized = true;
   
}

void History::traceState()
{
   ETG_TRACE_COMP(("  _initialized = %u", _initialized));
}


void History::convertToInitialRelease(TiXmlElement *release, bool isReset) {
   ETG_TRACE_COMP(("History::convertToInitialRelease() START"));

   Config *cfg=Config::instance();
   if(isReset) {
      swu::setTextChild(release, "NAME", cfg->cfg_RunningSwVersionDisplay.get());
      swu::setTextChild(release, "FROMVERSION", "unknown");
      swu::setTextChild(release, "FROMVERSION_CUSTOMER", "unknown");
      swu::setTextChild(release, "FROMVERSION_LABEL", "unknown");
      swu::setTextChild(release, "TOVERSION", cfg->cfg_RunningSwTrain.get());
      swu::setTextChild(release, "TOVERSION_CUSTOMER", cfg->cfg_RunningCustomerVersion.get());
      swu::setTextChild(release, "TOVERSION_LABEL", cfg->cfg_RunningSwLabel.get());
      swu::setTextChild(release, "SUCCESS", "1");
      swu::setTextChild(release, "TYPE", "other");
      swu::setTextChild(release, "COMPLETED", "1");
      swu::setTextChild(release, "TOTAL_DISTANCE", "0");
      updateDeviceProductionTime();
      timestamp(release);
   }
   else
   {
      if(cfg->cfg_InitialProgrammingDateRequested.get())
      {
          updateDeviceProductionTime();
          timestamp(release);
      }
   }
   ETG_TRACE_COMP(("History::convertToInitialRelease() END"));
}

TiXmlElement *History::createInitialRelease() {
   ETG_TRACE_COMP(("History::createInitialRelease() START"));
        
   Config *cfg=Config::instance();
   TiXmlElement *release = new TiXmlElement("RELEASE");
   swu::setTextChild(release, "DNL_SIZE", "0");
   swu::setUIntChild(release, "INDEX", 0);

   ETG_TRACE_COMP(("History::createInitialRelease() END"));

   return release;
}


// delete data for factory-reset, bot some data are kept
/*
  keep all special releases.
  check INITIAL release is there, if not create forced one.
  copy XXX_AFTER_RESET with FIRST_USB, FIRST_OTA, INITIAL.
  clear all FIRST_USB, FIRST_OTA
  delete all releases
  update the counters.
*/
void History::reset()
{
   ETG_TRACE_USR1(("History::reset"));
   
   SWU_ASSERT_RETURN(_initialized);
   SWU_ASSERT_RETURN(_releases);
   SWU_ASSERT_RETURN(_specialReleases);
   
   TiXmlElement* initialRelease = getSpecialRelease("INITIAL");
   if(!initialRelease) {
      initialRelease=createInitialRelease();  
      convertToInitialRelease(initialRelease, true);    
      setSpecialRelease("INITIAL",initialRelease, true);
   }
   setSpecialRelease("INITIAL_AFTER_RESET",initialRelease, true);

   TiXmlElement* firstUsbRel = getSpecialRelease("FIRST_USB");
   if(firstUsbRel) {
      setSpecialRelease("FIRST_USB_AFTER_RESET",firstUsbRel->Clone()->ToElement(), true);
      setSpecialRelease("FIRST_USB", 0, true);
   }

   TiXmlElement* firstOtaRel = getSpecialRelease("FIRST_OTA");
   if(firstOtaRel) {
      setSpecialRelease("FIRST_OTA_AFTER_RESET",firstOtaRel->Clone()->ToElement(), true);
      setSpecialRelease("FIRST_OTA", 0, true);
   }
 
   _releases->Clear();
   swu::setUIntChild(_root, "OVERALL_COUNT", getNumUpdates() + 1);
   swu::setUIntChild(_root, "SINCE_FACTORY", getNumUpdatesAfterReset() + 1);
   (void)persist();
}


// delete all, not only factory reset
/*
  it is a bosch production reset.
  req number: 22144
  req from RN-AIVI: In case a "set to delivery state" (this is the last step in the BOSCH production process) is performed the   Software Update shall delete all items in the Update history except for the last one. The last item becomes the first item with the Number "0".
 */
void History::clear()
{
   ETG_TRACE_USR1(("History::clear"));

   SWU_ASSERT_RETURN(_initialized);
   SWU_ASSERT_RETURN(_releases);
   SWU_ASSERT_RETURN(_specialReleases);

   //clear all special releases.   
   setSpecialRelease("FIRST_USB", 0, true);
   setSpecialRelease("FIRST_USB_AFTER_RESET", 0, true);
   setSpecialRelease("FIRST_OTA", 0, true);
   setSpecialRelease("FIRST_OTA_AFTER_RESET", 0, true);
   setSpecialRelease("INITIAL", 0, true);
   setSpecialRelease("INITIAL_AFTER_RESET", 0, true);

   //store last releases as oth index means initial
   //somehow create the INITIAL release 
   TiXmlElement *lastRel=getLastRelease();  
   if(lastRel) {
      convertToInitialRelease(lastRel);
      swu::setUIntChild(lastRel, "INDEX", 0);
      setSpecialRelease("INITIAL",lastRel->Clone()->ToElement(), true);      
   } else {
      TiXmlElement* initialRelease = getSpecialRelease("INITIAL");
      if(!initialRelease) {
         initialRelease=createInitialRelease();
         convertToInitialRelease(initialRelease, true);
         setSpecialRelease("INITIAL",initialRelease, true);
      }
   }   

   _releases->Clear();
   swu::setUIntChild(_root, "OVERALL_COUNT", 1);
   swu::setUIntChild(_root, "SINCE_FACTORY", 1);
   
   (void)persist();
}

// TODO: Look up initial and first USB/OTA updates and find the latest if release history is empty
TiXmlElement *History::getLastRelease() const
{
   TiXmlElement *elem = 0;
   TiXmlNode *node = _releases->LastChild("RELEASE");
   if (node) {
      elem = node->ToElement();
   }
   return elem;
}


bool History::addRelease(CtrlProgressSection const *progressSection)
{
   ETG_TRACE_USR1(("History::addRelease START"));

   bool status = false;
	
   if(Config::instance()->cfg_AddReleaseToHistory.get())
   {
      SWU_ASSERT_RETURN_FALSE(_initialized);
      SWU_ASSERT_RETURN_FALSE(_releases);

      CtrlProgressSection *prgSection = const_cast<CtrlProgressSection *>(progressSection);
      TiXmlElement *progXml = const_cast<TiXmlElement *>(prgSection->getProgressReleaseSection());
      const TiXmlElement *updateSource = prgSection->getOverallSection()->FirstChildElement("SOURCE_TYPE");
      tenSourceType sourceType = updateSource ? (tenSourceType)atoi(updateSource->GetText()) : tenSourceType_other;

      TiXmlElement *xml = 0;
      const TiXmlElement *last = getLastRelease();
      // In case of partial update clone the last release to fill in missing components
#if FORCE_PARTIAL_UPDATE
      if (last)
#else
         if (!progressSection->isFullUpdate() && last)
#endif
         {
            ETG_TRACE_USR1(("History::addRelease: Clone last update"));
            xml = last->Clone()->ToElement();
         }
         else {
            xml = copyBxmlSkeleton(progXml);
         }
      swu::ScopedPointer<TiXmlElement> scoped_xml(xml);
      SWU_ASSERT_RETURN_FALSE(xml);

      TiXmlElement *release = 0;
      tUInt relSize = swu::getUIntFromChild(progXml, "DNL_SIZE", 0);

      for (XmlItemAccessIter dstIter(xml, XmlItemAccessIter::enOnEnter); !dstIter.isDone(); dstIter.next())
      {

         XmlItemAccess dstAccess(dstIter);
         TiXmlElement *elem = getBoschXmlElem(progXml, dstAccess.getLongName());
         if (!elem) {
            continue; // partial update: element is not present in progressSection
         }
         BoschXmlNode srcAccess(elem);
         // In case of partial update replace already cloned elements
         switch (srcAccess.enGetType())
         {
            case  XmlItemAccess::enType_Release:
            {
               release = dstAccess.getXml();
               swu::setUIntChild(release, "INDEX", getNumUpdates()+1);

               swu::setTextChild(release, "FROMVERSION", 
                                 swu::getTextFromChild(srcAccess.getXml(), "FROMVERSION", false, "unknown"));
               swu::setTextChild(release, "FROMVERSION_CUSTOMER", 
                                 swu::getTextFromChild(srcAccess.getXml(), "FROMVERSION_CUSTOMER", false, "unknown"));
               swu::setTextChild(release, "FROMVERSION_LABEL", 
                                 swu::getTextFromChild(srcAccess.getXml(), "FROMVERSION_LABEL", false, "unknown"));
               swu::setTextChild(release, "TOVERSION", 
                                 swu::getTextFromChild(srcAccess.getXml(), "TOVERSION", false, "unknown"));
               swu::setTextChild(release, "TOVERSION_CUSTOMER", 
                                 swu::getTextFromChild(srcAccess.getXml(), "TOVERSION_CUSTOMER", false, "unknown"));
               swu::setTextChild(release, "TOVERSION_LABEL", 
                                 swu::getTextFromChild(srcAccess.getXml(), "TOVERSION_LABEL", false, "unknown"));
               swu::setTextChild(release, "MUVERSION", swu::getTextFromChild(srcAccess.getXml(), "MUVERSION", false, "unknown"));

               //                                swu::setUIntChild(release, "SUCCESS", prgSection->isUpdateOK() ? 1 : 0);
               swu::setUIntChild(release, "SUCCESS", swu::getUIntFromChild(srcAccess.getXml(), "ALL_OK"));

               swu::setUIntChild(release, "DNL_SIZE", relSize);
               swu::setTextChild(release, "TYPE", sourceTypeToString(sourceType));
               swu::setTextChild(release, "COMPLETED", "1");

               TiXmlNode *e = release->FirstChild("UTC");
               if (e) {
                  release->RemoveChild(e);
               }
               SWU_ASSERT_RETURN_FALSE(timestamp(release));

               break;
            }

            case  XmlItemAccess::enType_Submodule:
            {
               swu::setTextChild(dstAccess.getXml(), "VERSION" , 
                                 swu::getTextFromChild(srcAccess.getXml(), "VERSION", false, "unknown"));
               swu::setTextChild(dstAccess.getXml(), "FROMVERSION" , 
                                 swu::getTextFromChild(srcAccess.getXml(), "FROMVERSION", false, "unknown"));
                           
               swu::setTextChild(dstAccess.getXml(), "RESULT", 
                                 swu::Constants::XML::update_progress_state__not_affected);
               swu::setTextChild(dstAccess.getXml(), "LAST_AFFECTED_RESULT", 
                                 swu::Constants::XML::update_progress_state__not_affected);
               /* get LAST_AFFECTED_RESULT from previous version(s)" */
               TiXmlElement *itemsItem=srcAccess.getItemsItem();
               TiXmlElement *item=0;
               if (itemsItem) {
                  item = srcAccess.getItemsItem()->FirstChildElement("ITEM");
                  while (item && item->NextSiblingElement("ITEM")) {
                     item=item->NextSiblingElement("ITEM");
                  }
               }
               if (item)
               {
                  swu::setTextChild(dstAccess.getXml(), "RESULT", 
                                    swu::getTextFromChild(item, "STATE", false, swu::Constants::XML::update_progress_state__not_affected.c_str()));
                  swu::setTextChild(dstAccess.getXml(), "LAST_AFFECTED_RESULT", 
                                    swu::getTextFromChild(item, "STATE", false, swu::Constants::XML::update_progress_state__not_affected.c_str()));
               }
               swu::setTextChild(dstAccess.getXml(), "NAME_DISPLAY", 
                                 swu::getTextFromChild(srcAccess.getXml(), "NAME_DISPLAY", false, "unknown"));
               break;
            }
            case  XmlItemAccess::enType_Device:
            case  XmlItemAccess::enType_Module:
            {
               swu::setTextChild(dstAccess.getXml(), "NAME_DISPLAY", 
                                 swu::getTextFromChild(srcAccess.getXml(), "NAME_DISPLAY", false, "unknown"));

               break;
            }

            default:
               break;
         }
      }

      SWU_ASSERT_RETURN_FALSE(release);
      swu::setUIntChild(release, "TOTAL_DISTANCE", Config::instance()->cfg_UpdateStartDistanceTotalizer.get());
      swu::setUIntChild(_root, "OVERALL_COUNT", getNumUpdates() + 1);      

      if (sourceType == tenSourceType_USB)
      {
         TiXmlElement* firstUsbRel = getChildForced(_specialReleases, "FIRST_USB")->FirstChildElement("RELEASE");
         if(!firstUsbRel) {
            setSpecialRelease("FIRST_USB", xml);
         }
      }
      else if (sourceType == tenSourceType_OTA)
      {
         TiXmlElement* firstOtaRel = getChildForced(_specialReleases, "FIRST_OTA")->FirstChildElement("RELEASE");         
         if(!firstOtaRel) {
            setSpecialRelease("FIRST_OTA", xml);
         }
      }

      _releases->LinkEndChild(xml);
      scoped_xml.release();

      if (getNumReleases() >= k_MAX_ENTRIES)
      {
         _releases->RemoveChild(_releases->FirstChildElement("RELEASE"));
         ETG_TRACE_USR1(("History truncated"));
      }

      status = persist();
	 
   }
   else
   {
      ETG_TRACE_USR1(("History::addRelease cfg_AddReleaseToHistory is set to false"));
   }
   ETG_TRACE_USR1(("History::addRelease END"));

   return status;
}

TiXmlElement *History::getChildForced(TiXmlElement *parent, std::string childName) {
   ETG_TRACE_USR1(("History::getChildForced: child==%s", childName.c_str()));
   TiXmlElement *child=parent->FirstChildElement(childName.c_str());
   if (!child) {
      ETG_TRACE_USR1(("History::getChildForced: create!"));
      child=new TiXmlElement(childName.c_str());
      parent->LinkEndChild(child);
   }
   return child;
}

void History::setSpecialRelease(std::string relType, TiXmlElement *release, bool force) {
   // get root-node of special
   ETG_TRACE_USR1(("History::setSpecialRelease: release=%p force=%u type=%s", 
                   release, force, relType.c_str()));
   TiXmlElement *specialRels=getChildForced(_root, "SPECIAL_RELEASES");
   TiXmlElement *specialRel=getChildForced(specialRels, relType);
   TiXmlElement *rel=specialRel->FirstChildElement("RELEASE");
   ETG_TRACE_USR1(("History::setSpecialRelease: specialRels=%p specialRel=%p rel=%p",
                   specialRels, specialRel, rel));

   if (force && rel) {
      ETG_TRACE_USR1(("History::setSpecialRelease: remove existing (force)"));
      specialRel->Clear();
      rel=0;
   }

   if (!rel && release) {
      ETG_TRACE_USR1(("History::setSpecialRelease: add new"));
      specialRel->InsertEndChild(*release);
   }
}



bool History::persist()
{
   swu::RobustFile rf(getPersistentHistoryFileName());
   ETG_TRACE_USR1(("History::persist(): %s", getPersistentHistoryFileName().c_str()));
   SWU_ASSERT_RETURN_FALSE(rf.init());
   bool status = _doc.SaveFile(getPersistentHistoryFileName().c_str());
   SWU_ASSERT_RETURN_FALSE(rf.sync());

   return status;
}


int History::getNumReleases() const
{
   return swu::getNumChildren(_releases, "RELEASE");
}

int History::getNumUpdates() const
{
   return swu::getUIntFromChild(_root, "OVERALL_COUNT", 0);
}

int History::getNumUpdatesAfterReset() const
{
   return swu::getUIntFromChild(_root, "SINCE_FACTORY", 0);
}

TiXmlElement *History::getSpecialRelease(std::string relType)
{
   ETG_TRACE_USR1(("History::getSpecialRelease START (relType=%s)", relType.c_str()));   
   TiXmlElement *specRelease=getChildForced(_specialReleases, relType);
   return specRelease->FirstChildElement("RELEASE");
}


bool History::store(const char *path, bool sinceFactoryReset)
{
   ETG_TRACE_USR1(("History::store() START sinceReset=%u path=%s", sinceFactoryReset, path));

   SWU_ASSERT_RETURN_FALSE(_initialized);

   TiXmlDocument *doc = new TiXmlDocument(_doc);
   swu::ScopedPointer<TiXmlDocument> scoped_doc(doc);
   SWU_ASSERT_RETURN_FALSE(doc);

   if (sinceFactoryReset)
   {
      TiXmlElement *root = doc->RootElement();
      SWU_ASSERT_RETURN_FALSE(root);
      root->RemoveChild(root->FirstChildElement("SPECIAL_RELEASES"));
      root->RemoveChild(root->FirstChildElement("OVERALL_COUNT"));
   }

   bool status = doc->SaveFile(path);

   ETG_TRACE_USR4(("History::store END (status %u)", (int)status));
   return status;
}

bool History::timestamp(TiXmlElement *release)
{
   ETG_TRACE_USR4(("History::timestamp START"));

   swu::LocalTime utc(fcswupdate::Config::instance()->cfg_UpdateStartTime.get());
   TiXmlNode *e = utc.toXml();
   if (e) {
      swu::removeChildren(release, "UTC");
      release->LinkEndChild(e);
   }
   ETG_TRACE_USR4(("History::timestamp END"));
   return e ? true:false;
}

void History::updateDeviceProductionTime() {

   ETG_TRACE_USR4(("History::updateDeviceProductionTime START"));
   Config *cfg=Config::instance();
   std::vector<tU8> vecDate=cfg->cfg_InitialProgrammingDate.get();

   swu::LocalTime lc(swu::bcdToU8(vecDate[0]) + 2000,
                     swu::bcdToU8(vecDate[1]),
                     swu::bcdToU8(vecDate[2]),
                     swu::bcdToU8(vecDate[3]),
                     swu::bcdToU8(vecDate[4]),
                     swu::bcdToU8(vecDate[5]));

   fcswupdate::Config::instance()->cfg_UpdateStartTime.set(lc.toString());
   ETG_TRACE_USR4(("History::updateDeviceProductionTime END"));

}

}

