#ifndef FCSWUPD_CTRL_PROGRESS_H
#define FCSWUPD_CTRL_PROGRESS_H

#include <string>
#include "util/fcswupd_types.hpp"
#include "util/swu_util.hpp"
#include "util/swu_xmlDoc.h"
#include "util/swu_constants.hpp"
#include "util/swu_singleton.hpp"
#include "util/swu_memberList.hpp"
#include "config/fcswupd_config.hpp"
#include "util/swu_types.h"
#include "util/fcswupd_bxml.h"
#include "util/swu_localtime.hpp"
#include "ctrl/fcswupd_ctrlXmlUtil.h"

namespace fcswupdate {
struct Msg_CurrentBoschXmlChanged;
class CurrentBoschXml;
/* configuration that is fetched on start of update and stored persistently in progress-section if needed
   configuration stays valid for whole update, also when we change to recovery-mode
 */
class CtrlPersistentConfig:public swu::XmlItemContainer {
public:
   CtrlPersistentConfig():
      usePersistency(this, "USE_PERSISTENCY", Config::instance()->cfg_CtrlCreateWithPersistentProgress.get()),
      enInitialMode(this, "INITIAL_MODE", tenSwUpdateMode_Invalid),
      enMode(this, "UPDATE_MODE", tenSwUpdateMode_Invalid),
      needsResultOkInHmiMode(this, "NEEDS_RESULT_IN_HMI_MODE", false),
      needsResultAfterActivation(this, "NEEDS_RESULT_AFTER_ACTIVATION", false),
#if 1
      needsPreResult(this, "NEEDS_PRE_RESULT", false),
      forceShutdown(this, "FORCE_SHUTDOWN", Config::instance()->cfg_DoShutdownAfterUpdate.readAsBool()),
      waitForStickRemovedOnSuccess(this, "WAIT_STICK_REMOVED_ON_SUCCESS", false),
      needsUpdateLock(this, "NEEDS_UPDATE_LOCK", false),
      maxWaitTimeForStickRemovedSec(this, "WAIT_STICK_REMOVED_SECONDS", 0),
      waitForStickRemovedLine1(this, "WAIT_STICK_REMOVED_MSG1", "Please wait for the system to shutdown"),
      waitForStickRemovedLine2(this, "WAIT_STICK_REMOVED_MSG2", "System will Reboot once the Update Media is removed"),
      maxModTries(this, "MAX_MOD_TRIES", Config::instance()->cfg_CtrlMaxModRetries.get())
#endif

   {}

   swu::XmlItem<tUInt, bool> usePersistency;
   swu::XmlItem<tUInt, tenSwUpdateMode> enInitialMode;
   swu::XmlItem<tUInt, tenSwUpdateMode> enMode;
   swu::XmlItem<tUInt, bool> needsResultOkInHmiMode;
   swu::XmlItem<tUInt, bool> needsResultAfterActivation;
#if 1
   swu::XmlItem<tUInt, bool> needsPreResult; // maybe this is identical with needsResultAfterActivation
   swu::XmlItem<tUInt, bool> forceShutdown;
   swu::XmlItem<tUInt, bool> waitForStickRemovedOnSuccess;
   swu::XmlItem<tUInt, bool> needsUpdateLock;
   swu::XmlItem<tUInt> maxWaitTimeForStickRemovedSec; // recovery: Config::instance()->cfg_CtrlRecoveryResultSupervisionSec.get()
   swu::XmlItem<std::string> waitForStickRemovedLine1;  // recovery
   swu::XmlItem<std::string> waitForStickRemovedLine2; // recovery
   swu::XmlItem<tUInt> maxModTries;

#endif

};


/* configuration that is fetched on startup
   configuration stays valid for this power-cycle.
 */
struct CtrlStaticConfig {
   CtrlStaticConfig():
      needsSpmSwUpdateMode(false), // parameter for resource-manager
      interactOnError(false),
      interactOnResult(false),
      defaultDecision(update_decision_setting_type__force)
   {
   }
   // next not needed, all adapters should use Config directly
   bool needsSpmSwUpdateMode;
   bool interactOnError;
   bool interactOnResult;
   update_decision_setting_type defaultDecision;
};

struct Msg_ProgressTimer {
};

// message we will broadcast, when progress-section has changed.
struct Msg_ProgressSectionChanged: public swu::Msg< Msg_ProgressSectionChanged, FcSwUpdRoot > {
   virtual tVoid vTrace(){};
};



class CtrlProgressSection;

class CtrlProgItem {
   friend class CtrlProgressSection;
private:
   bool _valid;   
   CtrlProgressSection *_progressSection;
   // MOD can be RELEASE, DEVICE, MODULE, SUBMODULE
   TiXmlElement *_progressItem; // <MOD><INFO><ITEMS><ITEM>
   TiXmlElement *_progBoschXmlElem; // <MOD>
   std::string _name; // "IMX/MAIN/APPLICATION"
   std::string _itemType; // APP or PREPARE or FINALIZE
   tUInt _subItemIndexMax;
   
public:

   CtrlProgItem(CtrlProgressSection *progressSection, TiXmlElement *progressItem);
   CtrlProgItem():_valid(false), _progressSection(NULL), _progressItem(NULL), _progBoschXmlElem(NULL), _subItemIndexMax(0) {};  //Coverity fix for 17707

   // defined actions
   void start();
   void finalize(bool success);
   void scheduleRetry() {
      setState("pending");
   }

   // accessors
   std::string const &getName() const {
      return _name;
   }

   std::string getVersion() {
      return swu::getTextFromChild(_progBoschXmlElem, "VERSION", false, "unknown");;
   }

   std::string getFromVersion() {
      return swu::getTextFromChild(_progBoschXmlElem, "FROMVERSION", false, "unknown");
   }
  
   /* childName
    LOCK_STATE
    */
   void setItemUIntChild(std::string childName, tUInt val) {
      swu::setUIntChild(_progBoschXmlElem, childName.c_str(), val);
   }

   bool isItemChildValid(std::string childName) {
      return swu::getUIntFromChild(_progBoschXmlElem, childName.c_str()) ? true : false;
   }

   
   tUInt getUpdateTime() const;

   std::string getRefKey() const {
      std::string res=_name + ":" + _itemType +":" + swu::getTextFromChild(_progressItem, "INDEX", false, "0");
      return res ;
   }

   std::string const &getType() const {
      return _itemType;
   }

   XmlItemAccess::tenType getLevel() {
      SWU_ASSERT_RETURN_VAL(isValid(), XmlItemAccess::enType_Invalid);
      XmlItemAccess cur(_progBoschXmlElem);
      return cur.enGetType();
   }

   std::string getState() const {
      SWU_ASSERT_RETURN_VAL(_progressItem, "");
      return swu::getTextFromChild(_progressItem, "STATE", true, "");  //Coverity fix for 69978
   }

   void setScomoPhaseDone();
   tUInt getScomoPhaseDone() const {
      SWU_ASSERT_RETURN_VAL(_progressItem, 0);
      return swu::getUIntFromChild(_progressItem, "SCOMO_PHASE_DONE");
   }


   tUInt getSubItemIndex() const {
      SWU_ASSERT_RETURN_VAL(isValid(), 0);
      return swu::getUIntFromChild(_progressItem, "INDEX", 0);
   }

   tUInt getNumSubItems() const {
      return _subItemIndexMax + 1;
   }

   bool isValid() const;

   bool needsRecoveryMode();   

   TiXmlElement *getProgBoschXmlItem();

   std::string getScriptName();

   std::string getFileNameAndPath(TiXmlElement *elem);
   std::string getToolsPath() {
      return getFileNameAndPath(getToolsXml());
   }

   std::string getDataPath() {
      SWU_ASSERT_RETURN_VAL(isValid(), "");
      if (_itemType=="APP") {         
         return getFileNameAndPath(swu::getNthChild(getRelXmlAccess().getXml()->FirstChildElement("FLASH_ITEMS"), "FLASH_ITEM", getSubItemIndex()));
      }
      else {
         return getToolsPath();
      }
   }

   TiXmlElement *getDataChecksum() {
      SWU_ASSERT_RETURN_VAL(isValid(), 0);
      if (_itemType=="APP") {
         TiXmlElement *item=swu::getNthChild(getRelXmlAccess().getXml()->FirstChildElement("FLASH_ITEMS"), "FLASH_ITEM", getSubItemIndex());
         return item->FirstChildElement("CKSUM");
      }
      else {
         return getToolsChecksum();
      }
   }

   TiXmlElement *getToolsChecksum() {
      SWU_ASSERT_RETURN_VAL(isValid(), 0);
      return getToolsXml()->FirstChildElement("CKSUM");
   }

   std::string getNameOfLevel(XmlItemAccess::tenType level) {
      SWU_ASSERT_RETURN_VAL(isValid(), "");
      XmlItemAccess cur(_progBoschXmlElem);
      if ((tU32)cur.enGetType()<(tU32)level) {
         return "";
      }
      while ((tU32)cur.enGetType()>(tU32)level) {
         cur=cur.getParent() ;
      }
      return cur.getName();
   }

   std::string getLongNameOfLevel(XmlItemAccess::tenType level) {
      SWU_ASSERT_RETURN_VAL(isValid(), "");
      XmlItemAccess cur(_progBoschXmlElem);
      if ((tU32)cur.enGetType()<(tU32)level) {
         return "";
      }
      while ((tU32)cur.enGetType()>(tU32)level) {
         cur=cur.getParent() ;
      }
      return cur.getLongName();
   }
      
   std::string getSourceFileType();
   std::string getDataFileDestOffset();
   std::string getDataFileSize();

   void getExtraArgs(std::map<std::string, std::string> &kvPairs);

   bool needsPostInstall();
   std::vector<std::string> getRequiredLocks();


private:
   XmlItemAccess getRelXmlAccess();
   XmlItemAccess getProgressRelXmlAccess();

   bool needsRecoveryMode(XmlItemAccess &access);
   void needsRescheduling(XmlItemAccess &access);

   bool needsCurrentScomoPhase();
   bool needsScomoPhase(tUInt scomoPhase); // todo:implementation
   std::string getConfiguredScomoPhases();
   static  bool flashOnEnter(std::string const &type) {
      return (type!="FINALIZE");
   }

   bool flashOnEnter() {
      return (getType()!="FINALIZE");
   }

   TiXmlElement *getProgressItem() {
      SWU_ASSERT_RETURN_VAL(isValid(), 0);
      return _progressItem;
   }

   void setState(std::string state);

   TiXmlElement *getToolsXml();
   std::string getReleasePath();

   bool isPhaseStart();
};


class ProgressFile: public swu::XmlFile {
public:
   virtual ~ProgressFile();
   ProgressFile(std::string filename);
};

class CtrlProgressSection : public swu::Member < FcSwUpdRoot >
{
   friend class CtrlProgItem;
public:
   class Statistics {

   public:
      Statistics(CtrlProgressSection *progressSection):
         _progressSection(progressSection),
         _bPhaseStarted(false),
           _bPhaseEnded(false)
      {}

      struct Times {
         tUInt tAll;
         tUInt tDone;
         Times():tAll(0),tDone(0) {};
         bool operator!=(Times const &rhs) const {
            return ((tAll != rhs.tAll) ||
                    (tDone != rhs.tDone));
         }
      };
      typedef enum {
         enAll,
         enPhase,
         enCurrent
      } tenScope;

      struct Counters {
         tUInt numPending;
         tUInt numRunning;
         tUInt numFailed;
         tUInt numOk;
         tUInt numRetries;
         tUInt numAll;
         Times _times;
         Times _timesPhase;
         Times _timesCurrent;

         Counters()
         {
            clear();
         };

         static const char *scopeToCStr(tenScope scope) {
            switch (scope) {
               case enAll: return "enAll";
               case enPhase: return "enPhase";
               case enCurrent: return "enCurrent";
               default: return "invalid";
            }
         }

         Times const &getTimes(tenScope enScope) const {
            return const_cast<Counters *>(this)->getTimesMutable(enScope);
         }
         Times &getTimesMutable(tenScope enScope) {
            switch (enScope) {
               case enPhase:
                  return _timesPhase;
               case enCurrent:
                  return _timesCurrent;
               case enAll:
               default:
                  return _times;
            }
         }


         float getFraction(tenScope enScope) const;
         tUInt getPercent(tenScope enScope) const {
            return (tUInt)round(getFraction(enScope) * 100);
         };

         void clear() {
            numPending=0;
            numRunning=0;
            numFailed=0;
            numOk=0;
            numRetries=0;
            numAll=0;
            _times=Times();
            _timesPhase=Times();
            _timesCurrent=Times();
         }

         bool operator!=(Counters const &rhs) const {
            return (
                    (numPending != rhs.numPending) ||
                    (numRunning != rhs.numRunning) ||
                    (numFailed  != rhs.numFailed) ||
                    (numOk      != rhs.numOk) ||
                    (numAll      != rhs.numAll) ||
                    (_times      != rhs._times) ||
                    (_timesPhase      != rhs._timesPhase) ||
                    (_timesCurrent      != rhs._timesCurrent)
                    );
         }
      };

      Counters const &get() const {
         return _counters;
      }

      void clear() {
         _bPhaseStarted=false;
         _bPhaseEnded=false;
         _counters.clear();
      }
      void add( CtrlProgItem &progItem);

      void print(const char *action="PRINT");
      void onTimeout();

   private:
      void addTimes(tenScope enScope, CtrlProgItem const &progItem, bool bPhaseStarted, bool bPhaseEnded);

      CtrlProgressSection *_progressSection;
      Counters _counters;
      bool _bPhaseStarted;
      bool _bPhaseEnded;
      tUInt getUpdateTimerSec() {
         return 10;
      }
   }; // class Statistics



private:
   std::set<std::string> _changedGroups; // pass also to sw-update-core
   ProgressFile *_progressFile; // robust-file holding the xml-progress-section
   TiXmlElement *_progressRoot;
   TiXmlElement *_progressSection;
   TiXmlElement *_progressReleaseSection;
   TiXmlElement *_overallSection;
   TiXmlElement *_releaseSection;
   TiXmlElement *_decisionsSection;
   CurrentBoschXml *_currentBoschXmlStick;
   CtrlProgItem _currentCtrlProgItem;
   update_decision_setting_type _defaultDecision;

   Statistics _statistics;
   std::string _lastStartedRefKey;
   tUInt _curRetries; 
   bool _isCompatUpdate;
   bool _abort;
   bool _silent;
   bool _recoveryNeeded;
   bool _rescheduleNeeded;

   static bool _doHandleRecoveryMode;
   std::string _startingItemRefKey;
   CtrlPersistentConfig *_persistentConfig;
   CtrlStaticConfig *_staticConfig;
   CtrlPersistentConfig *_persistentConfigAlloc;
   CtrlStaticConfig *_staticConfigAlloc;


public:

   CtrlProgressSection(std::string progressFileName);
   virtual ~CtrlProgressSection();
  
   Statistics::Counters const &getStatistics() {
      return _statistics.get();
   }

   tUInt getCurRetries() {
      return _curRetries;
   }

   void setPersistentConfig(CtrlPersistentConfig &config);
   void setStaticConfig(CtrlStaticConfig &config);
   CtrlPersistentConfig const &setPersistentConfig();
   CtrlStaticConfig const &setStaticConfig();

   tUInt getMaxScomoPhase() const {
      return 3;
   }
   char const *getMaxScomoPhaseStr() const {
      return "3";
   }

   TiXmlElement const *getOverallSection(bool doAssert=true) {
      if (doAssert) {
         SWU_ASSERT_RETURN_VAL(_overallSection, 0);
      }
      return _overallSection;
   }
   typedef enum {
      enState_Initial,
      enState_Restoring,
      enState_WaitBoschXml, //todo: this state is not used.
      enState_WaitStart,
      enState_Running,
      enState_ScomoPending,
      enState_Reboot, // todo    
      enState_Result,     
      enState_Done,     
   } tenState;
   tenState _enState;

   virtual tVoid traceState();

   TiXmlElement *getReleaseSection() {
      return _releaseSection;
   }
   TiXmlElement *getProgressReleaseSection() const {
      return _progressReleaseSection;
   }

public:
   bool isScomoMode() {
      return (Config::instance()->cfg_EnterFullOperation.get() &&
              (getInitialMode()==tenSwUpdateMode_Scomo));
   }

   bool isScomoModeNormal() {
      return isScomoMode() && !isFirmwareComponentId();
   }
   bool isScomoModeFirmware() {
      return isScomoMode() && isFirmwareComponentId();
   }

   bool isFirmwareComponentId() {
      return (getComponentId() == "FIRMWARE");
   }

   tU32 getRequestedScomoPhase();
   bool isLastScomoPhase();
   std::string getComponentId(); 
   std::string getSubModuleListConfiguredToUpdate();
  
private:
   void setRequestedScomoPhase(tU32 phase);
   void setComponentId(std::string compId);

   void setPersistentState(std::string newState) {
      swu::setTextChild(_progressSection, "STATE", newState);
   }
   void setSubModuleListConfiguredToUpdate(std::string subModuleName);

public:
   // methods that can be used on progress-section after call of restore(bool peekOnly=true)
   // once there is an error, this goes to false, currently no chance to resume
   bool isUpdateOK() {
      return swu::getUIntFromChild(_progressSection, "ALL_OK") ? true : false;            
   }
   // stored state "pending", "running", "result", "scomoPending"
   std::string getPersistentState() {
      return swu::getTextFromChild(_progressSection, "STATE", false, "unknown");
   }
      
   tenSwUpdateMode getInitialMode() const;

   void storePhaseStart();

   bool isCurItem(CtrlProgItem const *item);

   std::string getProgXmlState() {
      if (!_progressSection) {
         return "INVALID";
      }
      return swu::getTextFromChild(_progressSection, "STATE", false, "INVALID");
   }

   std::string getSpecialMode() {
      if (!_progressReleaseSection) {
         return "";
      }
      return swu::getTextFromChildOrEmpty(_progressReleaseSection, "SPECIAL_MODE");
   }
   
   void setPostInstallMode(bool doSet=true);

   bool isInPostInstallMode();


   void setSpecialMode(std::string specialMode) {
      if (!_progressReleaseSection) {
         return;
      }
      swu::setTextChild(_progressReleaseSection, "SPECIAL_MODE", specialMode);
   }

   bool allOk() {
      return _statistics.get().numFailed ? false : true;
   }


   bool retry() {
      _enState=enState_Reboot;
   }

   std::string getToVersion() {
         return swu::getTextFromChild(_progressReleaseSection, "TOVERSION", false, "unknown");         
   }

   std::string getToLabel() {
         return swu::getTextFromChild(_progressReleaseSection, "TOVERSION_LABEL", false, "unknown");         
   }

   std::string getToVersionCustomer() {
         return swu::getTextFromChild(_progressReleaseSection, "TOVERSION_CUSTOMER", false, "unknown");         
   }

   std::string getFinalName() {
      if (Config::instance()->cfg_UseReleaseNameFromRunningSwLabel.get()) {
         return swu::getTextFromChild(_progressReleaseSection, "TOVERSION_LABEL", false, "unknown");
      } else {
         return swu::getTextFromChild(_progressReleaseSection, "TOVERSION", false, "unknown");         
      }
   }

   std::string getFinalNameCustomer() {
      std::string name = swu::getTextFromChild(_progressReleaseSection, "TOVERSION_CUSTOMER", false, "");
	  if (name == "" || name == "unknown") {
		return getFinalName();
	  }
	  return name;
   }
   
   tUInt getBxmlVersion() {      
      return swu::getUIntFromChild(_releaseSection->FirstChildElement("DEVICE"), "BXML_VERSION");
   }

   bool switchBootChainOnDone() {
      return swu::getUIntFromChild(_progressReleaseSection, "SWITCH_BOOTCHAIN");
   }


   void create(TiXmlDocument *boschXmlDocIn, CtrlPersistentConfig *_persitentConfig, CtrlStaticConfig *_staticConfig);
   bool restore(bool peekOnly=false, CtrlPersistentConfig *_persitentConfig=0, CtrlStaticConfig *_staticConfig=0);
   // tell progress-section, that we start now (public)
   bool startUpdate();
   TiXmlElement * collectProgItems();
   void evalProgXmlStates();
   tVoid store();
   
   bool reOpen() {
      _enState=enState_Reboot;
      return true;
   }

   CtrlProgItem peekNextItem() {
      return peekNextItem(_currentCtrlProgItem);
   }
   
   CtrlProgItem peekNextItem(CtrlProgItem curItem);
   CtrlProgItem setNextItem(bool firstItem = false);
   
   // get the current item
   CtrlProgItem currentItem() {
      return _currentCtrlProgItem;
   }

   bool requestFullUpdate();
   void setRecoveryNeeded() {
      _recoveryNeeded=true;
   }
   void setReschedulesNeeded() {
      _rescheduleNeeded=true;
   }
   bool needsSwitchToRecovery();
   static void handleRecoveryMode(bool doHandle);
   
   bool isUpdateDone();
   void setUpdateDone(unsigned int updateResult);

   void finalizeRelease();
   bool isStoredToHistory();
   void setIsStoredToHistory();
   
 
   void remove() {
      if(exists()) {
         _progressFile->remove();
      }
      enterState(enState_Done);
   }

   void storeProgressFile();

   bool exists();   

   // query state of update
   std::string getUpdateState() {
      if (!_progressSection) {
         return "IDLE";
      }
      return swu::getTextFromChild(_progressSection, "STATE");
   }

   void vProcess(Msg_CurrentBoschXmlChanged *msg);
   tVoid vProcess(Msg_ProgressTimer *pMsg) {
      _statistics.onTimeout();
   }

   tenState getState();

   tenSwUpdateMode getUpdateMode() {
      if (_enState==enState_Initial) {
         return  tenSwUpdateMode_Invalid;
      }
      return (tenSwUpdateMode) atoi(swu::getTextFromChild(_progressSection, "UPDATE_MODE", false));
   }

   void skip(XmlItemAccess::tenType level);
   void abort();

   bool isLastOf(XmlItemAccess::tenType level); // todo: should also be defined for class CtrlProgItem
   bool isCompatUpdate() const;
   bool isPostImageMode() const;

   bool isPartialRelease() const;
   bool isFullUpdate() const; 
   bool isFirmwareUpdate() const {
      return swu::getUIntFromChild(_progressReleaseSection, "FIRMWARE_UPDATE");
   }

   std::string getMediaPath();
   tenSourceType getSourceType();
   bool isOtaUpdate();

   void startProgressTimer(tUInt sec);
   void stopProgressTimer();
   void notifyClients();

   std::string const &getStartingItemRefKey() const {
      return _startingItemRefKey;
   }
   
   std::string getDefaultDecision() {
      if (!_decisionsSection) {
         return "Invalid";
       }
      return swu::getTextFromChild(_decisionsSection, "DEFAULT_MODE",false, "Invalid");
   }
   
private:
   void configure();
   void cleanup();
   void startItem(CtrlProgItem *progItem);

   CtrlProgItem nextItem(CtrlProgItem curItem);

   TiXmlElement *nextUpdatedBoschXmlItem(TiXmlElement *boschXmlItem=0, bool onEnter=true);

   void bubbleUpDoUpdate(BoschXmlNode access);
   void reserveLock(TiXmlElement *modelem, BoschXmlNode access);
   void restoreProgItems();
   bool readProgressFile();
   void subscribeForStick();
   bool restoreCurrentRelease(TiXmlElement *overallXml);
   void setOverallSection(TiXmlElement *overallSection);
   void enterState(tenState enState);   

   void evalCompatIndex();
   void setInitialRelease(TiXmlDocument *boschXmlDocIn, CtrlPersistentConfig *persistentConfig);


   void fetchExtraArgs(TiXmlElement *xmlElem, std::map< std::string, std::string>&res);

   void fetchExtraArgsRecurse(TiXmlElement *xmlElem, std::map< std::string, std::string>&res);

   update_decision_setting_type decisionFromString(std::string decision);

   update_decision_setting_type getDecisionFromDecisionSection(std::string longName);

   update_decision_setting_type getNodeDecision(TiXmlElement *node);

   update_decision_setting_type getReleaseDecision(TiXmlElement *node);

   TiXmlElement *getXmlProgressItem(std::string longName);

   bool isSubmoduleUpdateConfigured(TiXmlElement *subModule);

   bool needsUpdate(TiXmlElement *elem) {
      return swu::getUIntFromChild(elem->FirstChildElement("INFO"), "DO_UPDATE") ? true : false;
   }

   bool needsUpdate(std::string const &longName) {
      TiXmlElement *elem=getBoschXmlElem(_releaseSection, longName);
      return swu::getUIntFromChild(elem->FirstChildElement("INFO"), "DO_UPDATE") ? true : false;
   }

   void setNeedsUpdateAllSubmodules(BoschXmlNode access);

   bool addDependencies();

   bool handleGroups();

   TiXmlElement *createProgItem(std::string const &longName, std::string const &appType);

   bool createScriptItems(BoschXmlNode &access, std::string scriptType );

   bool createProgItems(bool onStartOnly=false);   

   TiXmlElement *getCurrentProgressRelease() {
      return _progressSection->FirstChildElement("RELEASE");
   }

   void evalScopes(BoschXmlNode &boschXmlNode);
   swu::SwUpdTimer< CtrlProgressSection, Msg_ProgressTimer > _progressTimer;


};

} 


#if 0

<DEVICE>
<SWRELEASE>0002</SWRELEASE>
<NAME_DISPLAY>unknown</NAME_DISPLAY>
<NAME>IMX</NAME>
<MODULE>
<NAME>MAIN</NAME>
<NAME_DISPLAY>MAIN</NAME_DISPLAY>
<LANG>Default</LANG>
<SUBMODULE>
<TYPE>BOOTLOADER</TYPE>
<NAME_DISPLAY>unknown</NAME_DISPLAY>
<VERSION>2</VERSION>
<PREPARE>
<SCRIPT>
<NAME>bot.fp.tar.gz</FILENAME>
</SCRIPT>
</PREPARE>
<FLASH_ITEMS>
<FLASH_ITEM>
<FILENAME>bot.fp.tar.gz</FILENAME>
<PATH>./rfs/rnaivi/</PATH>
<FOUND>bot.fp.tar.gz</FOUND>
<CKSUM>
<SIZE>32286720</SIZE>
<TYPE>SHA256</TYPE>
<SEGLEN>0</SEGLEN>
<VAL>6d3b256a959ebecb725a60a13b90f99bef6ce0eec24ad7df8674ad3590fe93f9</VAL>
</CKSUM>
</FLASH_ITEM>
</FLASH_ITEMS>
</SUBMODULE>



<PREPARE>
<SCRIPT>CHECK_PARTITIONS</SCRIPT>
<EXTRA_ARGS></EXTRA_ARGS>
</PREPARE>

<PROGRESS>
<UPDATE_MODE>enRecovery</UPDATE_MODE>
<ID>0</ID>
<RELEASE>
<ITEM>
<NAME>IMX</NAME>
<TYPE>"FINALIZE"</TYPE>
<INDEX>0</INDEX>
<STATE>done<STATE>
<RES>passed</RES>
</ITEM>
<ITEM>

<STATE>running<STATE>
<ITEMS>
<ITEM>
<NAME>IMX_INIT_APPLICATION</NAME>
<TYPE>"APP"</TYPE>
<INDEX>0</INDEX>
<STATE>done<STATE>
<RES>passed</RES>
</ITEM>
<ITEM>
<TYPE>"PREPARE"</TYPE>
<NAME>IMX_MAIN_APPLICATION</NAME> 
<STATE>running<STATE>
<RES></RES>
<INDEX>2</INDEX>
</ITEM>
<TYPE>"APP"</TYPE>
<NAME>IMX_MAIN_APPLICATION</NAME> 
<STATE>running<STATE>
<RES></RES>
<INDEX></INDEX>
</ITEM>
<ITEM>
<TYPE>"FINALIZE"</TYPE>
<NAME>IMX_MAIN_APPLICATION</NAME> 
<STATE>pending<STATE>
<RES></RES>
<INDEX></INDEX>
<SCRIPT></SCRIPT>
</ITEM>
</ITEMS>
</RELEASE>
<PROGRESS>

<INFO>
<NUM_UPDATED_CHILDREN></NUM_UPDATED_CHILDREN>
<HAS_UNCHANGED_CHILDREN></HAS_UNCHANGED_CHILDREN>
<ITEMS>
<ITEM>
<NAME>IMX_INIT_APPLICATION</NAME>
<TYPE>"APP"</TYPE>
<INDEX>0</INDEX>
<STATE>done<STATE>
<RES>passed</RES>
</ITEM>
<ITEM>
<TYPE>"APP"</TYPE>
<NAME>IMX_MAIN_APPLICATION</NAME>
<STATE>running<STATE>
<RES></RES>
<INDEX></INDEX>
</ITEM>
<ITEM>
<TYPE>"FINALIZE"</TYPE>
<NAME>IMX_MAIN_APPLICATION</NAME>
<STATE>pending<STATE>
<RES></RES>
<INDEX></INDEX>
</ITEM>


</ITEMS>
<INFO>

#endif
#endif
