
#include <string>
using namespace std;
#include "config/fcswupd_config.hpp"
#include "main/fcswupd_systemData.h"
#include "main/fcswupd_bxmlCheck.h"
#include "main/fcswupd_resourcemanager.h"
#include "ctrl/fcswupd_ctrlXmlUtil.h"
#include "ctrl/fcswupd_ctrlProgress.h"
#include "util/swu_execCommand.h"
#include "util/swu_filesystem.h"
#include "main/fcswupd_campaignmanager.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_CTRL
#include "trcGenProj/Header/fcswupd_ctrlProgress.cpp.trc.h"
#endif

namespace fcswupdate {

bool CtrlProgressSection::_doHandleRecoveryMode = true;

ProgressFile::ProgressFile(std::string filename) :
   swu::XmlFile(filename) {
   ETG_TRACE_ERR(("ProgressFile CTOR"));
}


ProgressFile::~ProgressFile() {
   ETG_TRACE_ERR(("ProgressFile DTOR"));
}


CtrlProgItem::CtrlProgItem(CtrlProgressSection *progressSection, TiXmlElement *progressItem):
   _valid(true),
   _progressSection(progressSection),
   _progressItem(progressItem),
   _subItemIndexMax(0)
{
   ETG_TRACE_ERR(("CtrlProgItem CTOR START"));
   
   _progBoschXmlElem=_progressItem->Parent()->Parent()->Parent()->ToElement();
   SWU_ASSERT_RETURN(_progBoschXmlElem);
   BoschXmlNode access(_progBoschXmlElem);
   _name=access.getLongName();
   ETG_TRACE_ERR(("CtrlProgItem CTOR 4"));
   _itemType=swu::getTextFromChild(_progressItem, "TYPE");
   TiXmlElement *items=access.getItemsItem();
   ETG_TRACE_ERR(("CtrlProgItem CTOR 6"));
   _subItemIndexMax=swu::getNumChildrenWithTxtChild(items, "ITEM", "TYPE", _itemType) - 1;
   _valid=true;
   ETG_TRACE_ERR(("CtrlProgItem CTOR END"));

}

void CtrlProgItem::setState(std::string state) {
   ETG_TRACE_USR4(("CtrlProgItem::setState(%20s) name=%30s subitemIndex=%u _subItemIndexMax=%u",
                   state.c_str(), getName().c_str(), getSubItemIndex(), _subItemIndexMax));
   swu::setTextChild(_progressItem, "STATE", state);

}



void CtrlProgItem::start() {
   string strRefkey = getRefKey();  //Coverity fix for 209382
   ETG_TRACE_USR4(("CtrlProgItem::start() name=%30s subitemIndex=%u _subItemIndexMax=%u",
                   strRefkey.c_str(), getSubItemIndex(), _subItemIndexMax));
   setState("running");
   _progressSection->startItem(this);
}

void CtrlProgItem::finalize(bool success) {
   tUInt subItemIndex=getSubItemIndex();
   std::string oldState=swu::getTextFromChild(_progressItem, "STATE");
   ETG_TRACE_USR4(("CtrlProgItem::finalize(success=%u) name=%30s subitemIndex=%u LAST SUBITEM item=%p", 
                   success, getName().c_str(), subItemIndex, _progressItem));
   char const *newState=success ? "ok" : "failed";
   if (success) {
      setScomoPhaseDone();
   }
   setState(newState);
   _progressSection->store();
}

bool CtrlProgItem::isValid() const {
   bool valid=_valid && _progressSection && _progressItem && _progBoschXmlElem  && !_itemType.empty();
   if (!valid) {
      ETG_TRACE_ERR(("CtrlProgItem::isValid: false: "));
      ETG_TRACE_ERR(("CtrlProgItem::  _valid=%u", _valid));
      ETG_TRACE_ERR(("CtrlProgItem::  _progressSection=%p", _progressSection));
      ETG_TRACE_ERR(("CtrlProgItem::  _progressItem=%p", _progressItem));
      ETG_TRACE_ERR(("CtrlProgItem::  _progBoschXmlElem=%p", _progBoschXmlElem));
      ETG_TRACE_ERR(("CtrlProgItem::  _name='%s'", _name.c_str()));
      ETG_TRACE_ERR(("CtrlProgItem::  _itemType='%s'", _itemType.c_str()));
   }
   return valid; 
}

void CtrlProgItem::setScomoPhaseDone() {
   SWU_ASSERT_RETURN(_progressItem);
   swu::setTextChild(_progressItem, "SCOMO_PHASE_DONE",swu::intToString(_progressSection->getRequestedScomoPhase()));
}

tUInt CtrlProgItem::getUpdateTime() const {
   tUInt updateTime=0;
   if (getType()=="APP") {
      updateTime=swu::getUIntFromChild(_progBoschXmlElem, "UPDATE_TIME")/ getNumSubItems();
   } else if(CurrentBoschXml::instance()->isValid()) {
      CtrlProgItem *_thisMutable=const_cast<CtrlProgItem *>(this);
      TiXmlElement *relBxmlItem=_thisMutable->getRelXmlAccess().getXml();
      if (!relBxmlItem) {return updateTime;};
      TiXmlElement *relScriptItems=relBxmlItem->FirstChildElement(getType().c_str());
      if (!relScriptItems) {return updateTime;};
      TiXmlElement *relScriptItem=swu::getNthChild(relScriptItems, "SCRIPT", getSubItemIndex());
      if (!relScriptItem) {return updateTime;};
      updateTime=swu::getUIntFromChild(relScriptItem, "UPDATE_TIME", false);
      if (updateTime) {
         ETG_TRACE_USR4(("CtrlProgItem::got update-time from script:%u", updateTime));
      }
   }
   return updateTime;
}

bool CtrlProgItem::needsScomoPhase(tUInt scomoPhase) {
   tUInt scomoPhaseDone = getScomoPhaseDone();
   if (scomoPhaseDone == scomoPhase) {
      return false;
   }
   std::set<std::string> neededScomoPhases=swu::splitString(getConfiguredScomoPhases(), ',');
   for (std::set<std::string>::iterator phaseIter=neededScomoPhases.begin(); phaseIter!=neededScomoPhases.end(); ++phaseIter) {
      tUInt configuredScomoPhase=swu::stringToUInt(*phaseIter);
      if (configuredScomoPhase <= scomoPhase) {
         return true;
      }
   }
   return false;
}

bool CtrlProgItem::needsCurrentScomoPhase() {
   return needsScomoPhase(_progressSection->getRequestedScomoPhase() );
}

// Scomo-phase needed by a progress-item, does not reflect global needs.
std::string CtrlProgItem::getConfiguredScomoPhases() {
   ETG_TRACE_USR4(("CtrlProgItem::getConfiguredScomoPhases - START"));
   std::string res=_progressSection->getMaxScomoPhaseStr();
   tenScomoPhaseHandlingMode enScomoPhaseHandlingMode =(tenScomoPhaseHandlingMode)Config::instance()->cfg_ScomoPhaseHandlingMode.get();

   switch (enScomoPhaseHandlingMode) {
      case tenScomoPhaseHandlingMode_None: {
         break;
      }
      case tenScomoPhaseHandlingMode_Default: {         
         res=getTextFromChildRecursive(getRelXmlAccess().getXml(), "NEEDED_SCOMO_PHASES", "2,3", true);
         break;
      }
      case tenScomoPhaseHandlingMode_1None2Neutral3All: {
         if (needsRecoveryMode() || _progressSection->isLastOf(XmlItemAccess::enType_Release)) {
            res = "3";
         }
         else {
            res = "2";
         }
         break;
      }
      default: {
         break;
      }
   }
   ETG_TRACE_USR4(("CtrlProgItem::getConfiguredScomoPhases:RESULT-%s", res.c_str()));
   return res;
}


bool CtrlProgItem::needsRecoveryMode() {
   if (!isValid() || !_progressSection->getReleaseSection()) {
      ETG_TRACE_ERR(("CtrlProgItem::needsRecoveryMode(): false (invalid)"));      
      return false;
   }
   XmlItemAccess relAccess=getRelXmlAccess();
   bool res=needsRecoveryMode(relAccess);
   
   string strRefkey = getRefKey();   //Coverity fix for 209502
   ETG_TRACE_USR1(("CtrlProgItem::needsRecoveryMode(%50s): res=%u", strRefkey.c_str(),res));
   return res;
}

bool CtrlProgItem::needsRecoveryMode(XmlItemAccess &access) {
   std::string name=access.getLongName();
   bool res=swu::getUIntFromChild(access.getXml(), "NEEDS_RECOVERY_MODE");
   string strRefkey = getRefKey();   //Coverity fix for 209521
   if (access.isRelease()) {
      ETG_TRACE_ERR(("CtrlProgItem::needsRecoveryMode(%40s): isRelease, res=%u", strRefkey.c_str(), res));
      return res;
   }
   if (!res) {
      XmlItemAccess parent=access.getParent();
      ETG_TRACE_ERR(("CtrlProgItem::needsRecoveryMode(%40s): recurse to %s", strRefkey.c_str(), parent.getLongName().c_str()));
      return needsRecoveryMode(parent);
   }
   ETG_TRACE_ERR(("CtrlProgItem::needsRecoveryMode(%40s): res=%u", strRefkey.c_str(), res));
   return res;
}


TiXmlElement *CtrlProgItem::getProgBoschXmlItem() {
   return _progBoschXmlElem;
}


XmlItemAccess CtrlProgItem::getRelXmlAccess() {
   SWU_ASSERT(_progressSection);
   TiXmlElement *releaseSection=_progressSection->getReleaseSection();
   SWU_ASSERT(releaseSection);
   return XmlItemAccess(releaseSection, _name);
}

XmlItemAccess CtrlProgItem::getProgressRelXmlAccess() {
   SWU_ASSERT(_progressSection);
   TiXmlElement *releaseSection=_progressSection->getProgressReleaseSection();
   SWU_ASSERT(releaseSection);
   return XmlItemAccess(releaseSection, _name);
}



std::string CtrlProgItem::getReleasePath() {
  SWU_ASSERT(_progressSection);
  TiXmlElement *releaseSection=_progressSection->getReleaseSection();  
  TiXmlElement *deviceSection=releaseSection->FirstChildElement("DEVICE");
  SWU_ASSERT_RETURN_VAL(deviceSection, "");
  return swu::getTextFromChild(deviceSection, "BASEPATH");
}

TiXmlElement *CtrlProgItem::getToolsXml() {
   XmlItemAccess curAccess(getRelXmlAccess());
   TiXmlElement *toolsXml=0;
   while (curAccess.enGetType()>=XmlItemAccess::enType_Release) {
      TiXmlElement *curXml=curAccess.getXml();
      toolsXml=curXml->FirstChildElement("TOOLS");
      if (toolsXml) {
         break;
      }
      SWU_ASSERT_RETURN_VAL(curAccess.enGetType()!=XmlItemAccess::enType_Release, 0);
      curAccess=curAccess.getParent();
   }
   return toolsXml;
}

std::string CtrlProgItem::getSourceFileType() {
   XmlItemAccess curAccess(getRelXmlAccess());
   while (curAccess.enGetType()>=XmlItemAccess::enType_Release) {
      TiXmlElement *curXml=curAccess.getXml();
      char const *sourceFileType=swu::getTextFromChild(curXml, "SOURCE_FILE_TYPE", false);
      if (sourceFileType) {
         return sourceFileType;
      }
      if (curAccess.enGetType()==XmlItemAccess::enType_Release) {
         break;
      }
      curAccess=curAccess.getParent();
   }
   return "tar";
}

std::string CtrlProgItem::getDataFileDestOffset(){
   SWU_ASSERT_RETURN_VAL(isValid(), "");
   std::string fileOffset;
   if (_itemType=="APP") {
      TiXmlElement *item=swu::getNthChild(getRelXmlAccess().getXml()->FirstChildElement("FLASH_ITEMS"), "FLASH_ITEM", getSubItemIndex());
      fileOffset = swu::getTextFromChildOrEmpty(item, "FILEOFFSET");
   }
   return fileOffset;
}

std::string CtrlProgItem::getDataFileSize(){
   SWU_ASSERT_RETURN_VAL(isValid(), "");
   std::string fileSize;
   if (_itemType=="APP") {
      TiXmlElement *item=getDataChecksum();
      fileSize = swu::getTextFromChild(item, "SIZE", true, "" );  //Coverity fix for 83551
   }
   return fileSize;
}
void CtrlProgItem::getExtraArgs(std::map<std::string, std::string> &kvPairs) {
   SWU_ASSERT(_progressSection);
   _progressSection->fetchExtraArgsRecurse(getRelXmlAccess().getXml(), kvPairs);
   kvPairs["DataFileIndex"]= swu::intToString(getSubItemIndex());
   kvPairs["DataFileCount"]= swu::intToString(_subItemIndexMax + 1);
   kvPairs["BxmlVersion"]= swu::intToString(_progressSection->getBxmlVersion());
   kvPairs["PartialUpdate"]=_progressSection->isFullUpdate() ? "0" : "1";
   kvPairs["MediaType"]=sourceTypeToString(_progressSection->getSourceType());
   kvPairs["SourceFileType"]=getSourceFileType();
   kvPairs["DataFileDestOffset"]=getDataFileDestOffset();
   kvPairs["DataFileSize"]=getDataFileSize();   
   kvPairs["ReleaseHasDeltaUpdate"]=swu::getUIntFromChild(getRelXmlAccess().getXml(), "HAS_DELTA") ?  "1" : "0";
   kvPairs["UpdateOption"]=_progressSection->getDefaultDecision();
   kvPairs["SubModuleListConfiguredToUpdate"] = _progressSection->getSubModuleListConfiguredToUpdate();

   //assuming that 0 is a invalid phase for scomo.
   if((tenScomoPhaseHandlingMode)Config::instance()->cfg_ScomoPhaseHandlingMode.get() ==tenScomoPhaseHandlingMode_Default) {
      tUInt scomoPhaseDone=getScomoPhaseDone();
      std::set<std::string> neededScomoPhases=swu::splitString(getConfiguredScomoPhases(), ',');
      for (std::set<std::string>::iterator phaseIter=neededScomoPhases.begin(); 
           phaseIter!=neededScomoPhases.end();
           ++phaseIter) {
         tUInt scomoPhase=swu::stringToUInt(*phaseIter);
         if (scomoPhase<=_progressSection->getRequestedScomoPhase() && scomoPhase > scomoPhaseDone) {
            std::string strneededPhase = "execute_phase_" + swu::intToString(scomoPhase);
            kvPairs.insert(std::make_pair(strneededPhase, "1"));
         }
      }
   } else {
      // if not indicated otherwise
      kvPairs["execute_phase_all"] = "1";
   }
     
   if (Config::instance()->cfg_HandleRecoveryMagicsInFcSwUpd.get()) {
      kvPairs["DisableForceRecoveryMode"]="1";
   }

   if (_progressSection->getSpecialMode()=="COMPAT_UPDATE") {
      kvPairs["CompatUpdate"]="1";
   }

   
   if (_progressSection->isLastOf(XmlItemAccess::enType_Release)) {
      // last item of this release
      if (_progressSection->switchBootChainOnDone()) {
         // let target-scripting switch the boot-chain
         kvPairs["SwitchBootChainOnSuccess"]= "1";
      }
      if (!_progressSection->isCompatUpdate()) {
         // let target-scripting enable normal mode again
         kvPairs["LeaveRecoveryModeOnSuccess"]= "1";         
      }
   }

   if (_progressSection->isPostImageMode()) {
      kvPairs["PostImageMode"]= "1"; 
   }

   //in recovery update, we need to clear the oem partition to avoid inconsistency.
   if(_progressSection->getInitialMode() == tenSwUpdateMode_Emergency) {
      kvPairs["force_oem_init"]= "1";
   }
   
   char const *nameDisplay=swu::getTextFromChild(getRelXmlAccess().getXml(), "NAME_DISPLAY", false);
   if (nameDisplay) {
      kvPairs["NameDisplay"]=nameDisplay;
   } else {
      kvPairs["NameDisplay"]=getRelXmlAccess().getLongName();;
   }
}

std::string CtrlProgItem::getFileNameAndPath(TiXmlElement *elem) {
   SWU_ASSERT(_progressSection);
   ETG_TRACE_USR1(("CtrlProgItem::getFileNameAndPath: mediaPath=%s", _progressSection->getMediaPath().c_str()));
   ETG_TRACE_USR1(("CtrlProgItem::getFileNameAndPath: releasePath=%s",  getReleasePath().c_str()));

   std::string path = swu::getTextFromChild(elem, "PATH");
   ETG_TRACE_USR1(("CtrlProgItem::getFileNameAndPath: itemPath=%s",  path.c_str()));

   //check if it's absolute path then take as it is.
   std::string res;
   if(swu::isAbsolutePath(path)) {
      res = path + "/" + swu::getTextFromChild(elem, "FILENAME");
   } else {
      res = _progressSection->getMediaPath() + "/" + getReleasePath() + "/" + swu::getTextFromChild(elem, "PATH") + "/" + swu::getTextFromChild(elem, "FILENAME");
   }

     ETG_TRACE_USR1(("CtrlProgItem::getFileNameAndPath: res=%s", res.c_str()));
 
   return res;

}

bool CtrlProgItem::needsPostInstall() {
   tenSourceType enSourceType=_progressSection->getSourceType();
   bool res=false;
   ETG_TRACE_USR4(("CtrlProgItem::needsPostInstall() START sourceType=%u type=%s", 
                   enSourceType, getType().c_str() ));

   if (Config::instance()->cfg_EnterEngineering.get()) {
      // no post-install in developer-mode
      return false;
   }
   /*
   if (enSourceType==tenSourceType_USB ||
       enSourceType==tenSourceType_SD) {
      // no post-install for external sources needed
      return false;
   }
   */
   if (getType()=="APP") {
      // post-install is only defined for scripts
      return false;
   }
   TiXmlElement *relBxmlItem=getRelXmlAccess().getXml();
   SWU_ASSERT_RETURN_VAL(relBxmlItem, false);
   TiXmlElement *relScriptItems=relBxmlItem->FirstChildElement(getType().c_str());
   SWU_ASSERT_RETURN_VAL(relScriptItems, false);
   TiXmlElement *relScriptItem=swu::getNthChild(relScriptItems, "SCRIPT", getSubItemIndex());
   SWU_ASSERT_RETURN_VAL(relScriptItem, false);
   res =swu::getUIntFromChild(relScriptItem, "POST_INSTALL_MODE", false);
   if (res) {
      ETG_TRACE_USR4(("CtrlProgItem::needsPostInstall() TRUE")); 
   }
   return res;
}

std::string CtrlProgItem::getScriptName() {
   ETG_TRACE_USR4(("CtrlProgItem::getScriptName() START type=%s", getType().c_str() ));
   std::string script_name=_name;
   if (getType() == "RELEASE") {
      script_name="RELEASE";
   } 
   if (getType()!="APP") {
      script_name+="_" + getType();
      TiXmlElement *relBxmlItem=getRelXmlAccess().getXml();
      SWU_ASSERT_RETURN_VAL(relBxmlItem, "");
      TiXmlElement *relScriptItems=relBxmlItem->FirstChildElement(getType().c_str());
      SWU_ASSERT_RETURN_VAL(relScriptItems, "");
      TiXmlElement *relScriptItem=swu::getNthChild(relScriptItems, "SCRIPT", getSubItemIndex());
      SWU_ASSERT_RETURN_VAL(relScriptItem, "");
      char const *customName=swu::getTextFromChild(relScriptItem, "NAME", false);
      if (customName) {
         // use default-name.
         script_name =customName;
         ETG_TRACE_USR4(("CtrlProgItem::getScriptName() using customName"));
      }
   }
   for (uint32_t i = 0; i < script_name.size(); ++i) {
      if (script_name[i] == '/') {
         script_name[i] = '_';
      } else if ((script_name[i] >= 'A') && (script_name[i] <= 'Z')) {
        script_name[i] = static_cast<char> (script_name[i] + 'a' - 'A' );
      }
   }
   ETG_TRACE_USR4(("CtrlProgItem::getScriptName(): %s", script_name.c_str()));
   return script_name;
}

bool CtrlProgItem::isPhaseStart() {
   std::string phaseStartModule=swu::getTextFromChildOrEmpty(_progressSection->getProgressReleaseSection(), "PHASE_START");
   ETG_TRACE_USR4(("CtrlProgItem::isPhaseStart():phaseStartModule=%50s", phaseStartModule.c_str()));
   if (!isValid()) {
      ETG_TRACE_USR4(("CtrlProgItem::isPhaseStart():item invalid"));
      return false;
      
   }

   if (phaseStartModule.empty()) {
      return true;
   }
   ETG_TRACE_USR4(("CtrlProgItem::isPhaseStart():refKey=%s", getRefKey().c_str()));
   if (phaseStartModule == getRefKey()) {
      ETG_TRACE_USR4(("CtrlProgItem::isPhaseStart():matched refKey"));
      return true;
   }

   if (Config::instance()->cfg_EnterFullOperation.get() && 
       ((_progressSection->needsSwitchToRecovery() && _progressSection->isCurItem(this)) || needsRecoveryMode())) {
      return true;
   }
   ETG_TRACE_USR4(("CtrlProgItem::isPhaseStart():false"));
   return false;
}

std::vector<std::string> CtrlProgItem::getRequiredLocks()
{
  ETG_TRACE_USR4(("CtrlProgItem::getRequiredLocks() START" ));
  std::vector <std::string> bgLocksList;
   
  XmlItemAccess curAccess(getRelXmlAccess());

  while (curAccess.enGetType()>=XmlItemAccess::enType_Module) {
     TiXmlElement *curXml=curAccess.getXml();
     const char *bgLocks = swu::getTextFromChild(curXml, "BG_LOCKS", false);
     if (bgLocks){ 
          std::set <std::string> locksList = swu::splitString(bgLocks); 
          std::copy(locksList.begin(), locksList.end(),std::back_insert_iterator<std::vector<std::string> >(bgLocksList));
     }
     curAccess=curAccess.getParent();
   }  
   // Print the Locks for the Tracebality   
   std::vector<std::string>::iterator it;
   for (it = bgLocksList.begin(); it != bgLocksList.end(); ++it){
      ETG_TRACE_USR1(("Locks that are required %s",it->c_str()));
   }
   
    ETG_TRACE_USR4(("CtrlProgItem::getRequiredLocks() END" ));    
    return bgLocksList;   
}
CtrlProgressSection::CtrlProgressSection(std::string progressFileName):
   _progressRoot(0),
   _progressSection(0),
   _progressReleaseSection(0),
   _overallSection(0),
   _releaseSection(0),
   _decisionsSection(0),
   _currentBoschXmlStick(0),
   _enState(enState_Initial),
   _statistics(this),
   _curRetries(0),
   _isCompatUpdate(false),
   _abort(false),
   _silent(false),
   _recoveryNeeded(false),
   _rescheduleNeeded(false),
   _persistentConfig(0),
   _staticConfig(0),
   _persistentConfigAlloc(0),
   _staticConfigAlloc(0),
   _progressFile(0),
   _defaultDecision( update_decision_setting_type__invalid )
{
   ETG_TRACE_USR4(("CtrlProgressSection::CTOR START"));
   SWU_ASSERT_RETURN(progressFileName.size());
   _progressFile = new ProgressFile(progressFileName);
   ETG_TRACE_USR4(("CtrlProgressSection::CTOR END (%s)", progressFileName.c_str()));
}

CtrlProgressSection::~CtrlProgressSection() {
   ETG_TRACE_USR1(("CtrlProgressSection DTOR - START"));
   _progressRoot = 0;
   _progressSection = 0;
   _progressReleaseSection = 0;
   _overallSection = 0;
   _releaseSection = 0;
   _decisionsSection = 0; 

   delete _progressFile;
   _progressFile = 0;

   if(_persistentConfigAlloc) {
      delete _persistentConfigAlloc;
      _persistentConfigAlloc = 0;
   }

   if(_staticConfigAlloc) {
      delete _staticConfigAlloc;
      _staticConfigAlloc = 0;
   }

   ETG_TRACE_USR1(("CtrlProgressSection DTOR - END"));
}


tVoid CtrlProgressSection::traceState() {
   ETG_TRACE_COMP(("  _enState=%u", ETG_CENUM(CtrlProgressSection::tenState, _enState)));
   ETG_TRACE_COMP(("  _curRetries=%u", _curRetries));
   ETG_TRACE_COMP(("  _isCompatUpdate=%u", _isCompatUpdate));
   ETG_TRACE_COMP(("  _silent=%u", _silent));
   ETG_TRACE_COMP(("  _recoveryNeeded=%u",_recoveryNeeded));
   ETG_TRACE_COMP(("  _doHandleRecoveryMode=%u", _doHandleRecoveryMode));
}


// copy all INFO-items for _releaseSection to _progressReleaseSection
TiXmlElement * CtrlProgressSection::collectProgItems() {
   ETG_TRACE_ERR(("CtrlProgressSection::collectProgItems() START"));
   SWU_ASSERT_RETURN_VAL(_releaseSection, 0);

   TiXmlElement *progRelease=copyBxmlSkeleton(_releaseSection);
   for (XmlItemAccessIter srcIter(_releaseSection, XmlItemAccessIter::enOnEnter);!srcIter.isDone();srcIter.next()) {
      XmlItemAccess srcAccess(srcIter);
      BoschXmlNode dstAccess(progRelease, srcAccess.getLongName());

      switch (srcAccess.enGetType()) {
         case XmlItemAccess::enType_Release: {
            dstAccess.copyChildFrom(srcAccess, "NAME_DISPLAY");
            dstAccess.copyChildFrom(srcAccess, "DNL_SIZE");
            dstAccess.copyChildFrom(srcAccess, "TYPE"); // todo: USB / FOTA / DEVELOPER
            swu::addTextChild(dstAccess.getXml(), "COMPLETED", "1");
            break;
         }
         case XmlItemAccess::enType_Device: {
            dstAccess.copyChildFrom(srcAccess, "NAME_DISPLAY");
            dstAccess.copyChildFrom(srcAccess, "VERSION");
            dstAccess.copyChildFrom(srcAccess, "MUVERSION");
            break;
         }
         case XmlItemAccess::enType_Module: {          
            dstAccess.copyChildFrom(srcAccess, "NAME_DISPLAY");
            break;
         }
         case XmlItemAccess::enType_Submodule: {
            dstAccess.copyChildFrom(srcAccess, "UPDATE_TIME");
            dstAccess.copyChildFrom(srcAccess, "NAME_DISPLAY");
            dstAccess.copyChildFrom(srcAccess, "VERSION");
            swu::setUIntChild(dstAccess.getInfoItem(), "DO_UPDATE", 0);
            break;
         }
         default: {
            break;
         }
      }

   }

   
   ETG_TRACE_USR4(("CtrlProgressSection::collectProgItems END _enState=%u _releaseSection=%p", _enState, _releaseSection));

   return progRelease;
   
}

// set state of current release and overall according to counters
void CtrlProgressSection::evalProgXmlStates() {
   ETG_TRACE_USR4(("evalProgXmlStates:START"));
   _statistics.clear();

   // collect statistics
   for (XmlItemAccessIter iter(_progressReleaseSection, XmlItemAccessIter::enOnBoth);!iter.isDone();iter.next()) {
      BoschXmlNode access(iter);  
      TiXmlElement *itemsItem=access.getInfoItem()->FirstChildElement("ITEMS");
      for (TiXmlElement *item=itemsItem->FirstChildElement("ITEM");
           item;
           item=item->NextSiblingElement("ITEM")) {
         std::string itemType=swu::getTextFromChild(item, "TYPE");
         XmlItemAccessIter::tenDirection direction=iter.getDirection();

         if ((XmlItemAccessIter::enEnter==direction && itemType!="FINALIZE") ||
             (XmlItemAccessIter::enLeave==direction && itemType=="FINALIZE")) {
            ETG_TRACE_USR4(("copy to progRelease:item=%p name=%30s type=%30s state=%30s index=%u",
                            item,
                            swu::getTextFromChild(item, "NAME"),
                            swu::getTextFromChild(item, "TYPE"),
                            swu::getTextFromChild(item, "STATE"),
                            swu::getUIntFromChild(item, "INDEX")));
            
            CtrlProgItem progItem = CtrlProgItem(this, item);
            _statistics.add(progItem);
         }
      }
   }
   ETG_TRACE_USR4(("evalProgXmlStates:copy to progRelease:all Items done"));


   Statistics::Counters const &counters=_statistics.get();

   if (!counters.numRunning) {
      stopProgressTimer();
   }
   // check errors
   if (counters.numFailed) {
      // something went wrong
      // overall state follows, because by now we don't continue if anything went wrong
      swu::setTextChild(_progressReleaseSection, "ALL_OK", "0");
      stopProgressTimer();
   } else  {
      swu::setTextChild(_progressReleaseSection, "ALL_OK", "1");
   }
   
   // eval state-strings for overall and current
   if (counters.numRunning || counters.numPending) {
      // there is still something to be done
      if(_enState == enState_ScomoPending) {
         setPersistentState("scomoPending");
      } else if (_abort) {
         setPersistentState("result");
      }
      else {
         setPersistentState("running");
      }
   } else {
      if (!counters.numFailed ) {
         if (isCompatUpdate()) {
            // update must be repeated as full update
            setPersistentState("result");
         } else  if (isScomoMode() && (getRequestedScomoPhase() < getMaxScomoPhase())) {
            // todo, check up to which phase is required
            setPersistentState("scomoPending");
         }
         else {
            setPersistentState("result");
         }
      } else { // update failed
         setPersistentState("result");
      }
   }
   ETG_TRACE_USR4(("evalProgXmlStates done"));

}


tVoid CtrlProgressSection::store() {
   ETG_TRACE_USR1(("CtrlProgressSection::store() START _releaseSection=%p", 
                   _releaseSection));
#ifdef LOCAL_BUILD_
   TiXmlDocument tmpDoc1;
   tmpDoc1.InsertEndChild(*_releaseSection);
   tmpDoc1.SaveFile("2"TMP_RELEASE_FILE);
#endif
   
   ETG_TRACE_USR1(("CtrlProgressSection::store() call evalProgXmlStates"));
   evalProgXmlStates(); 
   ETG_TRACE_USR1(("CtrlProgressSection::store() done evalProgXmlStates"));
   // evaluate overall state
   std::string overallStateString=getPersistentState();
   ETG_TRACE_USR4(("CtrlProgressSection::store overallStateString=%s", overallStateString.c_str()));

   if (!strcmp(swu::getTextFromChildOrEmpty(_progressReleaseSection, "SPECIAL_MODE"),"ENTER_FULL_UPDATE")) {
      // we were reqested, to start the udate again
      _enState=enState_Reboot;
   }
   else if (overallStateString=="result") {
      // here we no longer need a valid usb-stick
      _enState=enState_Result;
   }
   else if (_enState==enState_Restoring||
            _enState==enState_WaitStart ||
            _enState==enState_WaitBoschXml) {
      // keep state
   }
   else if (overallStateString=="scomoPending") {
      _enState=enState_ScomoPending;      
   }
   else if (overallStateString=="running") {
      
   }
   
   ETG_TRACE_USR4(("CtrlProgressSection::store: _silent=%u getXmlRoot=%p storeToStdOut=%u usePersistency=%u", 
                   _silent, 
                   _progressFile->getXmlRoot(), 
                   Config::instance()->cfg_CtrlLogXmlToStdOut.get(),
                   _persistentConfig->usePersistency.get()));

   if (!_silent) {
      if (_progressFile->getXmlRoot() && (Config::instance()->cfg_CtrlLogXmlToStdOut.get()) ) {
         swu::execCommand("echo STORE PROGRESS SECTION:");
         _progressFile->getXmlDoc().Print();
      }
      
      if (_progressFile->getXmlRoot() && _persistentConfig->usePersistency.get()) {
         _progressFile->store();
      }
      notifyClients();
   }
   ETG_TRACE_USR4(("CtrlProgressSection::store END _enState=%u _silent=%u", _enState, _silent));
}

void CtrlProgressSection::storeProgressFile() {
   ETG_TRACE_USR3(("CtrlProgressSection::storeProgressFile START"));
   
   if (!_silent) {    
      if (_progressFile->getXmlRoot() && _persistentConfig->usePersistency.get()) {
         _progressFile->store();
      }
   }

   ETG_TRACE_USR3(("CtrlProgressSection::storeProgressFile END"));
}


bool CtrlProgressSection::readProgressFile() {
   ETG_TRACE_USR4(("CtrlProgressSection::readProgressFile() START _persistentConfig=%p", _persistentConfig));
   
   _progressRoot=_progressFile->getXmlRoot();
   SWU_ASSERT_RETURN_FALSE(_progressRoot);
   _progressSection=_progressRoot->FirstChildElement("PROGRESS");
   SWU_ASSERT_RETURN_FALSE(_progressSection);
   std::string overallStateString=getPersistentState();
   SWU_ASSERT_RETURN_FALSE(_persistentConfig);
   _persistentConfig->restore(_progressSection);
   _progressReleaseSection=getCurrentProgressRelease();
   if(!_progressReleaseSection) {
      ETG_TRACE_USR4(("CtrlProgressSection::readProgressFile() no active release section found"));
      return true;
   }
   _overallSection=_progressRoot->FirstChildElement("OVERALL");
   SWU_ASSERT_RETURN_FALSE(_overallSection);

   _decisionsSection=_progressRoot->FirstChildElement("DECISIONS");
   //   SWU_ASSERT_RETURN_FALSE(_decisionsSection);
   return true;
}

void CtrlProgressSection::subscribeForStick() {
   ETG_TRACE_USR4(("CtrlProgressSection::subsribeForStick START"));
#ifndef LOCAL_BUILD_
   Msg_CurrentBoschXmlChanged::vSubscribe(this);
#endif
   if (!_currentBoschXmlStick) {
      _currentBoschXmlStick= CurrentBoschXml::instance();
   }
         
   _currentBoschXmlStick->setConfig(swu::getTextFromChildOrEmpty(_overallSection, "FINALNAME"), swu::getTextFromChildOrEmpty(_overallSection, "BXML_CHECKSUM"));

   ETG_TRACE_USR4(("CtrlProgressSection::subsribeForStick END"));

}

bool CtrlProgressSection::restore(bool peekOnly, CtrlPersistentConfig *persistentConfig, CtrlStaticConfig *staticConfig) {
   ETG_TRACE_USR4(("CtrlProgressSection::restore START"));
   SWU_ASSERT_RETURN_FALSE(_enState==enState_Initial);
   _persistentConfig=persistentConfig;
   if (!_persistentConfig) {
      ETG_TRACE_USR4(("CtrlProgressSection::restore: no persistentConfig given"));
      _persistentConfigAlloc=new CtrlPersistentConfig;
      _persistentConfig=_persistentConfigAlloc;
   }

   _staticConfig=staticConfig;
   if (!_staticConfig) {
      ETG_TRACE_USR4(("CtrlProgressSection::restore: no staticConfig given"));
      _staticConfigAlloc=new CtrlStaticConfig;
      _staticConfig=_staticConfigAlloc;
   }

   if (!_progressSection) {
      if (!_progressFile->restore()) {
         return false;
      }
      if (!readProgressFile()) {
         return false;
      }
      SWU_ASSERT_RETURN_FALSE(_progressSection);
   }
   

   if (peekOnly) {
      return true;
   } else {
      SWU_ASSERT_RETURN_FALSE(persistentConfig && staticConfig);
   }      


   std::string overallStateString=getPersistentState();
   ETG_TRACE_USR1(("overallStateString:%s", overallStateString.c_str()));
   if (overallStateString=="running") {
      // todo: assert that scomo-phase is unchanged      
      enterState(enState_Restoring);
   }
   else if (overallStateString=="pending" || overallStateString=="scomoPending" ) {
      if (isScomoMode()) {
         tUInt newScomoPhase=Config::instance()->cfg_ScomoPhase.get();
         // todo: handling if same scomo-phase is requested again.
         if (newScomoPhase > getRequestedScomoPhase()) {
            setRequestedScomoPhase(Config::instance()->cfg_ScomoPhase.get());
          }
      }
      enterState(enState_Restoring);
   } else {
      enterState(enState_Result);   
   }
   subscribeForStick();
   // now we have a progress-section, but the boschXml from the stick is still missing.

   // evaluate data of wanted stick:
   ETG_TRACE_USR4(("CtrlProgressSection::restore END"));
   return true;
}

void CtrlProgressSection::bubbleUpDoUpdate(BoschXmlNode access) {
   while (1) {
      ETG_TRACE_ERR(("CtrlProgressSection::bubbleUpDoUpdate: set DO_UPDATE for %s", access.getLongName().c_str()));
      swu::setUIntChild(access.getInfoItem(), "DO_UPDATE", 1);
      if (access.isRelease()) {
         // reached root node
         break;
      }
      access=BoschXmlNode(access.getParentXml());
      if (swu::getUIntFromChild(access.getInfoItem(), "DO_UPDATE")) {
         // from here up do_update is already set
         break;
      }
   }
}

void CtrlProgressSection::reserveLock(TiXmlElement *elem, BoschXmlNode access) {
   ETG_TRACE_USR2(("CtrlProgressSection::reserveLock - START"));

   Config *cfg = Config::instance();
   if(cfg->cfg_EnterFullOperation.get() && cfg->cfg_HandleCentralLock.get()) {

      std::string bgLocks = swu::getTextFromChildOrEmpty(elem, "CENTRAL_LOCKS");
      if(bgLocks.length() > 0) {
         ETG_TRACE_USR2(("reserveLock:%s", bgLocks.c_str())); 
         swu::setUIntChild(access.getXml(), "LOCK_STATE", 0);
         std::set<std::string> bgLockList=swu::splitString(bgLocks);                             
         ResourceManager::instance()->configure(_enReqMode_reserve, bgLockList);
      }      
   }   
   ETG_TRACE_USR2(("CtrlProgressSection::reserveLock - END"));
}




void CtrlProgressSection::restoreProgItems() {
   SWU_ASSERT_RETURN(_progressReleaseSection);
   for (XmlItemAccessIter iter(_progressReleaseSection, XmlItemAccessIter::enOnEnter);!iter.isDone();iter.next()) {
      BoschXmlNode access(iter);
      swu::setUIntChild(access.getInfoItem(), "DO_UPDATE", 0);
   }
   for (XmlItemAccessIter iter(_progressReleaseSection, XmlItemAccessIter::enOnEnter);!iter.isDone();iter.next()) {
      BoschXmlNode access(iter);
      for (TiXmlElement  *flashItem=access.getItemsItem()->FirstChildElement("ITEM");
           flashItem;
           flashItem=flashItem->NextSiblingElement("ITEM")) {
         if (!strcmp(swu::getTextFromChild(flashItem, "TYPE"),"PREPARE")) {
            continue;
         }
         std::string itemState=swu::getTextFromChild(flashItem, "STATE");
         if (itemState=="running" || itemState =="failed" || itemState=="pending") {
            bubbleUpDoUpdate(access);
         }
         else if (isScomoModeNormal() && itemState=="ok") {
            // item has allready been updated, lets check if it needs to be updated
            // again in this scomo-phase
            CtrlProgItem item(this, flashItem);
            if (item.needsCurrentScomoPhase() ) {
               swu::setTextChild(flashItem, "STATE", "pending");
               bubbleUpDoUpdate(access);
            }
         }
      }
   }
}



bool CtrlProgressSection::restoreCurrentRelease(TiXmlElement *overallXml) {
   ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease() START"));

   SWU_ASSERT_RETURN_FALSE(overallXml);
   tenSwUpdateError errorCode;
   TiXmlDocument *boschXmlDoc=FcSwUpdCore::instance()->getReleaseXml(*overallXml, errorCode);

   if (!boschXmlDoc) {
      ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease() getReleaseXml gave errorCode=%u",
                     errorCode));
      return false;
   }
   TiXmlElement *boschXmlRoot=boschXmlDoc->FirstChildElement();
   SWU_ASSERT_RETURN_FALSE(boschXmlRoot);
   if (_releaseSection) {
      delete _releaseSection;
   }
   _releaseSection=swu::Swu_UnlinkNode(boschXmlRoot->FirstChildElement("RELEASE"))->ToElement();
   SWU_ASSERT_RETURN_FALSE(_releaseSection);
   setOverallSection(boschXmlRoot->FirstChildElement("OVERALL"));
   ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease()"));


   evalCompatIndex();
   ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease() _isCompatUpdate=%u", _isCompatUpdate));

   boschXmlRoot=0;
   delete boschXmlDoc;
   boschXmlDoc=0;
   bool reconfigure=false;

   if (getPersistentState() == "result") {
      ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease() _progressReleaseSection STATE==result"));
      std::string specialMode=getSpecialMode();
      if ((specialMode=="COMPAT_UPDATE") || (specialMode=="ENTER_FULL_UPDATE")) {
         ETG_TRACE_ERR(("CtrlProgressSection::restoreCurrentRelease() _progressReleaseSection SPECIAL_MODE=%s", specialMode.c_str()));
         swu::setTextChild(_progressReleaseSection, "FULL_UPDATE", "1");
         reconfigure=true;
      }
   }
   if (reconfigure) {
      // if this compat-update
      Config::instance()->cfg_TestCompatUpdate.set(0);
      _defaultDecision=update_decision_setting_type__force;
      // special-mode was used to reconfigure the update, this is done now, remove special-mode
      swu::setTextChild(_progressReleaseSection, "SPECIAL_MODE", "NONE");
      
   }
   else {
      // now we got the correct release for our progress-section
      // prepare bosch-xml by adding info-node to every DEVICE MODULE SUBMODULE
      // copy the prog-items to the bosch.xml
      restoreProgItems();
      // add PREPARE-scripts to changed nodes
      createProgItems(true);
   
      //Mark the Scope as a Reference in Recovery Mode For different phase logic
      if(Config::instance()->cfg_EnterRecovery.readAsBool() || Config::instance()->cfg_EnterEngineering.readAsBool()) {
        BoschXmlNode boschXmlNode(_progressReleaseSection);
        evalScopes(boschXmlNode);
      }

   }
   store();
   return true;
}

void CtrlProgressSection::vProcess(Msg_CurrentBoschXmlChanged *msg) {
   bool valid=msg->_currentBoschXml->isValid();

   switch (_enState) {
      case enState_Restoring: {
         if (valid) {
            TiXmlElement *overallXml=msg->_currentBoschXml->getOverallXml();
            SWU_ASSERT_RETURN(overallXml);
            if (restoreCurrentRelease(overallXml)) {
               enterState(enState_WaitStart);
            }
         }
         break;
      }
      case enState_WaitBoschXml: {
         if (valid) {
            TiXmlElement *overallXml=msg->_currentBoschXml->getOverallXml();
            SWU_ASSERT_RETURN(overallXml);
            // copy new overall-section, media-path might have changed
            setOverallSection(overallXml);
            
            enterState(enState_Running);
            // todo: notify outer world
         }
         break;
      }
      case enState_Running: {
         if (valid) {
            TiXmlElement *overallXml=msg->_currentBoschXml->getOverallXml();
            SWU_ASSERT_RETURN(overallXml);
            // copy new overall-section, media-path might have changed
            setOverallSection(overallXml);
         }
         break;
      }

      default:
         break;
   }

}

CtrlProgressSection::tenState CtrlProgressSection::getState() {
   ETG_TRACE_ERR(("CtrlProgressSection::getState: %u", _enState));
   return _enState;
}


bool CtrlProgressSection::requestFullUpdate()  {
   if (!_progressReleaseSection) {
      return false;
   }
   setSpecialMode("ENTER_FULL_UPDATE");
   setRecoveryNeeded();
   finalizeRelease();
   return true;
}

bool CtrlProgressSection::isStoredToHistory()  {
   return swu::getUIntFromChild(_progressReleaseSection, "STORED_TO_HISTORY");
}

void CtrlProgressSection::setIsStoredToHistory() {
   swu::setUIntChild(_progressReleaseSection, "STORED_TO_HISTORY", 1);
}

bool CtrlProgressSection::exists() {
   // returns when a progress-file could be restored.
   // it will still no be connected to a current release.
   // this will be broadcasted via Msg_ProgressSectionChanged
   bool res=_progressFile->exists();
   ETG_TRACE_COMP(("CtrlProgressSection::exists() res=%u", res));
   return res;
}


bool CtrlProgressSection::isLastScomoPhase() {
   return getRequestedScomoPhase()==getMaxScomoPhase();
}

bool CtrlProgressSection::needsSwitchToRecovery() {
   bool res=false;
   if (!Config::instance()->cfg_EnterFullOperation.get()) {
      //      res= false;
   }
   else if (!_doHandleRecoveryMode) {
      //      res = false;
   }
   else if (isScomoMode() &&  !isLastScomoPhase()) {
      return false;
   }
   else if (_recoveryNeeded) {
      res = true;
   }
   else if (isCompatUpdate()) {
      res =true;
   }
   else if (_currentCtrlProgItem.needsRecoveryMode()) {
      res = true;
   }

   ETG_TRACE_ERR(("CtrlProgressSection::needsSwitchToRecovery() _recoveryNeeded=%u res=%u", 
                  _recoveryNeeded, res));
   return res;
}

void CtrlProgressSection::handleRecoveryMode(bool doHandle) {
   _doHandleRecoveryMode=doHandle;
}

bool CtrlProgressSection::isUpdateDone()  {
   return swu::getUIntFromChild(_progressReleaseSection, "UPDATE_DONE");
}
   
void CtrlProgressSection::setUpdateDone(unsigned int updateResult) {
   swu::setUIntChild(_progressReleaseSection, "UPDATE_DONE", updateResult);
    if (_progressFile->getXmlRoot() && _persistentConfig->usePersistency.get()) {
         _progressFile->store();
      } 
}

void CtrlProgressSection::finalizeRelease() {
   ETG_TRACE_ERR(("CtrlProgressSection::finalizeRelease START"));
   // set all remaining items to failed
   _silent=true;

   if (_progressReleaseSection) {
      for (;_currentCtrlProgItem.isValid();setNextItem()) {
         if (_currentCtrlProgItem.getState()=="pending" || _currentCtrlProgItem.getState()=="running") {
            _currentCtrlProgItem.finalize(false);
         }
      } 

      ETG_TRACE_ERR(("CtrlProgressSection::finalizeRelease call evalScopes"));
      BoschXmlNode boschXmlNode(_progressReleaseSection);
      evalScopes(boschXmlNode);
   }

   _silent=false;
   store();
   ETG_TRACE_ERR(("CtrlProgressSection::finalizeRelease END"));
}


void CtrlProgressSection::setOverallSection(TiXmlElement *overallSection) {
   SWU_ASSERT_RETURN(overallSection);
   SWU_ASSERT_RETURN(_overallSection);
   SWU_ASSERT_RETURN(_progressRoot);
   _overallSection=_progressRoot->ReplaceChild(_overallSection, *overallSection)->ToElement();
}

void CtrlProgressSection::cleanup() {
   ETG_TRACE_ERR(("CtrlProgressSection:cleanup() START _progressRoot=%p", _progressRoot));
   _progressRoot=0;
   _progressSection=0;
   _progressReleaseSection=0;
   _overallSection=0;
   _decisionsSection=0;
   ETG_TRACE_ERR(("CtrlProgressSection:cleanup() _releaseSection=%p", _releaseSection));
   swu::deleteAndZero(_releaseSection);
   ETG_TRACE_ERR(("CtrlProgressSection:cleanup() END"));
}

void CtrlProgressSection::startItem(CtrlProgItem *progItem) {
      if (_lastStartedRefKey != _currentCtrlProgItem.getRefKey()) {
         _lastStartedRefKey = _currentCtrlProgItem.getRefKey();
         _curRetries=0;
      } else {
         _curRetries++;
      }
      _startingItemRefKey=progItem->getRefKey();
      ETG_TRACE_ERR(("CtrlProgressSection:startItem() START _startingItemRefKey=%s", _startingItemRefKey.c_str()));
      store();
      ETG_TRACE_ERR(("CtrlProgressSection:startItem() END _startingItemRefKey=%s", _startingItemRefKey.c_str()));
      _startingItemRefKey="";
   }

void CtrlProgressSection::startProgressTimer(tUInt sec) {
   ETG_TRACE_USR1(("CtrlProgressSection:startProgressTimer(%u)", sec));
   if(_currentBoschXmlStick->isValid()) {
      _progressTimer.start(this, sec * 1000);
   }
}
void CtrlProgressSection::stopProgressTimer() {
   ETG_TRACE_USR1(("CtrlProgressSection:stopProgressTimer()"));
   _progressTimer.stop();
}


void CtrlProgressSection::notifyClients() {
   // async notification. client may delete progress-section on receipt
   Msg_ProgressSectionChanged *msg=new Msg_ProgressSectionChanged;
   msg->bNotifyLater();
   msg=0;
}

void CtrlProgressSection::enterState(tenState enState) {
   ETG_TRACE_ERR(("CtrlProgressSection::enterState(%u->%u) START",
                  ETG_CENUM(CtrlProgressSection::tenState, _enState), ETG_CENUM(CtrlProgressSection::tenState, enState)));
   tenState oldState=_enState;
   if (_enState != enState) {
      _enState=enState;
      if (_enState == enState_Initial ||
          _enState == enState_Restoring) {
         // no progress-file available
         notifyClients();
      }
      else if (_enState == enState_Done) {
         // no progress-file available
         notifyClients();
         cleanup();
      }
      else {
         // state will be derived from progress-
         store();
      }
   }
   ETG_TRACE_ERR(("CtrlProgressSection::enterState() END newState=%u", ETG_CENUM(CtrlProgressSection::tenState, _enState)));   
}


TiXmlElement *CtrlProgressSection::nextUpdatedBoschXmlItem(TiXmlElement *boschXmlItem, bool onEnter) {
   // we have to iterate boschxml
   ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() boschXmlItem=%p onEnter=%u START", boschXmlItem, onEnter));
   bool first=true;
   for (XmlItemAccessIter iter(_progressReleaseSection, XmlItemAccessIter::enOnBoth, boschXmlItem, onEnter);
        !iter.isDone(); 
        iter.next()) {
      if (first) {
         first=false;
         ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() skip first"));
         continue;
      }
      BoschXmlNode access(iter);
      ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() START item:%s", access.getLongName().c_str()));

      if ((tUInt)access.enGetType()>(tUInt)XmlItemAccess::enType_Submodule) {
         ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() skip  > Submodule"));
         continue;
      }
      if (!swu::getUIntFromChild(access.getInfoItem(), "DO_UPDATE")) {
         ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() skip DO_UPDATE==false"));
         continue;
      }

      bool onEnter=(XmlItemAccessIter::enEnter==iter.getDirection());

      for (TiXmlElement  *item=access.getItemsItem()->FirstChildElement("ITEM");
           item;
           item=item->NextSiblingElement("ITEM")) {
         ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() check item=%p", item));
         if (onEnter==CtrlProgItem::flashOnEnter(swu::getTextFromChild(item, "TYPE"))) {
            if (std::string(swu::getTextFromChild(item, "STATE")) != "ok") {
               ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() END found item=%p", item));
               return item;
            }
         }
      }
   }
   ETG_TRACE_ERR(("CtrlProgressSection::nextUpdatedBoschXmlItem() END found nothing"));
   return 0;
}

bool CtrlProgressSection::isLastOf(XmlItemAccess::tenType level) {
   CtrlProgItem next=peekNextItem();
   XmlItemAccess::tenType curLevel=_currentCtrlProgItem.getLevel();
   if (!next.isValid()) {
      return true;
   }
   XmlItemAccess::tenType nextLevel=next.getLevel();
   bool res=false;
   if (nextLevel<level) {
      //e.g. level=MODULE cur=IMX/MAIN/APPLICATION next=IMX
      res= true;
   }
   else if (curLevel<level) {
      //e.g. level=APPLICATION cur=IMX/MAIN
      ETG_TRACE_ERR(("CtrlProgressSection::isLastOf:called for invalid level"));
      res= true;
   }
   else {
      res = XmlItemAccess(_currentCtrlProgItem._progBoschXmlElem).getLongName(level)!=XmlItemAccess(next._progBoschXmlElem).getLongName(level);
   }
   ETG_TRACE_ERR(("CtrlProgressSection::isLastOf:res=%u level=%u, cur=%30s next=%s", 
                  res, (tU32)level, _currentCtrlProgItem.getName().c_str(), next.getName().c_str()));
   return res;   
}


void CtrlProgressSection::skip(XmlItemAccess::tenType level) {
   ETG_TRACE_ERR(("CtrlProgressSection::skip (level=%u) START", (tUInt)level));

   while(!isLastOf(level)) {
      setNextItem();
      ETG_TRACE_ERR(("CtrlProgressSection::skipping %s", _currentCtrlProgItem.getName().c_str()));
      _currentCtrlProgItem.finalize(false);
   }
   notifyClients();
   ETG_TRACE_ERR(("CtrlProgressSection::skip END"));
}

void CtrlProgressSection::abort() {
   ETG_TRACE_ERR(("CtrlProgressSection::abort() START"));
   _abort=true;
   _silent=true;
   while(_progressRoot && setNextItem().isValid()) {
      ETG_TRACE_ERR(("CtrlProgressSection::aborting %s", _currentCtrlProgItem.getName().c_str()));
      _currentCtrlProgItem.finalize(false);
   }
   _silent=false;
   notifyClients();
   ETG_TRACE_ERR(("CtrlProgressSection::abort() END"));
}

bool CtrlProgressSection::isCompatUpdate() const {
   return _isCompatUpdate;
}

bool CtrlProgressSection::isPostImageMode() const {
   return _overallSection && swu::getUIntFromChild(_overallSection, "POST_IMAGE_MODE") 
      || Config::instance()->cfg_PostImageInstallationRunning.get();
}


bool CtrlProgressSection::isPartialRelease() const {
   return _progressReleaseSection && swu::getUIntFromChild(_progressReleaseSection, "PARTIAL_RELEASE");
}
bool CtrlProgressSection::isFullUpdate() const {
   return _progressReleaseSection && swu::getUIntFromChild(_progressReleaseSection, 
                                                           "FULL_UPDATE");
}

std::string CtrlProgressSection::getMediaPath() {
   std::string mediaPath;
   if (!_overallSection) {
      return mediaPath;
   }
   if (Config::instance()->cfg_EnterEngineering.get()) {
      mediaPath="/tftp@" + Config::instance()->cfg_TftpServerIpAdr.get() + ":";
      std::string basepath = swu::getTextFromChildOrEmpty(_overallSection, "BASEPATH");
      ETG_TRACE_USR1(("getMediaPath:Found Base Path : %s", basepath.c_str()));
      if ((basepath.size() > 0) && (basepath != ".")) {
         mediaPath += basepath;
         if (mediaPath[mediaPath.length() - 1] != '/') {
            mediaPath += "/";
         }
      }

   }
   else {
      mediaPath += getMediaPathFromOverallSection(_overallSection);
   }
   ETG_TRACE_ERR(("CtrlProgressSection::getMediaPath(): %s", mediaPath.c_str()));
   return mediaPath;
}


void CtrlProgressSection::setRequestedScomoPhase(tU32 phase) {
   swu::setUIntChild(_progressSection, "REQUESTED_SCOMO_PHASE", phase);
}
tU32 CtrlProgressSection::getRequestedScomoPhase() {
   return swu::getUIntFromChild(_progressSection, "REQUESTED_SCOMO_PHASE");
}

std::string CtrlProgressSection::getComponentId() {
   return swu::getTextFromChild(_progressSection, "COMPID", false, "unknown");
}

void CtrlProgressSection::setComponentId(std::string compId) {
   swu::setTextChild(_progressSection, "COMPID", compId);
}

void CtrlProgressSection::setSubModuleListConfiguredToUpdate(std::string subModuleList) {                      
   swu::setTextChild(_progressSection, "SUBMODULE_CONFIGURED_TOUPDATE", subModuleList);
}

std::string CtrlProgressSection::getSubModuleListConfiguredToUpdate() {    
   return swu::getTextFromChildOrEmpty(_progressSection, "SUBMODULE_CONFIGURED_TOUPDATE");
}

tenSourceType CtrlProgressSection::getSourceType()
{
   tenSourceType sourceType = tenSourceType(swu::getUIntFromChild(_overallSection, "SOURCE_TYPE", tUInt(tenSourceType_other)));
   ETG_TRACE_USR4(("CtrlProgressSection::getSourceType(): %u", ETG_CENUM(tenSourceType, sourceType)));
   return sourceType;
}

bool CtrlProgressSection::isOtaUpdate()
{
   bool res=(getSourceType() == tenSourceType_OTA) || (getSourceType() == tenSourceType_SCOMO) || (getSourceType() == tenSourceType_SCOMO_INDEX);
   ETG_TRACE_USR4(("CtrlProgressSection::isOtaUpdate(): %u", res));
   return res;
}


CtrlProgItem CtrlProgressSection::peekNextItem(CtrlProgItem curItem) {
   ETG_TRACE_USR4(("CtrlProgressSection::peekNextItem() START"));
   CtrlProgItem next=nextItem(curItem);
   ETG_TRACE_USR4(("CtrlProgressSection::peekNextItem() END"));
   return next;
}

CtrlProgItem CtrlProgressSection::setNextItem(bool firstItem) {
   ETG_TRACE_USR4(("CtrlProgressSection::setNextItem() firstItem=%u START", firstItem));
   
   if(!firstItem) {
      if (_currentCtrlProgItem.getState()=="pending") {
         _curRetries++;
         ETG_TRACE_USR4(("CtrlProgressSection::setNextItem() END (retry)"));
         return _currentCtrlProgItem;
      }

      _curRetries=0;
      _currentCtrlProgItem=nextItem(_currentCtrlProgItem);
   }

   if (_currentCtrlProgItem.isPhaseStart()) {
      storePhaseStart();
   }

   /* - evaluate new scomo-phase.
      - store evaluated phase persistently
      - enter State "ScomoPending"
    */
   ETG_TRACE_USR1(("CtrlProgressSection::setNextItem, abort:%u", _abort));
   if (!_abort && isScomoMode()) {
      tU32 requestedScomoPhase=getRequestedScomoPhase();
      switch((tenScomoPhaseHandlingMode)Config::instance()->cfg_ScomoPhaseHandlingMode.get()) {
         case tenScomoPhaseHandlingMode_Default: {
            // scomo-phase has already been configured during initialization, all items needed for this phase
            // should be in state pending
            break;
         }
         case tenScomoPhaseHandlingMode_1None2Neutral3All: {
            // in this mode, only one scomo-phase should be configured
            ETG_TRACE_USR4(("CtrlProgressSection::setNextItem: requestedScomoPhase:%u getConfiguredScomoPhases:%u", 
                            requestedScomoPhase, swu::stringToUInt(_currentCtrlProgItem.getConfiguredScomoPhases())));
            if (requestedScomoPhase < swu::stringToUInt(_currentCtrlProgItem.getConfiguredScomoPhases())) {
               enterState(enState_ScomoPending);               
               store();         
            }
            break;
         }
         default:
            break;
            
      }
   }
   if(firstItem) {      
      if (_currentCtrlProgItem.isValid() && _currentCtrlProgItem.getState() != "pending") {
         _currentCtrlProgItem.scheduleRetry();
      }
   }

   ETG_TRACE_USR4(("CtrlProgressSection::setNextItem() END"));
   return _currentCtrlProgItem;
}
// todo: rework to have version with currentItem as argument
   CtrlProgItem CtrlProgressSection::nextItem(CtrlProgItem curItem) {
   ETG_TRACE_USR4(("CtrlProgressSection::nextItem() START"));

   CtrlProgItem next=curItem;
   do  {
      if (!curItem._valid) {
         break;
      }
      ETG_TRACE_USR4(("CtrlProgressSection::nextItem() index=%u _subItemIndexMax=%u current=%s", 
                      curItem.getSubItemIndex(), curItem._subItemIndexMax, 
                      curItem.getName().c_str()));
      
      bool onEnter=curItem.flashOnEnter();
      ETG_TRACE_USR4(("CtrlProgressSection::nextItem: onEnter=%u search for Sibling", onEnter));
      bool found=false;
      for (TiXmlElement *siblingXml=curItem.getProgressItem()->NextSiblingElement("ITEM");
           siblingXml;
           siblingXml=siblingXml->NextSiblingElement("ITEM")) {
         ETG_TRACE_USR4(("CtrlProgressSection::nextItem: check Sibling"));
         if (onEnter==CtrlProgItem::flashOnEnter(swu::getTextFromChild(siblingXml, "TYPE"))) {
            // there is another item connected to this node:<MOD><INFO><ITEMS><ITEM++><0>
            next=CtrlProgItem(this, siblingXml);
            ETG_TRACE_USR4(("CtrlProgressSection::nextItem: check Sibling: found!"));
            found=true;
            break;
         }
      }
      if (found) {
         break;
      }
      ETG_TRACE_USR4(("CtrlProgressSection::nextItem: search for next boschXmlItem"));
      TiXmlElement *nextXml=nextUpdatedBoschXmlItem(curItem.getProgBoschXmlItem(), onEnter);
      if (nextXml) {
         next=CtrlProgItem(this, nextXml);
         ETG_TRACE_USR4(("CtrlProgressSection::nextItem: found next boschXmlItem"));
         break;
      }
      next._valid=false;
      ETG_TRACE_USR4(("CtrlProgressSection::nextItem: No Items left!"));
   } 
   while (0);
   if (next._valid) {
      ETG_TRACE_USR4(("CtrlProgressSection::nextItem() END boschXmlItem=%30s (%u)", 
                      next.getName().c_str(), next.getSubItemIndex()));
   }
   return next;

}


void CtrlProgressSection::evalCompatIndex() {
   // evaluate needed compat-index
   std::map< std::string, std::string>relExtraArgs;
   BXmlAccess xmlAccess(_releaseSection);
   fetchExtraArgsRecurse(xmlAccess.getModuleSection(), relExtraArgs);

   tUInt neededSwuCompatIndex=0;
   if (relExtraArgs.find("neededSwuCompatIndex") != relExtraArgs.end()) {
      neededSwuCompatIndex=atoi(relExtraArgs["neededSwuCompatIndex"].c_str());
   }
   // evaluate, if we have an compat-update
   _isCompatUpdate=((Config::instance()->cfg_SwuCompatIndex.get()< neededSwuCompatIndex) && getBxmlVersion());
   if (_isCompatUpdate) {
      swu::setTextChild(_progressReleaseSection, "SPECIAL_MODE", "COMPAT_UPDATE");
   }
   ETG_TRACE_USR1(("CtrlProgressSection::evalCompatIndex(): _isCompatUpdate=%u", _isCompatUpdate));
}

void CtrlProgressSection::setInitialRelease(TiXmlDocument *boschXmlDocIn, CtrlPersistentConfig *persistentConfig) {
   ETG_TRACE_USR4(("CtrlProgressSection::setInitialRelease START enMode=%u", persistentConfig->enInitialMode.get()));
      
   SWU_ASSERT_RETURN(persistentConfig);
   _persistentConfig=persistentConfig;
  
   // access the boschxml from configurator
   TiXmlElement *boschXmlRoot=boschXmlDocIn->FirstChildElement();;
   
   // store the received release section (will never be stored persistently)
   _releaseSection=swu::Swu_UnlinkNode(boschXmlRoot->FirstChildElement("RELEASE"))->ToElement();
   BXmlAccess relAccess(_releaseSection);

   // populate progress-section
   // insert root-node
   _progressRoot=_progressFile->getXmlDoc().InsertEndChild(TiXmlElement("UPDATEXML"))->ToElement();
   // steal overall-section from boschXmlDocIn
   _overallSection=swu::Swu_UnlinkNode(boschXmlRoot->FirstChildElement("OVERALL"))->ToElement();

   //FINALNAME is not there in Bosch Overall, so take it from Module and Place in ProgressFile overall
   swu::setTextChild(_overallSection, "FINALNAME", relAccess.getTextFromModule("FINALNAME", false, "unknown"));  

   ETG_TRACE_USR1(("CtrlProgressSection::setInitialRelease sourceType=%u", (tUInt)getSourceType()));
   

   if (!_progressRoot->LinkEndChild(_overallSection)) {
      // make Coverity happy, since LinkEndChild return 0 if  _overallSection is of type TINYXML_DOCUMENT
      SWU_ASSERT_RETURN_ALWAYS();
   }
   ETG_TRACE_USR1(("CtrlProgressSection::setInitialRelease 2 sourceType=%u", (tUInt)getSourceType()));
   // steal decisions-section from boschXmlDocIn
   if (boschXmlRoot->FirstChildElement("DECISIONS")) {
      _decisionsSection=swu::Swu_UnlinkNode(boschXmlRoot->FirstChildElement("DECISIONS"))->ToElement();
      _progressRoot->LinkEndChild(_decisionsSection);
   }
   
   // create section PROGRESS
   _progressSection=_progressRoot->InsertEndChild(TiXmlElement("PROGRESS"))->ToElement();
   _persistentConfig->store(_progressSection);
#if 0
   swu::setUIntChild(_progressSection, "INITIAL_MODE", persistentConfig.enInitialMode);
   swu::setUIntChild(_progressSection, "NEEDS_RESULT_IN_HMI_MODE", 1);
   swu::setUIntChild(_progressSection, "USE_PERSISTENCY", usePersistency ? 1 : 0);
#endif
   swu::setUIntChild(_progressSection, "UPDATE_MODE", persistentConfig->enInitialMode.get()); // todo: where to fetch update-mode?
   setPersistentState("pending");

   TiXmlElement *curRelease=collectProgItems();
   
   /*
     "FROMVERSION"
     "FROMVERSION_CUSTOMER"
     "FROMVERSION_LABEL"
     
     "TOVERSION"
     "TOVERSION_CUSTOMER"
     "TOVERSION_LABEL"
   */
   swu::setTextChild(curRelease, "FROMVERSION_LABEL", Config::instance()->cfg_RunningSwLabel.get());
   swu::setTextChild(curRelease, "FROMVERSION", Config::instance()->cfg_RunningSwTrain.get());
   swu::setTextChild(curRelease, "FROMVERSION_CUSTOMER", Config::instance()->cfg_RunningCustomerVersion.get());
   swu::setTextChild(curRelease, "TOVERSION_LABEL", relAccess.getTextFromModule("BUILD_LABEL", false, "unknown"));
   swu::setTextChild(curRelease, "TOVERSION", relAccess.getTextFromModule("FINALNAME", false,  "unknown"));
   swu::setTextChild(curRelease, "TOVERSION_CUSTOMER", relAccess.getTextFromModule("FINALNAME_CUSTOMER", false, "unknown"));
   swu::setTextChild(curRelease, "CISSWVERSIONNAME", relAccess.getTextFromModule("CISSWVERSIONNAME", false, "unknown"));
   
   _progressReleaseSection=curRelease;
   evalCompatIndex();

   _progressSection->LinkEndChild(curRelease);

   if(isScomoMode()) {      
      setRequestedScomoPhase(Config::instance()->cfg_ScomoPhase.get());
      setComponentId(CampaignManager::instance()->getScomoComponentId());
   }

   ETG_TRACE_USR4(("CtrlProgressSection::setInitialRelease END"));

}


tenSwUpdateMode CtrlProgressSection::getInitialMode() const {
   SWU_ASSERT_RETURN_VAL(_persistentConfig, tenSwUpdateMode_Invalid);
   return _persistentConfig->enInitialMode.get();
}



void CtrlProgressSection::configure() {

   // store that we have a partial-release
   bool isPartialRelease=swu::getUIntFromChild(_releaseSection, "PARTIAL_RELEASE");
   swu::setUIntChild(_progressReleaseSection, "PARTIAL_RELEASE",  isPartialRelease ? 1 : 0);

   ETG_TRACE_USR4(("CtrlProgressSection::configure START defaultDecision=%u isCompatUpdate()=%u isPartialRelease=%u", _defaultDecision, _isCompatUpdate, isPartialRelease));

   ETG_TRACE_USR4(("CtrlProgressSection::configure(): find submodules to update:"));
   std::string subModList;
   // evaluate for all submodules, if an update is configured
   for (XmlItemAccessIter iterChildrenFirst(_releaseSection, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      BoschXmlNode access(iterChildrenFirst);
      ETG_TRACE_USR4(("CtrlProgressSection::configure(): check type: %u", (tU32)access.enGetType()));

      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         TiXmlElement *subModule=access.getXml();
         ETG_TRACE_USR4(("CtrlProgressSection::configure(): check submodules: to update: %s", access.getLongName().c_str()));

         if (isSubmoduleUpdateConfigured(subModule)) {
            std::string subModuleName = access.getLongName();
            size_t index = subModuleName.find_last_of('/');
            if (index != std::string::npos)
            {
               subModuleName =  subModuleName.substr(index + 1);                
               ETG_TRACE_USR1(("CtrlProgressSection::configure append subModulename: %s",
                         subModuleName.c_str()));

               subModList = subModList + subModuleName + ",";
            }

            ETG_TRACE_USR4(("CtrlProgressSection::configure():  update %s", access.getLongName().c_str()));
            BoschXmlNode access2(_progressReleaseSection, access.getLongName());
            CtrlProgItem progItem=CtrlProgItem(this, access2.getXml());
            if (isScomoModeNormal() && !progItem.needsCurrentScomoPhase()) {
               // todo: check if I get the correct xml-element
               progItem.setState("ok");
            } else {
               bubbleUpDoUpdate(access2);
            }
         }
      }
   }

   std::string subModulelistConfigured = getSubModuleListConfiguredToUpdate();
   if ( subModulelistConfigured.empty() && !subModList.empty())
   {
      setSubModuleListConfiguredToUpdate(subModList);
   }

   if (!isCompatUpdate()) {      
      // evaluate other submodules that have to be updated because of dependencies
      // following methods will return true, when no more changes have been applied
      while (!handleGroups() || !addDependencies()) {
         continue;
      }
   }

   
   if (_changedGroups.find("BOOTCHAIN")!= _changedGroups.end() || isCompatUpdate()) {
      swu::setTextChild(_progressReleaseSection, "SWITCH_BOOTCHAIN", "1");      
      swu::setTextChild(_progressReleaseSection, "FIRMWARE_UPDATE", "1");
   }

   // add scripts to changed items
   createProgItems();
   
   //Mark the Scope as a Reference in Recovery Mode For different phase logic
   if(Config::instance()->cfg_EnterRecovery.readAsBool() || Config::instance()->cfg_EnterEngineering.readAsBool()) {
     BoschXmlNode boschXmlNode(_progressReleaseSection);
     evalScopes(boschXmlNode);
   }
}

void CtrlProgressSection::create(TiXmlDocument *boschXmlDocIn, CtrlPersistentConfig *persistentConfig, 
                                 CtrlStaticConfig *staticConfig )
{
   ETG_TRACE_USR4(("CtrlProgressSection::create() START"));
   // create new progress-section from input boschXml
   _staticConfig=staticConfig;
   setInitialRelease(boschXmlDocIn, persistentConfig);
   if (_staticConfig->defaultDecision!=update_decision_setting_type__invalid) {
      _defaultDecision=_staticConfig->defaultDecision;
   }
   configure();

#ifdef LOCAL_BUILD_
   TiXmlDocument tmpDoc;
   tmpDoc.InsertEndChild(*_releaseSection);
   tmpDoc.SaveFile(TMP_RELEASE_FILE);
#endif
   subscribeForStick();
   enterState(enState_WaitStart);
   ETG_TRACE_USR4(("CtrlProgressSection::create() END"));

}

bool CtrlProgressSection::startUpdate() {
   // todo: assert for correct state
   SWU_ASSERT_RETURN_FALSE(_enState==enState_WaitStart);
   enterState(enState_Running);

   TiXmlElement *next=nextUpdatedBoschXmlItem();
   setPersistentState("running");
   _currentCtrlProgItem=CtrlProgItem(this, next);
   setNextItem(true);
   store();
   return _currentCtrlProgItem.isValid();
}




void CtrlProgressSection::evalScopes(BoschXmlNode &boschXmlNode) {
   ETG_TRACE_ERR(("CtrlProgressSection::evalScopes _xmlItem=%p isRoot()=%u type=%u isGood=%u START (%s)", 
                  boschXmlNode.getXml(), 
                  boschXmlNode.isRoot(), 
                  boschXmlNode.enGetType(), 
                  boschXmlNode.isGood(),
                  boschXmlNode.getLongName().c_str()));

   SWU_ASSERT_RETURN(boschXmlNode.isGood());
   // local scope
   bool updated=false;
   bool hasErrors=false;
   bool hasUnchangedChildren=false;
   Scope::tenScope scope;

   // check items directly connected to this node
   for (TiXmlElement *item=boschXmlNode.getItemsItem()->FirstChildElement("ITEM");
        item;
        item=item->NextSiblingElement("ITEM")) {
      std::string state=swu::getTextFromChild(item, "STATE", false, "NO");
      if (state=="failed") {
         updated=true;
         hasErrors=true;
      } else if (state=="ok") {
         updated=true;
      }
   }

   // check children
   ETG_TRACE_ERR(("CtrlProgressSection::evalScopes check children START (%s)", boschXmlNode.getLongName().c_str()));
   if ((tUInt)boschXmlNode.getAccess().enGetType()<(tUInt)XmlItemAccess::enType_Submodule) {
      std::list<TiXmlElement *> childenList=boschXmlNode.getAccess().getChildren();
      for (std::list<TiXmlElement *>::iterator iter=childenList.begin();
           iter!=childenList.end();
           ++iter) {
         ETG_TRACE_ERR(("CtrlProgressSection::evalScopes check child START"));

         BoschXmlNode boschXmlNodeChild(*iter);
         ETG_TRACE_ERR(("CtrlProgressSection::evalScopes check child: recurse for %s", boschXmlNodeChild.getLongName().c_str()));
         // recursion: eval sope of child
         evalScopes(boschXmlNodeChild);
         Scope::tenScope childSope=(Scope::tenScope)swu::getUIntFromChild(boschXmlNodeChild.getXml(), "SCOPE", (tUInt)Scope::enScope_None);
         if (childSope==Scope::enScope_No) {
            hasUnchangedChildren=true;
         }
         else if (childSope==Scope::enScope_Yes) {
            updated=true;
         }
         else if (childSope==Scope::enScope_NOK) {
            updated=true;
            hasErrors=true;
         }
         else if (childSope==Scope::enScope_Some) {
            updated=true;
            hasUnchangedChildren=true;
         }
         ETG_TRACE_ERR(("CtrlProgressSection::evalScopes check child END (%s)", boschXmlNodeChild.getLongName().c_str()));

      }
   }
   ETG_TRACE_ERR(("CtrlProgressSection::evalScopes check children END (%s)",boschXmlNode.getLongName().c_str()));

   Scope::tenScope res=Scope::enScope_No;
   if (hasErrors) {
      res=Scope::enScope_NOK;
   }
   else if (updated && hasUnchangedChildren) {
      res=Scope::enScope_Some;
   }
   else if (updated) {
      res=Scope::enScope_Yes;
   }
   swu::setUIntChild(boschXmlNode.getXml(), "SCOPE", (tUInt)res);
   ETG_TRACE_ERR(("CtrlProgressSection::evalScopes END (%s)", boschXmlNode.getLongName().c_str()));

}


void CtrlProgressSection::fetchExtraArgs(TiXmlElement *xmlElem, std::map< std::string, std::string>&res)
{
   //ETG_TRACE_USR4(("fetchExtraArgs() START")); 
   
   for (TiXmlElement *extraArg = xmlElem->FirstChildElement("EXTRA_ARG"); extraArg; extraArg = extraArg->NextSiblingElement("EXTRA_ARG"))
   {
      char const *key = swu::getTextFromChild(extraArg, "KEY", false, 0);
      char const *value = swu::getTextFromChild(extraArg, "VALUE", false, 0);
      if (key && value)
      {
         ETG_TRACE_USR4(("fetchExtraArgs() ADD: key=%30s value=%s", key, value));
         res[key] = value;
      }
   }
}


void CtrlProgressSection::fetchExtraArgsRecurse(TiXmlElement *xmlElem, std::map< std::string, std::string>&res) {
   //ETG_TRACE_USR4(("fetchExtraArgsRecurse() START")); 
   XmlItemAccess access(xmlElem);
   ETG_TRACE_USR4(("fetchExtraArgsRecurse() I am (%u) %s", access.enGetType(), access.getName()));
   if (access.isRelease()) {
      ETG_TRACE_USR4(("fetchExtraArgsRecurse(): adding extra-args from overall-section"));
      fetchExtraArgs(_overallSection, res);
   } else {
      if (access.isDevice()) {
         res["bxml_device"]=swu::toLower(access.getName());
         fetchExtraArgs(access.getXml(), res);
      }
      else if (access.isModule()) {
         res["bxml_module"]=swu::toLower(access.getName());
         fetchExtraArgs(access.getXml(), res);
      }
      else if (access.isSubModule()) {
         res["bxml_submodule"]=swu::toLower(access.getName());
         res["SubModuleVersion"]=swu::getTextFromChild(access.getXml(), "VERSION");
         BoschXmlNode access2(_progressReleaseSection, access.getLongName());

         res["SubModuleFromVersion"]=swu::getTextFromChild(access2.getXml(), "FROMVERSION", false, "unknown");
      }
      ETG_TRACE_USR4(("fetchExtraArgsRecurse() call fetchExtraArgs for parent"));
      TiXmlElement *parent=xmlElem->Parent()->ToElement();
      fetchExtraArgsRecurse(parent, res);
   }
   ETG_TRACE_USR4(("fetchExtraArgsRecurse() call fetchExtraArgs for self"));
   fetchExtraArgs(xmlElem, res);
   //ETG_TRACE_USR4(("fetchExtraArgsRecurse() DONE")); 
}


update_decision_setting_type CtrlProgressSection::getDecisionFromDecisionSection(string longName) {
   update_decision_setting_type configDecision=update_decision_setting_type__invalid;
   if (! _decisionsSection) {
      ETG_TRACE_USR4(("CtrlProgressSection::getDecisionFromDecisionSection(): _decisionsSection missing!"));
      return configDecision;
   }

   const char *defaultSettingString = swu::getTextFromChildOrEmpty(_decisionsSection, "DEFAULT_MODE");

   configDecision = CtrlUtil::decisionFromString(defaultSettingString);
   ETG_TRACE_USR4(("CtrlProgressSection::getDecisionFromDecisionSection(): configDecision=%u DEFAULT_MODE=%s", configDecision, defaultSettingString));

   TiXmlElement *currentDecision = _decisionsSection->FirstChildElement("DECISION");
   while (currentDecision) {
      std::string module = swu::getTextFromChild(currentDecision, "MODULE");
      if (module==longName) {
         std::string setting = swu::getTextFromChild(currentDecision, "SETTING");
         update_decision_setting_type decision = CtrlUtil::decisionFromString(setting);
         ETG_TRACE_USR4(("CtrlProgressSection::getDecisionFromDecisionSection(): found decision=%u for=%s", decision, module.c_str()));

         if (decision != update_decision_setting_type__invalid) {
            configDecision=decision;
         }
         break;
      }
      currentDecision = currentDecision->NextSiblingElement("DECISION");
   }
   ETG_TRACE_USR4(("CtrlProgressSection::getDecisionFromDecisionSection(%50s): res=%u",
                   longName.c_str(), configDecision));

   return configDecision;
}

update_decision_setting_type CtrlProgressSection::getNodeDecision(TiXmlElement *node) {
   char const *nodeDecisionStr = swu::getTextFromChildOrEmpty(node, "UPDATE_MODE");
   update_decision_setting_type nodeDecision = CtrlUtil::decisionFromString(nodeDecisionStr);
   return nodeDecision;   
}

update_decision_setting_type CtrlProgressSection::getReleaseDecision(TiXmlElement *node) {
   update_decision_setting_type decision=update_decision_setting_type__invalid;
   XmlItemAccess access(node);
   while (1)  {
      decision=getNodeDecision(access.getXml());
      if (decision != update_decision_setting_type__invalid) {
         break;
      }
      if (access.isRelease()) {
         break;
      }
      access=access.getParent();
   }
   return decision;
}

bool CtrlProgressSection::isSubmoduleUpdateConfigured(TiXmlElement *subModule) {

   bool res=false;

   XmlItemAccess access(subModule);
   std::string longName=access.getLongName();
   std::string strSpecialMode = getSpecialMode();  //Coverity fix for 209494
   ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured(%50s): isCompatUpdate=%u usedForCompatUpdate=%u isPostImageMode=%u specialMode=%20s isScomMode:%u", 
                   longName.c_str(), 
                   isCompatUpdate(), 
                   swu::getUIntFromChild(subModule, "COMPAT_UPDATE"),
                   isPostImageMode() ,
                   strSpecialMode.c_str(),
                   isScomoMode()));

   if (isCompatUpdate()) {
      res = swu::getUIntFromChild(subModule, "COMPAT_UPDATE") ? true : false;
   } else if (isFullUpdate()) {
      res = true;
   }
   else if (isPostImageMode()) {
      res = true;
   }
   else if (isScomoMode()) {
      res = true;
   }
   else {

      update_decision_setting_type resSetting=getDecisionFromDecisionSection(longName);
      ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():getDecisionFromDecisionSection()=%u", resSetting));
      if (resSetting == update_decision_setting_type__invalid) {
         resSetting=getReleaseDecision(subModule);
         ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():getReleaseDecision()=%u", 
                         resSetting));
      }
      
      if (resSetting == update_decision_setting_type__invalid) {
         resSetting=_defaultDecision;
         ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():use defaultDecision=%u", 
                         resSetting));
      }  


      
      std::string newVersion =access.getVersion();
      std::string runningVersion = SystemData::instance()->getCurrentModVersion(longName).c_str();
      ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():newVersion    =%s", newVersion.c_str()));
      ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():runningVersion=%s", runningVersion.c_str()));

      bool isDiff=(newVersion != runningVersion);
      bool isUpgrade=(newVersion != runningVersion);
      ETG_TRACE_USR4(("CtrlProgressSection::isSubmoduleUpdateConfigured():isDiff=%u isUpgrade=%u",
                         isDiff, isUpgrade));
      switch (resSetting) {
         case update_decision_setting_type__force:
            res=true;
            break;
         case update_decision_setting_type__diff:
            res=isDiff;
            break;
         case update_decision_setting_type__update:
            res=isUpgrade;
            break;
         case update_decision_setting_type__updateNewOrSame:
            res=isUpgrade || !isDiff;
            break;
         default:
            break;
      }
   }
   ETG_TRACE_USR4(("isSubmoduleUpdateConfigured(%50s): res=%u", longName.c_str(), res));
   
   return res;
}



// set DO_UPDATE=1 for given node, parent-nodes up to "RELEASE", and recursively to all children.
void CtrlProgressSection::setNeedsUpdateAllSubmodules(BoschXmlNode access) {
   ETG_TRACE_USR4(("setNeedsUpdateAllSubmodules(%s) ACCESS", access.getLongName().c_str()));
   if (access.enGetType()==XmlItemAccess::enType_Submodule) {
      ETG_TRACE_USR4(("setNeedsUpdateAllSubmodules:Set DO_UPDATE for %s", access.getLongName().c_str()));
      bubbleUpDoUpdate(access);
   } else {
      for (XmlItemAccess childAccess=access.getFirstChild();
           childAccess.isGood();
           childAccess=childAccess.getNextSibling()) {
         ETG_TRACE_USR4(("setNeedsUpdateAllSubmodules:call child %s", childAccess.getLongName().c_str()));
         setNeedsUpdateAllSubmodules(childAccess);
      }
   }
}

bool CtrlProgressSection::addDependencies()  {
   ETG_TRACE_USR4(("CtrlProgressSection::addDependencies(): START"));

   bool done=true;
   for (XmlItemAccessIter iterChildrenFirst(_progressReleaseSection, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      XmlItemAccess access = iterChildrenFirst.getAccess();
      XmlItemAccess releaseItem(_releaseSection, access.getLongName());

         if (access.enGetType()==XmlItemAccess::enType_Submodule) {
            char const *depends=swu::getTextFromChild(releaseItem.getXml(), "FORCES", false );
            if (depends) {
               std::set<std::string>dependSet=swu::splitString(depends);
               ETG_TRACE_USR4(("found depends=%s\n", depends));
               for (std::set<std::string>::iterator iter =dependSet.begin();
                    iter != dependSet.end();
                    ++iter) {
                  std::string depend=*iter;
                  ETG_TRACE_USR4(("check:%s\n", depend.c_str()));
                  if (!needsUpdate(depend)) {
                     BoschXmlNode childAccess(_progressReleaseSection, depend.c_str());
                     setNeedsUpdateAllSubmodules(childAccess);
                     done=false;
                  }
               }
            }
         }
   }	
   ETG_TRACE_USR4(("CtrlProgressSection::addDependencies(): END done=%u", done));
   //true: all dependencies added, false: must be called again
   return done;
}

bool CtrlProgressSection::handleGroups() {
   ETG_TRACE_USR4(("CtrlProgressSection::handleGroups(): START"));

   _changedGroups.clear();
   // find all changed groups (collect all groups belonging to items to be updated)
   for (XmlItemAccessIter iterChildrenFirst(_progressReleaseSection, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      XmlItemAccess access = iterChildrenFirst.getAccess();
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         TiXmlElement *xmlElem=access.getXml();
         TiXmlElement *info=xmlElem->FirstChildElement("INFO");
         XmlItemAccess releaseItem(_releaseSection, access.getLongName());
         char const *groupsStr=swu::getTextFromChild(releaseItem.getXml(), "GROUPS", false );
         if (groupsStr) {
            ETG_TRACE_USR4(("CtrlProgressSection::handleGroups(): check groupsStr (%30s) from %s", 
                            groupsStr, groupsStr));
            if (swu::getUIntFromChild(info, "DO_UPDATE")) {
               std::set<std::string>groups=swu::splitString(groupsStr);
               ETG_TRACE_USR4(("CtrlProgressSection::handleGroups(): add to changed groups:%30s from %s", 
                               groupsStr, access.getLongName().c_str()));
               swu::addToSet(_changedGroups, groups);
            }
         }
      }
   }

   // find items the belong to changed groups and are not marked as to be updated yet
   bool unchanged=true;
   for (XmlItemAccessIter iterChildrenFirst(_progressReleaseSection, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      BoschXmlNode access(iterChildrenFirst);
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         XmlItemAccess releaseItem(_releaseSection, access.getLongName());
         char const *groupsStr=swu::getTextFromChild(releaseItem.getXml(), "GROUPS", false );
         if (groupsStr) {
            std::set<std::string>groups=swu::splitString(groupsStr);
            if (swu::hasIntersection(_changedGroups, groups)) {
               BoschXmlNode access2(_progressReleaseSection, access.getLongName());
               if (!swu::getUIntFromChild(access2.getInfoItem(), "DO_UPDATE")) {
                  // found new item to be updated
                  ETG_TRACE_USR4(("CtrlProgressSection::handleGroups(): force update of:%s", access.getLongName().c_str()));
                  bubbleUpDoUpdate(access2);
                  unchanged=false;
               }
            }
         }
      }
   }
   ETG_TRACE_USR4(("CtrlProgressSection::handleGroups(): END unchanged=%u", unchanged));

   return unchanged;
}

TiXmlElement *CtrlProgressSection::createProgItem(string const &longName, std::string const &appType) {
   TiXmlElement *progItem=new TiXmlElement("ITEM");
   swu::addTextChild(progItem, "NAME", longName);
   swu::addTextChild(progItem, "TYPE", appType);
   swu::addUIntChild(progItem, "INDEX", 0);
   swu::addTextChild(progItem, "STATE", "pending");
   ETG_TRACE_USR4(("Initialized state pending to %30s, item=%p", longName.c_str(), progItem));

   return progItem;
}

bool CtrlProgressSection::createScriptItems(BoschXmlNode &progAccess, std::string scriptType ) {
   std::string name=progAccess.getLongName();
   BoschXmlNode relAccess(_releaseSection, name);
   TiXmlElement *relBxmlElem=relAccess.getXml();
   TiXmlElement *relScriptItems=relBxmlElem->FirstChildElement(scriptType.c_str());
   if (relScriptItems){
      ETG_TRACE_USR4(("createScriptItems:  %s", scriptType.c_str()));

      tUInt itemIndex=0;
      for (TiXmlElement *relScriptItem=relScriptItems->FirstChildElement("SCRIPT");
           relScriptItem;
           relScriptItem=relScriptItem->NextSiblingElement("SCRIPT")) {

         if (isCompatUpdate() && !swu::getUIntFromChild(relScriptItem, "COMPAT_UPDATE")) {
            continue;
         }
         std::string script_name=name;
         if (progAccess.enGetType() == XmlItemAccess::enType_Release) {
            script_name="RELEASE";
         } 

         ETG_TRACE_USR4(("createProgItems:ADD %30s/%s", name.c_str(), scriptType.c_str()));
         TiXmlElement *progItem=createProgItem(name, scriptType);
         swu::setUIntChild(progItem, "INDEX", itemIndex);
         ++itemIndex;

         BoschXmlNode progAccess2(_progressReleaseSection, name);
         progAccess2.getItemsItem()->LinkEndChild(progItem);
      }
   }
   return true;
}

bool CtrlProgressSection::createProgItems(bool onStartOnly) {
   ETG_TRACE_ERR(("CtrlProgressSection::createProgItems START"));
   bool foundSkippedItem=false;
   for (XmlItemAccessIter accessIter(_progressReleaseSection, XmlItemAccessIter::enOnBoth);!accessIter.isDone();accessIter.next()) {
      BoschXmlNode progAccess(accessIter);
      
      if ((tUInt)progAccess.enGetType()>(tUInt)XmlItemAccess::enType_Submodule) {
         continue;
      }
      std::string name=progAccess.getLongName();
      ETG_TRACE_USR4(("createProgItems: Type=%u %s", (tUInt)progAccess.enGetType(), name.c_str()));
      if (!swu::getUIntFromChild(progAccess.getInfoItem(), "DO_UPDATE")) {
         ETG_TRACE_USR4(("createProgItems: SKIP DO_UPDATE=false %s", name.c_str()));  
         if (!foundSkippedItem) {
            ETG_TRACE_ERR(("CtrlProgressSection::createProgItems found skipped item"));
            foundSkippedItem=true;
         }
         continue;
      }

      if (accessIter.getDirection()==XmlItemAccessIter::enEnter) {
         ETG_TRACE_USR4(("createProgItems: Dir=enEnter %s", name.c_str()));

         createScriptItems(progAccess, "PREPARE");
         if (onStartOnly) {
            ETG_TRACE_USR4(("createProgItems: onStartOnly %s", name.c_str()));
            continue;
         }
         createScriptItems(progAccess, "PREPARE_ONCE");

         if (progAccess.enGetType()==XmlItemAccess::enType_Submodule) {
            ETG_TRACE_USR4(("createProgItems: enType_Submodule %s", name.c_str()));
            // get access to the release-section from stick:
            BoschXmlNode relAccess(_releaseSection, name);
            TiXmlElement *elem=relAccess.getXml();
     	    tUInt allowOneFlashItem=swu::getUIntFromChild(elem, "ALLOW_ONE_FLASH_ITEM");
            TiXmlElement *flashItems=elem->FirstChildElement("FLASH_ITEMS");
            SWU_ASSERT_RETURN_FALSE(flashItems);
            tUInt itemIndex=0;
            for (TiXmlElement *flashItem=flashItems->FirstChildElement("FLASH_ITEM");
                 flashItem;
                 flashItem=flashItem->NextSiblingElement("FLASH_ITEM")) {
               TiXmlElement *progItem=createProgItem(name, "APP");
               swu::setUIntChild(progItem, "INDEX", itemIndex);
               ETG_TRACE_USR4(("createProgItems:ADD %30s/APP/%u", name.c_str(), itemIndex));
               progAccess.getItemsItem()->LinkEndChild(progItem);

               ++itemIndex;
	     // This is to ensure that only one  flash item should be added to the running XML and this is useful only
	    // when trigger to particular sub-module is required only once , but multiple flash items are present
	     if(allowOneFlashItem)
		break; 
            }
            swu::addTextChild(progAccess.getXml(), "FROMVERSION", 
                              SystemData::instance()->getCurrentModVersion(name));
         }
      }
      else { // leave
         if (onStartOnly) {
            continue;
         }
         createScriptItems(progAccess, "FINALIZE");
      }
   } 
   bool bIsPartialRelease=isPartialRelease();
   bool fullUpdate=!foundSkippedItem &&  !bIsPartialRelease; 
   ETG_TRACE_USR4(("CtrlProgressSection::createProgItems: fullUpdate=%u foundSkippedItem=%u bIsPartialRelease=%u", 
                  fullUpdate, foundSkippedItem, bIsPartialRelease));
   swu::setUIntChild(_progressReleaseSection, "FULL_UPDATE", fullUpdate ? 1 : 0);
   
   ETG_TRACE_ERR(("CtrlProgressSection::createProgItems END"));

   return true; 
}

void CtrlProgressSection::storePhaseStart() {
   if (_currentCtrlProgItem.isValid()) {
      string strRefKey = _currentCtrlProgItem.getRefKey();  //Coverity fix for 209429
      ETG_TRACE_USR1(("CtrlProgItem::storePhaseStart(%50s)", strRefKey.c_str()));
      swu::setTextChild(_progressReleaseSection, "PHASE_START", _currentCtrlProgItem.getRefKey()); 
   }
   else {
      ETG_TRACE_ERR(("CtrlProgItem::storePhaseStart(): item invalid"));
   }
}

bool CtrlProgressSection::isCurItem(CtrlProgItem const *item) {
   bool res=false;
   if (!_currentCtrlProgItem.isValid()) {
      ETG_TRACE_ERR(("CtrlProgItem::isCurItem(): _currentCtrlProgItem invalid"));
   }
   else if (!item->isValid()) {
      ETG_TRACE_ERR(("CtrlProgItem::isCurItem(): item invalid"));
   }
   else  {
      res=(_currentCtrlProgItem.getRefKey()==item->getRefKey());
      string strRefKey = _currentCtrlProgItem.getRefKey();  //Coverity fix for 209454
      string strItemRefKey =  item->getRefKey();
      ETG_TRACE_ERR(("CtrlProgItem::isCurItem(%50s,%50s):%u", 
                     strRefKey.c_str(),
                     strItemRefKey.c_str(),
                     res));
   }
   
   return res;
}


void CtrlProgressSection::setPostInstallMode(bool doSet) {
   ETG_TRACE_USR4(("CtrlProgressSection::setPostInstallMode():%u", doSet));
   if (_progressSection) {
      swu::setUIntChild(_progressSection, "POST_INSTALL_MODE", doSet ? 1 : 0);
   }
}

bool CtrlProgressSection::isInPostInstallMode() {
   bool res=false;
   if (_progressSection) {
      res=swu::getUIntFromChild(_progressSection, "POST_INSTALL_MODE");
   }
   ETG_TRACE_USR4(("CtrlProgressSection::isInPostInstallMode():%u", res));
   return res;
}


void CtrlProgressSection::Statistics::addTimes(tenScope enScope, CtrlProgItem const &progItem, bool bPhaseStarted, bool bPhaseEnded) {
   tUInt tItem=progItem.getUpdateTime();
   Times &times=_counters.getTimesMutable(enScope);
   std::string itemState=progItem.getState();
   bool bIsInPhase=(bPhaseStarted && !bPhaseEnded);
   string strRefKey = progItem.getRefKey(); //Coverity fix for 209410
   ETG_TRACE_USR4(("CtrlProgressSection::Statistics::addTimes(%50s):scope=%10s tItem=%u bPhaseStarted=%u bPhaseEnded=%u bIsInPhase=%u itemState=%s",
                   strRefKey.c_str(), _counters.scopeToCStr(enScope), tItem, bPhaseStarted, bPhaseEnded, bIsInPhase, itemState.c_str()));

   if (bIsInPhase) {
      ETG_TRACE_USR4(("CtrlProgressSection::Statistics::addTimes():add to tAll:%u", tItem));
      times.tAll+=tItem;
      if (itemState == "ok" || itemState=="failed") {
         ETG_TRACE_USR4(("CtrlProgressSection::Statistics::addTimes():add to tDone:%u", tItem));
         times.tDone+=tItem;
      }
   }
   
}

void CtrlProgressSection::Statistics::onTimeout() {

   Times &times=_counters._timesCurrent;
   tInt timeLeft=times.tAll - times.tDone;
   ETG_TRACE_USR4(("CtrlProgressSection::Statistics::onTimeout() timeLeft=%u", timeLeft));

   if (timeLeft > static_cast<int> (getUpdateTimerSec())) {    // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      times.tDone+=getUpdateTimerSec();
   }
   else if (timeLeft) {
      times.tDone=times.tAll;
   }
   else {
      // nothing changed, no need to restart timer.
      return;
   }
   print();
   _progressSection->notifyClients();
   if (timeLeft) {
      _progressSection->startProgressTimer(getUpdateTimerSec());
   }
}

void CtrlProgressSection::Statistics::add( CtrlProgItem &progItem) {
   ETG_TRACE_USR4(("CtrlStatistics add START"));
   std::string itemState=progItem.getState();
   ETG_TRACE_USR4(("CtrlStatistics::add(%50s) isPhaseStart=%u", 
                   progItem.getRefKey().c_str(), progItem.isPhaseStart()));
   Times &curTimes=_counters._timesCurrent;
   std::string const &startingItemRefKey(_progressSection->getStartingItemRefKey());
   ETG_TRACE_USR4(("startingItemRefKey=%s", 
                   startingItemRefKey.c_str()));

   ETG_TRACE_USR4(("CtrlStatistics add:itemState=%s", itemState.c_str()));
   if ( progItem.getRefKey()==startingItemRefKey) {
      _progressSection->startProgressTimer(getUpdateTimerSec());
      curTimes.tAll = progItem.getUpdateTime();
      curTimes.tDone=0;
   }
   if (progItem.isPhaseStart() && !_bPhaseStarted) {
      ETG_TRACE_USR4(("CtrlStatistics set _bPhaseStarted"));
      _bPhaseStarted=true;
   }
   else if (progItem.isPhaseStart() && _bPhaseStarted) {
      ETG_TRACE_USR4(("CtrlStatistics set bPhaseEnded"));
      _bPhaseEnded=true;
   }

   if (itemState=="pending") {
      _counters.numPending++;
   }
   else if (itemState=="running")  {
      _counters.numRunning++;
   }
   else if (itemState=="ok") {
      _counters.numOk++;
   }
   else if (itemState=="failed") {
      curTimes.tDone=curTimes.tAll;
      _counters.numFailed++;

   }
   _counters.numAll++;
   addTimes(Statistics::enAll, progItem, true, false);
   addTimes(Statistics::enPhase, progItem, _bPhaseStarted, _bPhaseEnded);

   ETG_TRACE_USR4(("CtrlStatistics::add:name=%30s state=%30s: result: numPending=%u numRunning=%u, numFailed=%u numOk=%u",
                   progItem.getName().c_str(), itemState.c_str(), _counters.numPending, _counters.numRunning, _counters.numFailed, _counters.numOk));
   print();
}

float CtrlProgressSection::Statistics::Counters::getFraction(tenScope enScope) const {
   Times const &times=getTimes(enScope);
   ETG_TRACE_USR4(("CtrlProgressSection::Statistics::getFraction() enScope=%s", 
                   scopeToCStr(enScope)));
   ETG_TRACE_USR4(("   tAll=%u", times.tAll));
   ETG_TRACE_USR4(("   tDone=%u",times.tDone));
   float res;
   float timeDone=static_cast<float> (times.tDone);
   if (enScope != enCurrent) {
      timeDone += static_cast<float> (_timesCurrent.tDone);       // gen3armmake, gen3x86make, gen4rcar, gen4lsim: conversion to 'float' from 'tUInt {aka unsigned int}' made explicit
      ETG_TRACE_USR4(("   add current=%u", _timesCurrent.tDone));
   }
   if (times.tAll) {
      res=timeDone/(static_cast<float> (times.tAll));
   }
   else {
      ETG_TRACE_USR4(("   tAll is Zero"));
      res = 1;
   }
   ETG_TRACE_USR4(("   res=%f", res));
   return res;
}

void CtrlProgressSection::Statistics::print(const char *action) {

#ifndef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_USR4(("CtrlProgressSection::Statistics:"));
   ETG_TRACE_USR4(("   pending=%u", _counters.numPending));
   ETG_TRACE_USR4(("   running=%u", _counters.numRunning));
   ETG_TRACE_USR4(("   failed =%u", _counters.numFailed));
   ETG_TRACE_USR4(("   ok     =%u", _counters.numOk));
   ETG_TRACE_USR4(("   retries=%u", _counters.numRetries));
   ETG_TRACE_USR4(("   all    =%u", _counters.numAll));
   ETG_TRACE_USR4(("   times all:"));
   ETG_TRACE_USR4(("      tAll    =%u", _counters._times.tAll));
   ETG_TRACE_USR4(("      tDone   =%u", _counters._times.tDone));
   ETG_TRACE_USR4(("      fraction=%u", _counters.getPercent(enAll) * 100));
   ETG_TRACE_USR4(("   times phase:"));
   ETG_TRACE_USR4(("      tAll    =%u", _counters._timesPhase.tAll));
   ETG_TRACE_USR4(("      tDone   =%u", _counters._timesPhase.tDone));
   ETG_TRACE_USR4(("      fraction=%u", _counters.getPercent(enPhase) * 100));
   ETG_TRACE_USR4(("   times current:"));
   ETG_TRACE_USR4(("      tAll    =%u", _counters._timesCurrent.tAll));
   ETG_TRACE_USR4(("      tDone   =%u", _counters._timesCurrent.tDone));
   ETG_TRACE_USR4(("      fraction=%u", _counters.getPercent(enCurrent) * 100));
#endif

}

}

/*
  rescheduling in Scomo:
  - Progress-section knows:
  - state: running/scomoPending
  - items: 
     - state: peding/running/failed (dynamic)
     - needed phases: default=last(3) (static on creation)
     - lastPhase passed, default:0 (dynamic)
     - problem: update fails during post-scripts-> assert that on restart same phase is requested
  
 */
