#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "util/swu_util.hpp"
#include "util/swu_member.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_execCommand.h"
#include "config/fcswupd_config.hpp"
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_globalLog.h"
#include "main/fcswupd_component.h"
#include "main/fcswupd_srv.h"
#include "main/fcswupd_propDevMgr.h"

#include "main/fcswupd_parserIf.h"
#include "main/fcswupd_mainMessages.hpp"

#include "util/fcswupd_cust_update_info.hpp"

#include "configurator/fcswupd_configuratorRecoveryGeneric.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_CONFIGURATOR
#include "trcGenProj/Header/fcswupd_configuratorRecoveryGeneric.cpp.trc.h"
#endif

#define FCSWUPD_CONFIGURATOR_INIT_TIMER_MS 120000

namespace fcswupdate {

ConfiguratorUtil::tenConfigOption ConfiguratorRecoveryGeneric::getOverallReleaseFilter(TiXmlElement const &overallSection) {
   ConfiguratorUtil::tenConfigOption overallReleaseFilterOption = _util.decisionFromString(swu::getTextFromChildOrEmpty(&overallSection, "RELEASE_FILTER"));

   if(overallReleaseFilterOption == ConfiguratorUtil::enConfigOption_NotDefined)
   {
      overallReleaseFilterOption = (ConfiguratorUtil::tenConfigOption)Config::instance()->cfg_RecoveryCfgDefaultReleaseFilter.get();
   }
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::getOverallReleaseFilter():XMLFilterOption=%u",
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, _enFilterOption)));

   return overallReleaseFilterOption;
}

// comparator to find best release
bool ConfiguratorRecoveryGeneric::RelInfo::operator<(ConfiguratorRecoveryGeneric::RelInfo const &r) const {

   std::string releaseName = swu::getTextFromChildOrEmpty(&overallSection,"FINALNAME");
   std::string releaseNameR = swu::getTextFromChildOrEmpty(&r.overallSection,"FINALNAME");

   if(releaseName.empty() || releaseNameR.empty()) {
      return false;
   } else {
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::operator< start l=%20s r=%s",
                 releaseName.c_str(), releaseNameR.c_str()));
      int res=releaseName.compare( releaseNameR );
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::string compare returned %d", res));
      return res < 0;
   }  
}

bool ConfiguratorRecoveryGeneric::filterPassed(RelInfo  const &checkRelInfo) {
   std::string runningReleaseName=Config::instance()->cfg_RunningFinalName.get();
   std::string checkReleaseName = swu::getTextFromChildOrEmpty(&checkRelInfo.overallSection, "FINALNAME");
   TiXmlElement runningReleaseOverall=TiXmlElement("OVERALL");
   swu::addTextChild(&runningReleaseOverall, "FINALNAME", runningReleaseName);

   RelInfo runningRelInfo;
   runningRelInfo.overallSection=runningReleaseOverall;

   bool isUpgrade=(runningRelInfo < checkRelInfo);
   bool isDowngrade=(checkRelInfo < runningRelInfo);
   bool isUnchanged=(!isUpgrade && !isDowngrade);
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::filterPassed:isUpgrade=%u isDowngrade=%u isUnchanged=%u sourceType=%u",
                   isUpgrade, isDowngrade, isUnchanged, ETG_CENUM(tenSourceType, checkRelInfo.rSourceInfo.enSourceType) ));

   bool res=false;
   _enFilterOption = getOverallReleaseFilter(checkRelInfo.overallSection);

   switch(_enFilterOption)
   {
      // if the release is newer than the current release
      case ConfiguratorUtil::enConfigOption_NewOnly:
      {
         res=isUpgrade;
         break;
      }
     // if the release is newer or same of the current release
     case ConfiguratorUtil::enConfigOption_NewAndSameOnly:
     {
       res=isUpgrade || isUnchanged;
       break;
     }
      // current and available releases are different
      case ConfiguratorUtil::enConfigOption_DifferentOnly:
      {
         res=!isUnchanged;
         break;
      }
      // current and available releases are different
      case ConfiguratorUtil::enConfigOption_Any:
      {
         res=true;
         break;
      }
      //default decision is to fail, to know undefined value is used.
      default:
      {
         res=false;
         break;
      }
   }

   if (!res && (checkRelInfo.rSourceInfo.enSourceType==tenSourceType_OTA)){
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::filterPassed:force acceptance of OTA-release"));
      res=true;
   }

   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::filterPassed():filterOption=%u res=%u :%30s -> %s ",
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, _enFilterOption),
                   res,
                   runningReleaseName.c_str(),
                   checkReleaseName.c_str() ));
   return res;
}

ConfiguratorRecoveryGeneric::ConfiguratorRecoveryGeneric() : 
   _enState(enState_Initial), 
   _releaseXml(0),
   _enConfigTrigger( enConfigTrigger_Invalid )
{
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric CTOR"));
   _enFilterOption=(ConfiguratorUtil::tenConfigOption)Config::instance()->cfg_RecoveryCfgDefaultReleaseFilter.get();
}

ConfiguratorRecoveryGeneric::~ConfiguratorRecoveryGeneric() {
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric DTOR"));
   vDeInit();
}

// unsubsribe will be done automatically on destruction of this class.
// Expect to skip  _enState = enState_Initial;
//
// It is possible that one or more of the options available
//  ( that is - on the media and the last overall provided by Parser in the overall list )
// we have to take care NOT to use dissimilar release names
//    - customer update vs non customer update.
/*
  peha: customer-updates shall never be used for emergency-update
 */
tVoid ConfiguratorRecoveryGeneric::vInit() {

   vSetSWUHMIState(tenPrjState_NotReady);
   _enConfigTrigger=enConfigTrigger_Invalid;
   //for emmc firmware version
   swu::execCommand("/opt/bosch/base/bin/swupdatecontrol_out.out --emmcinfo");

   Msg_NotifyDevManagerChanged::vSubscribe(this);
   _enState = enState_SearchRelease;
   findReleaseXml();
   _initTimer.start(this, FCSWUPD_CONFIGURATOR_INIT_TIMER_MS);
}

tVoid ConfiguratorRecoveryGeneric::vDeInit() {
   if (_releaseXml) {
      delete _releaseXml;
      _releaseXml = 0;
   }
   _enState=enState_Terminated;
}

tVoid ConfiguratorRecoveryGeneric::findReleaseXml() {

   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml"));
   SWU_ASSERT_RETURN(_enState == enState_SearchRelease);
   // set that will contain all releases from all sources:
   std::set<RelInfo> relInfos;
   //  get list of all sources
   std::map<std::string, trSourceInfo> sourcesList =PropDevMgr::instance()->get()._mapSources;   
   // no need to store in member
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: numSources=%u", sourcesList.size()));

   std::string relName;   
   tenSwUpdateError errorCode = tenSwUpdateError_OK;   
   bool anyReleaseFound=false;

   /*
     basically you have the states "enState_FoundRelease" and "enState_SearchRelease" after beeing initialized.
     whild no release was found, we can only wait for new information via Msg_NotifyDevManagerChanged. (enState_FoundRelease)
     So the handling should be similar as in customer-update:
     First we try to find a release from during initilization, later we update on each call of Msg_NotifyDevManagerChanged.
     When we have reached enState_FoundRelease we ignore further updates via Msg_NotifyDevManagerChanged
     So source-code to find best release must be put into separate fn like in customer-update
   */

   std::list<TiXmlElement> xmlRelList;   

   for (std::map<std::string, trSourceInfo>::iterator iterSrc= sourcesList.begin();
            iterSrc!=sourcesList.end();
            ++iterSrc) {
      trSourceInfo sourceInfo=iterSrc->second;
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: check source %s", sourceInfo.path.c_str()));
      FcSwUpdCore::instance()->getReleaseList(xmlRelList, sourceInfo, &errorCode);

      if(!hasMediumFormatSupported(sourceInfo.path)) {
         ETG_TRACE_USR3(("ConfiguratorHmiGeneric::findReleaseXml, medium's format is not supported"));

         if(!isIncompactMediumCanProceed(sourceInfo.path, errorCode)) {
            ETG_TRACE_USR3(("ConfiguratorRecoveryGeneric::findReleaseXml: incompatible medium, cannot proceed"));
            continue;
         }
      }

      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: found %u releases", xmlRelList.size()));

      // get release-list from sourcesList
      for ( std::list<TiXmlElement>::iterator iterRel=xmlRelList.begin();
            iterRel!=xmlRelList.end();
            ++iterRel) {
         TiXmlElement &overallSection=*iterRel;
         RelInfo relInfo;
         relInfo.overallSection=overallSection;
         relInfo.rSourceInfo=sourceInfo;
         
         // create  struct ReleaseInfo
         // add compare-operator to ReleaseInfo.
         // add release ReleaseInfo to map<ReleaseInfo>        
         if ( ! Config::instance()->cfg_RecoveryCfgUseXMLReleaseFilter.get() )
         {
            ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: insert release %s", relName.c_str()));
            relInfos.insert(relInfo);
         }
         else 
         {
            // If cfg_RecoveryCfgUseXMLReleaseFilter is set filter specified in XML should pass
            if( filterPassed(relInfo) )
            {
               ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: insert release %s", relName.c_str()));
               relInfos.insert(relInfo);
            }
         }
      }
   }
   
   relName="";
   if (sourcesList.empty()) {
      tenSwUpdateError_ERROR_MEDIA_UNAVAILABLE;
      vSetSWUHMIState(tenPrjState_Ready);
   }
   else if(errorCode == tenSwUpdateError_ERROR_INCOMPATIBLE_USB_FORMAT){
      vSetSWUHMIState(tenPrjState_SelectSource);
   }
   // No release Found, No parsing error detected
   else if(!errorCode && xmlRelList.empty()){
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric:: No Release Found, Error=ERROR_NO_RELEASE_FOUND"));
      errorCode = tenSwUpdateError_ERROR_NO_RELEASE_FOUND;
   }
   // valid release(s) found without parsing errors, but filtered out due to providing filtering release criteria.
   else if(!errorCode && (relInfos.empty())){
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric:: No valid Release Found, Error=ERROR_IMAGE_INCOMPATIBLE_VERSION"));
      errorCode = tenSwUpdateError_ERROR_IMAGE_INCOMPATIBLE_VERSION;
   }
   else {
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric:: Default case, tenPrjState_SelectRelease"));
      vSetSWUHMIState(tenPrjState_SelectRelease);
   }

   for (std::set<RelInfo>::iterator iterRelInfo=relInfos.begin();
        iterRelInfo!=relInfos.end();
        ++iterRelInfo) {
      RelInfo relInfo=*iterRelInfo;      
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: best release Found"));
      
      // fetch releaseXml from std::map<ReleaseInfo>.front().
      _releaseXml = FcSwUpdCore::instance()->getReleaseXml(relInfo.overallSection, errorCode);
      if (_releaseXml) {
         BXmlAccess relAccess(_releaseXml);
         std::string relName = relAccess.getTextFromModule("FINALNAME");      
         ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: best release= %s: valid", relName.c_str()));
         // If this does not work, try next in list.
         break;
      }
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: best release= %s: INVALID", relName.c_str()));
   }

   // To-do In case of multiple parsing errors with no appropriate release,
   // priority of errors need to be determined with individual release and pushed in parseError	
   //send parserError=tenSwUpdateError_OK  otherwise appropriate error
   FCSWUPD_NS_FCSWUPD_T::trErrorIds updateErrors;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::tenErrorId > parserError;
   parserError.push_back((FCSWUPD_NS_FCSWUPD_T::tenErrorId)errorCode);
   updateErrors.setErrorIds(parserError);
   
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric:: Sending updateErrors with errorCode=%d",errorCode));
   FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, updateErrors, UpdateErrors);		

   
   if (_releaseXml) {
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: found release= %s, call vNotifyReady()", relName.c_str()));
      vSetSWUHMIState(tenPrjState_Config);
      _enState =enState_FoundRelease;
      TiXmlElement *xmlRoot = _releaseXml->RootElement();
      SWU_ASSERT_RETURN( xmlRoot);
      TiXmlElement *xmlOverall=  xmlRoot->FirstChildElement("OVERALL");
      SWU_ASSERT_RETURN( xmlOverall);


      std::string forceDnlFile= getMediaPathFromOverallSection(xmlOverall) + "force.dnl";
      std::string pattern="^SHUTDOWN[:space:]*=[:space:]*0*[1..9]";
      
      if (swu::exists(forceDnlFile)) {
         ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::seach for force.dnl in %s", forceDnlFile.c_str()));
         bool forceShutdown=swu::bFindPatternInFile(pattern,forceDnlFile);
         Config::instance()->cfg_DoShutdownAfterUpdate.setFromBool(forceShutdown);
      }
      vNotifyReady( true );
   } else {
      //Config Empty using same logic in configuratorHMIGeneric
      ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::findReleaseXml: found no appropriate release"));
   }
   GlobalLog::logSwuErrorOnce(errorCode);

}

// When we didn't find a feasible release in initialization, we get notified here 
// about new data
tVoid ConfiguratorRecoveryGeneric::vProcess(Msg_NotifyDevManagerChanged *pMsg) {
	ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::vProcess(Msg_NotifyDevManagerChanged *pMsg)"));
	(void)pMsg;    
	if (_enState!=enState_SearchRelease) {
       return;
    }
   _enConfigTrigger=enConfigTrigger_DevManagerChanged;
    findReleaseXml();
   _enConfigTrigger=enConfigTrigger_Invalid;

}


void ConfiguratorRecoveryGeneric::traceState() {
   ETG_TRACE_COMP(("ConfiguratorRecoveryGeneric traceState"));

   ETG_TRACE_COMP(("  _enState=%u",ETG_CENUM(ConfiguratorRecoveryGeneric::tenState, _enState)));
}


// return our prepared boschXml
TiXmlDocument *ConfiguratorRecoveryGeneric::getXmlDoc() {
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::getXmlDoc START _enState=%u",
                   _enState));
   assert(_enState == enState_FoundRelease );
   _enState  = enState_Terminated;
   TiXmlDocument *relDocTmp=_releaseXml;
   _releaseXml = 0;
   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::getXmlDoc END"));
   return relDocTmp;
}

tVoid ConfiguratorRecoveryGeneric::vSetSWUHMIState(tenPrjState newState) {

   ETG_TRACE_USR1(("ConfiguratorRecoveryGeneric::vEnterState: newstate:%u trigger:%u", newState, _enConfigTrigger ));
   if(_enConfigTrigger == enConfigTrigger_DevManagerChanged) {
      _initTimer.stop();
      ETG_TRACE_USR4(("Sending SWUHMIState-%u", newState ));
      _util.vSetHmiState(newState);
   }
}

tVoid ConfiguratorRecoveryGeneric::vProcess(Msg_InitialConfiguratorTimer *pMsg) { 
   ETG_TRACE_USR4(("Msg_InitialConfiguratorTimer elasped"));
   _util.vSetHmiState(tenPrjState_Ready);
}

bool ConfiguratorRecoveryGeneric::hasMediumFormatSupported(std::string path) {
   ETG_TRACE_USR4(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported START"));
   std::string supportedFormats = Config::instance()->cfg_SupportedUSBFormats.get();
   if(supportedFormats.empty()) {
      ETG_TRACE_USR3(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported, supported format is empty allow all formats"));
      return true;
   }

   ETG_TRACE_USR2(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported: path:%s", path.c_str()));
   std::string mediumFormat = swu::getMediaType(path);
   ETG_TRACE_USR2(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported: mediumFormat:%s", mediumFormat.c_str()));

   std::set<std::string> formatList = swu::splitString(supportedFormats);
   for(std::set<std::string>::iterator Iter = formatList.begin(); Iter != formatList.end(); Iter++) {
      if(0 == mediumFormat.compare(*Iter)) {
         ETG_TRACE_USR4(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported format supported"));
         return true;
      }
   }

   ETG_TRACE_USR4(("ConfiguratorRecoveryGeneric::hasMediumFormatSupported format not supported"));
   return false;
}

bool ConfiguratorRecoveryGeneric::isIncompactMediumCanProceed(std::string path, tenSwUpdateError& status) {
   ETG_TRACE_USR4(("ConfiguratorRecoveryGeneric::isIncompactMediumCanProceed START path:%s", path.c_str()));

   bool bResult=false;
   if(swu::exists(path + "/bosch.xml")) {
      ETG_TRACE_USR2(("ConfiguratorRecoveryGeneric::isIncompactMediumCanProceed, it's firmware update stick"));

      if(swu::exists(path + "/force.dnl")) {
         ETG_TRACE_USR2(("ConfiguratorRecoveryGeneric::isIncompactMediumCanProceed, forcing update"));
         bResult=true;
      }

      if(!bResult) {
         status = tenSwUpdateError_ERROR_INCOMPATIBLE_USB_FORMAT;
      }
   }

   ETG_TRACE_USR4(("ConfiguratorRecoveryGeneric::isIncompactMediumCanProceed END bResult:%u", bResult));
   return bResult;
}

}// namespace
