#include <sstream>
#include "util/swu_execCommand.h"
#include "util/swu_util.hpp"
#include "util/swu_member.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_globallog.h"
#include "tinyxml/tinyxml.h"
#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "config/fcswupd_config.hpp"
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_globalLog.h"
#include "main/fcswupd_component.h"
#include "main/fcswupd_specialRelease.h"
#include "main/fcswupd_spmIf.h"
#include "ctrl/fcswupd_ctrl.h"
#include "main/fcswupd_srv.h"
#include "tinyxml/tinyxml.h"
#include "main/fcswupd_propDevMgr.h"
#include <sys/stat.h>
#include <sys/types.h>

#include "main/fcswupd_parserIf.h"
#include "main/fcswupd_mainMessages.hpp"
#include "main/fcswupd_campaignmanager.h"


#include "configurator/fcswupd_configuratorHmiGeneric.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_CONFIGURATOR
#include "trcGenProj/Header/fcswupd_configuratorHmiGeneric.cpp.trc.h"
#endif

namespace fcswupdate {

tVoid ConfiguratorHmiGeneric::vProcess(Msg_NotifyDevManagerChanged *pMsg) {
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::Msg_NotifyDevManagerChanged START"));     
   // find release xml
   _enConfigTrigger=enConfigTrigger_DevManagerChanged;
   findReleaseXml();
   _enConfigTrigger=enConfigTrigger_Invalid;
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::Msg_NotifyDevManagerChanged END"));     
}
ConfiguratorUtil::tenConfigOption ConfiguratorHmiGeneric::getOverallReleaseFilter(TiXmlElement const &overallSection) {
   ConfiguratorUtil::tenConfigOption overallReleaseFilterOption = _util.decisionFromString(swu::getTextFromChildOrEmpty(&overallSection, "RELEASE_FILTER"));
   if(overallReleaseFilterOption == ConfiguratorUtil::enConfigOption_NotDefined)
   {
      overallReleaseFilterOption = (ConfiguratorUtil::tenConfigOption)Config::instance()->cfg_HmiCfgDefaultReleaseFilter.get();
   }
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::getOverallReleaseFilter():XMLFilterOption=%u",
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, _enFilterOption)));
   return overallReleaseFilterOption;
}

bool ConfiguratorHmiGeneric::RelInfo::operator<(ConfiguratorHmiGeneric::RelInfo const &r) const {
   // for comparision allways take bosch-version FINALNAME. FINALNAME_CUSTOMER is only for display
   std::string releaseName = swu::getTextFromChildOrEmpty(&overallSection,"FINALNAME");
   std::string releaseNameR = swu::getTextFromChildOrEmpty(&r.overallSection,"FINALNAME");

   if(releaseName.empty() || releaseNameR.empty()) {
      return false;
   } else {
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::operator< start l=%20s r=%s",
                      releaseName.c_str(), releaseNameR.c_str()));
      int res=releaseName.compare( releaseNameR );
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::string compare returned %d", res));
      return res < 0;
   }
}

bool ConfiguratorHmiGeneric::filterPassed(RelInfo  const &checkRelInfo, tenSwUpdateError& status) {
   std::string runningReleaseName=Config::instance()->cfg_RunningFinalName.get();
   std::string checkReleaseName = swu::getTextFromChild(&checkRelInfo.overallSection, "FINALNAME", false);
   TiXmlElement runningReleaseOverall=TiXmlElement("OVERALL");
   swu::addTextChild(&runningReleaseOverall, "FINALNAME", runningReleaseName);

   RelInfo runningRelInfo;
   runningRelInfo.overallSection=runningReleaseOverall;
   
   bool isUpgrade=(runningRelInfo < checkRelInfo);
   bool isDowngrade=(checkRelInfo < runningRelInfo);
   bool isUnchanged=(!isUpgrade && !isDowngrade);
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::filterPassed:isUpgrade=%u isDowngrade=%u isUnchanged=%u sourceType=%u",
                   isUpgrade, isDowngrade, isUnchanged, ETG_CENUM(tenSourceType, checkRelInfo.rSourceInfo.enSourceType) ));
   if(isUpgrade)
	   _eUpdateDirection = FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Upgrade;
   else if(isDowngrade)
	   _eUpdateDirection = FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Downgrade;
   else if(isUnchanged)
	   _eUpdateDirection = FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Same;
   //Removed else condition for coverity fix 22848
   /*else
	   _eUpdateDirection = FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Unknown;*/

   if (isDowngrade) status = tenSwUpdateError_ERROR_IMAGE_INCOMPATIBLE_DOWNGRADE;
   else if (isUnchanged) status = tenSwUpdateError_ERROR_IMAGE_INCOMPATIBLE_SAME_VERSION;
   else status = tenSwUpdateError_OK;

   ConfiguratorUtil::tenConfigOption enFilterOption = _enFilterOption;
   // If HMI allows any release and XML filter overrides, get the decision from XML
   if(_enFilterOption == ConfiguratorUtil::enConfigOption_Any && Config::instance()->cfg_HmiCfgXMLOverridesHMIReleaseFilter.get())
      enFilterOption = getOverallReleaseFilter(checkRelInfo.overallSection);
  
   if(Config::instance()->cfg_AllowMisCmcRelease.get()){	
		bool isMisPendingMarker = checkMisPendingMarker();
		ETG_TRACE_USR1(("ConfiguratorHmiGeneric::filterPassed: isUnchanged=%u isMisPendingMarker=%u",
                   isUnchanged, isMisPendingMarker));
		if( isUnchanged && isMisPendingMarker && overallAllowsMisCmcRelease(&checkRelInfo.overallSection) ) {
			enFilterOption = ConfiguratorUtil::enConfigOption_NewAndSameOnly;
		}
	}
   
   bool res=false;
   switch(enFilterOption)
   {
      // if the release is newer than the current release
      case ConfiguratorUtil::enConfigOption_NewOnly:  
      {
         res=isUpgrade;
         break;
      }
	  // if the release is newer or same of the current release
	  case ConfiguratorUtil::enConfigOption_NewAndSameOnly:
	  {
		 res=isUpgrade || isUnchanged;
		 break;
	  }
      // current and available releases are different
      case ConfiguratorUtil::enConfigOption_DifferentOnly:
      {
         res=!isUnchanged;
         break;
      }
      // current and available releases are different
      case ConfiguratorUtil::enConfigOption_Any:
      {
         res=true;
         break;
      }
      //default decision is to fail, to know undefined value is used.
      default:
      {
         res=false;
         break;
      }					
   } 

   if (!res && (checkRelInfo.rSourceInfo.enSourceType==tenSourceType_OTA)){
       ETG_TRACE_USR1(("ConfiguratorHmiGeneric::filterPassed:force acceptance of OTA-release"));
	   res=true;
   }

   if (!res && isDowngrade && Config::instance()->cfg_ConfiguratorAllianceAllowDowngrade.readAsBool()) {
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::filterPassed:force acceptance of downgrade SW as configured in alliance.xml"));
      status = tenSwUpdateError_OK;
      res=true;
   }
   
   swu::LOG_GLOBALINFO("filterOption=%u res=%u", enFilterOption, res);
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::filterPassed():filterOption=%u res=%u :%30s -> %s ", 
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, enFilterOption),
                   res,
                   runningReleaseName.c_str(), 
                   checkReleaseName.c_str() ));	  
   return res;
}



void ConfiguratorHmiGeneric::rebootToRecoveryMode() {
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric: Forcing reboot into recovery mode."));  
   SpmIf::instance()->rebootToRecoveryMode();
}

bool ConfiguratorHmiGeneric::overallAllowsForceDnl(TiXmlElement *overallSection) {
   bool allowForceDnl=false;
   if (!overallSection) {
      return allowForceDnl;
   }

   // test for dedicated element
   allowForceDnl=swu::getUIntFromChild(overallSection, "ALLOW_FORCE_DNL");
   if (allowForceDnl) {
      return allowForceDnl;
   }

   // check for extra-arg
   TiXmlElement* extraArg = overallSection->FirstChildElement("EXTRA_ARG");
   ETG_TRACE_USR1(("%s EXTRA_ARG", (extraArg ? "Found" : "Did not find") ));
   for (extraArg; extraArg; extraArg = extraArg->NextSiblingElement("EXTRA_ARG") ) {
      std::string key(swu::getTextFromChildOrEmpty(extraArg, "KEY"));
      if (key == "allow_force_dnl" || key == "ALLOW_FORCE_DNL") {
         std::string value(swu::getTextFromChildOrEmpty(extraArg, "VALUE"));
         ETG_TRACE_USR1(("Found. Value=%s", value.c_str() ));
         if (value == "true" || value == "TRUE" || value == "1") {
            allowForceDnl = true;
         }
         break;
      }
   }

   return allowForceDnl;
}

bool ConfiguratorHmiGeneric::overallAllowsAnyRelease(TiXmlElement *overallSection) {
   bool allowAnyRelease = false;

   // check for extra-arg
   TiXmlElement* extraArg = overallSection->FirstChildElement("EXTRA_ARG");
   ETG_TRACE_USR1(("%s EXTRA_ARG", (extraArg ? "Found" : "Did not find") ));
   for (extraArg; extraArg; extraArg = extraArg->NextSiblingElement("EXTRA_ARG") ) {
      std::string key(swu::getTextFromChildOrEmpty(extraArg, "KEY"));
      if (key == "allow_any_release" || key == "ALLOW_ANY_RELEASE") {
         std::string value(swu::getTextFromChildOrEmpty(extraArg, "VALUE"));
         ETG_TRACE_USR1(("Found. Value=%s", value.c_str() ));
         if (value == "true" || value == "TRUE" || value == "1") {
            allowAnyRelease = true;
         }
         break;
      }
   }

   return allowAnyRelease;
}

bool ConfiguratorHmiGeneric::overallAllowsMisCmcRelease(const TiXmlElement *overallSection){
	bool hasAllowMisCmcRelease = false;
	if (!overallSection) {
	  ETG_TRACE_USR1(("ConfiguratorHmiGeneric::overallAllowsMisCmcRelease Overall section not present" ));	
	  return hasAllowMisCmcRelease;
	}
    // check for extra-arg
   const TiXmlElement* extraArg = overallSection->FirstChildElement("EXTRA_ARG");
   ETG_TRACE_USR1(("%s EXTRA_ARG", (extraArg ? "Found" : "Did not find") ));
   for (extraArg; extraArg; extraArg = extraArg->NextSiblingElement("EXTRA_ARG") ) {
      std::string key(swu::getTextFromChildOrEmpty(extraArg, "KEY"));
      if (key == "allow_mis_cmc_release" || key == "ALLOW_MIS_CMC_RELEASE") {
		 ETG_TRACE_USR2(("ConfiguratorHmiGeneric::overallAllowsMisCmcRelease:: allow_mis_cmc_release present in Overall section"));
         std::string value(swu::getTextFromChildOrEmpty(extraArg, "VALUE"));
         ETG_TRACE_USR1(("allow_mis_cmc_release Found. Value=%s", value.c_str() ));
         if (value == "true" || value == "TRUE" || value == "1") {
            hasAllowMisCmcRelease = true;
         }
         break;
      }
   }
   return hasAllowMisCmcRelease;
}

bool ConfiguratorHmiGeneric::checkMisPendingMarker(){
	static const std::string misPendingMarkerFile="/var/opt/bosch/persistent/fcswupdate/misUpdatePending";
	bool isMisPendingMarkerPresent = false;
	isMisPendingMarkerPresent = swu::exists(misPendingMarkerFile);
	ETG_TRACE_USR1(("ConfiguratorHmiGeneric::checkMisPendingMarker isMisPendingMarkerPresent = %u", isMisPendingMarkerPresent ));
	return isMisPendingMarkerPresent;
   }
   
tenSwUpdateError ConfiguratorHmiGeneric::findReleaseXml() {
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml START")); 

   Msg_ResourceManagerChanged::vUnSubscribe(this);
   if(!isOtaUpdate()) {
      handleResource();
   }
   
   // set that will contain all releases from all sources:
   std::set<RelInfo> relInfos;
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml after relInfos"));     

   tenSwUpdateError errorCode = tenSwUpdateError_OK;   
   tenSwUpdateError detailedStatus = tenSwUpdateError_OK;
   std::list<TiXmlElement> xmlRelList;   
   bool notifyDTC=true;
   bool usingAlternateUpdatePath=false;
   _eUpdateDirection = FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Unknown;
               
   Config *cfg = Config::instance();
   tBool isInScomoMode = CampaignManager::instance()->isScomoMode();
   //reset the downgrade on detecting new USB stick
   cfg->cfg_ConfiguratorAllianceAllowDowngrade.set(0);

   std::map<std::string, trSourceInfo> sourcesList=PropDevMgr::instance()->get()._mapSources;
   // no need to store in member
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: numSources=%u", sourcesList.size()));
   if (sourcesList.size()) {
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Configurator: numSources=%u", 
                          sourcesList.size()));
   }
   
   _srvStoragePath = "";	
   
   // iterate over all sources
   //TO-DO errors individually, needs to be send for each source or combined
   for (std::map<std::string, trSourceInfo>::iterator iterSrc= sourcesList.begin();
        iterSrc!=sourcesList.end();
        ++iterSrc) {
      trSourceInfo source=iterSrc->second;
      
      if(isInScomoMode && source.owner != "FcSwUpdateScomoOtaIf") {          
         //blocking the other medium processing.
         ETG_TRACE_USR3(("SCOMO-Update is in Progress, don't process the others:%s", source.owner.c_str()));
         continue;         
      } 

      if(!source.bHasSupportedFormatType) {
         ETG_TRACE_USR3(("ConfiguratorHmiGeneric::findReleaseXml, medium has not support format"));

         if(!isIncompactMediumCanProceed(source.path, errorCode)) {
            ETG_TRACE_USR3(("ConfiguratorHmiGeneric::findReleaseXml, incompact medium cannot be proceed"));
            continue;
         }
      }

      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: source.path=%s", source.path.c_str()));
      std::string updateFolder=source.path + "/" + cfg->cfg_AlternativeUpdatePath.get() + "/";
      
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: check source %s", updateFolder.c_str()));
      
      struct stat st;
      if(cfg->cfg_AlternativeUpdatePath.get().length())
      {
	 //Coverity fix for 18596
         if (-1 == stat(updateFolder.c_str(),&st))
	 {
		ETG_TRACE_USR1(("stat has failed for the path %s",updateFolder.c_str()));
	 }
            if (S_ISDIR(st.st_mode)) {
               ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: set usingAlternateUpdatePath"));
               // if  Alternate Update Path existing, update from Alternative path otherwise root path
               source.path = updateFolder;
               usingAlternateUpdatePath=true;
         }
      }
      //overwrite the last source in usb_storage file. In case of multiple source, decision needed from HMI	  
      _srvStoragePath = source.path;	        
      // get release-list of current source
      FcSwUpdCore::instance()->getReleaseList(xmlRelList, source, &errorCode);	   
      
      bool hasForceDnl=swu::exists(_srvStoragePath + "/force.dnl");
      // To-Do Only one release being handled at the moment in BOSCH parser
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: found %u releases errorCode=%d", xmlRelList.size(),errorCode));
      // iterate over all releases of current source
      for ( std::list<TiXmlElement>::iterator iterRel=xmlRelList.begin();
            iterRel!=xmlRelList.end();
            ++iterRel) {		 
         TiXmlElement &overallSection=*iterRel;
         std::string releaseName = swu::getTextFromChild(&overallSection, "FINALNAME", false);
         ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: check release %s", releaseName.c_str()));
         RelInfo relInfo;
         relInfo.overallSection=overallSection;
         
         if ((cfg->cfg_ConfiguratorAcceptAllowForceDnlForAnyRelease.get() || overallAllowsAnyRelease(&overallSection)) 
             && hasForceDnl && overallAllowsForceDnl(&overallSection)) {
            rebootToRecoveryMode();
            relInfos.clear();
            errorCode=tenSwUpdateError_OK;
            break;
         }
         
         relInfo.rSourceInfo=source;
         if(source.enSourceType == tenSourceType_SCOMO_INDEX || source.enSourceType == tenSourceType_SCOMO) {
            swu::LOG_GLOBALINFO("version check is skipped for SCOMO updates");
            relInfos.insert(relInfo);
         } else {
            if (filterPassed(relInfo, detailedStatus)) {
               if (_mnfPathAndName.length() ) {
                  std::string fileName;
                  if (!swu::getFileNamePortion(_mnfPathAndName, fileName)) {
                     continue;
                  }
                  std::string mediumDrive=swu::getTextFromChildOrEmpty( &overallSection, "MEDIUM_DRIVE");
                  std::string basePath=swu::getTextFromChildOrEmpty( &overallSection, "BASEPATH");
                  std::string fileOnMedium=mediumDrive + "/" + basePath + "/" + fileName;
                  if (!swu::isSameFile(fileOnMedium, _mnfPathAndName)) {
                     continue;
                  }
               }
               ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: insert release %s", releaseName.c_str()));
               ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, 
                                   "Configurator: insert release insert release %s", releaseName.c_str()));
               relInfos.insert(relInfo);
            }		 
         }
      }
   }


   // ########################################################   
   std::string relName="";
    
   if (relInfos.empty() && _enState==enState_ConfigFound && _enConfigTrigger==enConfigTrigger_DevManagerChanged && cfg->cfg_NotifyMediumRemoval.get()) {
      errorCode = tenSwUpdateError_ERROR_MEDIUM_REMOVED;
      ETG_TRACE_USR2(("ConfiguratorHmiGeneric::findReleaseXml, medium removed"));
   }
   else if (sourcesList.empty()) {
      errorCode = tenSwUpdateError_ERROR_MEDIA_UNAVAILABLE;
      ETG_TRACE_USR2(("ConfiguratorHmiGeneric::findReleaseXml, medium unavailable"));
   }
   // No release Found, No parsing error detected
   else if(xmlRelList.empty() &&  cfg->cfg_ConfiguratorHmiDefaultToErrNoReleaseFound.get()){
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric:: No Release Found, Error=ERROR_NO_RELEASE_FOUND"));	
      errorCode = tenSwUpdateError_ERROR_NO_RELEASE_FOUND;
   }
   // valid release(s) found without parsing errors, but filtered out due to providing filtering release criteria.
   else if(!errorCode  && relInfos.empty() && tenSwUpdateError_OK != detailedStatus)
   {
      if (cfg->cfg_ConfiguratorHmiNotifyVersionConflictDetails.get())
      {
         ETG_TRACE_USR1(("ConfiguratorHmiGeneric:: No valid Release Found, detailed Error=%u", detailedStatus));
         errorCode = detailedStatus;
      }
      else
      {
         ETG_TRACE_USR1(("ConfiguratorHmiGeneric:: No valid Release Found, Error=ERROR_IMAGE_INCOMPATIBLE_VERSION"));
         errorCode = tenSwUpdateError_ERROR_IMAGE_INCOMPATIBLE_VERSION;
      }
   }	
  
   // iterate starting from best release
   // Error code will be overwritten in case of parsing error 
   TiXmlDocument *releaseXml=0;
   _stSelectedSourceInfo=trSourceInfo();
   for (std::set<RelInfo>::reverse_iterator iterRelInfo=relInfos.rbegin();
        iterRelInfo!=relInfos.rend();
        ++iterRelInfo) {
      RelInfo relInfo=*iterRelInfo;
      relName = swu::getTextFromChild(&relInfo.overallSection, "FINALNAME", false);
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: checkRelease= %s", relName.c_str()));	  
      // fetch releaseXml from map<ReleaseInfo>.front().
      releaseXml = FcSwUpdCore::instance()->getReleaseXml(relInfo.overallSection, errorCode);		      
      if (releaseXml) {
    	  _stSelectedSourceInfo= relInfo.rSourceInfo;
         ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: best release= %s: valid", relName.c_str()));
         break;
      } 
      else {
         // something was not OK with this release, try the next one
         ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: best release= %s: INVALID", relName.c_str()));
      }
   }
   
   ETG_TRACE_USR4(("Update Direction:%u", _eUpdateDirection));
   ETG_TRACE_USR4(("Selected SourceType:%u", ETG_CENUM(tenSourceType, _stSelectedSourceInfo.enSourceType)));
   if (releaseXml) { // We found a valid release.
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Configurator: found valid release"));      
	  
      SWU_ASSERT_RETURN_VAL( _util.setDoc(releaseXml),  tenSwUpdateError_ERROR_UNKNOWN);
      _util.setDefaultDecision(_enUpdateOption);	  

      // Only check for a force.dnl file, if "allow_force_dnl" was true
      ETG_TRACE_USR1(("Looking for force.dnl file in %s", _srvStoragePath.c_str() ));
      if (overallAllowsForceDnl(_util.getOverall()) && swu::exists(_srvStoragePath + "/force.dnl")) {
         rebootToRecoveryMode();
      }
         // If there is a force.dnl in the update path, reboot to recovery mode.

      if(!Config::instance()->cfg_DoIgnoreNormalModeHMIInteractions.get()) {
         if(!Config::instance()->cfg_HandleCentralLock.get()) {
         vSendReleaseDoc();	  
         } else {            
            vSendPreReleaseDoc(true);	  
            Msg_ResourceManagerChanged::vSubscribe(this);
            handleResource(true);
         }
      }
      //valid release, overwrite if any parsing error from other release
      errorCode = tenSwUpdateError_OK;	  
      vEnterState(enState_ConfigFound);	   	 
   } else {
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric::findReleaseXml: found no appropriate release"));
      ETG_TRACE_USR1(("ConfiguratorHmiGeneric:: Sending Empty ReleaseDoc"));
      if(!isInScomoMode) {
         vSendPreReleaseDoc(false);
         sendEmpyReleaseDoc();
      }
      vEnterState(enState_ConfigEmpty);	   	  
   }
   

	
   // isDir shall be overwritten in case of multiple sources, so shall update it later. 
   // If alternative update path doesn't exist, then ignore tenSwUpdateError_ERROR_METAINFO_NOT_FOUND error.
   if(!cfg->cfg_ConfiguratorHmiNotifyMetaInfoNotFound.get() && (errorCode == tenSwUpdateError_ERROR_METAINFO_NOT_FOUND))
   {
      errorCode=tenSwUpdateError_OK;
      notifyDTC = false;			
      
   }
   if(!cfg->cfg_ConfiguratorHmiNotifyMetaInfoNotFound.get() && (errorCode == tenSwUpdateError_ERROR_METAINFO_NOT_FOUND) && usingAlternateUpdatePath)
   {
      errorCode=tenSwUpdateError_OK;
      notifyDTC = false;			
   }
   
   // individually sending at the moment for each source
   // To-Do Handling of multiple errors or prioritiy in case of no valid release but multiple release with errors found		
   // notify (registered) ITC codes in case of parsing errors
   if(notifyDTC)
   {
      vNotifyConfiguratorError(errorCode);	
   }
	
   // To-do In case of multiple parsing errors with no appropriate release,
   // priority of errors need to be determined with individual release and pushed in parseError	
   //send parserError=tenSwUpdateError_OK  otherwise appropriate error
   FCSWUPD_NS_FCSWUPD_T::trErrorIds updateErrors;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::tenErrorId > parserError;
   parserError.push_back((FCSWUPD_NS_FCSWUPD_T::tenErrorId)errorCode);
   updateErrors.setErrorIds(parserError);
	
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric:: Sending updateErrors with errorCode=%d",errorCode));
   if (!cfg->cfg_SilentUpdate.get()) {
      swu::LOG_INTERFACE("updateErrors(error:%u)", errorCode);
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, updateErrors, UpdateErrors);		
   }
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Configurator:  errorCode=%u",
                       ETG_CENUM(tenSwUpdateError, errorCode)));

   GlobalLog::logSwuErrorOnce(errorCode);

   return errorCode;
}   

tVoid ConfiguratorHmiGeneric::handleResource(bool bAcquireResource) {
   ETG_TRACE_USR3(("ConfiguratorHmiGeneric::handleResource() - START bAcquireResource:%u", bAcquireResource));

   if(Config::instance()->cfg_HandleCentralLock.get()) {
      if(bAcquireResource) {          
         ResourceManager::instance()->configure(_enReqMode_reserve, CENTRAL_LOCK_TEXT);
      } else {
         ResourceManager::instance()->releaseResources();
      }
   }  
   ETG_TRACE_USR3(("ConfiguratorHmiGeneric::handleResource() - END"));  
}

tVoid ConfiguratorHmiGeneric::vSendPreReleaseDoc(bool bvalidRelease) {
   ETG_TRACE_USR4(( "ConfiguratorHmi::vSendPreReleaseDoc: START,bvalidRelease:%u",bvalidRelease) );
   swu::LOG_INTERFACE("vSendPreReleaseDoc");
   FCSWUPD_NS_FCSWUPD_T::trPreReleaseDoc preReleaseDoc;
   preReleaseDoc.setBFoundRelease(bvalidRelease);
   if (!Config::instance()->cfg_SilentUpdate.get()) {
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, preReleaseDoc, PreReleaseDoc);	
   }
   ETG_TRACE_USR4(( "ConfiguratorHmi::vSendPreReleaseDoc: END") );
}

void ConfiguratorHmiGeneric::vSendReleaseDoc() {    
   ETG_TRACE_USR4(( "ConfiguratorHmi::vSendReleaseDoc: START") );
   swu::LOG_INTERFACE("SendReleaseDoc");

   //for ota-update, fc_ota will acquire the lock
   Config *cfg = Config::instance();

   TiXmlDocument *xmlDoc=_util.getDocAccess();
   tU32 estimatedUpdateTimeSec = 0;
   SWU_ASSERT_RETURN( xmlDoc );
   _util.logXmlDoc(xmlDoc, "ConfiguratorXmlDocFinal.xml");
	  
   // -----------------------------------------------------
   // ready to begin processing of the Bosch xml to
   // prepare the trReleaseDoc object
   // -----------------------------------------------------
   // ::std::vector< FCSWUPD_NS_FCSWUPD_T::trConfigProblem > configProblemVec;
   TiXmlElement * pRelease = _util.getReleaseCopy( );
   SWU_ASSERT_RETURN( pRelease );		// Must not be null, there has to ba a release.
   bool isDeltaUpdate=swu::getUIntFromChild(pRelease, "HAS_DELTA");
   _scopes.clear();

   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trDeviceData >  deviceDataVec;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > moduleDataVec;
   ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > subModuleDataVec;
	  
   for (XmlItemAccessIter iterChildrenFirst(pRelease, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      XmlItemAccess access = iterChildrenFirst.getAccess();
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {	
         ConfigScope scope(access, _enUpdateOption, _decisions);
         _scopes.insert(scope);				          	
      }
   }	

   TiXmlElement *release = _util.getRelease();
   SWU_ASSERT_RETURN( release );  // There has to ba an overall-section.

   BXmlAccess releaseAccess(release);
   std::string releaseName = releaseAccess.getTextFromModule("FINALNAME_CUSTOMER");
   if (releaseName.empty() || releaseName=="unknown") {
      // no customer-version available, take bosch-version
      releaseName=releaseAccess.getTextFromModule("FINALNAME");
   }


   ETG_TRACE_USR4(("vSendReleaseDoc: releaseName:%s", releaseName.c_str()));
   std::string    hwIxdex = _util.getHardwareId();

   bool bNeedsFlashing=false;

   ETG_TRACE_USR4(( "vSendReleaseDoc: getHardwareId:%s", hwIxdex.c_str()));
   for (XmlItemAccessIter iterChildrenFirst(pRelease, XmlItemAccessIter::enOnLeave);!iterChildrenFirst.isDone();iterChildrenFirst.next()) {
      XmlItemAccess access = iterChildrenFirst.getAccess();
      ETG_TRACE_USR4(("iterChildrenFirst: START with elem: access-type:%u nameDisplay=%30s version=%s",
                      access.enGetType(),
                      access.getName(),
                      access.getVersion()));
	  
	  
      // add from-version from target     
      Scope::tenScope enScope=_scopes.get(access.getLongName()).enScope;
      FCSWUPD_NS_FCSWUPD_T::tenScope enAsfScope= (FCSWUPD_NS_FCSWUPD_T::tenScope)_scopes.get(access.getLongName()).enScope;
      ETG_TRACE_USR4(( "vSendReleaseDoc: scope(%40s)=%u", access.getLongName().c_str(), enAsfScope));

      if((enScope == Scope::enScope_Yes) || (enScope == Scope::enScope_Some))
      {
         bNeedsFlashing=true;
      }

      //Irrespective of the Scope the updateTime will be calculated now
      estimatedUpdateTimeSec+=access.getUpdateTime();
	  
      if (access.enGetType()==XmlItemAccess::enType_Submodule) {
         std::string fromVersion=SystemData::instance()->getCurrentModVersion(access.getLongName());

         ETG_TRACE_USR1(("iterChildrenFirst: Found subModule: %s", access.getName()));
         char versionTrans[100];	 
         sprintf(versionTrans, "%s -> %s", fromVersion.c_str(), access.getVersion());
         ETG_TRACE_USR1(("versionTrans:%s", versionTrans));            
         bool bDoUpdate=(enScope==Scope::enScope_Yes);
	
         FCSWUPD_NS_FCSWUPD_T::trSubModuleData *submodule = new FCSWUPD_NS_FCSWUPD_T::trSubModuleData();
         SWU_ASSERT_RETURN( submodule );
         submodule->setName(access.getName());
         ETG_TRACE_USR1(("Submodule Name:%s", access.getName()));				
         submodule->setBSelected(bDoUpdate);
         ETG_TRACE_USR1(("setBSelected %u", bDoUpdate));
         submodule->setInfo(versionTrans);
         submodule->setEnScope(enAsfScope);			
         ETG_TRACE_USR1(("enAsfScope %u", enAsfScope));
         submodule->setFromVersion(fromVersion);
         submodule->setToVersion(access.getVersion());
         
         std::string strSizeBytes=swu::getTextFromChild(access.getXml(), "DNL_SIZE", false, "0");
         ETG_TRACE_USR1(("sizeBytes=%s", strSizeBytes.c_str());
         int64_t sizeBytes;
         std::istringstream ss(strSizeBytes);
         ss >> sizeBytes;
         submodule->setSizeBytes(sizeBytes));

         const TiXmlElement *compatInfo = access.getXml()->FirstChildElement("COMPAT_INFO");
         if (compatInfo)
         {
        	 std::string fmt = swu::getTextFromChildOrEmpty(compatInfo, "FORMAT");
             ETG_TRACE_USR1(("CompatInfo: format=%s", fmt.c_str()));
        	 std::transform(fmt.begin(), fmt.end(), fmt.begin(), ::toupper);
        	 std::string data = swu::getTextFromChild(compatInfo, "DATA");
        	 ::std::vector< uint8 > info;
        	 if (fmt == "BASE64")
        	 {
                 ETG_TRACE_USR1(("CompatInfo: Convert data"));
        		 info = swu::decodeBase64(data);
        	 }
        	 else
        		 info = ::std::vector< uint8 >(data.begin(), data.end());
             ETG_TRACE_USR1(("CompatInfo: data =%s", ::std::string(info.begin(), info.end()).c_str()));
             submodule->setCompatInfo(info);
         }

         subModuleDataVec.push_back(*submodule);    
      }
      else if (access.enGetType()==XmlItemAccess::enType_Module) {
         FCSWUPD_NS_FCSWUPD_T::trModuleData *module = new FCSWUPD_NS_FCSWUPD_T::trModuleData();		
         SWU_ASSERT_RETURN( module );
         module->setName(access.getName());
         ETG_TRACE_USR1(("Module Name:%s", access.getName()));				
         module->setHwIndex(hwIxdex);
         ETG_TRACE_USR1(("HwIxdex:%s", hwIxdex.c_str()));
         module->setSubModules(subModuleDataVec);
         module->setEnScope(enAsfScope);			
         ETG_TRACE_USR1(("enAsfScope %u", enAsfScope));
         moduleDataVec.push_back(*module);
         subModuleDataVec.clear();
	 
         TiXmlElement* temp = access.getXml();
         estimatedUpdateTimeSec = estimatedUpdateTimeSec + getUpdateTimeFromScripts(temp->FirstChildElement("PREPARE_ONCE"));
         estimatedUpdateTimeSec = estimatedUpdateTimeSec + getUpdateTimeFromScripts(temp->FirstChildElement("FINALIZE"));
      }
      else if (access.enGetType()==XmlItemAccess::enType_Device) {
         FCSWUPD_NS_FCSWUPD_T::trDeviceData *device = new FCSWUPD_NS_FCSWUPD_T::trDeviceData();
         SWU_ASSERT_RETURN( device );
         device->setName(access.getName());
         ETG_TRACE_USR1(("Device Name:%s", access.getName()));			
         device->setModules(moduleDataVec);
         device->setEnScope(enAsfScope);
         ETG_TRACE_USR1(("enAsfScope %u", enAsfScope));
         deviceDataVec.push_back(*device);
         moduleDataVec.clear();
      }
      else {
         ETG_TRACE_USR4(("iterChildrenFirst: unhandled access-type:%u nameDisplay=%30s version=%s",
                         access.enGetType(),
                         access.getName(),
                         access.getVersion()));
      }
   } 
   //print build device vector
   for (::std::vector<FCSWUPD_NS_FCSWUPD_T::trDeviceData>::iterator it = deviceDataVec.begin() ; it != deviceDataVec.end(); ++it)
   {
      ::std::vector< FCSWUPD_NS_FCSWUPD_T::trModuleData > moduleFetch;
      printf("\n\n Device Name = %s",it->getName().c_str());
      moduleFetch = it->getModules();
      for (::std::vector<FCSWUPD_NS_FCSWUPD_T::trModuleData>::iterator it2 = moduleFetch.begin() ; it2 != moduleFetch.end(); ++it2)
      {
         printf("\n Module Name = %s", it2->getName().c_str());
         ::std::vector< FCSWUPD_NS_FCSWUPD_T::trSubModuleData > subModuleFetch;
         subModuleFetch = it2->getSubModules();
         for (::std::vector<FCSWUPD_NS_FCSWUPD_T::trSubModuleData>::iterator it3 = subModuleFetch.begin() ; it3 != subModuleFetch.end(); ++it3)
            printf("\n sub Module Name = %s", it3->getName().c_str());			
      }
   }
   
   // search for matching config-option
   FCSWUPD_NS_FCSWUPD_T::trReleaseDoc releaseDoc;    
   releaseDoc.setName(releaseName);
   // property issue

   if (cfg->cfg_ConfiguratorHmiNeedsFlashingBasedOnTrain.get()) {
      BXmlAccess access(release);
      std::string finalName = access.getTextFromModule("FINALNAME");
      releaseDoc.setBNeedsFlashing(cfg->cfg_RunningFinalName.get() != finalName);
   } else {
      releaseDoc.setBNeedsFlashing(bNeedsFlashing);
   }
   releaseDoc.setBIsDeltaUpdate(isDeltaUpdate);
   releaseDoc.setEnUpdateDirection(_eUpdateDirection);
   if(_stSelectedSourceInfo.enSourceType == tenSourceType_SCOMO) {
      releaseDoc.setEnSourceType(FCSWUPD_NS_FCSWUPD_T::tenSourceType__Ota);
   } else {
      releaseDoc.setEnSourceType((FCSWUPD_NS_FCSWUPD_T::tenSourceType)_stSelectedSourceInfo.enSourceType);
   }

   releaseDoc.setDevices(deviceDataVec);
   releaseDoc.setEnUpdOptions((FCSWUPD_NS_FCSWUPD_T::tenUpdOptions)_enUpdateOption);
   releaseDoc.setU32EstimatedUpdateTimeSec(estimatedUpdateTimeSec);   
   if (!cfg->cfg_SilentUpdate.get()) {
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, releaseDoc, ReleaseDoc);	
   }
   ETG_TRACE_USR2(("ConfiguratorHmi::vSendReleaseDoc: END"));
}       

tU32 ConfiguratorHmiGeneric::getUpdateTimeFromScripts(TiXmlElement* elem) {
   tU32 updateTime = 0;
   if(!elem) {
      return 0;
   }
   for (TiXmlElement *relScriptItem=elem->FirstChildElement("SCRIPT");
        relScriptItem;
        relScriptItem=relScriptItem->NextSiblingElement("SCRIPT")) {
      updateTime = updateTime + swu::getUIntFromChild(relScriptItem, "UPDATE_TIME");
   }
   return updateTime;
}

bool ConfiguratorHmiGeneric::doUpdate(XmlItemAccess &item, ConfiguratorUtil::tenConfigOption enConfigOption) {
#if 1
   // todo: check decisions
   bool bRes = false;
   char const *itemState=swu::getTextFromChild(item.getXml(), "STATE");
   std::string fromVersion=item.getFromVersion();
   std::string toVersion=item.getVersion();
   if (enConfigOption==ConfiguratorUtil::enConfigOption_UserDefined) {
      bRes = _decisions[item.getLongName()];
   }
   else if (swu::streq(itemState, "STICK_ONLY")) {
      bRes = true;
   }
   else {
      switch (enConfigOption) {
         case ConfiguratorUtil::enConfigOption_MetaInfo:
         {

            bRes = (toVersion > fromVersion);
            if (!bRes) {
               char const *autoFollowName=swu::getTextFromChild(item.getXml(), "AUTO_FOLLOW", false);
               if (autoFollowName) {
                  ETG_TRACE_USR1(("ConfiguratorHmi::doUpdate: autoFollowName=%s", autoFollowName));
                  XmlItemAccess release=item.getRelease();
                  XmlItemAccess autoFollowItem=XmlItemAccess(getBoschXmlElem(release.getXml(), autoFollowName));
                  bRes = doUpdate(autoFollowItem, enConfigOption);
               }
               ETG_TRACE_USR1(("ConfiguratorHmi::doUpdate: after autoFollowName=%u", bRes));

            }

            break;
         }
         case ConfiguratorUtil::enConfigOption_NewOnly:
         {
            bRes = (toVersion > fromVersion);
            break;
         }
         case ConfiguratorUtil::enConfigOption_DifferentOnly:
         {
            bRes = (toVersion != fromVersion);
            break;
         }
         case ConfiguratorUtil::enConfigOption_Any:
         {
            bRes = true;
            break;
         }
         //Coverity fix for 21772. Removing the case ConfiguratorUtil::enConfigOption_UserDefined this case is handled in the above if condition

         default: 
         {
            break;
         }      
      }
   }
   std::string strName = item.getLongName();  //Coverity fix for 209436
   ETG_TRACE_USR1(("ConfiguratorHmi::doUpdate: item: name=%20s itemState=%15s enConfigOption=%u from:%20s to:%20s bRes=%u", 
                   strName.c_str(),
                   itemState,
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, enConfigOption),                   
                   fromVersion.c_str(),
                   toVersion.c_str(),
                   bRes));

   return bRes;
#else
   return true;
#endif
}

// ------------------------------------------------------------------------------------------
// Constructor
// ------------------------------------------------------------------------------------------
ConfiguratorHmiGeneric::ConfiguratorHmiGeneric() : 
   _enState(enState_Initial),
   _enConfigTrigger(enConfigTrigger_Invalid),
   _eUpdateDirection ( FCSWUPD_NS_FCSWUPD_T::tenUpdateDirection__Unknown ) //Coverity fix for 22862
{
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric CTOR this=%p", this));
   _enFilterOption=(ConfiguratorUtil::tenConfigOption)Config::instance()->cfg_HmiCfgDefaultReleaseFilter.get();
   _enUpdateOption=(ConfiguratorUtil::tenConfigOption)Config::instance()->cfg_HmiCfgDefaultUpdateOption.get();
   FCSWUPD_NS_FCSWUPD_T::trReleaseFilter asfReleaseFilter;
   asfReleaseFilter.setEnUpgradeDirectionFilter((FCSWUPD_NS_FCSWUPD_T::tenUpdOptions)_enFilterOption);
   if (!Config::instance()->cfg_SilentUpdate.get()) {
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, asfReleaseFilter, ReleaseFilter);		
   }

}

// ------------------------------------------------------------------------------------------
//
// ------------------------------------------------------------------------------------------
ConfiguratorHmiGeneric::~ConfiguratorHmiGeneric()
{
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric DTOR, this=%p", this));
}

// ------------------------------------------------------------------------------------------
// TODO:
//
// ------------------------------------------------------------------------------------------
void ConfiguratorHmiGeneric::traceState() {
   ETG_TRACE_COMP(("ConfiguratorHmiGeneric traceState"));

   ETG_TRACE_COMP(("  _enState=%u",ETG_CENUM(ConfiguratorHmiGeneric::tenState, _enState)));
   if (_enState==enState_ConfigFound) {
      // todo: fetch some information from release-doc
   }
}

// ------------------------------------------------------------------------------------------
// unsubsribe will be done automatically on destruction of this class.
// ------------------------------------------------------------------------------------------
tVoid ConfiguratorHmiGeneric::vInit() { 
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::vInit"));
        
   SWU_ASSERT_RETURN(_enState == enState_Initial);
   // subscribe for messages.
   Msg_ApplyDocRequest::vSubscribe(this);
   Msg_UpdateEndRequest::vSubscribe(this);
   Msg_NotifyDevManagerChanged::vSubscribe(this);
   Msg_UpdateSelectReleaseByFileRequest::vSubscribe(this);
   Msg_UpdateSetReleaseFilterRequest::vSubscribe(this);
   Msg_SetUpdateOptionsRequest::vSubscribe(this);

   _enState = enState_WaitMediaInfo;
   Config *cfg = Config::instance();
   if (cfg->cfg_EnterFullOperation.get() && cfg->cfg_SpecialReleasePathAndName.get().length()) {
      /*when we reach full operation again, we forget about the special release.
        we cope with the race-condition, that the specialRelease has been sent to ctrl,
        but the update has for some reasons not started before reset.
        we assume, that special releases are always flashed in recovery mode.
      */
      SpecialRelease::instance()->clear();
   }

   // start timer to make sure that we receive Msg_NotifyDevManagerChanged
   _tiggerDevMgrTimer.start(this, 1000);
   vEnterState(enState_WaitMediaInfo);
   FcSwUpdSrv::instance()->sendUpdateEnterResponse(TRUE);
}

tVoid ConfiguratorHmiGeneric::vDeInit() {
   vEnterState(enState_Terminated);
}

// Receiving upgrade direction from the HMI
tVoid ConfiguratorHmiGeneric::vProcess(Msg_UpdateSetReleaseFilterRequest *pMsg){
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSetReleaseFilterRequest) START"));
   _enConfigTrigger=enConfigTrigger_SetReleaseFilter;

   FCSWUPD_NS_FCSWUPD_T::trReleaseFilter const &releaseFilter = pMsg->payload->getFilter();
   _enFilterOption = (ConfiguratorUtil::tenConfigOption)releaseFilter.getEnUpgradeDirectionFilter();
   FCSWUPD_NS_FCSWUPD_T::trReleaseFilter asfReleaseFilter;
   asfReleaseFilter.setEnUpgradeDirectionFilter((FCSWUPD_NS_FCSWUPD_T::tenUpdOptions)_enFilterOption);
   if (!Config::instance()->cfg_SilentUpdate.get()) {
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, asfReleaseFilter, ReleaseFilter);
   }

   findReleaseXml();
 
  FcSwUpdSrv::instance()->sendUpdateSetReleaseFilterResponse(true,releaseFilter);	
   _enConfigTrigger=enConfigTrigger_Invalid;
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSetReleaseFilterRequest) END"));
}

// Receiving update options for submodule flashing
tVoid ConfiguratorHmiGeneric::vProcess(Msg_SetUpdateOptionsRequest *pMsg){
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_SetUpdateOptionsRequest) START"));
   _enConfigTrigger=enConfigTrigger_SetUdateOptions;
   FCSWUPD_NS_FCSWUPD_T::tenUpdOptions updateOption = pMsg->payload->getEnOption();
   _enUpdateOption=(ConfiguratorUtil::tenConfigOption)updateOption;
   findReleaseXml();

   FcSwUpdSrv::instance()->sendSetUpdateOptionsResponse(true,updateOption);      
   _enConfigTrigger=enConfigTrigger_Invalid;
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_SetUpdateOptionsRequest) END"));
}

tVoid ConfiguratorHmiGeneric::sendUpdateSelectReleaseByFileResponse(tenSwUpdateError error) {
   bool success=(error==tenSwUpdateError_OK);
   ETG_TRACE_COMP(( "ConfiguratorHmi::sendUpdateSelectReleaseByFileResponse(): success=%u error=%u", 
                    success, ETG_CENUM(tenSwUpdateError, error)));

   FcSwUpdSrv::instance()->sendUpdateSelectReleaseByFileResponse ( true,  (FCSWUPD_NS_FCSWUPD_T::tenErrorId)error);
   Msg_UpdateSelectReleaseByFileResult * msg=new Msg_UpdateSelectReleaseByFileResult(success);
   msg->bNotifyLater();
   msg=0;
}

tVoid ConfiguratorHmiGeneric::vProcess (Msg_UpdateSelectReleaseByFileRequest *pMsg) {
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest)"));
   FCSWUPD_NS_FCSWUPD_T::trReleaseAccess const &releaseAccess = pMsg->payload->getReleaseAccess();
   if (_enState!=enState_WaitMediaInfo &&
       _enState!=enState_ConfigEmpty &&
       _enState!=enState_ConfigDone) {
      ETG_TRACE_COMP(( "ConfiguratorHmi::vProcess(Msg_UpdateSelectReleaseByFileRequest): wrong state=%u", _enState));
      sendUpdateSelectReleaseByFileResponse(tenSwUpdateError_ERROR_UNKNOWN);
      return;
   }
   if (releaseAccess.getUseExtendedOptions() && releaseAccess.getPostImageInstallationMode()) {
      Config::instance()->cfg_PostImageInstallationRunning.set(1);
   }
   _mnfPathAndName=releaseAccess.getMountPoint() + "/" + releaseAccess.getPath() + "/" + releaseAccess.getFile(); 
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest):_mnfPathAndName=%s", _mnfPathAndName.c_str()));
   if (!swu::isFile(_mnfPathAndName)) {
      ETG_TRACE_ERR(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest):tenSwUpdateError_ERROR_METAINFO_NOT_FOUND"));
      sendUpdateSelectReleaseByFileResponse(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
      return;
      
   }
   SpecialRelease::instance()->set(_mnfPathAndName);

   if (releaseAccess.getUseExtendedOptions()) {
      FCSWUPD_NS_FCSWUPD_T::trReleaseFilter const &asfReleaseFilter = releaseAccess.getReleaseFilter();
      _enFilterOption = (ConfiguratorUtil::tenConfigOption)asfReleaseFilter.getEnUpgradeDirectionFilter();
      _enUpdateOption=(ConfiguratorUtil::tenConfigOption) releaseAccess.getUpdateOption();
   }
   
   tenSwUpdateError enError=findReleaseXml();

   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest):_enState=%u enState_ConfigFound=%u _mnfPathAndName=%s", _enState, enState_ConfigFound, _mnfPathAndName.c_str()));
   if (_enState!=enState_ConfigFound) {
      ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest: clear _mnfPathAndName"))
         _mnfPathAndName="";
      sendUpdateSelectReleaseByFileResponse(enError == tenSwUpdateError_OK ? tenSwUpdateError_ERROR_UNKNOWN : enError);
      return;
   }

   // from here on we assume, that no errors should occure
   sendUpdateSelectReleaseByFileResponse(tenSwUpdateError_OK);
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_UpdateSelectReleaseByFileRequest) END"))
         
   if (releaseAccess.getUseExtendedOptions() && releaseAccess.getDoApply()) {
      Msg_ApplyDocRequest *selfMsg;
      vProcess(selfMsg);
   }

}
// ------------------------------------------------------------------------------------------
// request ApplyDoc() : (boolean bOk)
// data in:  none
// response: using vNotifyReady( true );
// ------------------------------------------------------------------------------------------
tVoid ConfiguratorHmiGeneric::vProcess (Msg_ApplyDocRequest *pMsg) {

   //If mapDownloadLock is with Map Update, HMI will not give the ApplyDoc request.
   //Eventhough lock is with map, FcSwUpdate has to provide the ReleaseDoc.

   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric::vProcess(Msg_ApplyDocRequest)"));
   swu::LOG_INTERFACE("ApplyDocRequest");
   Msg_ResourceManagerChanged::vUnSubscribe(this);

   if (_enState==enState_ConfigFound) {
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Configurator:  applyDocRequest"));

      _tiggerDevMgrTimer.stop();
      Msg_NotifyDevManagerChanged::vUnSubscribe(this);      
      if (Config::instance()->cfg_EnterPimpedRecovery.get())
      {
         rebootToRecoveryMode();
      }

      // tell main that configuration is ready to be flashed
      vEnterState(enState_ConfigDone);
      vNotifyReady( true );	
   }
   else {
      ETG_TRACE_ERR_CLS((TR_CLASS_FCSWUPDATE_CUST, "Configurator:  applyDocRequest, but no release found"));
      // nothing to apply, send negative response to HMI
      FcSwUpdSrv::instance()->sendApplyDocResponse( false);
   }
}

tVoid ConfiguratorHmiGeneric::vProcess (Msg_ResourceManagerChanged *pMsg) {
   ETG_TRACE_COMP(("ConfiguratorHmiGeneric vProcess(Msg_ResourceManagerChanged)"));
   vSendReleaseDoc();	
}
// ------------------------------------------------------------------------------------------
// HMI terminates update-session.
// data in:   none
// response:  via vNotifyReady( false );
// ------------------------------------------------------------------------------------------
tVoid ConfiguratorHmiGeneric::vProcess (Msg_UpdateEndRequest *pMsg) {
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric vProcess(Msg_UpdateEndRequest)"));
   // send ack to caller, todo: check if this is also needed in configuratorHmi
   Msg_ResourceManagerChanged::vUnSubscribe(this);
   handleResource();
   vEnterState(enState_Terminated);
   FcSwUpdSrv::instance()->sendUpdateEndResponse (true);

}

tVoid ConfiguratorHmiGeneric::vProcess (ConfiguratorHmiGeneric::Msg_TiggerDevMgrTimer *pMsg) {
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric vProcess(Msg_TiggerDevMgrTimer) START"));
   if (_enState==enState_WaitMediaInfo) {
      ETG_TRACE_COMP(( "ConfiguratorHmiGeneric vProcess(Msg_TiggerDevMgrTimer) CALL"));
      Msg_NotifyDevManagerChanged msg;
      vProcess(&msg);
   }
   ETG_TRACE_COMP(( "ConfiguratorHmiGeneric vProcess(Msg_TiggerDevMgrTimer) END"));
}


/* Config is ready, client pulls the xmlDoc.
   Configurator should not be used from here on
*/
TiXmlDocument *ConfiguratorHmiGeneric::getXmlDoc() {
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::getXmlDoc START _mnfPathAndName=%s", _mnfPathAndName.c_str()));
   /*
     return our prepared boschXml
   */
   Config::instance()->cfg_UpdateStartDistanceTotalizer.set(FcSwUpdCore::getIfOrDefault<DistTotalizerIf>()->getDistTotal());

   swu::LocalTime lc;
   FcSwUpdCore::getIfOrDefault<LocalTimeIf>()->getLocalTime(lc);
   Config::instance()->cfg_UpdateStartTime.set(lc.toString());

   TiXmlDocument *doc=_util.pullDocAndClear();
   if (doc) {
      _util.logXmlDoc(doc, "getXmlDoc.xml");
   } else {
      ETG_TRACE_ERR(("ConfiguratorHmiGeneric::pullDocAndClear returned 0"));
   }
   if (_mnfPathAndName.length()) {
      ETG_TRACE_ERR(("ConfiguratorHmiGeneric::storing special release"));
      SpecialRelease::instance()->store();
   }

   vEnterState(enState_Terminated);
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::getXmlDoc END"));
   return doc;
}


tVoid ConfiguratorHmiGeneric::sendEmpyReleaseDoc() {
   swu::LOG_INTERFACE("sendEmpyReleaseDoc");

   FCSWUPD_NS_FCSWUPD_T::trReleaseDoc releaseDoc;    
   releaseDoc.setBNeedsFlashing(false);
   releaseDoc.setName("");
   releaseDoc.setEnUpdateDirection(_eUpdateDirection);
   if (!Config::instance()->cfg_SilentUpdate.get()) {
      FCSWUPD_SET_ASF_PROPERTY(FcSwUpdSrv, releaseDoc, ReleaseDoc);		
   }
}

// ------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------
tVoid ConfiguratorHmiGeneric::vEnterState(tenState newState) {
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::vEnterState: old state=%u new state=%u", _enState, newState ));
   _enState = newState;
   switch (_enState) {
      case enState_ConfigDone:
//         sendEmpyReleaseDoc();
         _util.vSetHmiState(tenPrjState_Config);
         break;       
      case enState_ConfigFound:
         _util.vSetHmiState(tenPrjState_Config);
         break;
      case enState_ConfigEmpty:
         _util.vSetHmiState(tenPrjState_SelectRelease);
         _util.clear();
         break;
      case enState_Initial:
      case enState_WaitMediaInfo:
         _util.vSetHmiState(tenPrjState_SelectRelease);
         _util.clear();
         break;
      case enState_Terminated:
         // we are no longer responsible for the HMI-state
         _util.clear();
         break;
   }
}

tVoid ConfiguratorHmiGeneric::vNotifyConfiguratorError(tenSwUpdateError errorCode) { 
   ETG_TRACE_USR1(("ConfiguratorHmiGeneric::vNotifyConfiguratorError:errorCode=%u", errorCode));
   std::list< uint32_t > errors;
   errors.push_back( (uint32_t) errorCode);  
  
   Msg_NotifyConfiguratorError *pMesg = OSAL_NEW Msg_NotifyConfiguratorError(errors);
   SWU_ASSERT_RETURN(pMesg);
   pMesg->bNotifyLater();
   pMesg = 0;

}

void ConfiguratorHmiGeneric::ConfigScope::addChild(ConfiguratorHmiGeneric::ConfigScope const &child) {
   itemName=getParentName(child.itemName);
   ETG_TRACE_USR1(("ConfigScope(%20s with scope=%u)::addChild(%20s with scope=%u)", itemName.c_str(), enScope, child.itemName.c_str(), child.enScope));
   enScope=child.enScope;
   if (enScope==enScope_Yes && child.enScope!=enScope_Yes) {
      enScope=enScope_Some;
   }
   if (enScope==enScope_NotApplicable) {
      enScope=child.enScope;
   }
   if (child.enScope==enScope_Some) {
      enScope=enScope_Some;
   }
   ETG_TRACE_USR1(("ConfigScope(%20s)::addChild, new scope=%u", 
                   itemName.c_str(), enScope));
}

ConfiguratorHmiGeneric::ConfigScope::ConfigScope( XmlItemAccess item, ConfiguratorUtil::tenConfigOption enConfigOption,std::map<std::string, bool> const &decisions):Scope(item.getLongName()) {

   std::string fromVersion=SystemData::instance()->getCurrentModVersion(item.getLongName());
   if(fromVersion == "unknown") {
     enScope = enScope_Yes;
     return;
   }
   std::string toVersion=item.getVersion();

   bool isUpgrade=(fromVersion < toVersion);
   bool isDowngrade=(toVersion < fromVersion);
   bool isUnchanged=(!isUpgrade && !isDowngrade);

   bool res=false;
   switch(enConfigOption)
   {
      // if the release is newer than the current release
      case ConfiguratorUtil::enConfigOption_NewOnly:  
      {
         enScope = (isUpgrade) ? enScope_Yes : enScope_No;
         break;
      }
	  
	  // if the release is newer or same than the current release
      case ConfiguratorUtil::enConfigOption_NewAndSameOnly:  
      {
         enScope = (isUpgrade || isUnchanged) ? enScope_Yes : enScope_No;
         break;
      }
      // current and available releases are different
      case ConfiguratorUtil::enConfigOption_DifferentOnly:
      {
         enScope = (isUnchanged) ? enScope_No : enScope_Yes;
         break;
      }			  
      //default decision any for the moment or  explicit option enConfigOption_Any
      default:
      {
         enScope =enScope_Yes;
         break;
      }					
   } 	  	  
   std::string strName = item.getLongName();  //Coverity fix for 209522
   ETG_TRACE_USR1(("ConfigScope:.ConfigScope(%20s), fromVersion=%30s toVersion=%30s" 
                   "configOption=%u new scope=%u", 
                   strName.c_str(), fromVersion.c_str(), toVersion.c_str(),
                   ETG_CENUM(ConfiguratorUtil::tenConfigOption, enConfigOption),
                   ETG_CENUM(Scope::tenScope, enScope)));

  
}

bool ConfiguratorHmiGeneric::isIncompactMediumCanProceed(std::string path, tenSwUpdateError& status) {
   ETG_TRACE_USR2(("ConfiguratorHmiGeneric::isIncompactMediumCanProceed START path:%s", path.c_str()));

   bool bResult=false;
   if(swu::exists(path + "/bosch.xml")) {
      ETG_TRACE_USR2(("ConfiguratorHmiGeneric::isIncompactMediumCanProceed, it's firmware update stick"));

      if(swu::exists(path + "/force.dnl")) {
         ETG_TRACE_USR2(("ConfiguratorHmiGeneric::isIncompactMediumCanProceed, forcing update"));
         bResult=true;
      }

      if(!bResult) {
         status = tenSwUpdateError_ERROR_INCOMPATIBLE_USB_FORMAT;
      }
   }

   ETG_TRACE_USR2(("ConfiguratorHmiGeneric::isIncompactMediumCanProceed END bResult:%u", bResult));
   return bResult;
}

ConfiguratorUtil::tenConfigOption ConfiguratorHmiGeneric::getConfigOptionFromMetaInfo(XmlItemAccess subModuleAccess) {
   // default is newOnly;
   ConfiguratorUtil::tenConfigOption configOption=ConfiguratorUtil::enConfigOption_NewOnly;
   // check if there is a special setting in metainfo
   char const *metainfoOption=swu::getTextFromChild(subModuleAccess.getXml(), "UPDATE_MODE", false);
   if (metainfoOption) {
      ETG_TRACE_USR4(("found metainfo-decision:%s", metainfoOption));
      if (!strcmp(metainfoOption,"Any")) {
         configOption=ConfiguratorUtil::enConfigOption_Any;
      }
      else if (!strcmp(metainfoOption,"DifferentOnly")) {
         configOption=ConfiguratorUtil::enConfigOption_DifferentOnly;         
      }
	  else if (!strcmp(metainfoOption,"NewAndSameOnly")) {
	    configOption=ConfiguratorUtil::enConfigOption_NewAndSameOnly;         		
      }
      /*
        "NewOnly" is already handled by initialization of configOption
      */
   }
   return configOption;
}
}	// namespace
