#include "util/swu_util.hpp"
#include "util/swu_member.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_execCommand.h"
#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "fcswupdatesrv/FcSwUpdateSrvConst.h"

#include "config/fcswupd_config.hpp"
#include <sys/stat.h>
#include <sys/types.h>
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_globalLog.h"
#include "main/fcswupd_component.h"
#include "main/fcswupd_propDevMgr.h"
#include "main/fcswupd_tftpIf.h"

/* I think, we won't have a device-mgr here
   for the time beeing be should hardcode the ip-address 127.17.0.53

   If we don't find a release the first time, we could use a timer here to try again.
*/

#include "main/fcswupd_parserIf.h"
#include "main/fcswupd_mainMessages.hpp"

#include "util/fcswupd_cust_update_info.hpp"

#include "configurator/fcswupd_configuratorEngineering.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_CONFIGURATOR
#include "trcGenProj/Header/fcswupd_configuratorEngineering.cpp.trc.h"
#endif


namespace fcswupdate {

static const tUInt _triggerTftpIntervalMs=1000; 


/* we expect the release-list to contain only one element, we don't have to sort */

ConfiguratorEngineering::ConfiguratorEngineering() : 
   _enState(enState_Initial), 
   _releaseXml(0)
{
   ETG_TRACE_USR1(("ConfiguratorEngineering CTOR"));
}

ConfiguratorEngineering::~ConfiguratorEngineering() 
{
   ETG_TRACE_USR1(("ConfiguratorEngineering DTOR"));
   vDeInit();
}

// unsubsribe will be done automatically on destruction of this class.
// Expect to skip  _enState = enState_Initial;
//
// It is possible that one or more of the options available
//  ( that is - on the media and the last overall provided by Parser in the overall list )
// we have to take care NOT to use dissimilar release names
//    - customer update vs non customer update.
/*
  peha: customer-updates shall never be used for emergency-update
*/
tVoid ConfiguratorEngineering::vInit() 
{
   _enState = enState_SearchRelease;
   //for emmc firmware version
   swu::execCommand("/opt/bosch/base/bin/swupdatecontrol_out.out --emmcinfo");
   Msg_NotifyDevManagerChanged::vSubscribe(this);
   TftpIf::instance()->start(1000);
}


tVoid ConfiguratorEngineering::vDeInit() 
{
   if (_releaseXml) 
   {
      delete _releaseXml;
      _releaseXml = 0;
   }
   _enState=enState_Terminated;
}


tVoid ConfiguratorEngineering::vProcess(Msg_NotifyDevManagerChanged *pMsg)
{
   (void)pMsg; 
   SWU_ASSERT_RETURN(_enState == enState_SearchRelease);

   //fetch bosch.xml via tftp and store it to tmp-file
   findReleaseXml();

   if (_enState != enState_SearchRelease) {
      TftpIf::instance()->stop();
   }
}




tVoid ConfiguratorEngineering::checkV850CompatUpdate(TiXmlDocument *releaseDoc) {
      ETG_TRACE_USR1(("ConfiguratorEngineering::checkV850CompatUpdate() START"));
   if (!Config::instance()->cfg_ConfiguratorDeveloperForceV850Compat.get()) {
      ETG_TRACE_USR1(("ConfiguratorEngineering::checkV850CompatUpdate(): cfg_ConfiguratorDeveloperForceV850Compat not set"));
      return;   
   }
   /* in developer-update, v850 and imx-boot a not allways in sync since the since 
      imx-boot is running from the new SW and the V850 is still the one from the old SW.
      So we only keep the V850 device for this update and for a reset afterwards via a extra-arg.
   */
   TiXmlElement *pRelease=getReleaseSection(releaseDoc);
   SWU_ASSERT_RETURN(pRelease);

   // find v850-device
   TiXmlElement *subModuleXml;   
   subModuleXml=getBoschXmlElem(pRelease, "IVI/FIRMWARE/V850");  
   if (!subModuleXml) {     
      ETG_TRACE_USR1(("ConfiguratorEngineering::checkV850CompatUpdate(): IVI/FIRMWARE/V850 not found"));         
      return;
   }
   XmlItemAccess access=XmlItemAccess(subModuleXml);

   // check if feature is enabled in bosch.xml (extra-arg in release-section)
   if (!access.getExtraArgsValAsBool("force_v850_version_in_developer_update")) {
      ETG_TRACE_USR1(("ConfiguratorEngineering::checkV850CompatUpdate():  enforcing of v850 version disabled"));
      return;
   }


   

   std::string stickVersion=access.getVersion();
   std::string installedVersion=SystemData::instance()->getCurrentModVersion(access.getLongName());
   ETG_TRACE_USR1(("ConfiguratorEngineering::compat-update configured for %50s "
                   "stickVersion=%30s installedVersion=%s", 
                   access.getLongName().c_str(), stickVersion.c_str(), installedVersion.c_str()));
   if (stickVersion == installedVersion) {
      // v850-version is unchanged
      return;
   }
   

   access.setExtraArg("DoHardResetOnSuccess", "1");
   
   
   if (access.getExtraArgsValAsBool("use_fw_partition")) {
      access.setExtraArg("simulate_fw_partition", "1");
   }
   
   TiXmlElement *deviceXml = pRelease->FirstChildElement("DEVICE");
   TiXmlElement *moduleXml = deviceXml->FirstChildElement("MODULE");

  // remove all other devices from bosch.xml
   for (TiXmlElement *submoduleXml=moduleXml->FirstChildElement("SUBMODULE");submoduleXml;) {
      TiXmlElement *tmp=submoduleXml;
      submoduleXml=submoduleXml->NextSiblingElement("SUBMODULE");
      if (tmp) {
         if (std::string(swu::getTextFromChildOrEmpty(tmp, "NAME")) != "V850") {
            moduleXml->RemoveChild(tmp);
         }
      }
   }   
  // remove all other devices from bosch.xml
   
}

tVoid ConfiguratorEngineering::findReleaseXml() 
{
   ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml"));
   SWU_ASSERT_RETURN(_enState == enState_SearchRelease);   

   std::map<std::string, trSourceInfo> sourcesList=PropDevMgr::instance()->get()._mapSources;
   std::list<TiXmlElement> xmlRelList;   

   for (std::map<std::string, trSourceInfo>::iterator iterSrc= sourcesList.begin();
        iterSrc!=sourcesList.end();
        ++iterSrc) {
      trSourceInfo sourceInfo=iterSrc->second; 



      FcSwUpdCore::instance()->getReleaseList(xmlRelList, sourceInfo);
      
      ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml: found %u releases", xmlRelList.size()));
      if (!xmlRelList.size()) {
         ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml: relList is empty"));
         _releaseXml=0;
         return;
      }
         
      TiXmlElement &overallSection= xmlRelList.front();     
      tenSwUpdateError errorCode;
      _releaseXml = FcSwUpdCore::instance()->getReleaseXml(overallSection, errorCode);  
      
      if (_releaseXml) {
         BXmlAccess relAccess(_releaseXml);
         std::string relName = relAccess.getTextFromModule("FINALNAME");
         ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml: check release %s", relName.c_str()));
         
         printf("ConfiguratorEngineering::findReleaseXml found valid release xml\n");
         checkV850CompatUpdate(_releaseXml);
         
         ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml: found release= %s, call vNotifyReady()", relName.c_str()));
         _enState =enState_FoundRelease;
         vNotifyReady( true );
      } 
      else {
         printf("ConfiguratorEngineering::findReleaseXml no valid release xml\n");
         ETG_TRACE_USR1(("ConfiguratorEngineering::findReleaseXml: found no appropriate release"));
      }
      GlobalLog::logSwuErrorOnce(errorCode);
   }
}



void ConfiguratorEngineering::traceState() 
{
   ETG_TRACE_COMP(("ConfiguratorEngineering traceState"));

   ETG_TRACE_COMP(("  _enState=%u",ETG_CENUM(ConfiguratorEngineering::tenState, _enState)));
}


// return our prepared boschXml
TiXmlDocument *ConfiguratorEngineering::getXmlDoc() 
{
   ETG_TRACE_USR1(("ConfiguratorEngineering::getXmlDoc START _enState=%u",
                   _enState));
   assert(_enState == enState_FoundRelease );
   _enState  = enState_Terminated;
   TiXmlDocument *relDocTmp=_releaseXml;
   _releaseXml = 0;
   ETG_TRACE_USR1(("ConfiguratorEngineering::getXmlDoc END"));
   return relDocTmp;
}


}	// namespace
