#include "util/swu_kds.h"
#include "util/swu_registry.h"
#include "util/swu_filesystem.h"
#include "util/swu_environment.hpp"
#include "util/swu_constants.hpp"
#include "util/swu_rawRegistry.h"
#include "util/swu_crypto.hpp"
#include "util/swu_otp.h"
#include "util/swu_bootChain.h"
#include "util/fcswupd_types.hpp"
#include "config/fcswupd_config.hpp"
#include "main/fcswupd_dataStore.h"
#include <algorithm>
#include <iostream>
#include <fstream>

#define ETG_I_FILE_PREFIX fcswupdate::Config::instance()->

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_config.cpp.trc.h"
#endif
using namespace std;

namespace fcswupdate {

// KDS entries
#define SWU_KDS_KEY_SWUPD_VARIANT_INFO (0xa070)
#define SWU_KDS_KEY_CMVARIANTCODING (0x0df4)
#define SWU_KDS_KEY_CUSTOMER_VARIANTCODING (0x0da2)
//#define SWU_KDS_KEY_HARDWARE_VERSION (0x010e)
#define SWU_KDS_KEY_ECU_PARTNUMBER (0x0105)
#define SWU_KDS_KEY_CM_NAVI_UNIT (0x2010)
#define SWU_KDS_KEY_SWUPD_VIN_INFO (0xA1C8)

// KDS entry fields for KDS_KEY_CMVARIANTCODING
#define SWU_KDS_CMVC_DISPLAYTYPE (2)
#define SWU_KDS_CMVC_OEM_TYPE (3)
#define SWU_KDS_CMVC_HW_ASSEMBLY (4)
//#define SWU_KDS_CMVC_ADR3_VARIANT (5)
#define SWU_KDS_CMVC_DTBID (13)

// KDS entry masks for KDS_CMVC_HW_ASSEMBLY
#define SWU_KDS_MASK_DAB_TUNER (0x10)
#define SWU_KDS_MASK_CD (0x08)
//#define SWU_KDS_MASK_AMP (0x40)

// KDS entry values for KDS_CMVC_HW_ASSEMBLY
//#define SWU_KDS_DISPLAYTYPE_COLOR (0x0A)
//#define SWU_KDS_DISPLAYTYPE_GREY (0x0B)



//static const std::string _emptyString= "";
static bool _default_allow_reboot_during_update = false;
static const std::string _logDirName = "/tmp";
static const std::string _persistentBase = FCSWUPD_PERSISTENT_ROOT;
static const std::string _tmpBase = FCSWUPD_TMP_ROOT;
static const std::string _staticBase = FCSWUPD_STATIC_ROOT;
static const std::string _progessFilePersistent = _persistentBase + "fcswupd_running.xml";
static const std::string _progessFileTmp = _tmpBase + "fcswupd_running.xml";
static const std::string _updateTxtPersistentPrefix = _persistentBase + "update.txt.";

static const std::string _historyFilePersistent = _persistentBase + "history.xml";
static const std::string _historyFileStatic = _staticBase + "history.xml";
static const std::string _dataStoreFilePersistent = _persistentBase + "datastore.xml";
static const std::string _dataStoreFileNonPersistent = _tmpBase + "datastore.xml";
static const std::string _dataStoreFileStatic = _staticBase + "datastore.xml";
static const std::string _defaultUpdadeKeyStatic = _staticBase + "public_key.pem";

static const std::string _swucoreBase = FCSWUPD_SWUPDCORE_PERSISTENT_ROOT;
static const std::string _hmiSettingsPath =  _swucoreBase + "hmi/";
static const std::string _hmiSettingsFilePersistent = _hmiSettingsPath + "hmisettings.xml";

static const std::string _regBasePath = "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT";
static const std::string _regVersionsPath = "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS";
static const std::string _regKeyAdrSwVersion = "ADR_SW_Version";
static const std::string _regKeyAdrSwVersionBosch = "ADR_SW_Version";


static bool _bSkipPropertyDeregister = FCSWUPD_SKIP_PROPERTY_DEREGISTER;
static tUInt _rebootWaitTimeMs=3000;
static tUInt _spmWaitTimeMs=20000;
static tUInt _spmWaitAvailTimeMs=2000;
static bool _usePersistentPart=false;

static tUInt _installTimeoutMs=1800000; // 30 Minutes
static tUInt _abortInstallTimeoutMs=1800000; // 30 Minutes

static const std::string _dataStoreKeyIgnoreCkSumAndSig="IgnoreCksumAndSig";
bool EnterFullOperationItemDescription::read(tU32 &value) {
   
   value = (Config::instance()->cfg_EnterRecovery.readAsBool() || Config::instance()->cfg_EnterEngineering.readAsBool()) ? false : true;
   return true;
}


bool AdrVersionItemDescription::read(std::string &value) {
   std::string versionLong=Config::instance()->cfg_AdrSwVersionLong.get();
   std::size_t pos=versionLong.find(" ");
   value=versionLong.substr(0,pos);
   return !value.empty();
}


bool ConfigItemFcidStr::read(std::string &value) {
   tU32 u32Fcid;
   bool res=Config::instance()->cfg_FcId.get(u32Fcid);
   char fcidStr[7];     
   sprintf(fcidStr,"%.4X",u32Fcid);
   value = fcidStr;
   return res;
}

bool ConfigItemFcidStr::write(std::string const &value) {
   std::string newVal=std::string("0x") + value;
   return Config::instance()->cfg_FcId.set(atoi(newVal.c_str()));
}


bool ConfigItemPrjIdStr::read(std::string &value)
{
   tU32 u32PrjId;
   bool res=Config::instance()->cfg_PrjId.get(u32PrjId);
   char prjIdStr[7];
   sprintf(prjIdStr, "%.2X", u32PrjId);
   value = prjIdStr;
   return res;
}

bool ConfigItemPrjIdStr::write(std::string const &value)
{
   std::string newVal=std::string("0x")+value;
   return Config::instance()->cfg_PrjId.set(atoi(newVal.c_str()));
}


bool RunningCustomerVersion::read(std::string &value) {
   ETG_TRACE_USR4(("RunningCustomerVersion::START"));

      std::string custVerRegFile="/var/opt/bosch/persistent/registry/customerversion.reg";
      if (!swu::exists(custVerRegFile)) {
         return false;
      }
      std::string line;
      ifstream inFile;
      inFile.open(custVerRegFile.c_str());
      if(!(inFile.is_open())) {
         ETG_TRACE_USR4(("RunningCustomerVersion::read:Unable to open %s", custVerRegFile.c_str()));
         return false;
      }
      while(getline(inFile, line)) {
         swu::trimString(line);
         ETG_TRACE_USR4(("RunningCustomerVersion::found  line:%s", line.c_str()));
         std::string str;
         for(size_t i = 0; i < line.size(); ++i) {
            if(line[i] != 34) {
               str += line[i];
            }
         }
         line =str;
         ETG_TRACE_USR4(("RunningCustomerVersion::replaced:  line:%s", line.c_str()));
         std::string customerVerison=line;
         swu::replaceSubString(customerVerison, "CUSTOMERVERSION=", "");
         if (line != customerVerison) {
            ETG_TRACE_USR4(("RunningCustomerVersion::found  customerVerison:%s", customerVerison.c_str()));
            if (customerVerison=="XXXX") {
               ETG_TRACE_USR4(("RunningCustomerVersion::found  customerVerison:XXXX (unknown)"));
               return false;
            }
            swu::trimString(customerVerison);
            value=customerVerison;
            return true;
         }
      }
      return false;
   }

bool RunningFinalName::read(std::string &value) {
   Config *cfg=Config::instance();
   ETG_TRACE_USR4(("RunningFinalName::read() START"));
   if (cfg->cfg_UseReleaseNameFromRunningSwLabel.get()) {
      value = cfg->cfg_RunningSwLabel.get();
   }
   else {
      value = cfg->cfg_RunningSwTrain.get();
   }
   ETG_TRACE_USR4(("RunningFinalName::read(): %s", value.c_str()));
   return true;
}

bool RunningSwVersionDisplay::read(std::string &value) {
   Config *cfg=Config::instance();
   ETG_TRACE_USR4(("RunningCustomerVersionDisplay::read() START"));
   value=cfg->cfg_RunningCustomerVersion.get();
   if (!value.empty() && value != "unknown") {
      // do nothing
   }
   else {
      value = cfg->cfg_RunningFinalName.get();
   }
   ETG_TRACE_USR4(("RunningCustomerVersionDisplay::read(): %s", value.c_str()));
   return true;
}

bool ConfigItemSecureBootModeEnable::read(tU32 &value)
{
   unsigned int  fuse;
   bool res=swu::OTPDriver::GetInstance()->GetData(&fuse, 0x0460);
   value=(fuse&SECURE_BOOT_MODE_ENABLED_BIT)?true:false;
   return res;
}

bool ConfigItemSRKHash::read(std::string &value)
{
   unsigned int  fuse;
   char  buff[9];
   bool  res=true;
   value="";
   for(int i=0x0580; i<=0x05f0; i+=0x0010)
   {
      res&=swu::OTPDriver::GetInstance()->GetData(&fuse, i);
      snprintf(buff, sizeof(buff), "%08x", fuse);
      value = value + buff[6] + buff[7] + buff[4] + buff[5] + buff[2] + buff[3] + buff[0] + buff[1];
   }
   return res;
}

bool ConfigItemSRKRevocation::read(tU32 &value)
{
   unsigned int  fuse;
   bool res=swu::OTPDriver::GetInstance()->GetData(&fuse, 0x06f0);
   value=fuse&SECURE_BOOT_REVOCATION_BITS;
   return res;
}

bool ConfigItemTeseoVersion::read(std::string &value)
{
   tU32 u32Version;
   bool res=Config::instance()->cfg_TeseoVersionU32.get(u32Version);
   char chrVersion[9];
   sprintf(chrVersion, "%.08x", u32Version);
   value = chrVersion;
   return res;
}


bool RunningOverallCISSWVer::read(std::string &value) {
   Config *cfg=Config::instance();
   ETG_TRACE_USR4(("RunningOverallCISSWVer::read() START"));
   
   std::string cisVerRegFile="/var/opt/bosch/persistent/registry/overallCISSWversion.reg";
   if (!swu::exists(cisVerRegFile)) {
      return false;
   }
   std::string line;
   ifstream inFile;
   inFile.open(cisVerRegFile.c_str());
   if(!(inFile.is_open())) {
      ETG_TRACE_USR4(("RunningOverallCISSWVer::read:Unable to open %s", cisVerRegFile.c_str()));
      return false;
   }
   while(getline(inFile, line)) {
      swu::trimString(line);
      ETG_TRACE_USR4(("RunningOverallCISSWVer::found  line:%s", line.c_str()));
      std::string str;
      for(size_t i = 0; i < line.size(); ++i) {
         if(line[i] != 34) {
            str += line[i];
         }
      }
      line =str;
      ETG_TRACE_USR4(("RunningOverallCISSWVer::replaced:  line:%s", line.c_str()));
      std::string cisVerison=line;
      swu::replaceSubString(cisVerison, "OVERALL_CIS_SW_VERSION=", "");
      if (line != cisVerison) {
         ETG_TRACE_USR4(("RunningOverallCISSWVer::found  cisVerison:%s", cisVerison.c_str()));
         if (cisVerison=="") {
            ETG_TRACE_USR4(("RunningOverallCISSWVer::found  cisVerison: (unknown)"));
            return false;
         }
         swu::trimString(cisVerison);
         value=cisVerison;
         ETG_TRACE_USR4(("RunningOverallCISSWVer::read(): %s", value.c_str()));
         return true;
      }
   }
   return false;

}


Config::Config():

   // Setting, Constant: tmp-dir for swupdate, not specific for fcswupdate, cleanup needed
   cfg_SwuTmpDir(this, "SwuTmpDir", "/tmp/swupd"),
   // Variable: indicate to outer world (SwuLcmComponent), that fcswupdate has started by setting to 0
   cfg_SwuStarting(this, "SwuStarting", 0,
                   new swu::MarkerFileItemDescription("/tmp/swupd/SwuStarting")),
   // Variable: indicate to outer world, that startupd of fcswupdate is completed by setting it to 0, todo: check why 2 config items are used here
   cfg_SwuStarted(this, "SwuStarted", 0,
                  new swu::MarkerFileItemDescription("/tmp/swupd/SwuStarted")),

   //marker file for fcota to find active bootchain, this is needed for fcota
   cfg_bootChainMarkerForFcOta(this, "bootChainMarkerForFcOta", 0,
                               new swu::MarkerFileItemDescription((swu::BootChain::usedBootChain() == 1) ? "/tmp/swupd/Bootchain1_active" : "/tmp/swupd/Bootchain2_active")),

   //For Aborting the SWU, a marker file is temporarily created
   cfg_SwuAbortLuaMarker(this, "SwuAbortLuaMarker", 0,
                  new swu::MarkerFileItemDescription("/tmp/swupd/abortLua")),
  
   // Settings, Constant: Central KDS-table of sw-update
   cfg_SwUpdVariantInfo(this, "SwUpdVariantInfo", std::vector<tU8>(16), 
                       new swu::KdsItemDescription<std::vector<tU8> >(SWU_KDS_KEY_SWUPD_VARIANT_INFO, 
                                                                 16,  0, 16), 
                       SWU_CFG_ITEM_EXPORT),
#if 0
   // demo of ConfigItemVectorSliceDescription
   cfg_SwUpdFcIdHex(this, "SwUpdFcIdHex", std::vector<tU8>(2), 
                       new swu::ConfigItemVectorSliceDescription(&cfg_SwUpdVariantInfo, 
                                                                 1,  2),
                       SWU_CFG_ITEM_EXPORT),
   cfg_SwUpdFcIdHex2(this, "SwUpdFcIdHex2", (tU32)0, 
                     new swu::ConfigItemSliceDescription(&cfg_SwUpdVariantInfo, 
                                                         1,  2),
                     SWU_CFG_ITEM_EXPORT, ".04x"),
#endif

   // CustomerSubProject, e.g. rnaivi,rivie,npivi
   cfg_CustomerSubProject(this, "CustomerSubProject", ""),
   
   cfg_VerifyAllianceXML(this, "VerifyAllianceXML", 0), 

   cfg_SupportedUSBFormats(this, "SupportedUSBFormats", ""),

   // Vehicale Identification number   
   cfg_VehicleId(this, "VIN", "",
            new swu::KdsItemDescription<std::string>(SWU_KDS_KEY_SWUPD_VIN_INFO, 
                                              17,  0, 17)), 

   // Fcid as U32
   cfg_FcId(this, "FcId", (tU32)(0),
            new swu::KdsItemDescription<tU32>(SWU_KDS_KEY_SWUPD_VARIANT_INFO, 
                                              16,  1, 2)),
   // Constant :Fcid as String, without prefix 0x
   cfg_FcIdStr(this, "SwUpdFcIdStr", "0000", 
                    new ConfigItemFcidStr()),

   // Constant :project-id as U32. should be set by each project. Like this, each project can use all possible FCIDs
   cfg_PrjId(this, "PrjId", (tU32)(0),
            new swu::KdsItemDescription<tU32>(SWU_KDS_KEY_SWUPD_VARIANT_INFO, 
                                              16,  15, 1)),
   // Constant :same project-id as string
   cfg_PrjIdStr(this, "SwUpdPrjIdStr", "00", 
                    new ConfigItemPrjIdStr()),
   
   // Constant:the board-id
   cfg_BoardId(this, "BoardId", "", new ConfigItemBoardId<Config>()),

   // NOT used, was MIB
   cfg_VariantId(this, "VariantId", (tU32)35126),
   // NOT used, was MIB
   cfg_RegionId(this, "RegionId",  (tU32)0x02),
   // NOT used, was MIB, check if it can be remmoved from .fidl
   cfg_HwId(this, "HwId",      "050"),


   // CmVariantId defines the target for diagnosis, as fcid for swupdate
   cfg_CmVariantId(this, "CmVariantId", "", 
                       new swu::KdsItemDescription<std::string>(0xA011, 
                                                                32,  0, 32), 
                       SWU_CFG_ITEM_EXPORT),

   // Sub-Items from KDS that get exported as extra-args to swUpdateCore
   cfg_CmVariant(this, "CmVariant", std::vector<tU8>(16), 
                       new swu::KdsItemDescription<std::vector<tU8> >(SWU_KDS_KEY_CMVARIANTCODING, 
                                                                 16,  0, 16), 
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmVariantWithDab(this, "CmVariantWithDab", (tU32)0, 
                       new swu::ConfigItemSliceDescription(&cfg_CmVariant, 
                                                           SWU_KDS_CMVC_HW_ASSEMBLY,  1, 
                                                           SWU_KDS_MASK_DAB_TUNER),
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmVariantWithCd(this, "CmVariantWithCd", (tU32)0, 
                       new swu::ConfigItemSliceDescription(&cfg_CmVariant, 
                                                           SWU_KDS_CMVC_HW_ASSEMBLY,  1, 
                                                           SWU_KDS_MASK_CD),
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmVariantDisplayType(this, "CmVariantDisplayType", (tU32)0, 
                       new swu::ConfigItemSliceDescription(&cfg_CmVariant, 
                                                           SWU_KDS_CMVC_DISPLAYTYPE,  1),
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmVariantOem(this, "CmVariantOem", (tU32)0, 
                       new swu::ConfigItemSliceDescription(&cfg_CmVariant, 
                                                           SWU_KDS_CMVC_OEM_TYPE,  1),
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmVariantDtbId(this, "CmVariantDtbId", (tU32)0, 
                       new swu::ConfigItemSliceDescription(&cfg_CmVariant, 
                                                           SWU_KDS_CMVC_DTBID,  1),
                       SWU_CFG_ITEM_EXPORT),
   cfg_CustomerVariant(this, "CustomerVariant", std::vector<tU8>(16), 
                       new swu::KdsItemDescription<std::vector<tU8> >(SWU_KDS_KEY_CUSTOMER_VARIANTCODING, 
                                                                 16,  0, 16), 
                       SWU_CFG_ITEM_EXPORT),
   cfg_EcuPartNr(this, "EcuPartNr", std::vector<tU8>(), 
                       new swu::KdsItemDescription<std::vector<tU8> >(SWU_KDS_KEY_ECU_PARTNUMBER, 
                                                                 16,  0, 16), 
                       SWU_CFG_ITEM_EXPORT),
   cfg_CmNaviUnit(this, "CmNaviUnit", std::vector<tU8>(),
                       new swu::KdsItemDescription<std::vector<tU8> >(SWU_KDS_KEY_CM_NAVI_UNIT,
                                                                 8, 0, 8),
                       SWU_CFG_ITEM_EXPORT),
	// cfg_EcuCmVariantId defines the target for diagnosis, and this is part number for INF4CV
   cfg_EcuCmVariantId(this, "EcuCmVariantId", "", 
                       new swu::KdsItemDescription<std::string>(0xA011, 
                                                                32,  0, 32), 
                       SWU_CFG_ITEM_EXPORT),
   


   // demo-items for internal tests
   cfg_DemoU32Item1(this, "DemoU32Item1", (tU32)0),
   cfg_DemoU32Item2(this, "DemoU32Item2", (tU32)0),
   cfg_DemoU32Item3(this, "DemoU32Item3", (tU32)0),
   cfg_DemoStringItem1(this, "DemoStringItem1", ""),
   cfg_DemoStringItem2(this, "DemoStringItem2", ""),
   cfg_DemoVectorItem1(this, "DemoVectorItem1", std::vector<tU8>(16)),
   cfg_DemoVectorItem2(this, "DemoVectorItem2", std::vector<tU8>(16)),
                               
   // Constant Version like 0242_160926, registry-key=BUILDVERSION_CUSTVERSTRING, but it is no customer-version!
   cfg_RunningSwTrain(this, "RunningSwTrain", "unknown SW-Train", 
                      new swu::RawRegistryStringItemDescription(swu::Constants::Registry::REGISTRY_VERSION_ID)),
   // Label of the release, e.g. AI_PRJ_RN_AIVI_16.0V62, registry-key: BUILDVERSION_LABEL
   cfg_RunningSwLabel(this, "RunningSwLabel", "unknown SW-Label", 
                      new swu::RawRegistryStringItemDescription(swu::Constants::Registry::REGISTRY_LABEL_ID)),

   // version for the customer, not part of SW-build
   // check item-description for details.
   cfg_RunningCustomerVersion(this, "RunningCustomerVersion", "unknown", 
                              new RunningCustomerVersion),


   // depending on UseReleaseNameFromRunningSwLabel either  RunningCustomerVersion or RunningSwLabel
   // ONLY this item shall be used for version-compare
   cfg_RunningFinalName(this, "RunningFinalName", "unknown", 
                              new RunningFinalName),

   // if set, RunningCustomerVersion, else RunningFinalName
   cfg_RunningSwVersionDisplay(this, "RunningSwVersionDisplay", "unknown", 
                              new RunningSwVersionDisplay),

   // if set, use own automounter in recovery-mode
   cfg_UseOwnAutomountInRecovery(this, "UseOwnAutomountInRecovery", 1),


   // date that shall be set in initial history, when resetting to factory-settings
   cfg_InitialProgrammingDate(this, "InitialProgrammingDate", std::vector<tU8>(8),
                              new swu::KdsItemDescription<std::vector<tU8> >(0xA010,
                                                                             6, 0, 6,
                                                                             swu::enKsdEntryType_Data),
                              SWU_CFG_ITEM_VOLATILE),


   // build-time from registry, currently not used
   cfg_RunningBuildTime(this, "RunningBuildTime", "0", 
                        new swu::RawRegistryStringItemDescription(swu::Constants::Registry::REGISTRY_TIMESTAMP_ID)),

   // project can decide, which "version" will be used by cfg_RunningFinalName
   cfg_UseReleaseNameFromRunningSwLabel(this,"UseReleaseNameFromRunningSwLabel",(tU32)false),
   
   /* AllowRebootDuringInstall indicates, that sw-update is not allowd to reboot the target during update. Needed for background-update, where it has to be stored persitently, since background-update can continue in next power-cycle, see project-specific setting in RNAIVI
      todo: check if we can put this setting to the progress-section and not to data-store in RNAIVI
    */
   cfg_AllowRebootDuringInstall(this, "AllowRebootDuringInstall",  (tU32)FCSWUPD_ALLOW_REBOOT_DURING_INSTALL),

   /*
    Check whether fcswupdate needs to interact with Normal Mode HMI in config & running State.
   */
   cfg_DoIgnoreNormalModeHMIInteractions(this, "DoIgnoreNormalModeHMIInteractions",  (tU32)false),

   // simulate the KDS, no data will be read from real KDS.
   cfg_DoSimKds(this, "DoSimKds", 0,
                new swu::MarkerFileItemDescription(FCSWUPD_MARKER_FILE_SIM_KDS)),

   /* ignore checksums in bosch.xml and signature of bosch.xml
      todo: handling aparently broken, please repair
      also check handling of skip-cheksum in bosch.xml
    */
   cfg_DoIgnoreCkSumAndSig(this, "DoIgnoreCkSumAndSig",  (tU32)false),


   /* setting of file-type the is searched on update-medium, 
      todo: only used in auto-mounter but not by prm-if.
      check if we can remove this for all projects.
   */
   cfg_ManifestFileType(this, "ManifestFileType", "xml"),
   // check if this is used by any project and remove this item
   cfg_ManifestSubDir(this, "ManifestSubDir", ""),
   /*
     file we search in medium for developer-update
    */
   cfg_ParserTftpFilesDeveloperMode(this, "ParserTftpFilesDeveloperMode", "bosch.xml"),
   /*
     configure if we use encrypted files on update-medium
    */
   cfg_ParserUseEncryption(this, "ParserUseEncryption", (tU32)false),
   /*
     configure project-specific sub-dir on update-medium where we search for the bosch.xml
     Used by PSA-project.
    */
   cfg_AlternativeUpdatePath(this,"AlternativeUpdatePath",""),
   // is not empty, we will search for this iso-file on update-media and mount it as additional update-medium
   cfg_UpdateIsoFileName(this,"UpdateIsoFileName",""),
   
   /*
     indicates, that a update that started in hmi-mode shall end in hmi-mode an present the update-result here, even when the real update already finished in recovery-mode
    */
   cfg_NotifyHmiResultOnSuccess(this,"NotifyHmiResultOnSuccess",(tU32)false),
   // not used in aivi context, check if item can be removed
   cfg_NotifyCtrlError(this,"NotifyCtrlError",(tU32)true),
   /*
     if set, notify hmi, when we previously had a valid release the has been lost since the stick was removed.
    */
   cfg_NotifyMediumRemoval(this,"NotifyMediumRemoval",(tU32)false),
   //fixme: hpe2hi: activate ConfiguratorHmiNotifyMetaInfoNotFound for PSA
   /*
     if not set, wrong version will lead to error-code tenSwUpdateError_ERROR_IMAGE_INCOMPATIBLE_VERSION,
     if set, we distinguish between ERROR_IMAGE_INCOMPATIBLE_DOWNGRAD and ERROR_IMAGE_INCOMPATIBLE_SAME_VERSION, as required by AIVI.
    */
   cfg_ConfiguratorHmiNotifyVersionConflictDetails (this,"ConfiguratorHmiNotifyVersionConflictDetails",(tU32)false),
   /*
     indicate, if error ERROR_METAINFO_NOT_FOUND shall be raised (no release found on update-medium)
    */
   cfg_ConfiguratorHmiNotifyMetaInfoNotFound(this,"ConfiguratorHmiNotifyMetaInfoNotFound",(tU32)true),
   /*
     same as above but a medium contained the alternative update-path. (introduced by PSA)
    */
   cfg_ConfiguratorHmiNotifyMetaInfoNotFoundInAlternatePath(this,"ConfiguratorHmiNotifyMetaInfoNotFoundInAlternatePath",(tU32)true),
   /*
     if set, the need for flashing is evaluated by the main-version,
     otherwise from the fact that at least one submodule-version is different between stick and target.
    */
   cfg_ConfiguratorHmiNeedsFlashingBasedOnTrain(this,"ConfiguratorHmiNeedsFlashingBasedOnTrain",(tU32)true),
   // todo: enable ConfiguratorHmiDefaultToErrReleaseNotFound for PSA
   cfg_ConfiguratorHmiDefaultToErrNoReleaseFound(this,"ConfiguratorHmiDefaultToErrNoReleaseFound",(tU32)false),
   /*
     Configures, that any release with a force.dnl will be accepted, even if is does not pass the filters of the HmiConfigurator.
     todo: check if it is better to created a progress-section before resetting.
    */
   cfg_ConfiguratorAcceptAllowForceDnlForAnyRelease(this,"ConfiguratorAcceptAllowForceDnlForAnyRelease",(tU32)true),

   /*
    Renault doesn't have DTM & STM to downgrade to lower SW-version.
    so, they will have this feature in Alliance.xml to downgrade software. this is applicable only without force.dnl
    if, force.dnl is present it takes the priority.
    */
   cfg_ConfiguratorAllianceAllowDowngrade(this,"ConfiguratorAllianceAllowDowngrade",(tU32)false),
   /*
     If set, the developer update will first update the v850 in case it is changed and then start the complete update again. This prevents incompatible imx/v850 running together (the running imx-sw comes from the new sw on the stick, the running v850 from the old version on the target)
    */
   cfg_ConfiguratorDeveloperForceV850Compat(this, "ConfiguratorDeveloperForceV850Compat", (tU32)false),

   /*
     Internal value:
     Store the system-time before entering recovery-mode, to use it in the history when the update is finished. In recovery-mode, we don't have a valid time!
     todo: better store this value in progress-section, be aware of force.dnl (see above)
    */
   cfg_UpdateStartTime(this,"UpdateStartTime","0", 
                       new swu::DataStoreItemDescription<std::string>(DataStore::instance(), 
                                                                      "UpdateStartTime")),
   /*
     Internal value:
     Store the distance-total before entering recovery-mode, to use it in the history when the update is finished. In recovery-mode, we don't have this value.
     todo: better store this value in progress-section, be aware of force.dnl (see above)
    */
   cfg_UpdateStartDistanceTotalizer(this, "UpdateStartDistanceTotalizer", 0,  
                                    new swu::DataStoreItemDescription<tU32>(DataStore::instance(), 
                                                                            "UpdateStartDistanceTotalizer")),

   /*
     decide, which parser type shall be used in recovery-mode and hmi-mode (currently cms or bosch)
    */
   cfg_ParserType(this, "ParserType", swu::ParserTypeBosch),
   /*
     accepts cms-files when we don't have a ca-cert (needed during development)
    */
   cfg_CmsParserAllowEmptyCACert(this, "CmsParserAllowEmptyCACert", (tU32)true),

   /*
     set, where the cert is stored
    */
   cfg_CertificateAndKeySource(this, "CertificateAndKeySource", swu::CertificateAndKeySourceNOR),

   /*
     storage of authentic time of last accepted certificate as own time.
     We don't have a valid time in recovery-mode!
     This time is used to check certificates in the future.
     todo: get correct explanation from Phillip!
    */
   cfg_LastAuthenticTimeFormat(this, "LastAuthenticTimeFormat", "%Y-%m-%d %H:%M:%S"),
   cfg_LastAuthenticTime(this, "LastAuthenticTime", "1970-01-01 00:00:00"),
 

   /*
    SCOMO IVI Installer Type 
   */
   cfg_ScomoIVIInstallerType(this, "ScomoIVIInstallerType", "IVI"),


   /*
    For SCOMO-OTA, Set this Config Item For IVI-FIRMWARE
   */
   cfg_ScomoCompIdAppend(this, "ScomoCompIdAppend", "", 
                         new swu::DataStoreItemDescription<std::string>(DataStore::instance(), 
                                                                  "ScomoCompIdAppend")),

   /*
    For SCOMO-OTA, Set this Config Item For IVI-FIRMWARE
   */
   cfg_ScomoFirmwareCompId(this, "ScomoFirmwareCompId", "FIRMWARE"),

   /*
    For SCOMO-WCS, OEM Updater
   */
   cfg_ScomoWelcomeSequenceCompId(this, "ScomoWelcomeSequenceCompId", "WCS"),

   /*
    For SCOMO-SVS, OEM Updater
    For SCOMO-Visuals, OEM Updater
   */
   cfg_ScomoVisualsCompId(this, "ScomoVisualsCompId", "SVS"),
   cfg_ScomoSpecialVisualsCompId(this, "ScomoSpecialVisualsCompId", "SVS"),

   /*
    For SCOMO-SNM, OEM Updater
    */
   cfg_ScomoSNMCompId(this, "ScomoSNMCompId", "SNM"),

   /*
    For SCOMO-extra data, OEM Updater
   */
   cfg_ScomoExtraDataCompId(this, "ScomoExtraDataCompId", "CDPD"),
   
   /*
    For SCOMO-Radio logo, OEM Updater
    */
   cfg_ScomoRadioLogoCompId(this, "ScomoRadioLogoCompId", "RDL"),
   
   /*
    For SCOMO-Car Visuals, OEM Updater
    */
   cfg_ScomoCarVisualsCompId(this, "ScomoCarVisualsCompId", "CVS"),

   /*
   
   /*
     Indication in which mode the system is currently running
    */
   cfg_EnterRecovery(this, "EnterRecovery", (tU32)0,
                     new swu::EnvironmentItemDescription<tU32>("SW_UPDATE_RECOVERY_MODE")),

   cfg_EnterEngineering(this, "EnterEngineering", (tU32)0,
                        new swu::EnvironmentItemDescription<tU32>("SW_UPDATE_ENGINEERING_MODE")),

   cfg_EnterFullOperation(this, "EnterFullOperation", (tU32)0,
                          new EnterFullOperationItemDescription()),

   cfg_EnterPimpedRecovery(this, "EnterPimpedRecovery", (tU32)false),

   /*
     setting for tftp-connection used in developer-update
    */
   cfg_TftpServerIpAdr(this, "TftpServerIpAdr", "172.17.0.8"),
   cfg_TftpRootPath(this, "TftpRootPath", ""),
   cfg_TftpLocalPath(this, "TftpLocalPath", "/tmp/tftpRelease"),

					  
   /*
     todo: not used in aivi, check if it can be deleted
    */
   // config-item to be poputlated by project
   cfg_RegionIdCustomer(this, "RegionIdCustomer", ""),
   /*
     todo: not used in aivi, check if it can be deleted
    */
   cfg_TargetIdCustomer(this, "TargetIdCustomer", ""),

   /*
     indicates, that spm has to be asked before entering update in HMI-mode
    */
   cfg_UseSpmUpdateMode(this, "UseSpmUpdateMode", (tU32)FCSWUPD_USE_SPM_UPDATE_MODE),
   /*
     configure that a update can be resumed after power-cycle/reset
     todo: not used in aivi, check if it can be deleted.
     Should be configured now be existence of per persistent progress-section.
    */
   cfg_EnableReEnterStateRunning(this, "EnableReEnterStateRunning", (tU32)FCSWUPD_ENABLE_RE_ENTER_START_RUNNING),
   /*
     project can set the variable to let the v850 reset when going to idle again.
     Details by Wolfgang Fandrych
    */
   cfg_V850RebootNeeded(this, "V850RebootNeeded", 0, 0, SWU_CFG_ITEM_VOLATILE),
   /*
     todo: clarify shutdown-handling (shutdown.dnl)!!!
     shutdown.dnl should work in every project? clarify witch Sascha.
    */
   cfg_DoShutdownAfterUpdate(this, "DoShutdownAfterUpdate", 0),
   /*
     if compat-index in bosch.xml is higher than the installed compat-index, we have to do a compat-update.
     This means: Only update imx-boot and v850, reset to new boot-chain and start update allover again.
     The new SW must fullfill the required compat-index or we will end up in endless updates!
     Needed in case of a change in imx-boot/v850 to enable update of other sub-processors.
     compat-index is set in project-specific code by a macro.
     To get a compat-update, increase this macro und increase the neededSwuCompatIndex in bosch.xml.
     todo: add specific element in bosch.xml for compat-index?
    */
   cfg_SwuCompatIndex(this, "SwuCompatIndex", 0),

   /*
     marker-file to simulate a compat-update.
     Once the first part of the compat-update reaches state result, 
     the marker-file will be deleted and the target is reset to the new boot-chain.
    */
   cfg_TestCompatUpdate(this, "TestCompatUpdate", (tU32)false,
                        new swu::MarkerFileItemDescription(FCSWUPD_PERSISTENT_ROOT "/testSwuCompatIndex")),

   /*
     this will read the RFS variant Id, this needed for scomo update.
   */
   cfg_RfsVariantId(this, "RfsVariantId", "",
                     new swu::FileItemDescription<std::string>("/VariationID.txt")),  
   /*
     If element "FCID" is set in bosch.xml and it does not match the current fcid, the new fcid will be written to KDS after update
    */
   cfg_WriteFcIdAfterUpdate(this, "WriteFcIdAfterUpdate", 0),
   /*
     If element "FCID" is set in bosch.xml and it does not match the current fcid, the new fcid will be written to KDS before update
    */
   cfg_WriteFcIdBeforeUpdate(this, "WriteFcIdBeforeUpdate", 0),

   /*
     configure handling of GetUpdateHistoryRequest. If set, the initial release will be included in the release-list. Normally after reset to factory, the release-list is empty.
    */
   cfg_IncludeInitialReleaseInHistoryList(this, "IncludeInitialReleaseInHistoryList", (tU32)true),
   /*
     configure handling of GetUpdateHistoryScopedRequest. If set, the initial release will be included in the release-list. Normally after reset to factory, the release-list is empty.
    */
   cfg_IncludeInitialReleaseInScopedHistoryList(this, "IncludeInitialReleaseInScopedHistoryList", (tU32)true),

   
   /*
     handling currently not implemented
     Tells HMI-configurator to automatically select a source
     currently we always check all available sources
    */
   cfg_HmiCfgAutoSelectSource(this, "HmiCfgAutoSelectSource", (tU32)true),
   /*
     handling currently not implemented
     Tells HMI-configurator to automatically select a release from list of found releases
     currently we always check all available sources
     Might be required as false for cherry-project
    */
   cfg_HmiCfgAutoSelectRelease(this, "HmiCfgAutoSelectRelease", (tU32)true),
   /*
     Default filter for acceptance of releases according to fidl:UpdateSetReleaseFilter:trReleaseFilter:tenUpdOptions
    */
   cfg_HmiCfgDefaultReleaseFilter(this, "HmiCfgDefaultReleaseFilter", (tU32)0),
   /*
     Default options to decide which items will be updated when a release has been accepted by ReleaseFileter. Value according to fidl:SetUpdateOptions:tenUpdOptions
    */

   cfg_HmiCfgDefaultUpdateOption(this, "HmiCfgDefaultUpdateOption", (tU32)2), // Any, see fcswup_configuratorUtil.h

    cfg_HmiCfgXMLOverridesHMIReleaseFilter(this, "HmiCfgXMLOverridesHMIReleaseFilter", (tU32)false),
   /*
     indicates that a post-image-installation is running
    */
   cfg_PostImageInstallationRunning(this, "PostImageInstallationRunning", (tU32)false,
                                    new swu::DataStoreItemDescription<tU32>(DataStore::instance(), 
                                                                                   "postImageInstallationRunning")),

   /*
     Handling of postImageRelease. If set to false, we will lool for a postImageRelease in the
     persistent partition and conduct a simulated update to get an inital history. 
     No install-request will be sent to sw-update-core.
     set to false to enable feature in configPrj and add data-store item-description,
     so the fcSwUpdate can set it to true during runtime, when the postImageInstallation is done.
   */
   cfg_AutoHandlePostImgDone(this, "AutoHandlePostImgDone", (tU32)true),
   /* marker-file that PostImage shall be "flashed"
      This marker-file has to be put to the production-image of the persistent partition.
      It will be deleted when the post image installation is done.
    */
   cfg_AutoHandlePostImgPending(this, "AutoHandlePostImgPending", (tU32)false,
                                new swu::MarkerFileItemDescription(FCSWUPD_PERSISTENT_ROOT "/postImage/autoHandlePostImgPending")),
   /* 
      global variable to disable messages to HMI, used for auto post image installation.
    */
   cfg_SilentUpdate(this, "SilentUpdate", (tU32)false),
  
   /*
    central lock handling, this applicable for scope 2.1 and above.
   */
   cfg_HandleCentralLock(this, "HandleCentralLock", (tU32)false),
  /*
   after acquiring lock for restricted background update before goes to recovery mode touch this file.
   this file is used by SPM on next normal mode startup.
  */
   cfg_SpmMarkerForLock(this, "SpmMarkerForLock", (tU32)false,
                        new swu::MarkerFileItemDescription("/var/opt/bosch/persistent/deletelockflag.dnl", 7006, 7006, 0755)),


   /*
     Tells us to allways update in recovery-mode
     Will be overruled by cfg_AllowRebootDuringInstall
    */
   cfg_CtrlForceRecovery(this, "CtrlForceRecovery", (tU32)true),

   /*
    this config item tells whether XMLFilterChain can be applied or not.
    this XMLFilterChain filter bosch.xml as a file not as tiny xml.
   */
   cfg_ApplyXMLFilterChain(this, "ApplyXMLFilterChain", (tU32)false),


   /*
     Configured, if recovery-magics are mastered by target-scripting of by fcSwUpdate.
     Recovery-magics have to be handled in fcSwUpdate for Background-update.
    */
   cfg_HandleRecoveryMagicsInFcSwUpd(this, "HandleRecoveryMagicsInFcSwUpd", (tU32)false),
   /*
     Decides if a removed stick will lead to a failed update or if we wait for reinsertion to continue
     the update
    */
   cfg_CtrlRetryOnRemovedStick(this, "CtrlRetryOnRemovedStick", (tU32)false),

   //config-item to decide on adding history.
   cfg_AddReleaseToHistory(this, "cfg_AddReleaseToHistory", (tU32)true),
   
   /*
   cfg_CtrlLogXmlToStdOut(this, "CtrlLogXmlToStdOut", (tU32)false,
                          new swu::DataStoreItemDescription<tU32>(DataStore::instance(), "CtrlLogXmlToStdOut")),
   */

   /*
     tells ctrl to log the progress-section to stdout.
     Data-store does not work persitently for developer-update, since the datastore is erase at start
     of developer update.
    */
   cfg_CtrlLogXmlToStdOut(this, "CtrlLogXmlToStdOut", (tU32)false,
                          new swu::MarkerFileItemDescription(FCSWUPD_MARKER_FILE_CTRL_LOG_XML_TO_STDOUT)),
   /*
     Only if set, update can be resumed after power-cycle.
    */
   cfg_CtrlCreateWithPersistentProgress(this, "CtrlCreateWithPersistentProgress", (tU32)false),

   /*
     hack to tell fcOta after USB-update that a probably existing ota-delta-update is no longer valid 
     since we changed the from-version.
     Relalized by clearing to ota-partition.
     todo: handle by functional interface.
    */
   cfg_NotifyOtaResultAfterPowerCycle(this, "NotifyOtaResultAfterPowerCycle", (tU32)false),


   /*
     hack to tell fcOta after USB-update that a probably existing ota-delta-update is no longer valid 
     since we changed the from-version.
     Relalized by clearing to ota-partition.
     todo: handle by functional interface.
    */
   cfg_ClearOutdatedOtaUpdate(this, "ClearOutdatedOtaUpdate", (tU32)false),


   /*
     Notify fc-ota that the ota update failed in fg-phase.
     Since we will only recover via a usb-update, the ota-update has to be regarded as failed.
    */
   cfg_MarkerOtaFgUpdateFailed(this, "MarkerOtaFgUpdateFailed", (tU32)false,
                           new swu::MarkerFileItemDescription("/var/opt/bosch/persistent/fcota/ota_update_failed.txt", 10600, 10600 /*aid_fota*/)),


   /*
     marker to tell fcOta that an USB-update happened that inavlidate current OTA-session
     file must be evaluated by fcOta on startup.
    */
   cfg_MarkerUsbUpdateDone(this, "MarkerUsbUpdateDone", (tU32)false,
                           new swu::MarkerFileItemDescription("/var/opt/bosch/persistent/fcota/usb_update_done.txt", 10600, 10600 /*aid_fota*/)),
   /*
     Configures, that after finalizing Recovery-mode we shall automatically reboot the HMI-mode.
     (without waiting for removal of stick)
     Must be disabled during runtime if update was started by force.dnl ... or we would end up 
     in endless updates.
    */
   cfg_CtrlAutoReEnterHmiMode(this, "CtrlAutoReEnterHmiMode", (tU32)true),
   /*
     if cfg_NotifyHmiResultOnSuccess is set and HMI does not send a UpdateEndRequest, we never finalize
     the update, no further updates are possible.
     By setting a value, Ctrl will automatically finalize the update after the configured time.
    */
   cfg_CtrlFullOpResultSupervisionSec(this, "CtrlFullOpResultSupervisionSec", (tU32)0),
   /*
     if recovery update is a success and USB stick is not removed, the target is not powered down
     By setting a value, Ctrl will automatically reboot the target after the configured time.
    */
   
   cfg_CtrlRecoveryResultSupervisionSec(this, "CtrlRecoveryResultSupervisionSec", (tU32)0),

   // simple global variable, media-type will be written by configurator and controll
   cfg_CurrentMediaType(this, "CurrentMediaType", (tU32)tenSourceType_other),

   /* not used in rnaivi, check if it can be removed. 
      will set DoShutdownAfterUpdate at startup in recovery mode
   */
   cfg_ForceShutdownRequested(this, "ForceShutdownRequested", 0,
                              new swu::MarkerFileItemDescription(FCSWUPD_MARKER_FILE_FORCESHUTDOWN_REQUESTED)),
   /* currently not evaluated. If set, the decisions-section of the bosch.xml should be removed/ignored
      normally decision-section should only be created by configurator
    */
   cfg_UseBoschXmlDecisionsSection(this, "UseBoschXmlDecisionsSection", (tU32)true),

   /* indicate from recovery-mode to normal-mode that the recovery-update was 
      successfull.
      Needed when we don't reboot into normal mode in result-state
   */
   cfg_RevoveryPassed(this, "RevoveryPassed", 0,
                      new swu::MarkerFileItemDescription(FCSWUPD_MARKER_FILE_RECOVERY_PASSED)),


   /*
     Configure that scomo-interface shall be used instead of ota-interface.
    */
   cfg_UseScomo(this, "UseScomo", 0),

   /*
     Configure phase-handlng of scomo, see enum tenScomoPhaseHandlingMode.
    */
   cfg_ScomoPhaseHandlingMode(this, "ScomoPhaseHandlingMode", 0),

   /*
     currently running scomo-phase
    */
   cfg_ScomoPhase(this, "ScomoPhase", 0),

   /*
     DataPath where the ota-update is stored.
     We need persistent storage here to resume ota-update after power-cycle or in recovery-mode, because fc-ota will not tell us again.
    */   
   cfg_OtaDataPath(this, "OtaDataPath", "",
                   new swu::DataStoreItemDescription<std::string>(DataStore::instance(), 
                                                                  "otaDataPath")),
    /*
     directory where the iso-container of the ota-update will be mounted.
     todo: rename, we don't unpack.
    */
   cfg_OtaUnpackedDir(this, "OtaUnpackedDir", _tmpBase + "/otaUnpacked"),

   /*
     introduced for GM-project. Decides how often an ota-update shall be retries in case of failure.
     Currently not evaluated.
     todo: analyze current handling of failed ota-updates
     todo: remove if really not needed
    */
   cfg_OtaMaxNumTries(this, "OtaMaxNumTries", 0),
   /*
     persitent storage, how often we already tried
     todo: remove if really not needed
    */
   cfg_OtaNumTries(this, "OtaNumTries", 0,
                   new swu::DataStoreItemDescription<tU32>(DataStore::instance(), "OtaNumTries")),

   /*
     ASF-configuration on which port we shall connect to ota-service
    */
   cfg_OtaProxyPort(this, "OtaProxyPort", "fcOtaPort"),

   /*
     If set to true, scomo-proxy shall be used for ota-update
    */
   cfg_OtaEnableScomo(this, "OtaEnableScomo", 0),

   /*
     File stores information how to mount scomo-input for ivi/firmware in recovery-mode
     Note that in recovery-mode dp.iso ecu.iso and comp.iso are not mounted by fcOta.
    */
   cfg_ScomoFirmwareMountCfg(this, "ScomoFirmwareMountCfg", ""),

   /*
     Path and name of tools-container on target to be used by source-type SCOMO_INDEX
    */
   cfg_DefaultToolsContainerPathAndName(this, "DefaultToolsContainerPathAndName",
                                        std::string("/var/opt/bosch/firmware/") + ((swu::BootChain::usedBootChain() == 1) ? "1" : "2") + "/ivi/firmware/swutools/container.iso.bin"),

   /*
     persistent storage of a special location of an update-medium.
     Used for post-image-mode, see Msg_UpdateSelectReleaseByFileRequest
    */
   cfg_SpecialReleasePathAndName(this, "SpecialReleasePathAndName", "",
                                 new swu::DataStoreItemDescription<std::string>(DataStore::instance(),
                                                                                "SpecialReleasePathAndName")),

   /*
     tells ctrl, how often flashing of a module shall be retried until the complete update is set to failed.
    */
   cfg_CtrlMaxModRetries(this, "CtrlMaxModRetries", 0),
   
   /*
    use the updateTime & Percentage for Phase to Update to HMI.
   */
   cfg_UseEstimatedTimePerPhaseOnly(this, "UseEstimatedTimePerPhaseOnly", 0),


   /*
     config-items to retrieve versions of sub-processors.
     Attention: some of them currently only work in normal-mode.
     e.g. Versions are only written to registry in normal-mode
     todo: cleanup of handling needed
     Currently ModVersionIf should be used in the code.

    */
   cfg_AdrSwVersionLong(this, "AdrSwVersionLong", "",
                    new swu::RegistryItemDescription<std::string>(getRegVersionsPath(), getRegKeyAdrSwVersion())),
   cfg_AdrSwVersion(this, "AdrSwVersion", "", 
                    new AdrVersionItemDescription()),
   cfg_AdrSwVersionBosch(this, "AdrSwVersionBosch", "", 
                    new AdrVersionItemDescription()),
   cfg_V850AppVersion(this, "V850AppVersion", "",
                      new swu::RegistryItemDescription<std::string>(getRegVersionsPath(), "SCC_APP_LABEL")),
   cfg_V850BlVersion(this, "V850BlVersion", "",
                      new swu::RegistryItemDescription<std::string>(getRegVersionsPath(), "SCC_BL_LABEL")),
   cfg_TeseoVersionU32(this, "TeseoVersionU32", 0,
                       new swu::RegistryItemDescription<tU32>(getRegBasePath()+"/GNSS_DEVICE_FW_CRC_CHECKSUM", "GNSS_FW_CRC_CHECKSUM")),
   cfg_TeseoVersion(this, "TeseoVersion", "",
                    new ConfigItemTeseoVersion()),

   cfg_SxmVersion(this, "SxmVersion", "",
                    new swu::RegistryItemDescription<std::string>("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/PROCSXM/FC_SXM", "SXM_SW_VER")),

   cfg_DtvVersion(this, "DtvVersion", "",
                  new swu::RegistryItemDescription<std::string>("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS", "DTV_FW_VERSION")),


   /*
     in normal-mode, we get the HW-Type of the SXM-module from the registry.
     We need the hw-type to select the correct sxm-firmware in the updater and to get the correct version from the bosch.xml.
    */
   cfg_SxmHwType(this, "SxmHwType", "",
                    new swu::RegistryItemDescription<std::string>("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS", "SXM_TYPE")),

   // version of the currently installed pd-config
   cfg_PdConfigVersion(this, "PdConfigVersion", std::vector<tU8>(8),
                              new swu::KdsItemDescription<std::vector<tU8> >(0xD30,
                                                                             20, 0, 20,
                                                                             swu::enKsdEntryType_Data),
                              SWU_CFG_ITEM_EXPORT),


   // version of the currently installed cd-default-config
   cfg_CdDefConfigVersion(this, "CdDefConfigVersion", std::vector<tU8>(8),
                          new swu::KdsItemDescription<std::vector<tU8> >(0xA175,
                                                                         20, 0, 20,
                                                                         swu::enKsdEntryType_Data),
                          SWU_CFG_ITEM_EXPORT),

   // version of the currently installed cd-config
   cfg_CdConfigVersion(this, "CdConfigVersion", std::vector<tU8>(8),
                              new swu::KdsItemDescription<std::vector<tU8> >(0xA115,
                                                                             20, 0, 20,
                                                                             swu::enKsdEntryType_Data),
                              SWU_CFG_ITEM_EXPORT),

   /*
     Idle time in recovery-mode when no update is started.
     On timeout the system shall reset or shutdown
     todo: check where behavior reset/shutdown is configured
    */
   cfg_RecoveryMaxIdleTimeMs(this, "RecoveryMaxIdleTimeMs", (tU32)FCSWUPD_IDLE_TIMER_MS),
   /*
     same as cfg_RecoveryMaxIdleTimeMs but for the case of an ota-update
    */
   cfg_RecoveryOtaMaxIdleTimeMs(this, "RecoveryOtaMaxIdleTimeMs", (tU32)FCSWUPD_IDLE_TIMER_OTA_RECOVERY_MS),
   /*
     todo: remove
    */
   cfg_ImxBootRfsFailure(this,"ImxBootRfsFailure", (tU32)0),
   /*
     enable multi-language support of swu_hmi (recovery-mode)
    */
   cfg_MultiLanguageSupportStatus(this,"MultiLanguageSupportStatus",(tU32)0),
   /*
     location to store fonts and hmi_settings.xml to make them accessible in recovery mode swu_hmi ().
     not all images can be made accessible in recovery-mode due to limited storage (initramfs or persistent partition)
    */
   cfg_MultiLangLocInNormalMode(this,"MultiLangLocInNormalMode", ""),
   /*
     location to store images to make them accessible in recovery mode swu_hmi ().
    */
   cfg_ImageLocInNormalMode(this, "ImageLocInNormalMode", ""),
  
   cfg_RecoveryCfgUseXMLReleaseFilter(this, "RecoveryCfgUseXMLReleaseFilter", (tU32)false),
   cfg_RecoveryCfgDefaultReleaseFilter(this, "RecoveryCfgDefaultReleaseFilter", (tU32)2), // Any, see fcswup_configuratorUtil.h

   cfg_HmiCtrlAutoRebootOnUpdateFailure(this, "CtrlHmiAutoRebootOnUpdateFailure", (tU32)false),
   cfg_HmiCtrlFullOpIfSuccessRebootOnStickRemoval(this, "CtrlHmiFullOpIfSuccessRebootOnStickRemoval", (tU32)false),
   cfg_HmiCfgNeedsSecureBootErrors(this, "HmiCfgNeedsSecureBootErrors", (tU32)false),
   cfg_HmiCfgReportMissingCMSAsSignatureError(this, "HmiCfgReportMissingCMSAsSignatureError", (tU32)false),
   // config items from OTP
   cfg_SecureBootModeEnabled(this, "SecureBootModeEnabled", (tU32)false, new ConfigItemSecureBootModeEnable()),
   cfg_SRKHash(this, "SRKHash", "", new ConfigItemSRKHash()),
   cfg_SRKRevocation(this, "SRKRevocation", (tU32)false, new ConfigItemSRKRevocation()),
 
   cfg_AllowMisCmcRelease(this,"AllowMisCmcRelease",(tU32)false),
   
   cfg_InitialProgrammingDateRequested(this,"InitialProgrammingDateRequested",(tU32)false),

   //If set RunningOverallCISSWVer version 
   cfg_RunningOverallCISSWVer(this, "RunningOverallCISSWVer", "", 
                              new RunningOverallCISSWVer)
   
{
   ETG_I_REGISTER_FILE();
}
Config::~Config() {
   ETG_I_UNREGISTER_FILE();
}






void Config::traceState() {
   printf("Config::traceState START\n");
   ETG_TRACE_COMP(("  logDirName              =%s", _logDirName.c_str()));
   ETG_TRACE_COMP(("  progessFilePersistent   =%s", _progessFilePersistent.c_str()));
   ETG_TRACE_COMP(("  historyFilePersistent   =%s", _historyFilePersistent.c_str()));
   ETG_TRACE_COMP(("  historyFileStatic       =%s", _historyFileStatic.c_str()));
   ETG_TRACE_COMP(("  dataStoreFilePersistent =%s", _dataStoreFilePersistent.c_str()));
   ETG_TRACE_COMP(("  dataStoreFileStatic     =%s", _dataStoreFileStatic.c_str()));
   ETG_TRACE_COMP(("  defaultUpdadeKeyStatic  =%s", _defaultUpdadeKeyStatic.c_str()));
   ETG_TRACE_COMP(("  hmiSettingsFilePersistent  =%s", _hmiSettingsFilePersistent.c_str()));
   ETG_TRACE_COMP(("  _bSkipPropertyDeregister=%u", _bSkipPropertyDeregister));
   ETG_TRACE_COMP(("  _rebootWaitTimeMs       =%u", _rebootWaitTimeMs));
   ETG_TRACE_COMP(("  _spmWaitTimeMs          =%u", _spmWaitTimeMs));
   ETG_TRACE_COMP(("  _spmWaitAvailTimeMs     =%u", _spmWaitAvailTimeMs));
   ETG_TRACE_COMP(("  IgnoreCkSumsAndSig      =%u", getIgnoreCkSumAndSig()));
   ETG_TRACE_COMP(("  SimKds                  =%u", getSimKds()));


   swu::ConfigBase::traceState();
   printf("Config::traceState END\n");

}



tUInt Config::getInstallTimeoutMs() {
   return _installTimeoutMs;
}

tUInt Config::getAbortInstallTimeoutMs() {
   return _abortInstallTimeoutMs;
}


std::string const &Config::getLogDirName() {
   return _logDirName;
}


std::string const &Config::getPersitentDirName() {
   return _persistentBase;
}

std::string const &Config::getProgressFileName() {
   return _usePersistentPart ? _progessFilePersistent : _progessFileTmp;
}

std::string const &Config::getUpdateTxtPersistentPrefix() {
   return _updateTxtPersistentPrefix;
}

std::string const &Config::getPersistentHistoryFileName() {
   return _historyFilePersistent;
}

std::string const &Config::getStaticHistoryFileName() {
   return _historyFileStatic;
}

std::string const &Config::getPersistentDataStoreFileName() {
   return _dataStoreFilePersistent;

}

std::string const &Config::getNonPersistentDataStoreFileName() {
   return _dataStoreFileNonPersistent;
}

std::string const &Config::getStaticDataStoreFileName() {
   return _dataStoreFileStatic;
}

std::string const &Config::getStaticUpdateKeyFileName() {
   return _defaultUpdadeKeyStatic;
}

std::string const &Config::getRegBasePath() {
   return _regBasePath;
}

std::string const &Config::getRegVersionsPath() {
   return _regVersionsPath;
}
std::string const &Config::getRegKeyAdrSwVersion() {
   return _regKeyAdrSwVersion;
}
std::string const &Config::getRegKeyAdrSwVersionBosch() {
   return _regKeyAdrSwVersionBosch;
}

std::string const &Config::getHMISettingsPath() {
  return _hmiSettingsPath;
}

std::string const &Config::getHMISettingsFile() {
  return _hmiSettingsFilePersistent;
}

bool Config::getUseSpmUpdateMode() {
   return cfg_UseSpmUpdateMode.readAsBool();
}

void Config::setUseSpmUpdateMode(bool bUseSpmUpdateMode) {
   cfg_UseSpmUpdateMode.setFromBool(bUseSpmUpdateMode);
}

bool Config::getImxBootRfsFailure() {
   return cfg_ImxBootRfsFailure.readAsBool();
}
void Config::setImxBootRfsFailure(bool bUseImxBootRfsFailure) {
   cfg_ImxBootRfsFailure.setFromBool(bUseImxBootRfsFailure);
}
bool Config::getSkipPropertyDeregister() {
   return _bSkipPropertyDeregister;
}

void Config::setSkipPropertyDeregister(bool bSkipPropertyDeregister) {
   _bSkipPropertyDeregister=bSkipPropertyDeregister;
}

tUInt Config::getRebootWaitTimeMs() {
   return _rebootWaitTimeMs;
}

void Config::setRebootWaitTimeMs(tUInt rebootWaitTimeMs) {
   _rebootWaitTimeMs=rebootWaitTimeMs;
}

tUInt Config::getSpmWaitTimeMs() {
   return _spmWaitTimeMs;
}

void Config::setSpmWaitTimeMs(tUInt spmWaitTimeMs) {
   _spmWaitTimeMs=spmWaitTimeMs;
}

tUInt Config::getSpmWaitAvailTimeMs() {
   return _spmWaitAvailTimeMs;
}

void Config::setSpmWaitAvailTimeMs(tUInt spmWaitAvailTimeMs) {
   _spmWaitAvailTimeMs=spmWaitAvailTimeMs;
}

bool Config::getUsePersistentPart() {
   return _usePersistentPart;
}

void Config::setUsePersistentPart(bool doUse) {
   _usePersistentPart=doUse;
}

bool Config::getIgnoreCkSumAndSig() {
#if 1
   return true;
#else
   tU32 doIgnore;
   bool res= Config::instance()->getConfigVal(enConfigItem_DoIgnoreCkSumAndSig, doIgnore);
   ETG_TRACE_USR1(("Config::getIgnoreCkSumAndSig(): res=%u doIgnore=%u", res, doIgnore));
   return res && doIgnore;
#endif
}

void Config::setIgnoreCkSumAndSig(bool doIgnore) {
   Config::instance()->cfg_DoIgnoreCkSumAndSig.setFromBool(doIgnore, 1);
}

bool Config::getSimKds() {
   return  Config::instance()->cfg_DoSimKds.readAsBool();
}

void Config::setSimKds(bool doSimulate) {
   Config::instance()->cfg_DoSimKds.setFromBool(doSimulate);
}

// gen3armmake, gen3x86make, gen4rcar, gen4lsim: conversion to 'tU16 {aka short unsigned int}' from 'tU32 {aka unsigned int}'
// The values of key, size are stored as tU16 so their type is changed from tU32 to uint_least16_t
void Config::setKdsString(uint_least16_t key, uint_least16_t size, const char *val) {
   ETG_TRACE_USR4(("Config::setKdsString() key=0x%X, size=%u value=%s", key, size, val));

   swu::KdsItemDescription<std::vector<tU8> > kdsItem(key, size, 0, size, swu::enKsdEntryType_Data);
   std::string kdsValString(val);
   std::vector<tU8> kdsU8VecValue(kdsValString.begin(), kdsValString.end());
   //std::copy(kdsValString.begin(), kdsValString.end(), std::back_inserter(kdsU8VecValue));
   bool status = kdsItem.write(kdsU8VecValue);
   ETG_TRACE_COMP(("Config::setKdsString() result %u for %s", status, kdsValString.c_str()));
}





ETG_I_CMD_DEFINE((setUseSpmUpdateMode, "setUseSpmUpdateMode %u", tU8))
ETG_I_CMD_DEFINE((setSkipPropertyDeregister, "setSkipPropertyDeregister %u", tU8))
ETG_I_CMD_DEFINE((setRebootWaitTimeMs, "setRebootWaitTimeMs %u", tU32))
ETG_I_CMD_DEFINE((setSpmWaitTimeMs, "setSpmWaitTimeMs %u", tU32))
ETG_I_CMD_DEFINE((setSpmWaitAvailTimeMs, "setSpmWaitAvailTimeMs %u", tU32))
ETG_I_CMD_DEFINE((setIgnoreCkSumAndSig, "setIgnoreCkSumAndSig %u", tU32))
ETG_I_CMD_DEFINE((setSimKds, "setSimKds %u", tU8))


ETG_I_CMD_DEFINE((setCfgStr, "setCfgStr %50s %50s", ETG_I_STRING, ETG_I_STRING))
ETG_I_CMD_DEFINE((setCfgStr, "setCfgStr %50s", ETG_I_STRING, ETG_I_CONST_ARG("")))

ETG_I_CMD_DEFINE((setCfgU32, "setCfgU32 %50s %u", ETG_I_STRING, tU32))
ETG_I_CMD_DEFINE((setCfgVectorU8, "setCfgVectorU8 %50s %50s", ETG_I_STRING, ETG_I_STRING))

ETG_I_CMD_DEFINE((traceCfgStr,      "traceCfgStr %50s", ETG_I_STRING))
ETG_I_CMD_DEFINE((traceCfgU32,      "traceCfgU32 %50s", ETG_I_STRING))
ETG_I_CMD_DEFINE((traceCfgVectorU8, "traceCfgVectorU8 %50s", ETG_I_STRING))


ETG_I_CMD_DEFINE((setKdsString, "setKdsString %x %u %50s", uint_least16_t, uint_least16_t, ETG_I_STRING))


}
