/**
 *\file     SWUCoreComponent.h
 *\brief    SW Update Core ASF Component
 *
 *\par Copyright:
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 */

#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <iostream>
#include <fstream>
#include <cstdlib>
#ifndef __SW_UPDATE_UNIT_TESTING__
#include <systemd/sd-daemon.h>
#endif
#include "util/swu_filesystem.h"
#include "SWUCoreComponent.h"
#include "base/imp/ErrorHandler.h"
#include "base/imp/SystemCall.h"
#include "base/imp/checksum.hpp"
#include "base/imp/DownloadPipe.h"
#include "util/swu_securityEngine.h"
#include "base/imp/gen2_FileOperations.h"


#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_SWUPDATE
#define ETG_I_TTFIS_CMD_PREFIX "DNL_"
#define ETG_I_FILE_PREFIX SWUCoreComponent::
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_CORE
#include "trcGenProj/Header/SWUCoreComponent.cpp.trc.h"
#endif

using ai_sw_update::common::SystemCall;

namespace ai_sw_update {
namespace common {
namespace core {
namespace imp {

static bool setupSecurityEngine(const trCkSum &, swu::SecurityEngine &);

SWUCoreComponent *SWUCoreComponent::_instance=0;

DEFINE_CLASS_LOGGER_AND_LEVEL("swupdate/swupdatecore", SWUCoreComponentStub, Info);

static bool fileEmpty(const char *filename) {
   FILE *fp = fopen(filename, "r"); /* try to open file */
   if (fp == NULL)
      return false; /* open failed */
   fseek(fp, 0L, SEEK_END);
   size_t sz = ftell(fp);
   fclose(fp);
   return sz == 0;
}

static tUInt u32GetEnvVarVal(std::string key)
{
   tUInt foundVal=0;
   char *envValue = getenv(key.c_str());
   if (envValue) {
      foundVal = (tUInt)atoi(envValue);
   }
   ETG_TRACE_USR2(("u32GetEnvVarVal(key=%30s)=%u", key.c_str(), foundVal));
   return foundVal;
}

SWUCoreComponent::SWUCoreComponent() :
      ::asf::core::BaseComponent(), _componentStub() {
   vInitPlatformEtg();
   ETG_I_REGISTER_FILE();
   ETG_I_REGISTER_CHN(0);
   _instance = this;

   const std::string ENV_RECOVERY_MODE("SW_UPDATE_RECOVERY_MODE");
   const std::string ENV_ENGINEERING_MODE("SW_UPDATE_ENGINEERING_MODE");
   if(u32GetEnvVarVal(ENV_ENGINEERING_MODE)!=1 && u32GetEnvVarVal(ENV_RECOVERY_MODE)!=1)
   {
      // @todo: Notification is done in full operation mode only because there is no systemd in recovery mode
      // @todo: When systemd finally becomes available in recovery mode, the notification has to be send too
#ifndef __SW_UPDATE_UNIT_TESTING__
      sd_notify(0, "READY=1");
#endif
   }

   ETG_TRACE_USR2(("SWUCoreComponent: Ctor"));
}

SWUCoreComponent::~SWUCoreComponent() {
   ETG_TRACE_USR2(("SWUCoreComponent: Dtor"));
}

SWUCoreComponentStub::SWUCoreComponentStub() :
   SWUCoreServiceStub("SWUCoreServicePort"),
   _LuaInterpreter(),
   _isoContainer("/tmp/iso", "/tmp/swupdate/container.iso"),
   _last_mod_succeded(true),
   _isoContainerValid(false) {
   ETG_TRACE_USR2(("SWUCoreComponentStub: Ctor"));
}

SWUCoreComponentStub::~SWUCoreComponentStub() {
   ETG_TRACE_USR2(("SWUCoreComponentStub: Dtor"));
}
void SWUCoreComponentStub::onEnterUpdateSessionRequest(const ::boost::shared_ptr < SWUCoreService::EnterUpdateSessionRequest >& /*request*/) {
   bool success=false;
   ETG_TRACE_COMP(("SWUCoreComponentStub: EnterUpdateSession"));
   LOG_INFO("SWUCoreComponentStub: EnterUpdateSession");

   printToDownloadPipe("*****************************************************************");
   printToDownloadPipe("** Will start download");
   printToDownloadPipe("*****************************************************************");
   
   const char* initalPermPersisMarker="/var/opt/bosch/persistent/initial_permissions_set.txt";
   const char* initalPermDynamicMarker="/var/opt/bosch/dynamic/initial_permissions_set.txt";
   const char* postImageRegMarker="/var/opt/bosch/persistent/fcswupdate/postImage/autoHandlePostImgPending";

   if(swu::exists(initalPermPersisMarker)) {
      swu::removeFile(initalPermPersisMarker);
   }
   if(swu::exists(initalPermDynamicMarker)) {
      swu::removeFile(initalPermDynamicMarker);
   }
   if(swu::exists(postImageRegMarker)) {
      swu::removeFile(postImageRegMarker);
   }

   
   // Create Ramdisk for download
   // returns 0 on success
   if(!mkdir("/tmp/swupdate", 0755)){
      if (SystemCall::exec("mount -n -o mode=0755,size=100M -t tmpfs none /tmp/swupdate")) {
         ETG_TRACE_COMP(("SWUCoreComponentStub: mounted Ramdisk"));
         sendEnterUpdateSessionResponse(true);
         success=true;
      }
   }	
   // In case of mkdir/mount failure, send the session response as false
   if(!success){
      ETG_TRACE_ERR(("SWUCoreComponentStub: mkdir/mounted Ramdisk failure"));
      sendEnterUpdateSessionResponse(false);
   }
}

void SWUCoreComponentStub::onExitUpdateSessionRequest(const ::boost::shared_ptr < SWUCoreService::ExitUpdateSessionRequest >& /*request*/) {
   ETG_TRACE_COMP(("SWUCoreComponentStub: ExitUpdateSession"));
   LOG_INFO("SWUCoreComponentStub: ExitUpdateSession");

   // unmount ISO container -> OK: IsoContainer
   _isoContainer.umount();
   unlink(_isoContainer.getSrcFile().c_str());
   _isoContainerValid=false;

   //Release Ramdisk
   if (!SystemCall::exec("umount /tmp/swupdate")) {
      ETG_TRACE_ERR(("SWUCoreComponentStub: Could not umount Ramdisk"));
//      sendExitUpdateSessionResponse(false);
   }
   rmdir("/tmp/swupdate");

   printToDownloadPipe("*****************************************************************");
   printToDownloadPipe("** download completed");
   if (_last_mod_succeded) {
      printToDownloadPipe("** Last mod was successful");
   } else {
      printToDownloadPipe("** Last mod failed");
   }
   printToDownloadPipe("*****************************************************************");
   sendExitUpdateSessionResponse(true);
}

void SWUCoreComponentStub::onAbortInstallRequest(const ::boost::shared_ptr < SWUCoreService::AbortInstallRequest >& request) {
   ETG_TRACE_COMP(("SWUCoreComponentStub: AbortInstall"));
   LOG_INFO("SWUCoreComponentStub: AbortInstall");
   //ToDo: change with implementation later, this snippet is only for first initial response.
   std::string outModId = request->getInModId();
   // you have to decide if called class has to deal with asf-types or if you wrap everything
   sendAbortInstallResponse(true, outModId);
}

void SWUCoreComponentStub::onRequestRecoveryDownloadRequest(const ::boost::shared_ptr < SWUCoreService::RequestRecoveryDownloadRequest >& /*request*/) {
   ETG_TRACE_COMP(("SWUCoreComponentStub: RequestRecoveryDownload"));
   LOG_INFO("SWUCoreComponentStub: RequestRecoveryDownload");
   //ToDo: change with implementation later, this snippet is only for first initial response.
   sendRequestRecoveryDownloadResponse(true);
}

void SWUCoreComponentStub::onSetIsoPathRequest(const ::boost::shared_ptr < SWUCoreService::SetIsoPathRequest >& request) {
   ETG_TRACE_COMP(("SWUCoreComponentStub: SetIsoPath"));
   LOG_INFO("SWUCoreComponentStub: SetIsoPath");
   //ToDo: change with implementation later, this snippet is only for first initial response.
   std::string outIsoPath = request->getInIsoPath();
   sendSetIsoPathResponse(true, outIsoPath);
}

void SWUCoreComponentStub::onStartInstallRequest(const ::boost::shared_ptr < SWUCoreService::StartInstallRequest >& request) {
   int iErrorCount = 0;
   int iCoreInstallResult = 0;
   tenInstallResult enInstallResultValue = tenInstallResult__OK;

   ETG_TRACE_FATAL(("SWUCoreComponentStub: onStartInstallRequest"));
   ETG_TRACE_COMP(("SWUCoreComponentStub: modID (%s)", request->getInModId().c_str()));
   ETG_TRACE_COMP(("SWUCoreComponentStub: toolsContainer (%s)", request->getToolsContainer().c_str()));
   ETG_TRACE_COMP(("SWUCoreComponentStub: dataPath (%s)", request->getDataFile().c_str()));
   ETG_TRACE_COMP(("SWUCoreComponentStub: usb log path (%s)", request->getUsbLogPath().c_str()));
   LOG_INFO("SWUCoreComponentStub: onStartInstallRequest");
   LOG_INFO("SWUCoreComponentStub: modID (%s)", request->getInModId().c_str());
   LOG_INFO("SWUCoreComponentStub: toolsContainer (%s)", request->getToolsContainer().c_str());
   LOG_INFO("SWUCoreComponentStub: dataPath (%s)", request->getDataFile().c_str());
  
    
   sendStartInstallResponse(request->getInModId(), tenInstallResult__OK);
   
   _last_mod_succeded = false;
    usbLogPath.assign(request->getUsbLogPath());
   ETG_TRACE_COMP(("SWUCoreComponentStub: usb log Saved (%s)", usbLogPath.c_str()));

   /*
    Example:
    module imx_main_application
    toolspath /testing/media/usb//testing/media/usb/IMx/tools/150/default/file/tools.iso
    datapath /testing/media/usb//testing/media/usb/IMx/Main/150/default/Application/IMx_Application.tar.gz

    Sketch of actions

    SWUCoreServiceStub::InstallModule()
    - mount ISO container -> mount toolsPath
    - locate lua script -> OK
    - set lua environment -> OK
    - execute lua script -> OK: LuaInterpreter
    - unmount ISO container -> OK: IsoContainer
    - sendResponse


    --- script.lua ---
    mediumDownloadPath = mediumPathISO;


    --- config_tooling.lua ---
    mediumStickPath = "/dev/media/usbms1"; -> mount point not used in VW 'app mode'
    mediumLUAPath = mediumDownloadPath.."/scripts/lua/";
    mediumSHPath = mediumDownloadPath.."/scripts/sh/";

    shell_executeable = "/bin/sh ";
    dnlid = os.getenv("dnlid");         // @CF3: how to get in in normal mode?


    -- Switches for MOD_IDs to update ( 1 active / 0 inactive )
    MOD_ID_IMX = 1
    MOD_ID_ADR = 1
    MOD_ID_V850 = 1
    MOD_ID_TESEO = 1
    --- config_tooling: ---


    _writeMagic (RECOVERY_DOWNLOAD_MAGIC_ADDRESS, RECOVERY_DOWNLOAD_MAGIC_ACTIVE);

    --- update_imx.lua ---
    // is overridden in config_imx.lua ???
    bootchain_magic = {_readMagic(BOOTCHAIN_MAGIC_ADDRESS)};

    --- config_imx.lua ---
    MOD_ID_IMX_PARTITIONING = 1
    MOD_ID_IMX_BOOTCHAIN = 1
    MOD_ID_IMX_RFS = 1

    //@Bosse: should be dependant on active bootchain, or? Diff between Virgin and non-virgin mode?
    bootchain_magic = BOOTCHAIN_MAGIC_2_active

    BOOTCHAIN_VAR= ...
    MKFS_OPTIONS_VAR=...
    RFS_SOURCE_VAR=...
    */
   std::string scriptDir = _isoContainer.getMountPoint() + "/scripts/lua/";
   bool bMountIso = true;
   bool bLoadIso=true;
   SystemCall::clearError();

#ifdef LUA_RESTART
   _LuaInterpreter.LuaInitFramework();
#endif
   _LuaInterpreter.luaInitEnv("SwuEnv");
   if (request->getInModId() == "testLua") {
      // todo: check if this can be removed
      ETG_TRACE_USR4(("execute testLua"));
      // special test-command
      bMountIso = false;
      _LuaInterpreter.setGlobalVariable("tooling_location", request->getToolsContainer());
      scriptDir = request->getToolsContainer() + "/scripts/lua/";
   }

   // mount tools ISO container
   else if (fileEmpty(request->getToolsContainer().c_str())) {
      ETG_TRACE_COMP(("SWUCoreComponentStub: InstallResponse OK (patch for empty tools-container)"));
      LOG_INFO("SWUCoreComponentStub: InstallResponse OK (patch for empty tools-container)");
      _last_mod_succeded = true;
      printToDownloadPipe("*****************************************************************");
      printToDownloadPipe("*** SWUCoreComponent:received empty tools-container");
      printToDownloadPipe("*** Send StartInstallRespose(OK)");
      printToDownloadPipe("*****************************************************************");
      sendStartInstallResponseSignal(request->getInModId(), tenInstallResult__OK);
      return;
   }
   else if (_isoContainerValid && _lastContainerCkSum==request->getToolsCksum()) {
      // container is unchanged and already mounted
      bLoadIso=false;
   }

   // Initialize the error handler array
   iInitializeSwUpdateErrorArray();
   swu::SecurityEngine dataEngine;
   std::string errMsg;
   if (bMountIso) {
      // invalidate current container-checksum
      do {
         if (bLoadIso) {
            // todo: check if lua-container is only loaded once, if not fix it.
            bool isoOk=false;
            _isoContainerValid=false;
            swu::SecurityEngine containerEngine;
            if (false == setupSecurityEngine(request->getToolsCksum(), containerEngine))
            {
               errMsg="Can't extract ISO ChecksumData from request";
               enInstallResultValue=tenInstallResult__ERR_UNKNOWN;
               break;
            }
            if(!containerEngine.processAndCatenate(request->getToolsContainer(),
                                                   "/tmp/swupdate/container.iso")) 
            {
               errMsg="ISO container copy through verify failed";
               enInstallResultValue=tenInstallResult__ERR_UNKNOWN;
               break;
            }
            // mount tools ISO container
            _isoContainerValid=_isoContainer.mount();
            if (!_isoContainerValid) {
               errMsg="Can't mount ISO";
               enInstallResultValue=tenInstallResult__ERR_IMAGE_READ;
               break;
            }
            
            // container is validated and mounted, store it, so we don't have to copy, check and mount it again
            _lastContainerCkSum=request->getToolsCksum();
         }
         
         if (false == setupSecurityEngine(request->getDataFileDataCksum(), dataEngine)) 
         {
            errMsg="Can't extract Archive ChecksumData from ";
            enInstallResultValue=tenInstallResult__ERR_UNKNOWN;
            break;
         }
         if (false == dataEngine.writeParameters(dataEngine.PARAMETER_FILENAME())) 
         {
            errMsg="Can't store Archive ChecksumData as a file";
            enInstallResultValue=tenInstallResult__ERR_UNKNOWN;
            break;
         }
      } while (false);
      if (enInstallResultValue != tenInstallResult__OK) {
         printToDownloadPipe("*****************************************************************");
         ETG_TRACE_ERR(("*** %s", errMsg.c_str()));
         printToDownloadPipe(std::string("*** Send StartInstallRespose(ERROR) :" + errMsg));
         printToDownloadPipe("*****************************************************************");
         sendStartInstallResponseSignal(request->getInModId(), enInstallResultValue);
         return;
      }

      _LuaInterpreter.addToSwuEnvTable("CHECKSUM_FILENAME", dataEngine.PARAMETER_FILENAME());

      // locate lua script -> TBC map id to lua script
      // set lua environment. this shall be the only global varible ...
      _LuaInterpreter.setGlobalVariable("tooling_location", _isoContainer.getMountPoint());

   }

   // ... all other parameter are passed to the lua-table SwuEnv
    _LuaInterpreter.addToSwuEnvTable("data_location", request->getDataFile());
    //   ::boost::ptr_vector < trKeyValuePair > const &pairVec = request->getKeyValuePairs().getRKeyValuePair();
   ::std::vector < trKeyValuePair > const &pairVec = request->getKeyValuePairs().getRKeyValuePair();
   for (unsigned int i = 0; i < pairVec.size(); ++i) {
      trKeyValuePair const &pair = pairVec[i];
      ETG_TRACE_USR4(("Adding to lua environment %20s = %20s", pair.getKey().c_str(), pair.getValue().c_str()));
      LOG_INFO("SWUCoreComponentStub: %20s = %20s", pair.getKey().c_str(), pair.getValue().c_str());

      _LuaInterpreter.addToSwuEnvTable(pair.getKey(), pair.getValue());

#if 0
      if (pair.getKey() == "dnlid") {
         ETG_TRACE_USR4(("Adding to OS environment %20s = %20s", pair.getKey().c_str(), pair.getValue().c_str()));
         LOG_INFO("SWUCoreComponentStub: to OS %20s = %20s", pair.getKey().c_str(), pair.getValue().c_str());
         if (setenv(pair.getKey().c_str(), pair.getValue().c_str(), 1)) {
            ETG_TRACE_ERR(("Error setting OS env: %u", errno));
            ETG_TRACE_ERR(("Error setting OS env: %u", strerror(errno)));
         }
      }
#endif
   }
   _LuaInterpreter.luaSaveEnv();
   //@todo setGlobalVariable can only set simple name (string) / value (string) pairs. TBC How complex structures can be given to the lua environment
   // _LuaInterpreter.setGlobalVariable("datafile_checksum", request->getDataFileDataCksum());

   // Derive file name of lua script to execute for this modid from the modid name provided by the client.
   // The tooling generating the content of the update medium must use the identical file name.
   const std::string luaScriptFileName = request->getInModId() + ".lua";

   // execute the lua script
   _LuaInterpreter.execScript(scriptDir + luaScriptFileName);

   ETG_TRACE_COMP(("SWUCoreComponentStub: InstallResponse"));

   LOG_INFO("SWUCoreComponentStub: InstallResponse");
   //@todo evaluate runtime errors and report correct error code. Currently true is returned in all cases.
   // Get the error count from ErrorHandler and print the result to /tmp/ErrorHandler.txt:
   iErrorCount = iGetSwUpdateError("/tmp/ErrorHandler.txt");
 
   if ((iErrorCount == 0) && (!SystemCall::hasError())) {
      printToDownloadPipe("*****************************************************************");
      printToDownloadPipe("SWUCoreComponentStub: InstallResponse OK");
      printToDownloadPipe("*****************************************************************");
      ETG_TRACE_COMP(("SWUCoreComponentStub: InstallResponse OK"));
      LOG_INFO("SWUCoreComponentStub: InstallResponse OK");

      _last_mod_succeded = true;
      sendStartInstallResponseSignal(request->getInModId(), tenInstallResult__OK);

   } else {
      printToDownloadPipe("*****************************************************************");
      printToDownloadPipe("SWUCoreComponentStub: InstallResponse ERROR");
      printToDownloadPipe("*****************************************************************");
      ETG_TRACE_COMP(("SWUCoreComponentStub: InstallResponse ERROR"));
      LOG_INFO("SWUCoreComponentStub: InstallResponse ERROR");
      // Get the core install result from the error handler
      iCoreInstallResult = iGetSwUpdateCoreInstallResult();

      enInstallResultValue=mapCoreInstallResultToAsf(iCoreInstallResult);

      sendStartInstallResponseSignal(request->getInModId(), enInstallResultValue);
	  

      SystemCall::clearError();
   }
}

void SWUCoreComponentStub::onConfigUpdateSessionRequest(const ::boost::shared_ptr < SWUCoreService::ConfigUpdateSessionRequest >& /*request*/) {
   ETG_TRACE_COMP(("SWUCoreComponentStub: ConfigUpdateSession"));
   LOG_INFO("SWUCoreComponentStub: ConfigUpdateSession");
   //ToDo: change with implementation later, this snippet is only for first initial response.
   // under construction => how to use this template?
   //::boost::ptr_vector<>
   //trModIds modIds;
   //modIds.hasModIds();
   //std::string modID = "TEST_MOD_ID_1";
   //modIds.setModIds(modID);
   //sendConfigUpdateSessionResponse (modIds);
}


// todo: if no longer used, please remove
void SWUCoreComponentStub::ondisplayText(std::string str1, std::string str2, std::string str3){
       trDisplayText enDisplayTextValue;

        ETG_TRACE_USR4(("SWUCoreComponentStub: ondisplayText"));

       //set display text property
	if(!str1.empty()){
	       enDisplayTextValue.setLine1(str1);}

	if(!str2.empty()){
	       enDisplayTextValue.setLine2(str2);}

	if(!str3.empty()){
	       enDisplayTextValue.setLuaCmd(str3);}

       setDisplayText(enDisplayTextValue);
#ifdef VARIANT_S_FTR_ENABLE_SWU_DBUS
       // notify to all the subscribed components
       sendDisplayTextUpdate();
#else
       sendDisplayTextNotification();
#endif
 }

// todo: if no longer used, please remove
void SWUCoreComponentStub::onbgImageDetails(std::string str1, std::string str2){
       trDisplayBgImage enDisplayBgImageValue;
	   
        ETG_TRACE_USR4(("SWUCoreComponentStub: onbgImageDetails"));

        enDisplayBgImageValue.setImgDirPath(str1);

        enDisplayBgImageValue.setImgName(str2);

        setDisplayBgImage(enDisplayBgImageValue);
#ifdef VARIANT_S_FTR_ENABLE_SWU_DBUS
       // notify to all the subscribed components
	   sendDisplayBgImageUpdate();
#else
		sendDisplayBgImageNotification();
#endif
 }

static bool setupSecurityEngine(const trCkSum &securityParam, swu::SecurityEngine &engine)
{
   engine._fileLength = securityParam.getU64Len();
   engine._blockLength = securityParam.getU32SegSize();
   engine._digests = securityParam.getChkSumChunks();
   engine._encryptParam = securityParam.getEncryptParam();

   switch (securityParam.getEnType()) {
   case tenCkSumType__NONE:
      engine._enDigestType = swu::tenDigestTypeNone;
      break;
   case tenCkSumType__SHA1:
      engine._enDigestType = swu::tenDigestTypeSHA1;
      break;
   case tenCkSumType__MD5:
      engine._enDigestType = swu::tenDigestTypeMD5;
      break;
   case tenCkSumType__SHA256:
      engine._enDigestType = swu::tenDigestTypeSHA256;
      break;
   default:
      ETG_TRACE_ERR(("%d is not supported yet as a digest", securityParam.getEnType()));
      return false;
   }
   //printf("XXX:size_t:FileSize:%ju \n", engine._fileLength);
   //printf("XXX:sizeof(off_t):%u \n", sizeof(off_t));
   
   ETG_TRACE_USR2(("!!!!!!!!!!!! file len %u block len %d", (int) engine._fileLength, engine._blockLength));
   ETG_TRACE_USR2(("!!!!!!!!!!!! DigestType %1d digests quantity %3d", engine._enDigestType, engine._digests.size()));
   ETG_TRACE_USR2(("!!!!!!!!!!!! encrypt Param %s", engine._encryptParam.c_str()));
   return true;
}







tenInstallResult SWUCoreComponentStub::mapCoreInstallResultToAsf(int iErrorType) {
   tenInstallResult enResult=tenInstallResult__ERR_UNKNOWN;
   // Find the corresponding core result code
   switch (iErrorType)
   {
      case SW_UPDATE_ERROR_TYPE_NOT_RDY:
         enResult = tenInstallResult__ERR_DEVICE_NOT_READY;
         break;
         
      case SW_UPDATE_ERROR_TYPE_READ:
         enResult = tenInstallResult__ERR_READ;
         break;
         
      case SW_UPDATE_ERROR_TYPE_ERASE:
         enResult = tenInstallResult__ERR_IMAGE_FLASHING;
         break;
         
      case SW_UPDATE_ERROR_TYPE_WRITE:
         enResult = tenInstallResult__ERR_IMAGE_FLASHING;
         break;
         
      case SW_UPDATE_ERROR_TYPE_VERIFY:
         enResult =tenInstallResult__ERR_IMAGE_FLASHING;
         break;
         
      case SW_UPDATE_ERROR_TYPE_CHECKSUM:
         enResult = tenInstallResult__ERR_DATA_CKSUM;
         break;
         
      case SW_UPDATE_ERROR_TYPE_MEMORY_SIZE:
         enResult = tenInstallResult__ERR_IMAGE_SIZE;
         break;
         
      case SW_UPDATE_ERROR_TYPE_INVALID:
         enResult = tenInstallResult__ERR_IMAGE_INVALID_SECTOR;
         break;
         
      case SW_UPDATE_ERROR_TYPE_INVALID_SECTORHEADER:
         enResult = tenInstallResult__ERR_IMAGE_INVALID_SECTOR;
         break;

      case SW_UPDATE_ERROR_TYPE_MSG_DIG_SIG_MISMATCH:
         enResult = tenInstallResult__ERR_TOOLS_CKSUM;
         break;

      case SW_UPDATE_ERROR_TYPE_LIC_DIG_SIG_MISMATCH:
         enResult = tenInstallResult__ERR_TOOLS_CKSUM;
         break;

      case SW_UPDATE_ERROR_TYPE_FILES_NOT_AVAILABLE:
         enResult = tenInstallResult__ERR_DATA_NOT_FOUND;
         break;

      case SW_UPDATE_ERROR_TYPE_NOT_AVAILABLE:
         enResult = tenInstallResult__ERR_TOOLS_NOT_FOUND;
         break;

      case SW_UPDATE_ERROR_TYPE_COMPATIBLE_MISMATCH:
         enResult = tenInstallResult__ERR_IMAGE_INCOMPATIBLE_VERSION;
         break;
			
      case SW_UPDATE_ERROR_TYPE_CHANGE_BOOTMODE:
         enResult = tenInstallResult__ERR_DEVICE_NOT_READY;
         break;

      case SW_UPDATE_ERROR_TYPE_SCRIPT_LOAD:
         enResult = tenInstallResult__ERR_TOOLS_NOT_FOUND;
         break;

      case SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE:
         enResult = tenInstallResult__ERR_IMAGE_FLASHING;
         break;

      case SW_UPDATE_ERROR_TYPE_UNKNOWN_ERROR:
         enResult = tenInstallResult__ERR_UNKNOWN;
         break;

      case SW_UPDATE_ERROR_TYPE_RETRY_RECOVERY:
         enResult = tenInstallResult__RETRY_RECOVERY;
         break;

      case SW_UPDATE_ERROR_TYPE_RETRY_ALL_RECOVERY:
         enResult = tenInstallResult__RETRY_ALL_RECOVERY;
         break;

      default:
         enResult = tenInstallResult__ERR_UNKNOWN;
         break;
   }
   return enResult;
		
}

} // namespace imp
} // namespace core
} // namespace common
} // namespace ai_sw_update

